# Copyright (C) 2011-2018 A S Lewis
#
# This program is free software: you can redistribute it and/or modify it under the terms of the GNU
# General Public License as published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
# even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with this program. If not,
# see <http://www.gnu.org/licenses/>.
#
#
# Games::Axmud::EditWin::XXX
# All 'edit' windows, inheriting from GA::Generic::ConfigWin

{ package Games::Axmud::EditWin::Buffer::Display;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::Generic::EditWin Games::Axmud::Generic::ConfigWin
        Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::Buffer::Display')) {
            return undef;
        } else {
            return 1;
        }
    }

    sub enableButtons {

        # Called by $self->drawWidgets
        # We only need a single button so, instead of calling the generic ->enableButtons, call a
        #   method that creates just one button
        #
        # Expected arguments
        #   $hBox       - The horizontal packing box in which the buttons live (not yet stored as
        #                   an IV)
        #   $tooltips   - A Gtk2::Tooltips object for the buttons (not yet stored as an IV)
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise, a list containing the Gtk::Button object created

        my ($self, $hBox, $tooltips, $check) = @_;

        # Local variables
        my @emptyList;

        # Check for improper arguments
        if (! defined $hBox || ! defined $tooltips || defined $check) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->enableButtons', @_);
            return @emptyList;
        }

        return $self->enableSingleButton($hBox, $tooltips);
    }

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

    sub setupNotebook {

        # Called by $self->winEnable
        # Creates the first tab for the notebook. The remaining tabs are created by
        #   $self->expandNotebook
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setupNotebook', @_);
        }

        # Tab setup, using the standard table size
        my ($vBox, $table) = $self->addTab('_Line', $self->notebook);

        # Set up the rest of the tab
        $self->lineTab($table);

        # Set up the remaining tabs
        $self->expandNotebook();

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        $self->tagsTab();
        $self->partsTab();
        $self->mxpTab();

        return 1;
    }

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    # Notebook tabs

    sub lineTab {

        # Line tab - called by $self->setupNotebook
        #
        # Expected arguments
        #   $table  - The Gtk2::Table
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $table, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->lineTab', @_);
        }

#       # Tab setup (already created by the calling function)
#       my ($vBox, $table) = $self->addTab('_Line', $self->notebook);

        # Left column
        $self->addLabel($table, '<b>Display buffer line</b>',
            0, 6, 0, 1);
        $self->addLabel($table, 'Session',
            1, 3, 1, 2);
        my $entry = $self->addEntry($table, undef, FALSE,
            3, 6, 1, 2, 8, 8);
        $entry->set_text($self->editObj->session->number);

        $self->addLabel($table, 'Buffer location',
            1, 3, 2, 3);
        my $entry2 = $self->addEntry($table, undef, FALSE,
            3, 6, 2, 3, 8, 8);
        if ($self->editObj->parent eq 'session') {
            $entry2->set_text('Session');
        } else {
            $entry2->set_text('Combined');
        }
        $self->addLabel($table, 'Line number',
            1, 3, 3, 4);
        $self->addEntry($table, 'number', FALSE,
            3, 6, 3, 4, 8, 8);
        $self->addLabel($table, 'Ends with newline character',
            1, 5, 4, 5);
        my $checkButton = $self->addCheckButton($table, 'newLineFlag', FALSE,
            5, 6, 4, 5, 1, 0.5);

        # Right column
        $self->addLabel($table, 'Time received',
            7, 9, 1, 2);
        $self->addEntry($table, 'time', FALSE,
            9, 12, 1, 2, 8, 8);
        $self->addLabel($table, 'No. of packets',
            7, 9, 2, 3);
        my $entry3 = $self->addEntry($table, undef, FALSE,
            9, 12, 2, 3);
        $entry3->set_text($self->editObj->ivNumber('partList'));
        $self->addLabel($table, 'Empty line',
            7, 11, 4, 5);
        my $checkButton2 = $self->addCheckButton($table, 'emptyFlag', FALSE,
            11, 12, 4, 5, 1, 0.5);

        # Bottom section
        # (The entry boxes are sensitive, but not editable, so the user can scroll along to see text
        #   that's too big to fit in the window)
        $self->addLabel($table, '<u>Text received</u>',
            1, 12, 5, 6);
        $self->addLabel($table, 'Original line (received from the world)',
            1, 12, 6, 7);
        my $entry4 = $self->addEntry($table, undef, TRUE,
            1, 12, 7, 8);
        $entry4->set_editable(FALSE);
        $entry4->set_text($self->editObj->line);

        $self->addLabel($table, 'Stripped line (escape sequences removed)',
            1, 12, 8, 9);
        my $entry5 = $self->addEntry($table, undef, TRUE,
            1, 12, 9, 10);
        $entry5->set_editable(FALSE);
        $entry5->set_text($self->editObj->stripLine);

        $self->addLabel($table, 'Modified line (after being modified by any matching interfaces)',
            1, 12, 10, 11);
        my $entry6 = $self->addEntry($table, 'modLine', TRUE,
            1, 12, 11, 12);
        $entry6->set_editable(FALSE);
        $entry6->set_text($self->editObj->modLine);

        my $button = $self->addButton(
            $table,
            'Update',
            'Update the stored data',
            undef,
            10, 12, 12, 13,
        );
        $button->signal_connect('clicked' => sub {

            # Update the widgets
            $entry3->set_text($self->editObj->ivNumber('partList'));
            $entry4->set_text($self->editObj->line);
            $entry5->set_text($self->editObj->stripLine);
            $entry6->set_text($self->editObj->modLine);

            $checkButton->set_active($self->editObj->newLineFlag);
            $checkButton2->set_active($self->editObj->emptyFlag);

            # If the received line of text doesn't end with a newline character, the next batch of
            #   text received will update this object's IVs. Otherwise, no further updates are
            #   possible, and the button should be desensitised
            if ($self->editObj->newLineFlag) {

                $button->set_sensitive(FALSE);
            }
        });

        # The button starts desensitised if a newline character has already been received
        if ($self->editObj->newLineFlag) {

            $button->set_sensitive(FALSE);
        }

        return 1;
    }

    sub tagsTab {

        # Tags tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (@columnList, @columnList2);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->tagsTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('_Tags', $self->notebook);

        # Axmud colour/style tags
        $self->addLabel($table, '<b>' . $axmud::SCRIPT . ' colour/style tags</b>',
            0, 12, 0, 1);
        $self->addLabel(
            $table,
            '<i>List of tags corresponding to escape sequences in the modified line</i>',
            1, 6, 1, 2);

        # Add a simple list
        @columnList = (
            'Tag', 'text',
            'Offsets (positions) at which the tag occurs', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 6, 2, 10);

        # Add an entry
        $self->addLabel($table, 'Length of line (text only; not including escape sequences)',
            1, 5, 10, 11);
        my $entry = $self->addEntry($table, undef, FALSE,
            5, 6, 10, 11,
            8, 8);

        # Add textviews
        $self->addLabel(
            $table,
            '<i>List of tags applied at end of previous line</i>',
            7, 12, 1, 2);
        my $textView = $self->addTextView($table, undef, FALSE,
            7, 12, 2, 4,
            TRUE, TRUE, TRUE, FALSE,
            -1, 80);       # Fixed height

        $self->addLabel(
            $table,
            '<i>List of tags applied at beginning of this line</i>',
            7, 12, 4, 5);
        my $textView2 = $self->addTextView($table, undef, FALSE,
            7, 12, 5, 7,
            TRUE, TRUE, TRUE, FALSE,
            -1, 80);       # Fixed height

        $self->addLabel(
            $table,
            '<i>List of tags applied at end of this line</i>',
            7, 12, 7, 8);
        my $textView3 = $self->addTextView($table, undef, FALSE,
            7, 12, 8, 10,
            TRUE, TRUE, TRUE, FALSE,
            -1, 80);       # Fixed height

        # Initialise the simple list and textview
        $self->tagsTab_refreshList(
            $slWidget,
            (scalar @columnList / 2),
            $textView,
            $textView2,
            $textView3,
            $entry,
        );

        # Add a button
        my $button = $self->addButton(
            $table,
            'Refresh lists',
            'Refresh the lists of colour/style tags',
            undef,
            10, 12, 10, 11,
        );
        $button->signal_connect('clicked' => sub {

            # Refresh the simple list
            $self->tagsTab_refreshList(
                $slWidget,
                (scalar @columnList / 2),
                $textView,
                $textView2,
                $textView3,
                $entry,
            );

            # If the received line of text doesn't end with a newline character, the next batch of
            #   text received will update this object's IVs. Otherwise, no further updates are
            #   possible, and the button should be desensitised
            if ($self->editObj->newLineFlag) {

                $button->set_sensitive(FALSE);
            }
        });

        # The button starts desensitised if a newline character has already been received
        if ($self->editObj->newLineFlag) {

            $button->set_sensitive(FALSE);
        }

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub tagsTab_refreshList {

        # Called by $self->tagsTab to refresh the GA::Obj::Simple::List and Gtk2::TextView
        #
        # Expected arguments
        #   $slWidget   - The GA::Obj::Simple::List
        #   $columns    - The number of columns in the list
        #   $textView, $textView2, $textView3
        #               - The Gtk2::TextViews used
        #   $entry      - The Gtk2::Entry used
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $slWidget, $columns, $textView, $textView2, $textView3, $entry, $check) = @_;

        # Local variables
        my (
            @tagList, @dataList,
            %tagHash,
        );

        # Check for improper arguments
        if (
            ! defined $slWidget || ! defined $columns || ! defined $textView || ! defined $textView2
            || ! defined $textView3 || ! defined $entry || defined $check
        ) {
            return $axmud::CLIENT->writeImproper($self->_objClass . '->tagsTab_refreshList', @_);
        }

        # Import this line's hash of tags (for speed)
        %tagHash = $self->editObj->tagHash;

        # Sort them, so they appear in order of offset
        @tagList = sort {

            my $listRef = $tagHash{$a};
            my $listRef2 = $tagHash{$b};

            if ($$listRef[0] <= $$listRef2[0]) {
                -1;
            } elsif ($$listRef[0] >= $$listRef2[0]) {
                1;
            } else {
                $a cmp $b;
            }

        } (keys %tagHash);

        # Compile the simple list data
        foreach my $tag (@tagList) {

            my $listRef = $tagHash{$tag};
            push (@dataList, $tag, join(' ', @$listRef));
        }

        # Reset the simple list
        $self->resetListData($slWidget, [@dataList], $columns);

        # Reset the textviews
        $textView->get_buffer->set_text(join("\n", $self->editObj->previousTagList));
        $textView2->get_buffer->set_text(join("\n", $self->editObj->initialTagList));
        $textView3->get_buffer->set_text(join("\n", $self->editObj->finalTagList));
        # Reset the entry box
        $entry->set_text(length($self->editObj->stripLine));

        return 1;
    }

    sub partsTab {

        # Parts tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->partsTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('_Parts', $self->notebook);

        # Line parts
        $self->addLabel($table, '<b>Line parts</b>',
            0, 12, 0, 2);
        $self->addLabel(
            $table,
            '<i>List of line parts, each originating in a single data packet received from the'
            . ' world</i>',
            1, 12, 2, 4);

        my $textView = $self->addTextView($table, 'partList', FALSE,
            1, 12, 4, 10,
            TRUE, TRUE, TRUE, FALSE,
            -1, 300);       # Fixed height
        my $button = $self->addButton(
            $table,
            'Refresh list',
            'Refresh the list of colour/style tags',
            undef,
            10, 12, 10, 12,
        );
        $button->signal_connect('clicked' => sub {

            $textView->buffer->set_text(join("\n", $self->editObj->partList));
        });

        # The button starts desensitised if a newline character has already been received
        if ($self->editObj->newLineFlag) {

            $button->set_sensitive(FALSE);
        }

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub mxpTab {

        # MXP tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->mxpTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('_MXP', $self->notebook);

        # MXP tag properties
        $self->addLabel($table, '<b>MXP tag properties</b>',
            0, 12, 0, 1);
        $self->addLabel(
            $table,
            '<i>List of tag properties (defined by custom elements) which terminate on this'
            . 'line</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Tag property', 'text',
            'Stored text', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 300);      # Fixed height

        # Initialise the list
        $self->mxpTab_refreshList($slWidget, (scalar @columnList / 2));

        # Add a button
        my $button = $self->addButton(
            $table,
            'Refresh list',
            'Refresh the list of MXP tag properties',
            undef,
            10, 12, 10, 11,
        );
        $button->signal_connect('clicked' => sub {

            $self->mxpTab_refreshList($slWidget, (scalar @columnList / 2));
        });

        # The button starts desensitised if a newline character has already been received
        if ($self->editObj->newLineFlag) {

            $button->set_sensitive(FALSE);
        }

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub mxpTab_refreshList {

        # Called by $self->mxpTab to refresh the GA::Obj::Simple::List
        #
        # Expected arguments
        #   $slWidget   - The GA::Obj::Simple::List
        #   $columns    - Number of columns in the simple list
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $slWidget, $columns, $check) = @_;

        # Local variables
        my (
            @standardList, @dataList,
            %hash,
        );

        # Check for improper arguments
        if (! defined $slWidget || ! defined $columns || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->mxpTab_refreshList', @_);
        }

        # Compile the simple list data. Display standard tag properties first, then any remaining
        #   ones
        @standardList = (
            'RoomName',
            'RoomDesc',
            'RoomExit',
            'RoomNum',
            'Prompt',
            'Set',
        );

        %hash = $self->getEditHash_hashIV('mxpFlagTextHash');

        foreach my $prop (@standardList) {

            if (exists $hash{$prop}) {

                push (@dataList, $prop, $hash{$prop});
            }

            delete ($hash{$prop});
        }

        foreach my $prop (sort {$a cmp $b} (keys %hash)) {

            push (@dataList, $prop, $hash{$prop});
        }

        # Reset the simple list
        $self->resetListData($slWidget, [@dataList], $columns);

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::Buffer::Instruct;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::Generic::EditWin Games::Axmud::Generic::ConfigWin
        Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::Buffer::Instruct')) {
            return undef;
        } else {
            return 1;
        }
    }

    sub enableButtons {

        # Called by $self->drawWidgets
        # We only need a single button so, instead of calling the generic ->enableButtons, call a
        #   method that creates just one button
        #
        # Expected arguments
        #   $hBox       - The horizontal packing box in which the buttons live (not yet stored as
        #                   an IV)
        #   $tooltips   - A Gtk2::Tooltips object for the buttons (not yet stored as an IV)
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise, a list containing the Gtk::Button object created

        my ($self, $hBox, $tooltips, $check) = @_;

        # Local variables
        my @emptyList;

        # Check for improper arguments
        if (! defined $hBox || ! defined $tooltips || defined $check) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->enableButtons', @_);
            return @emptyList;
        }

        return $self->enableSingleButton($hBox, $tooltips);
    }

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

    sub setupNotebook {

        # Called by $self->winEnable
        # Creates the first tab for the notebook. The remaining tabs are created by
        #   $self->expandNotebook
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setupNotebook', @_);
        }

        # Tab setup, using the standard table size
        my ($vBox, $table) = $self->addTab('_Item', $self->notebook);

        # Set up the rest of the tab
        $self->itemTab($table);

#        # Set up the remaining tabs
#        $self->expandNotebook();       # (No more tabs to set up)

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

#   sub expandNotebook {}       # Inherited from GA::Generic::ConfigWin

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    # Notebook tabs

    sub itemTab {

        # Item tab - called by $self->setupNotebook
        #
        # Expected arguments
        #   $table  - The Gtk2::Table
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $table, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->itemTab', @_);
        }

#       # Tab setup (already created by the calling function)
#       my ($vBox, $table) = $self->addTab('_Item', $self->notebook);

        # Left column
        $self->addLabel($table, '<b>Instruction buffer item</b>',
            0, 6, 0, 1);
        $self->addLabel($table, 'Session',
            1, 3, 1, 2);
        my $entry = $self->addEntry($table, undef, FALSE,
            3, 6, 1, 2, 8, 8);
        $entry->set_text($self->editObj->session->number);

        $self->addLabel($table, 'Buffer location',
            1, 3, 2, 3);
        my $entry2 = $self->addEntry($table, undef, FALSE,
            3, 6, 2, 3, 8, 8);
        if ($self->editObj->parent eq 'session') {
            $entry2->set_text('Session');
        } else {
            $entry2->set_text('Combined');
        }
        $self->addLabel($table, 'Item number',
            1, 3, 3, 4);
        $self->addEntry($table, 'number', FALSE,
            3, 6, 3, 4);

        # Right column
        $self->addLabel($table, 'Time processed',
            7, 9, 1, 2);
        $self->addEntry($table, 'time', FALSE,
            9, 12, 1, 2);
        $self->addLabel($table, 'Instruction type',
            7, 9, 2, 3);
        $self->addEntry($table, 'type', FALSE,
            9, 12, 2, 3);

        # Bottom section
        # (The entry boxes are sensitive, but not editable, so the user can scroll along to see text
        #   that's too big to fit in the window)
        $self->addLabel($table, 'Instruction',
            1, 12, 4, 5);
        my $entry3 = $self->addEntry($table, 'instruct', TRUE,
            1, 12, 5, 6);
        $entry3->set_editable(FALSE);

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::Buffer::Cmd;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::Generic::EditWin Games::Axmud::Generic::ConfigWin
        Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::Buffer::Cmd')) {
            return undef;
        } else {
            return 1;
        }
    }

    sub enableButtons {

        # Called by $self->drawWidgets
        # We only need a single button so, instead of calling the generic ->enableButtons, call a
        #   method that creates just one button
        #
        # Expected arguments
        #   $hBox       - The horizontal packing box in which the buttons live (not yet stored as
        #                   an IV)
        #   $tooltips   - A Gtk2::Tooltips object for the buttons (not yet stored as an IV)
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise, a list containing the Gtk::Button object created

        my ($self, $hBox, $tooltips, $check) = @_;

        # Local variables
        my @emptyList;

        # Check for improper arguments
        if (! defined $hBox || ! defined $tooltips || defined $check) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->enableButtons', @_);
            return @emptyList;
        }

        return $self->enableSingleButton($hBox, $tooltips);
    }

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

    sub setupNotebook {

        # Called by $self->winEnable
        # Creates the first tab for the notebook. The remaining tabs are created by
        #   $self->expandNotebook
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setupNotebook', @_);
        }

        # Tab setup, using the standard table size
        my ($vBox, $table) = $self->addTab('_Item', $self->notebook);

        # Set up the rest of the tab
        $self->itemTab($table);

        # Set up the remaining tabs
        $self->expandNotebook();

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        $self->movesTab();

        return 1;
    }

#   sub expandNotebook {}       # Inherited from GA::Generic::ConfigWin

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    # Notebook tabs

    sub itemTab {

        # Item tab - called by $self->setupNotebook
        #
        # Expected arguments
        #   $table  - The Gtk2::Table
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $table, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->itemTab', @_);
        }

#       # Tab setup (already created by the calling function)
#       my ($vBox, $table) = $self->addTab('_Item', $self->notebook);

        # Left column
        $self->addLabel($table, '<b>Command buffer item</b>',
            0, 6, 0, 1);
        $self->addLabel($table, 'Session',
            1, 3, 1, 2);
        my $entry = $self->addEntry($table, undef, FALSE,
            3, 6, 1, 2, 8, 8);
        $entry->set_text($self->editObj->session->number);

        $self->addLabel($table, 'Buffer location',
            1, 3, 2, 3);
        my $entry2 = $self->addEntry($table, undef, FALSE,
            3, 6, 2, 3, 8, 8);
        if ($self->editObj->parent eq 'session') {
            $entry2->set_text('Session');
        } else {
            $entry2->set_text('Combined');
        }
        $self->addLabel($table, 'Item number',
            1, 3, 3, 4);
        $self->addEntry($table, 'number', FALSE,
            3, 6, 3, 4, 8, 8);

        # Right column
        $self->addLabel($table, 'Time sent',
            7, 9, 1, 2);
        $self->addEntry($table, 'time', FALSE,
            9, 12, 1, 2);

        # Bottom section
        # (The entry boxes are sensitive, but not editable, so the user can scroll along to see text
        #   that's too big to fit in the window)
        $self->addLabel($table, 'Command',
            1, 12, 4, 5);
        my $entry3 = $self->addEntry($table, 'cmd', TRUE,
            1, 12, 5, 6);
        $entry3->set_editable(FALSE);

        return 1;
    }

    sub movesTab {

        # Moves tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->movesTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('_Moves', $self->notebook);

        # Move settings
        $self->addLabel($table, '<b>Move settings</b>',
            0, 12, 0, 1);
        $self->addLabel($table, 'Is a \'look\' command',
            1, 5, 1, 2);
        $self->addCheckButton($table, 'lookFlag', FALSE,
            5, 6, 1, 2, 1, 0.5);
        $self->addLabel($table, 'Is a \'glance\' command',
            1, 5, 2, 3);
        $self->addCheckButton($table, 'glanceFlag', FALSE,
            5, 6, 2, 3, 1, 0.5);
        $self->addLabel($table, 'Is a movement command',
            1, 5, 3, 4);
        $self->addCheckButton($table, 'moveFlag', FALSE,
            5, 6, 3, 4, 1, 0.5);
        $self->addLabel($table, 'Is a redirect mode command',
            1, 5, 5, 6);
        $self->addCheckButton($table, 'redirectFlag', FALSE,
            5, 6, 5, 6, 1, 0.5);
        $self->addLabel($table, 'Is an assisted move',
            1, 5, 6, 7);
        $self->addCheckButton($table, 'assistedFlag', FALSE,
            5, 6, 6, 7, 1, 0.5);
        $self->addLabel($table, 'Is a teleport',
            1, 5, 9, 10);
        $self->addCheckButton($table, 'teleportFlag', FALSE,
            5, 6, 9, 10, 1, 0.5);

        # Right column
        $self->addLabel($table, 'Move direction',
            7, 9, 3, 4);
        $self->addEntry($table, 'moveDir', FALSE,
            9, 12, 3, 4);
        $self->addLabel($table, 'Move verb',
            7, 9, 4, 5);
        $self->addEntry($table, 'moveVerb', FALSE,
            9, 12, 4, 5);
        $self->addLabel($table, 'Substituted command',
            7, 9, 5, 6);
        $self->addEntry($table, 'redirectCmd', FALSE,
            9, 12, 5, 6);
        $self->addLabel($table, 'Standard primary direction',
            7, 9, 6, 7);
        $self->addEntry($table, 'assistedPrimary', FALSE,
            9, 12, 6, 7);

        $self->addLabel($table, 'Assisted command list',
            7, 9, 7, 8);
        my $entry3 = $self->addEntry($table, undef, FALSE,
            9, 12, 7, 8);
        $entry3->set_text(join($axmud::CLIENT->cmdSep, $self->editObj->assistedList));

        $self->addLabel($table, 'Exit number',
            7, 9, 8, 9);
        my $entry4 = $self->addEntry($table, undef, FALSE,
            9, 12, 8, 9);
        if ($self->editObj->assistedExitObj) {

            $entry4->set_text($self->editObj->assistedExitObj->number);
        }

        $self->addLabel($table, 'Destination room (if known)',
            7, 9, 9, 10);
        $self->addEntry($table, 'teleportDestRoom', FALSE,
            9, 12, 9, 10);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::Generic::Cage;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::Generic::EditWin Games::Axmud::Generic::ConfigWin
        Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

#   sub checkEditObj {}         # Inherited from GA::Generic::ConfigWin

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

#   sub expandNotebook {}       # Inherited from GA::Generic::ConfigWin

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    sub compileCages {

        # Compiles a list of cages, starting with this one, continuing with its inferior cage, and
        #   with that cage's inferior cage (and so on)
        # Returns the list, with this cage in first position, and the least superior cage in last
        #   position
        # If this cage has no inferior, the list will contain only this cage
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise, the list described above

        my ($self, $check) = @_;

        # Local variables
        my (
            $lowerCage, $lowestCage,
            @cageList,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->compileCages', @_);
        }

        # Check this cage's inferior, and that cage's inferior, and so on until the least superior
        #   cage is found
        $lowerCage = $self->editObj;

        do {

            my $thisCage = $self->session->ivShow('inferiorCageHash', $lowerCage->name);

            if (defined $thisCage) {

                push (@cageList, $lowerCage);
                $lowerCage = $thisCage;

            } else {

                push (@cageList, $lowerCage);
                $lowestCage = $lowerCage;
            }

        } until ($lowestCage);

        return @cageList;
    }

    # Notebook tabs used by more than one cage 'edit' window

    sub nameTab {

        # Name tab - called by $self->setupNotebook
        #
        # Expected arguments
        #   $table  - The Gtk2::Table
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $table, $check) = @_;

        # Local variables
        my $obj;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->nameTab', @_);
        }

#       # Tab setup (already created by the calling function)
#       my ($vBox, $table) = $self->addTab('_Name', $self->notebook);

        # Left column
        $self->addLabel($table, '<b>Cage settings</b>',
            0, 12, 0, 1);

        $self->addLabel($table, '<i>Cage</i>',
            1, 12, 1, 2);
        $self->addLabel($table, 'Cage name',
            2, 5, 2, 3);
        $self->addEntry($table, 'name', FALSE,
            5, 12, 2, 3, 42, 42);               # Cage's max names are 42 characters

        $self->addLabel($table, 'Cage type',
            2, 5, 3, 4);
        $self->addEntry($table, 'cageType', FALSE,
            5, 7, 3, 4, 8, 8);

        $self->addLabel($table, 'Inferior cage',
            2, 5, 4, 5);
        my $entry = $self->addEntry($table, undef, 0,
            5, 12, 4, 5);
        if ($self->session->ivExists('inferiorCageHash', $self->editObj->name)) {

            $obj = $self->session->ivShow('inferiorCageHash', $self->editObj->name);
            $entry->set_text($obj->name);
        }

        $self->addLabel($table, '<i>Associated profile</i>',
            1, 12, 5, 6);

        $self->addLabel($table, 'Profile name',
            2, 5, 6, 7);
        $self->addEntry($table, 'profName', FALSE,
            5, 7, 6, 7, 16, 16);

        # Right column
        $self->addLabel($table, 'Standard cage?',
            8, 10, 3, 4);
        $self->addCheckButton($table, 'standardFlag', FALSE,
            10, 12, 3, 4);

        $self->addLabel($table, 'Profile category',
            8, 10, 6, 7);
        $self->addEntry($table, 'profCategory', FALSE,
            10, 12, 6, 7);

#       # Tab complete (handled by the calling function)
#       $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::Cage::Cmd;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::EditWin::Generic::Cage Games::Axmud::Generic::EditWin
        Games::Axmud::Generic::ConfigWin Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win
        Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::Cage::Cmd')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook, depending on which type of cage is being edited
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        $self->commandsTab();
        $self->wordsTab();

        return 1;
    }

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    # Notebook tabs

    sub commandsTab {

        # Commands tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (@columnList, @cageList);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->commandsTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('_Commands', $self->notebook);

        # Interpolated command strings
        $self->addLabel($table, '<b>Interpolated command strings</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of command strings which can be interpolated in a different way for each'
            . ' world</i>',
            1, 12, 1, 2);

        # Add a simple list. Column three (onwards) contains all inferior cages. Each column (except
        #   the first two) has a title containing the cage's associated profile
        @cageList = $self->compileCages();
        # Use different text in the column for the first item in @cageList
        shift @cageList;

        @columnList = (
            'Command', 'text',
            'Replacement', 'text',
        );

        foreach my $cage (@cageList) {

            push (@columnList,
                '[' . $cage->profCategory . ': ' . $cage->profName . ']',
                'text',
            );
        }

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 300);     # Fixed height

        # Initialise the list
        $self->refreshList($slWidget, (scalar @columnList / 2), 'cmdHash');

        # Add entry boxes and edit buttons
        $self->addLabel($table, 'For the selected command:',
            1, 4, 10, 12);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            4, 6, 10, 11);

        $self->addCageButtons($table, $slWidget, $entry, 'cmdHash', (scalar @columnList / 2));

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub wordsTab {

        # Words tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (@columnList, @cageList);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->wordsTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('_Words', $self->notebook);

        # Interpolated word strings
        $self->addLabel($table, '<b>Interpolated word strings</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of word strings which are automatically interpolated for each world</i>',
            1, 12, 1, 2);

        # Add a simple list. Column three (onwards) contains all inferior cages. Each column (except
        #   the first two) has a title containing the cage's associated profile
        @cageList = $self->compileCages();
        # Use different text in the column for the first item in @cageList
        shift @cageList;

        @columnList = (
            'Word', 'text',
            'Replacement', 'text',
        );

        foreach my $cage (@cageList) {

            push (@columnList,
                '[' . $cage->profCategory . ': ' . $cage->profName . ']',
                'text',
            );
        }

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 300);     # Fixed height

        # Initialise the list
        $self->refreshList($slWidget, (scalar @columnList / 2), 'wordHash');

        # Add entry boxes and edit buttons
        $self->addLabel($table, 'For the selected word:',
            1, 4, 10, 12);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            4, 6, 10, 11);

        $self->addCageButtons($table, $slWidget, $entry, 'wordHash', (scalar @columnList / 2));

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub addCageButtons {

        # Called by $self->commandsTab and $self->wordsTab to create the editing buttons
        #   beneath the GA::Obj::Simple::List
        #
        # Expected arguments
        #   $table      - The current Gtk2::Table displayed in the notebook
        #   $slWidget   - The GA::Obj::Simple::List
        #   $entry      - A Gtk2::Entry needed by one of the buttons
        #   $iv         - The IV being edited in this tab - 'cmdHash' or 'wordHash'
        #   $columns    - The number of columns
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $table, $slWidget, $entry, $iv, $columns, $check) = @_;

        # Check for improper arguments
        if (
            ! defined $table || ! defined $slWidget || ! defined $entry || ! defined $iv
            || ! defined $columns || defined $check
        ) {
            return $axmud::CLIENT->writeImproper($self->_objClass . '->addCageButtons', @_);
        }

        # Add entry boxes and edit buttons
        my $button = $self->addButton(
            $table,
            'Set new value',
            'Set the current value of the selected command',
            undef,
            6, 8, 10, 11,
        );
        $button->signal_connect('clicked' => sub {

            my ($cmd, $value);

            ($cmd) = $self->getSimpleListData($slWidget, 0);
            if (defined $cmd && $self->checkEntryIcon($entry)) {

                # Set the new value of the selected command
                $value = $entry->get_text();
                # Save the key-value pair in the hash IV
                $self->modifyEditHash_hashIV($iv, $cmd, $value);

                # Refresh the simple list and reset entry boxes
                $self->refreshList($slWidget, $columns, $iv);
                $self->resetEntryBoxes($entry);
            }
        });

        my $button2 = $self->addButton(
            $table,
            'Don\'t use',
            'Don\'t use this command string (set its value to \'undef\'',
            undef,
            8, 10, 10, 11,
        );
        $button2->signal_connect('clicked' => sub {

            my ($cmd) = $self->getSimpleListData($slWidget, 0);
            if (defined $cmd) {

                # Save the key-value pair in the hash IV
                $self->modifyEditHash_hashIV($iv, $cmd, undef);

                # Refresh the simple list and reset entry boxes
                $self->refreshList($slWidget, $columns, $iv);
                $self->resetEntryBoxes($entry);
            }
        });

        my $button3 = $self->addButton(
            $table,
            'Clear all',
            'Clear all command strings from this cage',
            undef,
            10, 12, 10, 11,
        );
        $button3->signal_connect('clicked' => sub {

            my %ivHash;

            foreach my $cmd ($self->editObj->ivKeys($iv)) {

                $ivHash{$cmd} = undef;
            }

            $self->ivAdd('editHash', $iv, \%ivHash);

            # Refresh the simple list and reset entry boxes
            $self->refreshList($slWidget, $columns, $iv);
            $self->resetEntryBoxes($entry);
        });

        return 1;
    }

    sub refreshList {

        # Called by $self->commandsTab and $self->wordsTab to reset the
        #   GA::Obj::Simple::List
        #
        # Expected arguments
        #   $slWidget   - The GA::Obj::Simple::List
        #   $columns    - The number of columns in the list
        #   $iv         - The IV being edited in this tab - 'cmdHash' or 'wordHash'
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $slWidget, $columns, $iv, $check) = @_;

        # Local variables
        my (
            @sortedList, @cageList, @dataList,
            %ivHash,
        );

        # Check for improper arguments
        if (! defined $slWidget || ! defined $columns || ! defined $iv || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->refreshList', @_);
        }

        # Get a sorted list of commands from the cage
        @sortedList = sort {lc($a) cmp lc($b)} ($self->editObj->ivKeys($iv));
        # Get a list of columns in the simple list, which depends on how many cages there are
        @cageList = $self->compileCages();
        # (We already know the first cage)
        shift @cageList;

        # Import the IV
        %ivHash = $self->getEditHash_hashIV($iv);

        # Compile the simple list data
        OUTER: foreach my $cmd (@sortedList) {

            # First two columns, showing the command and its interpolated form...
            push (@dataList, $cmd, $ivHash{$cmd});

            # Remaining columns
            INNER: foreach my $cage (@cageList) {

                if ($cage->ivExists($iv, $cmd)) {

                    # By supplying only two arguments, we instruct ->ivShow to give us the value
                    #   stored in the cage, and not to consult its inferiors (if no value is stored
                    #   there)
                    push (@dataList, $cage->ivShow($iv, $cmd));

                } else {

                    # $cage doesn't store an interpolated form of the command
                    push (@dataList, undef);
                }
            }
        }

        # Reset the simple list
        $self->resetListData($slWidget, [@dataList], $columns);

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::Generic::InterfaceCage;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::EditWin::Generic::Cage Games::Axmud::Generic::EditWin
        Games::Axmud::Generic::ConfigWin Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win
        Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

#   sub checkEditObj {}         # Inherited from GA::Generic::ConfigWin

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

    sub setupNotebook {

        # Called by $self->winEnable
        # Creates the first tab for the notebook. The remaining tabs are created by
        #   $self->expandNotebook
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setupNotebook', @_);
        }

        # Tab setup, using the standard table size
        my ($vBox, $table) = $self->addTab('_Name', $self->notebook);

        # Interface cage 'edit' windows have some non-standard IVs, used to allow one tab to affect
        #   the contents of others and to allow the same code to be inherited efficiently by
        #   different types of interface cage 'edit' window
        # The values for each IV are set in the inheriting object's ->interfacesTab function
        $self->{slWidget1} = undef;
        $self->{slWidget2} = undef;
        $self->{singular} = undef;
        $self->{plural} = undef;
        $self->{interfaceModelObj} = undef;
        $self->{stimulus} = undef;
        $self->{response} = undef;

        # Set up the rest of the tab
        $self->nameTab($table);

        # Set up the remaining tabs
        $self->expandNotebook();

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook, depending on which type of cage is being edited
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        $self->interfacesTab();

        return 1;
    }

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    # Notebook tabs used by more than one cage 'edit' window

    sub interfaces1Tab {

        # Interfaces1 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->interfaces1Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _1', $innerNotebook);

        # Interace list
        $self->addLabel($table, '<b>' . ucfirst($self->singular) . ' list</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of ' . $self->plural . ' belonging to this cage (showing '
            . $self->{singular} . ' attributes)</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            ucfirst($self->singular) . ' name', 'text',
            'Enabled', 'bool',
            'Stimulus', 'text',
            'Response', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 170);      # Fixed height

        # Unusual step - save the list reference in an IV
        $self->ivPoke('slWidget2', $slWidget);

        # Initialise the list
        $self->interfaces1Tab_refreshList();

        # Add some buttons and entry boxes
        $self->interfacesTab_addButtons($table, $slWidget, (scalar @columnList / 2));

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub interfaces1Tab_refreshList {

        # Called by $self->interfaces1Tab (etc) to reset the GA::Obj::Simple::List
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (@sortedList, @dataList);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->interfaces1Tab_refreshList',
                @_,
            );
        }

        # Get a sorted list of interface objects from the cage
        @sortedList
            = sort {lc($a->name) cmp lc($b->name)} ($self->editObj->ivValues('interfaceHash'));

        # Compile the simple list data
        foreach my $interfaceObj (@sortedList) {

            push (@dataList,
                $interfaceObj->name,
                $interfaceObj->enabledFlag,
                $interfaceObj->stimulus,
                $interfaceObj->response,
            );
        }

        # Reset the simple list (page 2 has four columns)
        $self->resetListData($self->slWidget2, [@dataList], 4);

        return 1;
    }

    sub interfaces2Tab {

        # Interfaces2 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (@columnList, @cageList);

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->interfaces2Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _2', $innerNotebook);

        # Interface list
        $self->addLabel($table, '<b>' . ucfirst($self->singular) . ' list</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of ' . $self->plural . ' belonging to this cage (compared with lower-priority'
            . ' cages)</i>',
            1, 12, 1, 2);

        # Add a simple list. The number of columns is equal to the number of cages between this one
        #   and the least superior cage. (If this cage has no inferior, there's only one column)
        # Each column (except the first one) has a title containing the cage's associated profile
        @cageList = $self->compileCages();

        # Use different text in the column for the first item in @cageList
        @columnList = (ucfirst($self->singular) . ' name', 'text');
        shift @cageList;

        foreach my $cage (@cageList) {

            push (@columnList,
                '[' . $cage->profCategory . ': ' . $cage->profName . ']',
                'text',
            );
        }

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 170);      # Fixed height

        # Unusual step - save the list reference in an IV
        $self->ivPoke('slWidget1', $slWidget);

        # Initialise the list
        $self->interfaces2Tab_refreshList();

        # Add some buttons and entry boxes
        $self->interfacesTab_addButtons($table, $slWidget, (scalar @columnList / 2));

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub interfaces2Tab_refreshList {

        # Called by $self->interfaces2Tab (etc) to reset the GA::Obj::Simple::List
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (@sortedList, @cageList, @dataList);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->interfaces2Tab_refreshList',
                @_,
            );
        }

        # Get a sorted list of interface names from the cage
        @sortedList = sort {lc($a) cmp lc($b)} ($self->editObj->ivKeys('interfaceHash'));
        # Get a list of columns in the simple list, which depends on how many cages there are
        @cageList = $self->compileCages();

        # Compile the simple list data
        OUTER: foreach my $interfaceName (@sortedList) {

            INNER: foreach my $cage (@cageList) {

                my $interfaceObj;

                if ($cage->ivExists('interfaceHash', $interfaceName)) {

                    # By supplying only two arguments, we instruct ->ivShow to give us the value
                    #   stored in the cage, and not to consult its inferiors (if no value is stored
                    #   there)
                    $interfaceObj = $cage->ivShow('interfaceHash', $interfaceName);
                    push (@dataList, $interfaceObj->name);

                } else {

                    # An interface called $interfaceName isn't stored in $cage
                    push (@dataList, undef);
                }
            }
        }

        # Reset the simple list
        $self->resetListData($self->slWidget1, [@dataList], scalar @cageList);

        return 1;
    }

    sub interfacesTab_addButtons {

        # Called by $self->interfaces1Tab and $self->interfaces2Tab to create the editing buttons
        #   beneath the GA::Obj::Simple::List
        #
        # Expected arguments
        #   $table      - The current Gtk2::Table displayed in the notebook
        #   $slWidget   - The GA::Obj::Simple::List
        #   $columns    - The number of columns
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $table, $slWidget, $columns, $check) = @_;

        # Local variables
        my @comboList;

        # Check for improper arguments
        if (! defined $table || ! defined $slWidget || ! defined $columns || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->interfacesTab_addButtons',
                @_,
            );
        }

        # Add entry boxes and edit buttons
        $self->addLabel(
            $table, ucfirst($self->singular) . ' stimulus <i>(' . $self->stimulus . ')</i>',
            1, 3, 8, 9);
        # For hooks, use a combo; for everything else, use an entry. For aliases, leave room for an
        #   extra button beside the entry
        my ($entry, $combo);
        if ($self->editObj->cageType eq 'trigger') {

            $entry = $self->addEntryWithIcon($table, undef, 'regex', 1, undef,
                3, 12, 8, 9);

        } elsif ($self->editObj->cageType eq 'alias') {

            $entry = $self->addEntryWithIcon($table, undef, 'regex', 1, undef,
                3, 10, 8, 9);

        } elsif ($self->editObj->cageType eq 'macro') {

            $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
                3, 12, 8, 9);

        } elsif ($self->editObj->cageType eq 'timer') {

            $entry = $self->addEntryWithIcon(
                $table, undef, \&interfacesTab_checkInterval, undef, undef,
                3, 12, 8, 9);

        } elsif ($self->editObj->cageType eq 'hook') {

            @comboList = sort {$a cmp $b} ($self->interfaceModelObj->ivKeys('hookEventHash'));
            $combo = $self->addComboBox($table, undef, \@comboList, '',
                TRUE,               # No 'undef' value used
                3, 6, 8, 9);
        }

        my $label = $self->addLabel(
            $table, ucfirst($self->singular) . ' response <i>(' . $self->response . ')</i>',
            1, 3, 9, 10);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            3, 12, 9, 10);

        $self->addLabel($table, 'Name <i>(optional)</i>',
            1, 3, 10, 11);

        my ($entry3, $checkButton, $checkButton2, $checkButton3, $checkButton4);
        if ($self->singular eq 'trigger') {

            $entry3 = $self->addEntryWithIcon($table, undef, 'string', undef, undef,
                3, 4, 10, 11);

            $self->addLabel($table, 'Starts enabled',
                4, 5, 10, 11);
            $checkButton = $self->addCheckButton($table, undef, TRUE,
                5, 6, 10, 11, 1, 0.5);
            $self->addLabel($table, 'Splitter',
                6, 7, 10, 11);
            $checkButton2 = $self->addCheckButton($table, undef, TRUE,
                7, 8, 10, 11, 1, 0.5);
            $self->addLabel($table, 'Rewriter',
                8, 9, 10, 11);
            $checkButton3 = $self->addCheckButton($table, undef, TRUE,
                9, 10, 10, 11, 1, 0.5);
            $self->addLabel($table, 'Temp',
                10, 11, 10, 11);
            $checkButton4 = $self->addCheckButton($table, undef, TRUE,
                11, 12, 10, 11, 1, 0.5);

            $checkButton2->signal_connect('toggled' => sub {

                if ($checkButton2->get_active()) {
                    $self->ivPoke('response', 'pattern');
                } elsif ($checkButton3->get_active()) {
                    $self->ivPoke('response', 'substitution');
                } else {
                    $self->ivPoke('response', 'instruction');
                }

                $label->set_markup(
                    ucfirst($self->singular) . ' response <i>(' . $self->response . ')</i>',
                );
            });

            $checkButton3->signal_connect('toggled' => sub {

                if ($checkButton2->get_active()) {
                    $self->ivPoke('response', 'pattern');
                } elsif ($checkButton3->get_active()) {
                    $self->ivPoke('response', 'substitution');
                } else {
                    $self->ivPoke('response', 'instruction');
                }

                $label->set_markup(
                    ucfirst($self->singular) . ' response <i>(' . $self->response . ')</i>',
                );
            });

        } else {

            $entry3 = $self->addEntryWithIcon($table, undef, 'string', undef, undef,
                3, 6, 10, 11);

            $self->addLabel($table, 'Starts enabled',
                6, 8, 10, 11);
            $checkButton = $self->addCheckButton($table, undef, TRUE,
                8, 9, 10, 11, 1, 0.5);

            $self->addLabel($table, 'Temporary',
                9, 11, 10, 11);
            $checkButton4 = $self->addCheckButton($table, undef, TRUE,
                11, 12, 10, 11, 1, 0.5);
        }
        # New interfaces should be enabled by default
        $checkButton->set_active(TRUE);

        my $button = $self->addButton(
            $table,
            'Add',
            'Add the ' . $self->singular . ' to this cage',
            undef,
            1, 2, 11, 12,
            TRUE,           # Irreversible
        );
        $button->signal_connect('clicked' => sub {

            my (
                $stimulus, $response, $interfaceName, $enabledFlag, $stimulusSwitch,
                $responseSwitch, $nameSwitch, $enabledSwitch, $splitterFlag, $splitterSwitch,
                $splitterString, $rewriterFlag, $rewriterSwitch, $rewriterString, $tempSwitch,
                $tempString, $result,
            );

            # Get the interface stimulus
            if ($self->editObj->cageType eq 'hook' && $self->checkEntryIcon($entry2, $entry3)) {
                $stimulus = $combo->get_active_text();      # Compulsory
            } elsif ($self->checkEntryIcon($entry, $entry2, $entry3)) {
                $stimulus = $entry->get_text();             # Compulsory
            }

            if (defined $stimulus && $self->editObj->profName) {

                # Get the interface response and name
                $response = $entry2->get_text();            # Compulsory
                $interfaceName = $entry3->get_text();       # Optional
                if ($checkButton->get_active()) {
                    $enabledFlag = 1;
                } else {
                    $enabledFlag = 0;
                }

                $stimulusSwitch
                    = $self->interfaceModelObj->ivShow('compulsorySwitchHash', 'stimulus');
                $responseSwitch
                    = $self->interfaceModelObj->ivShow('compulsorySwitchHash', 'response');
                $nameSwitch
                    = $self->interfaceModelObj->ivShow('optionalSwitchHash', 'name');
                $enabledSwitch
                    = $self->interfaceModelObj->ivShow('optionalSwitchHash', 'enabled');

                if ($self->singular eq 'trigger') {

                    if ($checkButton2->get_active()) {      # Non-standard attribute
                        $splitterFlag = 1;
                    } else {
                        $splitterFlag = 0;
                    }

                    $splitterSwitch
                        = $self->interfaceModelObj->ivShow('optionalSwitchHash', 'splitter');
                    $splitterString = ' -' . $splitterSwitch . ' ' . $splitterFlag;

                    if ($checkButton3->get_active()) {      # Non-standard attribute
                        $rewriterFlag = 1;
                    } else {
                        $rewriterFlag = 0;
                    }

                    $rewriterSwitch
                        = $self->interfaceModelObj->ivShow('optionalSwitchHash', 'rewriter');
                    $rewriterString = ' -' . $rewriterSwitch . ' ' . $rewriterFlag;

                } else {

                    $splitterString = '';
                    $rewriterString = '';
                }

                if ($checkButton4->get_active()) {

                    $tempSwitch
                        = $self->interfaceModelObj->ivShow('optionalSwitchHash', 'temporary');
                    $tempString = ' -' . $tempSwitch . ' 1';

                } else {

                    $tempString = '';
                }

                if ($stimulus =~ m/[\<|\>]/ || $response =~ m/[\<|\>]/) {

                    $self->showMsgDialogue(
                        'Add ' . $self->singular,
                        'error',
                        'This version of ' . $axmud::SCRIPT . ' can\'t process diamond bracket'
                        . ' characters; please use . characters in your regular expressions'
                        . ' instead',
                        'ok',
                    );

                # Try to add a new trigger/alias/macro/timer/hook object belonging to this cage
                } elsif ($interfaceName) {

                    $result = $self->session->pseudoCmd(
                        'add' . $self->singular
                        . ' -' . $stimulusSwitch . ' <' . $stimulus
                        . '> -' . $responseSwitch . ' <' . $response
                        . '> -' . $enabledSwitch . ' ' . $enabledFlag
                        . $splitterString . $rewriterString . $tempString
                        . ' -' . $nameSwitch . ' <' . $interfaceName
                        . '> -d ' . $self->editObj->profName,
                        $self->pseudoCmdMode,
                    );

                } else {

                    $result = $self->session->pseudoCmd(
                        'add' . $self->singular
                        . ' -' . $stimulusSwitch . ' <' . $stimulus
                        . '> -' . $responseSwitch . ' <' . $response
                        . '> -' . $enabledSwitch . ' ' . $enabledFlag
                        . $splitterString . $rewriterString . $tempString
                        . ' -d ' . $self->editObj->profName,
                        $self->pseudoCmdMode,
                    );
                }

                if ($result) {

                    # Refresh (both) lists
                    $self->interfaces1Tab_refreshList();
                    $self->interfaces2Tab_refreshList();
                }
            }
        });

        my $button2 = $self->addButton(
            $table,
            'Edit...',
            'Edit the selected ' . $self->singular,
            undef,
            2, 3, 11, 12,
        );
        $button2->signal_connect('clicked' => sub {

            my ($interfaceName, $interfaceObj, $childWinObj);

            ($interfaceName) = $self->getSimpleListData($slWidget, 0);
            if (defined $interfaceName) {

                # Check that there's an interface with that name stored in this cage
                if ($self->editObj->ivExists('interfaceHash', $interfaceName)) {

                    # Open up an interface 'edit' window to edit the (inactive) interface
                    $interfaceObj = $self->editObj->ivShow('interfaceHash', $interfaceName);

                    $childWinObj = $self->createFreeWin(
                        'Games::Axmud::EditWin::Interface::' . ucfirst($self->singular),
                        $self,
                        $self->session,
                        'Edit ' . $self->singular . ' interface \'' . $interfaceName . '\'',
                        $interfaceObj,
                        FALSE,                          # Not temporary
                    );

                    if ($childWinObj) {

                        # When the 'edit' window closes, update widgets and/or IVs
                        $self->add_childDestroy(
                            $childWinObj,
                            'interfaces1Tab_refreshList',
                            [],         # No arguments required
                        );

                        $self->add_childDestroy(
                            $childWinObj,
                            'interfaces2Tab_refreshList',
                            [],         # No arguments required
                        );
                    }
                }
            }
        });

        my $button3 = $self->addButton(
            $table,
            'Delete',
            'Delete the selected ' . $self->singular,
            undef,
            3, 4, 11, 12,
            TRUE,               # Irreversible
        );
        $button3->signal_connect('clicked' => sub {

            my ($interfaceName, $profName);

            ($interfaceName) = $self->getSimpleListData($slWidget, 0);
            $profName = $self->editObj->profName;

            if (defined $interfaceName && defined $profName) {

                # Check that there's an interface with that name stored in this cage
                if ($self->editObj->ivExists('interfaceHash', $interfaceName)) {

                    # Delete the interface
                    $self->session->pseudoCmd(
                        'delete' . $self->singular . ' ' . $interfaceName . ' -d ' . $profName,
                        $self->pseudoCmdMode,
                    );

                    # Refresh (both) lists
                    $self->interfaces1Tab_refreshList();
                    $self->interfaces2Tab_refreshList();
                }
            }
        });

        my $button4 = $self->addButton(
            $table,
            'Dump',
            'Display this list of ' . $self->plural . ' in the \'main\' window',
            undef,
            6, 9, 11, 12,
        );
        $button4->signal_connect('clicked' => sub {

            my $profName = $self->editObj->profName;
            if ($profName) {

                # Display the data
                $self->session->pseudoCmd(
                    'list' . $self->singular . ' -d ' . $profName,
                    $self->pseudoCmdMode,
                );

                # Refresh (both) lists
                $self->interfaces1Tab_refreshList();
                $self->interfaces2Tab_refreshList();
            }
        });

        my $button5 = $self->addButton(
            $table,
            'Refresh list',
            'Refresh the list of ' . $self->plural,
            undef,
            9, 12, 11, 12,
        );
        $button5->signal_connect('clicked' => sub {

            # Refresh (both) lists
            $self->interfaces1Tab_refreshList();
            $self->interfaces2Tab_refreshList();
        });

        if ($self->editObj->cageType eq 'alias') {

            my $button6 = $self->addButton(
                $table,
                'Recommended: add ^...$',
                'Adds symbols to match the pattern against the whole world command',
                undef,
                10, 12, 8, 9,
            );

            $button6->signal_connect('clicked' => sub {

                my $string = $entry->get_text();

                if ($self->checkEntryIcon($entry)) {

                    $string = $entry->get_text();
                    if (! ($string =~ m/^\^/)) {

                        $string = '^' . $string;
                    }

                    if (! ($string =~ m/\$$/)) {

                        $string .= '$';
                    }

                    $entry->set_text($string);
                }
            });
        }

        return 1;
    }

    sub interfacesTab_checkInterval {

        # Called by $self->interfacesTab_addButtons to check a timer interval is valid
        #
        # Expected arguments
        #   $value      - The value to check, should be a number (minimum value 0.01), or a 24-hour
        #                   clock time in the form HH::MM
        #
        # Return values
        #   'undef' on improper arguments or if $value is invalid
        #   1 if $value is valid

        my ($self, $value, $check) = @_;

        # Check for improper arguments
        if (! defined $value || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->interfacesTab_checkInterval',
                @_,
            );
        }

        if (
            ! $axmud::CLIENT->floatCheck($value, 0.1)
            && ! ($value =~ m/^([01]?[0-9]|2[0-3]):[0-5][0-9]$/)
            && ! ($value =~ m/^99\:[0-5][0-9]$/)
        ) {
            return undef;
        } else {
            return 1;
        }
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get

    sub slWidget1
        { $_[0]->{slWidget1} }
    sub slWidget2
        { $_[0]->{slWidget2} }
    sub singular
        { $_[0]->{singular} }
    sub plural
        { $_[0]->{plural} }
    sub interfaceModelObj
        { $_[0]->{interfaceModelObj} }
    sub stimulus
        { $_[0]->{stimulus} }
    sub response
        { $_[0]->{response} }
}

{ package Games::Axmud::EditWin::Cage::Trigger;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::EditWin::Generic::InterfaceCage Games::Axmud::EditWin::Generic::Cage
        Games::Axmud::Generic::EditWin Games::Axmud::Generic::ConfigWin
        Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::Cage::Trigger')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

#   sub expandNotebook {}       # Inherited from GA::Generic::ConfigWin

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    # Notebook tabs

    sub interfacesTab {

        # Interfaces tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my $interfaceModelObj;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->interfacesTab', @_);
        }

        # Unusual step - we need to remember both simple lists displayed in these tabs, so create a
        #   few extra IVs
        $self->ivPoke('slWidget1', undef);
        $self->ivPoke('slWidget2', undef);

        # We also need to store a few strings for this kind of interface (triggers, in this case)
        $self->ivPoke('singular', 'trigger');
        $self->ivPoke('plural', 'triggers');

        $interfaceModelObj = $axmud::CLIENT->ivShow('interfaceModelHash', 'trigger');
        $self->ivPoke('interfaceModelObj', $interfaceModelObj);
        $self->ivPoke('stimulus', $interfaceModelObj->stimulusName);    # i.e. 'pattern'
        $self->ivPoke('response', $interfaceModelObj->responseName);    # i.e. 'instruction'

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('_Triggers', $self->notebook);

        # Add tabs to the inner notebook
        $self->interfaces1Tab($innerNotebook);
        $self->interfaces2Tab($innerNotebook);

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::Cage::Alias;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::EditWin::Generic::InterfaceCage Games::Axmud::EditWin::Generic::Cage
        Games::Axmud::Generic::EditWin Games::Axmud::Generic::ConfigWin
        Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::Cage::Alias')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

#   sub expandNotebook {}       # Inherited from GA::Generic::ConfigWin

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    # Notebook tabs

    sub interfacesTab {

        # Interfaces tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my $interfaceModelObj;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->interfacesTab', @_);
        }

        # Unusual step - we need to remember both simple lists displayed in these tabs, so create a
        #   few extra IVs
        $self->ivPoke('slWidget1', undef);
        $self->ivPoke('slWidget2', undef);

        # We also need to store a few strings for this kind of interface (aliases, in this case)
        $self->ivPoke('singular', 'alias');
        $self->ivPoke('plural', 'aliases');

        $interfaceModelObj = $axmud::CLIENT->ivShow('interfaceModelHash', 'alias');
        $self->ivPoke('interfaceModelObj', $interfaceModelObj);
        $self->ivPoke('stimulus', $interfaceModelObj->stimulusName);    # i.e. 'pattern'
        $self->ivPoke('response', $interfaceModelObj->responseName);    # i.e. 'substitution'

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('_Aliases', $self->notebook);

        # Add tabs to the inner notebook
        $self->interfaces1Tab($innerNotebook);
        $self->interfaces2Tab($innerNotebook);

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::Cage::Macro;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::EditWin::Generic::InterfaceCage Games::Axmud::EditWin::Generic::Cage
        Games::Axmud::Generic::EditWin Games::Axmud::Generic::ConfigWin
        Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::Cage::Macro')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

#   sub expandNotebook {}       # Inherited from GA::Generic::ConfigWin

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    # Notebook tabs

    sub interfacesTab {

        # Interfaces tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my $interfaceModelObj;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->interfacesTab', @_);
        }

        # Unusual step - we need to remember both simple lists displayed in these tabs, so create a
        #   few extra IVs
        $self->ivPoke('slWidget1', undef);
        $self->ivPoke('slWidget2', undef);

        # We also need to store a few strings for this kind of interface (macros, in this case)
        $self->ivPoke('singular', 'macro');
        $self->ivPoke('plural', 'macros');

        $interfaceModelObj = $axmud::CLIENT->ivShow('interfaceModelHash', 'macro');
        $self->ivPoke('interfaceModelObj', $interfaceModelObj);
        $self->ivPoke('stimulus', $interfaceModelObj->stimulusName);    # i.e. 'keycode'
        $self->ivPoke('response', $interfaceModelObj->responseName);    # i.e. 'instruction'

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('_Macros', $self->notebook);

        # Add tabs to the inner notebook
        $self->interfaces1Tab($innerNotebook);
        $self->interfaces2Tab($innerNotebook);

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::Cage::Timer;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::EditWin::Generic::InterfaceCage Games::Axmud::EditWin::Generic::Cage
        Games::Axmud::Generic::EditWin Games::Axmud::Generic::ConfigWin
        Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::Cage::Timer')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

#   sub expandNotebook {}       # Inherited from GA::Generic::ConfigWin

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    # Notebook tabs

    sub interfacesTab {

        # Interfaces tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my $interfaceModelObj;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->interfacesTab', @_);
        }

        # Unusual step - we need to remember both simple lists displayed in these tabs, so create a
        #   few extra IVs
        $self->ivPoke('slWidget1', undef);
        $self->ivPoke('slWidget2', undef);

        # We also need to store a few strings for this kind of interface (timers, in this case)
        $self->ivPoke('singular', 'timer');
        $self->ivPoke('plural', 'timers');

        $interfaceModelObj = $axmud::CLIENT->ivShow('interfaceModelHash', 'timer');
        $self->ivPoke('interfaceModelObj', $interfaceModelObj);
        $self->ivPoke('stimulus', $interfaceModelObj->stimulusName);    # i.e. 'interval'
        $self->ivPoke('response', $interfaceModelObj->responseName);    # i.e. 'instruction'

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('T_imers', $self->notebook);

        # Add tabs to the inner notebook
        $self->interfaces1Tab($innerNotebook);
        $self->interfaces2Tab($innerNotebook);

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::Cage::Hook;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::EditWin::Generic::InterfaceCage Games::Axmud::EditWin::Generic::Cage
        Games::Axmud::Generic::EditWin Games::Axmud::Generic::ConfigWin
        Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::Cage::Hook')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

#   sub expandNotebook {}       # Inherited from GA::Generic::ConfigWin

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    # Notebook tabs

    sub interfacesTab {

        # Interfaces tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my $interfaceModelObj;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->interfacesTab', @_);
        }

        # Unusual step - we need to remember both simple lists displayed in these tabs, so create a
        #   few extra IVs
        $self->ivPoke('slWidget1', undef);
        $self->ivPoke('slWidget2', undef);

        # We also need to store a few strings for this kind of interface (hooks, in this case)
        $self->ivPoke('singular', 'hook');
        $self->ivPoke('plural', 'hooks');

        $interfaceModelObj = $axmud::CLIENT->ivShow('interfaceModelHash', 'hook');
        $self->ivPoke('interfaceModelObj', $interfaceModelObj);
        $self->ivPoke('stimulus', $interfaceModelObj->stimulusName);    # i.e. 'hook_event'
        $self->ivPoke('response', $interfaceModelObj->responseName);    # i.e. 'instruction'

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('_Hooks', $self->notebook);

        # Add tabs to the inner notebook
        $self->interfaces1Tab($innerNotebook);
        $self->interfaces2Tab($innerNotebook);

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::Cage::Route;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::EditWin::Generic::InterfaceCage Games::Axmud::EditWin::Generic::Cage
        Games::Axmud::Generic::EditWin Games::Axmud::Generic::ConfigWin
        Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::Cage::Route')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

    sub setupNotebook {

        # Called by $self->winEnable
        # Creates the first tab for the notebook. The remaining tabs are created by
        #   $self->expandNotebook
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setupNotebook', @_);
        }

        # Tab setup, using the standard table size
        my ($vBox, $table) = $self->addTab('_Name', $self->notebook);

        # Route cage 'edit' windows have some non-standard IVs, used to allow one tab to affect the
        #   contents of others
        # The values for each IV are set in $self->routes1Tab and ->routes2Tab
        $self->{slWidget1} = undef;
        $self->{slWidget2} = undef;

        # Set up the rest of the tab
        $self->nameTab($table);

        # Set up the remaining tabs
        $self->expandNotebook();

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook, depending on which type of cage is being edited
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        $self->routesTab();

        return 1;
    }

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    # Notebook tabs

    sub routesTab {

        # Routes tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->routesTab', @_);
        }

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('_Routes', $self->notebook);

        # Add tabs to the inner notebook
        $self->routes1Tab($innerNotebook);
        $self->routes2Tab($innerNotebook);

        return 1;
    }

    sub routes1Tab {

        # Routes1 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (@columnList, @cageList, @typeList);

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->routes1Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _1', $innerNotebook);

        # Routes list
        $self->addLabel($table, '<b>Routes list</b>',
            0, 13, 0, 1);
        $self->addLabel($table,
            '<i>List of routes belonging to this cage (showing route attributes)</i>',
            1, 13, 1, 2);

        # Add a simple list
        @columnList = (
            'Stored as', 'text',
            'Type', 'text',
            'Hop', 'bool',
            'Steps', 'int',
            'Start tag', 'text',
            'Stop tag', 'text',
            'Circuit', 'text',
            'Route', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 13, 2, 8,
            -1, 170);      # Fixed height

        # Unusual step - save the list reference in an IV
        $self->ivPoke('slWidget2', $slWidget);

        # Initialise the list
        $self->routes1Tab_refreshList();

        # Add some buttons and entry boxes
        $self->routesTab_addEditButtons($table, $slWidget, (scalar @columnList / 2));

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub routes1Tab_refreshList {

        # Called by $self->routes1Tab to refresh the GA::Obj::Simple::List
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            @sortedList, @dataList,
            %routeHash,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->routes1Tab_refreshList', @_);
        }

        # Import the IV
        %routeHash = $self->editObj->routeHash;

        # Get a sorted list of route objects from the cage
        @sortedList = sort {lc($a) cmp lc($b)} (keys %routeHash);

        # Compile the simple list data
        foreach my $routeKey (@sortedList) {

            my $routeObj = $routeHash{$routeKey};

            push (@dataList,
                $routeKey,
                $routeObj->routeType,
                $routeObj->hopFlag,
                $routeObj->stepCount,
                $routeObj->startRoom,
                $routeObj->stopRoom,
                $routeObj->circuitName,
                $routeObj->route,
            );
        }

        # Reset the simple list (page 2 has four columns)
        $self->resetListData($self->{slWidget2}, [@dataList], 8);

        return 1;
    }

    sub routes2Tab {

        # Routes2 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (@columnList, @cageList, @typeList);

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->routes2Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _2', $innerNotebook);

        # Routes list
        $self->addLabel($table, '<b>Routes list</b>',
            0, 13, 0, 1);
        $self->addLabel($table,
            '<i>List of routes belonging to this cage (compared with lower-priority cages)</i>',
            1, 13, 1, 2);

        # Add a simple list. The number of columns is equal to the number of cages between this one
        #   and the least superior cage. (If this cage has no inferior, there's only one column)
        # Each column (except the first one) has a title containing the cage's associated profile
        @cageList = $self->compileCages();

        # Use different text in the column for the first item in @cageList
        @columnList = ('Stored as', 'text');
        shift @cageList;

        foreach my $cage (@cageList) {

            push (@columnList,
                '[' . $cage->profCategory . ': ' . $cage->profName . ']',
                'text',
            );
        }

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 13, 2, 8,
            -1, 170);      # Fixed height

        # Unusual step - save the list reference in an IV
        $self->ivPoke('slWidget1', $slWidget);

        # Initialise the list
        $self->routes2Tab_refreshList();

        # Add some buttons and entry boxes
        $self->routesTab_addEditButtons($table, $slWidget, (scalar @columnList / 2));

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub routes2Tab_refreshList {

        # Called by $self->routes2Tab to refresh the GA::Obj::Simple::List
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (@sortedList, @cageList, @dataList);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->routes2Tab_refreshList', @_);
        }

        # Get a sorted list of route keys from the cage
        @sortedList = sort {lc($a) cmp lc($b)} ($self->editObj->ivKeys('routeHash'));
        # Get a list of columns in the simple list, which depends on how many cages there are
        @cageList = $self->compileCages();

        # Compile the simple list data
        OUTER: foreach my $routeKey (@sortedList) {

            INNER: foreach my $cageObj (@cageList) {

                if ($cageObj->ivExists('routeHash', $routeKey)) {

                    # Don't display the route object's blessed reference - just display the route in
                    #   the form used by keys in the cage's ->routeHash
                    push (@dataList, $routeKey);

                } else {

                    # An route called $routeKey isn't stored in $cageObj
                    push (@dataList, undef);
                }
            }
        }

        # Reset the simple list
        $self->resetListData($self->slWidget1, [@dataList], scalar @cageList);

        return 1;
    }

    sub routesTab_addEditButtons {

        # Called by $self->routes1Tab and $self->routes2Tab to create the editing buttons beneath
        #   the GA::Obj::Simple::List
        #
        # Expected arguments
        #   $table      - The current Gtk2::Table displayed in the notebook
        #   $slWidget   - The GA::Obj::Simple::List
        #   $columns    - The number of columns
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $table, $slWidget, $columns, $check) = @_;

        # Local variables
        my @typeList;

        # Check for improper arguments
        if (! defined $table || ! defined $slWidget || ! defined $columns || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->routesTab_addEditButtons',
                @_,
            );
        }

        # Add comboboxes, check buttons and entries
        $self->addLabel($table, 'Route type',
            1, 4, 8, 9);
        @typeList = ('road', 'quick', 'circuit');
        my $combo = $self->addComboBox($table, undef, \@typeList, '',
            TRUE,               # No 'undef' value used
            4, 5, 8, 9);

        $self->addLabel($table, 'Hoppable',
            5, 6, 8, 9);
        my $checkButton = $self->addCheckButton($table, undef, TRUE,
            6, 7, 8, 9, 1, 0.5);
        # New routes should be hoppable by default
        $checkButton->set_active(TRUE);

        $self->addLabel($table, 'Start room tag',
            7, 10, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, 16,
            10, 13, 8, 9);

        $self->addLabel($table, 'Stop room tag',
            1, 4, 9, 10);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'string', 1, 16,
            4, 7, 9, 10);

        $self->addLabel($table, 'Circuit name',
            7, 10, 9, 10);
        my $entry3 = $self->addEntryWithIcon($table, undef, 'string', 1, 16,
            10, 13, 9, 10);

        $self->addLabel($table, 'Route',
            1, 4, 10, 11);
        my $entry4 = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            4, 13, 10, 11);

        # Add editing buttons
        my $button = $self->addButton(
            $table,
            'Add route',
            'Add a new pre-defined route',
            undef,
            3, 5, 11, 12,
            TRUE,           # Irreversible
        );
        $button->signal_connect('clicked' => sub {

            my ($routeType, $hopFlag, $startTag, $stopTag, $circuitName, $route, $cmd);

            $routeType = $combo->get_active_text();
            $hopFlag = $checkButton->get_active();
            $startTag = $entry->get_text();
            $stopTag = $entry2->get_text();
            $circuitName = $entry3->get_text();
            $route = $entry4->get_text();

            # Prepare the client command
            if ($routeType eq 'road') {

                if ($self->checkEntryIcon($entry, $entry2, $entry4)) {

                    $cmd = 'addroute ' . $startTag . ' ' . $stopTag . ' <' . $route . '> -o';
                }

            } elsif ($routeType eq 'quick') {

                if ($self->checkEntryIcon($entry, $entry2, $entry4)) {

                    $cmd = 'addroute ' . $startTag . ' ' . $stopTag . ' <' . $route . '> -q';
                }

            } else {

                if ($self->checkEntryIcon($entry, $entry3, $entry4)) {

                    $cmd = 'addroute ' . $startTag . ' ' . $circuitName . ' <' . $route . '> -t';
                }
            }

            if ($cmd) {

                $cmd .= ' -d ' . $self->editObj->profName;

                if (! $hopFlag) {

                    $cmd .= ' -h';
                }

                # Send the command
                $self->session->pseudoCmd($cmd, $self->pseudoCmdMode);

                # Refresh the simple lists in both tabs
                $self->routes1Tab_refreshList();
                $self->routes2Tab_refreshList();
            }
        });

        my $button2 = $self->addButton(
            $table,
            'Reverse route',
            'Set up a route in the reverse direction',
            undef,
            1, 3, 11, 12,
        );
        $button2->signal_connect('clicked' => sub {

            my ($startTag, $stopTag, $sequence, $reverse);

            # The user must have entered a start tag, stop tag and route
            if ($self->checkEntryIcon($entry, $entry2, $entry4)) {

                $startTag = $entry->get_text();
                $stopTag = $entry2->get_text();
                $sequence = $entry4->get_text();

                # See if we can reverse the sequence of world commands in $sequence
                ($reverse) = $self->session->worldModelObj->reversePath(
                    $self->session,
                    'no_abbrev',        # Don't abbreviate, use first opposite direction
                    $sequence,
                );

                if ($reverse) {

                    # Change the contents of the entry boxes; don't create a new route
                    $entry->set_text($stopTag);
                    $entry2->set_text($startTag);
                    $entry4->set_text($reverse);

                } else {

                    $self->showMessageDialogue(
                        'Reverse route',
                        'Error',
                        'Couldn\'t reverse the list of world commands',
                        'ok',
                        $self->winWidget,
                    );
                }

                # Refresh the simple lists in both tabs
                $self->routes1Tab_refreshList();
                $self->routes2Tab_refreshList();
            }
        });

        my $button3 = $self->addButton(
            $table,
            'Edit...',
            'Edit the selected pre-defined route',
            undef,
            5, 7, 11, 12,
        );
        $button3->signal_connect('clicked' => sub {

            my ($routeName, $routeObj, $string, $childWinObj);

            ($routeName) = $self->getSimpleListData($slWidget, 0);
            if ($routeName) {

                $routeObj = $self->editObj->ivShow('routeHash', $routeName);

                if (defined $routeObj) {

                    $string = 'Edit ' . $routeObj->routeType . ' route';
                    if ($routeObj->routeType eq 'circuit') {
                        $string .= ' \'' . $routeObj->circuitName . '\'';
                    } else {
                        $string .= ' (' . $routeObj->startRoom . ' > ' . $routeObj->stopRoom . ')';
                    }

                    # Open an 'edit' window for the route object
                    $childWinObj = $self->createFreeWin(
                        'Games::Axmud::EditWin::Route',
                        $self,
                        $self->session,
                        $string,
                        $routeObj,
                        FALSE,                  # Not temporary
                    );
                }

                if ($childWinObj) {

                    # When the 'edit' window closes, update widgets and/or IVs
                    $self->add_childDestroy(
                        $childWinObj,
                        'routes1Tab_refreshList',
                        [],             # Function requires no arguments
                    );

                    $self->add_childDestroy(
                        $childWinObj,
                        'routes2Tab_refreshList',
                        [],             # Function requires no arguments
                    );
                }
            }
        });

        my $button4 = $self->addButton(
            $table,
            'Delete',
            'Delete the selected pre-defined route',
            undef,
            7, 9, 11, 12,
            TRUE,           # Irreversible
        );
        $button4->signal_connect('clicked' => sub {

            my ($routeName, $routeObj, $cmd);

            ($routeName) = $self->getSimpleListData($slWidget, 0);
            $routeObj = $self->editObj->ivShow('routeHash', $routeName);

            if (defined $routeObj) {

                # Prepare the client command
                if ($routeObj->routeType eq 'road') {

                    $cmd = 'deleteroute ' . $routeObj->startRoom  . ' ' . $routeObj->stopRoom
                            . ' -o';

                } elsif ($routeObj->routeType eq 'quick') {

                    $cmd = 'deleteroute ' . $routeObj->startRoom  . ' ' . $routeObj->stopRoom
                            . ' -q';

                } else {

                    $cmd = 'deleteroute ' . $routeObj->startRoom  . ' ' . $routeObj->circuitName
                            . ' -t';
                }

                if ($cmd) {

                    $cmd .= ' -d ' . $self->editObj->profName;

                    # Delete the route
                    $self->session->pseudoCmd($cmd, $self->pseudoCmdMode);

                    # Refresh the simple lists in both tabs
                    $self->routes1Tab_refreshList();
                    $self->routes2Tab_refreshList();
                }
            }
        });

        my $button5 = $self->addButton(
            $table,
            'Dump routes',
            'Display this list of routes in the \'main\' window',
            undef,
            9, 11, 11, 12,
        );
        $button5->signal_connect('clicked' => sub {

            my $profName = $self->editObj->profName;
            if ($profName) {

                # Display the list of routes
                $self->session->pseudoCmd('listroute -d ' . $profName, $self->pseudoCmdMode);
            }
        });

        my $button6 = $self->addButton(
            $table,
            'Refresh list',
            'Refreshes this list of routes',
            undef,
            11, 13, 11, 12,
        );
        $button6->signal_connect('clicked' => sub {

            # Refresh the simple lists in both tabs
            $self->routes1Tab_refreshList();
            $self->routes2Tab_refreshList();
        });

        # Depending on the combo box setting, sensitise/desensitise the boxes, as needed
        $combo->signal_connect('changed' => sub {

            my $text = $combo->get_active_text();

            if ($text eq 'road' || $text eq 'quick') {

                $checkButton->set_sensitive(TRUE);  # Hoppable
                $entry->set_sensitive(TRUE);        # Start room
                $entry2->set_sensitive(TRUE);       # Stop room
                $entry3->set_text('');              # Circuit name
                $entry3->set_sensitive(FALSE);

            } else {

                $checkButton->set_sensitive(FALSE); # Hoppable
                $entry->set_sensitive(TRUE);        # Start room
                $entry2->set_text('');              # Stop room
                $entry2->set_sensitive(FALSE);
                $entry3->set_sensitive(TRUE);       # Circuit name
            }
        });

        # We start on the 'road' setting, so the circuit entry box is insensitive
        $entry3->set_sensitive(FALSE);              # Circuit name

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::ChatContact;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::Generic::EditWin Games::Axmud::Generic::ConfigWin
        Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::Obj::ChatContact')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

#   sub expandNotebook {}       # Inherited from GA::Generic::ConfigWin

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    # Notebook tabs

    sub nameTab {

        # Name tab - called by $self->setupNotebook
        #
        # Expected arguments
        #   $table  -  The Gtk2::Table which has already been created for this tab
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $table, $check) = @_;

        # Local variables
        my @comboList;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->nameTab', @_);
        }

#       # Tab setup (already created by the calling function)
#       my ($vBox, $table) = $self->addTab('_Name', $self->notebook);

        # Left column
        $self->addLabel($table, '<b>Chat contact settings</b>',
            0, 6, 0, 1);

        $self->addLabel($table, 'Unique name',
            1, 3, 1, 2);
        $self->addEntry($table, 'name', FALSE,
            3, 6, 1, 2);

        $self->addLabel($table, 'Protocol',
            1, 3, 2, 3);
        @comboList = ('MudMaster', 'zChat');
        my $comboBox = $self->addComboBox($table, undef, \@comboList, '',
            TRUE,               # No 'undef' value used
            3, 6, 2, 3);
        $comboBox->set_active($self->editObj->protocol);    # IV is set to either 0 or 1
        $comboBox->signal_connect('changed' => sub {

            my $text = $comboBox->get_active_text();
            if ($text eq 'MudMaster') {
                $self->ivAdd('editHash', 'protocol', 0);
            } else {
                $self->ivAdd('editHash', 'protocol', 1);
            }
        });

        $self->addLabel($table, 'IP address',
            1, 3, 3, 4);
        $self->addEntryWithIcon($table, 'ip', 'string', 1, undef,
            3, 6, 3, 4);
        $self->addLabel($table, 'Port',
            1, 3, 4, 5);
        $self->addEntryWithIcon($table, 'port', 'int', 0, 65535,
            3, 6, 4, 5);
        $self->addLabel($table, 'Email',
            1, 3, 5, 6);
        $self->addEntry($table, 'email', FALSE,
            3, 12, 5, 6);

        # Right column
        $self->addImage($table, undef, $self->editObj->lastIcon,
            FALSE,          # Don't use a scrolled window
            128, 128,
            7, 12, 1, 5);

#       # Tab complete (handled by the calling function)
#       $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::ColourScheme;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::Generic::EditWin Games::Axmud::Generic::ConfigWin
        Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::Obj::ColourScheme')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        $self->colourTab();
        $self->overrideTab();
        $self->textTab();

        return 1;
    }

    sub saveChanges {

        # Called by $self->buttonOK and $self->buttonSave (usually for 'edit' windows only, not
        #   'pref' windows)
        # Saves any changes made to data stored by the edit object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->saveChanges', @_);
        }

        if ($self->editHash) {

            # Store the changes the user has made
            foreach my $key ($self->ivKeys('editHash')) {

                $self->editObj->{$key} = $self->ivShow('editHash', $key);
            }

            # The changes can now be cleared
            $self->ivEmpty('editHash');

            # Mark the object's corresponding file object as needing to be saved, if it exists
            if ($self->editObj->_parentFile) {

                $self->editObj->doModify('saveChanges');
            }

            # Update the current session's object viewer window, if it is open
            if ($self->session->viewerWin) {

                $self->session->viewerWin->updateNotebook();
            }

            # Update all 'internal' windows using this colour scheme
            $self->session->pseudoCmd(
                'updatecolourscheme ' . $self->editObj->name,
                $self->pseudoCmdMode,
            );
        }

        return 1;
    }

    # Notebook tabs

    sub nameTab {

        # Name tab - called by $self->setupNotebook
        #
        # Expected arguments
        #   $table  - The Gtk2::Table
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $table, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->nameTab', @_);
        }

#       # Tab setup (already created by the calling function)
#       my ($vBox, $table) = $self->addTab('_Name', $self->notebook);

        # Colour scheme
        $self->addLabel($table, '<b>Colour scheme settings</b>',
            0, 12, 0, 1);
        $self->addLabel($table, 'Name',
            1, 3, 1, 2);
        $self->addEntry($table, 'name', 0,
            3, 6, 1, 2);
        $self->addLabel($table, 'Standard colour scheme',
            7, 11, 1, 2);
        my $checkButton = $self->addCheckButton($table, undef, FALSE,
            11, 12, 1, 2, 1, 0.5);
        if (
            $axmud::CLIENT->ivExists('constGridWinTypeHash', $self->editObj->name)
            || $axmud::CLIENT->ivExists('constFreeWinTypeHash', $self->editObj->name)
        ) {
            $checkButton->set_active(TRUE);
        }

#       # Tab complete (handled by the calling function)
#       $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub colourTab {

        # Colour tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->colourTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('_Colours', $self->notebook);

        # Colour settings
        $self->addLabel($table, '<b>Colour settings</b>',
            0, 12, 0, 1);

        my $hiddenButton = $self->colourTab_addRow($table, 'textColour', 'Text colour', 1);
        my $hiddenButton2 = $self->colourTab_addRow($table, 'underlayColour', 'Underlay colour', 3);
        my $hiddenButton3 = $self->colourTab_addRow(
            $table,
            'backgroundColour',
            'Background colour',
            5,
        );

        my $button = $self->addButton($table,
            'Set underlay to match background',
            'Update the underlay colour',
            undef,
            1, 12, 7, 8);
        $button->signal_connect('clicked' => sub {

            my $background = $self->getEditHash_scalarIV('backgroundColour');

            # Update IVs
            $self->ivAdd(
                'editHash',
                'underlayColour',
                $axmud::CLIENT->swapColours($background),
            );

            # Click the hidden button to redraw the Gnome2::Canvas/Gtk2::Entry objects
            $hiddenButton2->clicked();
        });

        my $button2 = $self->addButton($table,
            'Swap text and background colours',
            'Update the text and background colours',
            undef,
            1, 12, 8, 9);
        $button2->signal_connect('clicked' => sub {

            my ($text, $underlay, $background);

            $text = $self->getEditHash_scalarIV('textColour');
            $underlay = $axmud::CLIENT->swapColours($self->getEditHash_scalarIV('underlayColour'));
            $background = $self->getEditHash_scalarIV('backgroundColour');

            # Update IVs. If the background and underlay are the same colour, update that, too
            $self->ivAdd('editHash', 'textColour', $background);
            $self->ivAdd('editHash', 'backgroundColour', $text);
            if ($underlay eq $background) {

                $self->ivAdd('editHash', 'underlayColour', $axmud::CLIENT->swapColours($text));
            }

            # Click the hidden buttons to redraw the Gnome2::Canvas/Gtk2::Entry objects
            $hiddenButton->clicked();
            $hiddenButton2->clicked();
            $hiddenButton3->clicked();
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub colourTab_addRow {

        # Called by $self->nameTab
        # Adds a single row of labels, entry boxes and buttons to allow configuration of a single
        #   colour setting
        #
        # Expected arguments
        #   $table      - The Gtk2::Table for this tab
        #   $iv         - The IV to use, e.g. 'textColour'
        #   $text       - The label to use, e.g. 'Text colour'
        #   $row        - The number of the row in the Gtk2::Table displayed in this tab
        #
        # Return values
        #   'undef' on improper arguments
        #   Otherwise returns a Gtk2::Button which isn't visible in the tab, but which can be
        #       'clicked' by the calling function, and thereby change the displayed colour

        my ($self, $table, $iv, $text, $row, $check) = @_;

        # Local variables
        my (
            $title, $colour, $rgbColour, $constIV, $default, $rgbDefault,
            @tagList,
        );

        # Check for improper arguments
        if (
            ! defined $table || ! defined $iv || ! defined $text || ! defined $row
            || defined $check
        ) {
            return $axmud::CLIENT->writeImproper($self->_objClass . '->colourTab_addRow', @_);
        }

        # Initialise vars
        $title = 'Or select tag:';
        $colour = $self->editObj->$iv;
        $rgbColour = $axmud::CLIENT->returnRGBColour($colour);

        # (e.g. convert GA::Obj::ColourScheme->textColour to GA::Client->constTextColour)
        $constIV = 'const' . ucfirst($iv);
        $default = $axmud::CLIENT->$constIV;
        $rgbDefault = $axmud::CLIENT->returnRGBColour($default);

        # $rgbColour and $rgbDefault are used to fill the boxes with colour, so convert an Axmud
        #   underlay tag to a non-underlay tag
        $rgbColour =~ s/^[Uu]//;
        $rgbDefault =~ s/^[Uu]//;

        # Prepare a list of items for a combo
        if ($iv eq 'underlayColour') {

            foreach my $tag ($axmud::CLIENT->constColourTagList) {

                push (@tagList, 'ul_' . $tag);
            }

            foreach my $tag ($axmud::CLIENT->constBoldColourTagHash) {

                push (@tagList, 'UL_' . $tag);
            }

        } else {

            @tagList = ($axmud::CLIENT->constColourTagList, $axmud::CLIENT->constBoldColourTagList);
        }

        # Colour 1
        $self->addLabel($table, $text,
            1, 3, $row, ($row + 1));

        my ($frame, $canvas, $canvasObj) = $self->addSimpleCanvas($table, $rgbColour, undef,
            3, 4, $row, ($row + 1));

        my $entry = $self->addEntry($table, undef, FALSE,
            4, 6, $row, ($row + 1), 16, 16);
        $entry->set_text($colour);

        my $button = $self->addButton($table, 'Change', 'Change this colour', undef,
            6, 7, $row, ($row + 1));
        $button->signal_connect('clicked' => sub {

            # Prompt the user to select a new colour, using the existing colour as an initial value
            my $rgbModify = $self->showColourSelectionDialogue('Set ' . lc($text), $rgbColour);

            if ($rgbModify) {

                $rgbColour = $rgbModify;
                $canvasObj = $self->fillSimpleCanvas($canvas, $canvasObj, $rgbColour);

                # Convert a text colour tag to an underlay colour tag, if it's an underlay IV
                if ($iv eq 'underlayColour') {

                    $rgbModify = $axmud::CLIENT->swapColours($rgbModify);
                }

                $entry->set_text($rgbModify);

                # Update IVs
                $self->ivAdd('editHash', $iv, $rgbModify);
            }
        });

        # Colour 2
        my $combo = $self->addComboBox($table, undef, \@tagList, $title,
            TRUE,           # no 'undef'
            4, 6, ($row + 1), ($row + 2));

        my $button2 = $self->addButton($table, 'Use', 'Use this colour tag', undef,
            6, 7, ($row + 1), ($row + 2));
        $button2->signal_connect('clicked' => sub {

            my $text = $combo->get_active_text();
            if ($text && $text ne $title) {

                $canvasObj = $self->fillSimpleCanvas(
                    $canvas,
                    $canvasObj,
                    $axmud::CLIENT->returnRGBColour($text),
                );

                $entry->set_text($text);

                # Update IVs
                $self->ivAdd('editHash', $iv, $text);
            }
        });

        # Default colour
        $self->addSimpleCanvas($table, $rgbDefault, undef,
            8, 9, $row, ($row + 1));

        my $entry2 = $self->addEntry($table, undef, FALSE,
            9, 11, $row, ($row + 1), 16, 16);
        $entry2->set_text($default);

        my $button3 = $self->addButton($table, 'Use default', 'Use the default colour', undef,
            11, 12, $row, ($row + 1));
        $button3->signal_connect('clicked' => sub {

            $canvasObj = $self->fillSimpleCanvas($canvas, $canvasObj, $rgbDefault);
            $entry->set_text($default);

            # Update IVs
            $self->ivAdd('editHash', $iv, $default);
        });

        # Create a Gtk2::Button which isn't visible in the tab, but which can be 'clicked' by the
        #   calling function, and thereby change the displayed colour
        my $hiddenButton = Gtk2::Button->new('Hidden');
        $hiddenButton->signal_connect('clicked' => sub {

            $rgbColour = $self->getEditHash_scalarIV($iv);
            $canvasObj = $self->fillSimpleCanvas($canvas, $canvasObj, $rgbColour);

            $entry->set_text($rgbColour);

            # Update IVs
            $self->ivAdd('editHash', $iv, $rgbColour);
        });

        return $hiddenButton;
    }

    sub overrideTab {

        # Override tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->overrideTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Colour _overrides', $self->notebook);

        # Colour overrides
        $self->addLabel($table, '<b>Colour overrides</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of ' . $axmud::SCRIPT . ' colour tags which should be ignored or replaced when'
            . ' text is displayed using this colour scheme</i>',
            1, 12, 1, 2);
        my $checkButton = $self->addCheckButton($table, 'overrideAllFlag', TRUE,
            1, 12, 2, 3);
        $checkButton->set_label(
#            'Ignore all colour tags except the textview\'s default text, underlay and background'
#            . ' colours',
            'Ignore all colour tags except the normal text and background colours',
        );

        # Add a simple list
        @columnList = (
            'Original colour', 'text',
            'Replacement colour', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 3, 9,
            -1, 220);       # Fixed height

        # Initialise the simple list
        $self->overrideTab_refreshList($slWidget, scalar (@columnList / 2));

        # Add editing widgets
        $self->addLabel($table,
            'Colour tag  - Standard (e.g. <i>red, BLUE</i>), Xterm (e.g. <i>x128</i>)'
            . ' or RGB (e.g. <i>#ABCDEF</i>)',
            1, 9, 9, 10);
        my $entry = $self->addEntryWithIcon($table, undef, \&overrideTab_checkEntry, undef, undef,
            9, 12, 9, 10,
            16, 16);
        $self->addLabel($table,
            'Replacement tag - Standard, Xterm or RGB, or leave empty to use default colours',
            1, 9, 10, 11);
        my $entry2 = $self->addEntryWithIcon($table, undef, \&overrideTab_checkEntry2, undef, undef,
            9, 12, 10, 11,
            16, 16);
        # No text is an acceptable value, but the 'gtk-no' icon is currently visible. Force an
        #   update
        $self->setEntryIcon($entry2, TRUE);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'overrideHash',
            11,
            $entry, $entry2,
        );
        $button->signal_connect('clicked' => sub {

            my ($tag, $replace);

            $tag = $entry->get_text();
            $replace = $entry2->get_text();

            if ($self->checkEntryIcon($entry, $entry2)) {

                # To use the default colours, use an 'undef' value
                if ($replace eq '') {

                    $replace = undef;
                }

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('overrideHash', $tag, $replace);

                # Refresh the simple list and reset entry boxes
                $self->overrideTab_refreshList($slWidget, scalar (@columnList / 2));
                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub overrideTab_refreshList {

        # Called by $self->overrideTab to refresh the GA::Obj::Simple::List
        #
        # Expected arguments
        #   $slWidget   - The GA::Obj::Simple::List
        #   $columns    - The number of columns in the list
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $slWidget, $columns, $check) = @_;

        # Local variables
        my (
            @dataList,
            %ivHash,
        );

        # Check for improper arguments
        if (! defined $slWidget || ! defined $columns || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->overrideTab_refreshList',
                @_,
            );
        }

        # Import the IV
        %ivHash = $self->getEditHash_hashIV('overrideHash');

        # Compile the simple list data
        foreach my $key (sort {lc($a) cmp lc($b)} (keys %ivHash)) {

            if (defined $ivHash{$key}) {
                push (@dataList, $key, $ivHash{$key});
            } else {
                push (@dataList, $key, '<use normal colour>');
            }
        }

        # Reset the simple list
        $self->resetListData($slWidget, [@dataList], $columns);

        return 1;
    }

    sub overrideTab_checkEntry {

        # Callback from $self->overrideTab
        # Checks the contents of the Gtk2::Entry, returning FALSE or TRUE
        # Set up additional tabs for the notebook
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   Otherwise returns FALSE if the contents are not acceptable, TRUE if they are acceptable

        my ($self, $text, $check) = @_;

        # Local variables
        my $type;

        # Check for improper arguments
        if (! defined $text || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->overrideTab_checkEntry', @_);
        }

        ($type) = $axmud::CLIENT->checkColourTags($text);
        if (! defined $type) {
            return FALSE;
        } else {
            return TRUE;
        }
    }

    sub overrideTab_checkEntry2 {

        # Callback from $self->overrideTab
        # Checks the contents of the Gtk2::Entry, returning FALSE or TRUE
        # Set up additional tabs for the notebook
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   Otherwise returns FALSE if the contents are not acceptable, TRUE if they are acceptable

        my ($self, $text, $check) = @_;

        # Local variables
        my $type;

        # Check for improper arguments
        if (! defined $text || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->overrideTab_checkEntry', @_);
        }

        if ($text eq '') {

            return TRUE;

        } else {

            ($type) = $axmud::CLIENT->checkColourTags($text);
            if (! defined $type) {
                return FALSE;
            } else {
                return TRUE;
            }
        }
    }

    sub textTab {

        # Colour tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            $count, $current,
            @list, @comboList,
            %descripHash,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->textTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('_Text settings', $self->notebook);

        # Text settings
        $self->addLabel($table, '<b>Text settings</b>',
            0, 12, 0, 1);

        # Font settings
        $self->addLabel($table, '<i><u>Font settings</u></i>',
            1, 12, 1, 2);

        $self->addLabel($table, 'Font',
            1, 4, 2, 3);

        my $entry = $self->addEntry($table, undef, FALSE,
            4, 6, 2, 3);
        $entry->set_text($self->editObj->font . ' ' . $self->editObj->fontSize);

        my $button3 = $self->addButton($table, 'Change', 'Change this font', undef,
            6, 7, 2, 3);
        $button3->signal_connect('clicked' => sub {

            my $font = $self->showFontSelectionDialogue(
                'Colour scheme \'' . $self->editObj->name . '\' font',
            );

            if (defined $font) {

                # $font is a string in the form 'Monospace 10'. Separate the font name from the
                #   size
                if ($font =~ m/(.*)\s(.\d)$/) {

                    $self->ivAdd('editHash', 'font', $1);
                    $self->ivAdd('editHash', 'fontSize', $2);

                    $entry->set_text($font);
                }
            }
        });

        my $entry2 = $self->addEntry($table, undef, FALSE,
            9, 11, 2, 3);
        $entry2->set_text($axmud::CLIENT->constFont . ' ' . $axmud::CLIENT->constFontSize);

        my $button4 = $self->addButton($table, 'Use default', 'Use the default font', undef,
            11, 12, 2, 3);
        $button4->signal_connect('clicked' => sub {

            $self->ivAdd('editHash', 'font', $axmud::CLIENT->constFont);
            $self->ivAdd('editHash', 'fontSize', $axmud::CLIENT->constFontSize);

            $entry->set_text($axmud::CLIENT->constFont . ' ' . $axmud::CLIENT->constFontSize);
        });

        # Word wrap settings
        $self->addLabel($table, '<i><u>Word wrap settings</u></i>',
            1, 12, 3, 4);

        @list = (
            'no_wrap'           => 'Don\'t wrap text to fit the window',
            'wrap_char'         => 'Wrap text and split up words',
            'wrap_word'         => 'Wrap text and don\'t split up words',
            'wrap_word_char'    => 'Don\'t split up words unless it\'s necessary',
        );

        $count = -1;

        do {

            my ($mode, $descrip);

            $mode = shift @list;
            $descrip = shift @list;
            $count++;

            $descripHash{$descrip} = $mode;
            push (@comboList, $descrip);

            if ($mode eq $self->editObj->wrapMode) {

                $current = $count;
            }

        } until (! @list);

        my $comboBox = $self->addComboBox($table, undef, \@comboList, '',
            TRUE,               # No 'undef' value used
            1, 6, 4, 5);
        if (defined $current) {

            $comboBox->set_active($current);
        }
        $comboBox->signal_connect('changed' => sub {

            my $descrip = $comboBox->get_active_text();

            $self->ivAdd('editHash', 'wrapMode', $descripHash{$descrip});
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::Component;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::Generic::EditWin Games::Axmud::Generic::ConfigWin
        Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::Obj::Component')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        $self->locatorOptionsTab();
        $self->otherOptionsTab();
        if ($self->editObj->type ne 'verb_special') {

            $self->patternsTagsTab();
        }

        return 1;
    }

#   sub expandNotebook {}       # Inherited from GA::Generic::ConfigWin

    sub saveChanges {

        # Called by $self->buttonOK and $self->buttonSave
        # Saves the changes made to the component object. Check that any tags added are valid, and
        #   remove any that are invalid. (Dummy style tags are allowed, but not recommended.)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my @ivList;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->saveChanges', @_);
        }

        # A list of the IVs which contain lists of colour/style tags
        @ivList = (
            'startTagList',
            'startNoTagList',
            'skipTagList',
            'stopBeforeTagList',
            'stopBeforeNoTagList',
            'stopAtTagList',
            'stopAtNoTagList',
        );

        if ($self->editHash) {

            foreach my $iv (@ivList) {

                my (
                    $listRef,
                    @firstList, @modList,
                );

                if ($self->ivExists('editHash', $iv)) {

                    $listRef = $self->ivShow('editHash', $iv);
                    # $listRef could conceivably contain multiple colour/style tags on the same
                    #   line. Compile a list, with each item being a distinct colour/style tag
                    foreach my $line (@$listRef) {

                        push (@firstList, split(m/\s+/, $line));
                    }

                    # Remove any invalid colour/style tags (dummy style tags are allowed, though not
                    #   recommended)
                    foreach my $tag (@firstList) {

                        my ($type) = $axmud::CLIENT->checkColourTags($tag);

                        if (
                            $axmud::CLIENT->ivExists('constStyleTagHash', $tag)
                            || $axmud::CLIENT->ivExists('constDummyTagHash', $tag)
                            || $type
                        ) {
                            push (@modList, $tag);
                        }
                    }

                    # Update the IV
                    $self->ivPoke('editHash', $iv, \@modList);
                }
            }

            # Store the changes the user has made
            foreach my $key ($self->ivKeys('editHash')) {

                $self->editObj->{$key} = $self->ivShow('editHash', $key);
            }

            # The changes can now be cleared
            $self->ivEmpty('editHash');

            # Mark the object's corresponding file object as needing to be saved, if it exists
            if ($self->editObj->_parentFile) {

                $self->editObj->doModify('saveChanges');
            }

            # Update the current session's object viewer window, if it is open
            if ($self->session->viewerWin) {

                $self->session->viewerWin->updateNotebook();
            }
        }

        return 1;
    }

    # Notebook tabs

    sub nameTab {

        # Name tab - called by $self->setupNotebook
        #
        # Expected arguments
        #   $table  -   The Gtk2::Table
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $table, $check) = @_;

        # Local variables
        my (
            $allTextMsg, $current, $matchFlag,
            @comboList,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->nameTab', @_);
        }

#       # Tab setup (already created by the calling function)
#       my ($vBox, $table) = $self->addTab('_Name', $self->notebook);

        # Component name
        $self->addLabel($table, '<b>Component name</b>',
            0, 6, 0, 1);
        $self->addEntry($table, 'name', FALSE,
            1, 6, 1, 2, 16, 16);

        # Component type
        $self->addLabel($table, '<b>Component type</b>',
            7, 13, 0, 1);
        $self->addEntry($table, 'type', FALSE,
            8, 13, 1, 2, 16, 16);

        # Component expected size
        $self->addLabel($table, '<b>Component expected size</b>',
            0, 13, 2, 3);

        $self->addLabel(
            $table, 'Fixed size (range 0-256) (0 = use other properties to find the component)',
            1, 9, 3, 4);
        my $entry = $self->addEntryWithIcon($table, 'size', 'int', 0, 256,
            9, 13, 3, 4);

        $self->addLabel($table, 'Minimum size (range 0-256) (0 = this component is optional)',
            1, 9, 4, 5);
        my $entry2 = $self->addEntryWithIcon($table, 'minSize', 'int', 0, 256,
            9, 13, 4, 5);

        $self->addLabel($table, 'Maximum size (range 1-256)',
            1, 9, 5, 6);
        my $entry3 = $self->addEntryWithIcon($table, 'maxSize', 'int', 1, 256,
            9, 13, 5, 6);

        if ($self->editObj->type eq 'mudlib_path') {

            # 'mudlib_path' component cannot be longer than 1 line; don't let user change the
            #   component size
            $entry->set_sensitive(FALSE);
            $entry2->set_sensitive(FALSE);
            $entry3->set_sensitive(FALSE);
        }

        # Component analyse mode
        $self->addLabel($table, '<b>Component analyse mode</b>',
            0, 13, 6, 7);

        if ($self->editObj->type eq 'verb_special') {

            $self->addLabel(
                $table,
                '<i>For this component, lines are only checked against special contents'
                . ' patterns stored in the world profile</i>',
                1, 13, 7, 8);

        } else {

            my ($group, $radioButton) = $self->addRadioButton(
                $table, undef, '\'check_line\'', 'analyseMode',
                'check_line',   # IV set to this value when toggled
                TRUE,           # Sensitive widget
                1, 4, 8, 9);
            $self->addLabel(
                $table,
                'Check each line, one at a time, against all patterns/tags <i>(recommended)</i>',
                4, 12, 8, 9);

            ($group, $radioButton) = $self->addRadioButton(
                $table, $group, '\'check_pattern_tag\'', 'analyseMode', 'check_pattern_tag', TRUE,
                1, 4, 9, 10);
            $self->addLabel($table, 'Check each pattern/tag, one at a time, against all lines',
                4, 12, 9, 10);

            $self->addLabel(
                $table,
                '<i>NB Most patterns and tags are ignored if the component has a fixed size</i>',
                1, 13, 10, 11);
            $self->addLabel(
                $table,
                '<i>NB The two sets of start patterns/tags are always checked together</i>',
                1, 13, 11, 12);

        }

#       # Tab complete (handled by the calling function)
#       $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub locatorOptionsTab {

        # LocatorOptions tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            $allTextMsg, $current, $matchFlag,
            @comboList,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->locatorOptionsTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('_Locator options', $self->notebook);

        # Locator task options
        $self->addLabel($table, '<b>Locator task options</b>',
            0, 13, 0, 1);
        $self->addLabel(
            $table,
            '<i>Options which affect the way the Locator task interprets the text of a'
            . ' component</i>',
            1, 13, 1, 2);

        $self->addLabel($table, 'Locator doesn\'t use normal/bold colour tags interchangeably',
            1, 12, 2, 3);
        $self->addCheckButton($table, 'boldSensitiveFlag', TRUE,
            12, 13, 2, 3, 1, 0.5);

        $self->addLabel(
            $table,
            'Locator only considers colours/styles that applied at the beginning of the line',
            1, 12, 3, 4);
        my $checkButton = $self->addCheckButton($table, 'useInitialTagsFlag', TRUE,
            12, 13, 3, 4, 1, 0.5);

        $self->addLabel(
            $table,
            'Locator only considers colours/styles that were explicity specified on this line',
            1, 12, 4, 5);
        my $checkButton2 = $self->addCheckButton($table, 'useExplicitTagsFlag', TRUE,
            12, 13, 4, 5, 1, 0.5);

        $self->addLabel($table, 'Locator treats this component as if it were a single line',
            1, 12, 5, 6);
        $self->addCheckButton($table, 'combineLinesFlag', TRUE,
            12, 13, 5, 6, 1, 0.5);

        $self->addLabel($table, '<b>Locator task ignore options</b>',
            0, 13, 6, 7);
        $self->addLabel(
            $table,
            '<i>Options which allow the Locator task to ignore some parts of a line (if you'
            . ' modify several options, only one is used)</i>',
            1, 13, 7, 8);

        $self->addLabel($table, 'Locator ignores text that doesn\'t use this standard colour tag',
            1, 9, 8, 9);

        $allTextMsg = '(Use all colours)';
        @comboList = (
            $allTextMsg,
            $axmud::CLIENT->constColourTagList,
            $axmud::CLIENT->constBoldColourTagList,
        );

        foreach my $item ($axmud::CLIENT->constColourTagList) {

            push (@comboList, 'ul_' . $item);
        }

        foreach my $item ($axmud::CLIENT->constBoldColourTagList) {

            push (@comboList, 'UL_' . $item);
        }

        # (Don't use the generic combobox code so that the tags are always displayed in the same
        #   order)
        my $comboBox = $self->addComboBox($table, undef, \@comboList, '',
            TRUE,               # No 'undef' value used
            9, 13, 8, 9);

        $current = $self->getEditHash_scalarIV('useTextColour');
        if ($current) {

            # Display the current value of the IV in the combobox
            OUTER: for (my $count = 0; $count < scalar @comboList; $count++) {

                if ($comboList[$count] eq $current) {

                    $comboBox->set_active($count);
                    $matchFlag = TRUE;
                    last OUTER;
                }
            }

            if (! $matchFlag) {

                $comboBox->set_active(0);
            }
        }

       $comboBox->signal_connect('changed' => sub {

            my $text = $comboBox->get_active_text();

            # If the user has selected the title, ignore it
            if ($text eq $allTextMsg) {
                $self->ivAdd('editHash', 'useTextColour', undef);
            } else {
                $self->ivAdd('editHash', 'useTextColour', $text);
            }
        });

        $self->addLabel(
            $table,
            'Locator ignores the first <i>n</i> characters (0 - use whole line)',
            1, 9, 9, 10);
        $self->addEntryWithIcon($table, 'ignoreFirstChars', 'int', 0, undef,
            9, 13, 9, 10);
        $self->addLabel(
            $table,
            'Locator uses only the first <i>n</i> characters (0 - use whole line)',
            1, 9, 10, 11);
        $self->addEntryWithIcon($table, 'useFirstChars', 'int', 0, undef,
            9, 13, 10, 11);
        $self->addLabel(
            $table,
            'Locator uses only the contents of any group substring(s), if this pattern matches',
            1, 9, 11, 12);
        $self->addEntryWithIconButton($table, 'usePatternGroups', 'regex', 0, undef,
            1, 13, 12, 13);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub otherOptionsTab {

        # OtherOptions tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            $allTextMsg, $current, $matchFlag,
            @comboList,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->otherOptionsTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('_Other options', $self->notebook);

        # Non-extractable components
        $self->addLabel($table, '<b>Non-extractable components</b>',
            0, 12, 0, 1);
        $self->addLabel(
            $table,
            '<i>List of component types/names which should NOT be extracted, if this component can'
            . ' be extracted</i>',
            1, 12, 1, 2);

        $self->addTextView($table, 'noExtractList', TRUE,
            1, 12, 2, 6,
            TRUE, TRUE, TRUE, FALSE,    # Treat as list, remove empty lines, do remove whitespace
            -1, 150);                   # Fixed height

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub patternsTagsTab {

        # PatternsTags tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->patternsTagsTab', @_);
        }

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook)
            = $self->addInnerNotebookTab('_Patterns / tags', $self->notebook);

        # Add tabs to the inner notebook

        # Start patterns / tags
        $self->patternsTags1Tab(
            $innerNotebook,
            'Page _1',
            'startPatternList', 'startTagList', 'startAllFlag', 'startTagMode',
            '<b>Start patterns / tags</b>',
            '<i>The start of the component DOES match one of these patterns</i>',
            '<i>...DOES contain one of these tags</i>',
            '<i>The start of the component DOES match all of these patterns</i>',
            '<i>...DOES contain all of these tags</i>',
        );

        # Start patterns / tags (2)
        $self->patternsTags1Tab(
            $innerNotebook,
            'Page _2',
            'startNoPatternList', 'startNoTagList', 'startNoAllFlag', 'startNoTagMode',
            '<b>Start patterns / tags (2)</b>',
            '<i>The start of the component does NOT match one of these patterns</i>',
            '<i>...does NOT contain one of these tags</i>',
            '<i>The start of the component does NOT match all of these patterns</i>',
            '<i>...does NOT contain all of these tags</i>',
        );

        # Skip patterns / tags
        $self->patternsTags1Tab(
            $innerNotebook,
            'Page _3',
            'skipPatternList', 'skipTagList', 'skipAllFlag', 'skipTagMode',
            '<b>Skip patterns / tags</b>',
            '<i>Patterns which mean the line should be ignored</i>',
            '<i>Tags found on an ignorable line</i>',
            '<i>Patterns which mean the line should be ignored</i>',
            '<i>Tags found on an ignorable line</i>',
        );

        # Stop before patterns / tags
        $self->patternsTags1Tab(
            $innerNotebook,
            'Page _4',
            'stopBeforePatternList', 'stopBeforeTagList', 'stopBeforeAllFlag',
                'stopBeforeTagMode',
            '<b>Stop before patterns / tags</b>',
            '<i>Stop before the first line which DOES match one of these patterns</i>',
            '<i>...which DOES contain one of these tags</i>',
            '<i>Stop before the first line which DOES match all of these patterns</i>',
            '<i>...which DOES contain all of these tags</i>',
        );

        # Stop before patterns / tags (2)
        $self->patternsTags1Tab(
            $innerNotebook,
            'Page _5',
            'stopBeforeNoPatternList', 'stopBeforeNoTagList', 'stopBeforeNoAllFlag',
                'stopBeforeNoTagMode',
            '<b>Stop before patterns / tags (2)</b>',
            '<i>Stop before the first line which does NOT match one of these patterns</i>',
            '<i>...which does NOT contain one of these tags</i>',
            '<i>Stop before the first line which does NOT match all of these patterns</i>',
            '<i>...which does NOT contain all of these tags</i>',
        );

        # Stop at patterns / tags
        $self->patternsTags1Tab(
            $innerNotebook,
            'Page _6',
            'stopAtPatternList', 'stopAtTagList', 'stopAtAllFlag', 'stopAtTagMode',
            '<b>Stop at patterns / tags</b>',
            '<i>Stop at the first line which DOES match one of these patterns</i>',
            '<i>...which DOES contain one of these tags</i>',
            '<i>Stop at the first line which DOES match all of these patterns</i>',
            '<i>...which DOES contain all of these tags</i>',
        );

        # Stop at patterns / tags (2)
        $self->patternsTags1Tab(
            $innerNotebook,
            'Page _7',
            'stopAtNoPatternList', 'stopAtNoTagList', 'stopAtNoAllFlag', 'stopAtNoTagMode',
            '<b>Stop at patterns / tags (2)</b>',
            '<i>Stop at the first line which does NOT match one of these patterns</i>',
            '<i>... which does NOT contain one of these tags</i>',
            '<i>Stop at the first line which does NOT match all of these patterns</i>',
            '<i>...which does NOT contain all of these tags</i>',
        );

        # Stop at capitalised lines
        $self->patternsTags8Tab($innerNotebook);
        # Stop before mode
        $self->patternsTags9Tab($innerNotebook);
        # Stop at mode
        $self->patternsTags10Tab($innerNotebook);

        return 1;
    }

    sub patternsTags1Tab {

        # PatternsTags1 tab
        #
        # Expected arguments
        #   $innerNotebook
        #       - The Gtk2::Notebook object inside $self->notebook
        #   $tabName
        #       - e.g. 'Page _1'
        #   $patternListIV, $tagListIV, $flagIV, $modeIV
        #       - The names of the IVs being set (e.g. 'startPatternList', 'startTagList',
        #               'startAllFlag', 'startTagMode'
        #   @stringList
        #       - List containing five strings. The first is used for the main Gtk2::Label at the
        #           top of the tab; two out of the remaining four are used in the other Gtk2::Labels
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my (
            $self, $innerNotebook, $tabName, $patternListIV, $tagListIV, $flagIV, $modeIV,
            @stringList,
        ) = @_;

        # Local variables
        my (@comboList, @comboList2, @comboList3);

        # Check for improper arguments
        if (
            ! defined $innerNotebook || ! defined $tabName || ! defined $patternListIV
            || ! defined $tagListIV || ! defined $flagIV || ! defined $modeIV
            || scalar @stringList != 5
        ) {
            return $axmud::CLIENT->writeImproper($self->_objClass . '->patternsTags1Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab($tabName, $innerNotebook);

        # Title label, e.g. Start patterns / tags
        $self->addLabel($table, $stringList[0],
            0, 4, 0, 1);

        # First group of radio buttons (at top; ->signal_connects appear below)
        my ($group, $radioButton, $radioButton2);
        ($group, $radioButton) = $self->addRadioButton(
            $table, undef, 'Line contains at least one pattern / tag', undef, undef, TRUE,
            4, 9, 0, 1);
        ($group, $radioButton2) = $self->addRadioButton(
            $table, $group, 'All patterns / tags', undef, undef, TRUE,
            9, 12, 0, 1);
        if ($self->editObj->$flagIV) {

            $radioButton2->set_active(TRUE);
        }

        # Second group of radio buttons (in middle; ->signal_connects appear below)
        my ($group2, $radioButton11, $radioButton12, $radioButton13, $radioButton14);
        ($group2, $radioButton11) = $self->addRadioButton(
            $table, undef, 'Any tags', undef, undef, TRUE,
            4, 6, 6, 7);
        ($group2, $radioButton12) = $self->addRadioButton(
            $table, $group2, 'No colours', undef, undef, TRUE,
            6, 8, 6, 7);
        if ($self->editObj->$modeIV eq 'no_colour') {

            $radioButton12->set_active(TRUE);
        }
        ($group2, $radioButton13) = $self->addRadioButton(
            $table, $group2, 'No styles', undef, undef, TRUE,
            8, 10, 6, 7);
        if ($self->editObj->$modeIV eq 'no_style') {

            $radioButton13->set_active(TRUE);
        }
        ($group2, $radioButton14) = $self->addRadioButton(
            $table, $group2, 'No colours/styles', undef, undef, TRUE,
            10, 12, 6, 7);
        if ($self->editObj->$modeIV eq 'no_colour_style') {

            $radioButton14->set_active(TRUE);
        }

        # First label and textview, e.g. 'Patterns which mark the start of the component'
        my $label = $self->addLabel($table, '',
            1, 12, 1, 2);
        my $textView = $self->addTextView($table, $patternListIV, TRUE,
            1, 10, 2, 6,
            TRUE, TRUE, FALSE, FALSE,  # Treat as list, remove empty lines, don't remove whitespace
            -1, 100);                  # Fixed height
        $self->addRegexButton($table,
            [
                'list', $patternListIV,
            ],
            10, 12, 2, 3);
        $self->addLabel($table, '',    # Empty labels for spacing
            10, 12, 3, 4);
        $self->addLabel($table, '',    # Empty labels for spacing
            10, 12, 4, 5);
        $self->addLabel($table, '',    # Empty labels for spacing
            10, 12, 5, 6);

        # Second label and textview, e.g. 'Tags found at the start of the component'
        my $label2 = $self->addLabel($table, '',
            1, 4, 6, 7);
        my $textView2 = $self->addTextView($table, $tagListIV, TRUE,
            1, 4, 7, 12,
            TRUE, TRUE, TRUE, FALSE);  # Treat as list, remove empty lines, do remove whitespace
        my $buffer = $textView2->get_buffer();

        # (Set text for these two labels)
        if (! $self->getEditHash_scalarIV($flagIV)) {

            $label->set_markup($stringList[1]);
            $label2->set_markup($stringList[2]);

        } else {

            $label->set_markup($stringList[3]);
            $label2->set_markup($stringList[4]);
        }

        # ->signal_connects for first group of radiobuttons
        $radioButton->signal_connect('toggled' => sub {

            # Set the IV only if this radiobutton has been selected
            if ($radioButton->get_active()) {

                $self->ivAdd('editHash', $flagIV, FALSE);

                $label->set_markup($stringList[1]);
                $label2->set_markup($stringList[2]);
            }
        });

        $radioButton2->signal_connect('toggled' => sub {

            # Set the IV only if this radiobutton has been selected
            if ($radioButton2->get_active()) {

                $self->ivAdd('editHash', $flagIV, TRUE);

                $label->set_markup($stringList[3]);
                $label2->set_markup($stringList[4]);
            }
        });

        # Prepare lists for insertion into comboboxes
        @comboList = $axmud::CLIENT->constColourTagList;
        foreach my $item (@comboList) {

            push (@comboList2, 'ul_' . $item);
        }
        @comboList3 = $axmud::CLIENT->constStyleTagList;

        # Add editing combos / buttons in bottom-right corner
        # First set - standard text colour tags
        my $comboBox = $self->addComboBox($table, undef, \@comboList, 'Standard text colour tags:',
            TRUE,               # No 'undef' value used
            4, 10, 7, 8);
        my $button = $self->patternsTags1Tab_addButton(
            $table,
            7,
            $tagListIV,
            $buffer,
            $comboBox,
            'Standard text colour tags:',
        );
        my $button2 = $self->patternsTags1Tab_addBoldButton(
            $table,
            7,
            $tagListIV,
            $buffer,
            $comboBox,
            'Standard text colour tags:',
        );

        # Second set - standard underlay colour tags
        my $comboBox2 = $self->addComboBox(
            $table,
            undef,
            \@comboList2,
            'Standard underlay colour tags:',
            TRUE,               # No 'undef' value used
            4, 10, 8, 9);
        my $button3 = $self->patternsTags1Tab_addButton(
            $table,
            8,
            $tagListIV,
            $buffer,
            $comboBox2,
            'Standard underlay colour tags:',
        );
        my $button4 = $self->patternsTags1Tab_addBoldButton(
            $table,
            8,
            $tagListIV,
            $buffer,
            $comboBox2,
            'Standard underlay colour tags:',
        );

        # Third set - xterm tags
        $self->addLabel($table, 'xterm tags:',
            4, 6, 9, 10);
        my $entry = $self->addEntryWithIcon(
            $table,
            undef,
            \&patternsTags1Tab_checkEntry,
            undef,
            undef,
            6, 10, 9, 10,
            8, 8);

        # (Only one button of this type, for this tab, so it doesn't have its own function)
        my $button5 = $self->addButton(
            $table,
            'Add',
            'Add the selected xterm tag to the list',
            undef,
            10, 11, 9, 10,
        );
        $button5->signal_connect('clicked' => sub {

            my (
                $tag, $first, $second, $num, $modTag,
                @ivList,
            );

            if ($self->checkEntryIcon($entry)) {

                $tag = $entry->get_text();

                # Convert $tag to use standard capitalisation: xterm colour tags in the range
                #   'x0' to 'x255' or 'ux0' to 'ux255'
                $first = substr($tag, 0, 1);
                $second = substr($tag, 0, 2);
                if ($first eq 'x' || $first eq 'X') {

                    # (Make sure the nnn part is an integer without leading zeros, e.g. convert
                    #   001 to 1)
                    $num = substr($tag, 1);
                    $num += 0;
                    $modTag = 'x' . $num;

                } elsif ($second eq 'ux' || $second eq 'UX') {

                    $num = substr($tag, 2);
                    $num += 0;
                    $modTag = 'ux' . $num;
                }

                # Update the IV
                @ivList = $self->getEditHash_listIV($tagListIV);

                push (@ivList, $modTag);
                $self->ivAdd('editHash', $tagListIV, \@ivList);

                # Update the textview
                $buffer->set_text(join("\n", @ivList));
            }
        });

        # Fourth set - rgb tags
        $self->addLabel($table, 'RGB tags:',
            4, 6, 10, 11);
        my $entry2 = $self->addEntryWithIcon(
            $table,
            undef,
            \&patternsTags1Tab_checkEntry2,
            undef,
            undef,
            6, 10, 10, 11,
            8, 8);

        # (Only one button of this type, for this tab, so it doesn't have its own function)
        my $button6 = $self->addButton(
            $table,
            'Add',
            'Add the selected xterm tag to the list',
            undef,
            10, 11, 10, 11,
        );
        $button6->signal_connect('clicked' => sub {

            my (
                $tag, $first, $second, $num, $modTag,
                @ivList,
            );

            if ($self->checkEntryIcon($entry2)) {

                $tag = $entry2->get_text();

                # Convert $tag to use standard capitalisation: RGB colour tags in the range
                #   '#000000' to '#FFFFFF' or 'u#000000' to 'u#FFFFFF'
                $first = substr($tag, 0, 1);
                $second = substr($tag, 0, 2);
                if ($first eq '#') {

                    $modTag = uc($tag);

                } elsif ($second eq 'u#' || $second eq 'U#') {

                    $modTag = 'u#' . uc(substr($tag, 2));
                }

                # Update the IV
                @ivList = $self->getEditHash_listIV($tagListIV);

                push (@ivList, $modTag);
                $self->ivAdd('editHash', $tagListIV, \@ivList);

                # Update the textview
                $buffer->set_text(join("\n", @ivList));
            }
        });

        # Fifth set - style tags
        # NB 'Dummy' style tags can be added to the GA::Obj::Component object, although this is not
        #   recommended, so they are not included in the combobox
        my $comboBox3 = $self->addComboBox($table, undef, \@comboList3, 'Style tags:',
            TRUE,               # No 'undef' value used
            4, 10, 11, 12);
        my $button7 = $self->patternsTags1Tab_addButton(
            $table,
            11,
            $tagListIV,
            $buffer,
            $comboBox3,
            'Style tags:',
        );

        # Set initial sensitivity for these widgets
        $self->patternsTags1Tab_setSensitive(
            $self->getEditHash_scalarIV($modeIV),
            $textView2,
            [$comboBox, $button, $button2],
            [$comboBox2, $button3, $button4],
            [$entry, $button5],
            [$entry2, $button6],
            [$comboBox3, $button7],
        );

        # ->signal_connects for second group of radiobuttons
        $radioButton11->signal_connect('toggled' => sub {

            # Set the IV only if this radiobutton has been selected
            if ($radioButton11->get_active()) {

                $self->ivAdd('editHash', $modeIV, 'default');   # Any tags

                $self->patternsTags1Tab_setSensitive(
                    'default',
                    $textView2,
                    [$comboBox, $button, $button2],
                    [$comboBox2, $button3, $button4],
                    [$entry, $button5],
                    [$entry2, $button6],
                    [$comboBox3, $button6],
                );
            }
        });

        $radioButton12->signal_connect('toggled' => sub {

            # Set the IV only if this radiobutton has been selected
            if ($radioButton12->get_active()) {

                $self->ivAdd('editHash', $modeIV, 'no_colour');

                $self->patternsTags1Tab_setSensitive(
                    'no_colour',
                    $textView2,
                    [$comboBox, $button, $button2],
                    [$comboBox2, $button3, $button4],
                    [$entry, $button5],
                    [$entry2, $button6],
                    [$comboBox3, $button6],
                );
            }
        });

        $radioButton13->signal_connect('toggled' => sub {

            # Set the IV only if this radiobutton has been selected
            if ($radioButton13->get_active()) {

                $self->ivAdd('editHash', $modeIV, 'no_style');

                $self->patternsTags1Tab_setSensitive(
                    'no_style',
                    $textView2,
                    [$comboBox, $button, $button2],
                    [$comboBox2, $button3, $button4],
                    [$entry, $button5],
                    [$entry2, $button6],
                    [$comboBox3, $button6],
                );
            }
        });

        $radioButton14->signal_connect('toggled' => sub {

            # Set the IV only if this radiobutton has been selected
            if ($radioButton14->get_active()) {

                $self->ivAdd('editHash', $modeIV, 'no_colour_style');

                $self->patternsTags1Tab_setSensitive(
                    'no_colour_style',
                    $textView2,
                    [$comboBox, $button, $button2],
                    [$comboBox2, $button3, $button4],
                    [$entry, $button5],
                    [$entry2, $button6],
                    [$comboBox3, $button6],
                );
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub patternsTags1Tab_checkEntry {

        # Called by $self->patternsTags1Tab to check the text in the first Gtk2::Entry
        #
        # Expected arguments
        #   $text       - The contents of the Gtk2::Entry
        #
        # Return values
        #   'undef' on improper arguments or if $text is invalid
        #   1 if $text is valid

        my ($self, $text, $check) = @_;

        # Check for improper arguments
        if (! defined $text || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->patternsTags1Tab_checkEntry',
                @_,
            );
        }

        # $text can be an xterm colour tag (in the range 'x0' to 'x255', or 'ux0' to 'ux255'; xterm
        #   tags are case-insensitive
        if ($text =~ m/^u?x([0-9]+)$/i) {

            # (Don't permit 'x000005', but do permit 'x005' or 'x5')
            if ($1 >= 0 && $1 <= 255 && length ($1) <= 3) {
                return 1;
            } else {
                return undef;
            }

        } else {

            return undef;
        }
    }

    sub patternsTags1Tab_checkEntry2 {

        # Called by $self->patternsTags1Tab to check the text in the second Gtk2::Entry
        #
        # Expected arguments
        #   $text       - The contents of the Gtk2::Entry
        #
        # Return values
        #   'undef' on improper arguments or if $text is invalid
        #   1 if $text is valid

        my ($self, $text, $check) = @_;

        # Check for improper arguments
        if (! defined $text || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->patternsTags1Tab_checkEntry2',
                @_,
            );
        }

        # $text can be an RGB colour tag (in the range '#000000' to '#FFFFFF', or 'u#000000' to
        #   'u#FFFFFF'; RGB tags are case-insensitive
        if ($text =~ m/^u?\#[0-9A-F]{6}$/i) {
            return 1;
        } else {
            return undef;
        }
    }

    sub patternsTags1Tab_addButton {

        # Called by $self->patternsTags1Tab to add a Gtk2::Button next to a combobox
        #
        # Expected arguments
        #   $table          - The Gtk2::Table
        #   $row            - The position in the table of the buttons
        #   $iv             - The IV being edited (e.g. 'startTagList')
        #   $buffer         - The buffer of the Gtk2::TextView used to display the IV
        #   $comboBox       - The Gtk2::ComboBox used alongside the Gtk2::Button
        #   $title          - The title used at the top of the combobox
        #
        # Return values
        #   'undef' on improper arguments
        #   The Gtk2::Button created, otherwise

        my ($self, $table, $row, $iv, $buffer, $comboBox, $title, $check) = @_;

        # Check for improper arguments
        if (
            ! defined $table || ! defined $row || ! defined $iv || ! defined $buffer
            || ! defined $comboBox || ! defined $title || defined $check
        ) {
            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->patternsTags1Tab_addButton',
                @_,
            );
        }

        my $button = $self->addButton(
            $table,
            'Add',
            'Add the selected colour tag to the list',
            undef,
            10, 11, $row, ($row + 1),
        );
        $button->signal_connect('clicked' => sub {

            my (
                $tag,
                @ivList,
            );

            $tag = $comboBox->get_active_text();

            if ($tag && $tag ne $title) {

                # Update the IV
                @ivList = $self->getEditHash_listIV($iv);

                push (@ivList, $tag);
                $self->ivAdd('editHash', $iv, \@ivList);

                # Update the textview
                $buffer->set_text(join("\n", @ivList));
            }
        });

        return $button;
    }

    sub patternsTags1Tab_addBoldButton {

        # Called by $self->patternsTags1Tab to add another Gtk2::Button next to a combobox
        #
        # Expected arguments
        #   $table          - The Gtk2::Table
        #   $row            - The position in the table of the buttons
        #   $iv             - The IV being edited (e.g. 'startTagList')
        #   $buffer         - The buffer of the Gtk2::TextView used to display the IV
        #   $comboBox       - The Gtk2::ComboBox used alongside the Gtk2::Button
        #   $title          - The title used at the top of the combobox
        #
        # Return values
        #   'undef' on improper arguments
        #   The Gtk2::Button created, otherwise

        my ($self, $table, $row, $iv, $buffer, $comboBox, $title, $check) = @_;

        # Check for improper arguments
        if (
            ! defined $table || ! defined $row || ! defined $iv || ! defined $buffer
            || ! defined $comboBox || ! defined $title || defined $check
        ) {
            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->patternsTags1Tab_addBoldButton',
                @_,
            );
        }

        my $button = $self->addButton(
            $table,
            'Add bold',
            'Add the selected bold colour tag to the list',
            undef,
            11, 12, $row, ($row + 1),
        );
        $button->signal_connect('clicked' => sub {

            my (
                $tag,
                @ivList,
            );

            $tag = $comboBox->get_active_text();

            if ($tag && $tag ne $title) {

                # Update the IV
                @ivList = $self->getEditHash_listIV($iv);

                push (@ivList, uc($tag));
                $self->ivAdd('editHash', $iv, \@ivList);

                # Update the textview
                $buffer->set_text(join("\n", @ivList));
            }
        });

        return $button;
    }

    sub patternsTags1Tab_setSensitive {

        # Called by $self->patternsTags1Tab to check the text in the Gtk2::Entry
        #
        # Expected arguments
        #   $mode       - Matches a setting for $self->editObj->startTagMode, ->stopAtTagMode, etc
        #                   (set to one of 'default', 'no_colour', 'no_style', 'no_colour_style')
        #   $textView   - (Only) textview to be sensitised/desensitised
        #   $listRef, $listRef2, $listRef3, $listRef4, $listRef5
        #               - References to lists of widgets to be sensitised/desensitised
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my (
            $self, $mode, $textView, $listRef, $listRef2, $listRef3, $listRef4, $listRef5,
            $check,
        ) = @_;

        # Check for improper arguments
        if (
            ! defined $mode || ! defined $textView || ! defined $listRef || ! defined $listRef2
            || ! defined $listRef3 || ! defined $listRef4 || ! defined $listRef5 || defined $check
        ) {
            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->patternsTags1Tab_setSensitive',
                @_,
            );
        }

        # Textview
        if ($mode ne 'no_colour_style') {
            $textView->set_sensitive(TRUE);
        } else {
            $textView->set_sensitive(FALSE);
        }

        # First-fourth row
        if ($mode eq 'default' || $mode eq 'no_style') {

            foreach my $widget (@$listRef, @$listRef2, @$listRef3, @$listRef4) {

                $widget->set_sensitive(TRUE);
            }

        } else {

            foreach my $widget (@$listRef, @$listRef2, @$listRef3, @$listRef4) {

                $widget->set_sensitive(FALSE);
            }
        }

        # Fifth row
        if ($mode eq 'default' || $mode eq 'no_colour') {

            foreach my $widget (@$listRef5) {

                $widget->set_sensitive(TRUE);
            }

        } else {

            foreach my $widget (@$listRef5) {

                $widget->set_sensitive(FALSE);
            }
        }

        return 1;
    }

    sub patternsTags7Tab {

        # PatternsTags7 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (@comboList, @comboList2, @comboList3);

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->patternsTags7Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _7', $innerNotebook);

        # Stop at patterns (2)
        $self->addLabel($table, '<b>Stop at patterns (2)</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>Stop at the first line which DOES match one of these patterns</i>',
            1, 12, 1, 2);

        $self->addTextView($table, 'stopAtPatternList', TRUE,
            1, 12, 2, 6,
            TRUE, TRUE, FALSE, FALSE,  # Treat as list, remove empty lines, don't remove whitespace
            -1, 130);                  # Fixed height

        # Stop at tags (2)
        $self->addLabel($table, '<b>Stop at tags (2)</b>',
            0, 12, 6, 7);
        $self->addLabel($table,
            '<i>Stop at the first line which DOES contain one of these ' . $axmud::SCRIPT
            . ' colour/style tags</i>',
            1, 12, 7, 8);

        my $textView = $self->addTextView($table, 'stopAtTagList', TRUE,
            1, 6, 8, 12,
            undef, undef, undef, undef,
            -1, 130);       # Fixed height
        my $buffer = $textView->get_buffer();

        @comboList = $axmud::CLIENT->constColourTagList;
        @comboList3 = $axmud::CLIENT->constStyleTagList;
        foreach my $item (@comboList) {

            push (@comboList2, 'ul_' . $item);
        }

        # Add editing combos / buttons
        my $comboBox = $self->addComboBox($table, undef, \@comboList, 'Colour tags',
            TRUE,               # No 'undef' value used
            7, 11, 8, 9);
        my $button = $self->patternsTags1Tab_addButton(
            $table,
            11, 8,
            'stopAtTagList',
            $buffer,
            $comboBox,
            'Colour tags:',
        );

        my $comboBox2 = $self->addComboBox($table, undef, \@comboList2, 'Underlay tags',
            TRUE,               # No 'undef' value used
            7, 11, 9, 10);
        my $button2 = $self->patternsTags1Tab_addButton(
            $table,
            11, 9,
            'stopAtTagList',
            $buffer,
            $comboBox2,
            'Underlay tags:',
        );

        my $comboBox3 = $self->addComboBox($table, undef, \@comboList3, 'Style tags',
            TRUE,               # No 'undef' value used
            7, 11, 10, 11);
        my $button3 = $self->patternsTags1Tab_addButton(
            $table,
            11, 10,
            'stopAtTagList',
            $buffer,
            $comboBox3,
            'Style tags:',
        );

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub patternsTags8Tab {

        # PatternsTags8 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->patternsTags8Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _8', $innerNotebook);

        # Stop at capitalised lines
        $self->addLabel($table, '<b>Stop at capitalised lines</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            'Stop at the n-th line which starts with a capital letter (0 - ignore)',
            1, 8, 1, 2);
        $self->addEntryWithIcon($table, 'upperCount', 'int', 0, 16,
            8, 12, 1, 2);

        $self->addLabel($table,
            'Stop at the n-th line which starts with a lower-case letter or number (0 - ignore)',
            1, 8, 2, 3);
        $self->addEntryWithIcon($table, 'otherCount', 'int', 0, 16,
            8, 12, 2, 3);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub patternsTags9Tab {

        # PatternsTags9 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->patternsTags9Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _9', $innerNotebook);

        # Stop before mode
        $self->addLabel($table, '<b>Stop before mode</b>',
            0, 12, 0, 1);

        my ($group, $radioButton) = $self->addRadioButton(
            $table, undef, '\'default\'', 'stopBeforeMode',
            'default',      # IV set to this value when toggled
            TRUE,           # Sensitive widget
            1, 4, 1, 2);
        $self->addLabel($table, 'Don\'t use \'Stop before mode\'',
            4, 12, 1, 2);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'no_char\'', 'stopBeforeMode', 'no_char', TRUE,
            1, 4, 2, 3);
        $self->addLabel(
            $table, 'Stop one line before the first line containing no characters at all',
            4, 12, 2, 3);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'no_letter_num\'', 'stopBeforeMode', 'no_letter_num', TRUE,
            1, 4, 3, 4);
        $self->addLabel(
            $table, 'Stop one line before the first line containing no alphanumeric characters',
            4, 12, 3, 4);

        ($group, $radioButton) = $self->addRadioButton(
            $table,
            $group,
            '\'no_start_letter_num\'',
            'stopBeforeMode',
            'no_start_letter_num',
            TRUE,
            1, 4, 4, 5);
        $self->addLabel(
            $table,
            'Stop one line before the first line which doesn\'t start with an alphanumeric'
            . ' character',
            4, 12, 4, 5);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'no_tag\'', 'stopBeforeMode', 'no_tag', TRUE,
            1, 4, 5, 6);
        $self->addLabel(
            $table,
            'Stop one line before the first line containing no colour/style tags at all',
            4, 12, 5, 6);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'has_letter_num\'', 'stopBeforeMode', 'has_letter_num', TRUE,
            1, 4, 6, 7);
        $self->addLabel(
            $table,
            'Stop one line before the first line which DOES contain alphanumeric characters',
            4, 12, 6, 7);

        ($group, $radioButton) = $self->addRadioButton(
            $table,
            $group,
            '\'has_start_letter_num\'',
            'stopBeforeMode',
            'has_start_letter_num',
            TRUE,
            1, 4, 7, 8);
        $self->addLabel(
            $table,
            'Stop one line before the first line which DOES start with an alphanumeric character',
            4, 12, 7, 8);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'has_tag\'', 'stopBeforeMode', 'has_tag', TRUE,
            1, 4, 8, 9);
        $self->addLabel(
            $table,
            'Stop one line before the first line which DOES contain colour/style tags',
            4, 12, 8, 9);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub patternsTags10Tab {

        # PatternsTags10 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->patternsTags10Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page 1_0', $innerNotebook);

        # Stop at mode
        $self->addLabel($table, '<b>Stop at mode</b>',
            0, 12, 0, 1);

        my ($group, $radioButton) = $self->addRadioButton(
            $table, undef, '\'default\'', 'stopAtMode',
            'default',      # IV set to this value when toggled
            TRUE,           # Sensitive widget
            1, 4, 1, 2);
        $self->addLabel($table, 'Don\'t use \'Stop at mode\'',
            4, 12, 1, 2);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'no_char\'', 'stopAtMode', 'no_char', TRUE,
            1, 4, 2, 3);
        $self->addLabel(
            $table, 'Stop at the first line containing no characters at all',
            4, 12, 2, 3);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'no_letter_num\'', 'stopAtMode', 'no_letter_num', TRUE,
            1, 4, 3, 4);
        $self->addLabel(
            $table, 'Stop at the first line containing no alphanumeric characters',
            4, 12, 3, 4);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'no_start_letter_num\'', 'stopAtMode', 'no_start_letter_num', TRUE,
            1, 4, 4, 5);
        $self->addLabel(
            $table, 'Stop at the first line which doesn\'t start with an alphanumeric character',
            4, 12, 4, 5);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'no_tag\'', 'stopAtMode', 'no_tag', TRUE,
            1, 4, 5, 6);
        $self->addLabel(
            $table,
            'Stop at the first line containing no colour/style tags at all',
            4, 12, 5, 6);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'has_letter_num\'', 'stopAtMode', 'has_letter_num', TRUE,
            1, 4, 6, 7);
        $self->addLabel(
            $table, 'Stop at the first line which DOES contain alphanumeric characters',
            4, 12, 6, 7);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'has_start_letter_num\'', 'stopAtMode', 'has_start_letter_num', TRUE,
            1, 4, 7, 8);
        $self->addLabel(
            $table, 'Stop at the first line which DOES start with an alphanumeric character',
            4, 12, 7, 8);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'has_tag\'', 'stopAtMode', 'has_tag', TRUE,
            1, 4, 8, 9);
        $self->addLabel(
            $table,
            'Stop at the first line which DOES contain colour/style tags',
            4, 12, 8, 9);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::Dict;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::Generic::EditWin Games::Axmud::Generic::ConfigWin
        Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::Obj::Dict')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

    sub setupNotebook {

        # Called by $self->winEnable
        # Creates the first tab for the notebook. The remaining tabs are created by
        #   $self->expandNotebook, once the user has specified the dictionary's name
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setupNotebook', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('_Name', $self->notebook);

        # Dictionary name
        $self->addLabel($table, '<b>Dictionary name</b>',
            0, 6, 0, 1);

        # Explanatory label - contents to be set to 'Choose a name and press enter', in a moment
        my $label = $self->addLabel($table, '',
            1, 6, 2, 3);

        # Entry box for the dictionary name
        my $entry;

        if ($self->tempFlag) {

            # $self->editObj is a temporary dictionary, so the entry box must be sensitive
            $entry = $self->addEntry($table, undef, TRUE,
                1, 6, 1, 2, 16, 16);

            $entry->signal_connect (activate => sub {

                my $text = $entry->get_text();

                # Check that the name is acceptable
                if ($axmud::CLIENT->ivExists('dictHash', $text)) {

                    $label->set_markup('<i>Dictionary named \'' . $text . '\' already exists</i>');
                    $self->winShowAll($self->_objClass . '->setupNotebook');

                } elsif (! $axmud::CLIENT->nameCheck($text, 16)) {

                    $label->set_markup('<i>Max 16 chars: A-Z a-z _ 0-9</i>');
                    $self->winShowAll($self->_objClass . '->setupNotebook');

                } else {

                    # Replace the explanatory label
                    $label->set_markup('<i>You may now edit the dictionary</i>');
                    # Desensitise the entry box - once the dictionary name is entered, it's final
                    $entry->set_sensitive(FALSE);
                    # Make sure the OK/reset/save buttons are now sensitised
                    $self->okButton->set_sensitive(TRUE);
                    $self->resetButton->set_sensitive(TRUE);
                    $self->saveButton->set_sensitive(TRUE);

                    # Store the name until all IVs are copied to the profile
                    $self->ivAdd('editHash', 'name', $text);

                    # Set up the remaining tabs
                    $self->expandNotebook();

                    # Render the changes
                    $self->winShowAll($self->_objClass . '->setupNotebook');
                }
            });

            $entry->signal_connect('changed' => sub {

                my ($value, $result);

                $value = $entry->get_text();
                # Check whether $text is a valid value, or not
                if ($axmud::CLIENT->nameCheck($value, 16)) {
                    $entry->set_icon_from_stock('secondary', 'gtk-yes');
                } else {
                    $entry->set_icon_from_stock('secondary', 'gtk-no');
                }
            });

            $label->set_markup('<i>Choose a name and press \'enter\'</i>');
            # (Entry is initially empty)
            $entry->set_icon_from_stock('secondary', 'gtk-no');

        } else {

            # $self->editObj is an existing dictionary, so the entry box must be insensitive
            $entry = $self->addEntry($table, 'name', FALSE,
                1, 6, 1, 2, 16, 16);

            $label->set_markup('<i>(The dictionary name can\'t be changed)</i>');

            # Set up the remaining tabs
            $self->expandNotebook();
        }

        # Set up the rest of the tab
        $self->nameTab($table);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        # If this is a current dictionary, set a flag
        if ($self->session->currentDict eq $self->editObj) {

            $self->ivPoke('currentFlag', TRUE);
        }

        # Expand the notebook
        $self->articlesTab();
        $self->conjunctionsTab();
        $self->numbersTab();
        $self->timeTab();
        $self->nounsTab();
        $self->pluralsTab();
        $self->adjectivesTab();
        $self->pseudoTab();
        $self->contentsTab();
        $self->directionsTab();

        return 1;
    }

    sub saveChanges {

        # Called by $self->buttonOK and $self->buttonSave
        # Saves the changes made to the dictionary. If it was a temporary dictionary, create a real
        #   dictionary, and save the changes to the latter
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->saveChanges', @_);
        }

        if ($self->editHash) {

            # For a temporary dictionary...
            if ($self->tempFlag) {

                my ($dictName, $result, $dictObj, $number);

                $dictName = $self->ivShow('editHash', 'name');

                # The user made changes, so we can now create a permanent dictionary and give it IVs
                #   specified in $self->editHash
                $result = $self->session->pseudoCmd(
                    'adddictionary ' . $dictName,
                    $self->pseudoCmdMode,
                );

                if ($result) {

                    # Get the blessed reference of the new dictionary
                    $dictObj = $axmud::CLIENT->ivShow('dictHash', $dictName);

                    # Store the changes the user has made (ignore the 'name' IV, which has aleady
                    #   been set)
                    $self->ivDelete('editHash', 'name');

                    foreach my $key ($self->ivKeys('editHash')) {

                        $dictObj->{$key} = $self->ivShow('editHash', $key);
                    }

                    # If the 'save' button was clicked, the window won't be closed. Before
                    #   re-drawing all those tabs, set the object being edited to the one we just
                    #   created
                    $self->ivPoke('editObj', $dictObj);
                    $self->ivPoke('tempFlag', FALSE);
                }

            # For an existing dictionary...
            } else {

                # Store the changes the user has made
                foreach my $key ($self->ivKeys('editHash')) {

                    $self->editObj->{$key} = $self->ivShow('editHash', $key);
                }

                # Update the dictionary's combined hashes. The FALSE argument means that the
                #   dictionary's parent file object doesn't need to have its ->modifyFlag set
                #   (because we're about to do it anyway)
                $self->editObj->createCombHashes($self->session, FALSE);
            }

            # The changes can now be cleared
            $self->ivEmpty('editHash');

            # Mark the object's corresponding file object as needing to be saved, if it exists
            if ($self->editObj->_parentFile) {

                $self->editObj->doModify('saveChanges');
            }

            # Update the current session's object viewer window, if it is open
            if ($self->session->viewerWin) {

                $self->session->viewerWin->updateNotebook();
            }
        }

        return 1;
    }

    # Notebook tabs

    sub nameTab {

        # Name tab - called by $self->setupNotebook
        #
        # Expected arguments
        #   $table  -  The Gtk2::Table which has already been created for this tab
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $table, $check) = @_;

        # Local variables
        my (
            @pbList, @comboList,
            %comboHash,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->nameTab', @_);
        }

#       # Tab setup (already created by the calling function)
#       my ($vBox, $table) = $self->addTab('_Name', $self->notebook);

        # Current dictionary
        $self->addLabel($table, '<b>Current dictionary</b>',
            7, 12, 0, 1);

        my $checkbutton = Gtk2::CheckButton->new();
        if ($self->session->currentDict eq $self->editObj) {

            $checkbutton->set_active(TRUE);
        }
        $checkbutton->set_sensitive(FALSE);
        $table->attach_defaults($checkbutton, 8, 9, 1, 2);

        # Language
        $self->addLabel($table, '<b>Language</b>',
            0, 6, 3, 4);
        $self->addEntryWithIcon($table, 'language', 'string', 1, undef,
            1, 6, 4, 5, 16, 16);

        @pbList = sort {lc($a->targetName) cmp lc($b->targetName)}
            ($axmud::CLIENT->ivValues('constPhrasebookHash'));

        foreach my $pbObj (@pbList) {

            push (@comboList, $pbObj->targetName);
            $comboHash{$pbObj->targetName} = $pbObj;
        }

        my $combo = $self->addComboBox($table, undef, \@comboList, '',
            TRUE,               # No 'undef' value used
            1, 4, 5, 6);
        my $button = $self->addButton(
            $table,
            'Switch language',
            'Switch language and update some dictionary data',
            undef,
            4, 6, 5, 6,
            TRUE);          # Irreversible
        $button->signal_connect('clicked' => sub {

            my ($targetName, $pbObj, $choice);

            $targetName = $combo->get_active_text();
            $pbObj = $comboHash{$targetName};

            # Prompt user
            $choice = $self->showMsgDialogue(
                'Switch language',
                'question',
                "Are you sure you want to switch to " . ucfirst($pbObj->name) . "?\n\n(This action"
                . " will save any changes you\'ve already made, and then replace the dictionary\'s"
                . " primary directions, articles, conjunctions and some number words)",
                'yes-no',
            );

            if (defined $choice && $choice eq 'yes') {

                # Update the dictionary saving any changes made in this 'edit' window
                $self->saveChanges();

                # Perform the operation
                $self->editObj->uploadPhrasebook($pbObj);

                # Redraw the edit window
                $self->buttonReset();
            }
        });

        # Word order
        $self->addLabel($table, '<b>Word order</b>',
            7, 12, 3, 4);

        my ($group, $radioButton) = $self->addRadioButton(
            $table, undef, '\'noun_adj\'', 'nounPosn',
            'noun_adj',     # IV set to this value when toggled
            TRUE,           # Sensitive widget
            8, 10, 4, 5);
        $self->addLabel($table, 'Noun-adjective (e.g. French, Spanish)',
            10, 12, 4, 5);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'adj_noun\'', 'nounPosn', 'adj_noun', TRUE,
            8, 10, 5, 6);
        $self->addLabel($table, 'Adjective-noun (e.g. English, German)',
            10, 12, 5, 6);

#       # Tab complete (handled by the calling function)
#       $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub articlesTab {

        # Articles tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my @list;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->articlesTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('_Articles', $self->notebook);

        # Definite articles
        $self->addLabel($table, '<b>Definite articles</b>',
            0, 6, 0, 2);
        $self->addLabel($table, '<i>Words for \'the\' in the dictionary\'s language</i>',
            1, 6, 2, 4);
        $self->addTextView($table, 'definiteList', TRUE,
            1, 6, 4, 6);

        # Indefinite articles
        $self->addLabel($table, '<b>Indefinite articles</b>',
            0, 6, 6, 8);
        $self->addLabel($table, '<i>Words for \'a\' / \'an\' in the dictionary\'s language</i>',
            1, 6, 8, 10);
        $self->addTextView($table, 'indefiniteList', TRUE,
            1, 6, 10, 12);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub conjunctionsTab {

        # Conjunctions tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my @list;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->conjunctionsTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('_Conjunctions', $self->notebook);

        # Conjunctions
        $self->addLabel($table, '<b>Conjunctions</b>',
            0, 6, 0, 2);
        $self->addLabel($table, '<i>Words for \'and\' in the dictionary\'s language</i>',
            1, 6, 2, 4);
        $self->addTextView($table, 'andList', TRUE,
            1, 6, 4, 6);

        $self->addLabel($table, '<i>Words for \'or\' in the dictionary\'s language</i>',
            1, 6, 8, 10);
        $self->addTextView($table, 'orList', TRUE,
            1, 6, 10, 12);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub numbersTab {

        # Numbers tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->numbersTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('N_umbers', $self->notebook);

        # Number terms
        $self->addLabel($table, '<b>Number terms</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of terms for ordinal numbers, fractions and indeterminate numbers</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Term', 'text',
            'Number', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 260);      # Fixed height

        # Initialise the list
        $self->numbersTab_refreshList($slWidget, (scalar @columnList / 2), 'numberHash');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Term',
            1, 4, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            4, 7, 8, 9);

        $self->addLabel($table, 'Number (indeterminate = -1):',
            7, 10, 8, 9);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'float', 0, undef,
            10, 12, 8, 9);

        # Add buttons that resemble those from GA::Generic::EditWin->addSimpleListButtons_hashIV
        my $button = $self->addButton($table, 'Add term', 'Add a new number term', undef,
            1, 3, 9, 10);
        $button->signal_connect('clicked' => sub {

            my ($term, $value);

            if ($self->checkEntryIcon($entry, $entry2)) {

                $term = $entry->get_text();
                $value = $entry2->get_text();

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('numberHash', $term, $value);

                # Refresh the simple list and reset entry boxes
                $self->numbersTab_refreshList($slWidget, (scalar @columnList / 2), 'numberHash');
                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        my $button2 = $self->addButton($table, 'Delete term', 'Delete the selected term', undef,
            6, 8, 9, 10);
        $button2->signal_connect('clicked' => sub {

            my ($term) = $self->getSimpleListData($slWidget, 0);
            if (defined $term) {

                # Delete the key-value pair from the hash
                $self->modifyEditHash_hashIV('numberHash', $term, undef, TRUE);

                # Refresh the simple list and reset entry boxes
                $self->numbersTab_refreshList($slWidget, (scalar @columnList / 2), 'numberHash');
                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        my $button3 = $self->addButton($table,
            'Reset', 'Reset the hash which stores the terms', undef,
            8, 10, 9, 10);
        $button3->signal_connect('clicked' => sub {

            # Remove this IV from $self->editHash, so that the IV in $self->editObj takes over
            $self->ivDelete('editHash', 'numberHash');

            # Refresh the simple list and reset entry boxes
            $self->numbersTab_refreshList($slWidget, (scalar @columnList / 2), 'numberHash');
            $self->resetEntryBoxes($entry, $entry2);
        });

        my $button4 = $self->addButton($table,
            'Clear', 'Clear the hash which stores the terms', undef,
            10, 12, 9, 10);
        $button4->signal_connect('clicked' => sub {

            # Add an empty hash to $self->editHash
            $self->ivAdd('editHash', 'numberHash', {});

            # Refresh the simple list and reset entry boxes
            $self->numbersTab_refreshList($slWidget, (scalar @columnList / 2), 'numberHash');
            $self->resetEntryBoxes($entry, $entry2);
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub numbersTab_refreshList {

        # Called by $self->numbersTab to refresh the GA::Obj::Simple::List, ordering the data by
        #   each key-value pair's value (instead of ordering it by key, as usual)
        # Also called by $self->clock1Tab, ->clock2Tab and ->clock3Tab to do the same job
        #
        # Expected arguments
        #   $slWidget   - The GA::Obj::Simple::List
        #   $columns    - The number of columns in the list
        #   $iv         - The IV being edited
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $slWidget, $columns, $iv, $check) = @_;

        # Local variables
        my (
            @sortedList, @dataList,
            %ivHash,
        );

        # Check for improper arguments
        if (! defined $slWidget || ! defined $columns || ! defined $iv || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->numbersTab_refreshList', @_);
        }

        # Import the IV
        %ivHash = $self->getEditHash_hashIV($iv);

        # Get a list of the keys in %ivHash, sorted in order of each key's value
        @sortedList = sort {

            if ($ivHash{$a} == $ivHash{$b}) {

                # Values are equal, so sort by key
                return (lc($a) cmp lc($b));

            } else {

                # Sort by value
                return $ivHash{$a} <=> $ivHash{$b};
            }

        } (keys %ivHash);

        # Compile the simple list data
        foreach my $key (@sortedList) {

            push (@dataList,
                $key,
                $ivHash{$key},
            );
        }

        # Reset the simple list
        $self->resetListData($slWidget, [@dataList], $columns);

        return 1;
    }

    sub timeTab {

        # Clock tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->clockTab', @_);
        }

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('_Time', $self->notebook);

        # Add tabs to the inner notebook
        $self->time1Tab($innerNotebook);
        $self->time2Tab($innerNotebook);
        $self->time3Tab($innerNotebook);
        $self->time4Tab($innerNotebook);

        return 1;
    }

    sub time1Tab {

        # Time1 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (%timeHash, %timePluralHash);

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->time1Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _1', $innerNotebook);


        # Make a local copy of the dictionary's time hashes
        %timeHash = $self->getEditHash_hashIV('timeHash');
        %timePluralHash = $self->getEditHash_hashIV('timePluralHash');

        # Time units
        $self->addLabel($table, '<b>Time units</b>',
            0, 4, 0, 1);
        $self->addLabel($table, '<i>Singular</i>',
            4, 8, 0, 1);
        $self->addLabel($table, '<i>Plural</i>',
            8, 12, 0, 1);

        $self->addLabel($table, 'Seconds',
            1, 4, 1, 2);
        $self->time1Tab_addEntryWithIcon($table, 'timeHash', 'second',
            4, 8, 1, 2);
        $self->time1Tab_addEntryWithIcon($table, 'timePluralHash', 'second',
            8, 12, 1, 2);

        $self->addLabel($table, 'Minutes',
            1, 4, 2, 3);
        $self->time1Tab_addEntryWithIcon($table, 'timeHash', 'minute',
            4, 8, 2, 3);
        $self->time1Tab_addEntryWithIcon($table, 'timePluralHash', 'minute',
            8, 12, 2, 3);

        $self->addLabel($table, 'Hours',
            1, 4, 3, 4);
        $self->time1Tab_addEntryWithIcon($table, 'timeHash', 'hour',
            4, 8, 3, 4);
        $self->time1Tab_addEntryWithIcon($table, 'timePluralHash', 'hour',
            8, 12, 3, 4);

        $self->addLabel($table, 'Days',
            1, 4, 4, 5);
        $self->time1Tab_addEntryWithIcon($table, 'timeHash', 'day',
            4, 8, 4, 5);
        $self->time1Tab_addEntryWithIcon($table, 'timePluralHash', 'day',
            8, 12, 4, 5);

        $self->addLabel($table, 'Weeks',
            1, 4, 5, 6);
        $self->time1Tab_addEntryWithIcon($table, 'timeHash', 'week',
            4, 8, 5, 6);
        $self->time1Tab_addEntryWithIcon($table, 'timePluralHash', 'week',
            8, 12, 5, 6);

        $self->addLabel($table, 'Months',
            1, 4, 6, 7);
        $self->time1Tab_addEntryWithIcon($table, 'timeHash', 'month',
            4, 8, 6, 7);
        $self->time1Tab_addEntryWithIcon($table, 'timePluralHash', 'month',
            8, 12, 6, 7);

        $self->addLabel($table, 'Years',
            1, 4, 7, 8);
        $self->time1Tab_addEntryWithIcon($table, 'timeHash', 'year',
            4, 8, 7, 8);
        $self->time1Tab_addEntryWithIcon($table, 'timePluralHash', 'year',
            8, 12, 7, 8);

        $self->addLabel($table, 'Decades',
            1, 4, 8, 9);
        $self->time1Tab_addEntryWithIcon($table, 'timeHash', 'decade',
            4, 8, 8, 9);
        $self->time1Tab_addEntryWithIcon($table, 'timePluralHash', 'decade',
            8, 12, 8, 9);

        $self->addLabel($table, 'Centuries',
            1, 4, 9, 10);
        $self->time1Tab_addEntryWithIcon($table, 'timeHash', 'century',
            4, 8, 9, 10);
        $self->time1Tab_addEntryWithIcon($table, 'timePluralHash', 'century',
            8, 12, 9, 10);

        $self->addLabel($table, 'Millennia',
            1, 4, 10, 11);
        $self->time1Tab_addEntryWithIcon($table, 'timeHash', 'millennium',
            4, 8, 10, 11);
        $self->time1Tab_addEntryWithIcon($table, 'timePluralHash', 'millennium',
            8, 12, 10, 11);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub time1Tab_addEntryWithIcon {

        # Adds a Gtk2::Entry at the specified position on the tab, containing a stock icon to show
        #   whether the current contents of the entry is allowed
        # The stock icons used are 'gtk-yes' (for an acceptable value) and 'gtk-no' (for a forbidden
        #   value)
        # This is a customised version of the ->addEntryWithIcon method found in
        #   GA::Generic::EditWin
        #
        # Expected arguments
        #   $table      - The Gtk2::Table object in which the entry boxes are placed
        #   $iv         - A string naming the hash IV set (either 'timeHash' or 'timePluralHash')
        #   $key        - The key being set in the hash IV
        #   $leftAttach, $rightAttach, $topAttach, $bottomAttach
        #               - The position of the entry box in the table
        #
        # Return values
        #   'undef' on improper arguments
        #   Otherwise returns the Gtk2::Entry created

        my (
            $self, $table, $iv, $key, $leftAttach, $rightAttach, $topAttach, $bottomAttach, $check
        ) = @_;

        # Local variables
        my %ivHash;

        # Check for improper arguments
        if (
            ! defined $table || ! defined $iv || ! defined $key || ! defined $leftAttach
            || ! defined $rightAttach || ! defined $topAttach || ! defined $bottomAttach
            || defined $check
        ) {
            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->time1Tab_addEntryWithIcon',
                @_,
            );
        }

        # Import the hash being edited
        %ivHash = $self->getEditHash_hashIV($iv);

        # Create the entry
        my $entry = Gtk2::Entry->new();

        # Display the existing value of the IV
        if (defined $self->editObj->$iv) {

            $entry->append_text($ivHash{$key});

            if ($ivHash{$key}) {
                $entry->set_icon_from_stock('secondary', 'gtk-yes');
            } else {
                $entry->set_icon_from_stock('secondary', 'gtk-no');
            }

        } else {

            # We need to set the icon for an empty box
            $entry->set_icon_from_stock('secondary', 'gtk-no');
        }

        # Customise the entry
        $entry->signal_connect('changed' => sub {

            my $value = $entry->get_text();

            $self->modifyEditHash_hashIV($iv, $key, $value);

            if ($value) {
                $entry->set_icon_from_stock('secondary', 'gtk-yes');
            } else {
                $entry->set_icon_from_stock('secondary', 'gtk-no');
            }
        });

        # Add the entry to the table
        $table->attach_defaults($entry, $leftAttach, $rightAttach, $topAttach, $bottomAttach);

        return $entry;
    }

    sub time2Tab {

        # Time2 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (@columnList, @comboList);

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->time2Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _2', $innerNotebook);

        # Clock patterns - time of day
        $self->addLabel($table, '<b>Clock patterns - time of day</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>Use a value of 0 for a.m., 1 for p.m.</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Pattern', 'text',
            'Value', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 220);      # Fixed height

        # Initialise the list
        $self->numbersTab_refreshList($slWidget, (scalar @columnList / 2), 'clockDayHash');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Pattern:',
            1, 4, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'regex', 1, undef,
            4, 7, 8, 9);

        $self->addLabel($table, 'Value:',
            7, 10, 8, 9);
        @comboList = (0, 1);
        my $combo = $self->addComboBox($table, undef, \@comboList, '',
            TRUE,               # No 'undef' value used
            10, 12, 8, 9);

        # Add buttons that resemble those from GA::Generic::EditWin->addSimpleListButtons_hashIV
        my $button = $self->addButton($table, 'Add pattern', 'Add a new clock pattern', undef,
            1, 3, 9, 10);
        $button->signal_connect('clicked' => sub {

            my ($pattern, $value);

            if ($self->checkEntryIcon($entry)) {

                $pattern = $entry->get_text();
                $value = $combo->get_active_text();

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('clockDayHash', $pattern, $value);

                # Refresh the simple list and reset entry boxes
                $self->numbersTab_refreshList(
                    $slWidget,
                    (scalar @columnList / 2),
                    'clockDayHash',
                );

                $self->resetEntryBoxes($entry);
            }
        });

        my $button2 = $self->addButton($table,
            'Delete pattern', 'Delete the selected clock pattern', undef,
            6, 8, 9, 10);
        $button2->signal_connect('clicked' => sub {

            my ($term) = $self->getSimpleListData($slWidget, 0);
            if (defined $term) {

                # Delete the key-value pair from the hash
                $self->modifyEditHash_hashIV('clockDayHash', $term, undef, TRUE);

                # Refresh the simple list and reset entry boxes
                $self->numbersTab_refreshList(
                    $slWidget,
                    (scalar @columnList / 2),
                    'clockDayHash',
                );

                $self->resetEntryBoxes($entry);
            }
        });

        my $button3 = $self->addButton($table,
            'Reset', 'Reset the hash which stores the clock patterns', undef,
            8, 10, 9, 10);
        $button3->signal_connect('clicked' => sub {

            # Remove this IV from $self->editHash, so that the IV in $self->editObj takes over
            $self->ivDelete('editHash', 'clockDayHash');

            # Refresh the simple list and reset entry boxes
            $self->numbersTab_refreshList($slWidget, (scalar @columnList / 2), 'clockDayHash');
            $self->resetEntryBoxes($entry);
        });

        my $button4 = $self->addButton($table,
            'Clear', 'Clear the hash which stores the clock patterns', undef,
            10, 12, 9, 10);
        $button4->signal_connect('clicked' => sub {

            # Add an empty hash to $self->editHash
            $self->ivAdd('editHash', 'clockDayHash', {});

            # Refresh the simple list and reset entry boxes
            $self->numbersTab_refreshList($slWidget, (scalar @columnList / 2), 'clockDayHash');
            $self->resetEntryBoxes($entry);
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub time3Tab {

        # Time3 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->time3Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _3', $innerNotebook);

        # Clock patterns - hours
        $self->addLabel($table, '<b>Clock patterns - hours</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>Use a value of 1-12, but use 0 for midnight</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Pattern', 'text',
            'Value', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8, -1, 220);      # Fixed height

        # Initialise the list
        $self->numbersTab_refreshList($slWidget, (scalar @columnList / 2), 'clockHourHash');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Pattern',
            1, 4, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'regex', 1, undef,
            4, 7, 8, 9);

        $self->addLabel($table, 'Value',
            7, 10, 8, 9);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'float', 0, undef,
            10, 12, 8, 9);

        # Add buttons that resemble those from GA::Generic::EditWin->addSimpleListButtons_hashIV
        my $button = $self->addButton($table, 'Add pattern', 'Add a new clock pattern', undef,
            1, 3, 9, 10);
        $button->signal_connect('clicked' => sub {

            my ($term, $value);

            if ($self->checkEntryIcon($entry, $entry2)) {

                $term = $entry->get_text();
                $value = $entry2->get_text();

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('clockHourHash', $term, $value);

                # Refresh the simple list and reset entry boxes
                $self->numbersTab_refreshList(
                    $slWidget,
                    (scalar @columnList / 2),
                    'clockHourHash',
                );

                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        my $button2 = $self->addButton($table,
            'Delete pattern', 'Delete the selected clock pattern', undef,
            6, 8, 9, 10);
        $button2->signal_connect('clicked' => sub {

            my ($term) = $self->getSimpleListData($slWidget, 0);
            if (defined $term) {

                # Delete the key-value pair from the hash
                $self->modifyEditHash_hashIV('clockHourHash', $term, undef, TRUE);

                # Refresh the simple list and reset entry boxes
                $self->numbersTab_refreshList(
                    $slWidget,
                    (scalar @columnList / 2),
                    'clockHourHash',
                );

                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        my $button3 = $self->addButton($table,
            'Reset', 'Reset the hash which stores the clock patterns', undef,
            8, 10, 9, 10);
        $button3->signal_connect('clicked' => sub {

            # Remove this IV from $self->editHash, so that the IV in $self->editObj takes over
            $self->ivDelete('editHash', 'clockHourHash');

            # Refresh the simple list and reset entry boxes
            $self->numbersTab_refreshList($slWidget, (scalar @columnList / 2), 'clockHourHash');
            $self->resetEntryBoxes($entry, $entry2);
        });

        my $button4 = $self->addButton($table,
            'Clear', 'Clear the hash which stores the clock patterns', undef,
            10, 12, 9, 10);
        $button4->signal_connect('clicked' => sub {

            # Add an empty hash to $self->editHash
            $self->ivAdd('editHash', 'clockHourHash', {});

            # Refresh the simple list and reset entry boxes
            $self->numbersTab_refreshList($slWidget, (scalar @columnList / 2), 'clockHourHash');
            $self->resetEntryBoxes($entry, $entry2);
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub time4Tab {

        # Time4 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->time4Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _4', $innerNotebook);

        # Clock patterns - minutes
        $self->addLabel($table, '<b>Clock patterns - minutes</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>Use positive values for minutes past, negative for minutes to (0 for hours)</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Pattern', 'text',
            'Value', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8, -1, 220);      # Fixed height

        # Initialise the list
        $self->numbersTab_refreshList($slWidget, (scalar @columnList / 2), 'clockMinuteHash');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Pattern',
            1, 4, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'regex', 1, undef,
            4, 7, 8, 9);

        $self->addLabel($table, 'Value',
            7, 10, 8, 9);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'float', 0, undef,
            10, 12, 8, 9);

        # Add buttons that resemble those created by
        #   GA::Generic::EditWin->addSimpleListButtons_hashIV
        my $button = $self->addButton($table, 'Add pattern', 'Add a new clock pattern', undef,
            1, 3, 9, 10);
        $button->signal_connect('clicked' => sub {

            my ($term, $value);

            if ($self->checkEntryIcon($entry, $entry2)) {

                $term = $entry->get_text();
                $value = $entry2->get_text();

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('clockMinuteHash', $term, $value);

                # Refresh the simple list and reset entry boxes
                $self->numbersTab_refreshList(
                    $slWidget,
                    (scalar @columnList / 2),
                    'clockMinuteHash',
                );

                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        my $button2 = $self->addButton($table,
            'Delete pattern', 'Delete the selected clock pattern', undef,
            6, 8, 9, 10);
        $button2->signal_connect('clicked' => sub {

            my ($term) = $self->getSimpleListData($slWidget, 0);
            if (defined $term) {

                # Delete the key-value pair from the hash
                $self->modifyEditHash_hashIV('clockMinuteHash', $term, undef, TRUE);

                # Refresh the simple list and reset entry boxes
                $self->numbersTab_refreshList(
                    $slWidget,
                    (scalar @columnList / 2),
                    'clockMinuteHash',
                );

                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        my $button3 = $self->addButton($table,
            'Reset', 'Reset the hash which stores the clock patterns', undef,
            8, 10, 9, 10);
        $button3->signal_connect('clicked' => sub {

            # Remove this IV from $self->editHash, so that the IV in $self->editObj takes over
            $self->ivDelete('editHash', 'clockMinuteHash');

            # Refresh the simple list and reset entry boxes
            $self->numbersTab_refreshList($slWidget, (scalar @columnList / 2), 'clockMinuteHash');
            $self->resetEntryBoxes($entry, $entry2);
        });

        my $button4 = $self->addButton($table,
            'Clear', 'Clear the hash which stores the clock patterns', undef,
            10, 12, 9, 10);
        $button4->signal_connect('clicked' => sub {

            # Add an empty hash to $self->editHash
            $self->ivAdd('editHash', 'clockMinuteHash', {});

            # Refresh the simple list and reset entry boxes
            $self->numbersTab_refreshList($slWidget, (scalar @columnList / 2), 'clockMinuteHash');
            $self->resetEntryBoxes($entry, $entry2);
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub nounsTab {

        # Nouns tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->nounsTab', @_);
        }

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('N_ouns', $self->notebook);

        # Add tabs to the inner notebook
        $self->nouns1Tab($innerNotebook);
        $self->nouns2Tab($innerNotebook);
        $self->nouns3Tab($innerNotebook);
        $self->nouns4Tab($innerNotebook);
        $self->nouns5Tab($innerNotebook);
        $self->nouns6Tab($innerNotebook);
        $self->nouns7Tab($innerNotebook);
        $self->nouns8Tab($innerNotebook);
        $self->nouns9Tab($innerNotebook);
        $self->nouns10Tab($innerNotebook);

        return 1;
    }

    sub nouns1Tab {

        # Nouns1 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->nouns1Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _1', $innerNotebook);

        # Sentients
        $self->addLabel($table, '<b>Sentients</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>List of words for sentient beings</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Word', 'text',
            'Type', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8, -1, 220);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'sentientHash');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Word',
            1, 4, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            4, 7, 8, 9);

        $self->addLabel($table, 'Type',
            7, 10, 8, 9);
        my $entry2 = $self->addEntry($table, undef, FALSE,
            10, 12, 8, 9);
        $entry2->set_text('sentient');

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'sentientHash',
            9,
            $entry,
        );
        $button->signal_connect('clicked' => sub {

            my ($word, $type);

            $word = $entry->get_text();
            $type = $entry2->get_text();

            if ($self->checkEntryIcon($entry)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('sentientHash', $word, $type);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'sentientHash');
                $self->resetEntryBoxes($entry);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub nouns2Tab {

        # Nouns2 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->nouns2Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _2', $innerNotebook);

        # Creatures
        $self->addLabel($table, '<b>Creatures</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>List of words for non-sentient beings</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Word', 'text',
            'Type', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8, -1, 220);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'creatureHash');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Word',
            1, 4, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            4, 7, 8, 9);

        $self->addLabel($table, 'Type',
            7, 10, 8, 9);
        my $entry2 = $self->addEntry($table, undef, FALSE,
            10, 12, 8, 9);
        $entry2->set_text('creature');

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'creatureHash',
            9,
            $entry,
        );
        $button->signal_connect('clicked' => sub {

            my ($word, $type);

            $word = $entry->get_text();
            $type = $entry2->get_text();

            if ($self->checkEntryIcon($entry)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('creatureHash', $word, $type);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'creatureHash');
                $self->resetEntryBoxes($entry);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub nouns3Tab {

        # Nouns3 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->nouns3Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _3', $innerNotebook);

        # Portable types
        $self->addLabel($table, '<b>Portable types</b>',
            0, 10, 0, 1);
        $self->addLabel($table, '<i>Objects which can normally be picked up</i>',
            1, 10, 1, 2);
        my $textView = $self->addTextView($table, 'portableTypeList', TRUE,
            1, 12, 2, 5,
            TRUE, TRUE, TRUE, FALSE,  # Treat as list, remove empty lines, do remove whitespace
            -1, 130);
        my $buffer = $textView->get_buffer();

        my $button = Gtk2::Button->new('Use default list');
        $button->signal_connect('clicked' => sub {

            my @list = $self->editObj->constPortableTypeList;

            $self->ivAdd('editHash', \@list);
            $buffer->set_text(join("\n", $self->editObj->ivPeek('constPortableTypeList')));
        });
        $self->tooltips->set_tip($button, 'Use the default list of Portable types');
        $table->attach_defaults($button, 10, 12, 0, 2);

        # Decoration types
        $self->addLabel($table, '<b>Decoration types</b>',
            0, 10, 5, 6);
        $self->addLabel($table, '<i>Objects which can\'t normally be picked up</i>',
            1, 10, 6, 7);
        my $textView2 = $self->addTextView($table, 'decorationTypeList', TRUE,
            1, 12, 7, 10,
            TRUE, TRUE, TRUE, FALSE,  # Treat as list, remove empty lines, do remove whitespace
            -1, 130);
        my $buffer2 = $textView2->get_buffer();

        my $button2 = Gtk2::Button->new('Use default list');
        $button2->signal_connect('clicked' => sub {

            my @list = $self->editObj->constDecorationTypeList;

            $self->ivAdd('editHash', \@list);
            $buffer2->set_text(join("\n", $self->editObj->ivPeek('constDecorationTypeList')));
        });
        $self->tooltips->set_tip($button2, 'Use the default list of Decoration types');
        $table->attach_defaults($button2, 10, 12, 5, 7);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub nouns4Tab {

        # Nouns4 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (@columnList, @comboList);

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->nouns4Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page 4', $innerNotebook);

        # Portables
        $self->addLabel($table, '<b>Portables</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of portable words (objects which can usually be picked up)</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Word', 'text',
            'Type', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 220);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'portableTypeHash');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Word',
            1, 4, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            4, 7, 8, 9);

        $self->addLabel($table, 'Type',
            7, 10, 8, 9);

        @comboList = $self->getEditHash_listIV('portableTypeList');
        my $comboBox = $self->addComboBox($table, undef, \@comboList, '',
            TRUE,               # No 'undef' value used
            10, 12, 8, 9);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'portableTypeHash',
            9,
            $entry,
        );
        $button->signal_connect('clicked' => sub {

            my ($word, $type);

            $word = $entry->get_text();
            $type = $comboBox->get_active_text();

            if ($self->checkEntryIcon($entry) && $type) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('portableTypeHash', $word, $type);
                # (We have a second hash to update)
                $self->modifyEditHash_hashIV('portableHash', $word, 'portable');

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV(
                    $slWidget,
                    scalar (@columnList / 2),
                    'portableTypeHash',
                );

                $self->resetEntryBoxes($entry);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub nouns5Tab {

        # Nouns5 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (@columnList, @comboList);

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->nouns5Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _5', $innerNotebook);

        # Decorations
        $self->addLabel($table, '<b>Decorations</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of decoration words (objects which can\'t usually be picked up)</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Word', 'text',
            'Type', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 220);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'decorationTypeHash');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Word',
            1, 4, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            4, 7, 8, 9);

        $self->addLabel($table, 'Type',
            7, 10, 8, 9);

        @comboList = $self->getEditHash_listIV('decorationTypeList');
        my $comboBox = $self->addComboBox($table, undef, \@comboList, '',
            TRUE,               # No 'undef' value used
            10, 12, 8, 9);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'decorationTypeHash',
            9,
            $entry,
        );
        $button->signal_connect('clicked' => sub {

            my ($word, $type);

            $word = $entry->get_text();
            $type = $comboBox->get_active_text();

            if ($self->checkEntryIcon($entry) && $type) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('decorationTypeHash', $word, $type);
                # (We have a second hash to update)
                $self->modifyEditHash_hashIV('decorationHash', $word, 'decoration');

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV(
                    $slWidget,
                    scalar (@columnList / 2),
                    'decorationTypeHash',
                );

                $self->resetEntryBoxes($entry);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub nouns6Tab {

        # Nouns6 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->nouns6Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _6', $innerNotebook);

        # Guilds
        $self->addLabel($table, '<b>Guilds</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>List of guild words</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Word', 'text',
            'Type', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 220);          # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'guildHash');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Word',
            1, 4, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            4, 7, 8, 9);

        $self->addLabel($table, 'Type',
            7, 10, 8, 9);
        my $entry2 = $self->addEntry($table, undef, FALSE,
            10, 12, 8, 9);
        $entry2->set_text('guild');

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'guildHash',
            9,
            $entry,
        );
        $button->signal_connect('clicked' => sub {

            my ($word, $type);

            $word = $entry->get_text();
            $type = $entry2->get_text();

            if ($self->checkEntryIcon($entry)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('guildHash', $word, $type);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'guildHash');
                $self->resetEntryBoxes($entry);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub nouns7Tab {

        # Nouns7 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->nouns7Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _7', $innerNotebook);

        # Races
        $self->addLabel($table, '<b>Races</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>List of race words</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Word', 'text',
            'Type', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 220);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'raceHash');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Word',
            1, 4, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            4, 7, 8, 9);

        $self->addLabel($table, 'Type',
            7, 10, 8, 9);
        my $entry2 = $self->addEntry($table, undef, FALSE,
            10, 12, 8, 9);
        $entry2->set_text('race');

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'raceHash',
            9,
            $entry,
        );
        $button->signal_connect('clicked' => sub {

            my ($word, $type);

            $word = $entry->get_text();
            $type = $entry2->get_text();

            if ($self->checkEntryIcon($entry)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('raceHash', $word, $type);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'raceHash');
                $self->resetEntryBoxes($entry);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub nouns8Tab {

        # Nouns8 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->nouns8Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _8', $innerNotebook);

        # Weapons
        $self->addLabel($table, '<b>Weapons</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>List of weapon words</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Word', 'text',
            'Type', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 220);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'weaponHash');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Word',
            1, 4, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            4, 7, 8, 9);

        $self->addLabel($table, 'Type',
            7, 10, 8, 9);
        my $entry2 = $self->addEntry($table, undef, FALSE,
            10, 12, 8, 9);
        $entry2->set_text('weapon');

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'weaponHash',
            9,
            $entry,
        );
        $button->signal_connect('clicked' => sub {

            my ($word, $type);

            $word = $entry->get_text();
            $type = $entry2->get_text();

            if ($self->checkEntryIcon($entry)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('weaponHash', $word, $type);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'weaponHash');
                $self->resetEntryBoxes($entry);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub nouns9Tab {

        # Nouns9 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->nouns9Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _9', $innerNotebook);

        # Armour
        $self->addLabel($table, '<b>Armour</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>List of armour words</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Word', 'text',
            'Type', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 220);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'armourHash');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Word',
            1, 4, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            4, 7, 8, 9);

        $self->addLabel($table, 'Type',
            7, 10, 8, 9);
        my $entry2 = $self->addEntry($table, undef, FALSE,
            10, 12, 8, 9);
        $entry2->set_text('armour');

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'armourHash',
            9,
            $entry,
        );
        $button->signal_connect('clicked' => sub {

            my ($word, $type);

            $word = $entry->get_text();
            $type = $entry2->get_text();

            if ($self->checkEntryIcon($entry)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('armourHash', $word, $type);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'armourHash');
                $self->resetEntryBoxes($entry);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub nouns10Tab {

        # Nouns10 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->nouns10Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page 1_0', $innerNotebook);

        # Garments
        $self->addLabel($table, '<b>Garments</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>List of garment words</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Word', 'text',
            'Type', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 220);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'garmentHash');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Word',
            1, 4, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            4, 7, 8, 9);

        $self->addLabel($table, 'Type',
            7, 10, 8, 9);
        my $entry2 = $self->addEntry($table, undef, 0,
            10, 12, 8, 9);
        $entry2->set_text('garment');

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'garmentHash',
            9,
            $entry,
        );
        $button->signal_connect('clicked' => sub {

            my ($word, $type);

            $word = $entry->get_text();
            $type = $entry2->get_text();

            if ($self->checkEntryIcon($entry)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('garmentHash', $word, $type);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'garmentHash');
                $self->resetEntryBoxes($entry);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub pluralsTab {

        # Plurals tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->pluralsTab', @_);
        }

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('_Plurals', $self->notebook);

        # Add tabs to the inner notebook
        $self->plurals1Tab($innerNotebook);
        $self->plurals2Tab($innerNotebook);
        $self->plurals3Tab($innerNotebook);

        return 1;
    }

    sub plurals1Tab {

        # Plurals1 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->plurals1Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _1', $innerNotebook);

        # Plural ending patterns
        $self->addLabel($table, '<b>Plural ending patterns</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>Plural patterns and their equivalent singular endings</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Plural pattern', 'text',
            'Singular ending', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 220);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'pluralEndingHash');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Plural pattern',
            1, 4, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'regex', 1, undef,
            4, 7, 8, 9);

        $self->addLabel($table, 'Singular ending',
            7, 10, 8, 9);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'string', 0, undef,
            10, 12, 8, 9);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'pluralEndingHash',
            9,
            $entry, $entry2,
        );
        $button->signal_connect('clicked' => sub {

            my ($pattern, $singular);

            $pattern = $entry->get_text();
            $singular = $entry2->get_text();

            if ($self->checkEntryIcon($entry, $entry2)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('pluralEndingHash', $pattern, $singular);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'pluralEndingHash');
                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub plurals2Tab {

        # Plurals2 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->plurals2Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _2', $innerNotebook);

        # Singular ending patterns
        $self->addLabel($table, '<b>Singular ending patterns</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>Singular patterns and their equivalent plural endings</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Singular pattern', 'text',
            'Plural ending', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 220);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'reversePluralEndingHash');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Singular pattern',
            1, 4, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'regex', 0, undef,
            4, 7, 8, 9);

        $self->addLabel($table, 'Plural ending',
            7, 10, 8, 9);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            10, 12, 8, 9);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'reversePluralEndingHash',
            9,
            $entry, $entry2,
        );
        $button->signal_connect('clicked' => sub {

            my ($pattern, $singular);

            $pattern = $entry->get_text();
            $singular = $entry2->get_text();

            if ($self->checkEntryIcon($entry, $entry2)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('reversePluralEndingHash', $pattern, $singular);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV(
                    $slWidget,
                    scalar (@columnList / 2),
                    'reversePluralEndingHash',
                );

                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub plurals3Tab {

        # Plurals3 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->plurals3Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _3', $innerNotebook);

        # Plural nouns
        $self->addLabel($table, '<b>Plural nouns</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>List of nouns that are exceptions to the usual rules</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Singular noun', 'text',
            'Plural noun', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 13, 2, 8,
            -1, 220);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'pluralNounHash');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Singular noun',
            1, 4, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            4, 7, 8, 9);

        $self->addLabel($table, 'Plural noun',
            7, 10, 8, 9);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            10, 13, 8, 9);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'pluralNounHash',
            9,
            $entry, $entry2,
        );
        $button->signal_connect('clicked' => sub {

            my ($singular, $plural);

            $singular = $entry->get_text();
            $plural = $entry2->get_text();

            if ($self->checkEntryIcon($entry, $entry2)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('pluralNounHash', $singular, $plural);
                # (We have a second hash to update)
                $self->modifyEditHash_hashIV('reversePluralNounHash', $plural, $singular);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'pluralNounHash');
                $self->resetEntryBoxes($entry);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub adjectivesTab {

        # Adjectives tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->adjectivesTab', @_);
        }

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('Adj_ectives', $self->notebook);

        # Add tabs to the inner notebook
        $self->adjectives1Tab($innerNotebook);
        $self->adjectives2Tab($innerNotebook);
        $self->adjectives3Tab($innerNotebook);
        $self->adjectives4Tab($innerNotebook);

        return 1;
    }

    sub adjectives1Tab {

        # Adjectives1 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->adjectives1Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _1', $innerNotebook);

        # Adjectives
        $self->addLabel($table, '<b>Adjectives</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>List of adjective words</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Word', 'text',
            'Type', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 220);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'adjHash');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Word',
            1, 4, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            4, 7, 8, 9);

        $self->addLabel($table, 'Type',
            7, 10, 8, 9);
        my $entry2 = $self->addEntry($table, undef, FALSE,
            10, 12, 8, 9);
        $entry2->set_text('adj');

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'adjHash',
            9,
            $entry,
        );
        $button->signal_connect('clicked' => sub {

            my ($word, $type);

            $word = $entry->get_text();
            $type = $entry2->get_text();

            if ($self->checkEntryIcon($entry)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('adjHash', $word, $type);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'adjHash');
                $self->resetEntryBoxes($entry);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub adjectives2Tab {

        # Adjectives2 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->adjectives2Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _2', $innerNotebook);

        # Declined adjective patterns
        $self->addLabel($table, '<b>Declined adjective patterns</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>Declined adjective endings and their equivalent undeclined endings</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Declined ending pattern', 'text',
            'Undeclined ending', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 220);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'adjEndingHash');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Declined ending pattern',
            1, 4, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'regex', 1, undef,
            4, 7, 8, 9);

        $self->addLabel($table, 'Undeclined ending',
            7, 10, 8, 9);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            10, 12, 8, 9);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'adjEndingHash',
            9,
            $entry, $entry2,
        );
        $button->signal_connect('clicked' => sub {

            my ($declined, $undeclined);

            $declined = $entry->get_text();
            $undeclined = $entry2->get_text();

            if ($self->checkEntryIcon($entry, $entry2)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('adjEndingHash', $declined, $undeclined);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'adjEndingHash');
                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub adjectives3Tab {

        # Adjectives3 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->adjectives3Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _3', $innerNotebook);

        # Undeclined adjective patterns
        $self->addLabel($table, '<b>Undeclined adjective patterns</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>Undeclined adjective endings and their equivalent declined endings</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Undeclined ending pattern', 'text',
            'Declined ending', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 220);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'reverseAdjEndingHash');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Undeclined ending pattern',
            1, 4, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'regex', 1, undef,
            4, 7, 8, 9);

        $self->addLabel($table, 'Declined ending',
            7, 10, 8, 9);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            10, 12, 8, 9);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'reverseAdjEndingHash',
            9,
            $entry, $entry2,
        );
        $button->signal_connect('clicked' => sub {

            my ($declined, $undeclined);

            $declined = $entry->get_text();
            $undeclined = $entry2->get_text();

            if ($self->checkEntryIcon($entry, $entry2)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('reverseAdjEndingHash', $declined, $undeclined);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV(
                    $slWidget,
                    scalar (@columnList / 2),
                    'reverseAdjEndingHash',
                );

                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub adjectives4Tab {

        # Adjectives4 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->adjectives4Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _4', $innerNotebook);

        # Declined adjectives
        $self->addLabel($table, '<b>Declined adjectives</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of adjectives that are exceptions to the usual declension rules</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Undeclined form', 'text',
            'Declined form', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 13, 2, 8,
            -1, 220);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'declinedAdjHash');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Undeclined form',
            1, 4, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            4, 7, 8, 9);

        $self->addLabel($table, 'Declined form',
            7, 10, 8, 9);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            10, 13, 8, 9);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'declinedAdjHash',
            9,
            $entry, $entry2,
        );
        $button->signal_connect('clicked' => sub {

            my ($declined, $undeclined);

            $declined = $entry->get_text();
            $undeclined = $entry2->get_text();

            if ($self->checkEntryIcon($entry, $entry2)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('declinedAdjHash', $declined, $undeclined);
                # (We have a second hash to update)
                $self->modifyEditHash_hashIV(
                    'reverseDeclinedAdjHash',
                    $undeclined,
                    $declined,
                );

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'declinedAdjHash');
                $self->resetEntryBoxes($entry);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub pseudoTab {

        # Pseudo tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->pseudoTab', @_);
        }

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('P_seudo', $self->notebook);

        # Add tabs to the inner notebook
        $self->pseudo1Tab($innerNotebook);
        $self->pseudo2Tab($innerNotebook);
        $self->pseudo3Tab($innerNotebook);

        return 1;
    }

    sub pseudo1Tab {

        # Pseudo1 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->pseudo1Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _1', $innerNotebook);

        # Pseudo nouns
        $self->addLabel($table, '<b>Pseudo nouns</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>Groups of words which represent a single noun</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Pseudo noun', 'text',
            'Replacement noun', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 200);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'pseudoNounHash');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Pseudo noun (pattern)',
            1, 4, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'regex', 1, undef,
            4, 12, 8, 9);

        $self->addLabel($table, 'Replacement noun',
            1, 4, 9, 10);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            4, 12, 9, 10);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'pseudoNounHash',
            10,
            $entry, $entry2,
        );
        $button->signal_connect('clicked' => sub {

            my ($pseudo, $replacement);

            $pseudo = $entry->get_text();
            $replacement = $entry2->get_text();

            if ($self->checkEntryIcon($entry, $entry2)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('pseudoNounHash', $pseudo, $replacement);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'pseudoNounHash');
                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub pseudo2Tab {

        # Pseudo2 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->pseudo2Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _2', $innerNotebook);

        # Pseudo objects
        $self->addLabel($table, '<b>Pseudo objects</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>Groups of words which represent a single object</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Pseudo object', 'text',
            'Replacement object', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 200);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'pseudoObjHash');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Pseudo object (pattern)',
            1, 4, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'regex', 1, undef,
            4, 12, 8, 9);

        $self->addLabel($table, 'Replacement object (if any)',
            1, 4, 9, 10);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'string', 0, undef,
            4, 12, 9, 10);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'pseudoObjHash',
            10,
            $entry, $entry2,
        );
        $button->signal_connect('clicked' => sub {

            my ($pseudo, $replacement);

            $pseudo = $entry->get_text();
            $replacement = $entry2->get_text();

            if ($self->checkEntryIcon($entry, $entry2)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('pseudoObjHash', $pseudo, $replacement);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'pseudoObjHash');
                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub pseudo3Tab {

        # Pseudo3 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->pseudo3Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _3', $innerNotebook);

        # Pseudo adjectives
        $self->addLabel($table, '<b>Pseudo adjectives</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>Groups of words which represent a single adjective</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Pseudo adjective', 'text',
            'Replacement adjective', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 200);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'pseudoAdjHash');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Pseudo adjective (pattern)',
            1, 4, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'regex', 1, undef,
            4, 12, 8, 9);

        $self->addLabel($table, 'Replacement adjective (if any)',
            1, 4, 9, 10);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'string', 0, undef,
            4, 12, 9, 10);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'pseudoAdjHash',
            10,
            $entry, $entry2,
        );
        $button->signal_connect('clicked' => sub {

            my ($pseudo, $replacement);

            $pseudo = $entry->get_text();
            $replacement = $entry2->get_text();

            if ($self->checkEntryIcon($entry, $entry2)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('pseudoAdjHash', $pseudo, $replacement);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'pseudoAdjHash');
                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub contentsTab {

        # Contents tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->contentsTab', @_);
        }

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('_Contents', $self->notebook);

        # Add tabs to the inner notebook
        $self->contents1Tab($innerNotebook);
        $self->contents2Tab($innerNotebook);
        $self->contents3Tab($innerNotebook);

        return 1;
    }

    sub contents1Tab {

        # Contents1 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->contents1Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _1', $innerNotebook);

        # Death words
        $self->addLabel($table, '<b>Death words</b>',
            0, 6, 0, 2);
        $self->addLabel($table, '<i>List of words which mark an object as dead</i>',
            1, 6, 2, 4);
        my ($textView) = $self->addTextView($table, undef, TRUE,
            1, 6, 4, 6,
            TRUE, TRUE, TRUE, FALSE,  # Treat as list, remove empty lines, do remove whitespace
            -1, 125);
        my $buffer = $textView->get_buffer();
        $buffer->set_text(join("\n", $self->editObj->ivKeys('deathWordHash')));
        $buffer->signal_connect('changed' => sub {

            # (The 'edit' window presents this IV as a list, but the dictionary object stores it as
            #   a hash)

            my (
                $text,
                @list, @finalList,
                %hash,
            );

            $text = $axmud::CLIENT->desktopObj->bufferGetText($buffer);

            # Split the contents of the textview into a list of lines, separated by newline
            #   characters
            @list = split("\n", $text);
            # Remove any empty lines and leading/trailing whitespace
            foreach my $line (@list) {

                if ($line) {

                    $line =~ s/^\s*//;      # Remove leading whitespace
                    $line =~ s/\s*$//;      # Remove trailing whitepsace
                    push (@finalList, $line);
                }

                # From @finalList, compile the hash
                foreach my $word (@finalList) {

                    $hash{$word} = undef;
                }

                # Set the IV using the hash
                $self->ivAdd('editHash', 'deathWordHash', \%hash);
            }
        });

        # Ignore words
        $self->addLabel($table, '<b>Ignore words</b>',
            0, 6, 6, 8);
        $self->addLabel($table,
            '<i>List of words that should be ignored (never added to the unknown words list)</i>',
            1, 6, 8, 10);
        my $textView2 = $self->addTextView($table, undef, TRUE,
            1, 6, 10, 12,
            TRUE, TRUE, TRUE, FALSE,  # Treat as list, remove empty lines, do remove whitespace
            -1, 125);
        my $buffer2 = $textView2->get_buffer();
        $buffer2->set_text(join("\n", $self->editObj->ivKeys('ignoreWordHash')));
        $buffer2->signal_connect('changed' => sub {

            # (The 'edit' window presents this IV as a list, but the dictionary object stores it as
            #   a hash)

            my (
                $text,
                @list, @finalList,
                %hash,
            );

            $text = $axmud::CLIENT->desktopObj->bufferGetText($buffer);

            # Split the contents of the textview into a list of lines, separated by newline
            #   characters
            @list = split("\n", $text);
            # Remove any empty lines and leading/trailing whitespace
            foreach my $line (@list) {

                if ($line) {

                    $line =~ s/^\s*//;      # Remove leading whitespace
                    $line =~ s/\s*$//;      # Remove trailing whitepsace
                    push (@finalList, $line);
                }

                # From @finalList, compile the hash
                foreach my $word (@finalList) {

                    $hash{$word} = 'ignore_word';
                }

                # Set the IV using the hash
                $self->ivAdd('editHash', 'ignoreWordHash', \%hash);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub contents2Tab {

        # Contents2 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (
            $listRef,
            @columnList, @comboList,
            %ivHash,
        );

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->contents2Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _2', $innerNotebook);

        # Unknown words
        $self->addLabel($table, '<b>Unknown words</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>Gathered words which aren\'t yet defined in this dictionary <b>(use the \'Save\''
            . ' button before editing other other tabs)</b></i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Unknown word', 'text',
        );

        my $slWidget = Games::Axmud::Obj::Simple::List->new(@columnList);
        $slWidget->get_selection->set_mode('multiple');

        # Make the simple list scrollable
        my $scroller = Gtk2::ScrolledWindow->new;
        $scroller->set_policy('automatic', 'automatic');
        $scroller->set_size_request(-1, 200);
        $scroller->add($slWidget);
        $table->attach_defaults($scroller, 1, 6, 2, 8);

        # Initialise the list. Import the IV
        %ivHash = $self->getEditHash_hashIV('unknownWordHash');
        @{$slWidget->{data}} = sort {lc($a) cmp lc($b)} (keys %ivHash);

        $self->contents2Tab_addButtons($table,
            $slWidget, 'Guild', 'Selected words are guilds', 'addword -g ', undef,
            6, 8, 2, 3);
        $self->contents2Tab_addButtons($table,
            $slWidget, 'Race', 'Selected words are races', 'addword -r ', undef,
            8, 10, 2, 3);
        $self->contents2Tab_addButtons($table,
            $slWidget, 'Weapon', 'Selected words are weapons', 'addword -w ', undef,
            10, 12, 2, 3);
        $self->contents2Tab_addButtons($table,
            $slWidget, 'Armour', 'Selected words are armours', 'addword -a ', undef,
            6, 8, 3, 4);
        $self->contents2Tab_addButtons($table,
            $slWidget, 'Garment', 'Selected words are garments', 'addword -e ', undef,
            8, 10, 3, 4);
        $self->contents2Tab_addButtons($table,
            $slWidget, 'Sentient', 'Selected words are sentient beings', 'addword -s ', undef,
            10, 12, 3, 4);
        $self->contents2Tab_addButtons($table,
            $slWidget, 'Creature', 'Selected words are non-sentient beings', 'addword -k ', undef,
            6, 8, 5, 6);
        $self->contents2Tab_addButtons($table,
            $slWidget, 'Adjective', 'Selected words are adjectives', 'addword -j ', undef,
            8, 10, 5, 6);
        $self->contents2Tab_addButtons($table,
            $slWidget, 'Ignore', 'Selected words can be ignored in future', 'addword -i ', undef,
            10, 12, 5, 6);

        # Portables
        if ($self->ivExists('editHash', 'portableTypeList')) {

            $listRef = $self->ivShow('editHash', 'portableTypeList');
            @comboList = @$listRef;

        } else {

            @comboList = $self->editObj->portableTypeList;
        }

        my $comboBox = $self->addComboBox($table, undef, \@comboList, '',
            TRUE,               # No 'undef' value used
            6, 9, 6, 7);

        $self->contents2Tab_addButtons($table,
            $slWidget, 'Portable', 'Selected words are portables', 'addword -p ', $comboBox,
            9, 12, 6, 7);

        # Decorations
        if ($self->ivExists('editHash', 'decorationTypeList')) {

            $listRef = $self->ivShow('editHash', 'decorationTypeList');
            @comboList = @$listRef;

        } else {

            @comboList = $self->editObj->decorationTypeList;
        }

        my $comboBox2 = $self->addComboBox($table, undef, \@comboList, '',
            TRUE,               # No 'undef' value used
            6, 9, 7, 8);

        $self->contents2Tab_addButtons($table,
            $slWidget, 'Decoration', 'Selected words are decorations', 'addword -d ', $comboBox2,
            9, 12, 7, 8);

        # Delete word
        $self->contents2Tab_addButtons($table,
            $slWidget, 'Delete word(s)', 'Selected words are deleted from the list', undef, undef,
            9, 12, 8, 9);

        # Widgets can't be manipulated when this isn't a current dictionary (buttons are
        #   desensitised in the calls to $self->contents2Tab_addButtons)
        if (! $self->currentFlag) {

            $comboBox->set_sensitive(FALSE);
            $comboBox2->set_sensitive(FALSE);
        }

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub contents2Tab_addButtons {

        # Called by $self->words2Tab
        # Adds a single button that, when clicked, runs the client command ';addword' on the
        #   selected item(s) in the GA::Obj::Simple::List displayed in this tab
        #
        # Expected arguments
        #   $table      - The Gtk2::Table for the tab
        #   $slWidget   - The GA::Obj::Simple::List displayed
        #   $title      - The button's title
        #   $tooltips   - Tooltips for the button
        #   $cmd        - The text of the command to send, e.g. 'addword -r' to which the selected
        #                   items are appended (e.g. 'addword -r troll'). If 'undef', no command is
        #                   sent (so the word is simply removed from the list)
        #   $widget     - The Gtk::ComboBox linked to this button ('undef' if not used)
        #   $leftAttach, $rightAttach, $topAttach, $bottomAttach
        #               - The position of the button in the tab's table
        #
        # Return values
        #   'undef' on improper arguments
        #   Otherwise the Gtk2::Button created created

        my (
            $self, $table, $slWidget, $title, $tooltips, $cmd, $widget, $leftAttach, $rightAttach,
            $topAttach, $bottomAttach, $check
        ) = @_;

        # Local variables
        my (
            @indexList,
            %ivHash,
        );

        # Check for improper arguments
        if (
            ! defined $table || ! defined $slWidget|| ! defined $title || ! defined $tooltips
            || ! defined $leftAttach || ! defined $rightAttach || ! defined $topAttach
            || ! defined $bottomAttach || defined $check
        ) {
            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->contents2Tab_addButtons',
                @_,
            );
        }

        # Create the button
        my $button = $self->addButton($table, $title, $tooltips, undef,
            $leftAttach, $rightAttach, $topAttach, $bottomAttach,
            TRUE,               # Irreversible
        );
        $button->signal_connect('clicked' => sub {

            my ($arg, $arg2);

            # Get the current hash of unknown words by importing the IV
            %ivHash = $self->getEditHash_hashIV('unknownWordHash');

            # Get the second argument to the 'addword' command, if any
            if (! defined $widget) {

                $arg2 = '';     # Will be appended to $arg

            } elsif ($widget->get_active_text()) {

                $arg2 = ' ' . $widget->get_active_text();
            }

            if (defined $arg2) {

                # (Allow multiple selected words; deal with each one in turn)
                @indexList = $slWidget->get_selected_indices();
                foreach my $index (@indexList) {

                    $arg = ${$slWidget->{data}}[$index][0];
                    delete $ivHash{$arg};

                    if ($cmd) {

                        $self->session->pseudoCmd($cmd . $arg . $arg2, $self->pseudoCmdMode);
                    }
                }

                @{$slWidget->{data}} = sort {lc($a) cmp lc($b)} (keys %ivHash);
                $self->ivAdd('editHash', 'unknownWordHash', \%ivHash);
            }
        });

        # Widgets can't be manipulated when this isn't a current profile
        if (! $self->currentFlag) {

            $button->set_sensitive(FALSE);
        }

        return $button;
    }

    sub contents3Tab {

        # Contents3 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (
            $unknownFlag,
            @columnList,
        );

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->contents3Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _3', $innerNotebook);

        # Contents lines
        $self->addLabel($table, '<b>Contents lines</b>',
            0, 12, 0, 1);
        $self->addLabel(
            $table,
            '<i>Contents lines gathered by the Locator task <b>(use the \'Save\' button before'
            . ' editing other tabs)</b></i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Contents line', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 180);      # Fixed height

        # Initialise the list
        $self->contents3Tab_refreshList($slWidget, (scalar @columnList / 2));

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Pattern',
            1, 3, 9, 10);
        my $entry = $self->addEntryWithIcon($table, undef, 'regex', 1, undef,
            3, 12, 9, 10);

        $self->addLabel($table, 'Replacement',
            1, 3, 10, 11);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            3, 6, 10, 11);

        # Add a checkbutton right at the bottom
        my $checkButton = $self->addCheckButton($table, undef, TRUE,
            6, 7, 11, 12);
        $unknownFlag = FALSE;       # Button starts deselected
        $checkButton->signal_connect('toggled' => sub {

            if ($checkButton->get_active()) {
                $unknownFlag = TRUE;
            } else {
                $unknownFlag = FALSE;
            }
        });
        $self->addLabel($table, 'Also add the replacement as unknown word(s)',
            7, 12, 11, 12);

        # Add buttons
        my $button = $self->addButton($table,
            'Use selected', 'Copy the selected contents line into the entry box below', undef,
            1, 3, 8, 9);
        $button->signal_connect('clicked' => sub {

            my ($key) = $self->getSimpleListData($slWidget, 0);
            if (defined $key) {

                $entry->set_text($key);
                $entry2->set_text('');
            }
        });

        my $button2 = $self->addButton($table,
            'Delete selected', 'Delete the selected contents line', undef,
            8, 10, 8, 9);
        $button2->signal_connect('clicked' => sub {

            my ($key) = $self->getSimpleListData($slWidget, 0);
            if (defined $key) {

                # Delete the key-value pair from the hash
                $self->modifyEditHash_hashIV('contentsLinesHash', $key, undef, TRUE);

                # Refresh the simple list and reset entry boxes
                $self->contents3Tab_refreshList($slWidget, (scalar @columnList / 2));
                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        my $button3 = $self->addButton($table,
            'Delete all', 'Empty the hash which stores contents lines', undef,
            10, 12, 8, 9);
        $button3->signal_connect('clicked' => sub {

            # Add an empty hash to $self->editHash
            $self->ivAdd('editHash', 'contentsLinesHash', {});

            # Refresh the simple list and reset entry boxes
            $self->contents3Tab_refreshList($slWidget, (scalar @columnList / 2));
            $self->resetEntryBoxes($entry, $entry2);
        });

        my $button4 = $self->addButton($table,
            'Add pseudo-noun', 'Add the pattern and replacement as a pseudo-noun', undef,
            6, 8, 10, 11,
            TRUE,               # Irreversible
        );
        $button4->signal_connect('clicked' => sub {

            my ($pattern, $replacement);

            if ($self->checkEntryIcon($entry, $entry2)) {

                $pattern = $entry->get_text();
                $replacement = $entry2->get_text();

                $self->session->pseudoCmd(
                    'addword -x <' . $replacement . '> <' . $pattern . '>',
                    $self->pseudoCmdMode,
                );

                if ($unknownFlag) {

                    $self->session->pseudoCmd(
                        'addword -u ' . join(' ', split(/\s+/, $replacement)),
                        $self->pseudoCmdMode,
                    );
                }

                # (Reset the replacement, but not the pattern)
                $entry2->set_text('');
            }
        });

        my $button5 = $self->addButton($table,
            'Add pseudo-obj', 'Add the pattern and replacement as a pseudo-object', undef,
            8, 10, 10, 11,
            TRUE,               # Irreversible
        );
        $button5->signal_connect('clicked' => sub {

            my ($pattern, $replacement);

            if ($self->checkEntryIcon($entry, $entry2)) {

                $pattern = $entry->get_text();
                $replacement = $entry2->get_text();

                $self->session->pseudoCmd(
                    'addword -v <' . $replacement . '> <' . $pattern . '>',
                    $self->pseudoCmdMode,
                );

                if ($unknownFlag) {

                    $self->session->pseudoCmd(
                        'addword -u ' . join(' ', split(/\s+/, $replacement)),
                        $self->pseudoCmdMode,
                    );
                }

                # (Reset the replacement, but not the pattern)
                $entry2->set_text('');
            }
        });

        my $button6 = $self->addButton($table,
            'Add pseudo-adj', 'Add the pattern and replacement as a pseudo-adjective', undef,
            10, 12, 10, 11,
            TRUE,               # Irreversible
        );
        $button6->signal_connect('clicked' => sub {

            my ($pattern, $replacement);

            if ($self->checkEntryIcon($entry, $entry2)) {

                $pattern = $entry->get_text();
                $replacement = $entry2->get_text();

                $self->session->pseudoCmd(
                    'addword -y <' . $replacement . '> <' . $pattern . '>',
                    $self->pseudoCmdMode,
                );

                if ($unknownFlag) {

                    $self->session->pseudoCmd(
                        'addword -u ' . join(' ', split(/\s+/, $replacement)),
                        $self->pseudoCmdMode,
                    );
                }

                # (Reset the replacement, but not the pattern)
                $entry2->set_text('');
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub contents3Tab_refreshList {

        # Called by $self->contentsTab to refresh the GA::Obj::Simple::List, displaying only keys
        #
        # Expected arguments
        #   $slWidget   - The GA::Obj::Simple::List
        #   $columns    - The number of columns in the list
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $slWidget, $columns, $check) = @_;

        # Local variables
        my (
            @sortedList, @dataList,
            %ivHash,
        );

        # Check for improper arguments
        if (! defined $slWidget || ! defined $columns || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->contents3Tab_refreshList',
                @_,
            );
        }

        # Import the IV
        %ivHash = $self->getEditHash_hashIV('contentsLinesHash');

        # Compile the simple list data
        push (@dataList, (sort {lc($a) cmp lc($b)} (keys %ivHash)));

        # Reset the simple list
        $self->resetListData($slWidget, [@dataList], $columns);

        return 1;
    }

    sub directionsTab {

        # Directions tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->directionsTab', @_);
        }

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('_Directions', $self->notebook);

        # Add tabs to the inner notebook
        $self->directions1Tab($innerNotebook);
        $self->directions2Tab($innerNotebook);
        $self->directions3Tab($innerNotebook);
        $self->directions4Tab($innerNotebook);
        $self->directions5Tab($innerNotebook);

        return 1;
    }

    sub directions1Tab {

        # Directions1 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (
            $showAllFlag,
            @columnList,
        );

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->directions1Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _1', $innerNotebook);

        # Primary directions
        $self->addLabel($table, '<b>Primary directions</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of the standard compass directions, plus up/down</i>',
            1, 6, 1, 2);

        my $checkButton = $self->addCheckButton($table, undef, TRUE,
            6, 7, 1, 2);
        # (->signal_connect appears below)
        $self->addLabel($table, '<i>Show all intercardinal directions</i>',
            7, 12, 1, 2);

        # Create the simple list
        @columnList = (
            'Standard', 'text',
            'Custom', 'text',
            'Abbrev\'n', 'text',
            'Opposite dirs', 'text',
            'Opp abbrev\'d dirs', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 7,
            -1, 160);      # Fixed height

        # Initialise the list
        $self->directions1Tab_refreshList($slWidget, (scalar @columnList / 2), $showAllFlag);

        # Entry boxes
        $self->addLabel($table, 'Map direction',
            1, 4, 8, 9);
        my $entry = $self->addEntry($table, undef, FALSE,
            4, 6, 8, 9);

        $self->addLabel($table, 'Direction',
            6, 9, 8, 9);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            9, 12, 8, 9);

        $self->addLabel($table, 'Abbreviation',
            1, 4, 9, 10);
        my $entry3 = $self->addEntryWithIcon($table, undef, 'string', undef, undef,
            4, 6, 9, 10);

        $self->addLabel($table, 'Opposite dirs',
            6, 9, 9, 10);
        my $entry4 = $self->addEntryWithIcon($table, undef, 'string', undef, undef,
            9, 12, 9, 10);

        $self->addLabel($table, 'Opposite abbrev dirs',
            1, 4, 10, 11);
        my $entry5 = $self->addEntryWithIcon($table, undef, 'string', undef, undef,
            4, 6, 10, 11);

        # 'Edit direction' button
        my $button = $self->addButton($table,
            'Edit direction', 'Edit the selected direction', undef,
            1, 3, 7, 8);
        $button->signal_connect('clicked' => sub {

            my ($standard, $dir, $abbrev, $opp, $oppAbbrev)
                = $self->getSimpleListData($slWidget, 0, 1, 2, 3, 4);

            if ($standard) {

                # Sensitise the widgets (but not $entry, which is always desensitised)
                $self->sensitiseWidgets($entry2, $entry3, $entry4, $entry5);

                # Fill the entry boxes with the selected direction's data
                $entry->set_text($standard);
                $entry2->set_text($dir);
                $entry3->set_text($abbrev);
                $entry4->set_text($opp);
                $entry5->set_text($oppAbbrev);
            }
        });

        # 'Use changes' button
        my $button2 = $self->addButton($table,
            'Use changes', 'Use the modified directions', undef,
            10, 12, 10, 11);
        $button2->signal_connect('clicked' => sub {

            my (
                $standard, $dir, $abbrev, $opp, $oppAbbrev, $msg, $hashRef,
                %dirHash, %abbrevHash, %oppHash, %oppAbbrevHash, %secDirHash, %secAbbrevHash,
                %relDirHash, %relAbbrevHash,
                %combDirHash,
            );

            if ($self->checkEntryIcon($entry2, $entry3, $entry4, $entry5)) {

                $standard = $entry->get_text();
                # (Definitely need to remove any leading/trailing whitespace, as well as replace
                #   two or more whitespace characters in the middle with a single whitespace
                #   character)
                $dir = $axmud::CLIENT->trimWhitespace($entry2->get_text(), TRUE);
                $abbrev = $axmud::CLIENT->trimWhitespace($entry3->get_text(), TRUE);
                $opp = $axmud::CLIENT->trimWhitespace($entry4->get_text(), TRUE);
                $oppAbbrev = $axmud::CLIENT->trimWhitespace($entry5->get_text(), TRUE);

                # Need to check that the user hasn't added an existing custom primary or recognised
                #   secondary direction
                # Import data from the affected IVs
                if ($self->ivExists('editHash', 'primaryDirHash')) {

                    # (If any primary dir IVs have been changed, then they are ALL in ->editHash)
                    $hashRef = $self->ivShow('editHash', 'primaryDirHash');
                    %dirHash = %$hashRef;
                    $hashRef = $self->ivShow('editHash', 'primaryAbbrevHash');
                    %abbrevHash = %$hashRef;
                    $hashRef = $self->ivShow('editHash', 'primaryOppHash');
                    %oppHash = %$hashRef;
                    $hashRef = $self->ivShow('editHash', 'primaryOppAbbrevHash');
                    %oppAbbrevHash = %$hashRef;

                } else {

                    %dirHash = $self->editObj->primaryDirHash;
                    %abbrevHash = $self->editObj->primaryAbbrevHash;
                    %oppHash = $self->editObj->primaryOppHash;
                    %oppAbbrevHash = $self->editObj->primaryOppAbbrevHash;
                }

                if ($self->ivExists('editHash', 'secondaryDirList')) {

                    # (If any secondary dir IVs have been changed, then they are ALL in ->editHash)
                    $hashRef = $self->ivShow('editHash', 'secondaryDirHash');
                    %secDirHash = %$hashRef;
                    $hashRef = $self->ivShow('editHash', 'secondaryAbbrevHash');
                    %secAbbrevHash = %$hashRef;

                } else {

                    %secDirHash = $self->editObj->secondaryDirHash;
                    %secAbbrevHash = $self->editObj->secondaryAbbrevHash;
                }

                %relDirHash = $self->editObj->relativeDirHash;
                %relAbbrevHash = $self->editObj->relativeAbbrevHash;

                # (Compose our own temporary %combDirHash, in the form
                #   $combDirHash{custom_primary_dir} = undef
                #   $combDirHash{recognised_secondary_dir} = undef
                foreach my $value (values %dirHash) {

                    # (User is allowed to enter custom 'north' direction for the standard dir north,
                    #   so we don't add that to %combDirHash)
                    if ($dirHash{$standard} ne $value) {

                        $combDirHash{$value} = undef;   # Hash in form $hash{$key} = $value
                    }
                }

                foreach my $value (values %abbrevHash) {

                    if ($abbrevHash{$standard} ne $value) {

                        $combDirHash{$value} = undef;   # Hash in form $hash{$key} = $value
                    }
                }

                foreach my $key (keys %secDirHash) {

                    $combDirHash{$key} = undef;         # Hash in form $hash{$key} = $key
                }

                foreach my $key (keys %secAbbrevHash) {

                    $combDirHash{$key} = undef;         # Hash in form $hash{$key} = undef
                }

                foreach my $value (values %relDirHash) {

                    $combDirHash{$value} = undef;       # Hash in form $hash{$slot} = $value
                }

                foreach my $value (values %relAbbrevHash) {

                    $combDirHash{$value} = undef;       # Hash in form $hash{$slot} = $value
                }

                # Now we can test whether the directions entered by the user are already in use, or
                #   not
                if (exists $combDirHash{$dir}) {

                    $msg = '\'' . $dir . '\' is not available because it is already in use as a'
                            . ' custom primary or a secondary direction';

                } elsif ($abbrev && exists $combDirHash{$abbrev}) {

                    $msg = '\'' . $abbrev . '\' is not available because it is already in use as a'
                            . ' custom primary or a secondary direction';
                }

                if ($msg) {

                    # Don't make any changes
                    $self->showMsgDialogue(
                        'Use changes',
                        'error',
                        $msg,
                        'ok',
                    );

                } else {

                    # Add (or replace) an entry in the hashes
                    $dirHash{$standard} = $dir;
                    $abbrevHash{$standard} = $abbrev;
                    $oppHash{$standard} = $opp;
                    $oppAbbrevHash{$standard} = $oppAbbrev;

                    # Store the modified list/hashes. GA::Obj::Dict->createCombHashes is not called
                    #   until $self->saveChanges, of course
                    $self->ivAdd('editHash', 'primaryDirHash', \%dirHash);
                    $self->ivAdd('editHash', 'primaryAbbrevHash', \%abbrevHash);
                    $self->ivAdd('editHash', 'primaryOppHash', \%oppHash);
                    $self->ivAdd('editHash', 'primaryOppAbbrevHash', \%oppAbbrevHash);

                    # Refresh the list to show the changes
                    $self->directions1Tab_refreshList(
                        $slWidget,
                        (scalar @columnList / 2),
                        $showAllFlag,
                    );

                    # Empty the widgets below the simple list
                    $entry->set_text('');
                    $entry2->set_text('');
                    $entry3->set_text('');
                    $entry4->set_text('');
                    $entry5->set_text('');

                    # Desensitise the widgets below the simple list
                    $self->desensitiseWidgets($entry2, $entry3, $entry4, $entry5);
                }
            }
        });

        # 'Reset list' button
        my $button3 = $self->addButton($table,
            'Reset list', 'Reset the list of primary directions', undef,
            10, 12, 7, 8);
        $button3->signal_connect('clicked' => sub {

            # Remove the IVs from $self->editHash, so that the IVs in $self->editObj take over
            $self->ivDelete('editHash', 'primaryDirHash');
            $self->ivDelete('editHash', 'primaryAbbrevHash');
            $self->ivDelete('editHash', 'primaryOppHash');
            $self->ivDelete('editHash', 'primaryOppAbbrevHash');

            # Refresh the list to show the changes
            $self->directions1Tab_refreshList($slWidget, (scalar @columnList / 2), $showAllFlag);

            # Empty the widgets below the simple list
            $entry->set_text('');
            $entry2->set_text('');
            $entry3->set_text('');
            $entry4->set_text('');
            $entry5->set_text('');

            # Desensitise the widgets below the simple list
            $self->desensitiseWidgets($entry2, $entry3, $entry4, $entry5);
        });

        # Now we can defined the checkbutton's ->signal_connect
        $checkButton->signal_connect('toggled' => sub {

            if ($showAllFlag) {
                $showAllFlag = FALSE;
            } else {
                $showAllFlag = TRUE;
            }

            $self->directions1Tab_refreshList($slWidget, (scalar @columnList / 2), $showAllFlag);
        });

        # Widgets below the simple list start desensitised (NB $entry is always desensisited)
        $self->desensitiseWidgets($entry, $entry2, $entry3, $entry4, $entry5);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub directions1Tab_refreshList {

        # Called by $self->directions1Tab to refresh the GA::Obj::Simple::List
        #
        # Expected arguments
        #   $slWidget       - The GA::Obj::Simple::List
        #   $columns        - The number of columns in the list
        #
        # Optional arguments
        #   $showAllFlag    - If set to FALSE (or 'undef'), secondary-intercardinal directions
        #                       ('northnortheast', etc) are not shown. If set to TRUE, all primary
        #                       directions are shown
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $slWidget, $columns, $showAllFlag, $check) = @_;

        # Local variables
        my (
            $hashRef,
            @dirList, @dataList,
            %dirHash, %abbrevHash, %oppHash, %oppAbbrevHash,
        );

        # Check for improper arguments
        if (! defined $slWidget || ! defined $columns || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->directions1Tab_refreshList',
                @_,
            );
        }

        # Import the IVs
        if ($self->ivExists('editHash', 'primaryDirHash')) {

            $hashRef = $self->ivShow('editHash', 'primaryDirHash');
            %dirHash = %$hashRef;

            $hashRef = $self->ivShow('editHash', 'primaryAbbrevHash');
            %abbrevHash = %$hashRef;

            $hashRef = $self->ivShow('editHash', 'primaryOppHash');
            %oppHash = %$hashRef;

            $hashRef = $self->ivShow('editHash', 'primaryOppAbbrevHash');
            %oppAbbrevHash = %$hashRef;

        } else {

            %dirHash = $self->editObj->primaryDirHash;
            %abbrevHash = $self->editObj->primaryAbbrevHash;
            %oppHash = $self->editObj->primaryOppHash;
            %oppAbbrevHash = $self->editObj->primaryOppAbbrevHash;
        }

        @dirList = $axmud::CLIENT->constPrimaryDirList;

        # Compile the simple list data
        foreach my $dir (@dirList) {

            if ($showAllFlag || $axmud::CLIENT->ivExists('constShortPrimaryDirHash', $dir)) {

                push (@dataList,
                    $dir,
                    $dirHash{$dir},
                    $abbrevHash{$dir},
                    $oppHash{$dir},
                    $oppAbbrevHash{$dir},
                );
            }
        }

        # Reset the simple list
        $self->resetListData($slWidget, [@dataList], $columns);

        return 1;
    }

    sub directions2Tab {

        # Directions2 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (
            $thisDir, $noAutoString,
            @columnList, @comboList,
        );

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->directions2Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _2', $innerNotebook);

        # Primary directions
        $self->addLabel($table, '<b>Secondary directions</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>Customisable list of secondary directions</i>',
            1, 12, 1, 2);

        # Create the simple list
        @columnList = (
            'Custom', 'text',
            'Abbrev\'n', 'text',
            'Opposite dirs', 'text',
            'Opp abbrev\'d dirs', 'text',
            'Auto-allocated to', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 160);      # Fixed height

        # Initialise the list
        $self->directions2Tab_refreshList($slWidget, (scalar @columnList / 2));

        # Entry/combo boxes
        $self->addLabel($table, 'Direction',
            1, 2, 9, 10);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            2, 6, 9, 10);

        $self->addLabel($table, 'Abbreviation',
            6, 8, 9, 10);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'string', undef, undef,
            8, 12, 9, 10);

        $self->addLabel($table, 'Opposite dirs',
            1, 2, 10, 11);
        my $entry3 = $self->addEntryWithIcon($table, undef, 'string', undef, undef,
            2, 6, 10, 11);

        $self->addLabel($table, 'Opposite abbrev dirs',
            6, 8, 10, 11);
        my $entry4 = $self->addEntryWithIcon($table, undef, 'string', undef, undef,
            8, 12, 10, 11);

        @comboList = qw(
            north northeast east southeast south southwest west northwest up down
            northnortheast eastnortheast eastsoutheast southsoutheast
            southsouthwest westsouthwest westnorthwest northnorthwest
        );

        $noAutoString = 'Don\'t auto-allocate';
        unshift(@comboList, $noAutoString);

        $self->addLabel($table, 'Auto-allocate to',
            1, 2, 11, 12);
        my $combo = $self->addComboBox($table, undef, \@comboList, '',
            TRUE,               # No 'undef' value used
            2, 6, 11, 12);

        # 'Unselect' button
        my $button = $self->addButton($table,
            'Unselect', 'Unselect the selected direction', undef,
            1, 2, 8, 9);
        $button->signal_connect('clicked' => sub {

            $slWidget->get_selection->unselect_all();
        });

        # 'Add/Edit direction' button
        my $button2 = $self->addButton($table,
            'Add/Edit direction', 'Add a new direction or edit the selected one', undef,
            2, 4, 8, 9);
        $button2->signal_connect('clicked' => sub {

            my ($dir, $abbrev, $opp, $oppAbbrev, $auto, $posn);

            ($dir, $abbrev, $opp, $oppAbbrev, $auto)
                = $self->getSimpleListData($slWidget, 0, 1, 2, 3, 4);

            # Sensitise the widgets
            $self->sensitiseWidgets($entry, $entry2, $entry3, $entry4, $combo);

            if (! $dir) {

                # 'Add' direction. Make sure the entry boxes are empty
                $entry->set_text('');
                $entry2->set_text('');
                $entry3->set_text('');
                $entry4->set_text('');
                # Make sure the 'Don't auto-allocate' string is visible at the top of the combo
                $combo->set_active(0);

                # If the user clicks 'Use changes', it's a completely new secondary direction
                $thisDir = undef;

            } else {

                # 'Edit' direction. Fill the entry boxes with the selected direction's data
                $entry->set_text($dir);
                if (defined $abbrev) {

                    $entry2->set_text($abbrev);
                }
                $entry3->set_text($opp);
                $entry4->set_text($oppAbbrev);
                # Make the right choice visible in the combobox
                $posn = 0;  # Default is the 'don't auto-allocate' string
                if ($auto) {

                    OUTER: for (my $index = 0; $index < 18; $index++) {

                        if ($comboList[$index] eq $auto) {

                            $posn = $index;
                            last OUTER;
                        }
                    }
                }

                $combo->set_active($posn);
                $axmud::CLIENT->desktopObj->updateWidgets($self->_objClass . '->directions2Tab');

                # Store the direction that's being edited because the 'Use changes' button needs it
                $thisDir = $dir;
            }
        });

        # 'Delete direction' button
        my $button3 = $self->addButton($table,
            'Delete direction', 'Delete the selected direction', undef,
            4, 6, 8, 9);
        $button3->signal_connect('clicked' => sub {

            my (
                $dir, $listRef, $hashRef, $count,
                @dirList,
                %dirHash, %abbrevHash, %oppHash, %oppAbbrevHash, %autoHash,
            );

            ($dir) = $self->getSimpleListData($slWidget, 0);
            if ($dir) {

                # Import data from the affected IVs
                if ($self->ivExists('editHash', 'secondaryDirList')) {

                    $listRef = $self->ivShow('editHash', 'secondaryDirList');
                    @dirList = @$listRef;
                    $hashRef = $self->ivShow('editHash', 'secondaryDirHash');
                    %dirHash = %$hashRef;
                    $hashRef = $self->ivShow('editHash', 'secondaryAbbrevHash');
                    %abbrevHash = %$hashRef;
                    $hashRef = $self->ivShow('editHash', 'secondaryOppHash');
                    %oppHash = %$hashRef;
                    $hashRef = $self->ivShow('editHash', 'secondaryOppAbbrevHash');
                    %oppAbbrevHash = %$hashRef;
                    $hashRef = $self->ivShow('editHash', 'secondaryAutoHash');
                    %autoHash = %$hashRef;

                } else {

                    @dirList = $self->editObj->secondaryDirList;
                    %dirHash = $self->editObj->secondaryDirHash;
                    %abbrevHash = $self->editObj->secondaryAbbrevHash;
                    %oppHash = $self->editObj->secondaryOppHash;
                    %oppAbbrevHash = $self->editObj->secondaryOppAbbrevHash;
                    %autoHash = $self->editObj->secondaryAutoHash;
                }

                # Remove $dir from the list of secondary directions
                $count = -1;
                OUTER: foreach my $otherDir (@dirList) {

                    $count++;

                    if ($otherDir eq $dir) {

                        # Remove the entry
                        splice (@dirList, $count, 1);
                        last OUTER;
                    }
                }

                # Remove $dir from the corresponding hashes
                delete $dirHash{$dir};
                delete $abbrevHash{$dir};
                delete $oppHash{$dir};
                delete $oppAbbrevHash{$dir};
                delete $autoHash{$dir};

                # Store the modified list/hashes
                $self->ivAdd('editHash', 'secondaryDirList', \@dirList);
                $self->ivAdd('editHash', 'secondaryDirHash', \%dirHash);
                $self->ivAdd('editHash', 'secondaryAbbrevHash', \%abbrevHash);
                $self->ivAdd('editHash', 'secondaryOppHash', \%oppHash);
                $self->ivAdd('editHash', 'secondaryOppAbbrevHash', \%oppAbbrevHash);
                $self->ivAdd('editHash', 'secondaryAutoHash', \%autoHash);
                # (If the user was modifying a direction in the entry boxes, those changes are
                #   discarded)
                $thisDir = undef;

                # Refresh the list to show the changes
                $self->directions2Tab_refreshList($slWidget, (scalar @columnList / 2));

                # Empty the widgets below the simple list
                $entry->set_text('');
                $entry2->set_text('');
                $entry3->set_text('');
                $entry4->set_text('');

                # Desensitise the widgets below the simple list
                $self->desensitiseWidgets($entry, $entry2, $entry3, $entry4, $combo);
            }
        });

        # 'Move up' button
        my $button4 = $self->addButton($table,
            'Move up', 'Move the selected direction up the list', undef,
            6, 8, 8, 9);
        $button4->signal_connect('clicked' => sub {

            my (
                $rowNum, $listRef, $hashRef, $dir,
                @dirList,
                %dirHash, %abbrevHash, %oppHash, %oppAbbrevHash, %autoHash,
            );

            # Get the selected row
            ($rowNum) = $slWidget->get_selected_indices();
            # (Don't move a component already at the top of the list...)
            if (defined $rowNum && $rowNum > 0) {

                # Import data from the affected IVs (if we make a change to one, all of them must
                #   be moved into $self->editHash)
                if ($self->ivExists('editHash', 'secondaryDirList')) {

                    $listRef = $self->ivShow('editHash', 'secondaryDirList');
                    @dirList = @$listRef;
                    $hashRef = $self->ivShow('editHash', 'secondaryDirHash');
                    %dirHash = %$hashRef;
                    $hashRef = $self->ivShow('editHash', 'secondaryAbbrevHash');
                    %abbrevHash = %$hashRef;
                    $hashRef = $self->ivShow('editHash', 'secondaryOppHash');
                    %oppHash = %$hashRef;
                    $hashRef = $self->ivShow('editHash', 'secondaryOppAbbrevHash');
                    %oppAbbrevHash = %$hashRef;
                    $hashRef = $self->ivShow('editHash', 'secondaryAutoHash');
                    %autoHash = %$hashRef;


                } else {

                    @dirList = $self->editObj->secondaryDirList;
                    %dirHash = $self->editObj->secondaryDirHash;
                    %abbrevHash = $self->editObj->secondaryAbbrevHash;
                    %oppHash = $self->editObj->secondaryOppHash;
                    %oppAbbrevHash = $self->editObj->secondaryOppAbbrevHash;
                    %autoHash = $self->editObj->secondaryAutoHash;
                }

                # Move the item
                $dir = splice (@dirList, $rowNum, 1);
                splice (@dirList, ($rowNum - 1), 0, $dir);

                # Store the modified list/hashes
                $self->ivAdd('editHash', 'secondaryDirList', \@dirList);
                $self->ivAdd('editHash', 'secondaryDirHash', \%dirHash);
                $self->ivAdd('editHash', 'secondaryAbbrevHash', \%abbrevHash);
                $self->ivAdd('editHash', 'secondaryOppHash', \%oppHash);
                $self->ivAdd('editHash', 'secondaryOppAbbrevHash', \%oppAbbrevHash);
                $self->ivAdd('editHash', 'secondaryAutoHash', \%autoHash);

                # Refresh the simple list
                $self->directions2Tab_refreshList($slWidget, scalar (@columnList / 2));

                # The component should still be highlighted, after being moved up
                $slWidget->select($rowNum - 1);
            }
        });

        # 'Move down' button
        my $button5 = $self->addButton($table,
            'Move down', 'Move the selected direction down the list', undef,
            8, 10, 8, 9);
        $button5->signal_connect('clicked' => sub {

            my (
                $rowNum, $listRef, $hashRef, $dir,
                @dirList,
                %dirHash, %abbrevHash, %oppHash, %oppAbbrevHash, %autoHash,
            );

            # Get the selected row
            ($rowNum) = $slWidget->get_selected_indices();
            if (defined $rowNum) {

                # Import data from the affected IVs (if we make a change to one, all of them must
                #   be moved into $self->editHash)
                if ($self->ivExists('editHash', 'secondaryDirList')) {

                    $listRef = $self->ivShow('editHash', 'secondaryDirList');
                    @dirList = @$listRef;
                    $hashRef = $self->ivShow('editHash', 'secondaryDirHash');
                    %dirHash = %$hashRef;
                    $hashRef = $self->ivShow('editHash', 'secondaryAbbrevHash');
                    %abbrevHash = %$hashRef;
                    $hashRef = $self->ivShow('editHash', 'secondaryOppHash');
                    %oppHash = %$hashRef;
                    $hashRef = $self->ivShow('editHash', 'secondaryOppAbbrevHash');
                    %oppAbbrevHash = %$hashRef;
                    $hashRef = $self->ivShow('editHash', 'secondaryAutoHash');
                    %autoHash = %$hashRef;

                } else {

                    @dirList = $self->editObj->secondaryDirList;
                    %dirHash = $self->editObj->secondaryDirHash;
                    %abbrevHash = $self->editObj->secondaryAbbrevHash;
                    %oppHash = $self->editObj->secondaryOppHash;
                    %oppAbbrevHash = $self->editObj->secondaryOppAbbrevHash;
                    %autoHash = $self->editObj->secondaryAutoHash;
                }

                # (Don't move a component already at the bottom of the list...)
                if ($rowNum < (scalar @dirList - 1)) {

                    $dir = splice (@dirList, $rowNum, 1);
                    splice (@dirList, ($rowNum + 1), 0, $dir);

                    # Store the modified list/hashes
                    $self->ivAdd('editHash', 'secondaryDirList', \@dirList);
                    $self->ivAdd('editHash', 'secondaryDirHash', \%dirHash);
                    $self->ivAdd('editHash', 'secondaryAbbrevHash', \%abbrevHash);
                    $self->ivAdd('editHash', 'secondaryOppHash', \%oppHash);
                    $self->ivAdd('editHash', 'secondaryOppAbbrevHash', \%oppAbbrevHash);
                    $self->ivAdd('editHash', 'secondaryAutoHash', \%autoHash);

                    # Refresh the simple list
                    $self->directions2Tab_refreshList($slWidget, scalar (@columnList / 2));

                    # The component should still be highlighted, after being moved down
                    $slWidget->select($rowNum + 1);
                }
            }
        });

        # 'Use changes' button
        my $button6 = $self->addButton($table,
            'Use changes', 'Use the changes to the data', undef,
            8, 12, 11, 12);
        $button6->signal_connect('clicked' => sub {

            my (
                $dir, $abbrev, $opp, $oppAbbrev, $msg, $auto, $listRef, $hashRef, $count,
                @dirList,
                %primDirHash, %primAbbrevHash, %dirHash, %abbrevHash, %oppHash, %oppAbbrevHash,
                %autoHash, %relDirHash, %relAbbrevHash, %combDirHash,
            );

            if ($self->checkEntryIcon($entry, $entry2, $entry3, $entry4)) {

                $dir = $entry->get_text();
                $abbrev = $entry2->get_text();
                $opp = $entry3->get_text();
                $oppAbbrev = $entry4->get_text();

                # $abbrev should be 'undef', rather than an empty string; vice-versa for $oppAbbrev
                if (! $abbrev) {

                    $abbrev = undef;
                }

                if (! $oppAbbrev) {

                    $oppAbbrev = '';
                }

                # (Definitely need to remove any leading/trailing whitespace, as well as replace two
                #   or more whitespace characters in the middle with a single whitespace character
                $dir = $axmud::CLIENT->trimWhitespace($dir, TRUE);
                if ($abbrev) {

                    $abbrev = $axmud::CLIENT->trimWhitespace($abbrev, TRUE);
                }

                $opp = $axmud::CLIENT->trimWhitespace($opp, TRUE);
                if ($oppAbbrev) {

                    $oppAbbrev = $axmud::CLIENT->trimWhitespace($oppAbbrev, TRUE);
                }

                $auto = $combo->get_active_text();
                if ($auto eq $noAutoString) {

                    # User selected 'Don't auto-allocate' in the combobox; the dictionary's
                    #   ->secondaryAutoHash expects a key-value pair, with a value set to 'undef'
                    $auto = undef;
                }

                # Need to check that the user hasn't added a primary direction (standard or custom)
                #   or an existing secondary direction
                # Import data from the affected IVs. We can't use ->combDirHash because it hasn't
                #   been updated yet, if any changes have been made in this window
                if ($self->ivExists('editHash', 'primaryDirHash')) {

                    # (If any primary dir IVs have been changed, then they are ALL in ->editHash)
                    $hashRef = $self->ivShow('editHash', 'primaryDirHash');
                    %primDirHash = %$hashRef;
                    $hashRef = $self->ivShow('editHash', 'primaryAbbrevHash');
                    %primAbbrevHash = %$hashRef;

                } else {

                    %primDirHash = $self->editObj->primaryDirHash;
                    %primAbbrevHash = $self->editObj->primaryAbbrevHash;
                }

                %relDirHash = $self->editObj->relativeDirHash;
                %relAbbrevHash = $self->editObj->relativeAbbrevHash;

                if ($self->ivExists('editHash', 'secondaryDirList')) {

                    # (If any secondary dir IVs have been changed, then they are ALL in ->editHash)
                    $listRef = $self->ivShow('editHash', 'secondaryDirList');
                    @dirList = @$listRef;
                    $hashRef = $self->ivShow('editHash', 'secondaryDirHash');
                    %dirHash = %$hashRef;
                    $hashRef = $self->ivShow('editHash', 'secondaryAbbrevHash');
                    %abbrevHash = %$hashRef;
                    $hashRef = $self->ivShow('editHash', 'secondaryOppHash');
                    %oppHash = %$hashRef;
                    $hashRef = $self->ivShow('editHash', 'secondaryOppAbbrevHash');
                    %oppAbbrevHash = %$hashRef;
                    $hashRef = $self->ivShow('editHash', 'secondaryAutoHash');
                    %autoHash = %$hashRef;

                } else {

                    @dirList = $self->editObj->secondaryDirList;
                    %dirHash = $self->editObj->secondaryDirHash;
                    %abbrevHash = $self->editObj->secondaryAbbrevHash;
                    %oppHash = $self->editObj->secondaryOppHash;
                    %oppAbbrevHash = $self->editObj->secondaryOppAbbrevHash;
                    %autoHash = $self->editObj->secondaryAutoHash;
                }

                # (Compose our own temporary %combDirHash, in the form
                #   $combDirHash{custom_primary_dir} = undef
                #   $combDirHash{recognised_secondary_dir} = undef
                foreach my $value (values %primDirHash) {

                    $combDirHash{$value} = undef;       # Hash in form $hash{$key} = $value
                }

                foreach my $value (values %primAbbrevHash) {

                    $combDirHash{$value} = undef;       # Hash in form $hash{$key} = $value
                }

                foreach my $key (keys %dirHash) {

                    # (User is allowed to enter custom 'in' direction for the existing dir 'in',
                    #   so we don't add that to %combDirHash)
                    if (! $thisDir || $key ne $thisDir) {

                        $combDirHash{$key} = undef;     # Hash in form $hash{$key} = $key
                    }
                }

                foreach my $key (keys %abbrevHash) {

                    if (! $thisDir || $key ne $thisDir) {

                        $combDirHash{$key} = undef;     # Hash in form $hash{$key} = undef
                    }
                }

                foreach my $value (values %relDirHash) {

                    $combDirHash{$value} = undef;       # Hash in form $hash{$slot} = $value
                }

                foreach my $value (values %relAbbrevHash) {

                    $combDirHash{$value} = undef;       # Hash in form $hash{$slot} = $value
                }

                # Now we can test whether the directions entered by the user are already in use, or
                #   not
                if ($axmud::CLIENT->ivExists('constOppDirHash', $dir)) {

                    $msg = '\'' . $dir . '\' is not available because it is a standard primary'
                                . ' direction';

                } elsif ($abbrev && $axmud::CLIENT->ivExists('constOppDirHash', $abbrev)) {

                    $msg = '\'' . $abbrev . '\' is not available because it is a standard primary'
                                . ' direction';

                } elsif (exists $combDirHash{$dir}) {

                    $msg = '\'' . $dir . '\' is not available because it is already in use as a'
                            . ' custom primary or a secondary direction';

                } elsif ($abbrev && exists $combDirHash{$abbrev}) {

                    $msg = '\'' . $abbrev . '\' is not available because it is already in use as a'
                            . ' custom primary or a secondary direction';
                }

                if ($msg) {

                    # Don't make any changes
                    $self->showMsgDialogue(
                        'Use changes',
                        'error',
                        $msg,
                        'ok',
                    );

                } else {

                    if ($thisDir) {

                        # Find the position of the direction being edited, $thisDir, in the list
                        $count = -1;
                        OUTER: foreach my $otherDir (@dirList) {

                            $count++;

                            if ($otherDir eq $thisDir) {

                                # In the ordered list of secondary directions, replace $thisDir with
                                #   $dir (they might be the same, if the user didn't change the text
                                #   in $entry)
                                splice (@dirList, $count, 1, $dir);
                                last OUTER;
                            }
                        }

                    } else {

                        # Add the new direction to the end of the list
                        push (@dirList, $dir);
                    }

                    # Add (or replace) an entry in the hashes
                    $dirHash{$dir} = $dir;
                    $abbrevHash{$dir} = $abbrev;
                    $oppHash{$dir} = $opp;
                    $oppAbbrevHash{$dir} = $oppAbbrev;
                    $autoHash{$dir} = $auto;

                    if (defined $thisDir && $thisDir ne $dir) {

                        # Remove the old entry
                        delete $dirHash{$thisDir};
                        delete $abbrevHash{$thisDir};
                        delete $oppHash{$thisDir};
                        delete $oppAbbrevHash{$thisDir};
                        delete $autoHash{$thisDir};
                    }

                    # Store the modified list/hashes
                    $self->ivAdd('editHash', 'secondaryDirList', \@dirList);
                    $self->ivAdd('editHash', 'secondaryDirHash', \%dirHash);
                    $self->ivAdd('editHash', 'secondaryAbbrevHash', \%abbrevHash);
                    $self->ivAdd('editHash', 'secondaryOppHash', \%oppHash);
                    $self->ivAdd('editHash', 'secondaryOppAbbrevHash', \%oppAbbrevHash);
                    $self->ivAdd('editHash', 'secondaryAutoHash', \%autoHash);
                    # (We are no longer editing $thisDir)
                    $thisDir = undef;

                    # Refresh the list to show the changes
                    $self->directions2Tab_refreshList($slWidget, (scalar @columnList / 2));

                    # Reset the widgets below the simple list
                    $entry->set_text('');
                    $entry2->set_text('');
                    $entry3->set_text('');
                    $entry4->set_text('');
                    $combo->set_active(0);

                    # Desensitise the widgets below the simple list
                    $self->desensitiseWidgets($entry, $entry2, $entry3, $entry4, $combo);
                }
            }
        });

        # 'Reset list' button
        my $button7 = $self->addButton($table,
            'Reset list', 'Reset the list of secondary directions', undef,
            10, 12, 8, 9);
        $button7->signal_connect('clicked' => sub {

            # Remove the IVs from $self->editHash, so that the IVs in $self->editObj take over
            $self->ivDelete('editHash', 'secondaryDirList');
            $self->ivDelete('editHash', 'secondaryDirHash');
            $self->ivDelete('editHash', 'secondaryAbbrevHash');
            $self->ivDelete('editHash', 'secondaryOppHash');
            $self->ivDelete('editHash', 'secondaryOppAbbrevHash');
            $self->ivDelete('editHash', 'secondaryAutoHash');
            # (We are no longer editing $thisDir)
            $thisDir = undef;

            # Refresh the list to show the changes
            $self->directions2Tab_refreshList($slWidget, (scalar @columnList / 2));

            # Empty the widgets below the simple list
            $entry->set_text('');
            $entry2->set_text('');
            $entry3->set_text('');
            $entry4->set_text('');
            $combo->set_active(0);

            # Desensitise the widgets below the simple list
            $self->desensitiseWidgets($entry, $entry2, $entry3, $entry4, $combo);
        });

        # Widgets below the simple list start desensitised
        $self->desensitiseWidgets($entry, $entry2, $entry3, $entry4, $combo);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub directions2Tab_refreshList {

        # Called by $self->directions2Tab to refresh the GA::Obj::Simple::List
        #
        # Expected arguments
        #   $slWidget   - The GA::Obj::Simple::List
        #   $columns    - The number of columns in the list
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $slWidget, $columns, $check) = @_;

        # Local variables
        my (
            $slWidgetRef, $hashRef,
            @dirList, @dataList,
            %dirHash, %abbrevHash, %oppHash, %oppAbbrevHash, %autoHash,
        );

        # Check for improper arguments
        if (! defined $slWidget || ! defined $columns || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->directions2Tab_refreshList',
                @_,
            );
        }

        # Import the IVs
        if ($self->ivExists('editHash', 'secondaryDirList')) {

            $slWidgetRef = $self->ivShow('editHash', 'secondaryDirList');
            @dirList = @$slWidgetRef;

            $hashRef = $self->ivShow('editHash', 'secondaryDirHash');
            %dirHash = %$hashRef;

            $hashRef = $self->ivShow('editHash', 'secondaryAbbrevHash');
            %abbrevHash = %$hashRef;

            $hashRef = $self->ivShow('editHash', 'secondaryOppHash');
            %oppHash = %$hashRef;

            $hashRef = $self->ivShow('editHash', 'secondaryOppAbbrevHash');
            %oppAbbrevHash = %$hashRef;

            $hashRef = $self->ivShow('editHash', 'secondaryAutoHash');
            %autoHash = %$hashRef;

        } else {

            @dirList = $self->editObj->secondaryDirList;
            %dirHash = $self->editObj->secondaryDirHash;
            %abbrevHash = $self->editObj->secondaryAbbrevHash;
            %oppHash = $self->editObj->secondaryOppHash;
            %oppAbbrevHash = $self->editObj->secondaryOppAbbrevHash;
            %autoHash = $self->editObj->secondaryAutoHash;
        }

        # Compile the simple list data
        foreach my $dir (@dirList) {

            push (@dataList,
                $dirHash{$dir},
                $abbrevHash{$dir},
                $oppHash{$dir},
                $oppAbbrevHash{$dir},
                $autoHash{$dir},
            );
        }

        # Reset the simple list
        $self->resetListData($slWidget, [@dataList], $columns);

        return 1;
    }

    sub directions3Tab {

        # Directions3 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (
            @columnList, @list, @comboList,
            %descripHash,
        );

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->directions3Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _3', $innerNotebook);

        # Relative directions
        $self->addLabel($table, '<b>Relative directions</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of relative directions (which depend on the direction of movement)</i>',
            1, 12, 1, 2);

        # Create the simple list
        @columnList = (
            'Slot #', 'int',
            'Relative dir', 'text',
            'Abbreviation', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 200);      # Fixed height

        # Initialise the list
        $self->directions3Tab_refreshList($slWidget, (scalar @columnList / 2));

        # Add editing widgets
        @list = (
            'Slot 0 (equivalent of north)'      => 0,
            'Slot 1 (equivalent of northeast)'  => 1,
            'Slot 2 (equivalent of east)'       => 2,
            'Slot 3 (equivalent of southeast)'  => 3,
            'Slot 4 (equivalent of south)'      => 4,
            'Slot 5 (equivalent of southwest)'  => 5,
            'Slot 6 (equivalent of west)'       => 6,
            'Slot 7 (equivalent of northwest)'  => 7,
        );

        do {

            my ($descrip, $slot);

            $descrip = shift @list;
            $slot = shift @list;

            push (@comboList, $descrip);
            $descripHash{$descrip} = $slot;

        } until (! @list);

        $self->addLabel($table, 'Slot # (assuming movement from the south)',
            1, 6, 8, 9);
        my $combo = $self->addComboBox($table, undef, \@comboList, '',
            TRUE,               # No 'undef' value used
            6, 12, 8, 9);

        $self->addLabel($table, 'Direction',
            1, 2, 9, 10);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            2, 6, 9, 10);

        $self->addLabel($table, 'Abbreviation',
            6, 8, 9, 10);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'string', undef, undef,
            8, 12, 9, 10);

        # 'Move down' button
        my $button = $self->addButton($table,
            'Add', 'Add this relative direction', undef,
            1, 3, 10, 11);
        $button->signal_connect('clicked' => sub {

            my (
                $slot, $dir, $abbrev, $hashRef, $listRef, $msg,
                @dirList,
                %primDirHash, %primAbbrevHash, %dirHash, %abbrevHash, %oppHash, %oppAbbrevHash,
                %relDirHash, %relAbbrevHash, %combDirHash,
            );

            $slot = $descripHash{$combo->get_active_text()};
            $dir = lc($entry->get_text());
            $abbrev = lc($entry2->get_text());

            if ($self->checkEntryIcon($entry, $entry2)) {

                # $abbrev should be 'undef', rather than an empty string
                if (! $abbrev) {

                    $abbrev = undef;
                }

                # (Definitely need to remove any leading/trailing whitespace, as well as replace two
                #   or more whitespace characters in the middle with a single whitespace character
                $dir = $axmud::CLIENT->trimWhitespace($dir, TRUE);
                if ($abbrev) {

                    $abbrev = $axmud::CLIENT->trimWhitespace($abbrev, TRUE);
                }

                # Need to check that the user hasn't added a primary direction (standard or custom)
                #   or an existing secondary direction
                # Import data from the affected IVs. We can't use ->combDirHash because it hasn't
                #   been updated yet, if any changes have been made in this window
                if ($self->ivExists('editHash', 'primaryDirHash')) {

                    # (If any primary dir IVs have been changed, then they are ALL in ->editHash)
                    $hashRef = $self->ivShow('editHash', 'primaryDirHash');
                    %primDirHash = %$hashRef;
                    $hashRef = $self->ivShow('editHash', 'primaryAbbrevHash');
                    %primAbbrevHash = %$hashRef;

                } else {

                    %primDirHash = $self->editObj->primaryDirHash;
                    %primAbbrevHash = $self->editObj->primaryAbbrevHash;
                }

                if ($self->ivExists('editHash', 'secondaryDirList')) {

                    # (If any secondary dir IVs have been changed, then they are ALL in ->editHash)
                    $listRef = $self->ivShow('editHash', 'secondaryDirList');
                    @dirList = @$listRef;
                    $hashRef = $self->ivShow('editHash', 'secondaryDirHash');
                    %dirHash = %$hashRef;
                    $hashRef = $self->ivShow('editHash', 'secondaryAbbrevHash');
                    %abbrevHash = %$hashRef;
                    $hashRef = $self->ivShow('editHash', 'secondaryOppHash');
                    %oppHash = %$hashRef;
                    $hashRef = $self->ivShow('editHash', 'secondaryOppAbbrevHash');
                    %oppAbbrevHash = %$hashRef;

                } else {

                    @dirList = $self->editObj->secondaryDirList;
                    %dirHash = $self->editObj->secondaryDirHash;
                    %abbrevHash = $self->editObj->secondaryAbbrevHash;
                    %oppHash = $self->editObj->secondaryOppHash;
                    %oppAbbrevHash = $self->editObj->secondaryOppAbbrevHash;
                }

                %relDirHash = $self->editObj->relativeDirHash;
                %relAbbrevHash = $self->editObj->relativeAbbrevHash;

                # (Compose our own temporary %combDirHash, in the form
                #   $combDirHash{custom_primary_dir} = undef
                #   $combDirHash{recognised_secondary_dir} = undef
                foreach my $value (values %primDirHash) {

                    $combDirHash{$value} = undef;       # Hash in form $hash{$key} = $value
                }

                foreach my $value (values %primAbbrevHash) {

                    $combDirHash{$value} = undef;       # Hash in form $hash{$key} = $value
                }

                foreach my $key (keys %dirHash) {

                    $combDirHash{$key} = undef;         # Hash in form $hash{$key} = $key
                }

                foreach my $key (keys %abbrevHash) {

                    $combDirHash{$key} = undef;         # Hash in form $hash{$key} = undef
                }

                foreach my $value (values %relDirHash) {

                    $combDirHash{$value} = undef;       # Hash in form $hash{$slot} = $value
                }

                foreach my $value (values %relAbbrevHash) {

                    $combDirHash{$value} = undef;       # Hash in form $hash{$slot} = $value
                }

                # Now we can test whether the directions entered by the user are already in use, or
                #   not
                if ($axmud::CLIENT->ivExists('constOppDirHash', $dir)) {

                    $msg = '\'' . $dir . '\' is not available because it is a standard primary'
                                . ' direction';

                } elsif ($abbrev && $axmud::CLIENT->ivExists('constOppDirHash', $abbrev)) {

                    $msg = '\'' . $abbrev . '\' is not available because it is a standard primary'
                                . ' direction';

                } elsif (exists $combDirHash{$dir}) {

                    $msg = '\'' . $dir . '\' is not available because it is already in use as a'
                            . ' custom primary or a secondary direction';

                } elsif ($abbrev && exists $combDirHash{$abbrev}) {

                    $msg = '\'' . $abbrev . '\' is not available because it is already in use as a'
                            . ' custom primary or a secondary direction';
                }

                if ($msg) {

                    # Don't make any changes
                    $self->showMsgDialogue(
                        'Add relative direction',
                        'error',
                        $msg,
                        'ok',
                    );

                } else {

                    # Update IVs
                    $self->modifyEditHash_hashIV('relativeDirHash', $slot, $dir);
                    $self->modifyEditHash_hashIV('relativeAbbrevHash', $slot, $abbrev);
                    # Refresh the list to show the changes
                    $self->directions3Tab_refreshList($slWidget, (scalar @columnList / 2));
                }
            }
        });

        # 'Delete direction' button
        my $button2 = $self->addButton($table,
            'Delete', 'Empty the selected slot', undef,
            3, 5, 10, 11);
        $button2->signal_connect('clicked' => sub {

            my $slot;

            ($slot) = $self->getSimpleListData($slWidget, 0);
            if (defined $slot) {

                # Update IVs. The TRUE arguments means to delete the key-value pair
                $self->modifyEditHash_hashIV('relativeDirHash', $slot, undef, TRUE);
                $self->modifyEditHash_hashIV('relativeAbbrevHash', $slot, undef, TRUE);
                # Refresh the list to show the changes
                $self->directions3Tab_refreshList($slWidget, (scalar @columnList / 2));
            }
        });


        # 'Reset list' button
        my $button3 = $self->addButton($table,
            'Reset list', 'Reset the list of relative directions', undef,
            10, 12, 10, 11);
        $button3->signal_connect('clicked' => sub {

            # Remove the IVs from $self->editHash, so that the IVs in $self->editObj take over
            $self->ivDelete('editHash', 'relativeDirHash');
            $self->ivDelete('editHash', 'relativeAbbrevHash');
            # Refresh the list to show the changes
            $self->directions3Tab_refreshList($slWidget, (scalar @columnList / 2));

            # Empty the widgets below the simple list
            $combo->set_active(0);
            $entry->set_text('');
            $entry2->set_text('');
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub directions3Tab_refreshList {

        # Called by $self->directions3Tab to refresh the GA::Obj::Simple::List
        #
        # Expected arguments
        #   $slWidget   - The GA::Obj::Simple::List
        #   $columns    - The number of columns in the list
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $slWidget, $columns, $check) = @_;

        # Local variables
        my (
            @dataList,
            %dirHash, %abbrevHash,
        );

        # Check for improper arguments
        if (! defined $slWidget || ! defined $columns || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->directions3Tab_refreshList',
                @_,
            );
        }

        # Import the IVs
        %dirHash = $self->getEditHash_hashIV('relativeDirHash');
        %abbrevHash = $self->getEditHash_hashIV('relativeAbbrevHash');

        # Compile the simple list data
        for (my $index = 0; $index < 8; $index++) {

            push (@dataList,
                $index,
                $dirHash{$index},
                $abbrevHash{$index},
            );
        }

        # Reset the simple list
        $self->resetListData($slWidget, [@dataList], $columns);

        return 1;
    }

    sub directions4Tab {

        # Directions4 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->directions4Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _4', $innerNotebook);

        # Speedwalking characters
        $self->addLabel($table, '<b>Speedwalking characters</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of speedwalking characters and the movement commands they represent</i>',
            1, 12, 1, 2);

        # Create the simple list
        @columnList = (
            'Char', 'text',
            'Movement command', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10, -1, 220);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'speedDirHash');

        # Editing widgets
        $self->addLabel($table, 'Speedwalking character (a-z)',
            1, 4, 10, 11);
        my $entry = $self->addEntryWithIcon(
            $table,
            undef,
            \&directions4Tab_checkEntry,
            undef,
            undef,
            4, 6, 10, 11,
            1, 1);      # Max width
        $self->addLabel($table, 'Movement command',
            6, 8, 10, 11);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            8, 12, 10, 11);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'speedDirHash',
            11,
            $entry, $entry2,
        );
        $button->signal_connect('clicked' => sub {

            my ($char, $cmd);

            $char = lc($entry->get_text());
            $cmd = $entry2->get_text();

            if ($self->checkEntryIcon($entry, $entry2)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('speedDirHash', $char, $cmd);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'speedDirHash');
                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub directions4Tab_checkEntry {

        # Called by $self->directions4Tab to check the text in the Gtk2::Entry
        #
        # Expected arguments
        #   $text       - The contents of the Gtk2::Entry
        #
        # Return values
        #   'undef' on improper arguments or if $text is invalid
        #   1 if $text is valid

        my ($self, $text, $check) = @_;

        # Check for improper arguments
        if (! defined $text || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->directions4Tab_checkEntry',
                @_,
            );
        }

        if ($text =~ m/^[[:alpha:]]$/) {
            return 1;
        } else {
            return undef;
        }
    }

    sub directions5Tab {

        # Directions5 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (
            $cageObj, $replaceCmd,
            @columnList, @comboList, @sortedList,
        );

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->directions5Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _5', $innerNotebook);

        # Get the current highest-priority command cage
        $cageObj = $self->session->findHighestCage('cmd');

        # Speedwalking modifier characters
        $self->addLabel($table, '<b>Speedwalking modifier characters</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of speedwalking modifier characters and the standard commands they'
            . ' represent</i>',
            1, 12, 1, 2);

        # Create the simple list
        @columnList = (
            'Char', 'text',
            'Standard command', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 200);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'speedModifierHash');

        # Editing widgets
        $self->addLabel($table, 'Modifier character (A-Z)',
            1, 4, 8, 9);
        my $entry = $self->addEntryWithIcon(
            $table,
            undef,
            \&directions3Tab_checkEntry,
            undef,
            undef,
            4, 6, 8, 9,
            1, 1);      # Max width

        $self->addLabel($table, 'Standard command',
            6, 8, 8, 9);

        @comboList = (
            $cageObj->moveCmdList,
            'open_dir', 'close_dir', 'unlock', 'unlock_with', 'lock', 'lock_with', 'pick',
            'pick_with', 'break', 'break_with',
        );
        @sortedList = sort {$a cmp $b} (@comboList);
        my $combo = $self->addComboBox($table, undef, \@sortedList, '',
            TRUE,               # No 'undef' value used
            8, 12, 8, 9);
        # (->signal_connect appears below)

        $self->addLabel($table, 'Current replacement command',
            1, 4, 9, 10);
        my $entry2 = $self->addEntry($table, undef, FALSE,
            4, 12, 9, 10);
        $replaceCmd = $self->session->findCmd($combo->get_active_text());
        if ($replaceCmd) {

            $entry2->set_text($replaceCmd);
        }

        # (->signal_connect from above)
        $combo->signal_connect('changed' => sub {

            $replaceCmd = $self->session->findCmd($combo->get_active_text());
            if (! $replaceCmd) {
                $entry2->set_text('');
            } else {
                $entry2->set_text($replaceCmd);
            }
        });

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'speedModifierHash',
            10,
            $entry, $combo,
        );
        $button->signal_connect('clicked' => sub {

            my ($char, $cmd);

            $char = uc($entry->get_text());
            $cmd = $combo->get_active_text();

            if ($self->checkEntryIcon($entry)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('speedModifierHash', $char, $cmd);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV(
                    $slWidget,
                    scalar (@columnList / 2),
                    'speedModifierHash',
                );

                $self->resetEntryBoxes($entry);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::Exit;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::Generic::EditWin Games::Axmud::Generic::ConfigWin
        Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::Obj::Exit')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        $self->assistedMovesTab();
        $self->doorsTab();
        $self->randomTab();
        $self->ornamentsTab();
        $self->privateDataTab(       # Inherited from GA::Generic::EditWin
            'privateHash',
            '_Private',
            'Private properties hash',
            'Properties created by your own plugins and scripts, but stored in this type of object',
        );

        return 1;
    }

    sub saveChanges {

        # Called by $self->buttonOK and $self->buttonSave
        # Saves any changes made to data stored by the edit object
        # (We don't use the inherited function, because we need to tell Automapper windows to update
        #   themselves - even if $self->editHash is empty)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (
            $wmObj, $randomType, $newListRef,
            @oldList, @newList, @modNewList, @removeList, @addList,
            %hash,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->saveChanges', @_);
        }

        if ($self->editHash) {

            # Import the world model (for convenience)
            $wmObj = $self->session->worldModelObj;

            # Respond, if the user has been changes to ->randomType or ->randomDestList
            if ($self->ivExists('editHash', 'randomType')) {

                $randomType = $self->ivShow('editHash', 'randomType');
                if ($randomType ne $self->editObj->randomType) {

                    # New value for ->randomType, which requires a call to the world model
                    if ($randomType ne 'none') {

                        $wmObj->restoreRandomExit(
                            TRUE,               # Update Automapper windows
                            $self->editObj,
                        );

                    } else {

                        $wmObj->setRandomExit(
                            TRUE,               # Update Automapper windows
                            $self->editObj,
                            $randomType,
                        );
                    }
                }

                $self->ivDelete('editHash', 'randomType');
            }

            if ($self->ivExists('editHash', 'randomDestList')) {

                # Import the existing and new versions of ->randomDestList
                @oldList = $self->editObj->randomDestList;
                $newListRef = $self->ivShow('editHash', 'randomDestList');
                @newList = @$newListRef;

                # Check all the items in @newList, removing any items which are not valid room
                #   model numbers (and removing duplicates)
                foreach my $roomNum (@newList) {

                    if (
                        $wmObj->ivExists('roomModelHash', $roomNum)
                        && (! exists $hash{$roomNum})
                    ) {
                        push (@modNewList, $roomNum);
                        $hash{$roomNum} = undef;
                    }
                }

                # Compile a list of room numbers that were in the old ->randomDestList, but are not
                #   in the new one, and which therefore need to be removed as random exit
                #   destinations (stored in @removeList)
                OUTER: foreach my $oldRoomNum (@oldList) {

                    INNER: foreach my $newRoomNum (@newList) {

                        if ($oldRoomNum == $newRoomNum) {

                            next OUTER;
                        }
                    }

                    # $oldRoomNum is missing from @newList
                    push (@removeList, $oldRoomNum);
                }

                # Compile a list of room numbers that are in the new ->randomDestList, but were not
                #   in the old one, and which therefore need to be added as random exit
                #   destinations (stored in @addList)
                OUTER: foreach my $newRoomNum (@modNewList) {

                    INNER: foreach my $oldRoomNum (@oldList) {

                        if ($newRoomNum == $oldRoomNum) {

                            next OUTER;
                        }
                    }

                    # $newRoomNum is missing from @oldList
                    push (@addList, $newRoomNum);
                }

                # The rooms in @removeList must be informed that they are no longer destination
                #   rooms for a random exit
                foreach my $roomNum (@removeList) {

                    my $roomObj = $wmObj->ivShow('modelHash', $roomNum);

                    $wmObj->addRandomDestination($roomObj, $self->editObj);
                }

                # The rooms in @addList must be informed that they are now destination rooms for a
                #   random exit
                foreach my $roomNum (@addList) {

                    my $roomObj = $wmObj->ivShow('modelHash', $roomNum);

                    $wmObj->removeRandomDestination($roomObj, $self->editObj);
                }

                # Now set the new value of ->randomDestList. The order matters, so we don't sort
                #   the list
                $self->editObj->ivPoke('randomDestList', @modNewList);

                $self->ivDelete('editHash', 'randomDestList');
            }

#            if ($self->ivExists('editHash', 'altDir')) {
#
#                if ($self->ivShow('editHash', 'altDir') eq '') {
#
#                    # Use 'undef' rather than an empty string
#                    $self->editObj->ivUndef('altDir');
#
#                } else {
#
#                    $self->editObj->ivPoke('altDir', $self->ivShow('editHash', 'altDir'));
#                }
#
#                $self->ivDelete('editHash', 'altDir');
#            }

            # Store the changes the user has made
            foreach my $key ($self->ivKeys('editHash')) {

                $self->editObj->{$key} = $self->ivShow('editHash', $key);
            }

            # The changes can now be cleared
            $self->ivEmpty('editHash');

            # Mark the object's corresponding file object as needing to be saved, if it exists
            if ($self->editObj->_parentFile) {

                $self->editObj->doModify('saveChanges');
            }

            # Update the current session's object viewer window, if it is open
            if ($self->session->viewerWin) {

                $self->session->viewerWin->updateNotebook();
            }
        }

        # For objects stored in the exit model (first checking that they are still there)...
        if (
            $self->editObj->modelFlag
            && $self->session->worldModelObj->ivExists('exitModelHash', $self->editObj->number)
            && $self->session->worldModelObj->ivShow('exitModelHash', $self->editObj->number)
                eq $self->editObj
        ) {
            # Mark this room to be redrawn, if it is visible
            $self->session->worldModelObj->updateMaps('exit', $self->editObj);
        }

        return 1;
    }

    # Notebook tabs

    sub nameTab {

        # Name tab - called by $self->setupNotebook
        #
        # Expected arguments
        #   $table  - The Gtk2::Table
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $table, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->nameTab', @_);
        }

#       # Tab setup (already created by the calling function)
#       my ($vBox, $table) = $self->addTab('_Name', $self->notebook);

        # Left column
        $self->addLabel($table, '<b>General properties</b>',
            0, 6, 0, 1);
        $self->addLabel($table, 'Name',
            1, 3, 1, 2);
        $self->addEntry($table, 'name', FALSE,
            3, 6, 1, 2);
        $self->addLabel($table, 'Exit model #',
            1, 3, 2, 3);
        $self->addEntry($table, 'number', FALSE,
            3, 6, 2, 3);
        $self->addLabel($table, 'Parent room #',
            1, 3, 3, 4);
        $self->addEntry($table, 'parent', FALSE,
            3, 6, 3, 4);

        $self->addLabel($table, 'Exit type',
            1, 3, 4, 5);
        $self->addEntry($table, 'exitType', FALSE,
            3, 6, 4, 5);

        $self->addLabel($table, 'Nominal direction',
            1, 3, 5, 6);
        $self->addEntry($table, 'dir', FALSE,
            3, 6, 5, 6);
        $self->addLabel($table, 'Alternative direction(s)',
            1, 3, 6, 7);
        $self->addEntry($table, 'altDir', FALSE,
            3, 6, 6, 7);
        $self->addLabel($table, 'Mapped as direction',
            1, 3, 7, 8);
        my $entry = $self->addEntry($table, undef, FALSE,
            3, 6, 7, 8);
        if ($self->editObj->mapDir) {
            $entry->set_text($self->editObj->mapDir);
        } else {
            $entry->set_text('unallocatable');
        }
        $self->addLabel($table, 'Draw mode',
            1, 3, 8, 9);
        $self->addEntry($table, 'drawMode', FALSE,
            3, 6, 8, 9);
        $self->addLabel($table, 'Destination room #',
            1, 3, 9, 10);
        $self->addEntry($table, 'destRoom', FALSE,
            3, 6, 9, 10);
        $self->addLabel($table, 'Twin exit #',
            1, 3, 10, 11);
        $self->addEntry($table, 'twinExit', FALSE,
            3, 6, 10, 11);
        $self->addLabel($table, 'Shadow exit #',
            1, 3, 11, 12);
        $self->addEntry($table, 'shadowExit', FALSE,
            3, 6, 11, 12);


        # Right column
        $self->addLabel($table, 'Broken exit flag',
            7, 11, 1, 2);
        $self->addCheckButton($table, 'brokenFlag', FALSE,
            11, 12, 1, 2, 1, 0.5);
        $self->addLabel($table, 'Bent broken exit flag',
            7, 11, 2, 3);
        $self->addCheckButton($table, 'bentFlag', FALSE,
            11, 12, 2, 3, 1, 0.5);
        $self->addLabel($table, 'Region exit flag',
            7, 11, 3, 4);
        $self->addCheckButton($table, 'regionFlag', FALSE,
            11, 12, 3, 4, 1, 0.5);
        $self->addLabel($table, 'Region super-exit flag',
            7, 11, 4, 5);
        $self->addCheckButton($table, 'superFlag', FALSE,
            11, 12, 4, 5, 1, 0.5);
        $self->addLabel($table, 'One-way exit flag',
            7, 11, 5, 6);
        $self->addCheckButton($table, 'oneWayFlag', FALSE,
            11, 12, 5, 6, 1, 0.5);
        $self->addLabel($table, 'Incoming 1-way dir',
            7, 11, 6, 7);
        $self->addEntry($table, 'oneWayDir', FALSE,
            9, 12, 6, 7);
        $self->addLabel($table, 'Retracing exit flag',
            7, 11, 7, 8);
        $self->addCheckButton($table, 'retraceFlag', FALSE,
            11, 12, 7, 8, 1, 0.5);
        $self->addLabel($table, 'Hidden exit flag',
            7, 11, 8, 9);
        $self->addCheckButton($table, 'hiddenFlag', FALSE,
            11, 12, 8, 9);
        $self->addLabel($table, 'Exit tag',
            7, 9, 9, 10);
        $self->addEntry($table, 'exitTag', FALSE,
            9, 12, 9, 10);
        $self->addLabel($table, 'Exit tag X offset',
            7, 9, 10, 11);
        $self->addEntry($table, 'exitTagXOffset', FALSE,
            9, 12, 10, 11);
        $self->addLabel($table, 'Exit tag Y offset',
            7, 9, 11, 12);
        $self->addEntry($table, 'exitTagYOffset', FALSE,
            9, 12, 11, 12);

        return 1;
    }

    sub assistedMovesTab {

        # Assisted Moves tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (@columnList, @profList, @sortedList);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->assistedMovesTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('_Assisted moves', $self->notebook);

        # Assisted moves
        $self->addLabel($table, '<b>Assisted moves</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>Command sequences used in assisted moves for this exit, e.g.'
            . ' \'push button;north\'</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Profile', 'text',
            'Command sequence', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 230);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'assistedHash');

        # Get a sorted list of all non-world profiles...
        foreach my $profObj ($self->session->ivValues('profHash')) {

            if ($profObj->category ne 'world') {

                push (@profList, $profObj->name);
            }
        }

        @sortedList = sort {lc($a) cmp lc($b)} (@profList);
        # ...and put the current world profile at the top of the list
        unshift (@sortedList, $self->session->currentWorld->name);

        # Add entries/comboboxes for adding world model objects
        $self->addLabel($table, 'Profile',
            1, 3, 8, 9);
        my $combo = $self->addComboBox($table, undef, \@sortedList, '',
            TRUE,               # No 'undef' value used
            3, 6, 8, 9);

        $self->addLabel($table, 'Command sequence',
            1, 3, 9, 10);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            3, 12, 9, 10);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'assistedHash',
            10,
            $entry,
        );
        $button->signal_connect('clicked' => sub {

            my ($profName, $sequence);

            $profName = $combo->get_active_text();
            $sequence = $entry->get_text();

            if ($self->checkEntryIcon($entry)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('assistedHash', $profName, $sequence);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'assistedHash');
                $self->resetEntryBoxes($entry);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub doorsTab {

        # Doors tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (@columnList, @comboList);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->doorsTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('_Doors', $self->notebook);

        # Door commands
        $self->addLabel($table, '<b>Door commands</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>Command sequences used to get through this exit\'s door (if any)</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Door type', 'text',
            'Command sequence', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 230);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'doorHash');

        # Add entries/comboboxes for adding world model objects
        $self->addLabel($table, 'Profile',
            1, 3, 8, 9);
        @comboList = qw(break pick unlock open close lock);
        my $combo = $self->addComboBox($table, undef, \@comboList, '',
            TRUE,               # No 'undef' value used
            3, 6, 8, 9);

        $self->addLabel($table, 'Command sequence',
            1, 3, 9, 10);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            3, 12, 9, 10);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'assistedHash',
            10,
            $entry,
        );
        $button->signal_connect('clicked' => sub {

            my ($type, $sequence);

            $type = $combo->get_active_text();
            $sequence = $entry->get_text();

            if ($self->checkEntryIcon($entry)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('doorHash', $type, $sequence);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'doorHash');
                $self->resetEntryBoxes($entry);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub randomTab {

        # Random tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->randomTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('_Random', $self->notebook);

        # Left column
        $self->addLabel($table, '<b>Random exit type</b>',
            0, 6, 0, 1);

        my ($group, $radioButton) = $self->addRadioButton(
            $table, undef, '\'none\'', 'randomType',
            'none',         # IV set to this value when toggled
            TRUE,           # Sensitised widget
            1, 3, 1, 2);
        $self->addLabel($table, 'Not a random exit',
            3, 6, 1, 2);

        # Right column
        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'same_region\'', 'randomType', 'same_region', TRUE,
            7, 9, 1, 2);
        $self->addLabel($table, 'Leads to random room in same region',
            9, 12, 1, 2);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'any_region\'', 'randomType', 'any_region', TRUE,
            7, 9, 2, 3);
        $self->addLabel($table, 'Leads to random room anywhere',
            9, 12, 2, 3);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'temp_region\'', 'randomType', 'temp_region', TRUE,
            7, 9, 3, 4);
        $self->addLabel($table, 'Creates new room in temporary region',
            9, 12, 3, 4);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'room_list\'', 'randomType', 'room_list', TRUE,
            7, 9, 4, 5);
        $self->addLabel($table, 'Leads to random room from this list:',
            9, 12, 4, 5);

        $self->addTextView($table, 'randomDestList', TRUE,
            9, 12, 5, 11,
            undef, undef, undef, undef,
            -1, 220);               # Fixed width/height

        $self->addLabel($table, '<i>(Add a list of room numbers, one per line)</i>',
            9, 12, 11, 12);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub ornamentsTab {

        # Ornaments tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->ornamentsTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Orna_ments', $self->notebook);

        # Left column
        $self->addLabel($table, '<b>Exit ornaments</b>',
            0, 6, 0, 1);

        $self->addLabel($table, 'Breakable door',
            1, 5, 1, 2);
        my $button = $self->addCheckButton($table, undef, FALSE,
            5, 6, 1, 2);
        $self->addLabel($table, 'Pickable lock',
            1, 5, 2, 3);
        my $button2 = $self->addCheckButton($table, undef, FALSE,
            5, 6, 2, 3);
        $self->addLabel($table, 'Lockable door',
            1, 5, 3, 4);
        my $button3 = $self->addCheckButton($table, undef, FALSE,
            5, 6, 3, 4);
        $self->addLabel($table, 'Openable door',
            1, 5, 4, 5);
        my $button4 = $self->addCheckButton($table, undef, FALSE,
            5, 6, 4, 5);
        $self->addLabel($table, 'Exit impassable',
            1, 5, 5, 6);
        my $button5 = $self->addCheckButton($table, undef, FALSE,
            5, 6, 5, 6);
        $self->addLabel($table, 'Mystery exit',
            1, 5, 6, 7);
        my $button6 = $self->addCheckButton($table, undef, FALSE,
            5, 6, 6, 7);

        if ($self->editObj->exitOrnament eq 'break') {
            $button->set_active(TRUE);
        } elsif ($self->editObj->exitOrnament eq 'pick') {
            $button2->set_active(TRUE);
        } elsif ($self->editObj->exitOrnament eq 'lock') {
            $button3->set_active(TRUE);
        } elsif ($self->editObj->exitOrnament eq 'open') {
            $button4->set_active(TRUE);
        } elsif ($self->editObj->exitOrnament eq 'impass') {
            $button5->set_active(TRUE);
        } elsif ($self->editObj->exitOrnament eq 'mystery') {
            $button6->set_active(TRUE);
        }

        $self->addLabel(
            $table,
            '<b>Exit information</b>',
            0, 6, 7, 8);
        $self->addLabel(
            $table,
            '<i>(e.g. further details about the destination room)</i>',
            1, 6, 8, 9);
        $self->addEntryWithButton($table, 'exitInfo', TRUE,
            1, 6, 9, 10);

        # Right column
        $self->addLabel($table, '<b>Exit current state</b>',
            6, 12, 0, 1);

        my ($group, $radioButton) = $self->addRadioButton(
            $table, undef, '\'normal\'', 'exitState',
            'normal',              # IV set to this value when toggled
            FALSE,          # Insensitive widget
            7, 9, 1, 2);
        $self->addLabel($table, 'Exit passable/state unknown',
            9, 12, 1, 2);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'open\'', 'exitState', 'open', FALSE,
            7, 9, 2, 3);
        $self->addLabel($table, 'Exit open',
            9, 12, 2, 3);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'closed\'', 'exitState', 'closed', FALSE,
            7, 9, 3, 4);
        $self->addLabel($table, 'Exit closed',
            9, 12, 3, 4);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'locked\'', 'exitState', 'locked', FALSE,
            7, 9, 4, 5);
        $self->addLabel($table, 'Exit locked',
            9, 12, 4, 5);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'secret\'', 'exitState', 'secret', FALSE,
            7, 9, 5, 6);
        $self->addLabel($table, 'Secret exit',
            9, 12, 5, 6);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'secret_open\'', 'exitState', 'secret_open', FALSE,
            7, 9, 6, 7);
        $self->addLabel($table, 'Secret open exit',
            9, 12, 6, 7);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'secret_closed\'', 'exitState', 'secret_closed', FALSE,
            7, 9, 7, 8);
        $self->addLabel($table, 'Secret closed exit',
            9, 12, 7, 8);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'secret_locked\'', 'exitState', 'secret_locked', FALSE,
            7, 9, 8, 9);
        $self->addLabel($table, 'Secret locked exit',
            9, 12, 8, 9);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'impass\'', 'exitState', 'impass', FALSE,
            7, 9, 9, 10);
        $self->addLabel($table, 'Exit impassable',
            9, 12, 9, 10);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'dark\'', 'exitState', 'dark', FALSE,
            7, 9, 10, 11);
        $self->addLabel($table, 'Destination room is dark',
            9, 12, 10, 11);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'danger\'', 'exitState', 'danger', FALSE,
            7, 9, 11, 12);
        $self->addLabel($table, 'Destination room is dangerous',
            9, 12, 11, 12);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'emphasis\'', 'exitState', 'emphasis', FALSE,
            7, 9, 12, 13);
        $self->addLabel($table, 'Exit emphasised',
            9, 12, 12, 13);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'other\'', 'exitState', 'other', FALSE,
            7, 9, 13, 14);
        $self->addLabel($table, 'Other state',
            9, 12, 13, 14);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::Generic::Interface;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::Generic::EditWin Games::Axmud::Generic::ConfigWin
        Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

#   sub checkEditObj {}         # Inherited from GA::Generic::ConfigWin

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

    sub setupNotebook {

        # Called by $self->winEnable
        # Creates the first tab for the notebook. The remaining tabs are created by
        #   $self->expandNotebook
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setupNotebook', @_);
        }

        # Tab setup, using the standard table size
        my ($vBox, $table) = $self->addTab('_Name', $self->notebook);

        # Unusual step - create a hash IV for all 'edit' windows inheriting this one. Each key-value
        #   pair corresponds to a key-value pair in GA::Interface::Trigger->attribHash (etc)
        $self->{attribHash} = {};
        # Also create two list IVs which store the list of keys which will be saved in
        #   GA::Interface::Trigger->beforeHash or ->afterHash
        $self->{beforeList} = [];
        $self->{afterList} = [];
        # $self-saveChanges needs to know when those list IVs have been modified, so we need two
        #   flags, as well. (If the lists are emptied, that counts as a change, so we can't simply
        #   test whether ->beforeList and ->afterList are not empty)
        $self->{beforeFlag} = FALSE;
        $self->{afterFlag} = FALSE;

        # Set up the rest of the tab
        $self->nameTab($table);

        # Set up the remaining tabs
        $self->expandNotebook();

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

#   sub expandNotebook {}       # Inherited from GA::Generic::ConfigWin

    sub saveChanges {

        # Called by $self->buttonOK and $self->buttonSave
        # Saves any changes made to data stored by the edit object (which can be three of the four
        #   standard attributes, 'stimulus', 'response' and 'enabled'). Also saves any modified
        #   attributes stored in $self->attribHash
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (%attribHash, %beforeHash, %afterHash);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->saveChanges', @_);
        }

        if ($self->editHash || $self->attribHash || $self->beforeFlag || $self->afterFlag) {

            # Transfer the modified non-standard attributes $self->attribHash to the local variable
            #   %attribHash. At the same time, deal with any 'dummy' attributes stored in
            #   $self->attribHash
            foreach my $attrib ($self->ivKeys('attribHash')) {

                my $value = $self->ivShow('attribHash', $attrib);

                # Attributes beginning with an underline are 'dummy' attributes, used to help set
                #   the values of other attributes (e.g. '_substr_num', used to set 'style_mode')
                if (substr($attrib, 0, 1) ne '_') {

                    if ($attrib eq 'style_mode' && $value == -3) {

                        # The attribute's true value is stored in the dummy attribute, '_substr_num'
                        $attribHash{$attrib} = $self->ivShow('attribHash', '_substr_num');

                    } else {

                        # Normal attribute
                        $attribHash{$attrib} = $value;
                    }
                }
            }

            # Add any modified standard attributes to %attribHash. $self->editHash can contain the
            #   keys 'stimulus', 'response' and 'enabled'
            foreach my $attrib ($self->ivKeys('editHash')) {

                if ($attrib eq 'stimulus' || $attrib eq 'response') {
                    $attribHash{$attrib} = $self->ivShow('editHash', $attrib);
                } elsif ($attrib eq 'enabledFlag') {
                    $attribHash{'enabled'} = $self->ivShow('editHash', 'enabledFlag');
                }
            }

            if (%attribHash) {

                # Store the modified standard and non-standard attributes
                $self->editObj->modifyAttribs($self->session, %attribHash);

                # Update every active interface based on the inactive interface, $self->editObj. Do
                #   the update in every affected session, except this one
                foreach my $otherSession ($axmud::CLIENT->listSessions()) {

                    if (
                        $otherSession->currentWorld eq $self->session->currentWorld
                        && $otherSession ne $self->session
                    ) {
                        $otherSession->updateInterfaces($self->editObj, %attribHash);
                    }
                }

                # Now do the update for active interfaces in this session
                $self->session->updateInterfaces($self->editObj, %attribHash);
            }

            # If $self->beforeList or $self->afterList have been modified, store those changes, too
            if ($self->beforeFlag) {

                foreach my $item ($self->beforeList) {

                    $beforeHash{$item} = undef;
                }

                $self->editObj->ivPoke('beforeHash', %beforeHash);
            }

            if ($self->afterFlag) {

                foreach my $item ($self->afterList) {

                    $afterHash{$item} = undef;
                }

                $self->editObj->ivPoke('afterHash', %afterHash);
            }

            # The local IVs can now be reset
            $self->ivEmpty('editHash');
            $self->ivEmpty('attribHash');
            $self->ivEmpty('beforeList');
            $self->ivEmpty('afterList');
            $self->ivPoke('beforeFlag', FALSE);
            $self->ivPoke('afterFlag', FALSE);

            # Mark the object's corresponding file object as needing to be saved, if it exists
            if ($self->editObj->_parentFile) {

                $self->editObj->doModify('saveChanges');
            }

            # Update the current session's object viewer window, if it is open
            if ($self->session->viewerWin) {

                $self->session->viewerWin->updateNotebook();
            }
        }

        return 1;
    }

    # Notebook tabs used by more than one interface 'edit' window

    sub nameTab {

        # Name tab - called by $self->setupNotebook
        #
        # Expected arguments
        #   $table  -   The Gtk2::Table
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $table, $check) = @_;

        # Local variables
        my (
            $interfaceModelObj,
            @comboList,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->nameTab', @_);
        }

#       # Tab setup (already created by the calling function)
#       my ($vBox, $table) = $self->addTab('_Name', $self->notebook);

        # Get the corresponding interface model object
        $interfaceModelObj = $axmud::CLIENT->ivShow('interfaceModelHash', $self->editObj->category);

        # Standard interface attributes
        $self->addLabel($table, '<b>Standard interface attributes</b>',
            0, 12, 0, 1);
        $self->addLabel($table, 'Name',
            1, 3, 1, 2);
        $self->addEntry($table, 'name', FALSE,
            3, 6, 1, 2);
        $self->addLabel($table, 'Category',
            1, 3, 2, 3);
        $self->addEntry($table, 'category', FALSE,
            3, 6, 2, 3);

        $self->addLabel($table, 'Enabled',
            7, 11, 1, 2);
        $self->addCheckButton($table, 'enabledFlag', TRUE,
            11, 12, 1, 2);

        $self->addLabel($table, 'Stimulus <i>(' . $interfaceModelObj->stimulusName . ')</i>',
            1, 3, 3, 4);
        # For hooks, use a combo; for everything else, use an entry
        if ($self->editObj->category eq 'trigger' || $self->editObj->category eq 'alias') {

            $self->addEntryWithIcon($table, 'stimulus', 'regex', 1, undef,
                3, 12, 3, 4);

        } elsif ($self->editObj->category eq 'macro') {

            $self->addEntryWithIcon($table, 'stimulus', 'string', 1, undef,
                3, 12, 3, 4);

        } elsif ($self->editObj->category eq 'timer') {

            $self->addEntryWithIcon($table, 'stimulus', \&nameTab_checkInterval, undef, undef,
                3, 12, 3, 4);

        } elsif ($self->editObj->category eq 'hook') {

            @comboList = sort {$a cmp $b} ($interfaceModelObj->ivKeys('hookEventHash'));
            $self->addComboBox($table, 'stimulus', \@comboList, '',
                TRUE,               # No 'undef' value used
                3, 6, 3, 4);
        }

        # For triggers, the response can be any of 'instruction', 'pattern', 'substitution'
        $self->addLabel($table, 'Response <i>(' . $interfaceModelObj->responseName . ')</i>',
            1, 3, 4, 5);
        $self->addEntryWithIcon($table, 'response', 'string', 1, undef,
            3, 12, 4, 5);
        if ($interfaceModelObj->category eq 'trigger') {

            $self->addLabel(
                $table,
                '<i>NB Splitter triggers: the response is a pattern.</i>',
                3, 12, 5, 6);
            $self->addLabel(
                $table,
                '<i>NB Rewriter triggers: the response is a substitution</i>',
                3, 12, 6, 7);
        }

#       # Tab complete (handled by the calling function)
#       $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub nameTab_checkInterval {

        # Called by $self->nameTab to check a timer interval is valid
        #
        # Expected arguments
        #   $value      - The value to check, should be a number (minimum value 0.01), or a 24-hour
        #                   clock time in the form HH::MM
        #
        # Return values
        #   'undef' on improper arguments or if $value is invalid
        #   1 if $value is valid

        my ($self, $value, $check) = @_;

        # Check for improper arguments
        if (! defined $value || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->nameTab_checkInterval',
                @_,
            );
        }

        if (
            ! $axmud::CLIENT->floatCheck($value, 0.1)
            && ! ($value =~ m/^([01]?[0-9]|2[0-3]):[0-5][0-9]$/)
            && ! ($value =~ m/^99\:[0-5][0-9]$/)
        ) {
            return undef;
        } else {
            return 1;
        }
    }

    sub beforeAfterTab {

        # BeforeAfter tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my ($single, $plural);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->beforeAfterTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('_Before / after', $self->notebook);

        $single = $self->editObj->category;
        if ($single eq 'alias') {
            $plural = 'aliases';
        } else {
            $plural = $single . 's';
        }

        # Door patterns
        $self->addLabel($table, '<b>Before ' . $plural . '</b>',
            0, 12, 0, 2);
        $self->addLabel(
            $table,
            '<i>When this ' . $single . ' becomes active, it is tested before the following '
            . $plural . '</i>',
            1, 12, 2, 4);
        $self->beforeAfterTab_addTextView($table, 'before', 4);

        # Locked door patterns
        $self->addLabel($table, '<b>After ' . $plural . '</b>',
            0, 12, 6, 8);
        $self->addLabel(
            $table,
            '<i>When this ' . $single . ' becomes active, it is tested after the following '
            . $plural . '</i>',
            1, 12, 8, 10);
        $self->beforeAfterTab_addTextView($table, 'after', 10);

        # (Explanatory message for triggers only)
        if ($single eq 'trigger') {

            $self->addLabel(
                $table,
                '<i>(NB All splitter triggers are tested before any other kind of trigger is'
                . ' tested)</i>',
                1, 12, 12, 14);
        }

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub beforeAfterTab_addTextView {

        # Called by $self->beforeAfterTab to create a Gtk2::TextView whose contents are treated as
        #   lists, and saved in $self->beforeList or $self->afterList
        # (We don't use the standard GA::Generic::EditWin->addTextView because we need these
        #   textviews to create lists, which are saved as hashes in $self->editObj)
        #
        # Expected arguments
        #   $table      - The Gtk2::Table
        #   $type       - 'before' or 'after'
        #   $row        - The Gtk2::Table row where the textview is drawn
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $table, $type, $row, $check) = @_;

        # Local variables
        my @sortedList;

        # Check for improper arguments
        if (! defined $table || ! defined $type || ! defined $row || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->beforeAfterTab_addTextView',
                @_,
            );
        }

        # Create the textview
        my $textView = $self->addTextView($table, undef, TRUE,
            1, 12, $row, ($row + 2),
            TRUE, TRUE, TRUE, FALSE,    # Treat as list, remove empty lines, do remove whitespace
            -1, 140);                   # Fixed height
        my $buffer = $textView->get_buffer();

        # Display a list of the keys in $self->editObj->beforeHash / ->afterHash (sort them
        #   alphabetically, for good measure)
        @sortedList = sort {lc($a) cmp lc($b)} ($self->editObj->ivKeys($type . 'Hash'));
        $buffer->set_text(join("\n", @sortedList));

        # Store any changes the user makes in $self->$localIV
        $buffer->signal_connect('changed' => sub {

            my (
                $text,
                @list, @finalList,
            );

            $text = $axmud::CLIENT->desktopObj->bufferGetText($buffer);

            # Split the contents of the textview into a list of lines, separated by newline
            #   characters
            @list = split("\n", $text);
            # Remove any empty lines and leading/trailing whitespace, if allowed
            foreach my $line (@list) {

                $line =~ s/^\s+//;  # Remove leading whitespace
                $line =~ s/\s+$//;  # Remove trailing whitespace

                if ($line) {

                    push (@finalList, $line);
                }
            }

            # Store the contents, temporarily, so that they're available to $self->saveChanges
            $self->ivPoke($type . 'List', @finalList);
            # Set a flag, to mark that changes have been made
            $self->ivPoke($type . 'Flag', TRUE);
        });

        return $textView;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get

    sub attribHash
        { my $self = shift; return %{$self->{attribHash}}; }
    sub beforeList
        { my $self = shift; return @{$self->{beforeList}}; }
    sub afterList
        { my $self = shift; return @{$self->{afterList}}; }
    sub beforeFlag
        { $_[0]->{beforeFlag} }
    sub afterFlag
        { $_[0]->{afterFlag} }
}

{ package Games::Axmud::EditWin::Interface::Active;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::Generic::EditWin Games::Axmud::Generic::ConfigWin
        Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::Interface::Active')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

    sub setupNotebook {

        # Called by $self->winEnable
        # Creates the first tab for the notebook. The remaining tabs are created by
        #   $self->expandNotebook
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setupNotebook', @_);
        }

        # Tab setup, using the standard table size
        my ($vBox, $table) = $self->addTab('_Name', $self->notebook);

        # Unusual step - create a hash IV for all 'edit' windows inheriting this one. Each key-value
        #   pair corresponds to a key-value pair in GA::Interface::Trigger->attribHash (etc)
        $self->{attribHash} = {};

        # Set up the rest of the tab
        $self->nameTab($table);

        # Set up the remaining tabs
        $self->expandNotebook();

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my $func;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        $func = $self->editObj->category . 'AttributesTab';     # e.g. ->triggerAttributesTab
        $self->$func();

        $self->propertiesTab();

        return 1;
    }

    sub saveChanges {

        # Called by $self->buttonOK and $self->buttonSave. Function based on
        #   GA::EditWin::Generic::Interface->saveChanges
        # Saves any changes made to data stored by the edit object (which can be three of the four
        #   standard attributes, 'stimulus', 'response' and 'enabled'). Also saves any modified
        #   attributes stored in $self->attribHash
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my %attribHash;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->saveChanges', @_);
        }

        if ($self->editHash || $self->attribHash) {

            # Transfer the modified non-standard attributes $self->attribHash to the local variable
            #   %attribHash. At the same time, deal with any 'dummy' attributes stored in
            #   $self->attribHash
            foreach my $attrib ($self->ivKeys('attribHash')) {

                my $value = $self->ivShow('attribHash', $attrib);

                # Attributes beginning with an underline are 'dummy' attributes, used to help set
                #   the values of other attributes (e.g. '_substr_num', used to set 'style_mode')
                if (substr($attrib, 0, 1) ne '_') {

                    if ($attrib eq 'style_mode' && $value == -3) {

                        # The attribute's true value is stored in the dummy attribute, '_substr_num'
                        $attribHash{$attrib} = $self->ivShow('attribHash', '_substr_num');

                    } else {

                        # Normal attribute
                        $attribHash{$attrib} = $value;
                    }
                }
            }

            # Add any modified standard attributes to %attribHash. $self->editHash can contain the
            #   keys 'stimulus', 'response' and 'enabled'
            foreach my $attrib ($self->ivKeys('editHash')) {

                if ($attrib eq 'stimulus' || $attrib eq 'response') {
                    $attribHash{$attrib} = $self->ivShow('editHash', $attrib);
                } elsif ($attrib eq 'enabledFlag') {
                    $attribHash{'enabled'} = $self->ivShow('editHash', 'enabledFlag');
                }
            }

            # Store the modified standard and non-standard attributes
            $self->editObj->modifyAttribs($self->session, %attribHash);

            # The local hashes can now be emptied
            $self->ivEmpty('editHash');
            $self->ivEmpty('attribHash');

            # Mark the object's corresponding file object as needing to be saved, if it exists
            if ($self->editObj->_parentFile) {

                $self->editObj->doModify('saveChanges');
            }

            # Update the current session's object viewer window, if it is open
            if ($self->session->viewerWin) {

                $self->session->viewerWin->updateNotebook();
            }
        }

        return 1;
    }

    # Notebook tabs

    sub nameTab {

        # Name tab - called by $self->setupNotebook
        #
        # Expected arguments
        #   $table  - The Gtk2::Table
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $table, $check) = @_;

        # Local variables
        my (
            $interfaceModelObj,
            @comboList,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->nameTab', @_);
        }

#       # Tab setup (already created by the calling function)
#       my ($vBox, $table) = $self->addTab('_Name', $self->notebook);

        # Settings (left column)
        $self->addLabel($table, '<b>Settings</b>',
            0, 12, 0, 1);
        $self->addLabel($table, 'Name',
            1, 3, 1, 2);
        $self->addEntry($table, 'name', FALSE,
            3, 12, 1, 2);
        $self->addLabel($table, 'Number',
            1, 3, 2, 3);
        $self->addEntry($table, 'number', FALSE,
            3, 6, 2, 3);
        $self->addLabel($table, 'Category',
            1, 3, 3, 4);
        $self->addEntry($table, 'category', FALSE,
            3, 6, 3, 4);

        # (Right column)
        $self->addLabel($table, '(Inactive) parent',
            7, 9, 2, 3);
        my $entry = $self->addEntry($table, undef, FALSE,
            9, 12, 2, 3);
        if ($self->editObj->parent) {
            $entry->set_text($self->editObj->parent->name);
        } else {
            $entry->set_text('n/a');
        }

        $self->addLabel($table, 'Type',
            7, 9, 3, 4);
        my $entry2 = $self->addEntry($table, undef, FALSE,
            9, 12, 3, 4);
        if ($self->editObj->indepFlag) {
            $entry2->set_text('independent');
        } else {
            $entry2->set_text('dependent');
        }

        # Standard attributes
        $self->addLabel($table, '<b>Standard attributes</b>',
            0, 12, 4, 5);
        $interfaceModelObj = $axmud::CLIENT->ivShow('interfaceModelHash', $self->editObj->category);

        $self->addLabel($table, 'Stimulus <i>(' . $interfaceModelObj->stimulusName . ')</i>',
            1, 3, 5, 6);
        # For hooks, use a combo; for everything else, use an entry
        if ($self->editObj->category eq 'trigger' || $self->editObj->category eq 'alias') {

            $self->addEntryWithIcon($table, 'stimulus', 'regex', 1, undef,
                3, 12, 5, 6);

        } elsif ($self->editObj->category eq 'macro') {

            $self->addEntryWithIcon($table, 'stimulus', 'string', 1, undef,
                3, 12, 5, 6);

        } elsif ($self->editObj->category eq 'timer') {

            $self->addEntryWithIcon($table, 'stimulus', 'float', 0.01, undef,
                3, 12, 5, 6);

        } elsif ($self->editObj->category eq 'hook') {

            @comboList = sort {$a cmp $b} ($interfaceModelObj->ivKeys('hookEventHash'));
            $self->addComboBox($table, 'stimulus', \@comboList, '',
                TRUE,               # No 'undef' value used
                3, 6, 5, 6);
        }

        $self->addLabel($table, 'Response <i>(' . $interfaceModelObj->responseName . ')</i>',
            1, 3, 6, 7);
        $self->addEntryWithIcon($table, 'response', 'string', 1, undef,
            3, 12, 6, 7);
        $self->addLabel($table, 'Enabled',
            1, 11, 7, 8);
        $self->addCheckButton($table, 'enabledFlag', TRUE,
            11, 12, 7, 8);

        # Independent interface IVs
        $self->addLabel($table, '<b>Independent interface IVs</b>',
            0, 6, 8, 9);
        $self->addLabel($table, 'Associated profile',
            1, 3, 9, 10);
        $self->addEntry($table, 'assocProf', FALSE,
            3, 6, 9, 10);
        $self->addLabel($table, 'Profile category',
            1, 3, 10, 11);
        $self->addEntry($table, 'assocProfCategory', FALSE,
            3, 6, 10, 11);

        # Dependent interface IVs
        $self->addLabel($table, '<b>Dependent interface IVs</b>',
            7, 12, 8, 9);
        $self->addLabel($table, 'Call class',
            7, 9, 9, 10);
        $self->addEntry($table, 'callClass', 0,
            9, 12, 9, 10);
        $self->addLabel($table, '(...named)',
            7, 9, 10, 11);
        my $entry3 = $self->addEntry($table, undef, 0,
            9, 12, 10, 11);
        if (defined $self->editObj->callClass) {

            $entry3->set_text($self->editObj->callClass->_objName);
        }
        $self->addLabel($table, 'Call method',
            7, 9, 11, 12);
        $self->addEntry($table, 'callMethod', 0,
            9, 12, 11, 12);

#       # Tab complete (handled by the calling function)
#       $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub attributesTab {

        # Attributes tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->attributesTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('_Attributes', $self->notebook);

        # Attributes
        $self->addLabel($table, '<b>Attributes</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>List of attributes for this active interface</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Attribute', 'text',
            'Current value', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 250);     # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, (scalar @columnList / 2), 'attribHash');

        # Add a single button
        my $button = $self->addButton(
            $table,
            'Refresh list',
            'Refresh the list of attributes',
            undef,
            9, 12, 10, 11,
        );
        $button->signal_connect('clicked' => sub {

            # Refresh the simple list
            $self->refreshList_hashIV($slWidget, (scalar @columnList / 2), 'attribHash');
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub propertiesTab {

        # Properties tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->propertiesTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('_Properties', $self->notebook);

        # Quest solution
        $self->addLabel($table, '<b>Properties</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>List of custom properties for this active interface</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Key', 'text',
            'Value', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 300);     # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, (scalar @columnList / 2), 'propertyHash');

        # Add a single button
        my $button = $self->addButton(
            $table,
            'Refresh list',
            'Refresh the list of properties',
            undef,
            9, 12, 10, 11,
        );
        $button->signal_connect('clicked' => sub {

            # Refresh the simple list
            $self->refreshList_hashIV($slWidget, (scalar @columnList / 2), 'propertyHash');
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get

    sub attribHash
        { my $self = shift; return %{$self->{attribHash}}; }
}

{ package Games::Axmud::EditWin::Interface::Trigger;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::EditWin::Generic::Interface Games::Axmud::Generic::EditWin
        Games::Axmud::Generic::ConfigWin Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win
        Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::Interface::Trigger')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        $self->triggerAttributesTab();
        $self->beforeAfterTab();

        return 1;
    }

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::Interface::Alias;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::EditWin::Generic::Interface Games::Axmud::Generic::EditWin
        Games::Axmud::Generic::ConfigWin Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win
        Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::Interface::Alias')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        $self->aliasAttributesTab();
        $self->beforeAfterTab();

        return 1;
    }

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::Interface::Macro;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::EditWin::Generic::Interface Games::Axmud::Generic::EditWin
        Games::Axmud::Generic::ConfigWin Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win
        Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::Interface::Macro')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        $self->macroAttributesTab();
        $self->beforeAfterTab();

        return 1;
    }

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::Interface::Timer;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::EditWin::Generic::Interface Games::Axmud::Generic::EditWin
        Games::Axmud::Generic::ConfigWin Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win
        Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::Interface::Timer')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        $self->timerAttributesTab();
        $self->beforeAfterTab();

        return 1;
    }

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::Interface::Hook;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::EditWin::Generic::Interface Games::Axmud::Generic::EditWin
        Games::Axmud::Generic::ConfigWin Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win
        Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::Interface::Hook')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        $self->hookAttributesTab();
        $self->beforeAfterTab();

        return 1;
    }

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::InterfaceModel;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::Generic::EditWin Games::Axmud::Generic::ConfigWin
        Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::Generic::InterfaceModel')) {
            return undef;
        } else {
            return 1;
        }
    }

    sub enableButtons {

        # Called by $self->drawWidgets
        # We only need a single button so, instead of calling the generic ->enableButtons, call a
        #   method that creates just one button
        #
        # Expected arguments
        #   $hBox       - The horizontal packing box in which the buttons live (not yet stored as
        #                   an IV)
        #   $tooltips   - A Gtk2::Tooltips object for the buttons (not yet stored as an IV)
        #
        # Return values
        #   An empty list on improper arguments
        #   Otherwise, a list containing the Gtk::Button object created

        my ($self, $hBox, $tooltips, $check) = @_;

        # Local variables
        my @emptyList;

        # Check for improper arguments
        if (! defined $hBox || ! defined $tooltips || defined $check) {

            $axmud::CLIENT->writeImproper($self->_objClass . '->enableButtons', @_);
            return @emptyList;
        }

        return $self->enableSingleButton($hBox, $tooltips);
    }

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

    sub setupNotebook {

        # Called by $self->winEnable
        # Creates the first tab for the notebook. The remaining tabs are created by
        #   $self->expandNotebook
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setupNotebook', @_);
        }

        # Tab setup, using the standard table size
        my ($vBox, $table) = $self->addTab('_Category', $self->notebook);

        # Set up the rest of the tab
        $self->categoryTab($table);

        # Set up the remaining tabs
        $self->expandNotebook();

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        $self->attributesTab();

        return 1;
    }

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    # Notebook tabs

    sub categoryTab {

        # Category tab - called by $self->setupNotebook
        #
        # Expected arguments
        #   $table  - The Gtk2::Table
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $table, $check) = @_;

        # Local variables
        my @eventList;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->categoryTab', @_);
        }

#       # Tab setup (already created by the calling function)
#       my ($vBox, $table) = $self->addTab('_Name', $self->notebook);

        # Left column
        $self->addLabel($table, '<b>Interface category</b>',
            0, 6, 0, 1);
        $self->addLabel($table, 'Category',
            1, 3, 1, 2);
        $self->addEntry($table, 'category', FALSE,
            3, 6, 1, 2);

        # Right column
        $self->addLabel($table, '<b>' . ucfirst($self->editObj->category) . ' attributes</b>',
            7, 12, 0, 1);
        $self->addLabel($table, 'Stimulus name',
            7, 9, 1, 2);
        $self->addEntry($table, 'stimulusName', FALSE,
            9, 12, 1, 2);
        $self->addLabel($table, 'Response name',
            7, 9, 2, 3);
        $self->addEntry($table, 'responseName', FALSE,
            9, 12, 2, 3);

        # Hook events (for the hook interface model object only)
        if ($self->editObj->category eq 'hook') {

            # Hook events
            $self->addLabel($table, '<b>List of hook events</b>',
                0, 12, 3, 4);
            my $textView = $self->addTextView($table, undef, FALSE,
               1, 12, 4, 12,
               undef, undef, undef, undef,
               -1, 250);        # Fixed height

            @eventList = sort {$a cmp $b} ($self->editObj->ivKeys('hookEventHash'));

            my $buffer = $textView->get_buffer();
            $buffer->set_text(join("\n", @eventList));
        }

#       # Tab complete (handled by the calling function)
#       $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub attributesTab {

        # Attributes tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->attributesTab', @_);
        }

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('_Attributes', $self->notebook);

        # Add tabs to the inner notebook
        $self->attributes1Tab($innerNotebook);
        $self->attributes2Tab($innerNotebook);
        $self->attributes3Tab($innerNotebook);
        $self->attributes4Tab($innerNotebook);

        return 1;
    }

    sub attributes1Tab {

        # Attributes1 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->attributes1Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _1', $innerNotebook);

        # Quest solution
        $self->addLabel($table, '<b>Attribute types</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>Acceptable values for all attributes (standard and '
            . $self->editObj->category . ' attributes)</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Attribute', 'text',
            'Type', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 300);     # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, (scalar @columnList / 2), 'attribTypeHash');

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub attributes2Tab {

        # Attributes2 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->attributes2Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _2', $innerNotebook);

        # Quest solution
        $self->addLabel($table, '<b>Default values</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>Default values for ' . $self->editObj->category
            . ' attributes</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Attribute', 'text',
            'Default value', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 300);     # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, (scalar @columnList / 2), 'optionalAttribHash');

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub attributes3Tab {

        # Attributes3 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->attributes3Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _3', $innerNotebook);

        # Quest solution
        $self->addLabel($table, '<b>Compulsory switches</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>List of compulsory switches for the \'' . $axmud::CLIENT->cmdSep
            . 'add' . $self->editObj->category . '\' command</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Attribute', 'text',
            'Switch', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 300);     # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, (scalar @columnList / 2), 'compulsorySwitchHash');

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub attributes4Tab {

        # Attributes4 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->attributes4Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _4', $innerNotebook);

        # Quest solution
        $self->addLabel($table, '<b>Default values</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>List of optional switches for the \'' . $axmud::CLIENT->cmdSep
            . 'add' . $self->editObj->category . '\' command</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Attribute', 'text',
            'Switch', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 300);     # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, (scalar @columnList / 2), 'optionalSwitchHash');

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::Generic::CageMask;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::EditWin::Generic::Cage Games::Axmud::Generic::EditWin
        Games::Axmud::Generic::ConfigWin Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win
        Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

#   sub checkEditObj {}         # Inherited from GA::Generic::ConfigWin

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook, depending on which type of cage is being edited
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        $self->maskTab();

        return 1;
    }

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    # Notebook tabs used by more than one cage 'edit' window

    sub maskTab {

        # Mask tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (@cageList, @columnList);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->maskTab', @_);
        }

        # Tab setup
        my ($vbox, $table) = $self->addTab('_Mask', $self->notebook);

        # Cage mask
        $self->addLabel($table, '<b>Cage mask</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of properties (IVs) used by this cage mask</i>',
            1, 12, 1, 2);

        # Add a simple list. The number of columns is equal to the number of cages between this one
        #   and the least superior cage. (If this cage has no inferior, there's only one column)
        # Each column (except the first two) has a title containing the cage's associated profile
        @cageList = $self->compileCages();

        # Use different text in the column for the first item in @cageList
        @columnList = (
            'Property (IV)', 'text',
            'Value/Size', 'text',
        );

        shift @cageList;

        foreach my $cage (@cageList) {

            push (@columnList,
                '[' . $cage->profCategory . ': ' . $cage->profName . ']',
                'text',
            );
        }

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 300);      # Fixed height

        # Initialise the list
        $self->maskTab_refreshList($slWidget, scalar (@columnList / 2));

        # Add some buttons and entry boxes

        $self->addLabel($table, 'Selected property:',
            1, 3, 10, 11);

        # The IV's corresponding value is either 'undef', or a scalar/list/hash reference (not a
        #   plain scalar)
        my $button = $self->addButton($table, 'Edit as scalar', 'Edit value as a scalar', undef,
            3, 5, 10, 11);
        $button->signal_connect('clicked' => sub {

            my (
                $prop, $value, $refType, $refName,
                %maskHash,
            );

            ($prop) = $self->getSimpleListData($slWidget, 0);
            if (defined $prop) {

                # Check the corresponding value is either 'undef' or a scalar reference
                %maskHash = $self->getEditHash_hashIV('maskHash');
                if (defined $maskHash{$prop}) {

                    $refType = ref $maskHash{$prop};
                    if ($refType eq 'ARRAY') {
                        $refName = 'a list';
                    } elsif ($refType eq 'HASH') {
                        $refName = 'a hash';
                    } else {
                        $refName = 'a scalar';
                    }
                }

                if (defined $refType && ($refType eq 'ARRAY' || $refType eq 'HASH')) {

                    $self->showMsgDialogue(
                        'Edit scalar',
                        'error',
                        'The \'' . $prop . '\' property is currently ' . $refName
                        . ', not a scalar',
                        'ok',
                    );

                } else {

                    $self->promptScalar(
                        'maskHash',
                        $prop,
                        TRUE,                   # This is a cage mask!
                        $slWidget,
                        FALSE,                  # Not read-only
                        'maskTab_refreshList',
                        $slWidget,              # Pass these arguments to ->maskTab_refreshList
                        (scalar @columnList / 2),
                    );
                }
            }
        });

        my $button2 = $self->addButton($table, 'Edit as list', 'Edit value as a list', undef,
            5, 7, 10, 11);
        $button2->signal_connect('clicked' => sub {

            my (
                $prop, $value, $refType, $refName,
                %maskHash,
            );

            ($prop) = $self->getSimpleListData($slWidget, 0);
            if (defined $prop) {

                # Check the corresponding value is either 'undef' or a list reference
                %maskHash = $self->getEditHash_hashIV('maskHash');
                if (defined $maskHash{$prop}) {

                    $refType = ref $maskHash{$prop};
                    if ($refType eq 'ARRAY') {
                        $refName = 'a list';
                    } elsif ($refType eq 'HASH') {
                        $refName = 'a hash';
                    } else {
                        $refName = 'a scalar';
                    }
                }

                if (defined $refType && $refType ne 'ARRAY') {

                    $self->showMsgDialogue(
                        'Edit list',
                        'error',
                        'The \'' . $prop . '\' property is currently ' . $refName
                        . ', not a list',
                        'ok',
                    );

                } else {

                    $self->promptList(
                        'maskHash',
                        $prop,
                        $slWidget,
                        FALSE,                  # Not read-only
                        'maskTab_refreshList',
                        $slWidget,              # Pass these arguments to ->maskTab_refreshList
                        (scalar @columnList / 2),
                    );
                }
            }
        });

        my $button3 = $self->addButton($table, 'Edit as hash', 'Edit value as a hash', undef,
            7, 9, 10, 11);
        $button3->signal_connect('clicked' => sub {

            my (
                $prop, $value, $refType, $refName,
                %maskHash,
            );

            ($prop) = $self->getSimpleListData($slWidget, 0);
            if (defined $prop) {

                # Check the corresponding value is either 'undef' or a hash reference
                %maskHash = $self->getEditHash_hashIV('maskHash');
                if (defined $maskHash{$prop}) {

                    $refType = ref $maskHash{$prop};
                    if ($refType eq 'ARRAY') {
                        $refName = 'a list';
                    } elsif ($refType eq 'HASH') {
                        $refName = 'a hash';
                    } else {
                        $refName = 'a scalar';
                    }
                }

                if (defined $refType && $refType ne 'HASH') {

                    $self->showMsgDialogue(
                        'Edit list',
                        'error',
                        'The \'' . $prop . '\' property is currently ' . $refName
                        . ', not a hash',
                        'ok',
                    );

                } else {

                    $self->promptHash(
                        'maskHash',
                        $prop,
                        $slWidget,
                        FALSE,                  # Not read-only
                        'maskTab_refreshList',
                        $slWidget,              # Pass these arguments to ->maskTab_refreshList
                        (scalar @columnList / 2),
                    );
                }
            }
        });

        my $button4 = $self->addButton($table, 'Reset', 'Reset (don\'t use) this property', undef,
            10, 12, 10, 11);
        $button4->signal_connect('clicked' => sub {

            my (
                $prop,
                %maskHash,
            );

            ($prop) = $self->getSimpleListData($slWidget, 0);
            if (defined $prop) {

                %maskHash = $self->getEditHash_hashIV('maskHash');
                if (defined $maskHash{$prop}) {

                    $maskHash{$prop} = undef;
                    $self->ivAdd('editHash', 'maskHash', \%maskHash);

                    $self->maskTab_refreshList($slWidget, scalar (@columnList / 2));
                }
            }
        });

        # Tab complete
        $vbox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub maskTab_refreshList {

        # Called by $self->maskTab to reset the GA::Obj::Simple::List
        #
        # Expected arguments
        #   $slWidget       - The GA::Obj::Simple::List
        #   $columns        - The number of columns in the list
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $slWidget, $columns, $check) = @_;

        # Local variables
        my (
            @standardList, @cageList, @dataList,
            %ivHash,
        );

        # Check for improper arguments
        if (! defined $slWidget || ! defined $columns || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->maskTab_refreshList', @_);
        }

        # Get a list of IVs from the cage, in a standard order
        @standardList = $self->editObj->maskList;
        # Get a list of columns in the simple list, which depends on how many cages there are
        @cageList = $self->compileCages();
        # (We already know the first cage)
        shift @cageList;

        # Import the mask hash
        %ivHash = $self->getEditHash_hashIV('maskHash');

        # Compile the simple list data
        OUTER: foreach my $iv (@standardList) {

            my ($value, $ref);

            $value = $ivHash{$iv};
            if (! defined $value) {

                push (@dataList, $iv, undef);

            } else {

                $ref = ref $value;
                if ($ref eq 'HASH') {
                    push (@dataList, $iv, scalar (keys %$value));
                } elsif ($ref eq 'ARRAY') {
                    push (@dataList, $iv, scalar @$value);
                } else {
                    push (@dataList, $iv, $$value);
                }
            }

            # Remaining columns
            INNER: foreach my $cage (@cageList) {

                my ($thisValue, $thisRef);

                # The lack of a $self->session argument tells ->ivShow to give us the value stored
                #   in the cage, and not to consult its inferiors (if no value is stored there)
                $thisValue = $cage->ivShow('maskHash', $iv);
                if (! defined $thisValue) {

                    # $cage doesn't store an interpolated form of the command
                    push (@dataList, undef);

                } else {

                    $thisRef = ref $thisValue;
                    if ($thisRef eq 'HASH') {
                        push (@dataList, scalar (keys %$thisValue));
                    } elsif ($thisRef eq 'ARRAY') {
                        push (@dataList, scalar @$thisValue);
                    } else {
                        push (@dataList, $$thisValue);
                    }
                }
            }
        }

        # Reset the simple list
        $self->resetListData($slWidget, [@dataList], $columns);

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::MapLabelStyle;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::Generic::EditWin Games::Axmud::Generic::ConfigWin
        Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::Obj::MapLabelStyle')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

#   sub expandNotebook {}       # Inherited from GA::Generic::ConfigWin

    sub saveChanges {

        # Called by $self->buttonOK and $self->buttonSave (usually for 'edit' windows only, not
        #   'pref' windows)
        # Saves any changes made to data stored by the edit object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->saveChanges', @_);
        }

        if ($self->editHash) {

            # If the style name has been changed, apply that change with a client command
            if ($self->ivExists('editHash', 'name')) {

                if (
                    ! $self->session->pseudoCmd(
                        'renamelabelstyle <' . $self->editObj->name . '> <'
                        . $self->ivShow('editHash', 'name'),
                    )
                ) {

                    # Unable to save changes; notify the user
                    $self->showMsgDialogue(
                        'Rename label style',
                        'error',
                        'The label style could not be renamed - save aborted',
                        'ok',
                    );

                    return undef;

                } else {

                    $self->ivDelete('editHash', 'name');
                }
            }

            # Store any remaining changes the user has made
            foreach my $key ($self->ivKeys('editHash')) {

                $self->editObj->{$key} = $self->ivShow('editHash', $key);
            }

            # The changes can now be cleared
            $self->ivEmpty('editHash');

            # Mark the object's corresponding file object as needing to be saved, if it exists
            if ($self->editObj->_parentFile) {

                $self->editObj->doModify('saveChanges');
            }

            # Update the current session's object viewer window, if it is open
            if ($self->session->viewerWin) {

                $self->session->viewerWin->updateNotebook();
            }

            # Redraw labels in maps in all Automapper windows using this world model
            $self->session->worldModelObj->updateMapLabels();
        }

        return 1;
    }

    # Notebook tabs

    sub nameTab {

        # Name tab - called by $self->setupNotebook
        #
        # Expected arguments
        #   $table  -  The Gtk2::Table which has already been created for this tab
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $table, $check) = @_;

        # Local variables
        my (
            $count, $index,
            @list, @comboList,
            %descripHash,
        );

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->nameTab', @_);
        }

#       # Tab setup (already created by the calling function)
#       my ($vBox, $table) = $self->addTab('_Name', $self->notebook);

        # Map label style
        $self->addLabel($table, '<b>Map label style</b>',
            0, 12, 0, 1);

        $self->addLabel($table, 'Style name (max 16 chars)',
            1, 4, 1, 2);
        $self->addEntryWithIcon($table, 'name', 'string', 1, 16,
            4, 8, 1, 2,
            16, 16);

        $self->addLabel($table, 'Text colour',
            1, 4, 2, 3);
        my ($frame, $canvas, $canvasObj) = $self->addSimpleCanvas($table,
            $self->editObj->textColour,
            undef,                          # No neutral colour
            4, 6, 2, 3);
        my $entry = $self->addEntry($table, 'textColour', FALSE,
            6, 8, 2, 3,
            7, 7);
        my $button = $self->addButton($table, 'Set', 'Set this colour', undef,
            8, 10, 2, 3);
        $button->signal_connect('clicked' => sub {

            # Prompt the user to select a new colour, using the existing colour as an initial value
            my $rgbModify = $self->showColourSelectionDialogue(
                'Set text colour',
                $self->editObj->textColour,
            );

            if ($rgbModify) {

                $canvasObj = $self->fillSimpleCanvas($canvas, $canvasObj, $rgbModify);
                $entry->set_text($rgbModify);

                # Update IVs
                $self->ivAdd('editHash', 'textColour', $rgbModify);
            }
        });
        my $button2 = $self->addButton($table, 'Reset', 'Reset this colour', undef,
            10, 12, 2, 3);
        $button2->signal_connect('clicked' => sub {

            $canvasObj = $self->fillSimpleCanvas($canvas, $canvasObj);
            $entry->set_text('');

            # Update IVs
            $self->ivAdd('editHash', 'textColour', undef);
        });

        $self->addLabel($table, 'Underlay colour',
            1, 4, 3, 4);
        my ($frame2, $canvas2, $canvasObj2) = $self->addSimpleCanvas($table,
            $self->editObj->underlayColour,
            undef,                          # No neutral colour
            4, 6, 3, 4);
        my $entry2 = $self->addEntry($table, 'underlayColour', FALSE,
            6, 8, 3, 4,
            7, 7);
        my $button3 = $self->addButton($table, 'Set', 'Set this colour', undef,
            8, 10, 3, 4);
        $button3->signal_connect('clicked' => sub {

            # Prompt the user to select a new colour, using the existing colour as an initial value
            my $rgbModify = $self->showColourSelectionDialogue(
                'Set underlay colour',
                $self->editObj->textColour,
            );

            if ($rgbModify) {

                $canvasObj2 = $self->fillSimpleCanvas($canvas2, $canvasObj2, $rgbModify);
                $entry2->set_text($rgbModify);

                # Update IVs
                $self->ivAdd('editHash', 'underlayColour', $rgbModify);
            }
        });
        my $button4 = $self->addButton($table, 'Reset', 'Reset this colour', undef,
            10, 12, 3, 4);
        $button4->signal_connect('clicked' => sub {

            $canvasObj2 = $self->fillSimpleCanvas($canvas2, $canvasObj2);
            $entry2->set_text('');

            # Update IVs
            $self->ivAdd('editHash', 'underlayColour', undef);
        });

        $self->addLabel($table, 'Relative size (0.5 - 10)',
            1, 4, 4, 5);
        $self->addEntryWithIcon($table, 'relSize', 'float', 0.5, 10,
            4, 8, 4, 5,
            8, 8);

        my $checkButton = $self->addCheckButton($table, 'italicsFlag', TRUE,
            1, 3, 5, 6,
            0, 0.5);
        $checkButton->set_label('Italics');
        my $checkButton2 = $self->addCheckButton($table, 'boldFlag', TRUE,
            3, 5, 5, 6,
            0, 0.5);
        $checkButton2->set_label('Bold');
        my $checkButton3 = $self->addCheckButton($table, 'underlineFlag', TRUE,
            5, 7, 5, 6,
            0, 0.5);
        $checkButton3->set_label('Underline');
        my $checkButton4 = $self->addCheckButton($table, 'strikeFlag', TRUE,
            7, 9, 5, 6,
            0, 0.5);
        $checkButton4->set_label('Strikethrough');
        my $checkButton5 = $self->addCheckButton($table, 'boxFlag', TRUE,
            9, 12, 5, 6,
            0, 0.5);
        $checkButton5->set_label('Draw box around label');

#        $self->addLabel($table, 'Orientation',
#            1, 4, 6, 7);
#
#        $count = -1;
#        @list = (
#            'south' => 'Normal',
#            'north' => 'Rotated 180 degrees',
#            'west'  => 'Rotated 90 degrees clockwise',
#            'east'  => 'Rotated 90 degrees anti-clockwise',
#        );
#
#        do {
#
#            my $gravity = shift @list;
#            my $descrip = shift @list;
#
#            push (@comboList, $descrip);
#            $descripHash{$descrip} = $gravity;
#
#            $count++;
#            if ($gravity eq $self->editObj->gravity) {
#
#                $index = $count;
#            }
#
#        } until (! @list);
#
#        my $combo = $self->addComboBox($table, undef, \@comboList, '',
#            TRUE,               # No 'undef' value used
#            4, 8, 6, 7,
#        );
#        $combo->set_active($index),
#        $combo->signal_connect('changed' => sub {
#
#            my $descrip = $combo->get_active_text();
#
#            $self->ivAdd('editHash', 'gravity', $descripHash{$combo->get_active_text});
#        });

#       # Tab complete (handled by the calling function)
#       $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::Mission;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::Generic::EditWin Games::Axmud::Generic::ConfigWin
        Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::Obj::Mission')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        $self->contentsTab();

        return 1;
    }

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    # Notebook tabs

    sub nameTab {

        # Name tab - called by $self->setupNotebook
        #
        # Expected arguments
        #   $table  - The Gtk2::Table
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $table, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->nameTab', @_);
        }

#       # Tab setup (already created by the calling function)
#       my ($vBox, $table) = $self->addTab('_Name', $self->notebook);

        # Name
        $self->addLabel($table, '<b>Name</b>',
            0, 12, 0, 1);
        $self->addLabel($table, 'Name',
            1, 4, 1, 2);
        $self->addEntry($table, 'name', 0,
            4, 6, 1, 2);
        $self->addLabel($table, 'Description',
            1, 4, 2, 3);
        $self->addEntryWithIcon($table, 'descrip', 'string', 1, 64,
            4, 12, 2, 3);

#       # Tab complete (handled by the calling function)
#       $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub contentsTab {

        # Contents tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->contentsTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('_Contents', $self->notebook);

        # Contents list
        $self->addLabel($table, '<b>Contents list</b>',
            0, 12, 0, 2);
        $self->addLabel($table,
            '<i>The contents of the mission; each line is a separate instruction</i>',
            1, 12, 2, 4);
        $self->addTextView($table, 'missionList', TRUE,
            1, 12, 4, 8,
            TRUE, FALSE, FALSE, FALSE,   # Treat as a list, don't remove empty lines or whitespace
            -1, 250);                    # Fixed height

        $self->addLabel($table, '<b>Commands:</b> ',
            1, 4, 8, 9);
        $self->addLabel($table,
            '<b>></b> <i>world command,</i> '
            . '<b>;</b> <i>client command,</i> '
            . '<b>.</b> <i>speedwalk command,</i> '
            . '<b>#</b> <i>\'main\' window comment</i>',
            4, 12, 8, 9);

        $self->addLabel($table, '<b>Breaks:</b> ',
            1, 4, 9, 10);
        $self->addLabel($table,
            '<b>@</b> <i>ordinary break,</i> '
            . '<b>t {pattern}</b> <i>trigger break,</i> '
            . '<b>p {interval}</b> <i>pause break,</i>'
            . '<b>l</b> <i>Locator break</i>',
            4, 12, 9, 10);

        $self->addLabel($table, '<b>Automated logins:</b> ',
            1, 4, 10, 11);
        $self->addLabel($table,
            '<b>n</b> <i>send current character\'s name,</i> '
            . '<b>w</b> <i>send password,</i>'
            . '<b>a</b> <i>send account name,</i>'
            . '<b>c</b> <i>send combined command</i>',
            4, 12, 10, 11);
        $self->addLabel($table,
            '<i>(in combined commands, </i><b>@name@</b><i>, </i><b>@password@</b><i> and'
            . ' </i><b>@account@</b><i> are substituted)</i>',
            4, 12, 11, 12);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::Generic::ModelObj;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::Generic::EditWin Games::Axmud::Generic::ConfigWin
        Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

#   sub checkEditObj {}         # Inherited from GA::Generic::ConfigWin

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook, depending on which category of model object is
        #   being edited
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        # Group 1
        $self->familyTab();
        $self->privateDataTab(       # Inherited from GA::Generic::EditWin
            'privateHash',
            '_Private',
            'Private properties hash',
            'Properties created by your own plugins and scripts, but stored in this type of object',
        );
        $self->notesTab();
        # Group 2
        $self->wordsTab();
        # Group 4
        $self->physicalTab();
        $self->exclusiveTab();

        return 1;
    }

    sub saveChanges {

        # Called by $self->buttonOK and $self->buttonSave
        # Saves any changes made to data stored by the edit object
        # (We don't use the inherited function, because for region and room objects, we need to
        #   tell Automapper windows to update themselves - even if $self->editHash is empty)
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->saveChanges', @_);
        }

        if ($self->editHash) {

            # Store the changes the user has made
            foreach my $key ($self->ivKeys('editHash')) {

                $self->editObj->{$key} = $self->ivShow('editHash', $key);
            }

            # The changes can now be cleared
            $self->ivEmpty('editHash');

            # Mark the object's corresponding file object as needing to be saved, if it exists
            if ($self->editObj->_parentFile) {

                $self->editObj->doModify('saveChanges');
            }

            # Update the current session's object viewer window, if it is open
            if ($self->session->viewerWin) {

                $self->session->viewerWin->updateNotebook();
            }
        }

        # For objects stored in the world model (first checking that they are still there)...
        if (
            $self->editObj->modelFlag
            && $self->session->worldModelObj->ivExists('modelHash', $self->editObj->number)
            && $self->session->worldModelObj->ivShow('modelHash', $self->editObj->number)
                eq $self->editObj
        ) {
            # For regions and rooms, update any Automapper windows
            if ($self->editObj->category eq 'region') {

                # Redraw this region, if drawn
                $self->session->worldModelObj->updateRegion($self->editObj->name);

            } elsif ($self->editObj->category eq 'room') {

                # Mark this room to be redrawn, if it is visible
                $self->session->worldModelObj->updateMaps('room', $self->editObj);
            }
        }

        return 1;
    }

    # Notebook tabs used by more than one model object 'edit' window

    # Group 1

    sub nameTab {

        # Name tab - called by $self->setupNotebook
        #
        # Expected arguments
        #   $table  - The Gtk2::Table
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $table, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->nameTab', @_);
        }

#       # Tab setup (already created by the calling function)
#       my ($vBox, $table) = $self->addTab('_Name', $self->notebook);

        # Left column
        $self->addLabel($table, '<b>General properties</b>',
            0, 6, 0, 1);
        $self->addLabel($table, 'Name',
            1, 3, 1, 2);
        $self->addEntry($table, 'name', FALSE,
            3, 6, 1, 2);
        $self->addLabel($table, 'Category',
            1, 3, 2, 3);
        $self->addEntry($table, 'category', FALSE,
            3, 6, 2, 3);
        $self->addLabel($table, 'In world model',
            1, 5, 3, 4);
        $self->addCheckButton($table, 'modelFlag', FALSE,
            5, 6, 3, 4, 1, 0.5);
        $self->addLabel($table, 'World model #',
            1, 3, 4, 5);
        $self->addEntry($table, 'number', FALSE,
            3, 6, 4, 5);

        $self->addLabel($table, '<b>Source code</b>',
            0, 12, 6, 7);
        if ($self->session->worldModelObj->mudlibPath) {

            $self->addLabel($table,
                '<i>The equivalent file in the mudlib, relative to '
                . $self->session->worldModelObj->mudlibPath . '</i>',
                1, 12, 7, 8);

        } else {

            $self->addLabel($table,
                '<i>The equivalent file in the mudlib, e.g.'
                . ' /home/myname/ds/lib/domains/town/room/vill_road2</i>',
                1, 12, 7, 8);
        }

        my $entry = $self->addEntry($table, 'sourceCodePath', TRUE,
            1, 12, 8, 9);

        my $button = $self->addButton($table,
            'View',
            'View this file',
            undef,
            9, 11, 9, 10);
        $button->signal_connect('clicked' => sub {

            my $flag;

            if ($self->getEditHash_scalarIV('virtualAreaPath')) {
                $flag = TRUE;
            } else {
                $flag = FALSE;
            }

            $self->quickFreeWin(
                'Games::Axmud::OtherWin::SourceCode',
                $self->session,
                # Config
                'model_obj' => $self->editObj,
                'virtual_flag' => $flag,
                'path' => $self->getEditHash_scalarIV('sourceCodePath'),
            );
        });

        my $button2 = $self->addButton($table,
            'Edit',
            'Edit this file',
            undef,
            11, 12, 9, 10);
        $button2->signal_connect('clicked' => sub {

            my ($cmd, $file);

            # Check that the GA::Client has a text editor command set, and that it is valid
            $cmd = $axmud::CLIENT->textEditCmd;
            if (! $cmd || ! ($cmd =~ m/%s/)) {

                # Show a 'dialogue' window to explain the problem
                $self->showMsgDialogue(
                    'Edit source code file',
                    'error',
                    'Can\'t edit the file: invalid external application command \'' . $cmd . '\'',
                    'ok',
                );

                return undef;
            }

            # Set the file to be opened. If the current world model defines a mudlib directory, the
            #   object's ->mudlibPath is relative to that; otherwise it's an absolute path
            if ($self->session->worldModelObj->mudlibPath) {
                $file = $self->session->worldModelObj->mudlibPath;
            } else {
                $file = '';
            }

            if ($self->getEditHash_scalarIV('virtualAreaPath')) {
                $file .= $self->getEditHash_scalarIV('virtualAreaPath');
            } else {
                $file .= $self->getEditHash_scalarIV('sourceCodePath');
            }

            # Add the file extension, if set
            if ($self->session->worldModelObj->mudlibExtension) {

                $file .= $self->session->worldModelObj->mudlibExtension;
            }

            # Check the file exists
            if (! (-e $file)) {

                $self->showMsgDialogue(
                    'Edit source code file',
                    'error',
                    'Can\'t find the file \'' . $file . '\'',
                    'ok',
                );

                return undef;
            }

            # Open the file in the external text editor
            $cmd =~ s/%s/$file/;

            system $cmd;
        });

        # This IV only available to rooms
        if ($self->editObj->category eq 'room') {

            $self->addLabel($table, '<i>For rooms in virtual areas, the virtual area\'s file</i>',
                1, 12, 10, 11);
            my $entry2 = $self->addEntry($table, 'virtualAreaPath', TRUE,
                1, 12, 11, 12);
            # 2nd entry box starts insensitive if the 1st entry box doesn't contain text
            if (! $entry->get_text()) {

                $entry2->set_sensitive(FALSE);
            }

            my $button3 = $self->addButton($table,
                'Copy to virtual path',
                'Copy the file path above into the entry box below',
                undef,
                7, 9, 9, 10);
            $button3->signal_connect('clicked' => sub {

                # Copy the contents of the first entry into the second when the button is clicked
                $entry2->set_text($entry->get_text());
            });

            # The second entry can only be edited when the first contains text
            $entry->signal_connect('changed' => sub {

                # Update the IV
                my $text = $entry->get_text();
                $self->ivAdd('editHash', 'sourceCodePath', $text);

                if (! $entry->get_text()) {

                    # First entry is now empty, so empty the second entry too
                    $entry2->set_text('');
                    $entry2->set_sensitive(FALSE);

                } else {

                    # Sensitise the second entry
                    $entry2->set_sensitive(TRUE);
                }
            });
        }

        # Right column
        $self->addLabel($table, 'Concrete (not abstract) object',
            7, 11, 1, 2);
        $self->addCheckButton($table, 'concreteFlag', FALSE,
            11, 12, 1, 2);
        $self->addLabel($table, 'Alive',
            7, 11, 2, 3);
        $self->addCheckButton($table, 'aliveFlag', FALSE,
            11, 12, 2, 3);
        $self->addLabel($table, 'Sentient',
            7, 11, 3, 4);
        $self->addCheckButton($table, 'sentientFlag', FALSE,
            11, 12, 3, 4);
        $self->addLabel($table, 'Portable (can be carried)',
            7, 11, 4, 5);
        $self->addCheckButton($table, 'portableFlag', FALSE,
            11, 12, 4, 5);
        $self->addLabel($table, 'Saleable (can be sold)',
            7, 11, 5, 6);
        $self->addCheckButton($table, 'saleableFlag', FALSE,
            11, 12, 5, 6);

#       # Tab complete (handled by the calling function)
#       $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub familyTab {

        # Family tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->familyTab', @_);
        }

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('_Family', $self->notebook);

        # Add tabs to the inner notebook
        $self->family1Tab($innerNotebook);
        $self->family2Tab($innerNotebook);

        return 1;
    }

    sub family1Tab {

        # Family1 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->family1Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _1', $innerNotebook);

        # Parent object
        $self->addLabel($table, '<b>Parent object</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List containing the world model object which \'owns\' this one (if any)</i>',
            1, 12, 1, 2);

        # Add a simple list - it never contains more than one object, but we use a simple list to
        #   keep this tab consistent with the next one
        @columnList = (
            'Model #', 'int',
            'Object', 'text',
            'Object name', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 100);     # Fixed height - only one row

        # Initialise the list
        $self->family1Tab_refreshList($slWidget, (scalar @columnList / 2));

        # All of the buttons (except 'Edit') are not available when $self->editObj is a region or
        #   a room
        my $entry;

        if ($self->editObj->category ne 'region' && $self->editObj->category ne 'room') {

            $self->addLabel($table, 'Object #:',
                1, 4, 10, 11);

            my $funcRef = sub {

                # Check whether a valid world model object number has been entered - also, a world
                #   model object can't add itself as its own parent/child add
                my $value = $entry->get_text();

                if (
                    $self->session->worldModelObj->ivExists('modelHash', $value)
                    && $value ne $self->editObj->number
                ) {
                    return 1;       # Valid
                } else {
                    return undef;   # Invalid
                }
            };
            $entry = $self->addEntryWithIcon($table, undef, $funcRef, undef, undef,
                4, 6, 10, 11);

            my $button = $self->addButton($table,
                'Set as parent', 'Set this world model object as the parent', undef,
                6, 9, 10, 11,
                TRUE);      # Irreversible
            $button->signal_connect('clicked' => sub {

                my $value;

                if ($self->checkEntryIcon($entry)) {

                    $value = $entry->get_text();

                    # Attempt to change the parent
                    if (
                        ! $self->session->worldModelObj->setParent(
                            TRUE,           # Update Automapper windows
                            $self->editObj->number,
                            $value,
                        )
                    ) {
                        $self->showMsgDialogue(
                            'Add model object',
                            'error',
                            'Failed to set the parent object',
                            'ok',
                        );

                    } else {

                        # Update the simple list
                        $self->family1Tab_refreshList($slWidget, (scalar @columnList / 2));
                    }

                    # In either case, reset the entry box
                    $self->resetEntryBoxes($entry);
                }
            });
        }

        my $button2 = $self->addButton(
            $table, 'Edit...', 'Edit the selected world model object', undef,
            9, 12, 10, 11);
        $button2->signal_connect('clicked' => sub {

            my ($number) = $self->getSimpleListData($slWidget, 0);
            if (defined $number && $self->session->worldModelObj->ivExists('modelHash', $number)) {

                # Open up an 'edit' window for the child object
                $self->openChildEditWin($number);

                # Refresh the simple lists and reset entry boxes
                $self->family1Tab_refreshList($slWidget, (scalar @columnList / 2));
                $self->resetEntryBoxes($entry);
            }
        });

        if ($self->editObj->category ne 'region' && $self->editObj->category ne 'room') {

            my $button3 = $self->addButton($table,
                'Remove parent', 'Remove the selected world model object from the list', undef,
                1, 4, 11, 12,
                TRUE);      # Irreversible
            $button3->signal_connect('clicked' => sub {

                my ($number) = $self->getSimpleListData($slWidget, 0);
                if (
                    defined $number
                    && $self->session->worldModelObj->ivExists('modelHash', $number)
                ) {
                    # Attempt to reset the parent
                    if (
                        ! $self->session->worldModelObj->setParent(
                            TRUE,           # Update Automapper windows
                            $self->editObj->number,
                            undef,          # Object now has no parent
                        )
                    ) {
                        $self->showMsgDialogue(
                            'Add model object',
                            'error',
                            'Failed to reset the parent object',
                            'ok',
                        );

                    } else {

                        # Update the simple list
                        $self->family1Tab_refreshList($slWidget, (scalar @columnList / 2));
                    }

                    # In either case, reset the entry box
                    $self->resetEntryBoxes($entry);
                }
            });
        }

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub family2Tab {

        # Family2 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->family2Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _2', $innerNotebook);

        # Child objects
        $self->addLabel($table, '<b>Child objects</b>',
            0, 12, 0, 1);
        $self->addLabel(
            $table, '<i>List of world model objects which are \'owned\' by this one</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Model #', 'int',
            'Category', 'text',
            'Object name', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 220);     # Fixed height

        # Initialise the list
        $self->family2Tab_refreshList($slWidget, (scalar @columnList / 2));

        # Add editing buttons
        $self->addLabel($table, 'Object #:',
            1, 4, 10, 11);

        my $entry;
        my $funcRef = sub {

            # Check whether a valid world model object number has been entered - also, a world model
            #   object can't add itself as its own parent/child add
            my $value = $entry->get_text();

            if (
                $self->session->worldModelObj->ivExists('modelHash', $value)
                && $value ne $self->editObj->number
            ) {
                return 1;       # Valid
            } else {
                return undef;   # Invalid
            }
        };
        $entry = $self->addEntryWithIcon($table, undef, $funcRef, undef, undef,
            4, 6, 10, 11);

        my $button = $self->addButton($table,
            'Add as child', 'Add the specified model object as a child of this object', undef,
            6, 9, 10, 11,
            TRUE);      # Irreversible
        $button->signal_connect('clicked' => sub {

            my ($number, $childObj);

            if ($self->checkEntryIcon($entry)) {

                $number = $entry->get_text();

                # If the edited object is a region or a room, we can't add a region or a room as a
                #   child from this window. The call to ->addChild already checks for this, but we
                #   can provide an informative 'dialogue' window to explain the problem
                $childObj = $self->session->worldModelObj->ivShow('modelHash', $number);
                if (
                    (
                        $self->editObj->category eq 'region'
                        && ($childObj->category eq 'region' || $childObj->category eq 'room')
                    ) || (
                        $self->editObj->category eq 'room'
                        && ($childObj->category eq 'region' || $childObj->category eq 'room')
                    )
                ) {
                    $self->showMsgDialogue(
                        'Add model object',
                        'warning',
                        'Regions and rooms cannot be given a child region/room from this tab',
                        'ok',
                    );

                # Otherwise, attempt to add a child
                } elsif (
                    ! $self->session->worldModelObj->addChild(
                        TRUE,           # Update Automapper windows
                        $self->editObj->number,
                        $number,
                    )
                ) {
                    $self->showMsgDialogue(
                        'Add model object',
                        'error',
                        'Failed to add a child object',
                        'ok',
                    );

                } else {

                    # Update the simple list
                    $self->family2Tab_refreshList($slWidget, (scalar @columnList / 2));
                }

                # In either case, reset the entry box
                $self->resetEntryBoxes($entry);
            }
        });

        my $button2 = $self->addButton(
            $table, 'Edit...', 'Edit the selected world model object', undef,
            9, 12, 10, 11);
        $button2->signal_connect('clicked' => sub {

            my ($number) = $self->getSimpleListData($slWidget, 0);
            if (defined $number && $self->session->worldModelObj->ivExists('modelHash', $number)) {

                # Open up an 'edit' window for the child object
                $self->openChildEditWin($number);

                # Refresh the simple lists and reset entry boxes
                $self->family2Tab_refreshList($slWidget, (scalar @columnList / 2));
                $self->resetEntryBoxes($entry);
            }
        });

        my $button3 = $self->addButton($table,
            'Remove child', 'Remove the selected world model object from the list', undef,
            1, 4, 11, 12,
            TRUE);      # Irreversible
        $button3->signal_connect('clicked' => sub {

            my ($number, $childObj);

            ($number) = $self->getSimpleListData($slWidget, 0);
            if (defined $number && $self->session->worldModelObj->ivExists('modelHash', $number)) {

                # If the edited object is a region, we can't remove a child region or room from
                #   this window. The call to ->removeChild already checks for this, but we can
                #   provide an informative dialogue to explain the problem
                $childObj = $self->session->worldModelObj->ivShow('modelHash', $number);
                if (
                    $self->editObj->category eq 'region'
                    && ($childObj->category eq 'region' || $childObj->category eq 'room')
                ) {
                    $self->showMsgDialogue(
                        'Remove child object',
                        'warning',
                        'A child region or room cannot be removed from a parent region using this'
                        . ' tab',
                        'ok',
                    );

                # Attempt to remove the child
                } elsif (
                    ! $self->session->worldModelObj->removeChild(
                        TRUE,           # Update Automapper windows
                        $self->editObj->number,
                        $number,
                    )
                ) {
                    $self->showMsgDialogue(
                        'Add model object',
                        'error',
                        'Failed to remove the child object',
                        'ok',
                    );

                } else {

                    # Update the simple list
                    $self->family2Tab_refreshList($slWidget, (scalar @columnList / 2));
                }

                # In either case, reset the entry box
                $self->resetEntryBoxes($entry);
            }
        });

        my $button4 = $self->addButton($table,
            'Delete from model', 'Delete the selected world model object', undef,
            4, 6, 11, 12,
            TRUE);      # Irreversible
        $button4->signal_connect('clicked' => sub {

            my ($number, $obj);

            ($number) = $self->getSimpleListData($slWidget, 0);
            if (defined $number && $self->session->worldModelObj->ivExists('modelHash', $number)) {

                $obj = $self->session->worldModelObj->ivShow('modelHash', $number);

                # Attempt to delete the child from the world model
                if (
                    ! $self->session->worldModelObj->deleteObj(
                        $self->session,
                        TRUE,           # Update Automapper windows
                        $obj,
                      )
                ) {
                    $self->showMsgDialogue(
                        'Delete model object',
                        'error',
                        'Failed to delete the child object',
                        'ok',
                    );

                } else {

                    # Update the simple list
                    $self->family2Tab_refreshList($slWidget, (scalar @columnList / 2));
                }

                # In either case, reset the entry box
                $self->resetEntryBoxes($entry);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub family1Tab_refreshList {

        # Called by $self->family1Tab to refresh the GA::Obj::Simple::List
        #
        # Expected arguments
        #   $slWidget   - The GA::Obj::Simple::List
        #   $columns    - The number of columns in the list
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $slWidget, $columns, $check) = @_;

        # Local variables
        my (
            $parentNum, $parentObj,
            @dataList,
        );

        # Check for improper arguments
        if (! defined $slWidget || ! defined $columns || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->family1Tab_refreshList', @_);
        }

        # Import the IV
        $parentNum = $self->getEditHash_scalarIV('parent');
        if ($parentNum) {

            $parentObj = $self->session->worldModelObj->ivShow('modelHash', $parentNum);
        }

        # Compile the simple list data
        if ($parentNum && $parentObj) {

            push (@dataList,
                $parentNum,
                $parentObj->category,
                $parentObj->name,
            );
        }

        # Reset the simple list
        $self->resetListData($slWidget, [@dataList], $columns);

        return 1;
    }

    sub family2Tab_refreshList {

        # Called by $self->family2Tab to refresh the GA::Obj::Simple::List
        #
        # Expected arguments
        #   $slWidget   - The GA::Obj::Simple::List
        #   $columns    - The number of columns in the list
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $slWidget, $columns, $check) = @_;

        # Local variables
        my (
            @sortedList, @dataList,
            %ivHash,
        );

        # Check for improper arguments
        if (! defined $slWidget || ! defined $columns || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->family2Tab_refreshList', @_);
        }

        # Import the IV
        %ivHash = $self->getEditHash_hashIV('childHash');

        # Get a sorted list of child object numbers
        @sortedList = sort {$a <=> $b} (keys %ivHash);

        # Compile the simple list data
        foreach my $childNum (@sortedList) {

            my $childObj = $self->session->worldModelObj->ivShow('modelHash', $childNum);

            push (@dataList,
                $childObj->number,
                $childObj->category,
                $childObj->name,
            );
        }

        # Reset the simple list
        $self->resetListData($slWidget, [@dataList], $columns);

        return 1;
    }

    sub notesTab {

        # Notes tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->notesTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('N_otes', $self->notebook);

        # Definite articles
        $self->addLabel($table, '<b>Notes</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>Any additional information about this object you want to retain between'
            . ' sessions</i>',
            1, 12, 1, 2);

        my $textView = $self->addTextView($table, 'notesList', TRUE,
            1, 12, 2, 10,
            TRUE, FALSE, FALSE, FALSE,  # Treat as list, don't remove empty lines or whitespace
            -1, 300);                   # Fixed height
        my $buffer = $textView->get_buffer();

        # Add a button to clear the list
        my $button = $self->addButton($table,
            'Clear notes', 'Delete your notes for this object', undef,
            10, 12, 10, 11);
        $button->signal_connect('clicked' => sub {

            # Reset the IV
            $self->ivAdd('editHash', 'notesList', []);
            # Empty the list displayed in the tab
            $buffer->set_text('');
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    # Group 2

    sub wordsTab {

        # Words tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->wordsTab', @_);
        }

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('_Words', $self->notebook);

        # Add tabs to the inner notebook
        $self->words1Tab($innerNotebook);
        $self->words2Tab($innerNotebook);
        $self->words3Tab($innerNotebook);
        $self->words4Tab($innerNotebook);

        return 1;
    }

    sub words1Tab {

        # Words1 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->words1Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _1', $innerNotebook);

        # Left column
        $self->addLabel($table, '<b>Words</b>',
            0, 12, 0, 1);
        $self->addLabel($table, 'Main noun',
            1, 3, 1, 2);
        $self->addEntryWithIcon($table, 'noun', 'string', 1, undef,
            3, 6, 1, 2);
        $self->addLabel($table, 'Noun tag',
            1, 3, 2, 3);
        $self->addEntry($table, 'nounTag', FALSE,
            3, 6, 2, 3);

        $self->addLabel($table, 'Base string',
            1, 3, 3, 4);
        $self->addEntryWithButton($table, 'baseString', TRUE,
            3, 12, 3, 4);
        $self->addLabel($table, 'Description',
            1, 3, 4, 5);
        $self->addEntryWithButton($table, 'descrip', TRUE,
            3, 12, 4, 5);

        $self->addLabel($table, '<b>Inventory task properties</b>',
            0, 12, 5, 6);
        $self->addLabel($table, 'Container model number',
            1, 3, 6, 7);
        $self->addEntry($table, 'container', FALSE,
            3, 6, 6, 7);
        $self->addLabel($table, 'Inventory type',
            1, 3, 7, 8);
        $self->addEntry($table, 'inventoryType', FALSE,
            3, 6, 7, 8);

        # Right column
        $self->addLabel($table, 'Multiple',
            7, 9, 1, 2);
        $self->addEntryWithIcon($table, 'multiple', 'float', 0, undef,
            9, 12, 1, 2);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub words2Tab {

        # Words2 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->words2Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _2', $innerNotebook);

        # Other nouns
        $self->addLabel($table, '<b>Other nouns</b>',
            0, 12, 0, 2);
        $self->addLabel($table,
            '<i>Other words for this object which are known to be nouns (besides the main one)</i>',
            1, 12, 2, 4);
        $self->addTextView($table, 'otherNounList', TRUE,
            1, 12, 4, 6);

        # Adjectives
        $self->addLabel($table, '<b>Adjectives</b>',
            0, 12, 6, 8);
        $self->addLabel($table, '<i>Words for this object that are known to be adjectives</i>',
            1, 12, 8, 10);
        $self->addTextView($table, 'adjList', TRUE,
            1, 12, 10, 12);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub words3Tab {

        # Words3 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->words3Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _3', $innerNotebook);

        # Pseudo adjectives
        $self->addLabel($table, '<b>Pseudo-adjectives</b>',
            0, 12, 0, 2);
        $self->addLabel($table,
            '<i>Adjectives for this object which are linked to longer terms (e.g. \'big\' for'
            . ' \'big-looking\')</i>',
            1, 12, 2, 4);
        $self->addTextView($table, 'pseudoAdjList', TRUE,
            1, 12, 4, 6);

        # Root adjectives
        $self->addLabel($table, '<b>Root adjectives</b>',
            0, 12, 6, 8);
        $self->addLabel($table, '<i>Root forms of declined adjectives (not found in English)</i>',
            1, 12, 8, 10);
        $self->addTextView($table, 'rootAdjList', TRUE,
            1, 12, 10, 12);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub words4Tab {

        # Words4 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->word4Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _4', $innerNotebook);

        # Unknown words
        $self->addLabel($table, '<b>Unknown words</b>',
            0, 12, 0, 2);
        $self->addLabel($table,
            '<i>Other words for this object which aren\'t known nouns or adjectives</i>',
            1, 12, 2, 4);
        $self->addTextView($table, 'unknownWordList', TRUE,
            1, 12, 4, 6);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    # Group 3

    sub attacksTab {

        # Attacks tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->attacksTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('_Attacks', $self->notebook);

        # Left column
        $self->addLabel($table, '<b>Properties for use by your code during attacks</b>',
            0, 6, 0, 1);
        $self->addLabel($table, 'Target status',
            1, 3, 1, 2);
        $self->addEntry($table, 'targetStatus', FALSE,
            3, 6, 1, 2);
        $self->addLabel($table, 'Target (attack) type',
            1, 3, 2, 3);
        $self->addEntry($table, 'targetType', FALSE,
            3, 6, 2, 3);
        $self->addLabel($table, 'Target path',
            1, 3, 3, 4);
        $self->addEntry($table, 'targetPath', FALSE,
            3, 6, 3, 4);
        $self->addLabel($table, 'Target room number',
            1, 3, 4, 5);
        $self->addEntry($table, 'targetRoomNum', FALSE,
            3, 6, 4, 5);

        # Right column
        $self->addLabel($table, 'Already attacked',
            7, 11, 1, 2);
        $self->addCheckButton($table, 'alreadyAttackedFlag', FALSE,
            11, 12, 1, 2, 1, 0.5);
        $self->addLabel($table, 'Explicit in description',
            7, 11, 2, 3);
        $self->addCheckButton($table, 'explicitFlag', TRUE,
            11, 12, 2, 3, 1, 0.5);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    # Group 4

    sub physicalTab {

        # Physical tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->physicalTab', @_);
        }

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('_Physical', $self->notebook);

        # Add tabs to the inner notebook
        $self->physical1Tab($innerNotebook);
        $self->physical2Tab($innerNotebook);

        return 1;
    }

    sub physical1Tab {

        # Physical1 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->physical1Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _1', $innerNotebook);

        # Left column
        $self->addLabel($table, '<b>Physical properties</b>',
            0, 12, 0, 1);
        $self->addLabel($table, 'Explicitly in description',
            1, 5, 1, 2);
        $self->addCheckButton($table, 'explicitFlag', TRUE,
            5, 6, 1, 2, 1, 0.5);
        $self->addLabel($table, 'Weight (if known)',
            1, 3, 2, 3);
        $self->addEntryWithButton($table, 'weight', TRUE,
            3, 6, 2, 3);
        $self->addLabel($table, 'Condition (0-100)',
            1, 3, 3, 4);
        $self->addEntryWithIcon($table, 'condition', 'int', 0, 100,
            3, 6, 3, 4, 4, 4);
        $self->addLabel($table, 'Condition change flag',
            1, 5, 4, 5);
        $self->addCheckButton($table, 'conditionChangeFlag', FALSE,
            5, 6, 4, 5, 1, 0.5);

        # Right column
        $self->addLabel($table, 'Fixable',
            7, 11, 1, 2);
        $self->addCheckButton($table, 'fixableFlag', TRUE,
            11, 12, 1, 2, 1, 0.5);
        $self->addLabel($table, 'Sellable',
            7, 11, 2, 3);
        $self->addCheckButton($table, 'sellableFlag', TRUE,
            11, 12, 2, 3, 1, 0.5);
        $self->addLabel($table, 'Value (when bought)',
            7, 9, 3, 4);
        $self->addEntryWithButton($table, 'buyValue', TRUE,
            9, 12, 3, 4);
        $self->addLabel($table, 'Value (when sold)',
            7, 9, 4, 5);
        $self->addEntryWithButton($table, 'sellValue', TRUE,
            9, 12, 4, 5);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub physical2Tab {

        # Physical2 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->physical2Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _2', $innerNotebook);

        # Bonus stats
        $self->addLabel($table, '<b>Bonus stats</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>Stat bonuses and penalties when using this object</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Stat', 'text',
            'Value', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 220);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'bonusHash');

        # Add entry boxes and edit buttons
        $self->addLabel($table, 'Stat',
            1, 4, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            4, 7, 8, 9);

        $self->addLabel($table, 'Bonus/penalty',
            7, 10, 8, 9);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'float', undef, undef,
            10, 12, 8, 9);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'bonusHash',
            9,
            $entry, $entry2,
        );
        $button->signal_connect('clicked' => sub {

            my ($unit, $value);

            $unit = $entry->get_text();
            $value = $entry2->get_text();

            if ($self->checkEntryIcon($entry, $entry2)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('bonusHash', $unit, $value);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'bonusHash');
                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub exclusiveTab {

        # Exclusive tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my (@columnList, @sortedList);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->exclusiveTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('_Exclusive', $self->notebook);

        # Exclusive profiles
        $self->addLabel($table, '<b>Exclusive profiles</b>',
            0, 6, 0, 1);
        $self->addLabel($table,
            'Only exclusive profiles (characters, races, guilds, etc) can use this object?',
            1, 11, 1, 2);
        $self->addCheckButton($table, 'exclusiveFlag', TRUE,
            11, 12, 1, 2, 1, 0.5);

        # Add a simple list
        @columnList = (
            'Exclusive profile', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 280);     # Fixed height

        # Initialise the list
        $self->exclusiveTab_refreshList($slWidget, scalar (@columnList / 2));

        # Prepare a combo box with all of the current session's profiles (i.e. excluding all worlds
        #   except the current one)
        @sortedList = sort {lc($a) cmp lc($b)} ($self->session->ivKeys('profHash'));
        # Add the combo box
        $self->addLabel($table, 'Exclusive profile:',
            1, 3, 10, 11);
        my $comboBox = $self->addComboBox($table, undef, \@sortedList, '',
            TRUE,               # No 'undef' value used
            3, 6, 10, 11);

        # Add buttons
        my $button = $self->addButton($table,
            'Exclusive', 'Make the chosen profile exclusive', undef,
            6, 8, 10, 11);
        $button->signal_connect('clicked' => sub {

            my $profName = $comboBox->get_active_text();
            if ($profName && $self->session->ivExists('profHash', $profName)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV(
                    'exclusiveHash',
                    $profName,
                    $self->session->ivShow('profHash', $profName),
                );

                # Refresh the simple list
                $self->exclusiveTab_refreshList($slWidget, scalar (@columnList / 2));
            }
        });

        my $button2 = $self->addButton($table,
            'Not exclusive', 'Make the selected profile un-exclusive', undef,
            8, 10, 10, 11);
        $button2->signal_connect('clicked' => sub {

            my ($profName) = $self->getSimpleListData($slWidget, 0);
            if (defined $profName) {

                # Delete the key-value pair from the hash
                $self->modifyEditHash_hashIV('exclusiveHash', $profName, undef, TRUE);

                # Refresh the simple list
                $self->exclusiveTab_refreshList($slWidget, scalar (@columnList / 2));
            }
        });

        my $button3 = $self->addButton($table,
            'Clear', 'Clear the list of exclusive profiles', undef,
            10, 12, 10, 11);
        $button3->signal_connect('clicked' => sub {

            # Add an empty hash to $self->editHash
            $self->ivAdd('editHash', 'exclusiveHash', {});

            # Refresh the simple list
            $self->exclusiveTab_refreshList($slWidget, scalar (@columnList / 2));
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub exclusiveTab_refreshList {

        # Resets the simple list displayed by $self->exclusiveTab
        #
        # Expected arguments
        #   $slWidget   - The GA::Obj::Simple::List
        #   $columns    - The number of columns
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $slWidget, $columns, $check) = @_;

        # Local variables
        my (
            @dataList,
            %ivHash,
        );

        # Check for improper arguments
        if (! defined $slWidget || ! defined $columns || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->exclusiveTab_refreshList',
                @_,
            );
        }

        # Import the IV
        %ivHash = $self->getEditHash_hashIV('exclusiveHash');

        # Compile the simple list data
        @dataList = sort {lc($a) cmp lc($b)} (keys %ivHash);

        # Reset the simple list
        $self->resetListData($slWidget, [@dataList], $columns);

        return 1;
    }

    # Group 5

    sub sentient1Tab {

        # Sentient1 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (@comboList, @comboList2);

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->sentient1Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _1', $innerNotebook);

        # Left column
        $self->addLabel($table,
            '<b>Properties of ' . $self->editObj->category . ' model objects</b>',
            0, 12, 0, 1);

        $self->addLabel($table, 'Guild',
            1, 3, 1, 2);
        @comboList = sort {lc($a) cmp lc($b)} ($self->session->currentDict->ivKeys('guildHash'));
        $self->addComboBox($table, 'guild', \@comboList, '',
            FALSE,              # 'undef' value allowed
            3, 6, 1, 2);

        $self->addLabel($table, 'Race',
            1, 3, 2, 3);
        @comboList = sort {lc($a) cmp lc($b)} ($self->session->currentDict->ivKeys('raceHash'));
        $self->addComboBox($table, 'race', \@comboList2, '',
            FALSE,              # 'undef' value allowed
            3, 6, 2, 3);

        # Right column
        $self->addLabel($table, 'Level (if known)',
            7, 10, 1, 2);
        $self->addEntryWithIcon($table, 'level', 'int', 0, undef,
            10, 12, 1, 2, 8, 8);
        $self->addLabel($table, '(not less than)',
            8, 10, 2, 3);
        $self->addEntryWithIcon($table, 'weakerLevel', 'int', 0, undef,
            10, 12, 2, 3, 8, 8);
        $self->addLabel($table, '(not more than)',
            8, 10, 3, 4);
        $self->addEntryWithIcon($table, 'strongerLevel', 'int', 0, undef,
            10, 12, 3, 4, 8, 8);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub sentient2Tab {

        # Sentient2 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->sentient2Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _2', $innerNotebook);

        # Inventory list
        $self->addLabel($table, '<b>Inventory list</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of things the ' . $self->editObj->category . ' typically carries in its'
            . ' inventory</i>',
            1, 12, 1, 2);

        $self->addTextView($table, 'inventoryList', TRUE,
            1, 12, 2, 12,
            TRUE, TRUE, TRUE, FALSE,    # Treat as list, remove empty lines, do remove whitespace
            -1, 300);                   # Fixed height

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub sentient3Tab {

        # Sentient3 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->sentient3Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _3', $innerNotebook);

        # Cash list
        $self->addLabel($table, '<b>Cash list</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>Amounts of cash found on the ' . $self->editObj->category . '\'s corpse (most'
            . ' recent ten amounts)</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Amount' => 'text',
        );

        my $slWidget = $self->addSimpleList($table, 'cashList', \@columnList,
            1, 12, 2, 10,
            -1, 270);       # Fixed height

        # Add a button
        my $button = $self->addButton($table, 'Clear list', 'Clear the list of cash amounts', undef,
            10, 12, 10, 11);
        $button->signal_connect('clicked' => sub {

            # Add an empty list to $self->editHash
            $self->ivAdd('editHash', 'cashList', []);

            # Refresh the simple list
            $self->refreshList_listIV($slWidget, scalar (@columnList / 2), 'cashList');
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub sentient4Tab {

        # Sentient4 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (@comboList, @comboList2);

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->sentient4Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _4', $innerNotebook);

        # Left column
        $self->addLabel($table,
            '<b>Properties of ' . $self->editObj->category . ' model objects (continued)</b>',
            0, 12, 0, 1);

        $self->addLabel($table, 'Ever initiated combat (unfriendly)',
            1, 5, 1, 2);
        $self->addCheckButton($table, 'unfriendlyFlag', TRUE,
            5, 6, 1, 2, 1, 0.5);

        $self->addLabel($table, 'Morality',
            1, 5, 2, 3);
        @comboList = ('good', 'neutral', 'evil');
        $self->addComboBox($table, 'morality', \@comboList, '',
            TRUE,               # No 'undef' value used
            3, 6, 2, 3);

        $self->addLabel($table, 'Attached to quest',
            1, 3, 3, 4);
        @comboList2 = $self->session->currentWorld->ivKeys('questHash');
        $self->addComboBox($table, 'questName', \@comboList2, '',
            FALSE,              # 'undef' value allowed
            3, 6, 3, 4);

        # Right column
        $self->addLabel($table, 'Tends to wander around',
            7, 11, 1, 2);
        $self->addCheckButton($table, 'wanderFlag', TRUE,
            11, 12, 1, 2, 1, 0.5);
        $self->addLabel($table, 'Ever fleed combat',
            7, 11, 2, 3);
        $self->addCheckButton($table, 'fleeFlag', TRUE,
            11, 12, 2, 3, 1, 0.5);
        $self->addLabel($table, 'Tends to flee combat quickly',
            7, 11, 3, 4);
        $self->addCheckButton($table, 'quickFleeFlag', TRUE,
            11, 12, 3, 4, 1, 0.5);
        $self->addLabel($table, 'Should <b>NEVER</b> be attacked',
            7, 11, 4, 5);
        $self->addCheckButton($table, 'noAttackFlag', TRUE,
            11, 12, 4, 5, 1, 0.5);
        $self->addLabel($table, 'Mercies opponents (doesn\'t kill)',
            7, 11, 5, 6);
        $self->addCheckButton($table, 'mercyFlag', TRUE,
            11, 12, 5, 6, 1, 0.5);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub sentient5Tab {

        # Sentient5 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->sentient5Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _5', $innerNotebook);

        # Action properties
        $self->addLabel($table, '<b>Action properties</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>Flag to be set if the ' . $self->editObj->category
            . ' has ever performed an action</i>',
            1, 10, 1, 2);
        $self->addCheckButton($table, 'actionFlag', TRUE,
            10, 12, 1, 2, 1, 0.5);

        $self->addLabel($table,
            '<i>List of strings received when the ' . $self->editObj->category
            . ' performs an action:</i>',
            1, 10, 2, 3);
        $self->addTextView($table, 'actionList', TRUE,
            1, 12, 3, 12,
            TRUE, TRUE, FALSE, FALSE,   # Treat as list, remove empty lines, don't remove whitespace
            -1, 250);                   # Fixed height

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub sentient6Tab {

        # Sentient6 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->sentient6Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _6', $innerNotebook);

        # Conversation properties
        $self->addLabel($table, '<b>Conversation properties</b>',
            0, 12, 0, 1);

        $self->addLabel($table,
            '<i>Flag to be set if the ' . $self->editObj->category
            . ' has ever said anything</i>',
            1, 10, 1, 2);
        $self->addCheckButton($table, 'talkativeFlag', TRUE,
            10, 12, 1, 2, 1, 0.5);

        $self->addLabel($table,
            '<i>List of things spoken by the ' . $self->editObj->category . ':</i>',
            1, 10, 2, 3);
        $self->addTextView($table, 'talkList', TRUE,
            1, 12, 3, 12,
            TRUE, TRUE, FALSE, FALSE,   # Treat as list, remove empty lines, don't remove whitespace
            -1, 250);                   # Fixed height

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub character1Tab {

        # Character1 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (@comboList, @comboList2, @comboList3, @comboList4);

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->character1Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _1', $innerNotebook);

        # Left column
        $self->addLabel($table, '<b>Properties of ' . $self->editObj->category
            . ' model objects</b>',
            0, 12, 0, 1);

        $self->addLabel($table, 'Guild',
            1, 3, 1, 2);
        @comboList = $self->session->currentDict->ivKeys('guildHash');
        $self->addComboBox($table, 'guild', \@comboList, '',
            FALSE,              # 'undef' value allowed
            3, 6, 1, 2);

        $self->addLabel($table, 'Race',
            1, 3, 2, 3);
        @comboList2 = $self->session->currentDict->ivKeys('raceHash');
        $self->addComboBox($table, 'race', \@comboList2, '',
            FALSE,              # 'undef' value allowed
            3, 6, 2, 3);

        if ($self->editObj->category eq 'char') {

            # Characters
            $self->addLabel($table, 'Own character',
                1, 5, 3, 4);
            $self->addCheckButton($table, 'ownCharFlag', TRUE,
                5, 6, 3, 4, 1, 0.5);
            $self->addLabel($table, 'Owner',
                1, 3, 4, 5);
            $self->addEntryWithButton($table, 'owner', TRUE,
                3, 6, 4, 5);

            $self->addLabel($table, 'Mortal status',
                1, 3, 5, 6);
            @comboList3 = ('mortal', 'wiz', 'test');
            $self->addComboBox($table, 'mortalStatus', \@comboList3, '',
                TRUE,               # No 'undef' value used
                3, 6, 5, 6);

            $self->addLabel($table, 'Diplomatic status',
                1, 3, 6, 7);
            @comboList4 = ('friendly', 'neutral', 'hostile');
            $self->addComboBox($table, 'diplomaticStatus', \@comboList4, '',
                TRUE,               # No 'undef' value used
                3, 6, 6, 7);

            $self->addLabel($table, 'Ever attacked you',
                1, 5, 7, 8);
            $self->addCheckButton($table, 'grudgeFlag', TRUE,
                5, 6, 7, 8, 1, 0.5);
            $self->addLabel($table, 'Fight back if attacked by this',
                1, 5, 8, 9);
            $self->addCheckButton($table, 'fightBackFlag', TRUE,
                5, 6, 8, 9, 1, 0.5);

        } elsif ($self->editObj->category eq 'minion') {

            # Minions
            $self->addLabel($table, 'Minion cost',
                1, 3, 3, 4);
            $self->addEntryWithIcon($table, 'value', 'string', 0, undef,
                3, 6, 3, 4);
            $self->addLabel($table, 'Own minion',
                1, 5, 4, 5);
            $self->addCheckButton($table, 'ownMinionFlag', TRUE,
                5, 6, 4, 5, 1, 0.5);
        }

        # Right column
        $self->addLabel($table, 'Level (if known)',
            7, 10, 1, 2);
        $self->addEntryWithIcon($table, 'level', 'int', 0, undef,
            10, 12, 1, 2, 8, 8);
        $self->addLabel($table, '(not less than)',
            7, 10, 2, 3);
        $self->addEntryWithIcon($table, 'weakerLevel', 'int', 0, undef,
            10, 12, 2, 3, 8, 8);
        $self->addLabel($table, '(not more than)',
            7, 10, 3, 4);
        $self->addEntryWithIcon($table, 'strongerLevel', 'int', 0, undef,
            10, 12, 3, 4, 8, 8);

        if ($self->editObj->category eq 'char') {

            $self->addLabel($table, 'Total XP (if known)',
                7, 10, 4, 5);
            $self->addEntryWithButton($table, 'totalXP', 1,
                10, 12, 4, 5);
            $self->addLabel($table, 'Total QP (if known)',
                7, 10, 5, 6);
            $self->addEntryWithButton($table, 'totalQP', 1,
                10, 12, 5, 6);
        }

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::ModelObj::Region;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::EditWin::Generic::ModelObj Games::Axmud::Generic::EditWin
        Games::Axmud::Generic::ConfigWin Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win
        Games::Axmud

    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::ModelObj::Region')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook, depending on which category of model object is
        #   being edited
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        # Group 1
        $self->familyTab();
        $self->privateDataTab(       # Inherited from GA::Generic::EditWin
            'privateHash',
            '_Private',
            'Private properties hash',
            'Properties created by your own plugins and scripts, but stored in this type of object',
        );
        $self->notesTab();
        # Group 5
        $self->regionTab();

        return 1;
    }

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    # Group 5

    sub regionTab {

        # Region tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->regionTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('_Region', $self->notebook);

        $self->addLabel($table, '<b>General properties</b>',
            0, 6, 0, 1);

        # Temporary region
        $self->addLabel($table, 'Temporary region',
            1, 11, 1, 2);
        $self->addCheckButton($table, 'tempRegionFlag', FALSE,
            11, 12, 1, 2, 1, 0.5);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::ModelObj::Room;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::EditWin::Generic::ModelObj Games::Axmud::Generic::EditWin
        Games::Axmud::Generic::ConfigWin Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win
        Games::Axmud

    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::ModelObj::Room')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook, depending on which category of model object is
        #   being edited
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        # Group 1
        $self->familyTab();
        $self->privateDataTab(       # Inherited from GA::Generic::EditWin
            'privateHash',
            '_Private',
            'Private properties hash',
            'Properties created by your own plugins and scripts, but stored in this type of object',
        );
        $self->notesTab();
        # Group 5
        $self->roomTab();
        $self->exitsTab();
        $self->contentsTab();
        $self->protocolsTab();

        return 1;
    }

    sub saveChanges {

        # Called by $self->buttonOK and $self->buttonSave (usually for 'edit' windows only, not
        #   'pref' windows)
        # Saves any changes made to data stored by the edit object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my ($wmObj, $hashRef);

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->saveChanges', @_);
        }

        # Import the world model (for convenience)
        $wmObj = $self->session->worldModelObj;

        if ($self->editHash) {

            # For changes to ->involuntaryExitPatternHash, other objects in the world model must be
            #   updated
            if ($self->ivExists('editHash', 'involuntaryExitPatternHash')) {

                # Simplest way to deal with changes to involuntary exits is to delete all the
                #   existing patterns (if any), then add new ones (if any)
                foreach my $pattern ($self->editObj->ivKeys('involuntaryExitPatternHash')) {

                    $wmObj->removeInvoluntaryExit($self->editObj, $pattern);
                }

                $hashRef = $self->ivShow('editHash', 'involuntaryExitPatternHash');
                foreach my $pattern (keys %$hashRef) {

                    $wmObj->addInvoluntaryExit($self->editObj, $pattern, $$hashRef{$pattern});
                }

                $self->ivDelete('editHash', 'involuntaryExitPatternHash');
            }

            # Same process for ->repulseExitPatternHash
            if ($self->ivExists('editHash', 'repulseExitPatternHash')) {

                # Simplest way to deal with changes to repulse exits is to delete all the existing
                #   patterns (if any), then add new ones (if any)
                foreach my $pattern ($self->editObj->ivKeys('repulseExitPatternHash')) {

                    $wmObj->removeRepulseExit($self->editObj, $pattern);
                }

                $hashRef = $self->ivShow('editHash', 'repulseExitPatternHash');
                foreach my $pattern (keys %$hashRef) {

                    $wmObj->addRepulseExit($self->editObj, $pattern, $$hashRef{$pattern});
                }

                $self->ivDelete('editHash', 'repulseExitPatternHash');
            }

            # Store the changes the user has made
            foreach my $key ($self->ivKeys('editHash')) {

                $self->editObj->{$key} = $self->ivShow('editHash', $key);
            }

            # The changes can now be cleared
            $self->ivEmpty('editHash');

            # Mark the object's corresponding file object as needing to be saved, if it exists
            if ($self->editObj->_parentFile) {

                $self->editObj->doModify('saveChanges');
            }

            # Update the current session's object viewer window, if it is open
            if ($self->session->viewerWin) {

                $self->session->viewerWin->updateNotebook();
            }

            # Redraw the room in every automapper window using the same world model
            $wmObj->updateMaps('room', $self->editObj);
        }

        return 1;
    }

    # Group 5

    sub roomTab {

        # Room tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->roomTab', @_);
        }

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('_Room', $self->notebook);

        # Add tabs to the inner notebook
        $self->room1Tab($innerNotebook);
        $self->room2Tab($innerNotebook);
        $self->room3Tab($innerNotebook);
        $self->room4Tab($innerNotebook);
        $self->room5Tab($innerNotebook);
        $self->room6Tab($innerNotebook);
        $self->room7Tab($innerNotebook);

        return 1;
    }

    sub room1Tab {

        # Room1 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->room1Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _1', $innerNotebook);

        # Left column
        $self->addLabel($table,
            '<b>Properties of ' . $self->editObj->category . ' model objects</b>',
            0, 6, 0, 1);

        $self->addLabel($table, 'Room tag',
            1, 2, 1, 2);
        $self->addEntry($table, 'roomTag', FALSE,
            2, 6, 1, 2);
        $self->addLabel($table, 'Offset on map',
            1, 2, 2, 3);
        $self->addLabel($table, 'X:',
            2, 3, 2, 3);
        $self->addEntry($table, 'roomTagXOffset', FALSE,
            3, 4, 2, 3, 6, 6);
        $self->addLabel($table, 'Y:',
            4, 5, 2, 3);
        $self->addEntry($table, 'roomTagYOffset', FALSE,
            5, 6, 2, 3, 6, 6);

        $self->addLabel($table, 'Room guild',
            1, 2, 3, 4);
        $self->addEntry($table, 'roomGuild', FALSE,
            2, 6, 3, 4);
        $self->addLabel($table, 'Offset on map',
            1, 2, 4, 5);
        $self->addLabel($table, 'X:',
            2, 3, 4, 5);
        $self->addEntry($table, 'roomGuildXOffset', FALSE,
            3, 4, 4, 5, 6, 6);
        $self->addLabel($table, 'Y:',
            4, 5, 4, 5);
        $self->addEntry($table, 'roomGuildYOffset', FALSE,
            5, 6, 4, 5, 6, 6);

        $self->addLabel($table, 'Unspecified room description',
            1, 5, 5, 6);
        $self->addCheckButton($table, 'unspecifiedFlag', FALSE,
            5, 6, 5, 6, 1, 0.5);
        $self->addLabel($table, 'Currently dark',
            1, 5, 6, 7);
        $self->addCheckButton($table, 'currentlyDarkFlag', FALSE,
            5, 6, 6, 7, 1, 0.5);

        $self->addLabel($table, '<b>Wilderness mode</b>',
            0, 6, 7, 8);
        my $entry = $self->addEntry($table, undef, FALSE,
            1, 6, 8, 9);
        if ($self->editObj->wildMode eq 'normal') {
            $entry->set_text('\'normal\' - exits are required between rooms');
        } elsif ($self->editObj->wildMode eq 'border') {
            $entry->set_text('\'border\' - assume exits exist between adjacent wild rooms');
        } elsif ($self->editObj->wildMode eq 'wild') {
            $entry->set_text('\'wild\' - assume exits exist between all adjacent rooms');
        }

        # Right column
        $self->addLabel($table, '<b>Regionmap grid position</b>',
            7, 13, 0, 1);
        $self->addLabel($table, 'X block',
            8, 11, 1, 2);
        $self->addEntry($table, 'xPosBlocks', FALSE,
            11, 13, 1, 2, 4, 4);
        $self->addLabel($table, 'Y block',
            8, 11, 2, 3);
        $self->addEntry($table, 'yPosBlocks', FALSE,
            11, 13, 2, 3, 4, 4);
        $self->addLabel($table, 'Z block',
            8, 11, 3, 4);
        $self->addEntry($table, 'zPosBlocks', FALSE,
            11, 13, 3, 4, 4, 4);

        # Axbasic list
        $self->addLabel($table, '<b>' . $axmud::BASIC_NAME . ' list</b>',
            7, 13, 4, 5);
        $self->addLabel(
            $table,
            '<i>List of ' . $axmud::BASIC_NAME . ' scripts run when the room is entered</i>',
            8, 13, 5, 6);
        $self->addTextView($table, 'arriveScriptList', TRUE,
            8, 13, 6, 12);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub room2Tab {

        # Room2 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Optional arguments
        #   $tabTitle   - When inherited by GA::EditWin::Painter, the tab's page number (a string)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $tabTitle, $check) = @_;

        # Local variables
        my (
            $wmObj, $roomFlag, $roomFlagObj, $colour, $noUpdateFlag,
            @columnList, @comboList, @comboList2,
        );

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->room2Tab', @_);
        }

        # Import the world model (for convenience)
        $wmObj = $self->session->worldModelObj;

        # Tab setup
        if (! $tabTitle) {

            $tabTitle = 'Page _2';      # This function wasn't called by GA::EditWin::Painter
        }

        my ($vBox, $table) = $self->addTab($tabTitle, $innerNotebook);

        # Room flags
        $self->addLabel($table, '<b>Room flags</b>',
            0, 6, 0, 1);
        $self->addLabel($table,
            '<i>Flags that determine how the room is drawn in the automapper</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Room flag', 'text',
            'Description', 'text',
            'Priority', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 230);     # Fixed height

        # Initialise the list
        $self->room2Tab_refreshList($slWidget, scalar (@columnList / 2));

        # Add editing widgets
        $self->addLabel($table, 'Room flag category (filter)',
            1, 4, 10, 11);
        @comboList = $axmud::CLIENT->constRoomFilterList;
        my $combo = $self->addComboBox($table, undef, \@comboList, '',
            TRUE,              # 'undef' value not used
            4, 6, 10, 11);
        # ->signal_connect appears below

        $self->addLabel($table, 'Room flag',
            6, 7, 10, 11);
        @comboList2
            = $self->session->worldModelObj->getRoomFlagsInFilter($combo->get_active_text());

        my $combo2 = $self->addComboBox($table, undef, \@comboList2, '',
            TRUE,              # 'undef' value not used
            7, 10, 10, 11);
        # ->signal_connect appears below

        $self->addLabel($table, 'Colour',
            10, 11, 10, 11);
        $roomFlag = $combo2->get_active_text();
        if ($roomFlag) {

            $roomFlagObj = $self->session->worldModelObj->ivShow('roomFlagHash', $roomFlag);
            if ($roomFlagObj) {

                $colour = $roomFlagObj->colour;
            }
        }
        my ($frame, $canvas, $canvasObj) = $self->addSimpleCanvas($table, $colour, undef,
            11, 12, 10, 11);

        # ->signal_connect from above
        $combo->signal_connect('changed' => sub {

            my ($text, $text2);

            $text = $combo->get_active_text();

            # Don't let the ->signal_connect below react before we're ready
            $noUpdateFlag = TRUE;
            $self->resetComboBox(
                $combo2,
                $self->session->worldModelObj->getRoomFlagsInFilter($text),
            );

            $noUpdateFlag = FALSE;

            $text2 = $combo2->get_active_text();
            if ($text2) {
                $roomFlagObj = $self->session->worldModelObj->ivShow('roomFlagHash', $text2);
            } else {
                $roomFlagObj = undef;
            }

            if ($roomFlagObj) {
                $canvasObj = $self->fillSimpleCanvas($canvas, $canvasObj, $roomFlagObj->colour);
            } else {
                $canvasObj = $self->fillSimpleCanvas($canvas, $canvasObj, undef);
            }
        });

        $combo2->signal_connect('changed' => sub {

            my $text2 = $combo2->get_active_text();

            if (! $noUpdateFlag) {

                if ($text2) {
                    $roomFlagObj = $self->session->worldModelObj->ivShow('roomFlagHash', $text2);
                } else {
                    $roomFlagObj = undef;
                }

                if ($roomFlagObj) {
                    $canvasObj = $self->fillSimpleCanvas($canvas, $canvasObj, $roomFlagObj->colour);
                } else {
                    $canvasObj = $self->fillSimpleCanvas($canvas, $canvasObj, undef);
                }
            }
        });

        $self->addLabel($table, 'Last flag drawn',
            1, 3, 11, 12);
        $self->addEntry($table, 'lastRoomFlag', FALSE,
            3, 6, 11, 12);

        my $button = $self->addButton($table,
            'Use', 'The automapper can use this room flag when drawing the room', undef,
            6, 8, 11, 12);
        $button->signal_connect('clicked' => sub {

            my $flag = $combo2->get_active_text();
            if ($flag) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('roomFlagHash', $flag, undef);

                # Refresh the simple list
                $self->room2Tab_refreshList($slWidget, scalar (@columnList / 2));
            }
        });

        my $button2 = $self->addButton($table,
            'Don\'t use', 'The automapper can\'t use this room flag when drawing the room', undef,
            8, 10, 11, 12);
        $button2->signal_connect('clicked' => sub {

            my $flag = $combo2->get_active_text();
            if ($flag) {

                # Delete the key-value pair from the hash
                $self->modifyEditHash_hashIV('roomFlagHash', $flag, undef, TRUE);

                # Refresh the simple list
                $self->room2Tab_refreshList($slWidget, scalar (@columnList / 2));
            }
        });

        my $button3 = $self->addButton($table, 'Clear', 'Clear the list of room flags', undef,
            10, 12, 11, 12);
        $button3->signal_connect('clicked' => sub {

            # Add an empty hash to $self->editHash
            $self->ivAdd('editHash', 'roomFlagHash', {});

            # Refresh the simple list
            $self->room2Tab_refreshList($slWidget, scalar (@columnList / 2));
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub room2Tab_refreshList {

        # Called by $self->room2Tab to refresh the GA::Obj::Simple::List
        #
        # Expected arguments
        #   $slWidget   - The GA::Obj::Simple::List
        #   $columns    - The number of columns in the list
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $slWidget, $columns, $check) = @_;

        # Local variables
        my (
            $wmObj,
            @list, @sortedList, @dataList,
            %ivHash,
        );

        # Check for improper arguments
        if (! defined $slWidget || ! defined $columns || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->room2Tab_refreshList', @_);
        }

        # Import the world model (for speed)
        $wmObj = $self->session->worldModelObj;
        # Import the IV, and sort by priority
        %ivHash = $self->getEditHash_hashIV('roomFlagHash');
        foreach my $key (keys %ivHash) {

            my $roomFlagObj = $wmObj->ivShow('roomFlagHash', $key);
            if ($roomFlagObj) {

                push (@list, $roomFlagObj);
            }
        }

        @sortedList = sort {$a->priority <=> $b->priority} (@list);

        # Compile the simple list data
        foreach my $roomFlagObj (@sortedList) {

            push (@dataList,
                $roomFlagObj->name,
                $roomFlagObj->descrip,
                $roomFlagObj->priority,
            );
        }

        # Reset the simple list
        $self->resetListData($slWidget, [@dataList], $columns);

        return 1;
    }

    sub room3Tab {

        # Room3 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->room3Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _3', $innerNotebook);

        # Room titles
        $self->addLabel($table, '<b>Room titles</b>',
            0, 6, 0, 2);
        $self->addLabel($table, '<i>List of room titles (brief descriptions) for this room</i>',
            1, 6, 2, 4);
        $self->addTextView($table, 'titleList', TRUE,
            1, 6, 4, 6);

        # Room commands
        $self->addLabel($table, '<b>Room commands</b>',
            6, 12, 0, 2);
        $self->addLabel($table, '<i>List of commands available in this room</i>',
            7, 12, 2, 4);
        $self->addTextView($table, 'roomCmdList', TRUE,
            7, 12, 4, 6);

        # Unspecified room patterns
        $self->addLabel($table, '<b>Unspecified room patterns</b>',
            0, 12, 6, 8);
        $self->addLabel(
            $table,
            '<i>Patterns seen when the character moves to a room without a recognisable room'
            . ' statement</i>',
            1, 12, 8, 10);
        $self->addTextView($table, 'unspecifiedPatternList', TRUE,
            1, 12, 10, 12);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub room4Tab {

        # Room4 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Optional arguments
        #   $tabTitle       - When inherited by GA::EditWin::Painter, the tab's page number (a
        #                       string)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $tabTitle, $check) = @_;

        # Local variables
        my (@columnList, @comboList);

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->room4Tab', @_);
        }

        # Tab setup
        if (! $tabTitle) {

            $tabTitle = 'Page _4';  # This function wasn't called by GA::EditWin::Painter
        }

        my ($vBox, $table) = $self->addTab($tabTitle, $innerNotebook);

        # Verbose descriptions
        $self->addLabel($table, '<b>Verbose descriptions</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>List of verbose descriptions for each light status</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Light status', 'text',
            'Verbose descrip', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 200);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'descripHash');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Light status',
            1, 3, 8, 9);
        @comboList = $self->session->worldModelObj->lightStatusList;
        my $combo = $self->addComboBox($table, undef, \@comboList, '',
            TRUE,               # No 'undef' value used
            3, 6, 8, 9);

        $self->addLabel($table, 'Verbose description',
            1, 3, 9, 10);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            3, 12, 9, 10);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'descripHash',
            10,
            $entry,
        );
        $button->signal_connect('clicked' => sub {

            my ($status, $descrip);

            $status = $combo->get_active_text();
            $descrip = $entry->get_text();

            if ($self->checkEntryIcon($entry)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('descripHash', $status, $descrip);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'descripHash');
                $self->resetEntryBoxes($entry);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub room5Tab {

        # Room5 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Optional arguments
        #   $tabTitle       - When inherited by painter object (a non-model GA::ModelObj::Room),
        #                       the tab's page number (a string)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $tabTitle, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->room5Tab', @_);
        }

        # Tab setup
        if (! $tabTitle) {

            $tabTitle = 'Page _5';  # This function wasn't called by the painter object
        }

        my ($vBox, $table) = $self->addTab($tabTitle, $innerNotebook);

        # Noun list
        $self->addLabel($table, '<b>Noun list</b>',
            0, 12, 0, 2);
        $self->addLabel($table,
            '<i>List of recognised nouns that appear in this room and its contents</i>',
            1, 12, 2, 4);
        $self->addTextView($table, 'nounList', TRUE,
            1, 12, 4, 6);

        # Adjective list
        $self->addLabel($table, '<b>Adjective list</b>',
            0, 12, 6, 8);
        $self->addLabel($table,
            '<i>List of recognised adjectives that appear in this room and its contents</i>',
            1, 12, 8, 10);
        $self->addTextView($table, 'adjList', TRUE,
            1, 12, 10, 12);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub room6Tab {

        # Room6 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (@columnList, @charList, @sortedList);

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->room6Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _6', $innerNotebook);

        # Character visits
        $self->addLabel($table, '<b>Character visits</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>Record of how many visits each character has made to this room</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Character', 'text',
            'Number of visits', 'int',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 220);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'visitHash');

        # Get a sorted list of characters, not including the current character (if any)
        foreach my $profObj ($self->session->ivValues('profHash')) {

            if (
                $profObj->category eq 'char'
                && (
                    ! $self->session->currentChar
                    || $self->session->currentChar ne $profObj
                )
            ) {
                push (@charList, $profObj->name);
            }
        }

        @sortedList = sort {lc($a) cmp lc($b)} (@charList);

        # Put the current char (if any) at the beginning of the list
        if ($self->session->currentChar) {

            unshift (@sortedList, $self->session->currentChar->name);
        }

        # Add entries/comboboxes for adding new key/value pairs
        $self->addLabel($table, 'Character',
            1, 3, 8, 9);
        my $combo = $self->addComboBox($table, undef, \@sortedList, '',
            TRUE,               # No 'undef' value used
            3, 6, 8, 9);

        $self->addLabel($table, 'Number of visits',
            7, 9, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'int', 0, undef,
            9, 12, 8, 9);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'visitHash',
            10,
            $entry,
        );
        $button->signal_connect('clicked' => sub {

            my ($char, $visitCount);

            $char = $combo->get_active_text();
            $visitCount = $entry->get_text();

            if ($self->checkEntryIcon($entry)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('visitHash', $char, $visitCount);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'visitHash');
                $self->resetEntryBoxes($entry);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub room7Tab {

        # Room7 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Optional arguments
        #   $tabTitle       - When inherited by GA::EditWin::Painter, the tab's page number (a
        #                       string)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $tabTitle, $check) = @_;

        # Local variables
        my (@columnList, @profList, @sortedList);

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->room7Tab', @_);
        }

        # Tab setup
        if (! $tabTitle) {

            $tabTitle = '_Page _7';     # This function wasn't called by GA::EditWin::Painter
        }

        my ($vBox, $table) = $self->addTab($tabTitle, $innerNotebook);

        # Exclusive profiles
        $self->addLabel($table, '<b>Exclusive profiles</b>',
            0, 6, 0, 1);
        $self->addLabel($table,
            'Only exclusive profiles (characters, races, guilds, etc) can enter this room?',
            1, 11, 1, 2);
        $self->addCheckButton($table, 'exclusiveFlag', TRUE,
            11, 12, 1, 2, 1, 0.5);

        # Add a simple list
        @columnList = (
            'Exclusive profile', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 250);     # Fixed height

        # Initialise the list (the function is inherited from GA::EditWin::Generic::ModelObj)
        $self->exclusiveTab_refreshList($slWidget, scalar (@columnList / 2));

        # Prepare a combo box with all of the current session's profiles (excluding all worlds
        #   including the current one)
        foreach my $profObj ($self->session->ivValues('profHash')) {

            if ($profObj->category ne 'world') {

                push (@profList, $profObj->name);
            }
        }

        @sortedList = sort {lc($a) cmp lc($b)} (@profList);

        # Add the combo box
        $self->addLabel($table, 'Exclusive profile:',
            1, 3, 10, 11);
        my $comboBox = $self->addComboBox($table, undef, \@sortedList, '',
            TRUE,               # No 'undef' value used
            3, 6, 10, 11);

        # Add buttons
        my $button = $self->addButton($table,
            'Exclusive', 'Make the chosen profile exclusive', undef,
            6, 8, 10, 11);
        $button->signal_connect('clicked' => sub {

            my $profName = $comboBox->get_active_text();
            if ($profName && $self->session->ivExists('profHash', $profName)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV(
                    'exclusiveHash',
                    $profName,
                    $self->session->ivShow('profHash', $profName),
                );

                # Refresh the simple list
                $self->exclusiveTab_refreshList($slWidget, scalar (@columnList / 2));
            }
        });

        my $button2 = $self->addButton($table,
            'Not exclusive', 'Make the selected profile un-exclusive', undef,
            8, 10, 10, 11);
        $button2->signal_connect('clicked' => sub {

            my ($profName) = $self->getSimpleListData($slWidget, 0);
            if (defined $profName) {

                # Delete the key-value pair from the hash
                $self->modifyEditHash_hashIV('exclusiveHash', $profName, undef, TRUE);

                # Refresh the simple list
                $self->exclusiveTab_refreshList($slWidget, scalar (@columnList / 2));
            }
        });

        my $button3 = $self->addButton($table,
            'Clear', 'Clear the list of exclusive profiles', undef,
            10, 12, 10, 11);
        $button3->signal_connect('clicked' => sub {

            # Add an empty hash to $self->editHash
            $self->ivAdd('editHash', 'exclusiveHash', {});

            # Refresh the simple list
            $self->exclusiveTab_refreshList($slWidget, scalar (@columnList / 2));
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub exitsTab {

        # Exits tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->exitsTab', @_);
        }

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('_Exits', $self->notebook);

        # Add tabs to the inner notebook
        $self->exits1Tab($innerNotebook);
        $self->exits2Tab($innerNotebook);
        $self->exits3Tab($innerNotebook);
        $self->exits4Tab($innerNotebook);
        $self->exits5Tab($innerNotebook);
        $self->exits6Tab($innerNotebook);
        $self->exits7Tab($innerNotebook);
        $self->exits8Tab($innerNotebook);
        $self->exits9Tab($innerNotebook);

        return 1;
    }

    sub exits1Tab {

        # Exits1 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->exits1Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _1', $innerNotebook);

        # Failed exit patterns
        $self->addLabel($table, '<b>Failed exit patterns</b>',
            0, 10, 0, 2);
        $self->addLabel(
            $table,
            '<i>Patterns which mean an exit in this room is (temporarily) unavailable</i>',
            1, 10, 2, 4);
        $self->addRegexButton($table,
            [
                'list', 'failExitPatternList',
                'list', 'specialDepartPatternList',
            ],
            10, 12, 0, 4);

        $self->addTextView($table, 'failExitPatternList', TRUE,
            1, 12, 4, 6,
            TRUE, TRUE, FALSE, FALSE,   # Treat as list, remove empty lines, don't remove whitespace
            -1, 130);                   # Fixed height

        # Special departure patterns
        $self->addLabel($table, '<b>Special departure patterns</b>',
            0, 12, 6, 8);
        $self->addLabel(
            $table,
            '<i>Patterns which mean we\'ve left the room (when a new room statement isn\'t'
            . ' sent)</i>',
            1, 12, 8, 10);

        $self->addTextView($table, 'specialDepartPatternList', TRUE,
            1, 12, 10, 12,
            TRUE, TRUE, FALSE, FALSE,   # Treat as list, remove empty lines, don't remove whitespace
            -1, 130);                   # Fixed height

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub exits2Tab {

        # Exits2 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->exits2Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _2', $innerNotebook);

        # Involuntary exit patterns
        $self->addLabel($table, '<b>Involuntary exit patterns</b>',
            0, 12, 0, 2);
        $self->addLabel(
            $table,
            '<i>Patterns which mean the character has left the room involuntarily</i>',
            1, 10, 2, 4);

        # Add a simple list
        @columnList = (
            'Pattern', 'text',
            'Optional direction or destination room #', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 220);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV(
            $slWidget,
            scalar (@columnList / 2),
            'involuntaryExitPatternHash',
        );

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Pattern',
            1, 3, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            3, 12, 8, 9);

        $self->addLabel($table, '(Optional) direction or destination room number',
            1, 4, 9, 10);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'string', undef, undef,
            4, 12, 9, 10);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'involuntaryExitPatternHash',
            10,
            $entry,
        );
        $button->signal_connect('clicked' => sub {

            my ($pattern, $value);

            $pattern = $entry->get_text();
            $value = $entry2->get_text();
            # Use 'undef' rather than an empty string
            if ($value eq '') {

                $value = undef;
            }

            if ($self->checkEntryIcon($entry)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('involuntaryExitPatternHash', $pattern, $value);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV(
                    $slWidget,
                    scalar (@columnList / 2),
                    'involuntaryExitPatternHash',
                );

                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub exits3Tab {

        # Exits3 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->exits3Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _3', $innerNotebook);

        # Repulse exit patterns
        $self->addLabel($table, '<b>Repulse exit patterns</b>',
            0, 12, 0, 2);
        $self->addLabel(
            $table,
            '<i>Patterns which mean the character has left the room involuntarily after a failed'
            . ' move</i>',
            1, 10, 2, 4);

        # Add a simple list
        @columnList = (
            'Pattern', 'text',
            'Optional direction or destination room #', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 220);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV(
            $slWidget,
            scalar (@columnList / 2),
            'repulseExitPatternHash',
        );

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Pattern',
            1, 3, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            3, 12, 8, 9);

        $self->addLabel($table, '(Optional) direction or destination room number',
            1, 4, 9, 10);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'string', undef, undef,
            4, 12, 9, 10);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'repulseExitPatternHash',
            10,
            $entry,
        );
        $button->signal_connect('clicked' => sub {

            my ($pattern, $value);

            $pattern = $entry->get_text();
            $value = $entry2->get_text();
            # Use 'undef' rather than an empty string
            if ($value eq '') {

                $value = undef;
            }

            if ($self->checkEntryIcon($entry)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('repulseExitPatternHash', $pattern, $value);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV(
                    $slWidget,
                    scalar (@columnList / 2),
                    'repulseExitPatternHash',
                );

                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub exits4Tab {

        # Exits4 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->exits4Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _4', $innerNotebook);

        # Exit objects
        $self->addLabel($table, '<b>Exit objects</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>List of exit objects for each exit in this room</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Exit #', 'text',
            'Direction', 'text',
            'Map direction', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 230);     # Fixed height

        # Initialise the list
        $self->exits4Tab_refreshList($slWidget, scalar (@columnList / 2));

        # Add entry box for adding new exit objects
        $self->addLabel($table, 'Direction',
            1, 3, 10, 11);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            3, 6, 10, 11);

        # Add editing buttons
        my $button = $self->addButton(
            $table,
            'Add',
            'Add the direction to the exit model',
            undef,
            6, 8, 10, 11,
            TRUE,           # Irreversible
        );
        $button->signal_connect('clicked' => sub {

            my $dir = $entry->get_text();

            if ($self->checkEntryIcon($entry)) {

                # If the button is sensitised, this must be a world model room, so add the direction
                #   to the exit model
                $self->session->pseudoCmd(
                    'addexit ' . $dir . ' ' . $self->editObj->number,
                    $self->pseudoCmdMode,
                );

                # Refresh the simple list
                $self->exits4Tab_refreshList($slWidget, scalar (@columnList / 2));
            }
        });

        my $button2 = $self->addButton(
            $table,
            'Edit...',
            'Edit the selected exit object',
            undef,
            8, 10, 10, 11,
        );
        $button2->signal_connect('clicked' => sub {

            my ($exitNum, $exitObj);

            ($exitNum) = $self->getSimpleListData($slWidget, 0);
            if (defined $exitNum) {

                $exitObj = $self->session->worldModelObj->ivShow('exitModelHash', $exitNum);

                # Open an 'edit' window for the exit object
                $self->createFreeWin(
                    'Games::Axmud::EditWin::Exit',
                    $self,
                    $self->session,
                    'Edit exit model object #' . $exitObj->number,
                    $exitObj,
                    FALSE,                          # Not temporary
                );

                # Refresh the simple list and reset entry boxes
                $self->exits4Tab_refreshList($slWidget, scalar (@columnList / 2));
                $self->resetEntryBoxes($entry);
            }
        });

        my $button3 = $self->addButton(
            $table,
            'Delete',
            'Delete the selected exit object',
            undef,
            10, 12, 10, 11,
            TRUE,           # Irreversible
        );
        $button3->signal_connect('clicked' => sub {

            my ($exitNum, $exitObj);

            ($exitNum) = $self->getSimpleListData($slWidget, 0);
            if (defined $exitNum) {

                $exitObj = $self->session->worldModelObj->ivShow('exitModelHash', $exitNum);

                # If the button is sensitised, this must be a world model room, so delete the exit
                #   from the exit model
                $self->session->pseudoCmd(
                    'deleteexit ' . $exitNum,
                    $self->pseudoCmdMode,
                );

                # Refresh the simple list and reset entry boxes
                $self->exits4Tab_refreshList($slWidget, scalar (@columnList / 2));
                $self->resetEntryBoxes($entry);
            }
        });

        my $button4 = $self->addButton(
            $table,
            'Delete exit and twin',
            'Delete the selected exit object and its twin, if any',
            undef,
            6, 9, 11, 12,
            TRUE,           # Irreversible
        );
        $button4->signal_connect('clicked' => sub {

            my ($exitNum, $exitObj);

            ($exitNum) = $self->getSimpleListData($slWidget, 0);
            if (defined $exitNum) {

                $exitObj = $self->session->worldModelObj->ivShow('exitModelHash', $exitNum);

                # If the button is sensitised, this must be a world model room, so delete the exit
                #   and its twin from the exit model
                $self->session->pseudoCmd(
                    'deleteexit -t ' . $exitNum,
                    $self->pseudoCmdMode,
                );

                # Refresh the simple list and reset entry boxes
                $self->exits4Tab_refreshList($slWidget, scalar (@columnList / 2));
                $self->resetEntryBoxes($entry);
            }
        });

        my $button5 = $self->addButton(
            $table,
            'Refresh list',
            'Refresh the list of exits',
            undef,
            9, 12, 11, 12,
            TRUE,       # Irreversible
        );
        $button5->signal_connect('clicked' => sub {

            # Refresh the simple list and reset entry boxes
            $self->exits4Tab_refreshList($slWidget, scalar (@columnList / 2));
            $self->resetEntryBoxes($entry);
        });

        # If it's a temporary room (not in the world model), desensitise the entry box and buttons
        if (! $self->editObj->modelFlag) {

            $entry->set_state('insensitive');
            $button->set_state('insensitive');
            $button2->set_state('insensitive');
            $button3->set_state('insensitive');
            $button4->set_state('insensitive');
        }

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub exits4Tab_refreshList {

        # Called by $self->exits3Tab to refresh the GA::Obj::Simple::List
        #
        # Expected arguments
        #   $slWidget   - The GA::Obj::Simple::List
        #   $columns    - The number of columns in the list
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $slWidget, $columns, $check) = @_;

        # Local variables
        my (
            @sortedExitList, @dataList,
            %exitNumHash,
        );

        # Check for improper arguments
        if (! defined $slWidget || ! defined $columns || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->exits4Tab_refreshList', @_);
        }

        # Import the IVs directly from the edited object (because changes are made directly, and
        #   not stored in $self->editHash)
        @sortedExitList = $self->editObj->sortedExitList;
        %exitNumHash = $self->editObj->exitNumHash;

        # Compile the simple list data
        foreach my $dir (@sortedExitList) {

            my ($number, $exitObj, $mapDir);

            $number = $exitNumHash{$dir};
            if (defined $number) {

                $exitObj = $self->session->worldModelObj->ivShow('exitModelHash', $number);
            }

            if (defined $exitObj) {

                if ($exitObj->mapDir) {
                    $mapDir = $exitObj->mapDir;
                } else {
                    $mapDir = 'unallocatable';
                }
            }

            if (defined $exitObj && $exitObj->modelFlag) {

                # In exit model
                push (@dataList,
                    $number,
                    $exitObj->dir,
                    $mapDir,
                );

            } else {

                # Not in exit model
                push (@dataList,
                    undef,
                    $dir,
                    undef,
                );
            }
        }

        # Reset the simple list
        $self->resetListData($slWidget, [@dataList], $columns);

        return 1;
    }

    sub exits5Tab {

        # Exits5 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->exits5Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _5', $innerNotebook);

        # Incoming uncertain exits
        $self->addLabel($table, '<b>Incoming uncertain exits</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of uncertain exits belonging to other rooms which point to this room</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Room #', 'text',
            'Region', 'text',
            'Exit #', 'text',
            'Direction', 'text',
            'Map direction', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 270);     # Fixed height

        # Initialise the list
        $self->exits5Tab_refreshList($slWidget, scalar (@columnList / 2), 'uncertainExitHash');

        # Add a single button
        my $button = $self->addButton(
            $table,
            'Refresh list',
            'Refresh the list of incoming uncertain exits',
            undef,
            9, 12, 10, 11,
        );
        $button->signal_connect('clicked' => sub {

            # Refresh the simple list
            $self->exits5Tab_refreshList(
                $slWidget,
                scalar (@columnList / 2),
                'uncertainExitHash',
            );
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub exits5Tab_refreshList {

        # Called by $self->exits4Tab, ->exits5Tab and ->exits6Tab to refresh the
        #   GA::Obj::Simple::List
        #
        # Expected arguments
        #   $slWidget   - The GA::Obj::Simple::List
        #   $columns    - The number of columns in the list
        #   $iv         - The IV to show
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $slWidget, $columns, $iv, $check) = @_;

        # Local variables
        my (@exitNumList, @dataList);

        # Check for improper arguments
        if (! defined $slWidget || ! defined $columns || ! defined $iv || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->exits5Tab_refreshList', @_);
        }

        # Import the IV
        @exitNumList = $self->editObj->ivKeys($iv);

        # Compile the simple list data
        foreach my $number (@exitNumList) {

            my ($exitObj, $roomObj, $regionObj);

            $exitObj = $self->session->worldModelObj->ivShow('exitModelHash', $number);
            $roomObj = $self->session->worldModelObj->ivShow('modelHash', $exitObj->parent);
            $regionObj = $self->session->worldModelObj->ivShow('modelHash', $roomObj->parent);

            push (@dataList,
                $roomObj->number,
                $regionObj->name,
                $exitObj->number,
                $exitObj->dir,
                $exitObj->mapDir,
            );
        }

        # Reset the simple list
        $self->resetListData($slWidget, [@dataList], $columns);

        return 1;
    }

    sub exits6Tab {

        # Exits6 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->exits6Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _6', $innerNotebook);

        # Incoming one-way exits
        $self->addLabel($table, '<b>Incoming one-way exits</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of one-way exits belonging to other rooms which point to this room</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Room #', 'text',
            'Region', 'text',
            'Exit #', 'text',
            'Direction', 'text',
            'Map direction', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 270);     # Fixed height

        # Initialise the list
        $self->exits5Tab_refreshList($slWidget, scalar (@columnList / 2), 'oneWayExitHash');

        # Add a single button
        my $button = $self->addButton(
            $table,
            'Refresh list',
            'Refresh the list of incoming one-way exits',
            undef,
            9, 12, 10, 11,
        );
        $button->signal_connect('clicked' => sub {

            # Refresh the simple list
            $self->exits5Tab_refreshList($slWidget, scalar (@columnList / 2), 'oneWayExitHash');
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub exits7Tab {

        # Exits7 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->exits7Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _7', $innerNotebook);

        # Incoming involuntary exits
        $self->addLabel($table, '<b>Incoming involuntary exits</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of rooms with involuntary exit patterns whose corresponding destination is'
            . ' this room</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Room #', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 270);     # Fixed height

        # Initialise the list
        $self->exits7Tab_refreshList($slWidget, scalar (@columnList / 2));

        # Add a single button
        my $button = $self->addButton(
            $table,
            'Refresh list',
            'Refresh the list of incoming involuntary exits',
            undef,
            9, 12, 10, 11,
        );
        $button->signal_connect('clicked' => sub {

            # Refresh the simple list
            $self->exits7Tab_refreshList($slWidget, scalar (@columnList / 2));
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub exits7Tab_refreshList {

        # Called by $self->exits6Tab to refresh the GA::Obj::Simple::List
        #
        # Expected arguments
        #   $slWidget   - The GA::Obj::Simple::List
        #   $columns    - The number of columns in the list
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $slWidget, $columns, $check) = @_;

        # Local variables
        my @dataList;

        # Check for improper arguments
        if (! defined $slWidget || ! defined $columns || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->exits7Tab_refreshList', @_);
        }

        # Import the IV
        @dataList = sort {$a <=> $b} ($self->editObj->ivValues('invRepExitHash'));

        # Reset the simple list
        $self->resetListData($slWidget, [@dataList], $columns);

        return 1;
    }

    sub exits8Tab {

        # Exits8 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->exits8Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _8', $innerNotebook);

        # Incoming random exits
        $self->addLabel($table, '<b>Incoming random exits</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of random exits belonging to other rooms which point to this room</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Room #', 'text',
            'Region', 'text',
            'Exit #', 'text',
            'Direction', 'text',
            'Map direction', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 270);     # Fixed height

        # Initialise the list
        $self->exits5Tab_refreshList($slWidget, scalar (@columnList / 2), 'randomExitHash');

        # Add a single button
        my $button = $self->addButton(
            $table,
            'Refresh list',
            'Refresh the list of incoming random exits',
            undef,
            9, 12, 10, 11,
        );
        $button->signal_connect('clicked' => sub {

            # Refresh the simple list
            $self->exits5Tab_refreshList(
                $slWidget,
                scalar (@columnList / 2),
                'randomExitHash',
            );
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub exits9Tab {

        # Exits9 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->exits9Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _9', $innerNotebook);

        # Checked directions
        $self->addLabel($table, '<b>Checked directions</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of directions the character has tried, but which generated a failed exit'
            . ' message</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Direction', 'text',
            'Number of failed attempts', 'int',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 270);     # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'checkedDirHash');

        # Add a single button
        my $button = $self->addButton(
            $table,
            'Refresh list',
            'Refresh the list of checked directions',
            undef,
            9, 12, 10, 11,
        );
        $button->signal_connect('clicked' => sub {

            # Refresh the simple list
            $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'checkedDirHash');
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub contentsTab {

        # Contents tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->contentsTab', @_);
        }

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('C_ontents', $self->notebook);

        # Add tabs to the inner notebook
        $self->contents1Tab($innerNotebook);
        $self->contents2Tab($innerNotebook);
        $self->contents3Tab($innerNotebook);

        return 1;
    }

    sub contents1Tab {

        # Contents1 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->contents1Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _1', $innerNotebook);

        # Temporary contents
        $self->addLabel($table, '<b>Temporary contents</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>List of objects which are currently in a non-model room</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            '#', 'text',
            'Object name', 'text',
            'Object category', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 270);     # Fixed height

        # Initialise the simple list
        $self->contents1Tab_refreshList($slWidget, scalar (@columnList / 2));

        # Add editing buttons
        my $button = $self->addButton(
            $table,
            'View',
            'View the selected temporary object',
            undef,
            1, 3, 10, 11,
        );
        $button->signal_connect('clicked' => sub {

            my (
                $num, $obj,
                @objList,
            );

            ($num) = $self->getSimpleListData($slWidget, 0);
            if (defined $num) {

                @objList = $self->getEditHash_listIV('tempObjList');
                $obj = $objList[$num - 1];
                if (defined $obj) {

                    # Open an 'edit' window for the exit object
                    $self->createFreeWin(
                        'Games::Axmud::EditWin::ModelObj::' . ucfirst($obj->category),
                        $self,
                        $self->session,
                        'Edit temporary ' . $obj->category . ' object',
                        $obj,
                        # Not temporary (at least not in the sense that 'edit' windows understand
                        #   an object to be temporary)
                        FALSE,
                    );
                }
            }

            # Refresh the simple list
            $self->contents1Tab_refreshList($slWidget, scalar (@columnList / 2));
        });

        my $button2 = $self->addButton(
            $table,
            'Reset',
            'Reset the list of objects',
            undef,
            8, 10, 10, 11,
        );
        $button2->signal_connect('clicked' => sub {

            # Remove the IV from $self->editHash, so that the IV in $self->editObj takes over
            $self->ivDelete('editHash', 'tempObjList');

            # Refresh the simple list
            $self->contents1Tab_refreshList($slWidget, scalar (@columnList / 2));
        });

        my $button3 = $self->addButton(
            $table,
            'Clear',
            'Empty the list of objects',
            undef,
            10, 12, 10, 11,
        );
        $button3->signal_connect('clicked' => sub {

            # Empty the list
            $self->ivAdd('editHash', 'tempObjList', []);

            # Refresh the simple list
            $self->contents1Tab_refreshList($slWidget, scalar (@columnList / 2));
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub contents1Tab_refreshList {

        # Called by $self->contents1Tab to refresh the GA::Obj::Simple::List
        #
        # Expected arguments
        #   $slWidget   - The GA::Obj::Simple::List
        #   $columns    - The number of columns in the list
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $slWidget, $columns, $check) = @_;

        # Local variables
        my (
            $count,
            @objList, @dataList,
        );

        # Check for improper arguments
        if (! defined $slWidget || ! defined $columns || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->contents1Tab_refreshList',
                @_,
            );
        }

        # Import the IV
        @objList = $self->getEditHash_listIV('tempObjList');

        # Compile the simple list data
        $count = 0;
        foreach my $obj (@objList) {

            $count++;

            push (@dataList,
                $count,
                $obj->name,
                $obj->category,
            );
        }

        # Reset the simple list
        $self->resetListData($slWidget, [@dataList], $columns);

        return 1;
    }

    sub contents2Tab {

        # Contents2 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->contents2Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _2', $innerNotebook);

        # Hidden objects
        $self->addLabel($table, '<b>Hidden objects</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of world model objects which are in the room, but not visible</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Object #', 'text',
            'Command to obtain it', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 200);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'hiddenObjHash');

        # Add entry boxes
        $self->addLabel($table, 'Object #',
            1, 3, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'int', 1, undef,
            3, 6, 8, 9);

        $self->addLabel($table, 'Command to obtain it (optional)',
            1, 3, 9, 10);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'string', undef, undef,
            3, 12, 9, 10);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'hiddenObjHash',
            10,
            $entry, $entry2,
        );
        $button->signal_connect('clicked' => sub {

            my ($number, $cmd);

            $number = $entry->get_text();
            $cmd = $entry2->get_text();

            if ($self->checkEntryIcon($entry, $entry2)) {

                # Check the model object exists
                if (! $self->session->worldModelObj->ivExists('modelHash', $number)) {

                    $self->showMsgDialogue(
                        'Add hidden object',
                        'error',
                        'The object #' . $number . ' does not exist in the world model',
                        'ok',
                    );

                } else {

                    # Add a new key-value pair
                    $self->modifyEditHash_hashIV('hiddenObjHash', $number, $cmd);

                    # Reset the entry boxes
                    $self->resetEntryBoxes($entry, $entry2);
                }

                # In either case, refresh the list
                $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'hiddenObjHash');
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub contents3Tab {

        # Contents3 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Optional arguments
        #   $tabTitle       - When inherited by GA::EditWin::Painter, the tab's page number (a
        #                       string)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $tabTitle, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->contents3Tab', @_);
        }
        # Tab setup
        if (! $tabTitle) {

            $tabTitle = 'Page _3';  # This function wasn't called by GA::EditWin::Painter
        }

        my ($vBox, $table) = $self->addTab($tabTitle, $innerNotebook);

        # Search responses
        $self->addLabel($table, '<b>Search responses</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of non-interacting things in the room description which can be'
            . ' searched/examined</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Search term', 'text',
            'Result of search/examine', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 200);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'searchHash');

        # Add entry boxes
        $self->addLabel($table, 'Thing\'s name',
            1, 3, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            3, 12, 8, 9);

        $self->addLabel($table, 'Response (optional)',
            1, 3, 9, 10);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'string', undef, undef,
            3, 12, 9, 10);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'searchHash',
            10,
            $entry, $entry2,
        );
        $button->signal_connect('clicked' => sub {

            my ($string, $response);

            $string = $entry->get_text();
            $response = $entry2->get_text();

            if ($self->checkEntryIcon($entry, $entry2)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('searchHash', $string, $response);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'searchHash');
                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub protocolsTab {

        # Protocols tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->protocolsTab', @_);
        }

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('Pr_otocols', $self->notebook);

        # Add tabs to the inner notebook
        $self->protocols1Tab($innerNotebook);
        $self->protocols2Tab($innerNotebook);

        return 1;
    }

    sub protocols1Tab {

        # Protocols1 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Optional arguments
        #   $tabTitle       - When inherited by GA::EditWin::Painter, the tab's page number (a
        #                       string)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $tabTitle, $check) = @_;

        # Local variables
        my (@columnList, @comboList);

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->protocols1Tab', @_);
        }

        # Tab setup
        if (! $tabTitle) {

            $tabTitle = 'Page _1';  # This function wasn't called by GA::EditWin::Painter
        }

        my ($vBox, $table) = $self->addTab($tabTitle, $innerNotebook);

        # Room data
        $self->addLabel($table, '<b>Room data</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>Data for this room supplied by various MUD protocols</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Name', 'text',
            'Value', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 300);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'protocolRoomHash');

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub protocols2Tab {

        # Protocols2 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Optional arguments
        #   $tabTitle       - When inherited by GA::EditWin::Painter, the tab's page number (a
        #                       string)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $tabTitle, $check) = @_;

        # Local variables
        my (@columnList, @comboList);

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->protocols2Tab', @_);
        }

        # Tab setup
        if (! $tabTitle) {

            $tabTitle = 'Page _2';  # This function wasn't called by GA::EditWin::Painter
        }

        my ($vBox, $table) = $self->addTab($tabTitle, $innerNotebook);

        # Exit data
        $self->addLabel($table, '<b>Exit data</b>',
            0, 12, 0, 1);
        $self->addLabel(
            $table,
            '<i>Data for this room\'s exits supplied by various MUD protocols</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Direction', 'text',
            'Destination vnum', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 300);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'protocolExitHash');

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::ModelObj::Weapon;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::EditWin::Generic::ModelObj Games::Axmud::Generic::EditWin
        Games::Axmud::Generic::ConfigWin Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win
        Games::Axmud

    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::ModelObj::Weapon')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook, depending on which category of model object is
        #   being edited
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        # Group 1
        $self->familyTab();
        $self->privateDataTab(       # Inherited from GA::Generic::EditWin
            'privateHash',
            '_Private',
            'Private properties hash',
            'Properties created by your own plugins and scripts, but stored in this type of object',
        );
        $self->notesTab();
        # Group 2
        $self->wordsTab();
        # Group 4
        $self->physicalTab();
        $self->exclusiveTab();
        # Group 5
        #   (no group 5 IVs for weapons)

        return 1;
    }

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    # Group 5

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::ModelObj::Armour;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::EditWin::Generic::ModelObj Games::Axmud::Generic::EditWin
        Games::Axmud::Generic::ConfigWin Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win
        Games::Axmud

    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::ModelObj::Armour')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook, depending on which category of model object is
        #   being edited
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        # Group 1
        $self->familyTab();
        $self->privateDataTab(       # Inherited from GA::Generic::EditWin
            'privateHash',
            '_Private',
            'Private properties hash',
            'Properties created by your own plugins and scripts, but stored in this type of object',
        );
        $self->notesTab();
        # Group 2
        $self->wordsTab();
        # Group 4
        $self->physicalTab();
        $self->exclusiveTab();
        # Group 5
        #   (no group 5 IVs for armours)

        return 1;
    }

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    # Group 5

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::ModelObj::Garment;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::EditWin::Generic::ModelObj Games::Axmud::Generic::EditWin
        Games::Axmud::Generic::ConfigWin Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win
        Games::Axmud

    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::ModelObj::Garment')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook, depending on which category of model object is
        #   being edited
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        # Group 1
        $self->familyTab();
        $self->privateDataTab(       # Inherited from GA::Generic::EditWin
            'privateHash',
            '_Private',
            'Private properties hash',
            'Properties created by your own plugins and scripts, but stored in this type of object',
        );
        $self->notesTab();
        # Group 2
        $self->wordsTab();
        # Group 4
        $self->physicalTab();
        $self->exclusiveTab();
        # Group 5
        #   (no group 5 IVs for garments)

        return 1;
    }

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    # Group 5

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::ModelObj::Char;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::EditWin::Generic::ModelObj Games::Axmud::Generic::EditWin
        Games::Axmud::Generic::ConfigWin Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win
        Games::Axmud

    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::ModelObj::Char')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook, depending on which category of model object is
        #   being edited
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        # Group 1
        $self->familyTab();
        $self->privateDataTab(       # Inherited from GA::Generic::EditWin
            'privateHash',
            '_Private',
            'Private properties hash',
            'Properties created by your own plugins and scripts, but stored in this type of object',
        );
        $self->notesTab();
        # Group 2
        $self->wordsTab();
        # Group 3
        $self->attacksTab();
        # Group 5
        $self->characterTab();

        return 1;
    }

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    # Group 5

    sub characterTab {

        # Creature tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->characterTab', @_);
        }

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('C_haracter', $self->notebook);

        # Add tabs to the inner notebook (uses some of the same tabs as sentients)
        $self->character1Tab($innerNotebook);
        $self->sentient2Tab($innerNotebook);

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::ModelObj::Minion;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::EditWin::Generic::ModelObj Games::Axmud::Generic::EditWin
        Games::Axmud::Generic::ConfigWin Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win
        Games::Axmud

    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::ModelObj::Minion')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook, depending on which category of model object is
        #   being edited
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        # Group 1
        $self->familyTab();
        $self->privateDataTab(       # Inherited from GA::Generic::EditWin
            'privateHash',
            '_Private',
            'Private properties hash',
            'Properties created by your own plugins and scripts, but stored in this type of object',
        );
        $self->notesTab();
        # Group 2
        $self->wordsTab();
        # Group 3
        $self->attacksTab();
        # Group 5
        $self->minionTab();

        return 1;
    }

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    # Group 5

    sub minionTab {

        # Minion tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->minionTab', @_);
        }

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('_Minion', $self->notebook);

        # Add tabs to the inner notebook (uses some of the same tabs as sentients and characters)
        $self->character1Tab($innerNotebook);
        $self->sentient2Tab($innerNotebook);

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::ModelObj::Sentient;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::EditWin::Generic::ModelObj Games::Axmud::Generic::EditWin
        Games::Axmud::Generic::ConfigWin Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win
        Games::Axmud

    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::ModelObj::Sentient')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook, depending on which category of model object is
        #   being edited
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        # Group 1
        $self->familyTab();
        $self->privateDataTab(       # Inherited from GA::Generic::EditWin
            'privateHash',
            '_Private',
            'Private properties hash',
            'Properties created by your own plugins and scripts, but stored in this type of object',
        );
        $self->notesTab();
        # Group 2
        $self->wordsTab();
        # Group 3
        $self->attacksTab();
        # Group 5
        $self->sentientTab();

        return 1;
    }

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    # Group 5

    sub sentientTab {

        # Sentient tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->sentientTab', @_);
        }

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('_Sentient', $self->notebook);

        # Add tabs to the inner notebook
        $self->sentient1Tab($innerNotebook);
        $self->sentient2Tab($innerNotebook);
        $self->sentient3Tab($innerNotebook);
        $self->sentient4Tab($innerNotebook);
        $self->sentient5Tab($innerNotebook);
        $self->sentient6Tab($innerNotebook);

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::ModelObj::Creature;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::EditWin::Generic::ModelObj Games::Axmud::Generic::EditWin
        Games::Axmud::Generic::ConfigWin Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win
        Games::Axmud

    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::ModelObj::Creature')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook, depending on which category of model object is
        #   being edited
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        # Group 1
        $self->familyTab();
        $self->privateDataTab(       # Inherited from GA::Generic::EditWin
            'privateHash',
            '_Private',
            'Private properties hash',
            'Properties created by your own plugins and scripts, but stored in this type of object',
        );
        $self->notesTab();
        # Group 2
        $self->wordsTab();
        # Group 3
        $self->attacksTab();
        # Group 5
        $self->creatureTab();

        return 1;
    }

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    # Group 5

    sub creatureTab {

        # Creature tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->creatureTab', @_);
        }

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('C_reature', $self->notebook);

        # Add tabs to the inner notebook
        $self->sentient1Tab($innerNotebook);
        $self->sentient2Tab($innerNotebook);
        $self->sentient3Tab($innerNotebook);
        $self->sentient4Tab($innerNotebook);
        $self->sentient5Tab($innerNotebook);

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::ModelObj::Portable;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::EditWin::Generic::ModelObj Games::Axmud::Generic::EditWin
        Games::Axmud::Generic::ConfigWin Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win
        Games::Axmud

    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::ModelObj::Portable')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook, depending on which category of model object is
        #   being edited
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        # Group 1
        $self->familyTab();
        $self->privateDataTab(       # Inherited from GA::Generic::EditWin
            'privateHash',
            '_Private',
            'Private properties hash',
            'Properties created by your own plugins and scripts, but stored in this type of object',
        );
        $self->notesTab();
        # Group 2
        $self->wordsTab();
        # Group 4
        $self->physicalTab();
        $self->exclusiveTab();
        # Group 5
        $self->portableTab();

        return 1;
    }

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    # Group 5

    sub portableTab {

        # Portable tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my @comboList;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->portableTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Po_rtable', $self->notebook);

        # Properties belonging only to portable world model objects
        $self->addLabel($table, '<b>Properties belonging only to portable model objects</b>',
            0, 6, 0, 1);
        $self->addLabel($table, 'Object\'s type (further classifies the type of portable object)',
            1, 8, 1, 2);
        @comboList = $self->session->currentDict->portableTypeList;
        $self->addComboBox($table, 'type', \@comboList, '',
            TRUE,               # No 'undef' value used
            8, 12, 1, 2);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::ModelObj::Decoration;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::EditWin::Generic::ModelObj Games::Axmud::Generic::EditWin
        Games::Axmud::Generic::ConfigWin Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win
        Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::ModelObj::Decoration')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook, depending on which category of model object is
        #   being edited
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        # Group 1
        $self->familyTab();
        $self->privateDataTab(       # Inherited from GA::Generic::EditWin
            'privateHash',
            '_Private',
            'Private properties hash',
            'Properties created by your own plugins and scripts, but stored in this type of object',
        );
        $self->notesTab();
        # Group 2
        $self->wordsTab();
        # Group 4
        $self->physicalTab();
        $self->exclusiveTab();
        # Group 5
        $self->decorationTab();

        return 1;
    }

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    # Group 5

    sub decorationTab {

        # Decoration tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my @comboList;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->decorationTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('_Decoration', $self->notebook);

        # Properties belonging only to decoration world model objects
        $self->addLabel($table, '<b>Properties belonging only to decoration model objects</b>',
            0, 6, 0, 1);
        $self->addLabel($table, 'Object\'s type (further classifies the type of decoration object)',
            1, 8, 1, 2);
        @comboList = $self->session->currentDict->decorationTypeList;
        $self->addComboBox($table, 'type', \@comboList, '',
            TRUE,               # No 'undef' value used
            8, 12, 1, 2);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::ModelObj::Custom;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::EditWin::Generic::ModelObj Games::Axmud::Generic::EditWin
        Games::Axmud::Generic::ConfigWin Games::Axmud::Generic::FreeWin
        Games::Axmud::Generic::Win Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::ModelObj::Custom')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook, depending on which category of model object is
        #   being edited
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        # Group 1
        $self->familyTab();
        $self->privateDataTab(       # Inherited from GA::Generic::EditWin
            'privateHash',
            '_Private',
            'Private properties hash',
            'Properties created by your own plugins and scripts, but stored in this type of object',
        );
        $self->notesTab();
        # Group 2
        $self->wordsTab();
        # Group 4
        $self->physicalTab();
        $self->exclusiveTab();
        # Group 5
        #   (no group 5 IVs for custom model objects)

        return 1;
    }

#   sub saveChanges {}          # Inherited from GA::Generic::ConfigWin

    # Group 5

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::Painter;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::EditWin::ModelObj::Room Games::Axmud::EditWin::Generic::ModelObj
        Games::Axmud::Generic::EditWin Games::Axmud::Generic::ConfigWin
        Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::ModelObj::Room')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

    sub setupNotebook {

        # Called by $self->enable
        # Creates the first tab for the notebook. The remaining tabs are created by
        #   $self->expandNotebook
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setupNotebook', @_);
        }

#       # Tab setup
#       my ($vBox, $table) = $self->addTab('_Room', $self->notebook);

        # Set up the rest of the first tab (all of it, in this case)
        $self->roomTab();

        # Set up the remaining tabs
        $self->expandNotebook();

#       # Tab complete
#       $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        $self->contentsTab();

        return 1;
    }

    sub saveChanges {

        # Called by $self->buttonOK and $self->buttonSave (usually for 'edit' windows only, not
        #   'pref' windows)
        # Saves any changes made to data stored by the edit object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->saveChanges', @_);
        }

        if ($self->editHash) {

            # Store the changes the user has made
            foreach my $key ($self->ivKeys('editHash')) {

                $self->editObj->{$key} = $self->ivShow('editHash', $key);
            }

            # The changes can now be cleared
            $self->ivEmpty('editHash');

            # Mark the object's corresponding file object as needing to be saved, if it exists
            if ($self->editObj->_parentFile) {

                $self->editObj->doModify('saveChanges');
            }

            # Update the current session's object viewer window, if it is open
            if ($self->session->viewerWin) {

                $self->session->viewerWin->updateNotebook();
            }

            # Redraw menu bars/toolbars in all automapper windows using this world model
            $self->session->worldModelObj->updateMapMenuToolbars();
        }

        return 1;
    }

    # Notebook tabs

    sub roomTab {

        # Room tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->roomTab', @_);
        }

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('_Room', $self->notebook);

        # Add tabs to the inner notebook (mostly inherited from GA::EditWin::ModelObj::Room, so we
        #   pass an extra argument to show the tab which page number to display)
        $self->ownRoom1Tab($innerNotebook);              # (not inherited)
        $self->room2Tab($innerNotebook, 'Page _2');     # ->roomFlagHash
        $self->room4Tab($innerNotebook, 'Page _3');     # ->descripHash
        $self->ownRoom2Tab($innerNotebook);              # (not inherited)
        $self->room7Tab($innerNotebook, 'Page _5');     # ->exclusiveFlag, ->exclusiveHash

        return 1;
    }

    sub ownRoom1Tab {

        # OwnRoom1Tab tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (@guildList, @comboList);

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->ownRoom1Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _1', $innerNotebook);

        # Wilderness mode
        $self->addLabel($table, '<b>Wilderness mode</b>',
            0, 12, 0, 1);

        my ($group, $radioButton) = $self->addRadioButton(
            $table, undef, '\'normal\'', 'wildMode',
            'normal',          # IV set to this value when toggled
            TRUE,       # Sensitive widget
            1, 4, 1, 2);
        $self->addLabel(
            $table,
            "The automapper expects an exit to exist in the world model for every exit in this"
            . " room\n(if an exit is visible in the automapper window, it exists in the world"
            . " model)",
            4, 12, 1, 2);
        $self->addLabel(
            $table,
            "<i>In \'update\' mode, the automapper window draws new exits</i>",
            4, 12, 2, 3);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'wild\'', 'wildMode', 'wild', TRUE,
            1, 4, 3, 4);
        $self->addLabel(
            $table,
            "The automapper assumes the world doesn\'t specify any exits for this room and that"
            . " the\ncharacter can move from this room to any adjacent room using any primary"
            . " direction",
            4, 12, 3, 4);
        $self->addLabel(
            $table,
            '<i>In \'update\' mode, the automapper window doesn\'t draw new exits</i>',
            4, 12, 4, 5);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'border\'', 'wildMode', 'border', TRUE,
            1, 4, 5, 6);
        $self->addLabel(
            $table,
            "The automapper assumes the world doesn\'t specify any exits for this room and that"
            . " the\ncharacter can move to any adjacent \'wild\' or \'border\' room, but movement"
            . " to a \'normal\'\nroom requires an exit in the world model",
            4, 12, 5, 6);
        $self->addLabel(
            $table,
            "<i>In \'update\' mode, the automapper window only draws exits between a \'border\'"
            . " room and\nan adjacent \'normal\' room</i>",
            4, 12, 6, 7);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub ownRoom2Tab {

        # OwnRoom2Tab tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (@guildList, @comboList);

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->ownRoom2Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _4', $innerNotebook);

        # Room titles
        $self->addLabel($table, '<b>Room titles</b>',
            0, 12, 0, 2);
        $self->addLabel($table, '<i>List of all known titles for this room</i>',
            1, 12, 2, 4);
        $self->addTextView($table, 'titleList', TRUE,
            1, 12, 4, 6);

        # Room guild
        $self->addLabel($table, '<b>Room guild</b>',
            0, 12, 6, 8);

        # Get a sorted list of existing guild profiles
        foreach my $profile ($self->session->ivValues('profHash')) {

            if ($profile->category eq 'guild') {

                push (@guildList, $profile->name);
            }
        }

        @comboList = sort {lc($a) cmp lc($b)} (@guildList);

        my $combo = $self->addComboBox($table, 'roomGuild', \@comboList, 'Choose a guild:',
            FALSE,              # 'undef' value allowed
            1, 6, 8, 10);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub contentsTab {

        # Contents tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->contentsTab', @_);
        }

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('_Contents', $self->notebook);

        # Add tabs to the inner notebook (inherited from GA::EditWin::ModelObj::Room, so we pass an
        #   extra argument to show the tab which page number to display)
        $self->contents3Tab($innerNotebook, 'Page _1');     # ->searchHash

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::Generic::Profile;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::Generic::EditWin Games::Axmud::Generic::ConfigWin
        Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

#   sub checkEditObj {}         # Inherited from GA::Generic::ConfigWin

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

    sub setupNotebook {

        # Called by $self->winEnable
        # Creates the first tab for the notebook. The remaining tabs are created by
        #   $self->expandNotebook, once the user has specified the profile's name
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->setupNotebook', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('_Name', $self->notebook);

        # Profile name
        $self->addLabel($table, '<b>Profile name</b>',
            0, 6, 0, 1);

        # Explanatory label - contents to be set to 'Choose a name and press enter', in a moment
        my $label = $self->addLabel($table, '',
            1, 6, 2, 3);

        # Entry box for the profile name
        my $entry;

        if ($self->tempFlag) {

            # $self->editObj is a temporary profile, so the entry box must be sensitive
            $entry = $self->addEntry($table, undef, TRUE,
                1, 6, 1, 2, 16, 16);

            $entry->signal_connect (activate => sub {

                my $text = $entry->get_text();

                # Check that the name is acceptable
                if (
                    (
                        $self->editObj->category eq 'world'
                        && $axmud::CLIENT->ivExists('worldProfHash', $text)
                    ) || (
                        $self->editObj->category ne 'world'
                        && $self->session->ivExists('profHash', $text)
                    )
                ) {
                    $label->set_markup('<i>Profile named \'' . $text . '\' already exists</i>');
                    $self->winShowAll($self->_objClass . '->setupNotebook');

                } elsif (! $axmud::CLIENT->nameCheck($text, 16)) {

                    $label->set_markup('<i>Max 16 chars: A-Z a-z _ 0-9</i>');
                    $self->winShowAll($self->_objClass . '->setupNotebook');

                } else {

                    # Replace the explanatory label
                    $label->set_markup('<i>You may now edit the profile</i>');
                    # Desensitise the entry box - once the profile name is entered, it's final
                    $entry->set_sensitive(FALSE);
                    # Make sure the OK/reset/save buttons are now sensitised
                    $self->okButton->set_sensitive(TRUE);
                    $self->resetButton->set_sensitive(TRUE);
                    $self->saveButton->set_sensitive(TRUE);

                    # Store the name until all IVs are copied to the profile
                    $self->ivAdd('editHash', 'name', $text);

                    # Set up the remaining tabs
                    $self->expandNotebook();

                    # Render the changes
                    $self->winShowAll($self->_objClass . '->setupNotebook');
                }
            });

            $entry->signal_connect('changed' => sub {

                my ($value, $result);

                $value = $entry->get_text();
                # Check whether $text is a valid value, or not
                if ($axmud::CLIENT->nameCheck($value, 16)) {
                    $entry->set_icon_from_stock('secondary', 'gtk-yes');
                } else {
                    $entry->set_icon_from_stock('secondary', 'gtk-no');
                }
            });

            $label->set_markup('<i>Choose a name and press \'enter\'</i>');
            # (Entry is initially empty)
            $entry->set_icon_from_stock('secondary', 'gtk-no');

        } else {

            # $self->editObj is an existing profile, so the entry box must be insensitive
            $entry = $self->addEntry($table, 'name', FALSE,
                1, 6, 1, 2, 16, 16);

            $label->set_markup('<i>(The profile name can\'t be changed)</i>');

            # Set up the remaining tabs
            $self->expandNotebook();
        }

        # Set up the rest of the tab
        $self->nameTab($table);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

#   sub expandNotebook {}       # Inherited from GA::Generic::ConfigWin

    sub saveChanges {

        # Called by $self->buttonOK and $self->buttonSave
        # Saves the changes made to the profile. If it was a temporary profile, create a real
        #   profile, and save the changes to the latter
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->saveChanges', @_);
        }

        if ($self->editHash) {

            # For a temporary profile...
            if ($self->tempFlag) {

                my ($profName, $result, $profObj, $number);

                $profName = $self->ivShow('editHash', 'name');

                # The user made changes, so we can now create a permanent profile and give it IVs
                #   specified in $self->editHash
                if ($self->editObj->category eq 'world') {

                    $result = $self->session->pseudoCmd(
                        'addworld ' . $profName,
                        $self->pseudoCmdMode,
                    );

                } elsif ($self->editObj->category eq 'guild') {

                    $result = $self->session->pseudoCmd(
                        'addguild ' . $profName,
                        $self->pseudoCmdMode,
                    );

                } elsif ($self->editObj->category eq 'race') {

                    $result = $self->session->pseudoCmd(
                        'addrace ' . $profName,
                        $self->pseudoCmdMode,
                    );

                } elsif ($self->editObj->category eq 'char') {

                    $result = $self->session->pseudoCmd(
                        'addchar ' . $profName,
                        $self->pseudoCmdMode,
                    );

                } else {

                    $result = $self->session->pseudoCmd(
                        'addcustomprofile ' . $profName . ' ' . $self->editObj->category,
                        $self->pseudoCmdMode,
                    );
                }

                if ($result) {

                    # Get the blessed reference of the new profile
                    if ($self->editObj->category eq 'world') {
                        $profObj = $axmud::CLIENT->ivShow('worldProfHash', $profName);
                    } else {
                        $profObj = $self->session->ivShow('profHash', $profName);
                    }

                    # Store the changes the user has made (ignore the 'name' IV, which has aleady
                    #   been set)
                    $self->ivDelete('editHash', 'name');

                    foreach my $key ($self->ivKeys('editHash')) {

                        $profObj->{$key} = $self->ivShow('editHash', $key);
                    }

                    # If the 'save' button was clicked, the window won't be closed. Before
                    #   re-drawing all those tabs, set the object being edited to the one we just
                    #   created
                    $self->ivPoke('editObj', $profObj);
                    $self->ivPoke('tempFlag', FALSE);
                }

            # For an existing profile...
            } else {

                # Store the changes the user has made
                foreach my $key ($self->ivKeys('editHash')) {

                    # Special case: for GA::Profile::World->worldCharSet IV, must use a client
                    #   command
                    if ($self->editObj->category eq 'world' && $key eq 'worldCharSet') {

                        $self->session->pseudoCmd(
                            'setcharset -w ' . $self->ivShow('editHash', $key),
                            $self->pseudoCmdMode,
                        );

                    } else {

                        $self->editObj->{$key} = $self->ivShow('editHash', $key);
                    }
                }
            }

            # The changes can now be cleared
            $self->ivEmpty('editHash');

            # Mark the object's corresponding file object as needing to be saved, if it exists
            if ($self->editObj->_parentFile) {

                $self->editObj->doModify('saveChanges');
            }

            # Update the current session's object viewer window, if it is open
            if ($self->session->viewerWin) {

                $self->session->viewerWin->updateNotebook();
            }
        }

        return 1;
    }

    # Notebook tabs used by more than one profile 'edit' window

    sub nameTab {

        # Name tab - called by $self->setupNotebook
        #
        # Expected arguments
        #   $table  -  The Gtk2::Table which has already been created for this tab
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $table, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->nameTab', @_);
        }

#       # Tab setup (already created by the calling function)
#       my ($vBox, $table) = $self->addTab('_Name', $self->notebook);

        # Profile category
        $self->addLabel($table, '<b>Profile category</b>',
            0, 6, 3, 4);
        $self->addEntry($table, 'category', FALSE,
            1, 6, 4, 5, 16, 16);

        # Parent world profile
        $self->addLabel($table, '<b>Parent world profile</b>',
            7, 13, 0, 1);
        $self->addEntry($table, 'parentWorld', FALSE,
            8, 13, 1, 2);

        # Current profile
        $self->addLabel($table, '<b>Current profile</b>',
            7, 13, 3, 4);

        my $checkbutton = Gtk2::CheckButton->new();
        if (
            $self->session->ivExists('currentProfHash', $self->editObj->category)
            && $self->session->ivShow('currentProfHash', $self->editObj->category) eq $self->editObj
        ) {
            $checkbutton->set_active(TRUE);
        }
        $checkbutton->set_sensitive(FALSE);
        $table->attach_defaults($checkbutton, 7, 13, 4, 5);

#       # Tab complete (handled by the calling function)
#       $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub commandsTab {

        # Commands tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my @list;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->commandsTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('_Commands', $self->notebook);

        # List of profile-specific commands
        $self->addLabel($table, '<b>List of profile-specific commands</b>',
            0, 12, 0, 2);
        $self->addLabel($table, '<i>(Character-independent)</i>',
            1, 12, 2, 4);
        $self->addTextView($table, 'cmdList', TRUE,
            1, 12, 4, 6,
            undef, undef, undef, undef,
            -1, 300);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub fightTab {

        # Fight tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->fightTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('_Fight', $self->notebook);

        # List of profile-specific fight patterns
        $self->addLabel($table, '<b>List of profile-specific fight patterns</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>Patterns seen during a fight</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Pattern', 'text',
            'Substring #', 'int',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 270);

        # Initialise the list
        $self->refreshList_listIV($slWidget, scalar (@columnList / 2), 'fightMsgList');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Pattern:',
            1, 2, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'regex', 1, undef,
            2, 8, 8, 9);

        $self->addLabel($table, 'Target substring #:',
            8, 9, 8, 9);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'int', 0, undef,
            9, 12, 8, 9, 4, 4);

        # Add buttons at various positions
        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_listIV(
            $table,
            $slWidget,
            'fightMsgList',
            9, 2,
            $entry, $entry2,
        );
        $button->signal_connect('clicked' => sub {

            my ($pattern, $grpNum);

            $pattern = $entry->get_text();
            $grpNum = $entry2->get_text();

            if ($self->checkEntryIcon($entry, $entry2)) {

                # Add new values to (the end of) the list IV
                $self->addEditHash_listIV('fightMsgList', undef, FALSE, $pattern, $grpNum);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_listIV($slWidget, scalar (@columnList / 2), 'fightMsgList');
                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub interactionTab {

        # Interaction tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->interactionTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Int_eraction', $self->notebook);

        # List of profile-specific interaction patterns
        $self->addLabel($table, '<b>List of profile-specific interaction patterns</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>Patterns seen during an interaction</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Pattern', 'text',
            'Substring #', 'int',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 270);       # Fixed height

        # Initialise the list
        $self->refreshList_listIV($slWidget, scalar (@columnList / 2), 'interactionMsgList');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Pattern:',
            1, 2, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'regex', 1, undef,
            2, 8, 8, 9);

        $self->addLabel($table, 'Target substring #:',
            8, 9, 8, 9);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'int', 0, undef,
            9, 12, 8, 9, 4, 4);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_listIV(
            $table,
            $slWidget,
            'interactionMsgList',
            9, 2,
            $entry, $entry2,
        );
        $button->signal_connect('clicked' => sub {

            my ($pattern, $grpNum);

            $pattern = $entry->get_text();
            $grpNum = $entry2->get_text();

            if ($self->checkEntryIcon($entry, $entry2)) {

                # Add new values to (the end of) the list IV
                $self->addEditHash_listIV(
                    'interactionMsgList',
                    undef,
                    FALSE,
                    $pattern,
                    $grpNum,
                );

                # Refresh the simple list and reset entry boxes
                $self->refreshList_listIV(
                    $slWidget,
                    scalar (@columnList / 2),
                    'interactionMsgList',
                );

                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub initialTab {

        # Initial tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->initialTab', @_);
        }

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('_Initial', $self->notebook);

        # Add tabs to the inner notebook
        $self->initial1Tab($innerNotebook);
        $self->initial2Tab($innerNotebook);
        $self->initial3Tab($innerNotebook);

        return 1;
    }

    sub initial1Tab {

        # Initial1 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->initial1Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _1', $innerNotebook);

        # Initial tasks
        $self->addLabel($table, '<b>Initial tasks</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of initial tasks that start when this profile becomes current</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Name' => 'text',       # ->uniqueName
            'Type' => 'text',       # ->category
            'Jealous' => 'bool',    # ->jealousyFlag
            'Stage' => 'text',      # ->stage
            'Status' => 'text',     # ->status
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 230);     # Fixed height

        # Initialise the list
        $self->initial1Tab_refreshList($slWidget, (scalar @columnList / 2));

        # Add editing buttons and combos
        my $combo = $self->addComboBox($table, undef, [], '',
            TRUE,           # No 'undef' value used
            1, 4, 10, 11);
        $self->initial1Tab_refreshCombo($combo);

        my $button = $self->addButton(
            $table,
            'Add initial task...',
            'Add the selected task as an initial task',
            undef,
            4, 6, 10, 11,
            TRUE,           # Irreversible
        );
        $button->signal_connect('clicked' => sub {

            my ($taskName, $childWinObj);

            $taskName = $combo->get_active_text();
            if (defined $taskName) {

                # Open up a task start 'pref' window to specify task settings
                $childWinObj = $self->createFreeWin(
                    'Games::Axmud::PrefWin::TaskStart',
                    $self,
                    $self->session,
                    '\'' . $taskName . '\' task preferences',
                    undef,                                  # No ->editObj
                    FALSE,                                  # Not temporary
                    # Config
                    'type'  => 'profile_initial',           # Profile initial tasklist
                    'task_name' => $taskName,
                    'prof_name' => $self->editObj->name,    # Parent profile
                );

                if ($childWinObj) {

                    # When the 'edit' window closes, update widgets and/or IVs
                    $self->add_childDestroy(
                        $childWinObj,
                        'initial1Tab_refreshList',
                        [$slWidget, (scalar @columnList / 2)],
                    );

                    $self->add_childDestroy(
                        $childWinObj,
                        'initial1Tab_refreshCombo',
                        [$combo],
                    );
                }
            }
        });

        my $button2 = $self->addButton(
            $table,
            'Edit...',
            'Edit the selected task',
            undef,
            6, 8, 10, 11,
        );
        $button2->signal_connect('clicked' => sub {

            my ($taskName, $taskObj);

            ($taskName) = $self->getSimpleListData($slWidget, 0);
            if (defined $taskName) {

                $taskObj = $self->editObj->ivShow('initTaskHash', $taskName);

                $self->createFreeWin(
                    'Games::Axmud::EditWin::Task',
                    $self,
                    $self->session,
                    'Edit ' . $taskObj->prettyName . ' task',
                    $taskObj,
                    FALSE,                          # Not temporary
                    # Config
                    'edit_flag' => TRUE,            # Some IVs for initial tasks are editable
                );
            }
        });

        my $button3 = $self->addButton(
            $table,
            'Move up',
            'Move the selected initial task up the list',
            undef,
            8, 10, 10, 11,
            TRUE,               # Irreversible
        );
        $button3->signal_connect('clicked' => sub {

            my (
                $taskName, $match, $count,
                @orderList, @modList,
            );

            ($taskName) = $self->getSimpleListData($slWidget, 0);
            if (defined $taskName) {

                # Import the existing order list
                @orderList = $self->editObj->initTaskOrderList;

                # Remove the selected task from the list
                $count = -1;
                foreach my $item (@orderList) {

                    $count++;

                    if ($item ne $taskName || $count == 0) {

                        push (@modList, $item);

                    } else {

                        $match = $count;
                    }
                }

                if (defined $match) {

                    # A task was removed. Replace it one place higher
                    $match--;
                    splice (@modList, $match, 0, $taskName);
                }

                # Update the list
                $self->editObj->ivPoke('initTaskOrderList', @modList);

                # Refresh the simple list and select the same line
                $self->initial1Tab_refreshList($slWidget, (scalar @columnList / 2));
                if (defined $match) {

                    $slWidget->select($match);
                }

                # Also refresh the combo
                $self->initial1Tab_refreshCombo($combo);
            }
        });

        my $button4 = $self->addButton(
            $table,
            'Move down',
            'Move the selected initial task down the list',
            undef,
            10, 12, 10, 11,
            TRUE,               # Irreversible
        );
        $button4->signal_connect('clicked' => sub {

            my (
                $taskName, $match, $count,
                @orderList, @modList,
            );

            ($taskName) = $self->getSimpleListData($slWidget, 0);
            if (defined $taskName) {

                # Import the existing order list
                @orderList = $self->editObj->initTaskOrderList;

                # Remove the selected script from the list
                $count = -1;
                foreach my $item (@orderList) {

                    $count++;

                    if ($item ne $taskName || $count == (scalar (@orderList) - 1)) {

                        push (@modList, $item);

                    } else {

                        $match = $count;
                    }
                }

                if (defined $match) {

                    # A task was removed. Replace it one place lower
                    $match++;
                    splice (@modList, $match, 0, $taskName);
                }

                # Update the list
                $self->editObj->ivPoke('initTaskOrderList', @modList);

                # Refresh the simple list and select the same line
                $self->initial1Tab_refreshList($slWidget, (scalar @columnList / 2));
                if (defined $match) {

                    $slWidget->select($match);
                }

                # Also refresh the combo
                $self->initial1Tab_refreshCombo($combo);
            }
        });

        my $button5 = $self->addButton(
            $table,
            'Delete',
            'Delete the selected task',
            undef,
            1, 3, 11, 12,
            TRUE,               # Irreversible
        );
        $button5->signal_connect('clicked' => sub {

            my ($taskName, $taskObj);

            ($taskName) = $self->getSimpleListData($slWidget, 0);
            if (defined $taskName) {

                $taskObj = $self->editObj->ivShow('initTaskHash', $taskName);

                $self->session->pseudoCmd(
                    'deleteinitialtask ' . $taskObj->uniqueName . ' ' . $taskObj->profName,
                    $self->pseudoCmdMode,
                );

                # Reset the simple list
                $self->initial1Tab_refreshList($slWidget, (scalar @columnList / 2));
                # Also refresh the combo
                $self->initial1Tab_refreshCombo($combo);
            }
        });

        my $button6 = $self->addButton(
            $table,
            'Delete all',
            'Delete all of this profile\'s initial tasks',
            undef,
            3, 5, 11, 12,
            TRUE,           # Irreversible
        );
        $button6->signal_connect('clicked' => sub {

            $self->session->pseudoCmd(
                'deleteinitialtask -a ' . $self->editObj->name,
                $self->pseudoCmdMode,
            );

            # Reset the simple list
            $self->initial1Tab_refreshList($slWidget, (scalar @columnList / 2));
            # Also refresh the combo
            $self->initial1Tab_refreshCombo($combo);
        });

        my $button7 = $self->addButton(
            $table,
            'Refresh list',
            'Refresh the list of initial tasks',
            undef,
            7, 10, 11, 12,
        );
        $button7->signal_connect('clicked' => sub {

            # Reset the simple list
            $self->initial1Tab_refreshList($slWidget, (scalar @columnList / 2));
            # Also refresh the combo, just in case the user has added a new type of task (very
            #   unlikely)
            $self->initial1Tab_refreshCombo($combo);
        });

        my $button8 = $self->addButton(
            $table,
            'Dump',
            'Display this profile\'s list of initial tasks in the \'main\' window',
            undef,
            10, 12, 11, 12,
        );
        $button8->signal_connect('clicked' => sub {

            $self->session->pseudoCmd(
                'listinitialtask ' . $self->editObj->name,
                $self->pseudoCmdMode,
            );

            # Reset the simple list
            $self->initial1Tab_refreshList($slWidget, (scalar @columnList / 2));
            # Also refresh the combo
            $self->initial1Tab_refreshCombo($combo);
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub initial1Tab_refreshList {

        # Called by $self->initial1Tab to refresh the GA::Obj::Simple::List
        #
        # Expected arguments
        #   $slWidget   - The GA::Obj::Simple::List
        #   $columns    - Number of columns in the simple list
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $slWidget, $columns, $check) = @_;

        # Local variables
        my @dataList;

        # Check for improper arguments
        if (! defined $slWidget || ! defined $columns || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->initial1Tab_refreshList',
                @_,
            );
        }

        # Compile the simple list data
        OUTER: foreach my $taskName ($self->editObj->initTaskOrderList) {

            my $taskObj = $self->editObj->ivShow('initTaskHash', $taskName);

            push (@dataList,
                $taskObj->uniqueName, $taskObj->category, $taskObj->jealousyFlag,
                $taskObj->stage, $taskObj->status,
            );
        }

        # Reset the simple list
        $self->resetListData($slWidget, [@dataList], $columns);

        return 1;
    }

    sub initial1Tab_refreshCombo {

        # Called by $self->initial1Tab to refresh the Gtk2::Combo
        #
        # Expected arguments
        #   $combo      - The combo box to refresh
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $combo, $check) = @_;

        # Local variables
        my (
            @comboList,
            %packageHash,
        );

        # Check for improper arguments
        if (! defined $combo || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->initial1Tab_refreshCombo',
                @_,
            );
        }

        # For each task package in GA::Client->taskPackageHash, create a temporary task so that we
        #   can access its ->storableFlag IV (only storable tasks can appear in the combo)
        %packageHash = $axmud::CLIENT->taskPackageHash;

        foreach my $name (sort {lc($a) cmp lc($b)} (keys %packageHash)) {

            my ($package, $tempObj);

            $package = $packageHash{$name};

            # Create the temporary task
            $tempObj = $package->new($self->session);
            if ($tempObj && $tempObj->storableFlag) {

                push (@comboList, $name);
            }
        }

        $self->resetComboBox($combo, @comboList);

        return 1;
    }

    sub initial2Tab {

        # Initial2 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (
            @columnList, @initList, @comboList,
            %comboHash,
        );

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->initial2Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _2', $innerNotebook);

        # Initial tasks
        $self->addLabel($table, '<b>Initial scripts</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of initial scripts that start when this profile becomes current</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Script' => 'text',
            'Run mode' => 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 13, 2, 8,
            -1, 200);     # Fixed height

        # Initialise the list
        $self->initial2Tab_refreshList($slWidget, (scalar @columnList / 2));

        # Add editing entry boxes/buttons
        $self->addLabel($table, 'Script name',
            1, 3, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            3, 7, 8, 9);

        $self->addLabel($table, 'Run mode',
            7, 9, 8, 9);

        @initList = (
            'no_task'       => 'Run without a task',
            'run_task'      => 'Run from within a task',
            'run_task_win'  => 'Run in \'forced window\' mode',
        );

        do {

            my ($value, $string, $descrip);

            $value = shift @initList;
            $string = shift @initList;
            $descrip = 'Mode \'' . $value . '\' - ' . $string;

            push (@comboList, $descrip);
            $comboHash{$descrip} = $value;

        } until (! @initList);

        my $combo = $self->addComboBox($table, undef, \@comboList, '',
            TRUE,           # No 'undef' value used
            9, 13, 8, 9);

        my $button = $self->addButton(
            $table,
            'Add',
            'Add this script to the list of initial scripts',
            undef,
            1, 3, 9, 10,
            TRUE,           # Irreversible
        );
        $button->signal_connect('clicked' => sub {

            my ($scriptName, $descrip, $mode, $switch);

            if ($self->checkEntryIcon($entry)) {

                $scriptName = $entry->get_text();
                $descrip = $combo->get_active_text();
                $mode = $comboHash{$descrip};

                if ($mode eq 'no_task') {
                    $switch = '-r';
                } elsif ($mode eq 'run_task') {
                    $switch = '-t';
                } else {
                    $switch = '-c';
                }

                $self->session->pseudoCmd(
                    'addinitialscript ' . $scriptName . ' ' . $self->editObj->name . ' ' . $switch,
                    $self->pseudoCmdMode,
                );

                # Reset the simple list and reset the entry box
                $self->initial2Tab_refreshList($slWidget, (scalar @columnList / 2));
                $self->resetEntryBoxes($entry);
            }
        });

        my $button2 = $self->addButton(
            $table,
            'Edit',
            'Open the selected script in a text editor',
            undef,
            3, 5, 9, 10,
            TRUE,           # Irreversible
        );
        $button2->signal_connect('clicked' => sub {

            my ($name) = $self->getSimpleListData($slWidget, 0);
            if (defined $name) {

                $self->session->pseudoCmd('editscript ' . $name, $self->pseudoCmdMode);

                # Reset the simple list and reset the entry box
                $self->initial2Tab_refreshList($slWidget, (scalar @columnList / 2));
                $self->resetEntryBoxes($entry);
            }
        });

        my $button3 = $self->addButton(
            $table,
            'Delete',
            'Delete the selected script',
            undef,
            9, 11, 9, 10,
            TRUE,           # Irreversible
        );
        $button3->signal_connect('clicked' => sub {

            my ($name) = $self->getSimpleListData($slWidget, 0);
            if (defined $name) {

                $self->session->pseudoCmd(
                    'deleteinitialscript ' . $name . ' ' . $self->editObj->name,
                    $self->pseudoCmdMode,
                );

                # Reset the simple list and reset the entry box
                $self->initial2Tab_refreshList($slWidget, (scalar @columnList / 2));
                $self->resetEntryBoxes($entry);
            }
        });

        my $button4 = $self->addButton($table,
            'Move up', 'Move the selected script up the list', undef,
            5, 7, 9, 10,
            TRUE);      # Irreversible
        $button4->signal_connect('clicked' => sub {

            my (
                $script, $match, $count,
                @orderList, @modList,
            );

            ($script) = $self->getSimpleListData($slWidget, 0);
            if (defined $script) {

                # Import the existing order list
                @orderList = $self->editObj->initScriptOrderList;

                # Remove the selected script from the list
                $count = -1;
                foreach my $item (@orderList) {

                    $count++;

                    if ($item ne $script || $count == 0) {

                        push (@modList, $item);

                    } else {

                        $match = $count;
                    }
                }

                if (defined $match) {

                    # A script was removed. Replace it one place higher
                    $match--;
                    splice (@modList, $match, 0, $script);
                }

                # Update the list
                $self->editObj->ivPoke('initScriptOrderList', @modList);

                # Refresh the simple list and select the same line
                $self->initial2Tab_refreshList($slWidget, scalar (@columnList / 2));
                if (defined $match) {

                    $slWidget->select($match);
                }
            }
        });

        my $button5 = $self->addButton($table,
            'Move down', 'Move the selected script down the list', undef,
            7, 9, 9, 10,
            TRUE);          # Irreversible
        $button5->signal_connect('clicked' => sub {

            my (
                $script, $match, $count,
                @orderList, @modList,
            );

            ($script) = $self->getSimpleListData($slWidget, 0);
            if (defined $script) {

                # Import the existing order list
                @orderList = $self->editObj->initScriptOrderList;

                # Remove the selected script from the list
                $count = -1;
                foreach my $item (@orderList) {

                    $count++;

                    if ($item ne $script || $count == (scalar (@orderList) - 1)) {

                        push (@modList, $item);

                    } else {

                        $match = $count;
                    }
                }

                if (defined $match) {

                    # A script was removed. Replace it one place lower
                    $match++;
                    splice (@modList, $match, 0, $script);
                }

                # Update the list
                $self->editObj->ivPoke('initScriptOrderList', @modList);

                # Refresh the simple list and select the same line
                $self->initial2Tab_refreshList($slWidget, scalar (@columnList / 2));
                if (defined $match) {

                    $slWidget->select($match);
                }
            }
        });

        my $button6 = $self->addButton(
            $table,
            'Refresh list',
            'Refresh the list of initial scripts',
            undef,
            11, 13, 9, 10);
        $button6->signal_connect('clicked' => sub {

            # Reset the simple list
            $self->initial2Tab_refreshList($slWidget, (scalar @columnList / 2));
        });

        my $button7 = $self->addButton(
            $table,
            'Dump',
            'Display this profile\'s list of initial tasks in the \'main\' window',
            undef,
            11, 13, 10, 11,
        );
        $button7->signal_connect('clicked' => sub {

            # Display initial scripts
            $self->session->pseudoCmd(
                'listinitialscript ' . $self->editObj->name,
                $self->pseudoCmdMode,
            );

            # Reset the simple list
            $self->initial2Tab_refreshList($slWidget, (scalar @columnList / 2));
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub initial2Tab_refreshList {

        # Called by $self->initial2Tab to refresh the GA::Obj::Simple::List
        #
        # Expected arguments
        #   $slWidget   - The GA::Obj::Simple::List
        #   $columns    - Number of columns in the simple list
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $slWidget, $columns, $check) = @_;

        # Local variables
        my @dataList;

        # Check for improper arguments
        if (! defined $slWidget || ! defined $columns || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->initial2Tab_refreshList',
                @_,
            );
        }

        # Compile the simple list data
        OUTER: foreach my $scriptName ($self->editObj->initScriptOrderList) {

            push (@dataList,
                $scriptName,
                $self->editObj->ivShow('initScriptHash', $scriptName),
            );
        }

        # Reset the simple list
        $self->resetListData($slWidget, [@dataList], $columns);

        return 1;
    }

    sub initial3Tab {

        # Initial3 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (@columnList, @comboList);

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->initial3Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _3', $innerNotebook);

        # Initial mission
        # (NB Labels stop at column 11; otherwise the 'undef' button takes up most of the row,
        #   rather than being positioned at the end, as it should be)
        $self->addLabel($table, '<b>Initial mission</b>',
            0, 11, 0, 1);
        $self->addLabel($table,
            '<i>The mission to start when a session begins with this as a current profile</i>',
            1, 11, 1, 2);
        $self->addLabel($table,
            '<i>(N.B. Current profiles are checked in priority order; only the first mission'
            . ' found is started)</i>',
            1, 11, 2, 3);
        $self->addEntryWithButton($table, 'initMission', TRUE,
            1, 12, 3, 4);

        # Initial command list
        $self->addLabel($table, '<b>Initial command list</b>',
            0, 11, 4, 5);
        $self->addLabel($table,
            '<i>List of commands executed when a session begins with this as a current'
            . ' profile</i>',
            1, 11, 5, 6);
        $self->addLabel($table,
            '<i>(N.B. Current profiles are checked in priority order; commands are sent from all'
            . ' profiles</i>',
            1, 11, 6, 7);
        $self->addLabel($table,
            '<i>but no duplicate commands are sent)</i>',
            2, 11, 7, 8);
        $self->addTextView($table, 'initCmdList', TRUE,
            1, 12, 8, 12,
            TRUE, TRUE, FALSE, FALSE,
            -1, 160);           # Fixed height

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub notesTab {

        # Notes tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->notesTab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('N_otes', $self->notebook);

        # Notes
        $self->addLabel($table, '<b>Notes</b>',
            0, 12, 0, 1);
        $self->addLabel(
            $table,
            '<i>Persistent notes for this profile (also used by the Notepad task)</i>',
            1, 12, 1, 2);

        my $textView = $self->addTextView($table, 'notepadList', TRUE,
            1, 12, 2, 11,
            TRUE, FALSE, FALSE, FALSE,     # Treat as a list, don't remove empty lines or whitespace
            -1, 300);                      # Fixed height
        my $button = $self->addButton($table,
            'Clear notes', 'Empty the persistent notes for this profile', undef,
            10, 12, 11, 12);
        $button->signal_connect('clicked' => sub {

            $textView->get_buffer->set_text('');
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    ##################
    # Accessors - set

    ##################
    # Accessors - get
}

{ package Games::Axmud::EditWin::Profile::World;

    use strict;
    use warnings;
    use diagnostics;

    use Glib qw(TRUE FALSE);

    our @ISA = qw(
        Games::Axmud::EditWin::Generic::Profile Games::Axmud::Generic::EditWin
        Games::Axmud::Generic::ConfigWin Games::Axmud::Generic::FreeWin Games::Axmud::Generic::Win
        Games::Axmud
    );

    ##################
    # Constructors

#   sub new {}                  # Inherited from GA::Generic::ConfigWin

    ##################
    # Methods

    # Standard window object functions

#   sub winSetup {}             # Inherited from GA::Generic::ConfigWin

#   sub winEnable {}            # Inherited from GA::Generic::ConfigWin

#   sub winDesengage {}         # Inherited from GA::Generic::FreeWin

#   sub winDestroy {}           # Inherited from GA::Generic::FreeWin

#   sub winShowAll {}           # Inherited from GA::Generic::Win

#   sub drawWidgets {}          # Inherited from GA::Generic::ConfigWin

#   sub redrawWidgets {}        # Inherited from GA::Generic::Win

    # ->signal_connects

    # Other functions

    sub checkEditObj {

        # Called by $self->winEnable
        # Checks that the object stored in $self->editObj is the correct class of object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments or if the check fails
        #   1 if the check succeeds

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->checkEditObj', @_);
        }

        if ($self->editObj && ! $self->editObj->isa('Games::Axmud::Profile::World')) {
            return undef;
        } else {
            return 1;
        }
    }

#   sub enableButtons {}        # Inherited from GA::Generic::ConfigWin

#   sub enableSingleButton {}   # Inherited from GA::Generic::ConfigWin

#   sub setupNotebook {}        # Inherited from GA::Generic::ConfigWin

    sub expandNotebook {

        # Called by $self->setupNotebook
        # Set up additional tabs for the notebook, depending on which category of profile is being
        #   being edited
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->expandNotebook', @_);
        }

        # The combobox in $self->rooms2Tab (used by 'Rooms' pages 2, 4 and 6) must be updated
        #   whenever the user adds/deletes a room statement component (in page 1). Use a special IV
        #   to store the comboboxes to update
        $self->{comboHash} = {};
        # The simple lists on these tabs must also be updated, when the user deletes a component on
        #   page 1. Use a special IV to store the simple lists to  update
        $self->{simpleListHash} = {};

        # If this is a current profile, set a flag
        if ($self->session->currentWorld eq $self->editObj) {

            $self->ivPoke('currentFlag', TRUE);
        }

        # Expand the notebook
        $self->settingsTab();
        $self->initialTab();            # Inherited from GA::EditWin::Generic::Profile
        $self->notesTab();              # Inherited from GA::EditWin::Generic::Profile
        $self->overrideTab();
        $self->roomsTab();
        $self->movesTab();
        $self->statusTab();
        $self->inventoryTab();
        $self->channelsTab();
        $self->attackTab();
        $self->worldFightTab();         # (Not inherited from GA::EditWin::Generic::Profile)
        $self->worldInteractionTab();   # (Not inherited from GA::EditWin::Generic::Profile)
        $self->worldCommandsTab();      # (Not inherited from GA::EditWin::Generic::Profile)
        $self->advanceTab();
        $self->currencyTab();
        $self->missionsTab();
        $self->questsTab();
        $self->statsTab();
        $self->privateDataTab(          # Inherited from GA::Generic::EditWin
            'privateHash',
            '_Private',
            'Private data hash',
            'Data created by your own plugins and scripts, but stored in this profile',
        );

        return 1;
    }

    sub saveChanges {

        # Called by $self->buttonOK and $self->buttonSave (usually for 'edit' windows only, not
        #   'pref' windows)
        # Saves any changes made to data stored by the edit object
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->saveChanges', @_);
        }

        if ($self->editHash) {

            # Store the changes the user has made
            foreach my $key ($self->ivKeys('editHash')) {

                $self->editObj->{$key} = $self->ivShow('editHash', $key);
            }

            # If there's a Channels/Divert task running, and the user has just changed some of
            #   the channel pattern IVs, tell the task to update its triggers
            if (
                $self->editObj->category eq 'world'
                && (
                    $self->ivExists('editHash', 'channelList')
                    || $self->ivExists('editHash', 'noChannelList')
                )
            ) {
                if (
                    $self->session->divertTask
                    && (
                        $self->session->divertTask->status eq 'running'
                        || $self->session->divertTask->status eq 'paused'
                    )
                    && $self->session->divertTask->stage > 2
                ) {
                    $self->session->divertTask->resetTriggers();

                # Channels/Divert tasks can't run at the same time
                } elsif (
                    $self->session->channelsTask
                    && (
                        $self->session->channelsTask->status eq 'running'
                        || $self->session->channelsTask->status eq 'paused'
                    )
                ) {
                    $self->session->channelsTask->resetTriggers();
                }
            }

            # If ->termOverrideHash has been updated, inform all sessions whose current world is
            #   this one
            if (
                $self->editObj->category eq 'world'
                && $self->ivExists('editHash', 'termOverrideHash')
            ) {
                foreach my $otherSession ($axmud::CLIENT->ivValues('sessionHash')) {

                    if ($otherSession->currentWorld eq $self->editObj) {

                        $self->session->textViewCursorUpdate();
                        $self->session->textViewKeysUpdate();
                    }
                }
            }

            # The changes can now be cleared
            $self->ivEmpty('editHash');

            # Mark the object's corresponding file object as needing to be saved, if it exists
            if ($self->editObj->_parentFile) {

                $self->editObj->doModify('saveChanges');
            }

            # Update the current session's object viewer window, if it is open
            if ($self->session->viewerWin) {

                $self->session->viewerWin->updateNotebook();
            }

            # Changes to ->basicMappingFlag effect menu bar/toolbar items in automapper windows,
            #   so update them
            foreach my $session ($axmud::CLIENT->findSessions($self->session->currentWorld->name)) {

                if ($session->mapWin) {

                    $session->mapWin->restrictWidgets();
                }
            }
        }

        # The most efficient way to keep ->verboseExitDelimiterList, etc in the right order
        #   (longest first) is to sort them, every time this window is saved
        $self->editObj->ivPoke(
            'verboseExitDelimiterList',
                sort {length ($b) <=> length ($a)} ($self->editObj->verboseExitDelimiterList)
        );

        $self->editObj->ivPoke(
            'briefExitDelimiterList',
                sort {length ($b) <=> length ($a)} ($self->editObj->briefExitDelimiterList)
        );

        $self->editObj->ivPoke(
            'roomCmdDelimiterList',
                sort {length ($b) <=> length ($a)} ($self->editObj->roomCmdDelimiterList)
        );

        return 1;
    }

    # Notebook tabs

    sub nameTab {

        # Name tab - called by $self->setupNotebook
        #
        # Expected arguments
        #   $table  -   The Gtk2::Table
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $table, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->nameTab', @_);
        }

#       # Tab setup (already created by the calling function)
#       my ($vBox, $table) = $self->addTab('_Name', $self->notebook);

        # Profile category
        $self->addLabel($table, '<b>Profile category</b>',
            0, 6, 3, 4);
        $self->addEntry($table, 'category', FALSE,
            1, 6, 4, 5, 16, 16);

        # World long name
        $self->addLabel($table, '<b>World long name</b>',
            7, 13, 0, 1);
        $self->addEntryWithIcon($table, 'longName', 'string', 1, undef,
            8, 13, 1, 2);

        $self->addLabel($table, '<i>No length or character limits</i>',
            8, 13, 2, 3);

        # Current profile
        $self->addLabel($table, '<b>Current profile</b>',
            7, 13, 3, 4);

        my $checkbutton = Gtk2::CheckButton->new();
        if (
            $self->session->ivExists('currentProfHash', $self->editObj->category)
            && $self->session->ivShow('currentProfHash', $self->editObj->category) eq $self->editObj
        ) {
            $checkbutton->set_active(TRUE);
        }
        $checkbutton->set_sensitive(FALSE);
        $table->attach_defaults($checkbutton, 7, 13, 4, 5);

#       # Tab complete (handled by the calling function)
#       $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub settingsTab {

        # Settings tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->settingsTab', @_);
        }

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('Settin_gs', $self->notebook);

        # Add tabs to the inner notebook
        $self->settings1Tab($innerNotebook);
        $self->settings2Tab($innerNotebook);
        $self->settings3Tab($innerNotebook);
        $self->settings4Tab($innerNotebook);
        $self->settings5Tab($innerNotebook);
        $self->settings6Tab($innerNotebook);
        $self->settings7Tab($innerNotebook);
        $self->settings8Tab($innerNotebook);
        $self->settings9Tab($innerNotebook);
        $self->settings10Tab($innerNotebook);
        $self->settings11Tab($innerNotebook);
        $self->settings12Tab($innerNotebook);

        return 1;
    }

    sub settings1Tab {

        # Settings1 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (@comboList, @comboList2);

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->settings1Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _1', $innerNotebook);

        # Connection details
        $self->addLabel($table, '<b>Connection details</b>',
            0, 6, 0, 1);
        $self->addLabel($table, 'DNS',
            1, 3, 1, 2);
        $self->addEntryWithIcon($table, 'dns', 'string', undef, undef,
            3, 6, 1, 2);
        $self->addLabel($table, 'IPV4',
            1, 3, 2, 3);
        $self->addEntryWithIcon($table, 'ipv4', \&ipv4Check, undef, undef,
            3, 6, 2, 3);
        $self->addLabel($table, 'IPV6',
            1, 3, 3, 4);
        $self->addEntryWithIcon($table, 'ipv6', \&ipv6Check, undef, undef,
            3, 6, 3, 4);
        $self->addLabel($table, 'Port',
            1, 3, 4, 5);
        $self->addEntryWithIcon($table, 'port', 'int', 0, 65535,
            3, 6, 4, 5);
        $self->addLabel($table, 'Protocol',
            1, 3, 5, 6);
        $self->addComboBox($table, 'protocol', ['telnet', 'ssh', 'ssl'], '',
            TRUE,           # No 'undef' value used
            3, 6, 5, 6);
        # SSH Connection details
        $self->addLabel($table, '<b>SSH Connection details</b>',
            0, 6, 6, 7);
        $self->addLabel($table, 'SSH username',
            1, 3, 7, 8);
        $self->addEntryWithIcon($table, 'sshUserName', 'string', undef, undef,
            3, 6, 7, 8);
        $self->addLabel($table, 'SSH password',
            1, 3, 8, 9);
        $self->addEntryWithIcon($table, 'sshPassword', 'string', undef, undef,
            3, 6, 8, 9);
        $self->addLabel($table, 'Specify port in SSH connections',
            1, 5, 9, 10);
        $self->addCheckButton($table, 'sshPortFlag', TRUE,
            5, 6, 9, 10, 1, 0.5);

        # Previous connections
        $self->addLabel($table, '<b>Previous connections</b>',
            7, 12, 0, 1);
        $self->addLabel($table, 'Last connect date',
            7, 9, 1, 2);
        $self->addEntry($table, 'lastConnectDate', FALSE,
            9, 12, 1, 2, 20, 20);        # Max chars
        $self->addLabel($table, 'Last connect time',
            7, 9, 2, 3);
        $self->addEntry($table, 'lastConnectTime', FALSE,
            9, 12, 2, 3, 20, 20);        # Max chars
        $self->addLabel($table, 'Last character used',
            7, 9, 3, 4);
        $self->addEntry($table, 'lastConnectChar', FALSE,
            9, 12, 3, 4, 20, 20);        # Max chars
        $self->addLabel($table, 'Total no. connections',
            7, 9, 4, 5);
        $self->addEntry($table, 'numberConnects', FALSE,
            9, 12, 4, 5, 20, 20);        # Max chars

        # General settings
        $self->addLabel($table, '<b>General settings</b>',
            7, 12, 5, 6);
        $self->addLabel($table, 'Ever a current profile?',
            7, 11, 6, 7);
        $self->addCheckButton($table, 'setupCompleteFlag', FALSE,
            11, 12, 6, 7, 1, 0.5);
        $self->addLabel($table, 'World can\'t be saved (during this session)',
            7, 11, 7, 8);
        $self->addCheckButton($table, 'noSaveFlag', FALSE,
            11, 12, 7, 8, 1, 0.5);
        $self->addLabel($table, 'Life status override',
            7, 11, 8, 9);
        $self->addCheckButton($table, 'lifeStatusOverrideFlag', TRUE,
            11, 12, 8, 9);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub settings2Tab {

        # Settings2 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (
            $allString, $reverseFlag,
            @columnList, @charList, @comboList,
        );

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->settings2Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _2', $innerNotebook);

        # Connection history
        $self->addLabel($table, '<b>Connection history</b>',
            0, 12, 0, 1);
        $self->addLabel(
            $table, '<i>Connection history for this world (updated only when the global flag is'
            . ' set. Estimated times in square brackets [...] )</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Character', 'text',
            'Attempted', 'text',
            'Current', 'bool',
            'Connected', 'text',
            'Disconnected', 'text',
            'Connection length', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10, -1, 270);

        # Initialise the list, which starts in reverse order
        $reverseFlag = TRUE;
        $self->settings2Tab_refreshList($slWidget, (scalar @columnList / 2), $reverseFlag);

        # Add editing buttons
        $self->addLabel($table, 'Filter by character:',
            1, 2, 10, 11);

        foreach my $profObj ($self->session->ivValues('profHash')) {

            if ($profObj->category eq 'char') {

                push (@charList, $profObj->name);
            }
        }

        @comboList = sort {lc($a) cmp lc($b)} (@charList);
        $allString = 'Use all characters';
        unshift (@comboList, $allString);

        my $combo = $self->addComboBox($table, undef, \@comboList, '', TRUE,
            2, 6, 10, 11);

        my $button = $self->addButton($table,
            'Filter', 'Filter the list', undef,
            6, 7, 10, 11);
        $button->signal_connect('clicked' => sub {

            my $char = $combo->get_active_text();
            if ($char eq $allString) {

                # Refresh the full list
                $self->settings2Tab_refreshList(
                    $slWidget,
                    (scalar @columnList / 2),
                    $reverseFlag,
                );

            } else {

                # Filter the list
                $self->settings2Tab_refreshList(
                    $slWidget,
                    (scalar @columnList / 2), $reverseFlag,
                    $char,
                );
            }
        });

        my $button2 = $self->addButton($table,
            'Reverse', 'Reverse the order of the list', undef,
            7, 8, 10, 11);
        $button2->signal_connect('clicked' => sub {

            my $char;

            if (! $reverseFlag) {
                $reverseFlag = TRUE;
            } else {
                $reverseFlag = FALSE;
            }

            $char = $combo->get_active_text();
            if ($char eq $allString) {

                # Refresh the full list
                $self->settings2Tab_refreshList(
                    $slWidget,
                    (scalar @columnList / 2),
                    $reverseFlag,
                );

            } else {

                # Filter the list
                $self->settings2Tab_refreshList(
                    $slWidget,
                    (scalar @columnList / 2), $reverseFlag,
                    $char,
                );
            }
        });

        my $button3 = $self->addButton($table,
            'Refresh', 'Refresh the list', undef,
            8, 10, 10, 11);
        $button3->signal_connect('clicked' => sub {

            my $char = $combo->get_active_text();
            if ($char eq $allString) {

                # Refresh the full list
                $self->settings2Tab_refreshList(
                    $slWidget,
                    (scalar @columnList / 2),
                    $reverseFlag,
                );

            } else {

                # Filter the list
                $self->settings2Tab_refreshList(
                    $slWidget,
                    (scalar @columnList / 2), $reverseFlag,
                    $char,
                );
            }
        });

        my $button4 = $self->addButton($table,
            'Clear', 'Clear the list', undef,
            10, 12, 10, 11);
        $button4->signal_connect('clicked' => sub {

            # Add an empty list to $self->editHash
            $self->ivAdd('editHash', 'connectHistoryList', []);

            # Refresh the simple list and reset entry boxes
            $self->settings2Tab_refreshList($slWidget, (scalar @columnList / 2), $reverseFlag);
            $combo->set_active(0);
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub settings2Tab_refreshList {

        # Resets the simple list displayed by $self->settings2Tab
        #
        # Expected arguments
        #   $slWidget       - The GA::Obj::Simple::List
        #   $columns        - The number of columns
        #   $reverseFlag    - If TRUE, the most recent connection is shown first. If FALSE, the
        #                       earliest connection is shown first
        #
        # Optional arguments
        #   $filter         - If set, filter the list showing only connections using the specified
        #                       character profile name. If 'undef', all connections are shown
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $slWidget, $columns, $reverseFlag, $filter, $check) = @_;

        # Local variables
        my (@dataList, @ivList);

        # Check for improper arguments
        if (! defined $slWidget || ! defined $columns || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->settings2Tab_refreshList',
                @_,
            );
        }

        # Import the IV
        @ivList = $self->getEditHash_listIV('connectHistoryList');
        if ($reverseFlag) {

            @ivList = reverse @ivList;
        }

        # Compile the simple list data
        foreach my $historyObj (@ivList) {

            my ($char, $cTime, $flag, $cdTime, $dcTime, $len);

            if (! $filter || ($historyObj->char && $historyObj->char eq $filter)) {

                if (! defined $historyObj->char) {
                    $char = '(not set)';
                } else {
                    $char = $historyObj->char;
                }

                if (defined $historyObj->connectingTime) {

                    $cTime = $axmud::CLIENT->convertTime($historyObj->connectingTime, 'date') . ', '
                                . $axmud::CLIENT->convertTime($historyObj->connectingTime, 'clock');
                }

                if (
                    $self->session->connectHistoryObj
                    && $self->session->connectHistoryObj eq $historyObj
                ) {
                    $flag = TRUE;
                } else {
                    $flag = FALSE;
                }

                if (defined $historyObj->connectedTime) {

                    $cdTime = $axmud::CLIENT->convertTime($historyObj->connectedTime, 'clock');
                }

                if (
                    ! $self->session->connectHistoryObj
                    || $self->session->connectHistoryObj ne $historyObj
                ) {
                    if (! defined $historyObj->disconnectedTime) {

                        # Axmud crashed or was terminated abruptly
                        $dcTime = '['
                            . $axmud::CLIENT->convertTime($historyObj->currentTime, 'clock') . ']';

                    } else {

                        $dcTime
                            = $axmud::CLIENT->convertTime($historyObj->disconnectedTime, 'clock');
                    }
                }

                if (defined $historyObj->connectionLength) {

                    if (
                        ! defined $historyObj->disconnectedTime
                        && (
                            ! $self->session->connectHistoryObj
                            || $self->session->connectHistoryObj ne $historyObj
                        )
                    ) {
                        $len = '[' . $axmud::CLIENT->getCounter($historyObj->connectionLength)
                                    . ']';

                    } else {

                        $len = $axmud::CLIENT->getCounter($historyObj->connectionLength);
                    }
                }

                push (@dataList, $char, $cTime, $flag, $cdTime, $dcTime, $len);
            }
        }

        # Reset the simple list
        $self->resetListData($slWidget, [@dataList], $columns);

        return 1;
    }

    sub settings3Tab {

        # Settings3 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->settings3Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _3', $innerNotebook);

        # Websites. ->signal_connects appear further below
        $self->addLabel($table, '<b>Websites</b>',
            0, 12, 0, 1);
        $self->addLabel($table, 'Website URL',
            1, 3, 1, 2);
        my $entry = $self->addEntry($table, 'worldURL', TRUE,
            3, 10, 1, 2);
        my $button = $self->addButton($table,
            'View', 'View this link', undef,
            10, 12, 1, 2);

        $self->addLabel($table, 'Referrer URL',
            1, 3, 2, 3);
        my $entry2 = $self->addEntry($table, 'referURL', TRUE,
            3, 10, 2, 3);
        my $button2 = $self->addButton($table,
            'View', 'View this link', undef,
            10, 12, 2, 3);

        $button->signal_connect('clicked' => sub {

            # Let the user open junk URLs, if they want to, but don't let them open a non-existent
            #   URL
            my $url = $entry->get_text();
            if ($url) {

                $axmud::CLIENT->openURL($url);
            }
        });

        $button2->signal_connect('clicked' => sub {

            my $url = $entry2->get_text();
            if ($url) {

                $axmud::CLIENT->openURL($url);
            }
        });

        $self->addLabel($table, '<b>World description</b>',
            0, 3, 3, 4);
        $self->addLabel($table, 'General description',
            1, 3, 4, 5);
        $self->addLabel($table, '',         # Empty labels to get the spacing right
            1, 3, 5, 6);
        $self->addLabel($table, '',
            1, 3, 6, 7);
        $self->addLabel($table, '',
            1, 3, 7, 8);
        $self->addLabel($table, '',
            1, 3, 8, 9);
        $self->addLabel($table, '',
            1, 3, 9, 10);
        $self->addTextView($table, 'worldDescrip', TRUE,
            3, 12, 4, 10,
            FALSE, undef, undef, TRUE,      # Treat as a scalar, not a list/forbid horizontal scroll
            -1, 200);                       # Fixed height

        $self->addLabel($table, 'World hint',
            1, 3, 10, 11);
        $self->addEntry($table, 'worldHint', TRUE,
            3, 12, 10, 11);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub settings4Tab {

        # Settings4 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->settings4Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _4', $innerNotebook);

        # Automatic login mode
        $self->addLabel($table, '<b>Automatic login mode</b>',
            0, 12, 0, 1);

        my ($group, $radioButton) = $self->addRadioButton(
            $table, undef, '\'none\'', 'loginMode',
            'none',          # IV set to this value when toggled
            TRUE,       # Sensitive widget
            1, 4, 1, 2);
        $self->addLabel($table, 'No automatic login (user must use \';login\' command)',
            4, 12, 1, 2);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'immediate\'', 'loginMode', 'immediate', TRUE,
            1, 4, 2, 3);
        $self->addLabel($table, 'Immediate login (not recommended)',
            4, 12, 2, 3);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'lp\'', 'loginMode', 'lp', TRUE,
            1, 4, 3, 4);
        $self->addLabel(
            $table, 'LP/Diku/AberMUD login (consecutive prompts for character/password)',
            4, 12, 3, 4);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'tiny\'', 'loginMode', 'tiny', TRUE,
            1, 4, 4, 5);
        $self->addLabel(
            $table, 'TinyMUD login (send \'connect char pass\' at first prompt)',
            4, 12, 4, 5);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'world_cmd\'', 'loginMode', 'world_cmd', TRUE,
            1, 4, 5, 6);
        $self->addLabel(
            $table, 'Send a sequence of world commands at the first prompt (see next tab)',
            4, 12, 5, 6);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'telnet\'', 'loginMode', 'telnet', TRUE,
            1, 4, 6, 7);
        $self->addLabel(
            $table, 'Basic telnet login (e.g. \'login:\', \'password:\')',
            4, 12, 6, 7);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'task\'', 'loginMode', 'task', TRUE,
            1, 4, 7, 8);
        $self->addLabel(
            $table, 'Run a task to handle the login (see next tab)',
            4, 12, 7, 8);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'script_task\'', 'loginMode', 'script_task', TRUE,
            1, 4, 8, 9);
        $self->addLabel(
            $table,
            'Run ' . $axmud::BASIC_ARTICLE . ' script from within a task to handle the login (see'
            . ' next tab)',
            4, 12, 8, 9);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'script\'', 'loginMode', 'script', TRUE,
            1, 4, 9, 10);
        $self->addLabel(
            $table, 'Run ' . $axmud::BASIC_ARTICLE . ' script to handle the login (see next tab)',
            4, 12, 9, 10);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'mission\'', 'loginMode', 'mission', TRUE,
            1, 4, 10, 11);
        $self->addLabel(
            $table, 'Start a mission to handle the login (see next tab)',
            4, 12, 10, 11);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub settings5Tab {

        # Settings5 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->settings5Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _5', $innerNotebook);

        # Automatic login mode (2)
        $self->addLabel($table, '<b>Automatic login mode (2)</b>',
            0, 10, 0, 1);
        $self->addRegexButton($table,
            [
                'list', 'loginConnectPatternList',
                'list', 'loginSuccessPatternList',
            ],
            10, 12, 0, 1);

        $self->addLabel(
            $table, '<i>(NB The \'Initial\' tab lists the tasks, scripts and missions that start'
            . ' when the login process is complete)</i>',
            1, 12, 1, 2);
        $self->addLabel(
            $table,
            '<i>(NB In mode \'world_cmd\', if world commands contain @name@, @account@ and/or'
            . ' @password@, they are substituted)</i>',
            1, 12, 2, 3);

        $self->addLabel($table, 'Mode \'tiny\': List of login expected patterns',
            1, 8, 3, 4);
        $self->addTextView($table, 'loginConnectPatternList', TRUE,
            1, 8, 4, 6,
            undef, undef, undef, undef,
            -1, 110);           # Fixed height

        $self->addLabel($table, 'Mode \'world_cmd\': List of world commands to send',
            8, 12, 3, 4);
        $self->addTextView($table, 'loginCmdList', TRUE,
            8, 12, 4, 6,
            undef, undef, undef, undef,
            -1, 110);           # Fixed height

        $self->addLabel(
            $table,
            'Mode \'lp\', \'tiny\', \'world_cmd\', \'telnet\': Login success patterns',
            1, 8, 7, 8);
        $self->addTextView($table, 'loginSuccessPatternList', TRUE,
            1, 8, 8, 12,
            TRUE, TRUE, FALSE, FALSE, # Treat as a list, remove empty lines, don't remove whitespace
            -1, 110);           # Fixed height

        $self->addLabel($table, 'Task/script/mission modes: task/script/mission to start',
            8, 12, 7, 8);
        my $entry = $self->addEntry($table, 'loginObjName', TRUE,
            8, 12, 8, 9);

        # Add a button
        my $button = $self->addButton($table,
            'View / Edit', 'View the task, or edit the script/mission', undef,
            10, 12, 9, 10);
        $button->signal_connect('clicked' => sub {

            my ($text, $mode);

            $text = $entry->get_text();
            $mode = $self->getEditHash_scalarIV('loginMode');

            # (Do nothing, if the entry box is empty)
            if ($text) {

                # Login mode 'task' - run a task (character is logged if the task calls
                #   GA::Session->doLogin)
                if ($mode eq 'task') {

                    # Open an 'edit' window for this task
                    $self->session->pseudoCmd('edittask ' . $text, $self->pseudoCmdMode);

                # Login mode 'script_task' - run an Axbasic script from within a task (character is
                #   logged in if the script executes a LOGIN statement)
                # Login mode 'script' - run an Axbasic script (character is logged in if the script
                #   executes a LOGIN statement)
                } elsif ($mode eq 'script_task' || $mode eq 'script') {

                    # Open the script in default text editor
                    $self->session->pseudoCmd('editscript ' . $text, $self->pseudoCmdMode);

                # Login mode 'mission' - start a mission (character is logged in if the mission uses
                #   the ';login' client command)
                } elsif ($mode eq 'mission') {

                    # Open an 'edit' window for this mission
                    $self->session->pseudoCmd('editmission ' . $text, $self->pseudoCmdMode);
                }
            }
        });

        $self->addLabel($table, '',     # Get table spacing right using empty labels
            8, 12, 10, 11);
        $self->addLabel($table, '',     # Get table spacing right using empty labels
            8, 12, 11, 12);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub settings6Tab {

        # Settings6 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->settings6Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _6', $innerNotebook);

        # Automatic login mode (3)
        $self->addLabel($table, '<b>Automatic login mode (3)</b>',
            0, 12, 0, 1);

        $self->addLabel($table,
            'Mode \'mission\': List of patterns matching lines with character names, and'
            . ' world command substitutions',
            1, 12, 1, 2);
        $self->addLabel($table,
            '    <i>e.g. line  <b>1 Gandalf</b>  -  pattern'
            . '  <b>^(\d+)\s(\w+)$</b>  -  group substring number  <b>2</b>  -  command'
            . '  <b>$1</b></i>',
            1, 12, 2, 3);

        # Add a simple list
        @columnList = (
            'Pattern', 'text',
            'Substring #', 'int',
            'Substitution', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 3, 9,
            -1, 180);       # Fixed height

        # Initialise the list
        $self->refreshList_listIV($slWidget, scalar (@columnList / 2), 'loginSpecialList');

        # Add entries for adding new patterns
        $self->addLabel($table, 'Pattern:',
            1, 3, 9, 10);
        my $entry = $self->addEntryWithIcon($table, undef, 'regex', 1, undef,
            3, 12, 9, 10);

        $self->addLabel($table, 'Substring #:',
            1, 3, 10, 11);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'int', 1, undef,
            3, 6, 10, 11);

        $self->addLabel($table, 'World command:',
            6, 8, 10, 11);
        my $entry3 = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            8, 12, 10, 11);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_listIV(
            $table,
            $slWidget,
            'loginSpecialList',
            11, 3,
            $entry, $entry2, $entry3,
        );
        $button->signal_connect('clicked' => sub {

            my ($pattern, $grpNum, $cmd);

            $pattern = $entry->get_text();
            $grpNum = $entry2->get_text();
            $cmd = $entry3->get_text();

            if ($self->checkEntryIcon($entry, $entry2, $entry3)) {

                # Add new values to (the end of) the list IV
                $self->addEditHash_listIV(
                    'loginSpecialList',
                    undef, FALSE,
                    $pattern, $grpNum, $cmd,
                );

                # Refresh the simple list and reset entry boxes
                $self->refreshList_listIV(
                    $slWidget,
                    scalar (@columnList / 2),
                    'loginSpecialList',
                );

                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub settings7Tab {

        # Settings7 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->settings7Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _7', $innerNotebook);

        # Automatic login mode
        $self->addLabel($table, '<b>Automatic quit mode</b>',
            0, 12, 0, 1);
        $self->addLabel(
            $table,
            '<i>What happens when you use the client commands \';quit\' , \';qquit\' or'
            . ' \';quitall\'</i>',
            1, 12, 1, 2);

        my ($group, $radioButton) = $self->addRadioButton(
            $table, undef, '\'normal\'', 'autoQuitMode',
            'normal',   # IV set to this value when toggled
            TRUE,       # Sensitive widget
            1, 3, 2, 3);
        $self->addLabel(
            $table,
            'Send standard \'quit\' world command',
            3, 6, 2, 3);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'world_cmd\'', 'autoQuitMode', 'world_cmd', TRUE,
            7, 9, 2, 3);
        $self->addLabel($table, 'Send a sequence of world commands',
            9, 12, 2, 3);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'task\'', 'autoQuitMode', 'task', TRUE,
            1, 3, 3, 4);
        $self->addLabel(
            $table, 'Run a task',
            3, 6, 3, 4);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'task_script\'', 'autoQuitMode', 'task_script', TRUE,
            7, 9, 3, 4);
        $self->addLabel(
            $table, 'Run ' . $axmud::BASIC_ARTICLE . ' script from within a task',
            9, 12, 3, 4);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'script\'', 'autoQuitMode', 'script', TRUE,
            1, 3, 4, 5);
        $self->addLabel(
            $table, 'Run an ' . $axmud::BASIC_ARTICLE . ' script',
            3, 6, 4, 5);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'mission\'', 'autoQuitMode', 'mission', TRUE,
            7, 9, 4, 5);
        $self->addLabel(
            $table, 'Start a mission',
            9, 12, 4, 5);

        $self->addLabel($table, 'Mode \'world_cmd\': List of world commands to send',
            1, 6, 5, 6);
        $self->addTextView($table, 'autoQuitCmdList', TRUE,
            1, 6, 6, 12,
            undef, undef, undef, undef,
            -1, 200);           # Fixed height

        $self->addLabel($table, 'Other modes: task, script or mission to start',
            7, 12, 5, 6);
        my $entry = $self->addEntry($table, 'autoQuitObjName', TRUE,
            7, 12, 6, 7);

        # Add a button
        my $button = $self->addButton($table,
            'View / Edit', 'View the task, or edit the script/mission', undef,
            10, 12, 7, 8);
        $button->signal_connect('clicked' => sub {

            my ($text, $mode);

            $text = $entry->get_text();
            $mode = $self->getEditHash_scalarIV('autoQuitMode');

            # (Do nothing, if the entry box is empty)
            if ($text) {

                # Auto-quit mode 'task' - run a task (the task is responsible for sending a 'quit'
                #   world command)
                if ($mode eq 'task') {

                    # Open an 'edit' window for this task
                    $self->session->pseudoCmd('edittask ' . $text, $self->pseudoCmdMode);

                # Auto-quit mode 'task_script' - run an Axbasic script from within a task (the
                #   script is responsible for sending a 'quit' world command)
                # Auto-quit mode 'script' - run an Axbasic script  (the script is responsible for
                #   sending a 'quit' world command)
                } elsif ($mode eq 'task_script' || $mode eq 'script') {

                    # Open the script in default text editor
                    $self->session->pseudoCmd('editscript ' . $text, $self->pseudoCmdMode);

                # Auto-quit mode 'mission' - start a mission (the mission is responsible for sending
                #   a 'quit' world command)
                } elsif ($mode eq 'mission') {

                    # Open an 'edit' window for this mission
                    $self->session->pseudoCmd('editmission ' . $text, $self->pseudoCmdMode);
                }
            }
        });

        $self->addLabel($table, '',     # Get table spacing right using empty labels
            7, 12, 8, 9);
        $self->addLabel($table, '',
            7, 12, 9, 10);
        $self->addLabel($table, '',
            7, 12, 10, 11);
        $self->addLabel($table, '',
            7, 12, 11, 12);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub settings8Tab {

        # Settings8 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (
            $obscureFlag,
            @columnList,
        );

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->settings8Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _8', $innerNotebook);

        # Character/password list
        $self->addLabel($table, '<b>Character/password list</b>',
            0, 6, 0, 1);

        $self->addLabel($table,
            '<i>(NB Setting the account login mode on the right is optional; it doesn\'t affect the'
            . ' automatic login process)</i>',
            1, 12, 1, 2);

        my $checkButton = $self->addCheckButton($table, undef, TRUE,
            1, 2, 2, 3);
        # Passwords are initially obscured
        $obscureFlag = TRUE;
        $checkButton->set_active($obscureFlag);
        # (Signal connect follows)

        $self->addLabel($table, 'Obscure passwords',
            2, 6, 2, 3);

        my ($group, $radioButton) = $self->addRadioButton(
            $table, undef, '\'unknown\'', 'loginAccountMode',
            'unknown',  # IV set to this value when toggled
            TRUE,       # Sensitive widget
            6, 7, 2, 3);
        $self->addLabel($table, 'Unknown if account name is also required to log in',
            7, 12, 2, 3);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'not_required\'', 'loginAccountMode', 'not_required', TRUE,
            6, 7, 3, 4);
        $self->addLabel($table, 'Account name not required',
            7, 12, 3, 4);

         ($group, $radioButton) = $self->addRadioButton(
            $table, $group, '\'required\'', 'loginAccountMode', 'required', TRUE,
            6, 7, 4, 5);
        $self->addLabel($table, 'Account name required',
            7, 12, 4, 5);

        # Add a simple list
        @columnList = (
            'Character', 'text',
            'Password', 'text',
            'Associated account name (if any)', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 5, 11, -1, 190);      # Fixed height

        # Initialise the list
        $self->settings8Tab_refreshList($slWidget, (scalar @columnList / 2), $obscureFlag);

        # signal_connect for earlier checkbutton
        $checkButton->signal_connect('toggled' => sub {

            $obscureFlag = $checkButton->get_active();
            $self->settings8Tab_refreshList($slWidget, (scalar @columnList / 2), $obscureFlag);
        });

        # Add editing buttons
        my $button = $self->addButton($table,
            'Edit password', 'Set or edit the password for the selected character', undef,
            1, 4, 11, 12);
        $button->signal_connect('clicked' => sub {

            my (
                $name, $pwd, $newPwd, $profObj,
                %passwordHash,
            );

            ($name) = $self->getSimpleListData($slWidget, 0);
            if (defined $name) {

                # Import the world's password hash
                %passwordHash = $self->getEditHash_hashIV('passwordHash');

                # (Protect against missing character profiles...)
                $profObj = $self->session->ivShow('profHash', $name);
                if (
                    exists $passwordHash{$name}
                    && $profObj
                    && $profObj->category eq 'char'
                ) {
                    # Get the corresponding password (if any)
                    $pwd = $passwordHash{$name};

                    # Prompt the user for a new password
                    $newPwd = $self->showEntryDialogue(
                        'Set password',
                        'Set the password for \'' . $name . '\'',
                        undef,              # No max chars
                        $pwd,               # Current password
                        TRUE,               # Obscure text in the entry box
                    );

                    # (If user clicked 'cancel' button, $newPwd is 'undef'; if the user entered no
                    #   password, $newPwd is an empty string)
                    if (defined $newPwd) {

                        if ($newPwd) {
                            $passwordHash{$name} = $newPwd;
                        } else {
                            $passwordHash{$name} = undef;
                        }

                        # Update the IV
                        $self->ivPoke('editHash', 'passwordHash', \%passwordHash);
                    }
                }
            }

            # Refresh the simple list
            $self->settings8Tab_refreshList($slWidget, (scalar @columnList / 2), $obscureFlag);
        });

        my $button2 = $self->addButton($table,
            'Edit account name', 'Set or edit the account name for the selected character', undef,
            4, 6, 11, 12);
        $button2->signal_connect('clicked' => sub {

            my (
                $name, $profObj, $newAccount,
                %accountHash,
            );

            ($name) = $self->getSimpleListData($slWidget, 0);
            if (defined $name) {

                # Import the world's account hash
                %accountHash = $self->getEditHash_hashIV('accountHash');

                # (Protect against missing character profiles...)
                $profObj = $self->session->ivShow('profHash', $name);
                if (
                    exists $accountHash{$name}
                    && $profObj
                    && $profObj->category eq 'char'
                ) {
                    # Prompt the user for a new account name
                    $newAccount = $self->showEntryDialogue(
                        'Set account name',
                        'Set the associated account name for \'' . $name . '\'',
                        undef,                  # No max chars
                        $accountHash{$name},    # Current account name - may be 'undef'
                    );

                    # (If user clicked 'cancel' button, $newAccount is 'undef'; if the user entered
                    #   no account name, $newAccount is an empty string)
                    if (defined $newAccount) {

                        if ($newAccount) {
                            $accountHash{$name} = $newAccount;
                        } else {
                            $accountHash{$name} = undef;
                        }

                        # Update the IV
                        $self->ivPoke('editHash', 'accountHash', \%accountHash);
                    }
                }
            }

            # Refresh the simple list
            $self->settings8Tab_refreshList($slWidget, (scalar @columnList / 2), $obscureFlag);
        });

        my $button3 = $self->addButton($table,
            'Refresh list', 'Refresh the list of characters', undef,
            9, 12, 11, 12);
        $button3->signal_connect('clicked' => sub {

            # Refresh the simple list
            $self->settings8Tab_refreshList($slWidget, (scalar @columnList / 2), $obscureFlag);
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub settings8Tab_refreshList {

        # Resets the simple list displayed by $self->settings8Tab
        #
        # Expected arguments
        #   $slWidget       - The GA::Obj::Simple::List
        #   $columns        - The number of columns
        #   $obscureFlag    - Flag set to TRUE if passwords should be obscured, FALSE otherwise
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $slWidget, $columns, $obscureFlag, $check) = @_;

        # Local variables
        my (
            @charList, @dataList,
            %passwordHash, %accountHash,
        );

        # Check for improper arguments
        if (
            ! defined $slWidget || ! defined $columns || ! defined $obscureFlag
            || defined $check
        ) {
            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->settings8Tab_refreshList',
                @_,
            );
        }

        # Import the IVs
        %passwordHash = $self->getEditHash_hashIV('passwordHash');
        %accountHash = $self->getEditHash_hashIV('accountHash');
        # Get a sorted list of character names
        @charList = sort {lc($a) cmp lc($b)} (keys %passwordHash);

        # Compile the simple list data
        foreach my $charName (@charList) {

            my $password;

            if ($obscureFlag) {

                if ($passwordHash{$charName}) {
                    $password = '********';
                } else {
                    $password = '';
                }

            } else {

                $password = $passwordHash{$charName};
            }

            push (@dataList, $charName, $password, $accountHash{$charName});
        }

        # Reset the simple list
        $self->resetListData($slWidget, [@dataList], $columns);

        return 1;
    }

    sub settings9Tab {

        # Settings9 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->settings9Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _9', $innerNotebook);

        # Missions list
        $self->addLabel($table, '<b>Associated profiles</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of profiles associated with this one (inclusive)</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Name', 'text',
            'Category', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10, -1, 270);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'profHash');

        # Add a button
        my $button = $self->addButton($table,
            'Refresh list', 'Refresh the list of associated profiles', undef,
            9, 12, 10, 11);
        $button->signal_connect('clicked' => sub {

            $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'profHash');
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub settings10Tab {

        # Settings10 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (
            $noUndefFlag,
            @comboList,
        );

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->settings10Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page 1_0', $innerNotebook);

        # Dictionary settings
        $self->addLabel($table, '<b>Dictionary settings</b>',
            0, 6, 0, 1);
        $self->addLabel($table, '<i>Dictionary used with this world</i>',
            1, 6, 1, 2);
        @comboList = sort {lc($a) cmp lc($b)} ($axmud::CLIENT->ivKeys('dictHash'));

        # World profiles which have never been a current profile can have their ->dict IV set to
        #   'undef'; we need to take account of that when calling ->addComboBox
        if (defined $self->editObj->dict) {
            $noUndefFlag = TRUE;        # Combo doesn't contain an item corresponding to 'undef'
        } else {
            $noUndefFlag = FALSE;       # Combo does contain an item corresponding to 'undef'
        }

        my $combo = $self->addComboBox($table, undef, \@comboList, '',
            $noUndefFlag,
            1, 4, 2, 3);
        $self->settings10Tab_refreshCombo($combo);

        my $button = $self->addButton($table, 'Edit...', 'Edit the selected dictionary', undef,
            4, 6, 2, 3);
        $button->signal_connect('clicked' => sub {

            my ($dictName, $dictObj, $childWinObj);

            # Get the dictionary name selected in the combo
            $dictName = $combo->get_active_text();
            if ($dictName) {

                # Get the blessed reference
                $dictObj = $axmud::CLIENT->ivShow('dictHash', $dictName);
            }

            if ($dictObj) {

                # Open up a dictionary 'edit' window to edit the existing dictionary
                $childWinObj = $self->createFreeWin(
                    'Games::Axmud::EditWin::Dict',
                    $self,
                    $self->session,
                    'Edit dictionary \'' . $dictName . '\'',
                    $dictObj,
                    FALSE,                  # Not temporary
                );
            }

            if ($childWinObj) {

                # When the 'edit' window closes, update widgets and/or IVs
                $self->add_childDestroy(
                    $childWinObj,
                    'settings10Tab_refreshCombo',
                    [$combo],
                );
            }
        });

        my $button2 = $self->addButton($table, 'New...', 'Add new dictionary', undef,
            1, 3, 3, 4,
            TRUE);          # Irreversible
        $button2->signal_connect('clicked' => sub {

            my ($dictObj, $childWinObj);

            # Create a temporary dictionary
            $dictObj = Games::Axmud::Obj::Dict->new($self->session, '<temp>', 'English', TRUE);
            if ($dictObj) {

                # Open up a dictionary 'edit' window to create the new dictionary, replacing the
                #   temporary one we've just created
                $self->createFreeWin(
                    'Games::Axmud::EditWin::Dict',
                    $self,
                    $self->session,
                    'Edit dictionary \'' . $dictObj->name . '\'',
                    $dictObj,
                    TRUE,           # Temporary object
                );

                if ($childWinObj) {

                    # When the 'edit' window closes, update widgets and/or IVs
                    $self->add_childDestroy(
                        $childWinObj,
                        'settings10Tab_refreshCombo',
                        [$combo],
                    );
                }
            }
        });

        my $button3 = $self->addButton(
            $table, 'Refresh list', 'Refresh the list of dictionaries', undef,
            3, 6, 3, 4);
        $button3->signal_connect('clicked' => sub {

            # Refresh the combo
            $self->settings10Tab_refreshCombo($combo);
        });

        # Multiples
        $self->addLabel($table, '<b>Multiples</b>',
            0, 12, 4, 5);
        $self->addLabel($table, 'Add numbers to similar objects',
            1, 5, 5, 6);
        $self->addCheckButton($table, 'numberedObjFlag', TRUE,
            5, 6, 5, 6);
        $self->addLabel($table, 'Multiple object pattern',
            1, 4, 6, 7);
        $self->addEntryWithIconButton($table, 'multiplePattern', 'regex', undef, undef,
            4, 6, 6, 7);

        # Consecutive empty line suppression
        $self->addLabel($table, '<b>Consecutive empty line suppression</b>',
            7, 13, 0, 1);
        $self->addLabel($table, 'Lines to suppress (0 - none, 1 - all)',
            8, 11, 1, 2);
        $self->addEntryWithIcon($table, 'suppressEmptyLineCount', 'int', 0, undef,
            11, 13, 1, 2, 8, 8);
        $self->addLabel($table, 'Suppress empty lines before login',
            8, 12, 2, 3);
        $self->addCheckButton($table, 'suppressBeforeLoginFlag', TRUE,
            12, 13, 2, 3);

        # Command separator
        $self->addLabel($table, '<b>Command separator \'' . $axmud::CLIENT->cmdSep . '\'</b>',
            7, 12, 3, 4);
        $self->addLabel($table, 'Auto-separate commands in this world',
            8, 12, 4, 5);
        $self->addCheckButton($table, 'autoSeparateCmdFlag', TRUE,
            12, 13, 4, 5);

        # Slowwalking
        $self->addLabel($table, '<b>Slowwalking</b>',
            7, 13, 5, 6);
        $self->addLabel($table, 'Max world commands (0 - no limit)',
            8, 11, 6, 7);
        $self->addEntryWithIcon($table, 'excessCmdLimit', 'int', 0, undef,
            11, 13, 6, 7, 6, 4);
        $self->addLabel($table, '...in time period (min 0.1 secs)',
            8, 11, 7, 8);
        $self->addEntryWithIcon($table, 'excessCmdDelay', 'float', 0.1, undef,
            11, 13, 7, 8, 6, 4);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub settings10Tab_refreshCombo {

        # Resets the combobox displayed by $self->settings10Tab
        #
        # Expected arguments
        #   $combo          - The Gtk2::Combo
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $combo, $check) = @_;

        # Local variables
        my (
            $currentDict,
            @dictList, @comboList,
        );

        # Check for improper arguments
        if (! defined $combo || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->settings10Tab_refreshCombo',
                @_,
            );
        }

        # Import the GA::Client's list of dictionaries, and sort them
        @dictList = sort {lc($a->name) cmp lc($b->name)} ($axmud::CLIENT->ivValues('dictHash'));

        # Get the profile's current dictionary (might be stored in $self->editHash, or the
        #   unmodified value stored in $self->editObj)
        $currentDict = $self->getEditHash_scalarIV('dict');

        # If $currentDict is defined (and it should be), remove it from @dictList, and insert it at
        #   the beginning, so it can appear at the top of the combo
        foreach my $dictObj (@dictList) {

            if (! $currentDict || $currentDict ne $dictObj->name) {

                push (@comboList, $dictObj->name);
            }
        }

        if ($currentDict) {

            unshift (@comboList, $currentDict);

        } else {

            # (If $currentDict was not, for some reason, defined, an empty string is inserted for
            #   setting the IV to 'undef')
            unshift (@comboList, '');
        }

        # Refill the combo
        $self->resetComboBox($combo, @comboList);

        # Operation complete
        return 1;
    }

    sub settings11Tab {

        # Settings11 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->settings11Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page 11', $innerNotebook);

        # Units
        $self->addLabel($table, '<b>Units</b>',
            0, 10, 0, 1);
        $self->addRegexButton($table,
            [
                'list', 'cmdPromptPatternList',
            ],
            10, 12, 0, 1);

        $self->addLabel($table, 'Character age units',
            1, 6, 1, 2);
        $self->addEntryWithButton($table, 'charAgeUnit', TRUE,
            1, 6, 2, 3);
        $self->addLabel($table, 'Standard weight unit',
            1, 6, 3, 4);
        $self->addEntryWithButton($table, 'weightUnit', TRUE,
            1, 6, 4, 5);
        $self->addLabel($table, 'Strings for weight units (inc. plurals)',
            7, 12, 1, 2);
        $self->addTextView($table, 'weightUnitStringList', TRUE,
            7, 12, 2, 6,
            TRUE, TRUE, FALSE, FALSE, # Treat as a list, remove empty lines, don't remove whitespace
            -1, 100);                 # Fixed width/height

        # Command prompt patterns
        $self->addLabel(
            $table,
            '<b>Command prompt patterns</b>',
            0, 5, 6, 7);              # Use 5, rather than 12, to avoid Gtk layout problems
        $self->addLabel(
            $table,
#            '<i>You can add an initial ^ character, but do not add a terminating $ character</i>',
            '<i>Do not add a terminating $ character</i>',
            1, 12, 7, 8);
        $self->addTextView($table, 'cmdPromptPatternList', TRUE,
            1, 12, 8, 12,
            TRUE, TRUE, FALSE, FALSE, # Treat as a list, remove empty lines, don't remove whitespace
            -1, 150);                 # Fixed height

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub settings12Tab {

        # Settings12 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (@columnList, @comboList);

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->settings12Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page 12', $innerNotebook);

        # Session logging settings
        $self->addLabel($table, '<b>Session logging settings</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of logfiles written by each session using this world profile</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Logfile', 'text',
            'Write?', 'bool',
            'Description', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 270);      # Fixed height

        # Initialise the list
        $self->settings12Tab_refreshList($slWidget, (scalar @columnList / 2));

        # Add a two combos and a button
        @comboList = $axmud::CLIENT->constSessionLogOrderList;
        my $comboBox = $self->addComboBox($table, undef, \@comboList, '',
            TRUE,                       # No 'undef' value used
            1, 4, 10, 11);

        my $button = $self->addButton($table,
            'Toggle', 'Change the setting for the selected logfile', undef,
            4, 6, 10, 11,
            TRUE,           # Irreversible
        );
        $button->signal_connect('clicked' => sub {

            my $logfile = $comboBox->get_active_text();

            # Update the IV
            $self->session->pseudoCmd('log ' . $logfile, $self->pseudoCmdMode);

            # Refresh the simple list
            $self->settings12Tab_refreshList($slWidget, (scalar @columnList / 2));
        });

        # Widgets can't be manipulated when this isn't a current profile
        if (! $self->currentFlag) {

            $comboBox->set_sensitive(FALSE);
            $button->set_sensitive(FALSE);
        }

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub settings12Tab_refreshList {

        # Resets the simple list displayed by $self->settings12Tab
        #
        # Expected arguments
        #   $slWidget   - The GA::Obj::Simple::List
        #   $columns    - The number of columns
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $slWidget, $columns, $check) = @_;

        # Local variables
        my (
            @dataList,
            %ivHash,
        );

        # Check for improper arguments
        if (! defined $slWidget || ! defined $columns || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->settings12Tab_refreshList',
                @_,
            );
        }

        # Import the IV
        %ivHash = $self->getEditHash_hashIV('logPrefHash');

        # Compile the simple list data
        foreach my $logfile ($axmud::CLIENT->constSessionLogOrderList) {

            push (@dataList,
                $logfile,
                $ivHash{$logfile},
                $axmud::CLIENT->ivShow('constLogDescripHash', $logfile),
            );
        }

        # Reset the simple list
        $self->resetListData($slWidget, [@dataList], $columns);

        return 1;
    }

    sub overrideTab {

        # Override tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->overrideTab', @_);
        }

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('_Override', $self->notebook);

        # Add tabs to the inner notebook
        $self->override1Tab($innerNotebook);
        $self->override2Tab($innerNotebook);
        $self->override3Tab($innerNotebook);
        $self->override4Tab($innerNotebook);
        $self->override5Tab($innerNotebook);

        return 1;
    }

    sub override1Tab {

        # Override1 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (@comboList, @comboList2);

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->override1Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _1', $innerNotebook);

        # Left column
        $self->addLabel($table, '<b>Override ' . $axmud::SCRIPT . ' terminal type</b>',
            0, 6, 0, 1);
        $self->addLabel($table, 'Own termtype',
            1, 3, 1, 2);
        @comboList = $axmud::CLIENT->constTermTypeList;
        $self->addComboBox($table, 'termType', \@comboList, '',
            FALSE,              # 'undef' value allowed
            3, 6, 1, 2);

        $self->addLabel($table, '<b>Override ' . $axmud::SCRIPT . ' character set</b>',
            0, 6, 2, 3);
        $self->addLabel($table, 'Own charset',
            1, 3, 3, 4);
        @comboList2 = $axmud::CLIENT->charSetList;
        $self->addComboBox($table, 'worldCharSet', \@comboList2, '',
            FALSE,              # 'undef' value allowed
            3, 6, 3, 4);

        # Right column
        $self->addLabel($table, '<b>Override real terminal size (disables NAWS)</b>',
            7, 13, 0, 1);
        $self->addLabel($table, 'Columns',
            8, 10, 1, 2);
        $self->addEntryWithIcon($table, 'columns', 'int', 0, undef,
            10, 13, 1, 2);
        $self->addLabel($table, 'Rows',
            8, 10, 2, 3);
        $self->addEntryWithIcon($table, 'rows', 'int', 0, undef,
            10, 13, 2, 3);
        $self->addLabel($table, 'Send size after login',
            8, 12, 3, 4);
        $self->addCheckButton($table, 'sendSizeInfoFlag', TRUE,
            12, 13, 3, 4, 1, 0.5);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub override2Tab {

        # Override2 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (@columnList, @comboList);

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->override2Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _2', $innerNotebook);

        # Telnet override settings
        $self->addLabel($table, '<b>Telnet override settings</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of telnet options/protocols which must not be enabled in a session when'
            . ' this is the current world</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Option/protocol', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 3, 10,
            -1, 230);       # Fixed height

        # Initialise the list
        $self->override2Tab_refreshList($slWidget, (scalar @columnList / 2));

        # Add editing widgets
        $self->addLabel($table, 'Option/protocol:',
            1, 3, 10, 11);

        @comboList = qw(
            echo sga ttype eor naws new_environ charset
            msdp mssp mccp msp mxp pueblo zmp aard102 atcp gmcp mtts mcp
        );

        my $comboBox = $self->addComboBox($table, undef, \@comboList, '',
            TRUE,                       # No 'undef' value used
            3, 5, 10, 11);

        my $button = $self->addButton(
            $table,
            'Disable for this world',
            'Stop this option/protocol from being enabled in this world\'s session',
            undef,
            5, 8, 10, 11);
        $button->signal_connect('clicked' => sub {

            my (
                $string,
                %hash,
            );

            $string = $comboBox->get_active_text();
            %hash = $self->getEditHash_hashIV('telnetOverrideHash');
            $hash{$string} = undef;
            $self->ivAdd('editHash', 'telnetOverrideHash', \%hash);

            # Refresh the simple list
            $self->override2Tab_refreshList($slWidget, (scalar @columnList / 2));
        });

        my $button2 = $self->addButton(
            $table,
            'Enable/disable according to global settings',
            'Enable/disable this option/protocol according to global settings',
            undef,
            8, 12, 10, 11);
        $button2->signal_connect('clicked' => sub {

            my (
                $string,
                %hash,
            );

            $string = $comboBox->get_active_text();
            %hash = $self->getEditHash_hashIV('telnetOverrideHash');
            delete $hash{$string};
            $self->ivAdd('editHash', 'telnetOverrideHash', \%hash);

            # Refresh the simple list
            $self->override2Tab_refreshList($slWidget, (scalar @columnList / 2));
        });

        my $button3 = $self->addButton(
            $table,
            'Reset',
            'Reset the list of override settings',
            undef,
            8, 10, 11, 12);
        $button3->signal_connect('clicked' => sub {

            # Remove this IV from $self->editHash, so that the IV in $self->editObj takes over
            $self->ivDelete('editHash', 'telnetOverrideHash');

            # Refresh the simple list
            $self->override2Tab_refreshList($slWidget, (scalar @columnList / 2));
        });

        my $button4 = $self->addButton(
            $table,
            'Clear',
            'Clear the list of override settings',
            undef,
            10, 12, 11, 12);
        $button4->signal_connect('clicked' => sub {

            # Add an empty hash to $self->editHash
            $self->ivAdd('editHash', 'telnetOverrideHash', {});

            # Refresh the simple list
            $self->override2Tab_refreshList($slWidget, (scalar @columnList / 2));
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub override2Tab_refreshList {

        # Resets the simple list displayed by $self->override2Tab
        #
        # Expected arguments
        #   $slWidget   - The GA::Obj::Simple::List
        #   $columns    - The number of columns
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $slWidget, $columns, $check) = @_;

        # Local variables
        my (
            @dataList,
            %ivHash,
        );

        # Check for improper arguments
        if (! defined $slWidget || ! defined $columns || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->override2Tab_refreshList',
                @_,
            );
        }

        # Import the IV
        %ivHash = $self->getEditHash_hashIV('telnetOverrideHash');

        # Compile the simple list data
        @dataList = sort {$a cmp $b} (keys %ivHash);

        # Reset the simple list
        $self->resetListData($slWidget, [@dataList], $columns);

        return 1;
    }

    sub override3Tab {

        # Override3 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (
            @columnList, @setupList, @comboList,
            %comboHash,
        );

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->override3Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _3', $innerNotebook);

        # MXP override settings
        $self->addLabel($table, '<b>MXP override settings</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of MXP settings which should be enabled/disabled when this is the current'
            . ' world, overriding global settings</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Setting type', 'text',
            'Enabled', 'bool',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 3, 10,
            -1, 230);       # Fixed height

        # Initialise the list
        $self->override3Tab_refreshList($slWidget, (scalar @columnList / 2));

        # Add editing widgets
        $self->addLabel($table, 'Setting type:',
            1, 3, 10, 11);

        @setupList = (
            'room'      => 'Use MXP room data',
        );

        do {

            my ($type, $descrip);

            $type = shift @setupList;
            $descrip = shift @setupList;

            push (@comboList, $descrip);
            $comboHash{$descrip} = $type;

        } until (! @setupList);

        my $comboBox = $self->addComboBox($table, undef, \@comboList, '',
            TRUE,                       # No 'undef' value used
            3, 5, 10, 11);

        my $button = $self->addButton(
            $table,
            'Enable for this world',
            'Enable this setting in this world\'s sessions',
            undef,
            5, 7, 10, 11);
        $button->signal_connect('clicked' => sub {

            my (
                $type,
                %hash,
            );

            $type = $comboHash{$comboBox->get_active_text()};
            %hash = $self->getEditHash_hashIV('mxpOverrideHash');
            $hash{$type} = TRUE;
            $self->ivAdd('editHash', 'mxpOverrideHash', \%hash);

            # Refresh the simple list
            $self->override3Tab_refreshList($slWidget, (scalar @columnList / 2));
        });

        my $button2 = $self->addButton(
            $table,
            'Disable for this world',
            'Disable this setting in this world\'s sessions',
            undef,
            7, 9, 10, 11);
        $button2->signal_connect('clicked' => sub {

            my (
                $type,
                %hash,
            );

            $type = $comboHash{$comboBox->get_active_text()};
            %hash = $self->getEditHash_hashIV('mxpOverrideHash');
            $hash{$type} = FALSE;
            $self->ivAdd('editHash', 'mxpOverrideHash', \%hash);

            # Refresh the simple list
            $self->override3Tab_refreshList($slWidget, (scalar @columnList / 2));
        });

        my $button3 = $self->addButton(
            $table,
            'Use global settings',
            'Enable/disable this setting according to global settings',
            undef,
            9, 12, 10, 11);
        $button3->signal_connect('clicked' => sub {

            my (
                $type,
                %hash,
            );

            $type = $comboHash{$comboBox->get_active_text()};
            %hash = $self->getEditHash_hashIV('mxpOverrideHash');
            delete $hash{$type};
            $self->ivAdd('editHash', 'mxpOverrideHash', \%hash);

            # Refresh the simple list
            $self->override3Tab_refreshList($slWidget, (scalar @columnList / 2));
        });

        my $button4 = $self->addButton(
            $table,
            'Reset',
            'Reset the list of override settings',
            undef,
            8, 10, 11, 12);
        $button4->signal_connect('clicked' => sub {

            # Remove this IV from $self->editHash, so that the IV in $self->editObj takes over
            $self->ivDelete('editHash', 'mxpOverrideHash');

            # Refresh the simple list
            $self->override3Tab_refreshList($slWidget, (scalar @columnList / 2));
        });

        my $button5 = $self->addButton(
            $table,
            'Clear',
            'Clear the list of override settings',
            undef,
            10, 12, 11, 12);
        $button5->signal_connect('clicked' => sub {

            # Add an empty hash to $self->editHash
            $self->ivAdd('editHash', 'mxpOverrideHash', {});

            # Refresh the simple list
            $self->override3Tab_refreshList($slWidget, (scalar @columnList / 2));
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub override3Tab_refreshList {

        # Resets the simple list displayed by $self->override3Tab
        #
        # Expected arguments
        #   $slWidget   - The GA::Obj::Simple::List
        #   $columns    - The number of columns
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $slWidget, $columns, $check) = @_;

        # Local variables
        my (
            @dataList,
            %ivHash,
        );

        # Check for improper arguments
        if (! defined $slWidget || ! defined $columns || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->override3Tab_refreshList',
                @_,
            );
        }

        # Import the IV
        %ivHash = $self->getEditHash_hashIV('mxpOverrideHash');

        # Compile the simple list data
        foreach my $type ('room') {

            my $iv;

            if (exists $ivHash{$type}) {

                push (@dataList, $type, $ivHash{$type});
            }
        }

        # Reset the simple list
        $self->resetListData($slWidget, [@dataList], $columns);

        return 1;
    }

    sub override4Tab {

        # Override4 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (
            @columnList, @initList, @comboList, @initList2, @comboList2, @comboList3,
            %comboHash, %comboHash2,
        );

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->override4Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _4', $innerNotebook);

        # Terminal type override settings
        $self->addLabel($table, '<b>Terminal type override settings</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of terminal type override settings which must be applied when this is the'
            . ' current world, overriding global settings</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Setting', 'text',
            'Value (1 - enabled)', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 3, 7,
            -1, 140);       # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, (scalar @columnList / 2), 'termOverrideHash');

        # Add editing widgets
        $self->addLabel($table, 'TTYPE negotiations',
            1, 3, 7, 8);

        @initList = (
            'send_nothing'          => 'Send nothing',
            'send_client'           => 'Send client name, then usual list',
            'send_client_version'   => 'Send client/name version, then usual list',
            'send_custom_client'    => 'Send custom client/name version, then usual list',
            'send_default'          => 'Send usual terminal type list',
            'send_unknown'          => 'Send \'unknown\'',
        );

        do {

            my ($mode, $descrip);

            $mode = shift @initList;
            $descrip = shift @initList;

            push (@comboList, $descrip);
            $comboHash{$descrip} = $mode;

        } until (! @initList);

        my $combo = $self->addComboBox($table, undef, \@comboList, '',
            TRUE,                       # No 'undef' value used
            3, 8, 7, 8);

        my $button = $self->addButton($table, 'Use', 'Use this setting in the session', undef,
            8, 10, 7, 8);
        $button->signal_connect('clicked' => sub {

            my (
                $mode,
                %hash,
            );

            $mode = $comboHash{$combo->get_active_text()};
            %hash = $self->getEditHash_hashIV('termOverrideHash');
            $hash{termTypeMode} = $mode;
            $self->ivAdd('editHash', 'termOverrideHash', \%hash);

            # Reset the combobox
            $combo->set_active(0);
            # Refresh the simple list
            $self->refreshList_hashIV($slWidget, (scalar @columnList / 2), 'termOverrideHash');
        });

        my $button2 = $self->addButton(
            $table,
            'Don\'t use',
            'Don\'t use this setting in the session',
            undef,
            10, 12, 7, 8);
        $button2->signal_connect('clicked' => sub {

            my %hash = $self->getEditHash_hashIV('termOverrideHash');
            delete $hash{'termTypeMode'};
            $self->ivAdd('editHash', 'termOverrideHash', \%hash);

            # Reset the combobox
            $combo->set_active(0);
            # Refresh the simple list
            $self->refreshList_hashIV($slWidget, (scalar @columnList / 2), 'termOverrideHash');
        });

        $self->addLabel($table, 'Custom client name',
            1, 3, 8, 9);
        my $entry = $self->addEntry($table, undef, TRUE,
            3, 8, 8, 9);

        my $button3 = $self->addButton($table, 'Use', 'Use this setting in the session', undef,
            8, 10, 8, 9);
        $button3->signal_connect('clicked' => sub {

            my (
                $string,
                %hash,
            );

            $string = $entry->get_text();
            if (! defined $string) {

                $string = '';
            }

            %hash = $self->getEditHash_hashIV('termOverrideHash');
            $hash{'customClientName'} = $string;
            $self->ivAdd('editHash', 'termOverrideHash', \%hash);

            # Reset the entry box
            $entry->set_text('');
            # Refresh the simple list
            $self->refreshList_hashIV($slWidget, (scalar @columnList / 2), 'termOverrideHash');
        });

        my $button4 = $self->addButton(
            $table,
            'Don\'t use',
            'Don\'t use this setting in the session',
            undef,
            10, 12, 8, 9);
        $button4->signal_connect('clicked' => sub {

            my %hash = $self->getEditHash_hashIV('termOverrideHash');
            delete $hash{'customClientName'};
            $self->ivAdd('editHash', 'termOverrideHash', \%hash);

            # Reset the entry box
            $entry->set_text('');
            # Refresh the simple list
            $self->refreshList_hashIV($slWidget, (scalar @columnList / 2), 'termOverrideHash');
        });

        $self->addLabel($table, 'Custom client version',
            1, 3, 9, 10);
        my $entry2 = $self->addEntry($table, undef, TRUE,
            3, 8, 9, 10);

        my $button5 = $self->addButton($table, 'Use', 'Use this setting in the session', undef,
            8, 10, 9, 10);
        $button5->signal_connect('clicked' => sub {

            my (
                $string,
                %hash,
            );

            $string = $entry2->get_text();
            if (! defined $string) {

                $string = '';
            }

            %hash = $self->getEditHash_hashIV('termOverrideHash');
            $hash{'customClientVersion'} = $string;
            $self->ivAdd('editHash', 'termOverrideHash', \%hash);

            # Reset the entry box
            $entry2->set_text('');
            # Refresh the simple list
            $self->refreshList_hashIV($slWidget, (scalar @columnList / 2), 'termOverrideHash');
        });

        my $button6 = $self->addButton(
            $table,
            'Don\'t use',
            'Don\'t use this setting in the session',
            undef,
            10, 12, 9, 10);
        $button6->signal_connect('clicked' => sub {

            my %hash = $self->getEditHash_hashIV('termOverrideHash');
            delete $hash{'customClientVersion'};
            $self->ivAdd('editHash', 'termOverrideHash', \%hash);

            # Reset the entry box
            $entry2->set_text('');
            # Refresh the simple list
            $self->refreshList_hashIV($slWidget, (scalar @columnList / 2), 'termOverrideHash');
        });

        @initList2 = (
            'useCtrlSeqFlag'          => 'Use VT100 control sequences',
            'useVisibleCursorFlag'    => 'Show visible cursor in default textview',
            'useDirectKeysFlag'       => 'Use direct keyboard input in terminal',
        );

        do {

            my ($iv, $descrip);

            $iv = shift @initList2;
            $descrip = shift @initList2;

            push (@comboList2, $descrip);
            $comboHash2{$descrip} = $iv;

        } until (! @initList2);

        $self->addLabel($table, 'Terminal emulation',
            1, 3, 10, 11);
        my $combo2 = $self->addComboBox($table, undef, \@comboList2, '',
            TRUE,                       # No 'undef' value used
            3, 6, 10, 11);

        @comboList3 = ('Enabled', 'Disabled');
        my $combo3 = $self->addComboBox($table, undef, \@comboList3, '',
            TRUE,                       # No 'undef' value used
            6, 8, 10, 11);

        my $button7 = $self->addButton($table, 'Use', 'Use this setting in the session', undef,
            8, 10, 10, 11);
        $button7->signal_connect('clicked' => sub {

            my (
                $iv, $mode, $flag,
                %hash,
            );

            $iv = $comboHash2{$combo2->get_active_text()};
            $mode = $combo3->get_active_text();
            if ($mode eq 'Enabled') {
                $flag = TRUE;
            } else {
                $flag = FALSE;
            }

            %hash = $self->getEditHash_hashIV('termOverrideHash');
            $hash{$iv} = $flag;
            $self->ivAdd('editHash', 'termOverrideHash', \%hash);

            # Reset the comboboxes
            $combo2->set_active(0);
            $combo3->set_active(0);
            # Refresh the simple list
            $self->refreshList_hashIV($slWidget, (scalar @columnList / 2), 'termOverrideHash');
        });

        my $button8 = $self->addButton(
            $table,
            'Don\'t use',
            'Don\'t use this setting in the session',
            undef,
            10, 12, 10, 11);
        $button8->signal_connect('clicked' => sub {

            my (
                $iv,
                %hash,
            );

            $iv = $comboHash2{$combo2->get_active_text()};

            %hash = $self->getEditHash_hashIV('termOverrideHash');
            delete $hash{$iv};
            $self->ivAdd('editHash', 'termOverrideHash', \%hash);

            # Reset the comboboxes
            $combo2->set_active(0);
            $combo3->set_active(0);
            # Refresh the simple list
            $self->refreshList_hashIV($slWidget, (scalar @columnList / 2), 'termOverrideHash');
        });

        my $button9 = $self->addButton(
            $table,
            'Reset',
            'Reset the list of override settings',
            undef,
            8, 10, 11, 12);
        $button9->signal_connect('clicked' => sub {

            # Remove this IV from $self->editHash, so that the IV in $self->editObj takes over
            $self->ivDelete('editHash', 'termOverrideHash');

            # Reset widgets
            $combo->set_active(0);
            $entry->set_text('');
            $entry2->set_text('');
            $combo2->set_active(0);
            $combo3->set_active(0);
            # Refresh the simple list
            $self->refreshList_hashIV($slWidget, (scalar @columnList / 2), 'termOverrideHash');
        });

        my $button10 = $self->addButton(
            $table,
            'Clear',
            'Clear the list of override settings',
            undef,
            10, 12, 11, 12);
        $button10->signal_connect('clicked' => sub {

            # Add an empty hash to $self->editHash
            $self->ivAdd('editHash', 'termOverrideHash', {});

            # Reset widgets
            $combo->set_active(0);
            $entry->set_text('');
            $entry2->set_text('');
            $combo2->set_active(0);
            $combo3->set_active(0);
            # Refresh the simple list
            $self->refreshList_hashIV($slWidget, (scalar @columnList / 2), 'termOverrideHash');
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub override5Tab {

        # Override5 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (
            @columnList, @sigilList, @comboList,
            %comboHash,
        );

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->override5Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _5', $innerNotebook);

        # Instruction override settings
        $self->addLabel($table, '<b>Instruction sigil override settings</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of instruction sigils which should be enabled/disabled when this is the'
            . ' current world, overriding global settings</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Instruction type', 'text',
            'Enabled', 'bool',
            'Sigil', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 3, 10,
            -1, 230);       # Fixed height

        # Initialise the list
        $self->override5Tab_refreshList($slWidget, (scalar @columnList / 2));

        # Add editing widgets
        $self->addLabel($table, 'Instruction type:',
            1, 3, 10, 11);


        @sigilList = qw(echo perl script multi speed bypass);
        foreach my $item (@sigilList) {

            my $string;

            if ($item eq 'speed') {
                $string = ucfirst($item) . 'walk commands';
            } else {
                $string = ucfirst($item) . ' commands';
            }

            push (@comboList, $string);
            $comboHash{$string} = $item;
        }

        my $comboBox = $self->addComboBox($table, undef, \@comboList, '',
            TRUE,                       # No 'undef' value used
            3, 5, 10, 11);

        my $button = $self->addButton(
            $table,
            'Enable for this world',
            'Enable the sigil in this world\'s sessions',
            undef,
            5, 7, 10, 11);
        $button->signal_connect('clicked' => sub {

            my (
                $string,
                %hash,
            );

            $string = $comboHash{$comboBox->get_active_text()};
            %hash = $self->getEditHash_hashIV('sigilOverrideHash');
            $hash{$string} = TRUE;
            $self->ivAdd('editHash', 'sigilOverrideHash', \%hash);

            # Refresh the simple list
            $self->override5Tab_refreshList($slWidget, (scalar @columnList / 2));
        });

        my $button2 = $self->addButton(
            $table,
            'Disable for this world',
            'Disable the sigil in this world\'s sessions',
            undef,
            7, 9, 10, 11);
        $button2->signal_connect('clicked' => sub {

            my (
                $string,
                %hash,
            );

            $string = $comboHash{$comboBox->get_active_text()};
            %hash = $self->getEditHash_hashIV('sigilOverrideHash');
            $hash{$string} = FALSE;
            $self->ivAdd('editHash', 'sigilOverrideHash', \%hash);

            # Refresh the simple list
            $self->override5Tab_refreshList($slWidget, (scalar @columnList / 2));
        });

        my $button3 = $self->addButton(
            $table,
            'Use global settings',
            'Enable/disable this sigil according to global settings',
            undef,
            9, 12, 10, 11);
        $button3->signal_connect('clicked' => sub {

            my (
                $string,
                %hash,
            );

            $string = $comboHash{$comboBox->get_active_text()};
            %hash = $self->getEditHash_hashIV('sigilOverrideHash');
            delete $hash{$string};
            $self->ivAdd('editHash', 'sigilOverrideHash', \%hash);

            # Refresh the simple list
            $self->override5Tab_refreshList($slWidget, (scalar @columnList / 2));
        });

        my $button4 = $self->addButton(
            $table,
            'Reset',
            'Reset the list of override settings',
            undef,
            8, 10, 11, 12);
        $button4->signal_connect('clicked' => sub {

            # Remove this IV from $self->editHash, so that the IV in $self->editObj takes over
            $self->ivDelete('editHash', 'sigilOverrideHash');

            # Refresh the simple list
            $self->override5Tab_refreshList($slWidget, (scalar @columnList / 2));
        });

        my $button5 = $self->addButton(
            $table,
            'Clear',
            'Clear the list of override settings',
            undef,
            10, 12, 11, 12);
        $button5->signal_connect('clicked' => sub {

            # Add an empty hash to $self->editHash
            $self->ivAdd('editHash', 'sigilOverrideHash', {});

            # Refresh the simple list
            $self->override5Tab_refreshList($slWidget, (scalar @columnList / 2));
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub override5Tab_refreshList {

        # Resets the simple list displayed by $self->override5Tab
        #
        # Expected arguments
        #   $slWidget   - The GA::Obj::Simple::List
        #   $columns    - The number of columns
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $slWidget, $columns, $check) = @_;

        # Local variables
        my (
            @dataList,
            %ivHash,
        );

        # Check for improper arguments
        if (! defined $slWidget || ! defined $columns || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->override5Tab_refreshList',
                @_,
            );
        }

        # Import the IV
        %ivHash = $self->getEditHash_hashIV('sigilOverrideHash');

        # Compile the simple list data
        foreach my $sigil ('echo', 'perl', 'script', 'multi', 'speed', 'bypass') {

            my $iv;

            if (exists $ivHash{$sigil}) {

                $iv = 'const'. ucfirst($sigil) . 'Sigil';       # e.g. constEchoSigil
                push (@dataList, $sigil, $ivHash{$sigil}, $axmud::CLIENT->$iv);
            }
        }

        # Reset the simple list
        $self->resetListData($slWidget, [@dataList], $columns);

        return 1;
    }

    sub roomsTab {

        # Verbose tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->roomsTab', @_);
        }

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('_Rooms', $self->notebook);

        # Add tabs to the inner notebook
        $self->rooms1Tab($innerNotebook);
        $self->rooms2Tab($innerNotebook, 'verboseComponentList');
        $self->rooms3Tab($innerNotebook, 'verboseAnchorPatternList', 'verboseAnchorOffset');
        $self->rooms4Tab($innerNotebook, 'verboseAnchorCheckList');
        $self->rooms2Tab($innerNotebook, 'shortComponentList');
        $self->rooms3Tab($innerNotebook, 'shortAnchorPatternList', 'shortAnchorOffset');
        $self->rooms4Tab($innerNotebook, 'shortAnchorCheckList');
        $self->rooms2Tab($innerNotebook, 'briefComponentList');
        $self->rooms3Tab($innerNotebook, 'briefAnchorPatternList', 'briefAnchorOffset');
        $self->rooms4Tab($innerNotebook, 'briefAnchorCheckList');
        $self->rooms11Tab($innerNotebook);
        $self->rooms12Tab($innerNotebook);
        $self->rooms13Tab($innerNotebook);
        $self->rooms14Tab($innerNotebook);
        $self->rooms15Tab($innerNotebook);
        $self->rooms16Tab($innerNotebook);
        $self->rooms17Tab($innerNotebook);
        $self->rooms18Tab($innerNotebook);
        $self->rooms19Tab($innerNotebook);
        $self->rooms20Tab($innerNotebook);
        $self->rooms21Tab($innerNotebook);

        return 1;
    }

    sub rooms1Tab {

        # Rooms1 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (@columnList, @comboList);

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->rooms1Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _1', $innerNotebook);

        # Room statement components
        $self->addLabel($table, '<b>Room statement components</b>',
            0, 12, 0, 1);
        $self->addLabel(
            $table, '<i>List of the components that make up room statements in this world</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Component name', 'text',
            'Type', 'text',
            'Fixed size', 'int',
            'Min. size', 'int',
            'Max. size', 'int',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 230);      # Fixed height

        # Initialise the list
        $self->rooms1Tab_refreshList($slWidget, (scalar @columnList / 2));

        # Add entries/comboboxes for adding new components
        $self->addLabel($table, 'Type:',
            1, 2, 8, 9);
        @comboList = $axmud::CLIENT->constComponentTypeList;
        # The first item in the list is 'anchor', which is a special kind of component which doesn't
        #   have a corresponding GA::Obj::Component; remove it from the list
        shift @comboList;
        my $comboBox = $self->addComboBox($table, undef, \@comboList, '',
            TRUE,           # No 'undef' value used
            2, 6, 8, 9);
        # ->signal_connect appears below...

        $self->addLabel($table, 'Name:',
            7, 8, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, 16,
            8, 12, 8, 9);

        $comboBox->signal_connect('changed' => sub {

            my (
                $type, $name,
                %compHash,
            );

            $type = $comboBox->get_active_text();

            # Suggest a name that isn't already taken, and set the $entry to display it
            # First import the hash IV
            %compHash = $self->getEditHash_hashIV('componentHash');
            if (! exists $compHash{$type}) {

                # $type is available as a name, so use the type as the name
                $name = $type;

            } else {

                # Try some alternative names, e.g. 'verb_descrip_1'
                OUTER: for (my $count = 1; $count < 100; $count++) {

                    my $suggestion = $type . '_' . $count;

                    if (! exists $compHash{$suggestion}) {

                        $name = $suggestion;
                        last OUTER;
                    }
                }
            }

            if (defined $name) {

                $entry->set_text($name);
            }
        });

        # Add standard editing buttons to the simple list
        my $button = $self->addButton($table, 'Add...', 'Add a new room statement component', undef,
            1, 3, 9, 10);
        $button->signal_connect('clicked' => sub {

            my (
                $type, $name, $compObj,
                @comboList,
                %ivHash,
            );

            if ($self->checkEntryIcon($entry)) {

                $type = $comboBox->get_active_text();
                $name = $entry->get_text();

                if ($name eq 'anchor') {

                    $self->showMsgDialogue(
                        'Add component',
                        'warning',
                        'The \'anchor\' component always exists, so it cannot be added to this'
                        . ' list',
                        'ok',
                    );

                } else {

                    # Create a new window component object
                    $compObj = Games::Axmud::Obj::Component->new(
                        $self->session,
                        $self->editObj,
                        $name,
                        $type,
                    );

                    if ($compObj) {

                        # Component object created. Update the IV (don't use ->modifyEditHash_hashIV
                        #   before we need to update the combos, in a moment)
                        %ivHash = $self->getEditHash_hashIV('componentHash');
                        $ivHash{$compObj->name} = $compObj;
                        $self->ivAdd('editHash', 'componentHash', \%ivHash);

                        # Refresh the simple list and reset entry boxes
                        $self->rooms1Tab_refreshList($slWidget, (scalar @columnList / 2));
                        $self->resetEntryBoxes($entry);

                        # Refresh the combo boxes in pages 2, 4 and 6
                        $self->rooms1Tab_refreshCombos(keys %ivHash);
                    }
                }
            }
        });

        my $button2 = $self->addButton($table, 'Edit...', 'Edit the selected component', undef,
            6, 9, 9, 10);
        $button2->signal_connect('clicked' => sub {

            my (
                $name, $compObj, $childWinObj,
                %ivHash,
            );

            ($name) = $self->getSimpleListData($slWidget, 0);
            if (defined $name) {

                if ($name eq 'anchor') {

                    # (The 'anchor' component shouldn't appear in this tab's simple list, but just
                    #   in case...)
                    $self->showMsgDialogue(
                        'Edit component',
                        'warning',
                        'The \'anchor\' component can\'t be edited',
                        'ok',
                    );

                } else {

                    # Get the corresponding GA::Obj::Component object
                    %ivHash = $self->getEditHash_hashIV('componentHash');
                    $compObj = $ivHash{$name};

                    if ($compObj) {

                        # Open an 'edit' window for the selected component
                        $childWinObj = $self->createFreeWin(
                            'Games::Axmud::EditWin::Component',
                            $self,
                            $self->session,
                            'Edit room component \'' . $name . '\'',
                            $compObj,
                            FALSE,          # Not temporary
                        );
                    }

                    if ($childWinObj) {

                        # When the 'edit' window closes, update widgets and/or IVs
                        $self->add_childDestroy(
                            $childWinObj,
                            'rooms1Tab_refreshList',
                            [$slWidget, (scalar @columnList / 2)],
                        );
                    }
                }
            }
        });

        my $button3 = $self->addButton(
            $table,
            'Dump',
            'Display the selected component in the \'main\' window',
            undef,
            9, 10, 9, 10);
        $button3->signal_connect('clicked' => sub {

            my ($name) = $self->getSimpleListData($slWidget, 0);
            if (defined $name) {

                $self->session->pseudoCmd('listroomcomponent ' . $name, $self->pseudoCmdMode);

                # Refresh the simple list and reset entry boxes
                $self->rooms1Tab_refreshList($slWidget, (scalar @columnList / 2));
                $self->resetEntryBoxes($entry);

            }
        });

        my $button4 = $self->addButton($table, 'Delete', 'Delete the selected component', undef,
            10, 11, 9, 10);
        $button4->signal_connect('clicked' => sub {

            my (
                $name, $hashRef,
                @ivList,
                %ivHash,
            );

            ($name) = $self->getSimpleListData($slWidget, 0);
            if (defined $name) {

                # Update the IV
                %ivHash = $self->getEditHash_hashIV('componentHash');
                delete $ivHash{$name};
                $self->ivAdd('editHash', 'componentHash', \%ivHash);

                # Refresh the simple list and reset entry boxes
                $self->rooms1Tab_refreshList($slWidget, (scalar @columnList / 2));
                $self->resetEntryBoxes($entry);

                # Remove the component from the world's ->verboseComponentList, ->shortComponentList
                #    and ->briefComponentList, if it exists there
                $self->rooms1Tab_updateWorld($name);
                # Refresh the combo boxes in pages 2, 4 and 6
                $self->rooms1Tab_refreshCombos(keys %ivHash);
                # Update the simple lists in pages 2, 4 and 6
                @ivList = ('verboseComponentList', 'shortComponentList', 'briefComponentList');
                foreach my $thisIv (@ivList) {

                    my $thisSimpleList = $self->ivShow('simpleListHash', $thisIv);
                    $self->rooms2Tab_refreshList($thisSimpleList, 2, $thisIv)
                }
            }
        });

        my $button5 = $self->addButton($table, 'Reset', 'Reset the list of components', undef,
            11, 12, 9, 10);
        $button5->signal_connect('clicked' => sub {

            my %ivHash;

            # Remove this IV from $self->editHash, so that the IV in $self->editObj takes over
            $self->ivDelete('editHash', 'componentHash');

            # Refresh the simple list and reset entry boxes
            $self->rooms1Tab_refreshList($slWidget, (scalar @columnList / 2));
            $self->resetEntryBoxes($entry);

            # Refresh the combo boxes in pages 2, 4 and 6
            %ivHash = $self->getEditHash_hashIV('componentHash');

            $self->rooms1Tab_refreshCombos(keys %ivHash);
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub rooms1Tab_refreshList {

        # Resets the simple list displayed by $self->rooms1Tab
        #
        # Expected arguments
        #   $slWidget   - The GA::Obj::Simple::List
        #   $columns    - The number of columns
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $slWidget, $columns, $check) = @_;

        # Local variables
        my (
            $hashRef,
            @compList, @list, @sortedList, @dataList,
            %ivHash,
        );

        # Check for improper arguments
        if (! defined $slWidget || ! defined $columns || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->rooms1Tab_refreshList',
                @_,
            );
        }

        # Import the IV
        %ivHash = $self->getEditHash_hashIV('componentHash');
        @compList = sort {lc($a->name) cmp lc($b->name)} (values %ivHash);

        # Sort the list of components, using the standard order supplied by GA::Profile::World
        OUTER: foreach my $type ($axmud::CLIENT->constComponentTypeList) {

            @list = @compList;
            @compList = ();

            INNER: foreach my $compObj (@list) {

                if ($compObj->type eq $type) {

                    push (@sortedList, $compObj);

                } else {

                    # Match this component against a different type in the next iteration of the
                    #   OUTER loop
                    push (@compList, $compObj);
                }
            }
        }

        # Add any stragglers (if there should be any...)
        if (@compList) {

            push (@sortedList, @compList);
        }

        # Compile the simple list data
        foreach my $compObj (@sortedList) {

            push (@dataList,
                $compObj->name,
                $compObj->type,
                $compObj->size,
                $compObj->minSize,
                $compObj->maxSize,
            );
        }

        # Reset the simple list
        $self->resetListData($slWidget, [@dataList], $columns);

        return 1;
    }

    sub rooms1Tab_updateWorld {

        # Updates the world's list of room statement components, when one of the components is
        #   removed from ->componentHash
        #
        # Expected arguments
        #   $name   - The name of the component to remove (e.g. 'verb_descrip_1')
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $name, $check) = @_;

        # Local variables
        my @ivList;

        # Check for improper arguments
        if (! defined $name || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->rooms1Tab_updateWorld', @_);
        }

        # The deleted component might still be in any (or, conceivably all) of these IVs
        @ivList = ('verboseComponentList', 'shortComponentList', 'briefComponentList');
        foreach my $iv (@ivList) {

            my (@compList, @modList);

            # Import the IV...
            @compList = $self->getEditHash_listIV($iv);
            foreach my $component (@compList) {

                if ($name ne $component) {

                    push (@modList, $component);
                }
            }

            # ...and store it again
            $self->ivAdd('editHash', $iv, \@modList);
        }

        return 1;
    }

    sub rooms1Tab_refreshCombos {

        # Resets the comboboxes stored in $self->comboHash
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Optional arguments
        #   @comboList  - A list of items to store in each combobox, once it has been emptied (can
        #       be an empty list)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, @comboList) = @_;

        # Local variables
        my @sortedList;

        # (No improper arguments to check)

        # Sort the list of components
        @sortedList = sort {lc($a) cmp lc($b)} (@comboList);

        # Update each combobox in turn
        foreach my $combo ($self->ivValues('comboHash')) {

            $self->resetComboBox($combo, @sortedList);
        }

        return 1;
    }

    sub rooms2Tab {

        # Rooms2 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #   $iv             - The IV to edit - one of 'verboseComponentList', 'shortComponentList'
        #                       and 'briefComponentList'
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $iv, $check) = @_;

        # Local variables
        my (@columnList, @comboList);

        # Check for improper arguments
        if (! defined $innerNotebook || ! defined $iv || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->rooms2Tab', @_);
        }

        # Tab setup
        my ($vBox, $table);

        if ($iv eq 'verboseComponentList') {

            ($vBox, $table) = $self->addTab('Page _2', $innerNotebook);

            # Verbose components
            $self->addLabel($table, '<b>Verbose components</b>',
                0, 12, 0, 1);
            $self->addLabel($table,
                '<i>List of components found in verbose room statements</i>',
                1, 12, 1, 2);

        } elsif ($iv eq 'shortComponentList') {

            ($vBox, $table) = $self->addTab('Page _5', $innerNotebook);

            # Short verbose components
            $self->addLabel($table, '<b>Short verbose components</b>',
                0, 12, 0, 1);
            $self->addLabel($table,
                '<i>List of components found in short verbose room statements</i>',
                1, 12, 1, 2);

        } else {

            ($vBox, $table) = $self->addTab('Page _8', $innerNotebook);

            # Short verbose components
            $self->addLabel($table, '<b>Brief components</b>',
                0, 12, 0, 1);
            $self->addLabel($table,
                '<i>List of components found in brief room statements</i>',
                1, 12, 1, 2);
        }

        # Add a simple list
        @columnList = (
            'Component name', 'text',
            'Type', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 230);     # Fixed height

        # Initialise the simple list
        $self->rooms2Tab_refreshList($slWidget, scalar (@columnList / 2), $iv);

        # Add editing combos and buttons
        @comboList = sort {lc($a) cmp lc($b)} ($self->editObj->ivKeys('componentHash'));
        push(@comboList, 'anchor');
        my $combo = $self->addComboBox($table, undef, \@comboList, '',
            TRUE,           # No 'undef' value used
            1, 3, 10, 11);

        # Unusual step - store the combo in an IV, so that $self->rooms1Tab can change the list
        #    reference in an IV
        $self->ivAdd('comboHash', $iv, $combo);
        # Also need to store the simple list for the same reason
        $self->ivAdd('simpleListHash', $iv, $slWidget);

        my $button = $self->addButton(
            $table, 'Add to list', 'Add the selected component to the list', undef,
            3, 5, 10, 11);
        $button->signal_connect('clicked' => sub {

            my (
                $compName, $matchFlag,
                @ivList,
            );

            $compName = $combo->get_active_text();
            if ($compName) {

                # Update the IV
                @ivList = $self->getEditHash_listIV($iv);
                # (If the list already contains an anchor, don't add another one)
                if ($compName eq 'anchor') {

                    OUTER: foreach my $item (@ivList) {

                        if ($item eq 'anchor') {

                            $matchFlag = TRUE;
                            last OUTER;
                        }
                    }

                    if ($matchFlag) {

                        $self->showMsgDialogue(
                            'Add component',
                            'warning',
                            'The list may contain only one \'anchor\' component',
                            'ok',
                        );

                    } else {

                        push (@ivList, $compName);
                        $self->ivAdd('editHash', $iv, \@ivList);
                    }

                } else {

                    push (@ivList, $compName);
                    $self->ivAdd('editHash', $iv, \@ivList);
                }

                # Refresh the simple list
                $self->rooms2Tab_refreshList($slWidget, scalar (@columnList / 2), $iv);
            }
        });

        my $button2 = $self->addButton($table, 'Edit...', 'Edit the selected component', undef,
            6, 8, 10, 11);
        $button2->signal_connect('clicked' => sub {

            my (
                $name, $compObj, $childWinObj,
                %ivHash,
            );

            ($name) = $self->getSimpleListData($slWidget, 0);
            if (defined $name) {

                if ($name eq 'anchor') {

                    $self->showMsgDialogue(
                        'Edit component',
                        'warning',
                        'The \'anchor\' component can\'t be edited',
                        'ok',
                    );

                } else {

                    # Get the corresponding GA::Obj::Component object
                    %ivHash = $self->getEditHash_hashIV('componentHash');
                    $compObj = $ivHash{$name};

                    if ($compObj) {

                        # Open an 'edit' window for the selected component
                        $childWinObj = $self->createFreeWin(
                            'Games::Axmud::EditWin::Component',
                            $self,
                            $self->session,
                            'Edit room component \'' . $name . '\'',
                            $compObj,
                            FALSE,          # Not temporary
                        );
                    }

                    if ($childWinObj) {

                        # When the 'edit' window closes, update widgets and/or IVs
                        $self->add_childDestroy(
                            $childWinObj,
                            'rooms2Tab_refreshList',
                            [$slWidget, (scalar @columnList / 2), $iv],
                        );
                    }
                }
            }
        });

        my $button3 = $self->addButton(
            $table,
            'Dump',
            'Display the selected component in the \'main\' window',
            undef,
            8, 10, 10, 11);
        $button3->signal_connect('clicked' => sub {

            my ($name) = $self->getSimpleListData($slWidget, 0);
            if (defined $name) {

                $self->session->pseudoCmd('listroomcomponent ' . $name, $self->pseudoCmdMode);

                # Refresh the simple list
                $self->rooms2Tab_refreshList($slWidget, scalar (@columnList / 2), $iv);
            }
        });

        my $button4 = $self->addButton($table,
            'Remove from list',
            'Remove the selected component from this list (but don\'t delete it)',
            undef,
            10, 12, 10, 11);
        $button4->signal_connect('clicked' => sub {

            my (
                $rowNum,
                @ivList,
            );

            # Get the selected row
            ($rowNum) = $slWidget->get_selected_indices();
            if (defined $rowNum) {

                # Update the IV
                @ivList = $self->getEditHash_listIV($iv);

                splice (@ivList, $rowNum, 1);
                $self->ivAdd('editHash', $iv, \@ivList);

                # Refresh the simple list
                $self->rooms2Tab_refreshList($slWidget, scalar (@columnList / 2), $iv);
            }
        });

        my $button5 = $self->addButton($table,
            'Move up', 'Move the selected component up the list', undef,
            1, 3, 11, 12);
        $button5->signal_connect('clicked' => sub {

            my (
                $rowNum, $compName,
                @ivList,
            );

            # Get the selected row
            ($rowNum) = $slWidget->get_selected_indices();
            # (Don't move a component already at the top of the list...)
            if (defined $rowNum && $rowNum > 0) {

                # Update the IV
                @ivList = $self->getEditHash_listIV($iv);

                $compName = splice (@ivList, $rowNum, 1);
                splice (@ivList, ($rowNum - 1), 0, $compName);
                $self->ivAdd('editHash', $iv, \@ivList);

                # Refresh the simple list
                $self->rooms2Tab_refreshList($slWidget, scalar (@columnList / 2), $iv);

                # The component should still be highlighted, after being moved up
                $slWidget->select($rowNum - 1);
            }
        });

        my $button6 = $self->addButton($table,
            'Move down', 'Move the selected component down the list', undef,
            10, 12, 11, 12);
        $button6->signal_connect('clicked' => sub {

            my (
                $rowNum, $listRef, $compName,
                @ivList,
            );

            # Get the selected row
            ($rowNum) = $slWidget->get_selected_indices();
            if (defined $rowNum) {

                # Update the IV
                @ivList = $self->getEditHash_listIV($iv);

                # Don't move a component that is already at the bottom of the list
                if ($rowNum < (scalar @ivList - 1)) {

                    $compName = splice (@ivList, $rowNum, 1);
                    splice (@ivList, ($rowNum + 1), 0, $compName);
                    $self->ivAdd('editHash', $iv, \@ivList);

                    # Refresh the simple list
                    $self->rooms2Tab_refreshList($slWidget, scalar (@columnList / 2), $iv);

                    # The component should still be highlighted, after being moved up
                    $slWidget->select($rowNum + 1);
                }
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub rooms2Tab_refreshList {

        # Resets the simple list displayed by $self->rooms2Tab
        #
        # Expected arguments
        #   $slWidget   - The GA::Obj::Simple::List
        #   $columns    - The number of columns
        #   $iv         - The IV to display - one of 'verboseComponentList', 'shortComponentList'
        #                   and 'briefComponentList'
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $slWidget, $columns, $iv, $check) = @_;

        # Local variables
        my (
            $slWidgetRef,
            @compList, @dataList,
            %compHash,
        );

        # Check for improper arguments
        if (! defined $slWidget || ! defined $columns || ! defined $iv || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->rooms2Tab_refreshList',
                @_,
            );
        }

        # Import the IV
        @compList = $self->getEditHash_listIV($iv);
        # The component object itself may have been created (or modified) since the 'edit' window
        #   was opened
        %compHash = $self->getEditHash_hashIV('componentHash');

        # Compile the simple list data
        foreach my $compName (@compList) {

            my $compObj;

            # 'anchor' is a special type of room statement component that doesn't have its own
            #   component object
            if ($compName eq 'anchor') {

                push (@dataList, 'anchor', 'anchor');

            } else {

                $compObj = $compHash{$compName};
                push (@dataList,
                    $compObj->name,
                    $compObj->type,
                );
            }
        }

        # Reset the simple list
        $self->resetListData($slWidget, [@dataList], $columns);

        return 1;
    }

    sub rooms3Tab {

        # Rooms1 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #   $listIV         - The list IV to edit - one of 'verboseAnchorPatternList',
        #                       'shortAnchorPatternList' and 'briefAnchorPatternList'
        #   $offsetIV       - The accompanying scalar IV to edit - one of 'verboseAnchorOffset',
        #                       'shortAnchorOffset' and 'briefAnchorOffset'
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $listIV, $offsetIV, $check) = @_;

        # Check for improper arguments
        if (
            ! defined $innerNotebook || ! defined $listIV || ! defined $offsetIV
            || defined $check
        ) {
            return $axmud::CLIENT->writeImproper($self->_objClass . '->rooms3Tab', @_);
        }

        # Tab setup
        my ($vBox, $table);

        if ($listIV eq 'verboseAnchorPatternList') {

            ($vBox, $table) = $self->addTab('Page _3', $innerNotebook);

            # Verbose components
            $self->addLabel($table, '<b>Verbose anchor patterns</b>',
                0, 10, 0, 1);
            $self->addLabel($table,
                '<i>List of patterns which mark a line as a verbose anchor line</i>',
                1, 10, 1, 2);

        } elsif ($listIV eq 'shortAnchorPatternList') {

            ($vBox, $table) = $self->addTab('Page _6', $innerNotebook);

            # Short verbose components
            $self->addLabel($table, '<b>Short verbose anchor patterns</b>',
                0, 10, 0, 1);
            $self->addLabel($table,
                '<i>List of patterns which mark a line as a short verbose anchor line</i>',
                1, 10, 1, 2);

        } elsif ($listIV eq 'briefAnchorPatternList') {

            ($vBox, $table) = $self->addTab('Page _9', $innerNotebook);

            # Short verbose components
            $self->addLabel($table, '<b>Brief anchor patterns</b>',
                0, 10, 0, 1);
            $self->addLabel($table,
                '<i>List of patterns which mark a line as a brief anchor line</i>',
                1, 10, 1, 2);
        }

        $self->addRegexButton($table,
            [
                'list', $listIV,
            ],
            10, 12, 0, 2);

        $self->addTextView($table, $listIV, TRUE,
            1, 12, 2, 6,
            TRUE, TRUE, FALSE, FALSE, # Treat as a list, remove empty lines, don't remove whitespace
            -1, 200);                 # Fixed height

        $self->addLabel($table,
            '<i>Anchor line\'s relationship with other components in the component list</i>',
            1, 12, 6, 7);

        my ($group, $radioButton) = $self->addRadioButton(
            $table, undef, 'Mode -1', $offsetIV,
            -1,         # IV set to this value when toggled
            TRUE,       # Sensitive widget
            1, 4, 7, 8);
        $self->addLabel($table, 'Anchor line shares a line with the component before it',
            4, 12, 7, 8);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, 'Mode 0', $offsetIV, 0, TRUE,
            1, 4, 8, 9);
        $self->addLabel($table, 'Anchor line does not share its line with any other component',
            4, 12, 8, 9);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, 'Mode +1', $offsetIV, 1, TRUE,
            1, 4, 9, 10);
        $self->addLabel($table, 'Anchor line shares a line with the component after it',
            4, 12, 9, 10);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub rooms4Tab {

        # Rooms4 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #   $iv             - The IV to edit - one of 'verboseAnchorCheckList',
        #                       'shortAnchorCheckList' and 'briefAnchorCheckList'
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $iv, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || ! defined $iv || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->rooms4Tab', @_);
        }

        # Tab setup
        my ($vBox, $table);

        if ($iv eq 'verboseAnchorCheckList') {

            ($vBox, $table) = $self->addTab('Page _4', $innerNotebook);

            # Verbose components
            $self->addLabel($table, '<b>Verbose check patterns</b>',
                0, 12, 0, 1);
            $self->addLabel($table,
                '<i>List of patterns found near a verbose anchor line</i>',
                1, 12, 1, 2);

        } elsif ($iv eq 'shortAnchorCheckList') {

            ($vBox, $table) = $self->addTab('Page _7', $innerNotebook);

            # Short verbose components
            $self->addLabel($table, '<b>Short check patterns</b>',
                0, 12, 0, 1);
            $self->addLabel($table,
                '<i>List of patterns found near a short verbose anchor line</i>',
                1, 12, 1, 2);

        } else {

            ($vBox, $table) = $self->addTab('Page 1_0', $innerNotebook);

            # Short verbose components
            $self->addLabel($table, '<b>Brief check patterns</b>',
                0, 12, 0, 1);
            $self->addLabel($table,
                '<i>List of patterns found near a brief anchor line</i>',
                1, 12, 1, 2);
        }

        # Add a simple list
        @columnList = (
            'Offset', 'int',
            'Pattern', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 230);     # Fixed height

        # Initialise the simple list
        $self->refreshList_listIV($slWidget, scalar (@columnList / 2), $iv);

        # Add editing widgets
        $self->addLabel($table, 'Line offset:',
            1, 3, 10, 11);
        my $entry = $self->addEntryWithIcon($table, undef, 'int', -16, 16,
            3, 6, 10, 11);
        $self->addLabel($table, 'Pattern:',
            7, 9, 10, 11);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'regex', 1, undef,
            9, 12, 10, 11);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_listIV(
            $table,
            $slWidget,
            $iv,
            11, 2,
            $entry, $entry2,
        );
        $button->signal_connect('clicked' => sub {

            my ($offset, $pattern);

            $offset = $entry->get_text();
            $pattern = $entry2->get_text();

            if ($self->checkEntryIcon($entry, $entry2)) {

                # Add new values to (the end of) the list IV
                $self->addEditHash_listIV($iv, undef, FALSE, $offset, $pattern);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_listIV($slWidget, scalar (@columnList / 2), $iv);
                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub rooms11Tab {

        # Rooms11 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->rooms11Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page 11', $innerNotebook);

        # Other room statement markers
        $self->addLabel($table, '<b>Other room statement markers</b>',
            0, 10, 0, 1);
        $self->addRegexButton($table,
            [
                'list', 'notAnchorPatternList',
                'scalar', 'verboseFinalPattern',
                'scalar', 'shortFinalPattern',
                'scalar', 'briefFinalPattern',
            ],
            10, 12, 0, 1);

        my ($group, $radioButton) = $self->addRadioButton(
            $table, undef, 'OFF', 'basicMappingFlag',
            FALSE,         # IV set to this value when toggled
            TRUE,       # Sensitive widget
            1, 3, 1, 2);
        $self->addLabel($table, 'Basic mapping mode not required <i>(suitable for most worlds)</i>',
            3, 12, 1, 2);

        ($group, $radioButton) = $self->addRadioButton(
            $table, $group, 'ON', 'basicMappingFlag', TRUE, TRUE,
            1, 3, 2, 3);
        $self->addLabel($table,
            'Use basic mapping <i>(room statements don\'t contain matchable text: usually, no list'
            . ' of exits)</i>',
            3, 12, 2, 3);

        # (Not anchor patterns)
        $self->addLabel($table,
            '<i>Patterns which mean a line is definitely not an anchor line</i>',
            1, 12, 3, 4);

        $self->addTextView($table, 'notAnchorPatternList', TRUE,
            1, 12, 4, 8,
            TRUE, TRUE, FALSE, FALSE,  # Treat as list, remove empty lines, don't remove whitespace
            -1, 130);                  # Fixed height

        # (Room statement termination patterns)
        $self->addLabel(
            $table,
            '<i>Patterns which mean the whole room statement has been received (usually not'
            . ' necessary)</i>',
            1, 12, 8, 9);

        $self->addLabel($table, 'Verbose statement terminator',
            1, 4, 9, 10);
        $self->addEntry($table, 'verboseFinalPattern', TRUE,
            4, 12, 9, 10);
        $self->addLabel($table, 'Short verbose statement terminator',
            1, 4, 10, 11);
        $self->addEntry($table, 'shortFinalPattern', TRUE,
            4, 12, 10, 11);
        $self->addLabel($table, 'Brief statement terminator',
            1, 4, 11, 12);
        $self->addEntry($table, 'briefFinalPattern', TRUE,
            4, 12, 11, 12);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub rooms12Tab {

        # Rooms12 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->rooms12Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page 12', $innerNotebook);

        # Verbose exit delimiter/marker strings
        $self->addLabel($table,
            '<b>Verbose exit delimiter strings/marker patterns</b>',
            0, 10, 0, 1);
        $self->addRegexButton($table,
            [
                'list', 'verboseExitLeftMarkerList',
                'list', 'verboseExitRightMarkerList',
            ],
            10, 12, 0, 1);

        my $checkButton = $self->addCheckButton($table, 'verboseExitSplitCharFlag', TRUE,
            1, 12, 1, 2);
        $checkButton->set_label('Split exits into single characters, e.g. for NSEW');

        $self->addLabel($table, '<i>Delimiters (<b>not</b> patterns/regular expressions)</i>',
            1, 6, 2, 3);
        $self->addTextView($table, 'verboseExitDelimiterList', TRUE,
            1, 6, 3, 6,
            TRUE, TRUE, FALSE, FALSE); # Treat as list, remove empty lines, don't remove whitespace
        $self->addLabel($table, '<i>Non-delimiters (<b>not</b> patterns/regular expressions)</i>',
            7, 12, 2, 3);
        $self->addTextView($table, 'verboseExitNonDelimiterList', TRUE,
            7, 12, 3, 6,
            TRUE, TRUE, FALSE, FALSE); # Treat as list, remove empty lines, don't remove whitespace

        $self->addLabel($table, '<i>Left-side marker patterns</i>',
            1, 6, 6, 7);
        $self->addTextView($table, 'verboseExitLeftMarkerList', TRUE,
            1, 6, 7, 12,
            TRUE, TRUE, FALSE, FALSE); # Treat as list, remove empty lines, don't remove whitespace
        $self->addLabel($table, '<i>Right-side marker patterns</i>',
            7, 12, 6, 7);
        $self->addTextView($table, 'verboseExitRightMarkerList', TRUE,
            7, 12, 7, 12,
            TRUE, TRUE, FALSE, FALSE); # Treat as list, remove empty lines, don't remove whitespace

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub rooms13Tab {

        # Rooms13 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->rooms13Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page 13', $innerNotebook);

        # Brief exit delimiter/marker strings
        $self->addLabel($table,
            '<b>Brief exit delimiter strings/marker patterns</b>',
            0, 12, 0, 1);
        $self->addRegexButton($table,
            [
                'list', 'briefExitLeftMarkerList',
                'list', 'briefExitRightMarkerList',
            ],
            10, 12, 0, 1);

        my $checkButton = $self->addCheckButton($table, 'briefExitSplitCharFlag', TRUE,
            1, 12, 1, 2);
        $checkButton->set_label('Split exits into single characters, e.g. for NSEW');

        $self->addLabel($table, '<i>Delimiters (<b>not</b> patterns/regular expressions)</i>',
            1, 6, 2, 3);
        $self->addTextView($table, 'briefExitDelimiterList', TRUE,
            1, 6, 3, 6,
            TRUE, TRUE, FALSE, FALSE); # Treat as list, remove empty lines, don't remove whitespace
        $self->addLabel($table, '<i>Non-delimiters (<b>not</b> patterns/regular expressions)</i>',
            7, 12, 2, 3);
        $self->addTextView($table, 'briefExitNonDelimiterList', TRUE,
            7, 12, 3, 6,
            TRUE, TRUE, FALSE, FALSE); # Treat as list, remove empty lines, don't remove whitespace

        $self->addLabel($table, '<i>Left-side marker patterns</i>',
            1, 6, 6, 7);
        $self->addTextView($table, 'briefExitLeftMarkerList', TRUE,
            1, 6, 7, 12,
            TRUE, TRUE, FALSE, FALSE); # Treat as list, remove empty lines, don't remove whitespace
        $self->addLabel($table, '<i>Right-side marker patterns</i>',
            7, 12, 6, 7);
        $self->addTextView($table, 'briefExitRightMarkerList', TRUE,
            7, 12, 7, 12,
            TRUE, TRUE, FALSE, FALSE); # Treat as list, remove empty lines, don't remove whitespace

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub rooms14Tab {

        # Rooms14 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (@columnList, @comboList);

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->rooms14Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page 14', $innerNotebook);

        # Exit state strings
        $self->addLabel($table, '<b>Exit state strings</b>',
            0, 12, 0, 1);
        $self->addLabel(
            $table,
            '<i>Strings (<b>not</b> patterns/regular expressions) which give more information about'
            . ' exits in an exit list</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Exit state', 'text',
            'Start string', 'text',
            'Middle string', 'text',
            'End string', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 180);     # Fixed height

        # Initialise the simple list
        $self->rooms14Tab_refreshList($slWidget, scalar (@columnList / 2));

        # Add editing widgets
        $self->addLabel($table, 'Exit state',
            1, 3, 8, 9);
        @comboList = (
            'open',
            'closed',
            'locked',
            'secret',
            'secret_open',
            'secret_closed',
            'secret_locked',
            'impass',
            'dark',
            'danger',
            'emphasis',
            'other',
            'ignore',
            # Put 'normal' at the end, since it's the default exit state (but is probably never
            #   added to $self->ediObj->exitStateStringList)
            'normal',
        );
        my $combo = $self->addComboBox($table, undef, \@comboList, '',
            TRUE,           # No 'undef' value used
            3, 6, 8, 9);

        $self->addLabel($table, '(or) specify an assisted move',
            7, 9, 8, 9);
        my $entry = $self->addEntry($table, undef, TRUE,
            9, 12, 8, 9);

        $self->addLabel($table, 'Start string',
            1, 3, 9, 10);
        my $entry2 = $self->addEntry($table, undef, TRUE,
            3, 6, 9, 10);

        $self->addLabel($table, 'Middle string',
            7, 9, 9, 10);
        my $entry3 = $self->addEntry($table, undef, TRUE,
            9, 12, 9, 10);

        $self->addLabel($table, 'End string',
            1, 3, 10, 11);
        my $entry4 = $self->addEntry($table, undef, TRUE,
            3, 6, 10, 11);

        my $button = $self->addButton(
            $table, 'Add', 'Add this exit state', undef,
            1, 4, 11, 12);
        $button->signal_connect('clicked' => sub {

            my (
                $text, $text2, $state, $start, $middle, $end,
                @ivList,
            );

            $text = $combo->get_active_text();
            $text2 = $entry->get_text();
            if (defined $text2 && $text2 ne '') {
                $state = $text2;
            } else {
                $state = $text;
            }

            $start = $entry2->get_text();
            if (! $start) {

                # Use an empty string, rather than 'undef'
                $start = '';
            }

            $middle = $entry3->get_text();
            if (! $middle) {

                $middle = '';
            }

            $end = $entry4->get_text();
            if (! $end) {

                $end = '';
            }

            # Update the IV
            @ivList = $self->getEditHash_listIV('exitStateStringList');
            push (@ivList, $state, $start, $middle, $end);
            $self->ivAdd('editHash', 'exitStateStringList', \@ivList);

            # Refresh the simple list
            $self->rooms14Tab_refreshList($slWidget, scalar (@columnList / 2));
        });

        my $button2 = $self->addButton(
            $table, 'Delete', 'Delete the selected exit state from the list', undef,
            4, 6, 11, 12);
        $button2->signal_connect('clicked' => sub {

            my (
                $rowNum,
                @ivList,
            );

            # Get the selected row
            ($rowNum) = $slWidget->get_selected_indices();
            if (defined $rowNum) {

                # Update the IV
                @ivList = $self->getEditHash_listIV('exitStateStringList');

                splice (@ivList, ($rowNum * 4), 4);
                $self->ivAdd('editHash', 'exitStateStringList', \@ivList);

                # Refresh the simple list
                $self->rooms14Tab_refreshList($slWidget, scalar (@columnList / 2));
            }
        });

        my $button3 = $self->addButton(
            $table, 'Reset', 'Reset the list of exit states', undef,
            10, 12, 11, 12);
        $button3->signal_connect('clicked' => sub {

            $self->ivDelete('editHash', 'exitStateStringList');

            # Refresh the simple list
            $self->rooms14Tab_refreshList($slWidget, scalar (@columnList / 2));
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub rooms14Tab_refreshList {

        # Resets the simple list displayed by $self->rooms14Tab
        #
        # Expected arguments
        #   $slWidget   - The GA::Obj::Simple::List
        #   $columns    - The number of columns
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $slWidget, $columns, $check) = @_;

        # Local variables
        my @dataList;

        # Check for improper arguments
        if (! defined $slWidget || ! defined $columns || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->rooms14Tab_refreshList',
                @_,
            );
        }

        # Import the IV
        @dataList = $self->getEditHash_listIV('exitStateStringList');
        # Reset the simple list
        $self->resetListData($slWidget, [@dataList], $columns);

        return 1;
    }

    sub rooms15Tab {

        # Rooms15 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (@columnList, @comboList);

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->rooms15Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page 15', $innerNotebook);

        # Exit state colour tags
        $self->addLabel($table, '<b>Exit state colour tags</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of colour tags which specify an exit state, when they occur at the beginning'
            . ' of an exit</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Colour tag', 'text',
            'Exit state', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 3, 9,
            -1, 200);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'exitStateTagHash');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Colour tag',
            1, 3, 9, 10);
        my $entry = $self->addEntryWithIcon($table, undef, \&rooms15Tab_checkEntry, undef, undef,
            3, 6, 9, 10);

        $self->addLabel($table, 'Exit state',
            1, 3, 10, 11);
        @comboList = (
            'open',
            'closed',
            'locked',
            'secret',
            'secret_open',
            'secret_closed',
            'secret_locked',
            'impass',
            'dark',
            'danger',
            'emphasis',
            'other',
            'ignore',
            # Put 'normal' at the end, since it's the default exit state (but is probably never
            #   added to $self->ediObj->exitStateStringList)
            'normal',
        );
        my $combo = $self->addComboBox($table, undef, \@comboList, '',
            TRUE,           # No 'undef' value used
            3, 6, 10, 11);

        $self->addLabel($table, '(or) specify an assisted move',
            7, 9, 10, 11);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            9, 12, 10, 11);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'exitStateTagHash',
            11,
            $entry, $entry2,
        );
        $button->signal_connect('clicked' => sub {

            my ($tag, $text, $text2, $state);

            $tag = $entry->get_text();

            $text = $combo->get_active_text();
            $text2 = $entry2->get_text();
            if (defined $text2 && $text2 ne '') {
                $state = $text2;
            } else {
                $state = $text;
            }

            if ($self->checkEntryIcon($entry, $entry2)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('exitStateTagHash', $tag, $state);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'exitStateTagHash');
                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub rooms15Tab_checkEntry {

        # Called by $self->rooms15Tab to check the text in the Gtk2::Entry
        #
        # Expected arguments
        #   $text       - The contents of the Gtk2::Entry
        #
        # Return values
        #   'undef' on improper arguments or if $text is invalid
        #   1 if $text is valid

        my ($self, $text, $check) = @_;

        # Local variables
        my $result;

        # Check for improper arguments
        if (! defined $text || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->rooms15Tab_checkEntry', @_);
        }

        # $text can be an empty string, or any valid Axmud colour tag
        if ($text eq '') {

            return 1;

        } else {

            ($result) = $axmud::CLIENT->checkColourTags($text);
            if (defined $result) {
                return 1;
            } else {
                return undef;
            }
        }
    }

    sub rooms16Tab {

        # Rooms16 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->rooms16Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page 16', $innerNotebook);

        # Exit aliases
        $self->addLabel($table, '<b>Exit aliases</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of patterns, which appear in exit lists, but which represent one or more other'
            . ' exits</i>',
            1, 12, 1, 2);
        $self->addLabel($table,
            '<i>e.g. \'compass\' - representing the exits \'n s e w nw ne sw se\'</i>',
            1, 12, 2, 3);

        # Add a simple list
        @columnList = (
            'Exit alias (pattern)', 'text',
            'Replacement string', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 3, 8,
            -1, 210);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'exitAliasHash');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Exit alias',
            1, 3, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'regex', 1, undef,
            3, 6, 8, 9);

        $self->addLabel($table, 'Replacement string',
            6, 9, 8, 9);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            9, 12, 8, 9);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'exitAliasHash',
            9,
            $entry, $entry2,
        );
        $button->signal_connect('clicked' => sub {

            my ($alias, $string);

            $alias = $entry->get_text();
            $string = $entry2->get_text();

            if ($self->checkEntryIcon($entry, $entry2)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('exitAliasHash', $alias, $string);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'exitAliasHash');
                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub rooms17Tab {     # split

        # Rooms17 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->rooms17Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page 17', $innerNotebook);

        # Exit state patterns
        $self->addLabel($table, '<b>Exit state patterns</b>',
            0, 10, 0, 1);
        $self->addRegexButton($table,
            [
                'list', 'exitStatePatternList',
            ],
            10, 12, 0, 1);

        $self->addLabel(
            $table,
            '<i>If exit delimiters interfere with state strings, matching parts of the exit area'
            . ' removed before delimiters are applied</i>',
            1, 12, 1, 2);
        $self->addTextView($table, 'exitStatePatternList', TRUE,
            1, 12, 2, 6,
            TRUE, TRUE, FALSE, FALSE,  # Treat as list, remove empty lines, don't remove whitespace
            -1, 150);                   # Fixed height

        # Duplicate exit replacement string
        $self->addLabel($table, '<b>Duplicate exit replacement string</b>',
            0, 12, 6, 7);
        $self->addLabel(
            $table,
            '<i>If an exit appears in an exit list twice, the second one is converted to this'
            . ' string. <b>@@@</b> is substituted for the duplicate</i>',
            0, 12, 7, 8);
        $self->addEntryWithButton($table, 'duplicateReplaceString', TRUE,
            1, 12, 8, 9,
            64, 64);                    # Use high max characters for layout reasons

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub rooms18Tab {

        # Rooms18 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->rooms18Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page 18', $innerNotebook);

        # Exit remove/info patterns
        $self->addLabel($table, '<b>Exit remove/info patterns</b>',
            0, 10, 0, 1);
        $self->addRegexButton($table,
            [
                'list', 'exitRemovePatternList',
                'list', 'exitInfoPatternList',
            ],
            10, 12, 0, 1);

        $self->addLabel(
            $table,
            '<i>Parts of the exit which match these patterns are removed, before the exit is'
            . ' processed</i>',
            1, 12, 1, 2);
        $self->addTextView($table, 'exitRemovePatternList', TRUE,
            1, 12, 2, 6,
            TRUE, TRUE, FALSE, FALSE,  # Treat as list, remove empty lines, don't remove whitespace
            -1, 140);                  # Fixed height

        $self->addLabel(
            $table,
            '<i>Parts of the exit which match these patterns are removed (but the first'
            . ' group substring is stored)</i>',
            1, 12, 6, 7);
        $self->addTextView($table, 'exitInfoPatternList', TRUE,
            1, 12, 7, 12,
            TRUE, TRUE, FALSE, FALSE,  # Treat as list, remove empty lines, don't remove whitespace
            -1, 140);                  # Fixed height

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub rooms19Tab {

        # Rooms19 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->rooms19Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page 19', $innerNotebook);

        # Contents list patterns
        $self->addLabel($table, '<b>Contents list patterns</b>',
            0, 10, 0, 2);
        $self->addLabel(
            $table, '<i>List of patterns seen in contents lists (which are not parsed)</i>',
            1, 10, 2, 4);
        $self->addRegexButton($table,
            [
                'list', 'contentPatternList',
            ],
            10, 12, 0, 4);

        $self->addTextView($table, 'contentPatternList', TRUE,
            1, 12, 4, 12,
            TRUE, TRUE, FALSE, FALSE,  # Treat as list, remove empty lines, don't remove whitespace
            -1, 300);                  # Fixed height

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub rooms20Tab {

        # Rooms20 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->rooms20Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page 20', $innerNotebook);

        # Status commands
        $self->addLabel($table, '<b>Special contents patterns</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>List of patterns matching any line in the \'verb_special\' component</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Pattern', 'text',
            'Replacement contents', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 3, 8,
            -1, 230);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'specialPatternHash');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Pattern',
            1, 2, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'regex', 1, undef,
            2, 6, 8, 9);

        $self->addLabel($table, 'Replacement',
            7, 8, 8, 9);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            8, 12, 8, 9);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'specialPatternHash',
            9,
            $entry, $entry2,
        );
        $button->signal_connect('clicked' => sub {

            my ($pattern, $replace);

            $pattern = $entry->get_text();
            $replace = $entry2->get_text();

            if ($self->checkEntryIcon($entry, $entry2)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('specialPatternHash', $pattern, $replace);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV(
                    $slWidget,
                    scalar (@columnList / 2),
                    'specialPatternHash',
                );

                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub rooms21Tab {

        # Rooms21 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->rooms21Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page 21', $innerNotebook);

        # Room command delimiter/marker strings
        $self->addLabel($table,
            '<b>Room command delimiter strings/marker patterns</b>',
            0, 12, 0, 1);
        $self->addRegexButton($table,
            [
                'list', 'roomCmdLeftMarkerList',
                'list', 'roomCmdRightMarkerList',
            ],
            10, 12, 0, 1);

        $self->addLabel($table, '<i>Delimiters (<b>not</b> patterns)</i>',
            1, 4, 1, 2);
        $self->addTextView($table, 'roomCmdDelimiterList', TRUE,
            1, 4, 2, 6,
            TRUE, TRUE, FALSE, FALSE); # Treat as list, remove empty lines, don't remove whitespace
        $self->addLabel($table, '<i>Non-delimiters (<b>not</b> patterns)</i>',
            4, 8, 1, 2);
        $self->addTextView($table, 'roomCmdNonDelimiterList', TRUE,
            4, 8, 2, 6,
            TRUE, TRUE, FALSE, FALSE); # Treat as list, remove empty lines, don't remove whitespace
        $self->addLabel($table, '<i>Ignorable commands (<b>not</b> patterns)</i>',
            8, 12, 1, 2);
        $self->addTextView($table, 'roomCmdIgnoreList', TRUE,
            8, 12, 2, 11,
            TRUE, TRUE, FALSE, FALSE); # Treat as list, remove empty lines, don't remove whitespace

        $self->addLabel($table, '<i>Left-side marker patterns</i>',
            1, 4, 6, 7);
        $self->addTextView($table, 'roomCmdLeftMarkerList', TRUE,
            1, 4, 7, 11,
            TRUE, TRUE, FALSE, FALSE); # Treat as list, remove empty lines, don't remove whitespace
        $self->addLabel($table, '<i>Right-side marker patterns</i>',
            4, 8, 6, 7);
        $self->addTextView($table, 'roomCmdRightMarkerList', TRUE,
            4, 8, 7, 11,
            TRUE, TRUE, FALSE, FALSE); # Treat as list, remove empty lines, don't remove whitespace

        my $checkButton = $self->addCheckButton($table, 'roomCmdSplitCharFlag', TRUE,
            1, 12, 11, 12);
        $checkButton->set_label('Split room commands into singl-character commands');

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub movesTab {

        # Moves tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->movesTab', @_);
        }

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('Mo_ves', $self->notebook);

        # Add tabs to the inner notebook
        $self->moves1Tab($innerNotebook);
        $self->moves2Tab($innerNotebook);
        $self->moves3Tab($innerNotebook);
        $self->moves4Tab($innerNotebook);
        $self->moves5Tab($innerNotebook);
        $self->moves6Tab($innerNotebook);

        return 1;
    }

    sub moves1Tab {

        # Moves1 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->moves1Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _1', $innerNotebook);

        # Failed exit patterns
        $self->addLabel($table, '<b>Failed exit patterns</b>',
            0, 10, 0, 2);
        $self->addLabel($table, '<i>Patterns seen when the character fails to move</i>',
            1, 10, 2, 4);
        $self->addRegexButton($table,
            [
                'list', 'failExitPatternList',
                'list', 'darkRoomPatternList',
            ],
            10, 12, 0, 4);

        $self->addTextView($table, 'failExitPatternList', TRUE,
            1, 12, 4, 6,
            TRUE, TRUE, FALSE, FALSE); # Treat as list, remove empty lines, don't remove whitespace

        # Dark room patterns
        $self->addLabel($table, '<b>Dark room patterns</b>',
            0, 12, 6, 8);
        $self->addLabel($table,
            '<i>Patterns seen when the character moves to a room with no light</i>',
            1, 12, 8, 10);
        $self->addTextView($table, 'darkRoomPatternList', TRUE,
            1, 12, 10, 12,
            TRUE, TRUE, FALSE, FALSE); # Treat as list, remove empty lines, don't remove whitespace

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub moves2Tab {

        # Moves2 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->moves2Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _2', $innerNotebook);

        # Door patterns
        $self->addLabel($table, '<b>Door patterns</b>',
            0, 10, 0, 2);
        $self->addLabel($table, '<i>Patterns seen when the character bumps into a door</i>',
            1, 10, 2, 4);
        $self->addRegexButton($table,
            [
                'list', 'doorPatternList',
                'list', 'lockedPatternList',
            ],
            10, 12, 0, 4);
        $self->addTextView($table, 'doorPatternList', TRUE,
            1, 12, 4, 6,
            TRUE, TRUE, FALSE, FALSE); # Treat as list, remove empty lines, don't remove whitespace

        # Locked door patterns
        $self->addLabel($table, '<b>Locked door patterns</b>',
            0, 12, 6, 8);
        $self->addLabel($table, '<i>Patterns seen when the character bumps into a locked door</i>',
            1, 12, 8, 10);
        $self->addTextView($table, 'lockedPatternList', TRUE,
            1, 12, 10, 12,
            TRUE, TRUE, FALSE, FALSE); # Treat as list, remove empty lines, don't remove whitespace

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub moves3Tab {

        # Moves3 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->moves3Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _3', $innerNotebook);

        # Involuntary exit patterns
        $self->addLabel($table, '<b>Involuntary exit patterns</b>',
            0, 10, 0, 2);
        $self->addLabel($table, '<i>Patterns seen when the character is moved involuntarily</i>',
            1, 10, 2, 4);
        $self->addRegexButton($table,
            [
                'list', 'involuntaryExitPatternList',
                'list', 'unspecifiedRoomPatternList',
            ],
            10, 12, 0, 4);
        $self->addTextView($table, 'involuntaryExitPatternList', TRUE,
            1, 12, 4, 6,
            TRUE, TRUE, FALSE, FALSE); # Treat as list, remove empty lines, don't remove whitespace

        # Unspecified room patterns
        $self->addLabel($table, '<b>Unspecified room patterns</b>',
            0, 12, 6, 8);
        $self->addLabel($table,
            '<i>Patterns seen when the character moves to a room without a recognisable room'
            . ' statement</i>',
            1, 12, 8, 10);
        $self->addTextView($table, 'unspecifiedRoomPatternList', TRUE,
            1, 12, 10, 12,
            TRUE, TRUE, FALSE, FALSE); # Treat as list, remove empty lines, don't remove whitespace

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub moves4Tab {

        # Moves4 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->moves4Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _4', $innerNotebook);

        # Transient exit patterns
        $self->addLabel($table, '<b>Transient exit patterns</b>',
            0, 12, 0, 1);
        $self->addLabel(
            $table,
            '<i>Patterns matching exits which appear unpredictably (e.g. the entrance to a moving'
            . ' vehicle)</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Transient exit pattern', 'text',
            'Destination room #', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 230);      # Fixed height

        # Initialise the list
        $self->refreshList_listIV($slWidget, scalar (@columnList / 2), 'transientExitPatternList');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Transient exit pattern',
            1, 3, 10, 11);
        my $entry = $self->addEntryWithIcon($table, undef, 'regex', 1, undef,
            3, 6, 10, 11);

        $self->addLabel($table, '(Optional) desination room #',
            6, 9, 10, 11);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'string', 0, undef,
            9, 12, 10, 11);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_listIV(
            $table,
            $slWidget,
            'transientExitPatternList',
            11, 2,
            $entry, $entry2,
        );
        $button->signal_connect('clicked' => sub {

            my ($pattern, $destRoom);

            $pattern = $entry->get_text();
            $destRoom = $entry2->get_text();
            if ($destRoom eq '' || $destRoom eq '0') {

                $destRoom = undef;
            }

            if ($self->checkEntryIcon($entry, $entry2)) {

                # Add new values to (the end of) the list IV
                $self->addEditHash_listIV('transientExitPatternList',
                    undef, FALSE,
                    $pattern, $destRoom,
                );

                # Refresh the simple list and reset entry boxes
                $self->refreshList_listIV(
                    $slWidget,
                    scalar (@columnList / 2),
                    'transientExitPatternList',
                );

                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub moves5Tab {

        # Moves5 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->moves5Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _5', $innerNotebook);

        # Follow patterns
        $self->addLabel($table, '<b>Follow patterns</b>',
            0, 12, 0, 1);
        $self->addLabel(
            $table,
            '<i>Patterns seen when the character moves by following someone, and a new room'
            . ' statement is then received</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Follow pattern', 'text',
            'Direction substring #', 'int',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 230);      # Fixed height

        # Initialise the list
        $self->refreshList_listIV($slWidget, scalar (@columnList / 2), 'followPatternList');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Follow pattern',
            1, 3, 10, 11);
        my $entry = $self->addEntryWithIcon($table, undef, 'regex', 1, undef,
            3, 6, 10, 11);

        $self->addLabel($table, 'Substring # for direction',
            6, 9, 10, 11);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'int', 1, undef,
            9, 12, 10, 11);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_listIV(
            $table,
            $slWidget,
            'followPatternList',
            11, 2,
            $entry, $entry2,
        );
        $button->signal_connect('clicked' => sub {

            my ($pattern, $grpNum);

            $pattern = $entry->get_text();
            $grpNum = $entry2->get_text();

            if ($self->checkEntryIcon($entry, $entry2)) {

                # Add new values to (the end of) the list IV
                $self->addEditHash_listIV('followPatternList',
                    undef, FALSE,
                    $pattern, $grpNum,
                );

                # Refresh the simple list and reset entry boxes
                $self->refreshList_listIV(
                    $slWidget,
                    scalar (@columnList / 2),
                    'followPatternList',
                );

                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub moves6Tab {

        # Moves6 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->moves6Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _6', $innerNotebook);

        # Follow patterns
        $self->addLabel($table, '<b>Follow anchor patterns</b>',
            0, 12, 0, 1);
        $self->addLabel(
            $table,
            '<i>Patterns seen when the character moves by following someone, but no new room'
            . ' statement is then received</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Follow anchor pattern', 'text',
            'Direction substring #', 'int',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 10,
            -1, 230);      # Fixed height

        # Initialise the list
        $self->refreshList_listIV($slWidget, scalar (@columnList / 2), 'followAnchorPatternList');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Follow anchor pattern',
            1, 3, 10, 11);
        my $entry = $self->addEntryWithIcon($table, undef, 'regex', 1, undef,
            3, 6, 10, 11);

        $self->addLabel($table, 'Substring # for direction',
            6, 9, 10, 11);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'int', 1, undef,
            9, 12, 10, 11);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_listIV(
            $table,
            $slWidget,
            'followAnchorPatternList',
            11, 2,
            $entry, $entry2,
        );
        $button->signal_connect('clicked' => sub {

            my ($pattern, $grpNum);

            $pattern = $entry->get_text();
            $grpNum = $entry2->get_text();

            if ($self->checkEntryIcon($entry, $entry2)) {

                # Add new values to (the end of) the list IV
                $self->addEditHash_listIV('followAnchorPatternList',
                    undef, FALSE,
                    $pattern, $grpNum,
                );

                # Refresh the simple list and reset entry boxes
                $self->refreshList_listIV(
                    $slWidget,
                    scalar (@columnList / 2),
                    'followAnchorPatternList',
                );

                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub statusTab {

        # Status tab
        #
        # Expected arguments
        #   (none besides $self)
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $check) = @_;

        # Local variables
        my $taskObj;

        # Check for improper arguments
        if (defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->statusTab', @_);
        }

        # Tab setup
        # Create a notebook within the main one, so that we have two rows of tabs
        my ($vBox, $innerNotebook) = $self->addInnerNotebookTab('S_tatus', $self->notebook);

        # Create a temporary status task, so we can access its IVs
        $taskObj = Games::Axmud::Task::Status->new($self->session);

        # Add tabs to the inner notebook
        $self->status1Tab($innerNotebook);
        $self->status2Tab($innerNotebook, $taskObj);
        $self->status3Tab($innerNotebook, $taskObj);
        $self->status4Tab($innerNotebook);
        $self->status5Tab($innerNotebook);
        $self->status6Tab($innerNotebook, $taskObj);
        $self->status7Tab($innerNotebook);
        $self->status8Tab($innerNotebook);
        $self->status9Tab($innerNotebook);
        $self->status10Tab($innerNotebook);
        $self->status11Tab($innerNotebook);
        $self->status12Tab($innerNotebook);
        $self->status13Tab($innerNotebook);
        $self->status14Tab($innerNotebook);

        return 1;
    }

    sub status1Tab {

        # Status1 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->status1Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _1', $innerNotebook);

        # Status task commands
        $self->addLabel($table, '<b>Status task commands</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>Default list of commands sent to the world to update the Status task\'s'
            . ' variables (inherited by each</i>',
            1, 12, 1, 2);
        $self->addLabel($table,
            '<i>new character profile - edit the current character profile to change the commands'
            . ' actually sent)</i>',
            1, 12, 2, 3);

        # Add a simple list
        @columnList = (
            'Command', 'text',
            'How often to send', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 3, 8,
            -1, 220);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'statusCmdHash');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Command',
            1, 3, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            3, 6, 8, 9);

        $self->addLabel($table, 'Sent how often (in seconds):',
            6, 9, 8, 9);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'int', 1, undef,
            9, 12, 8, 9);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'statusCmdHash',
            9,
            $entry, $entry2,
        );
        $button->signal_connect('clicked' => sub {

            my ($cmd, $time);

            $cmd = $entry->get_text();
            $time = $entry2->get_text();

            if ($self->checkEntryIcon($entry, $entry2)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('statusCmdHash', $cmd, $time);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'statusCmdHash');
                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub status2Tab {

        # Status2 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #   $taskObj        - A temporary Status task object
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $taskObj, $check) = @_;

        # Local variables
        my @comboList;

        # Check for improper arguments
        if (! defined $innerNotebook || ! defined $taskObj || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->status2Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _2', $innerNotebook);

        # Status task - display
        $self->addLabel($table, '<b>Status task window format</b>',
            0, 6, 0, 1);
        $self->addLabel(
            $table,
            '<i>How the Status task displays data in its task window (format: </i>ordinary text'
            . ' @variable_name@...<i> )</i>',
            1, 11, 1, 2);
        my $textView = $self->addTextView($table, 'statusFormatList', TRUE,
            1, 11, 2, 11,
            TRUE, TRUE, FALSE, FALSE,  # Treat as list, remove empty lines, don't remove whitespace
            -1, 270);                  # Fixed height

        # Add a combo box displaying the list of Status task display variables
        @comboList = sort {lc($a) cmp lc($b)} (
            $taskObj->ivKeys('constCharVarHash'),
            $taskObj->ivKeys('constFixedVarHash'),
            $taskObj->ivKeys('constPseudoVarHash'),
            $taskObj->ivKeys('constLocalVarHash'),
            $taskObj->ivKeys('constCounterVarHash'),
        );
        $self->addLabel($table, 'Variable:',
            1, 2, 11, 12);
        my $comboBox = $self->addComboBox($table, undef, \@comboList, '',
            TRUE,           # No 'undef' value used
            2, 3, 11, 12);

        # Create some customised buttons for this tab
        # Add a button to insert the currently selected display variable into the textview, wherever
        #   the cursor is now
        my $button = Gtk2::Button->new('Insert');
        $button->signal_connect('clicked' => sub {

            my ($text, $buffer);

            $text = $comboBox->get_active_text();
            $buffer = $textView->get_buffer();
            $buffer->insert_at_cursor('@' . $text . '@');
        });
        $self->tooltips->set_tip($button, 'Insert the selected variable at the cursor');
        $table->attach_defaults($button, 3, 5, 11, 12);

        # Add a button to reset the contents of the textview to that stored by the profile
        my $button2 = Gtk2::Button->new('Reset');
        $button2->signal_connect('clicked' => sub {

            my $buffer = $textView->get_buffer();
            $buffer->set_text(join("\n", $self->editObj->statusFormatList));
        });
        $self->tooltips->set_tip($button2, 'Reset the Status task display');
        $table->attach_defaults($button2, 5, 7, 11, 12);

        # Add a button to use the Status task's default display format list
        my $button3 = Gtk2::Button->new('Default');
        $button3->signal_connect('clicked' => sub {

            my $buffer = $textView->get_buffer();
            $buffer->set_text(join("\n", $taskObj->defaultFormatList));
        });
        $self->tooltips->set_tip($button3, 'Use the default Status task display');
        $table->attach_defaults($button3, 7, 9, 11, 12);

        # Add a button to empty the textview
        my $button4 = Gtk2::Button->new('Clear');
        $button4->signal_connect('clicked' => sub {

            my $buffer = $textView->get_buffer();
            $buffer->set_text('');
        });
        $self->tooltips->set_tip($button4, 'Clear the Status task display');
        $table->attach_defaults($button4, 9, 11, 11, 12);

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub status3Tab {

        # Status3 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #   $taskObj        - A temporary Status task object
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $taskObj, $check) = @_;

        # Local variables
        my (@columnList, @comboList, @colourList);

        # Check for improper arguments
        if (! defined $innerNotebook || ! defined $taskObj || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->status3Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _3', $innerNotebook);

        # Status task - display
        $self->addLabel(
            $table,
            '<b>Status task - gauges</b>'
            . '   <i>How the Status task displays gauges in the \'main\' window</i>',
            0, 12, 0, 1);
        # Add a simple list
        @columnList = (
            'Variable', 'text',
            'Max variable', 'text',
            'Add', 'bool',
            'Label', 'text',
            'Full col', 'text',
            'Empty col', 'text',
            'Label col', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 1, 6,
            -1, 120);      # Fixed height

        # Initialise the list
        $self->refreshList_listIV(
            $slWidget,
            scalar (@columnList / 2),
            'gaugeFormatList',
        );

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Variable:',
            1, 3, 6, 7);

        @comboList = sort {lc($a) cmp lc($b)} (
            $taskObj->ivKeys('constCharVarHash'),
            $taskObj->ivKeys('constFixedVarHash'),
        );
        my $comboBox = $self->addComboBox($table, undef, \@comboList, '',
            TRUE,               # No 'undef' value used
            3, 5, 6, 7);
        $self->addLabel($table, 'or custom:',
            5, 7, 6, 7);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            7, 9, 6, 7);

        $self->addLabel($table, 'Maximum var:',
            1, 3, 7, 8);
        my $comboBox2 = $self->addComboBox($table, undef, \@comboList, '',
            TRUE,               # No 'undef' value used
            3, 5, 7, 8);
        $self->addLabel($table, 'or custom:',
            5, 7, 7, 8);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            7, 9, 7, 8);

        my $checkButton = $self->addCheckButton($table, undef, TRUE,
            9, 12, 7, 8);
        $checkButton->set_label('Not maximum');

        @colourList = ($axmud::CLIENT->constColourTagList, $axmud::CLIENT->constBoldColourTagList);
        $self->addLabel($table, 'Full colour:',
            1, 3, 8, 9);
        my $comboBox3 = $self->addComboBox($table, undef, [@colourList], '',
            TRUE,               # No 'undef' value used
            3, 5, 8, 9);
        $self->addLabel($table, 'or xterm/RGB:',
            5, 7, 8, 9);
        my $entry3 = $self->addEntryWithIcon($table, undef, \&status3Tab_checkEntry, undef, undef,
            7, 9, 8, 9);

        $self->addLabel($table, 'Empty colour:',
            1, 3, 9, 10);
        my $comboBox4 = $self->addComboBox($table, undef, [@colourList], '',
            TRUE,               # No 'undef' value used
            3, 5, 9, 10);
        $self->addLabel($table, 'or xterm/RGB:',
            5, 7, 9, 10);
        my $entry4 = $self->addEntryWithIcon($table, undef, \&status3Tab_checkEntry, undef, undef,
            7, 9, 9, 10);

        $self->addLabel($table, 'Label colour:',
            1, 3, 10, 11);
        my $comboBox5 = $self->addComboBox($table, undef, [@colourList], '',
            TRUE,               # No 'undef' value used
            3, 5, 10, 11);
        $self->addLabel($table, 'or xterm/RGB:',
            5, 7, 10, 11);
        my $entry5 = $self->addEntryWithIcon($table, undef, \&status3Tab_checkEntry, undef, undef,
            7, 9, 10, 11);

        $self->addLabel($table, 'Label text <i>(optional)</i>:',
            9, 12, 9, 10);
        my $entry6 = $self->addEntry($table, undef, TRUE,
            9, 12, 10, 11,
            16, 16);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_listIV(
            $table,
            $slWidget,
            'gaugeFormatList',
            11, 7,
            $comboBox, $comboBox2, $comboBox3, $comboBox4, $comboBox5,
            $entry, $entry2, $entry3, $entry4, $entry5, $entry6,
        );
        $button->signal_connect('clicked' => sub {

            my ($var, $maxVar, $full, $empty, $label, $text, $flag);

            if ($self->checkEntryIcon($entry)) {
                $var = $entry->get_text();
            } else {
                $var = $comboBox->get_active_text();
            }

            if ($self->checkEntryIcon($entry2)) {
                $maxVar = $entry2->get_text();
            } else {
                $maxVar = $comboBox2->get_active_text();
            }

            if ($self->checkEntryIcon($entry3)) {
                $full = $entry3->get_text();
            } else {
                $full = $comboBox3->get_active_text();
            }

            if ($self->checkEntryIcon($entry4)) {
                $empty = $entry4->get_text();
            } else {
                $empty = $comboBox4->get_active_text();
            }

            if ($self->checkEntryIcon($entry5)) {
                $label = $entry5->get_text();
            } else {
                $label = $comboBox5->get_active_text();
            }

            if ($checkButton->get_active()) {
                $flag = TRUE;
            } else {
                $flag = FALSE;
            }

            $text = $entry6->get_text();

            if ($var && $maxVar && $full && $empty && $label) {

                # Add new values to (the end of) the list IV
                $self->addEditHash_listIV('gaugeFormatList',
                    undef, FALSE,
                    $var, $maxVar, $flag, $text, $full, $empty, $label,
                );

                # Refresh the simple list and reset entry boxes
                $self->refreshList_listIV(
                    $slWidget,
                    scalar (@columnList / 2),
                    'gaugeFormatList',
                );
                $self->resetEntryBoxes($entry, $entry2, $entry3, $entry4, $entry5, $entry6);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub status3Tab_checkEntry {

        # Called by $self->status3Tab to check the text in the Gtk2::Entry
        #
        # Expected arguments
        #   $text       - The contents of the Gtk2::Entry
        #
        # Return values
        #   'undef' on improper arguments or if $text is invalid
        #   1 if $text is valid

        my ($self, $text, $check) = @_;

        # Check for improper arguments
        if (! defined $text || defined $check) {

            return $axmud::CLIENT->writeImproper(
                $self->_objClass . '->status3Tab_checkEntry',
                @_,
            );
        }

        # $text can be an xterm colour tag (in the range 'x0' to 'x255', or 'ux0' to 'ux255'; xterm
        #   tags are case-insensitive
        # $text can also be an RGB colour tag (in the range '#000000' to '#FFFFFF')
        if ($text =~ m/^u?x([0-9]+)$/i) {

            # (Don't permit 'x000005', but do permit 'x005' or 'x5')
            if ($1 >= 0 && $1 <= 255 && length ($1) <= 3) {
                return 1;
            } else {
                return undef;
            }

        } elsif ($text =~ m/^\#[0-9A-Fa-f]{6}$/) {

            return 1;

        } else {

            return undef;
        }
    }

    sub status4Tab {

        # Status4 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->status4Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _4', $innerNotebook);

        # Status task MSDP variables
        $self->addLabel($table, '<b>Status task MSDP variables</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>MSDP variables that should be converted directly into Status task variables</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'MSDP variable', 'text',
            'Status task variable', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 240);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'msdpStatusVarHash');

        # Add entries/comboboxes for adding new variables
        $self->addLabel($table, 'MSDP variable',
            1, 3, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            3, 6, 8, 9);

        $self->addLabel($table, 'Status task variable',
            6, 9, 8, 9);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            9, 12, 8, 9);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'msdpStatusVarHash',
            9,
            $entry, $entry2,
        );
        $button->signal_connect('clicked' => sub {

            my ($msdpVar, $var);

            $msdpVar = $entry->get_text();
            $var = $entry2->get_text();

            if ($self->checkEntryIcon($entry, $entry2)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('msdpStatusVarHash', $msdpVar, $var);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV(
                    $slWidget,
                    scalar (@columnList / 2),
                    'msdpStatusVarHash',
                );

                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub status5Tab {

        # Status5 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my @columnList;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->status5Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _5', $innerNotebook);

        # Status task MXP variables
        $self->addLabel($table, '<b>Status task MXP variables</b>',
            0, 12, 0, 1);
        $self->addLabel($table,
            '<i>MXP entities whose values should be converted directly into Status task'
            . ' variables</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'MXP variable', 'text',
            'Status task variable', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 240);      # Fixed height

        # Initialise the list
        $self->refreshList_hashIV($slWidget, scalar (@columnList / 2), 'mxpStatusVarHash');

        # Add entries/comboboxes for adding new variables
        $self->addLabel($table, 'MXP entity',
            1, 3, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            3, 6, 8, 9);

        $self->addLabel($table, 'Status task variable',
            6, 9, 8, 9);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            9, 12, 8, 9);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_hashIV(
            $table,
            $slWidget,
            'mxpStatusVarHash',
            9,
            $entry, $entry2,
        );
        $button->signal_connect('clicked' => sub {

            my ($mxpEnt, $var);

            $mxpEnt = $entry->get_text();
            $var = $entry2->get_text();

            if ($self->checkEntryIcon($entry, $entry2)) {

                # Add a new key-value pair
                $self->modifyEditHash_hashIV('mxpStatusVarHash', $mxpEnt, $var);

                # Refresh the simple list and reset entry boxes
                $self->refreshList_hashIV(
                    $slWidget,
                    scalar (@columnList / 2),
                    'mxpStatusVarHash',
                );

                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub status6Tab {

        # Status6 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #   $taskObj        - A temporary Status task object
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $taskObj, $check) = @_;

        # Local variables
        my (
            $customString,
            @columnList, @comboList,
        );

        # Check for improper arguments
        if (! defined $innerNotebook || ! defined $taskObj || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->status6Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _6', $innerNotebook);

        # Status task - general patterns
        $self->addLabel($table, '<b>Status task - general patterns</b>',
            0, 12, 0, 1);
        $self->addLabel(
            $table,
            '<i>Patterns with group substrings, e.g. </i><b>(.*)</b> <i>and</i> <b>(\d+)</b><i>,'
            . ' one of which matches the data we need</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Pattern', 'text',
            'Substring #', 'int',
            'Variable', 'text',
            'Mode', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 170);      # Fixed height

        # Initialise the list
        $self->refreshList_listIV(
            $slWidget,
            scalar (@columnList / 2),
            'groupPatternList',
        );

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Pattern:',
            1, 3, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'regex', 1, undef,
            3, 12, 8, 9);

        $self->addLabel($table, 'Substring #:',
            1, 3, 9, 10);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'int', 1, undef,
            3, 6, 9, 10, 4, 4);

        $self->addLabel($table, 'Mode:',
            9, 10, 9, 10);
        my $comboBox = $self->addComboBox($table, undef, ['hide', 'show', 'choose'], '',
            TRUE,               # No 'undef' value used
            10, 12, 9, 10);

        $self->addLabel($table, 'Status task variable:',
            1, 6, 10, 11);

        $customString = 'Use a custom variable:';

        @comboList = sort {lc($a) cmp lc($b)} (
            $taskObj->ivKeys('constCharVarHash'),
            $taskObj->ivKeys('constFixedVarHash'),
            $taskObj->ivKeys('constPseudoVarHash'),
            $taskObj->ivKeys('constLocalVarHash'),
            $taskObj->ivKeys('constCounterVarHash'),
        );
        unshift (@comboList, $customString);
        my $comboBox2 = $self->addComboBox($table, undef, \@comboList, '',
            TRUE,               # No 'undef' value used
            6, 9, 10, 11);
        # ->signal_connect appears below

        my $entry3 = $self->addEntry($table, undef, TRUE,
            9, 12, 10, 11);
        $comboBox2->signal_connect('changed' => sub {

            my $text = $comboBox2->get_active_text();
            if ($text eq $customString ) {
                $entry3->set_sensitive(TRUE);
            } else {
                $entry3->set_sensitive(FALSE);
                $entry3->set_text('');
            }
        });

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_listIV(
            $table,
            $slWidget,
            'groupPatternList',
            11, 4,
            $entry, $entry2, $entry3,
        );
        $button->signal_connect('clicked' => sub {

            my ($pattern, $grpNum, $var, $mode);

            $pattern = $entry->get_text();
            $grpNum = $entry2->get_text();
            $mode = $comboBox->get_active_text();
            $var = $comboBox2->get_active_text();
            if ($var eq $customString) {

                $var = $entry3->get_text();
            }

            if ($self->checkEntryIcon($entry, $entry2)) {

                # Add new values to (the end of) the list IV
                $self->addEditHash_listIV('groupPatternList',
                    undef, FALSE,
                    $pattern, $grpNum, $var, $mode
                );

                # Refresh the simple list and reset entry boxes
                $self->refreshList_listIV(
                    $slWidget,
                    scalar (@columnList / 2),
                    'groupPatternList',
                );
                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub status7Tab {

        # Status7 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (@columnList, @comboList);

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->status7Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _7', $innerNotebook);

        # Status task - bar patterns
        $self->addLabel($table, '<b>Status task - bar patterns</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>Patterns containing data represented by multiple symbols</i>',
            1, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Pattern', 'text',
            'Substring #', 'int',
            'Variable', 'text',
            'Mode', 'text',
            'Unit', 'text',
            'Max units', 'int',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 210);      # Fixed height

        # Initialise the list
        $self->refreshList_listIV($slWidget, scalar (@columnList / 2), 'barPatternList');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Pattern:',
            1, 2, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'regex', 1, undef,
            2, 6, 8, 9);

        $self->addLabel($table, 'Unit:',
            6, 7, 8, 9);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'string', 1, undef,
            7, 9, 8, 9, 4, 4);

        $self->addLabel($table, 'Max units:',
            9, 10, 8, 9);
        my $entry3 = $self->addEntryWithIcon($table, undef, 'int', 1, undef,
            10, 12, 8, 9, 4, 4);

        $self->addLabel($table, 'Substring #:',
            1, 2, 9, 10);
        my $entry4 = $self->addEntryWithIcon($table, undef, 'int', 1, undef,
            2, 4, 9, 10, 4, 4);

        $self->addLabel($table, 'Data type:',
            4, 6, 9, 10);

        @comboList = (
            'health', 'magic', 'energy', 'guild', 'social', 'xp_current', 'xp_next_level',
            'qp_current', 'qp_next_level', 'op_current', 'op_next_level',
        );

        my $comboBox = $self->addComboBox($table, undef, \@comboList, '',
            TRUE,               # No 'undef' value used
            6, 8, 9, 10);

        $self->addLabel($table, 'Mode:',
            8, 10, 9, 10);
        my $comboBox2 = $self->addComboBox($table, undef, ['hide', 'show', 'choose'], '',
            TRUE,               # No 'undef' value used
            10, 12, 9, 10);

        # Add buttons at various positions
        my $button = $self->addSimpleListButtons_listIV(
            $table,
            $slWidget,
            'barPatternList',
            10, 6,
            $entry, $entry2, $entry3, $entry4,
        );
        $button->signal_connect('clicked' => sub {

            my ($pattern, $grpNum, $var, $mode, $unit, $maxUnits);

            $pattern = $entry->get_text();
            $grpNum = $entry4->get_text();
            $var = $comboBox->get_active_text();
            $mode = $comboBox2->get_active_text();
            $unit = $entry2->get_text();
            $maxUnits = $entry3->get_text();

            if ($self->checkEntryIcon($entry, $entry2, $entry3, $entry4)) {

                # Add new values to (the end of) the list IV
                $self->addEditHash_listIV(
                    'barPatternList',
                    undef, FALSE,
                    $pattern, $grpNum, $var, $mode, $unit, $maxUnits,
                );

                # Refresh the simple list and reset entry boxes
                $self->refreshList_listIV($slWidget, scalar (@columnList / 2), 'barPatternList');
                $self->resetEntryBoxes($entry, $entry2, $entry3, $entry4);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub status8Tab {

        # Status8 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (@columnList, @comboList);

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->status8Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _8', $innerNotebook);

        # Status task - character affect patterns
        $self->addLabel($table, '<b>Status task - character affect patterns</b>',
            0, 12, 0, 1);
        $self->addLabel(
            $table,
            '<i>Patterns containing data about spells (etc) affecting the character</i>',
            0, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Pattern', 'text',
            'Substring #', 'int',
            'Display', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 210);      # Fixed height

        # Initialise the list
        $self->refreshList_listIV($slWidget, scalar (@columnList / 2), 'affectPatternList');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Pattern:',
            1, 2, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'regex', 1, undef,
            2, 12, 8, 9);

        $self->addLabel($table, 'Substring #:',
            1, 2, 9, 10);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'int', 1, undef,
            2, 6, 9, 10, 4, 4);

        $self->addLabel($table, 'Mode:',
            9, 10, 9, 10);
        my $comboBox2 = $self->addComboBox($table, undef, ['hide', 'show', 'choose'], '',
            TRUE,               # No 'undef' value used
            10, 12, 9, 10);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_listIV(
            $table,
            $slWidget,
            'affectPatternList',
            10,  4,
            $entry, $entry2,
        );
        $button->signal_connect('clicked' => sub {

            my ($pattern, $grpNum, $mode);

            $pattern = $entry->get_text();
            $grpNum = $entry2->get_text();
            $mode = $comboBox2->get_active_text();

            if ($self->checkEntryIcon($entry, $entry2)) {

                # Add new values to (the end of) the list IV
                $self->addEditHash_listIV(
                    'affectPatternList',
                    undef, FALSE,
                    $pattern, $grpNum, $mode,
                );

                # Refresh the simple list and reset entry boxes
                $self->refreshList_listIV(
                    $slWidget,
                    scalar (@columnList / 2),
                    'affectPatternList',
                );
                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub status9Tab {

        # Status9 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Local variables
        my (@columnList, @comboList);

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->status9Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page _9', $innerNotebook);

        # Status task - character stat patterns
        $self->addLabel($table, '<b>Status task - character stat patterns</b>',
            0, 12, 0, 1);
        $self->addLabel($table, '<i>Patterns containing data about the character\'s stats</i>',
            0, 12, 1, 2);

        # Add a simple list
        @columnList = (
            'Pattern', 'text',
            'Substring #', 'int',
            'Stat', 'text',
            'Display', 'text',
        );

        my $slWidget = $self->addSimpleList($table, undef, \@columnList,
            1, 12, 2, 8,
            -1, 210);      # Fixed height

        # Initialise the list
        $self->refreshList_listIV($slWidget, scalar (@columnList / 2), 'statPatternList');

        # Add entries/comboboxes for adding new patterns
        $self->addLabel($table, 'Pattern:',
            1, 2, 8, 9);
        my $entry = $self->addEntryWithIcon($table, undef, 'regex', 1, undef,
            2, 12, 8, 9);

        $self->addLabel($table, 'Substring #:',
            1, 2, 9, 10);
        my $entry2 = $self->addEntryWithIcon($table, undef, 'int', 1, undef,
            2, 6, 9, 10, 4, 4);

        $self->addLabel($table, 'Stat:',
            6, 7, 9, 10);
        @comboList = $self->editObj->ivKeys('statHash');
        my $comboBox = $self->addComboBox($table, undef, \@comboList, '',
            TRUE,               # No 'undef' value used
            7, 9, 9, 10);

        $self->addLabel($table, 'Mode:',
            9, 10, 9, 10);
        my $comboBox2 = $self->addComboBox($table, undef, ['hide', 'show', 'choose'], '',
            TRUE,               # No 'undef' value used
            10, 12, 9, 10);

        # Add standard editing buttons to the simple list
        my $button = $self->addSimpleListButtons_listIV(
            $table,
            $slWidget,
            'statPatternList',
            10,  4,
            $entry, $entry2,
        );
        $button->signal_connect('clicked' => sub {

            my ($pattern, $grpNum, $stat, $mode);

            $pattern = $entry->get_text();
            $grpNum = $entry2->get_text();
            $stat = $comboBox->get_active_text();
            $mode = $comboBox2->get_active_text();

            if (! $stat) {

                $self->showMsgDialogue(
                    'Character stat error',
                    'error',
                    'No stat set (if the stat list is empty, you must create some stats in the'
                    . ' \'stats\' tab)',
                    'ok',
                );

            } elsif ($self->checkEntryIcon($entry, $entry2)) {

                # Add new values to (the end of) the list IV
                $self->addEditHash_listIV(
                    'statPatternList',
                    undef, FALSE,
                    $pattern, $grpNum, $stat, $mode,
                );

                # Refresh the simple list and reset entry boxes
                $self->refreshList_listIV($slWidget, scalar (@columnList / 2), 'statPatternList');
                $self->resetEntryBoxes($entry, $entry2);
            }
        });

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub status10Tab {

        # Status10 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->status10Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page 1_0', $innerNotebook);

        # Status task ignore patterns
        $self->addLabel($table, '<b>Status task ignore patterns</b>',
            0, 10, 0, 2);
        $self->addLabel($table, '<i>Patterns which should be ignored by the Status task</i>',
            1, 10, 2, 4);
        $self->addRegexButton($table,
            [
                'list', 'statusIgnorePatternList',
                'list', 'questCompletePatternList',
            ],
            10, 12, 0, 4);

        $self->addTextView($table, 'statusIgnorePatternList', TRUE,
            1, 12, 4, 6,
            TRUE, TRUE, FALSE, FALSE); # Treat as list, remove empty lines, don't remove whitespace

        # Quest complete patterns
        $self->addLabel($table, '<b>Quest complete patterns</b>',
            0, 12, 6, 8);
        $self->addLabel($table, '<i>Patterns announcing a quest is complete</i>',
            1, 12, 8, 10);
        $self->addTextView($table, 'questCompletePatternList', TRUE,
            1, 12, 10, 12,
            TRUE, TRUE, FALSE, FALSE); # Treat as list, remove empty lines, don't remove whitespace

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub status11Tab {

        # Status11 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->status11Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page 11', $innerNotebook);

        # Status task age patterns
        $self->addLabel($table, '<b>Status task age patterns</b>',
            0, 10, 0, 2);
        $self->addLabel($table, '<i>Patterns containing the character\'s age</i>',
            1, 10, 2, 4);
        $self->addRegexButton($table,
            [
                'list', 'agePatternList',
                'list', 'timePatternList',
            ],
            10, 12, 0, 4);

        $self->addTextView($table, 'agePatternList', TRUE,
            1, 12, 4, 6,
            TRUE, TRUE, FALSE, FALSE); # Treat as list, remove empty lines, don't remove whitespace

        # Status task time patterns
        $self->addLabel($table, '<b>Status task time patterns</b>',
            0, 12, 6, 8);
        $self->addLabel($table, '<i>Patterns containing the (in-game) time</i>',
            1, 12, 8, 10);
        $self->addTextView($table, 'timePatternList', TRUE,
            1, 12, 10, 12,
            TRUE, TRUE, FALSE, FALSE); # Treat as list, remove empty lines, don't remove whitespace

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub status12Tab {

        # Status12 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->status12Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page 12', $innerNotebook);

        # Life patterns
        $self->addLabel($table, '<b>Life patterns</b>',
            0, 10, 0, 2);
        $self->addLabel($table, '<i>Patterns seen when the character dies</i>',
            1, 10, 2, 4);
        $self->addRegexButton($table,
            [
                'list', 'deathPatternList',
                'list', 'resurrectPatternList',
            ],
            10, 12, 0, 4);

        $self->addTextView($table, 'deathPatternList', TRUE,
            1, 12, 4, 6,
            TRUE, TRUE, FALSE, FALSE); # Treat as list, remove empty lines, don't remove whitespace

        $self->addLabel($table, '<i>Patterns seen when the character is resurrected</i>',
            1, 12, 6, 8);
        $self->addTextView($table, 'resurrectPatternList', TRUE,
            1, 12, 8, 10,
            TRUE, TRUE, FALSE, FALSE); # Treat as list, remove empty lines, don't remove whitespace

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub status13Tab {

        # Status13 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->status13Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page 13', $innerNotebook);

        # Passing out patterns
        $self->addLabel($table, '<b>Passing out patterns</b>',
            0, 10, 0, 2);
        $self->addLabel($table, '<i>Patterns seen when the character passes out</i>',
            1, 10, 2, 4);
        $self->addRegexButton($table,
            [
                'list', 'passedOutPatternList',
                'list', 'comeAroundPatternList',
            ],
            10, 12, 0, 4);

        $self->addTextView($table, 'passedOutPatternList', TRUE,
            1, 12, 4, 6,
            TRUE, TRUE, FALSE, FALSE); # Treat as list, remove empty lines, don't remove whitespace

        $self->addLabel($table, '<i>Patterns seen when the character comes around</i>',
            1, 12, 6, 8);
        $self->addTextView($table, 'comeAroundPatternList', TRUE,
            1, 12, 8, 10,
            TRUE, TRUE, FALSE, FALSE); # Treat as list, remove empty lines, don't remove whitespace

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub status14Tab {

        # Status14 tab
        #
        # Expected arguments
        #   $innerNotebook  - The Gtk2::Notebook object inside $self->notebook
        #
        # Return values
        #   'undef' on improper arguments
        #   1 otherwise

        my ($self, $innerNotebook, $check) = @_;

        # Check for improper arguments
        if (! defined $innerNotebook || defined $check) {

            return $axmud::CLIENT->writeImproper($self->_objClass . '->status14Tab', @_);
        }

        # Tab setup
        my ($vBox, $table) = $self->addTab('Page 14', $innerNotebook);

        # Sleep patterns
        $self->addLabel($table, '<b>Sleep patterns</b>',
            0, 10, 0, 2);
        $self->addLabel($table, '<i>Patterns seen when the character falls asleep</i>',
            1, 10, 2, 4);
        $self->addRegexButton($table,
            [
                'list', 'fallAsleepPatternList',
                'list', 'wakeUpPatternList',
            ],
            10, 12, 0, 4);

        $self->addTextView($table, 'fallAsleepPatternList', TRUE,
            1, 12, 4, 6,
            TRUE, TRUE, FALSE, FALSE); # Treat as list, remove empty lines, don't remove whitespace

        $self->addLabel($table, '<i>Patterns seen when the character wakes up</i>',
            1, 12, 6, 8);
        $self->addTextView($table, 'wakeUpPatternList', TRUE,
            1, 12, 8, 10,
            TRUE, TRUE, FALSE, FALSE); # Treat as list, remove empty lines, don't remove whitespace

        # Tab complete
        $vBox->pack_start($table, 0, 0, 0);

        return 1;
    }

    sub inventoryTab {

        # Inventory tab
    