package Twilio;
use Perlmazing;
use LWP::UserAgent;
use JSON;
use Scalar::Util qw(refaddr);
use Moduler;
use IO::Handle;
our $VERSION = '1.000';

our $AUTOLOAD;
our $api_url = 'api.twilio.com:443';
our $api_realm = 'Twilio API';
our @api_versions = qw(2010-04-01);
our $Objects = {};

my $modules;
my $maps;

our @members = qw(
	Twilio::Accounts
	Twilio::Applications
	Twilio::AuthorizedConnectApps
	Twilio::AvailablePhoneNumbers
	Twilio::Calls
	Twilio::Conferences
	Twilio::ConnectApps
	Twilio::IncomingPhoneNumbers
	Twilio::Media
	Twilio::Messages
	Twilio::Notifications
	Twilio::Objects
	Twilio::OutgoingCallerIds
	Twilio::Queues
	Twilio::Recordings
	Twilio::Sandbox
	Twilio::ShortCodes
	Twilio::Sip
	Twilio::SmsMessages
	Twilio::Transcriptions
	Twilio::Usage
);

BEGIN {
	for my $i (Moduler->Get(__PACKAGE__)) {
		if (exists $maps->{$i->Name}) {
			warn "Warning: Mapping for module $i was ignored to avoid overriding\n";
			next;
		} else {
			$maps->{$i->Name} = $i->Module;
			$i->require;
		}
		if ($i->Module =~ /Twilio::\w+$/) {
			next if $i->Module eq 'Twilio::Twiml';
			$i->Module->import;
			$modules->{$i->Module} = $i->Path;
		}
	}
}

sub country_codes () { $country_codes }

sub get_mapping {
	my $self = shift;
	my $name = shift;
	$self->die("There's no mapping for name '$name'. Current mapping is:\n".dumped($maps)) unless exists $maps->{$name};
	$maps->{$name};
}

sub new {
	my $class = shift;
	my $self = {
		sid			=> undef,
		token		=> undef,
		trace		=> undef,
		debug		=> undef,
		use_threads	=> 0,
		testing		=> 0, # This is merely for installation tests, don't use it yourself
		@_,
	};
	my $testing = delete $self->{testing};
	bless $self, $class;
	if (exists $self->{Twilio}) {
		$self->Objects->{Twilio} = delete $self->{Twilio};
	}
	$self->validate;
	$self->Objects->{ua} = Twilio::UserAgent->new(testing => $testing);
	for my $i (@members) {
		my $name = substr $i, rindex($i, ':') + 1;
		$self->Objects->{$name} = $i->new (
			Twilio => $self,
		);
	}
	$self->trace("Instance created");
	$self;
}

sub Account {
	my $self = shift;
	$self->Accounts->get($self->sid);
}

sub get_trace_handler {
	my $self = shift;
	my $trace = eval {
		$self->{trace} or ($self->Objects->{Twilio} and $self->Objects->Twilio->{trace});
	};
	my $fh = *STDERR;
	if ($@ or not $trace) {
		return undef;
	}
	if (is_filehandle $trace) {
		$fh = $trace;
	}
	$fh->autoflush(1);
	$fh;
}

sub trace {
	my $self = shift;
	my $fh = $self->get_trace_handler;
	return unless $fh;
	for my $i (@_) {
		my $name = ref $self;
		say $fh "-: [$name] $i";
	}
}

sub debug {
	my $self = shift;
	$self->trace(@_) if $self->{debug};
}

sub DESTROY {
	my $self = shift;
	my $addr = refaddr $self;
	my $tied = tied %$self;
	$addr = refaddr($tied) if $tied;
	delete $Objects->{$addr};
	$addr;
}

sub Objects {
	my $self = shift;
	my $addr = refaddr $self;
	my $tied = tied %$self;
	$addr = refaddr($tied) if $tied;
	unless (exists $Objects->{$addr}) {
		$Objects->{$addr} = Twilio::Objects->new;
	}
	$Objects->{$addr};
}

sub copy_objects {
	my $self = shift;
	my $instance = shift;
	my $refaddr = shift;
	$self->die("Cannot find instance $instance in \$Objects") unless exists $Objects->{$instance};
	$Objects->{$refaddr} = $Objects->{$instance};
}

sub validate {
	my $self = shift;
	$self->trace("Validating object properties");
	for my $i (qw(sid token)) {
		$self->die("Property $i is not defined and it's needed for Twilio API calls") unless defined $self->{$i};
	}
}

sub root_url {
	my $self = shift;
	"https://$api_url";
}

sub api_uri {
	my $self = shift;
	"/$api_versions[$#api_versions]";
}

sub base_url {
	my $self = shift;
	$self->root_url.$self->api_uri;
}

sub account_uri {
	my $self = shift;
	$self->api_uri."/Accounts/".$self->sid;
}

sub account_url {
	my $self = shift;
	$self->root_url.$self->account_uri;
}

sub set_credentials {
	my $self = shift;
	$self->Objects->ua->credentials($api_url, $api_realm, $self->sid, $self->token);
}

sub get {
	my $self = shift;
	my ($url, $params) = @_;
	$self->set_credentials;
	normalize_request_params($params);
	for (my $i = 0; $i < @$params; $i += 2) {
		$url .= (($url =~ /\?/) ? '&' : '?').escape_uri($params->[$i]).'='.escape_uri( to_utf8 $params->[$i + 1]);
	}
	$self->trace("GETing $url");
	my $r = $self->Objects->ua->get($url);
	if ($r->is_success) {
		my $obj = from_json $r->decoded_content;
		unless ($obj) {
			$self->die("GETed empty object from $url");
		}
	} else {
		my $body = '';
		my $json = eval {
			from_json $r->decoded_content;
		};
		if ($@) {
			$body = $r->decoded_content;
		} else {
			$body = dumped $json;
		}
		$self->die("Error GETing $url:\nStatus: ".$r->status_line."\n".$body."\n");
	}
}

sub post {
	my $self = shift;
	my ($url, $params) = @_;
	$self->set_credentials;
	normalize_request_params($params);
	$self->trace("POSTing $url");
	for (my $i = 0; $i < @$params; $i += 2) {
		$params->[$i + 1] = to_utf8 $params->[$i + 1];
		$self->trace("	$params->[$i]: $params->[$i + 1]");
	}
	my $r = $self->Objects->ua->post($url, $params);
	if ($r->is_success) {
		return from_json $r->decoded_content;
	} else {
		my $body = '';
		my $json = eval {
			from_json $r->decoded_content;
		};
		if ($@) {
			$body = $r->decoded_content;
		} else {
			$body = dumped $json;
		}
		$self->die("Error POSTing $url:\nStatus: ".$r->status_line."\n".$body."\n");
	}
}

sub AUTOLOAD : lvalue {
	my $autoload = $AUTOLOAD;
	my $wantarray = wantarray;
	my $self = shift;
	my $name = substr $autoload, rindex($autoload, ':') + 1;
	return if $name eq 'DESTROY';
	my @call = caller(0);
	if (is_blessed($self) and $self->isa('HASH') and exists $self->{$name}) {
		if (not defined($wantarray) and is_blessed($self->{$name}) and $self->{$name}->isa('Twilio::Collection')) {
			warn "Useless call to $autoload in void context at $call[1] line $call[2]\n";
			return;
		}
		if (@_ or $wantarray and is_blessed($self->{$name}) and $self->{$name}->isa('Twilio::Collection')) {
			$self->debug("AUTOLOAD: $name -> autolisting in list context");
			return $self->{$name}->list(@_);
		} else {
			return $self->{$name};
		}
	} elsif (is_blessed($self) and exists $self->Objects->{$name}) {
		$self->debug("AUTOLOAD: $name -> returning self Objects property");
		if (not defined($wantarray) and is_blessed($self->Objects->{$name}) and $self->Objects->{$name}->isa('Twilio::Collection')) {
			warn "Useless call to $autoload in void context at $call[1] line $call[2]\n";
			return;
		}
		if (@_ or $wantarray and is_blessed($self->Objects->{$name}) and $self->Objects->{$name}->isa('Twilio::Collection')) {
			$self->debug("AUTOLOAD: $name -> autolisting in list context");
			return $self->Objects->{$name}->list(@_);
		} else {
			if ($name eq 'Sandbox') {
				my $resource = $self->Objects->{$name};
				return $resource if $resource and $resource->{phone_number};
				my ($url, $r);
				if ($resource->{uri}) {
					$url = $self->root_url.$resource->{uri};
				} else {
					$url = $self->account_url.'/Sandbox.json';
				}
				if ($self->Objects->{Twilio}) {
					$r = $self->Twilio->get($self->account_url.'/Sandbox.json');
				} else {
					$r = $self->get($self->account_url.'/Sandbox.json');
				}
				$self->Objects->Sandbox = Twilio::Sandbox->new (
					Twilio	=> $self,
					%$r,
				);
			} else {
				return $self->Objects->{$name};
			}
		}
	} else {
		if (is_blessed $self) {
			if (exists($self->Objects->{Twilio}) and is_blessed($self->Objects->Twilio) and exists $self->Objects->Twilio->{$name}) {
				$self->debug("AUTOLOAD: $name -> returning main Twilio property");
				if (not defined($wantarray) and is_blessed($self->Objects->Twilio->{$name}) and $self->Objects->Twilio->{$name}->isa('Twilio::Collection')) {
					warn "Useless call to $autoload in void context at $call[1] line $call[2]\n";
					return;
				}
				if (@_ or $wantarray and is_blessed($self->Objects->Twilio->{$name}) and $self->Objects->Twilio->{$name}->isa('Twilio::Collection')) {
					if (not defined $wantarray) {
						warn "Useless call in void context at $call[1] line $call[2]\n";
						return;
					}
					$self->debug("AUTOLOAD: $name -> autolisting in list context");
					return $self->Objects->Twilio->{$name}->list(@_);
				} else {
					return $self->Objects->Twilio->{$name};
				}
			} else {
				if (defined &{ref($self).'::autoload'}) {
					$self->debug("AUTOLOAD: $name -> returning self defined &autoload");
					return $self->autoload($autoload, @_);
				} else {
					die "Can't locate method or property '$name' via package ".ref($self)." at $call[1] line $call[2]\n";
				}
			}
		} else {
			die "Uknown subroutine &$AUTOLOAD called at $call[1] line $call[2]\n";
		}
	}
}

sub use_instead {
	my $self = shift;
	my @caller = caller(1);
	my $call = $caller[3];
	my ($class, $method) = (substr($call, 0, rindex($call, '::')), substr($call, rindex($call, ':') + 1));
	(my $alternatives = join ', ', map { "'$_'" } @_) =~ s/, ('\w+')$/ or $1/;
	$self->die("Instances of class $class don't have a '$method' method. Instead, use $alternatives - ");
}

sub die : method {
	my $self = shift;
	my $message = shift;
	$message = '[NO ERROR DESCRIPTION]' unless defined $message;
	my $this = ref($self);
	my $offset = 1;
	while (my @call = caller($offset++)) {
		next if $call[0] =~ /^(Twilio::|Twilio$)/;
		die "$message at $call[1] line $call[2]\n";
	}
}

1;

__END__
=pod

=head1 NAME

Twilio - Standards compliant interface to the Twilio API

=head1 SYNOPSIS

This module is under development and no POD is yet included completely on purpose. Its goal
on the long term is to follow the Twilio API exactly like the official documentation describes
it for other scripting languages.

In the meantime, if you are looking for a quick alternative, look at L<WWW::Twilio::API> and
L<WWW::Twilio::TwiML> for the Perl part, and read the API documentation at L<https://www.twilio.com/docs/api/rest>.
