package CPANPLUS::Dist::Slackware::Base;

use strict;
use warnings;

our $VERSION = '1.026';

use base qw(CPANPLUS::Dist::Base);

use CPANPLUS::Dist::Slackware::Util qw(run);
use CPANPLUS::Error;

use Config;
use Cwd qw(abs_path cwd);
use File::Fetch;
use File::Spec::Functions qw(catdir catfile);
use IPC::Cmd qw(can_run run);

use Data::Dumper;    # XXX

sub init {
    my $dist = shift;

    my $status = $dist->status;
    my $module = $dist->parent;

    $dist->SUPER::init() or return;

    $status->mk_accessors(qw(_distroprefs));

    eval {
        require CPAN::Distroprefs;
        require CPAN::Meta::YAML;
        *YAML::Module:: = \*CPAN::Meta::YAML::;
        $status->_distroprefs(_find_prefs($module));
    };

    return 1;
}

sub prepare {
    my ( $dist, @params ) = @_;

    my $status = $dist->status;
    my $module = $dist->parent;

    my $prefs = $status->_distroprefs;
    if (defined $prefs) {
        _patch($module, $prefs) or return;
    }

    $dist->SUPER::prepare(@params) or return;

    return $status->prepared(1);
}

sub _find_perl {
    my $perl    = $Config{perlpath};
    my $version = $Config{version};
    for my $path ( $perl . $version, $perl ) {
        if ( -x $path ) {
            return $path;
        }
    }
    return "";
}

sub _find_prefs_dirs {
    return ( "/home/andreas/.cpan/prefs", );
}

sub _find_patches_dir {
    my $prefs_dir = shift;

    return "/home/andreas/.cpan/sources/authors/id/R/RU";
}

sub _find_prefs {
    my $module = shift;

    my $pretty_id = $module->author->cpanid . '/' . $module->package;

    my $module_info = {
        distribution => $pretty_id,
        env          => \%ENV,
        perl         => \&_find_perl,
        perlconfig   => \%Config,
        module       => sub {
            [ map { $_->name } $module->contains() ];
        },
    };

    my $ext_map = {
        yml => 'YAML::Module',
    };

    my @prefs;
    for my $prefs_dir ( _find_prefs_dirs() ) {
        my $finder = CPAN::Distroprefs->find( $prefs_dir, $ext_map );
        while ( my $result = $finder->next ) {
            if ( $result->is_fatal ) {
                error( $result->as_string );
                return;
            }
            if ( $result->is_warning ) {
                msg( $result->as_string );
                next;
            }
            for my $pref ( @{ $result->prefs } ) {
                if ( $pref->matches($module_info) ) {
                    my $pref_info
                        = { file => $result->abs, %{ $pref->data } };
                    push @prefs, $pref_info;
                }
            }
        }
    }
    return \@prefs;
}

sub _fetch_patch {
    my ( $uri, $dir ) = @_;

    my $ff = File::Fetch->new( uri => $uri );
    my $file = $ff->fetch( to => $dir );
    return $file;
}

sub _patch_p_parameter {
    my $file = shift;

    my $fh;
    if (open $fh, '<', $file) {
        while (<$fh>) {
            next unless /^[\*\+]{3}\s(\S+)/;
            return '-p1' if ! -f $1;
        }
        close $fh;
    }

    return '-p0';
}

sub _patch {
    my ( $module, $prefs ) = @_;

    my $patch_cmd = can_run('patch');

    my $cwd = cwd();
    my $wrksrc = $module->status->extract;
    if (!chdir $wrksrc) {
        return
    }

    for my $pref_info (@{$prefs}) {
        my $file    = $pref_info->{file};
        my $comment = $pref_info->{comment};
        my $patches = $pref_info->{patches};
        if ($patches) {
            my $patches_dir = _find_patches_dir("XXX");
            for my $patch ( @{$patches} ) {

                # XXX file_name_is_absolute
                my $file = catfile( $patches_dir, $patch );
                my $p = _patch_p_parameter($file);
                my $cmd = [ $patch_cmd, '-N', $p, '<', $file ];
                msg($comment);
                #print STDERR Dumper($cmd);
                my ($ok, $err, $output) = run(command => $cmd, timeout => 300);
                if ($output) {
                    for (@{$output}) {
                        chomp;
                        msg($_);
                    }
                }
                if ($err) {
                    error($err);
                }
                return if ! $ok;
            }
        }
    }

    chdir $cwd or return;

    return 1;
}

1;
__END__

=head1 NAME

CPANPLUS::Dist::Slackware::Distroprefs - XXX

=head1 VERSION

This document describes CPANPLUS::Dist::Slackware::Distroprefs version 1.025.

=head1 SYNOPSIS

    use CPANPLUS::Dist::Slackware::Distroprefs;

    my $finder = CPANPLUS::Dist::Slackware::Distroprefs->new(
        module => $modobj,
        tag    => '_MYTAG',
        type   => 'txz'
    );

    $filename = $pkgdesc->filename();
    $summary  = $pkgdesc->summary();
    $desc     = $pkgdesc->slack_desc();
    @docfiles = $pkgdesc->docfiles();

=head1 DESCRIPTION

XXX

=head1 SUBROUTINES/METHODS

=over 4

=item B<< CPANPLUS::Dist::Slackware::Distroprefs->new(%attrs) >>

Returns a newly constructed object.

    my $finder = CPANPLUS::Dist::Slackware::Distroprefs->new(
        module => $modobj,
        %attrs
    );

=back

=head1 DIAGNOSTICS

None.

=head1 CONFIGURATION AND ENVIRONMENT

XXX

=head1 DEPENDENCIES

Requires the module CPAN::Distroprefs.

=head1 INCOMPATIBILITIES

None known.

=head1 SEE ALSO

CPANPLUS::Dist::Slackware

=head1 AUTHOR

Andreas Voegele E<lt>voegelas@cpan.orgE<gt>

=head1 BUGS AND LIMITATIONS

Please report any bugs to C<bug-cpanplus-dist-slackware at rt.cpan.org>, or
through the web interface at L<http://rt.cpan.org/>.

=head1 LICENSE AND COPYRIGHT

Copyright 2017 Andreas Voegele

This library is free software; you can redistribute it and/or modify it under
the same terms as Perl itself.

See http://dev.perl.org/licenses/ for more information.

=cut
