use Date::Manip;
use Test::More;


BEGIN { use_ok( 'Date::Manip::Range' ); }
require_ok( 'Date::Manip::Range' );

subtest 'Parsing range strings' => sub {
	my $range = new_ok( 'Date::Manip::Range' );
	
	subtest 'and' => sub {
		ok( $range->parse( '2015-01-01 and 2015-02-15' ), 'Parsed' );
		is( $range->start->printf( '%Y-%m-%d' ), '2015-01-01', 'Start' );
		is( $range->end->printf( '%Y-%m-%d' ), '2015-02-15', 'End' );
	};
	subtest 'to' => sub {
		ok( $range->parse( '2015-01-01 to 2015-02-15' ), 'Parsed' );
		is( $range->start->printf( '%Y-%m-%d' ), '2015-01-01', 'Start' );
		is( $range->end->printf( '%Y-%m-%d' ), '2015-02-15', 'End' );
	};
	subtest 'through' => sub {
		ok( $range->parse( '2015-01-01 through 2015-02-15' ), 'Parsed' );
		is( $range->start->printf( '%Y-%m-%d' ), '2015-01-01', 'Start' );
		is( $range->end->printf( '%Y-%m-%d' ), '2015-02-15', 'End' );
	};
	subtest 'thru' => sub {
		ok( $range->parse( '2015-01-01 thru 2015-02-15' ), 'Parsed' );
		is( $range->start->printf( '%Y-%m-%d' ), '2015-01-01', 'Start' );
		is( $range->end->printf( '%Y-%m-%d' ), '2015-02-15', 'End' );
	};
	subtest 'dash (-)' => sub {
		ok( $range->parse( '2015-01-01 - 2015-02-15' ), 'Parsed' );
		is( $range->start->printf( '%Y-%m-%d' ), '2015-01-01', 'Start' );
		is( $range->end->printf( '%Y-%m-%d' ), '2015-02-15', 'End' );
	};
	subtest 'ellipses (...)' => sub {
		ok( $range->parse( '2015-01-01 ... 2015-02-15' ), 'Parsed' );
		is( $range->start->printf( '%Y-%m-%d' ), '2015-01-01', 'Start' );
		is( $range->end->printf( '%Y-%m-%d' ), '2015-02-15', 'End' );
	};
	subtest 'ellipses (...), no spaces' => sub {
		ok( $range->parse( '2015-01-01...2015-02-15' ), 'Parsed' );
		is( $range->start->printf( '%Y-%m-%d' ), '2015-01-01', 'Start' );
		is( $range->end->printf( '%Y-%m-%d' ), '2015-02-15', 'End' );
	};
	subtest 'dots (..)' => sub {
		ok( $range->parse( '2015-01-01 .. 2015-02-15' ), 'Parsed' );
		is( $range->start->printf( '%Y-%m-%d' ), '2015-01-01', 'Start' );
		is( $range->end->printf( '%Y-%m-%d' ), '2015-02-15', 'End' );
	};
	subtest 'dots (..), no spaces' => sub {
		ok( $range->parse( '2015-01-01..2015-02-15' ), 'Parsed' );
		is( $range->start->printf( '%Y-%m-%d' ), '2015-01-01', 'Start' );
		is( $range->end->printf( '%Y-%m-%d' ), '2015-02-15', 'End' );
	};
	subtest 'between ... and ...' => sub {
		ok( $range->parse( 'between 2015-01-01 and 2015-02-15' ), 'Parsed' );
		is( $range->start->printf( '%Y-%m-%d' ), '2015-01-01', 'Start' );
		is( $range->end->printf( '%Y-%m-%d' ), '2015-02-15', 'End' );
	};
	subtest 'from ... to ...' => sub {
		ok( $range->parse( 'from 2015-01-01 to 2015-02-15' ), 'Parsed' );
		is( $range->start->printf( '%Y-%m-%d' ), '2015-01-01', 'Start' );
		is( $range->end->printf( '%Y-%m-%d' ), '2015-02-15', 'End' );
	};
	subtest 'from ... through ...' => sub {
		ok( $range->parse( 'from 2015-01-01 through 2015-02-15' ), 'Parsed' );
		is( $range->start->printf( '%Y-%m-%d' ), '2015-01-01', 'Start' );
		is( $range->end->printf( '%Y-%m-%d' ), '2015-02-15', 'End' );
	};
	subtest 'from ... thru ...' => sub {
		ok( $range->parse( 'from 2015-01-01 thru 2015-02-15' ), 'Parsed' );
		is( $range->start->printf( '%Y-%m-%d' ), '2015-01-01', 'Start' );
		is( $range->end->printf( '%Y-%m-%d' ), '2015-02-15', 'End' );
	};
	
	subtest 'Mispelled operators' => sub {
		ok( !$range->parse( 'from 2015-01-01 thorough 2015-02-15' ), 'thorough' );
		ok( !$range->parse( 'from 2015-01-01 thu 2015-02-15' ), 'thu' );
		ok( !$range->parse( 'from 2015-01-01-2015-02-15' ), 'No spaces' );
	};

	ok( !$range->parse( 'between and 2015-02-15' ), 'Nothing after between' );

	subtest 'Sort dates' => sub {
		my $range = new_ok( 'Date::Manip::Range' => [parse => '2015-02-15 to 2015-01-01', test => 1] );
		is( $range->start->printf( '%Y-%m-%d' ), '2015-01-01', 'Earliest = start' );
		is( $range->end->printf( '%Y-%m-%d' ), '2015-02-15', 'Latest = end' );
	};
};

subtest 'Constructor' => sub {
	my $range;

	$range = new_ok( 'Date::Manip::Range' => [parse => '2015-01-01 to 2015-02-15'] );
	ok( $range->is_valid(), 'Range string' );

	$range = new_ok( 'Date::Manip::Range' => [parse => ['2015-01-01', '2015-02-15']] );
	ok( $range->is_valid(), 'Range array' );

	$range = new_ok( 'Date::Manip::Range' => [start => Date::Manip::Date->new( '2015-01-01' )] );
	ok( !$range->is_valid(), 'Start date object' );

	$range = new_ok( 'Date::Manip::Range' => [end => Date::Manip::Date->new( '2015-02-15' )] );
	ok( !$range->is_valid(), 'End date object' );

	subtest 'Swap dates' => sub {
		my $range = new_ok( 'Date::Manip::Range' => [
			end   => Date::Manip::Date->new( '2015-01-01' ),
			start => Date::Manip::Date->new( '2015-02-15' ),
		] );
		is( $range->start->printf( '%Y-%m-%d' ), '2015-01-01', 'Earliest = start' );
		is( $range->end->printf( '%Y-%m-%d' ), '2015-02-15', 'Latest = end' );
	};
};

subtest 'Range includes a date' => sub {
	my $range = new_ok( 'Date::Manip::Range' => [parse => '2015-01-01 to 2015-02-15'] );
	
	ok( $range->includes( '2015-01-10' ), 'Date inside of range' );
	ok( !$range->includes( '2014-12-20' ), 'Date before the range' );
	ok( !$range->includes( '2015-03-10' ), 'Date after the range' );

	ok( $range->includes( '2015-01-01' ), 'Inclusive start' );
	$range->include_start( 0 );
	ok( !$range->includes( '2015-01-01' ), 'Exclusive start' );

	ok( $range->includes( '2015-02-15' ), 'Inclusive end' );
	$range->include_end( 0 );
	ok( !$range->includes( '2015-02-15' ), 'Exclusive end' );
};

subtest 'Invalid dates' => sub {
	my $range = new_ok( 'Date::Manip::Range' );

	ok( !$range->parse( 'invalid to 2015-02-15' ), 'Start date' );
	ok( !$range->parse( '2015-01-01 to invalid' ), 'End date' );
};

done_testing;
