#!/usr/bin/perl

# Copyright 2001-2010, Paul Johnson (pjcj@cpan.org)

# This software is free.  It is licensed under the same terms as Perl itself.

# The latest version of this software should be available from my homepage:
# http://www.pjcj.net

require 5.006001;

use strict;
use warnings;

use Cwd;
use ExtUtils::MakeMaker;
use ExtUtils::Manifest "maniread";
use File::Copy;

$| = 1;

my $Version  = "0.67";
my $Date     = "8th May 2010";
my $Author   = 'pjcj@cpan.org';

my @perlbug  = ("perlbug", "-a", $Author,
                           "-s", "Installation of Devel::Cover $Version");
my $Perlbug  = join " ", map { / / ? "'$_'" : $_ } @perlbug;

my @files    = sort keys %{maniread()}, "lib/Devel/Cover/Inc.pm";
my @versions = grep !/^(?:README|Makefile\.PL|\.gitignore)$/, @files;

my $base     = getcwd;

my %inc      = map { -d $_ ? (($_ eq "." ? $_ : Cwd::abs_path($_)) => 1) : () }
                   @INC;
my @inc      = sort keys %inc;

open I, ">lib/Devel/Cover/Inc.pm"
    or die "Cannot open lib/Devel/Cover/Inc.pm: $!";
print I <<"EOI";
# Copyright 2001-2010, Paul Johnson (pjcj\@cpan.org)

# This software is free.  It is licensed under the same terms as Perl itself.

# The latest version of this software should be available from my homepage:
# http://www.pjcj.net

# This file was automatically generated by Makefile.PL.

package Devel::Cover::Inc;

use strict;
use warnings;

our \$VERSION      = "$Version";
our \$Perl_version = '$]';
our \$Base         = '$base';
our \@Inc          = qw( @inc );
chomp (our \$Perl  = <<'EOV');  # Careful with \\\\ in the path
$^X
EOV

if (\$Perl_version ne \$])
{
    print STDERR <<"EOM";

This version of Devel::Cover was built with Perl version \$Perl_version.
It is now being run with Perl version \$].
Attempting to make adjustments, but you may find that some of your modules do
not have coverage data collected.  You may need to alter the +-inc, +-ignore
and +-select options.

EOM
    eval "use Cwd";
    my \%inc = map { -d \$_ ? ((\$_ eq "." ? \$_ : Cwd::abs_path(\$_)) => 1) : () }
                   \@INC;
    \@Inc = sort keys \%inc;
}

# TODO - check for threadedness, 64bits etc. ?

1
EOI
close I or die "Cannot close lib/Devel/Cover/Inc.pm: $!";

print "Writing tests ........ ";

unless (-d "t")
{
    mkdir "t" or die "Cannot mkdir t: $!";
}

opendir D, "tests" or die "Cannot opendir tests: $!";
for my $t (readdir D)
{
    next unless -f "tests/$t";
    next if $t =~ /\.(pm|pl|org|bak|uncoverable)$/;
    next if $t =~ /~$/;

    if ($t =~ /\.t/)
    {
        copy("tests/$t", "t/$t") or die "Cannot copy tests/$t to t/$t: $!";
        next
    }

    open T, ">t/a$t.t" or die "Cannot open t/a$t.t: $!";
    print T <<EOT;
#!$^X

# Copyright 2002-2010, Paul Johnson (pjcj\@cpan.org)

# This software is free.  It is licensed under the same terms as Perl itself.

# The latest version of this software should be available from my homepage:
# http://www.pjcj.net

use strict;
use warnings;

use lib "$base/lib";
use lib "$base/blib/lib";
use lib "$base/blib/arch";
use lib "$base/t";

use Devel::Cover::Test $Version;

Devel::Cover::Test->new("$t");
EOT
    close T or die "Cannot open t/a$t.t: $!";
}
closedir D or die "Cannot closedir tests: $!";

print "done\n\n";

my $e;

print "checking for Storable.pm                    ........ ";

$e = <<EOM;

Storable.pm is required to store the coverage database.  You can
download Storable from CPAN.

EOM

eval "use Storable";
if (my $m = $INC{"Storable.pm"})
{
    my $v = eval { no warnings; $Storable::VERSION };
    print "$v     $m\n";
}
else
{
    print "not found\n\n$e\n";
}

print "checking for Digest::MD5.pm                 ........ ";

$e = <<EOM;

Digest::MD5.pm is required to check whether covered files have changed.
You can download Digest::MD5 from CPAN.

EOM

eval "use Digest::MD5";
if (my $m = $INC{"Digest/MD5.pm"})
{
    my $v = eval { no warnings; $Digest::MD5::VERSION };
    print "$v     $m\n";
}
else
{
    print "not found\n\n$e\n";
}

print "checking for Template.pm       version 2.00 ........ ";

$e = <<EOM;

Template.pm 2.00 is required to run the some HTML backends to cover and
for cpancover.  Unless you have specific requirements this should not be
a problem, but you will not be able to use these reports until you
install the Template Toolkit, available from CPAN.  In the meantime you
may continue to use the rest of Devel::Cover.

EOM

eval "use Template";
if (my $m = $INC{"Template.pm"})
{
    my $v = eval { no warnings; $Template::VERSION };
    print $v < 2.00 ? "$v\n\n$e\n" : "$v     $m\n";
}
else
{
    print "not found\n\n$e\n";
}

print "checking for PPI::HTML         version 1.07 ........ ";

$e = <<EOM;

One of PPI::HTML 1.07 or Perl::Tidy 20060719 is required to add syntax
highlighting to some HTML backends to cover and for cpancover.  Unless
you have specific requirements this should not be a problem, but you will
not be able to use syntax highlighting in these reports until you install
PPI::HTML or Perl::Tidy, available from the CPAN.  In the meantime you
may continue to use the rest of Devel::Cover.

EOM

eval "use PPI::HTML";
if (my $m = $INC{"PPI/HTML.pm"})
{
    my $v = eval { no warnings; $PPI::HTML::VERSION };
    print $v < 1.07 ? "$v\n\n$e\n" : "$v     $m\n";
}
else
{
    print "not found\n\n$e\n";
}

print "checking for Perl::Tidy        version 20060719 .... ";

$e = <<EOM;

One of PPI::HTML 1.07 or Perl::Tidy 20060719 is required to add syntax
highlighting to some HTML backends to cover and for cpancover.  Unless
you have specific requirements this should not be a problem, but you will
not be able to use syntax highlighting in these reports until you install
PPI::HTML or Perl::Tidy, available from the CPAN.  In the meantime you
may continue to use the rest of Devel::Cover.

EOM

eval "use Perl::Tidy";
if (my $m = $INC{"Perl/Tidy.pm"})
{
    my $v = eval { no warnings; $Perl::Tidy::VERSION };
    print $v < 20060719 ? "$v\n\n$e\n" : "$v $m\n";
}
else
{
    print "not found\n\n$e\n";
}

print "checking for Pod::Coverage.pm  version 0.06 ........ ";

$e = <<EOM;

Pod::Coverage.pm 0.06 is required to do pod coverage.  This will tell
you how well you have documented your modules.  Pod coverage will be
unavailable until you install this module, available from CPAN.  In the
meantime, you may continue to use the rest of Devel::Cover.
EOM

eval "use Pod::Coverage";
if (my $m = $INC{"Pod/Coverage.pm"})
{
    my $v = eval { no warnings; $Pod::Coverage::VERSION };
    print $v < 0.06 ? "$v\n\n$e\n" : "$v     $m\n";

    print "checking for Pod::Coverage::CountParents.pm ........ ";

    $e = <<EOM;

    Pod::Coverage::CountParents.pm is used for Pod coverage if it is available.
    We will fall back to using Pod::Coverage.pm.  If you want to use
    Pod::Coverage::CountParents.pm, just install it from CPAN.
EOM

    eval "use Pod::Coverage::CountParents";
    if (my $m = $INC{"Pod/Coverage/CountParents.pm"})
    {
        my $v = eval { no warnings; $Pod::Coverage::CountParents::VERSION };
        $v ||= "    ";
        print "$v     $m\n";
    }
    else
    {
        print "not found\n\n$e\n";
    }

}
else
{
    print "not found\n\n$e\n";
}

print "checking for Test::Differences.pm           ........ ";

$e = <<EOM;

Test::Differences.pm is used to display output from failed tests.
Hopefully there won't be any failed tests, but if there are you will get
output that may not be a model of clarity.  If you do get test failures
and you fancy helping me by debugging them, then you might like to
consider installing Test::Differences.  You can download
Test::Differences from CPAN.

EOM

eval "use Test::Differences";
if (my $m = $INC{"Test/Differences.pm"})
{
    my $v = eval { no warnings; $Test::Differences::VERSION };
    print "$v     $m\n";
}
else
{
    print "not found\n\n$e\n";
}

my $latest_tested = "5.012000";
print <<EOM if $] > $latest_tested;

Devel::Cover $Version has not been tested with perl $].
Testing will take place against expected output from perl $latest_tested.
You may well find failing tests.

EOM

if ($] < 5.008)
{
    print <<EOM;

Devel::Cover $Version is not fully functional on perl $].  It should
mostly work, but there are some constructs for which coverage will not
be collected, and you may well encounter bugs which have been fixed in
subsequent versions of perl.  Perl versions 5.8.1 and above should work
better.

EOM

    print <<EOM if $^O eq "MSWin32";
And things are even worse under Windows.  You may well find random bugs
of various severities.

EOM
}
elsif ($] < 5.008002)
{
    print <<EOM;

Devel::Cover $Version mostly works on perl $], but you may encounter
strange behaviours (bugs) which have been fixed in subsequent versions
of perl.  This is particularly true if you are running a threaded perl.
Perl versions 5.8.2 and above should work better.

EOM
}

print "\n" x 3;

$ExtUtils::MakeMaker::Verbose = 0;

WriteMakefile
(
    NAME      => "Devel::Cover",
    VERSION   => $Version,
    AUTHOR    => 'Paul Johnson (pjcj@cpan.org)',
    ABSTRACT  => "Code coverage metrics for Perl",
    DIR       => [],
    EXE_FILES => [ qw( cover gcov2perl cpancover ) ],
    PREREQ_PM => {
                     Storable      => 0,
                     "Digest::MD5" => 0,
                 },
    dist      => { COMPRESS => "gzip --best --force" },
    clean     => { FILES => join " ",  "t/*" },
    depend    => { distdir => "@files" },
    realclean => $] < 5.008008 ?
                 {
                     FILES => "lib/Devel/Cover/Inc.pm",
                     POSTOP => "\$(RM_RF) cover_db t"
                 } :
                 { FILES => "lib/Devel/Cover/Inc.pm cover_db t" },
);

print "\n";
print <<EOM if 0;

--------------------------------------------------------------------------------

I like to have some idea of the distribution of this software.  To that end I
would be very grateful for mail from you.  This will be used only to satisfy my
curiosity and to help me make decisions which may involve changes to the
module.

If you can run perlbug you can send me a success report with "make ok".
Failure reports with "make nok" are also appreciated.

If you have any questions or comments, mailing list details are available in the
README file, or send mail to me directly.

EOM

sub MY::libscan
{
    my ($self, $path) = @_;
    (my $p = $path) =~ s/^\$\(INST_LIB\)/lib/;  # 5.6.1
    $p =~ s|\\|/|g if $^O eq "MSWin32";
    # print "$path $p\n";
    my $wanted = -d $p;                         # 5.9.0
    for my $f (@files)
    {
        # print "$p - $f\n";
        last if $wanted ||= $p =~ /$f$/;
    }
    $wanted && $path;
}

sub MY::postamble
{
    qq[
SET_VERSION = \$(PERL) -pi \\
    -e 's/(^\\s*(?:our\\s+)\\\$\$VERSION = ")\\d+\\.\\d+(";)/\$\${1}$Version\$\$2/;' \\
    -e 's/(Version )\\d+\\.\\d+( - ).*/\$\${1}$Version\$\${2}$Date/;' \\
    -e 's/(Devel::Cover Version )\\d+\\.\\d+/\$\${1}$Version/;' \\
    -e 's/(\\buse Devel::Cover(?:::\\w+)*\\s+)\\d+\\.\\d+/\$\${1}$Version/;'

tags : @files
\t ptags @files

@versions : Makefile.PL
\t \$(SET_VERSION) @versions

README : lib/Devel/Cover.pm
\t TERMCAP= COLUMNS=80 pod2text -s lib/Devel/Cover.pm | \\
    \$(PERL) -n \\
    -e 'print if (/NAME/         ... /^[A-Z ]+\$\$/) =~ /^\\d+\$\$/;' \\
    -e 'print if (/SYNOPSIS/     ... /^[A-Z ]+\$\$/) =~ /^\\d+\$\$/;' \\
    -e 'print if (/DESCRIPTION/  ... /^[A-Z ]+\$\$/) =~ /^\\d+\$\$/;' \\
    -e 'print if (/REQUIREMENTS/ ... /^[A-Z ]+\$\$/) =~ /^\\d+\$\$/;' \\
    > README

ppm : ppd pure_all
\t tar cf Devel-Cover.tar blib
\t gzip --best --force Devel-Cover.tar
\t \$(PERL) -pi.bak \\
    -e 's/(OS NAME=")[^"]*/\$\$1MSWin32/;' \\
    -e 's/(ARCHITECTURE NAME=")[^"]*/\$\$1MSWin32-x86-multi-thread/;' \\
    -e 's/(CODEBASE HREF=")[^"]*/\$\$1Devel-Cover.tar.gz/;' \\
    Devel-Cover.ppd

TAINT = -T
COVER_OPTIONS =

_run : pure_all
\t \$(PERL) \$(TAINT) -Iblib/lib -Iblib/arch -MDevel::Cover=-merge,0,`\$(PERL) -e '\$\$l = qx|grep __COVER__ \$\$ARGV[0]|; \$\$l =~ /__COVER__\\s+criteria\\s+(.*)/; (\$\$c = \$\$1 || "all") =~ s/\\s+/,/g; \$\$p = "\$\$1," if \$\$l =~ /__COVER__\\s+test_parameters\\s+(.*)/; print "\$\$p-coverage,\$\$c"' tests/\$(TEST)`,\$(COVER_OPTIONS) tests/\$(TEST)

COVER_PARAMETERS = \$(PERL) -e '\$\$l = qx|grep __COVER__ \$\$ARGV[0]|; \$\$u = "-uncoverable_file \$\$1" if \$\$l =~ /__COVER__\\s+uncoverable_file\\s+(.*)/; (\$\$p) = \$\$l =~ /__COVER__\\s+cover_parameters\\s+(.*)/; print "\$\$u \$\$p"' tests/\$(TEST)

html : _run
\t \$(PERL) -Mblib cover `\$(COVER_PARAMETERS)` -report html

basic : _run
\t \$(PERL) -Mblib cover `\$(COVER_PARAMETERS)` -report html_basic

out : _run
\t \$(PERL) -Mblib cover `\$(COVER_PARAMETERS)` -report text > \$(TEST).out

text : out
\t \$(VISUAL) \$(TEST).out

wrun : pure_all
\t \$(PERL) \$(TAINT) -Iblib/lib -Iblib/arch -MDevel::Cover=-ignore,blib,-merge,0 tests/\$(TEST)

DB = cover_db

dump :
\t \$(PERL) -Mblib cover -dump_db \$(DB)

FONT = 8x13
FONT = -sun-screen-medium-r-normal-*-*-70-*-*-m-*-sun-fontspecific
FONT = "Bitstream Vera Sans Mono 8"
GEOM = 260x85+0+0

diff : out
\t \$(PERL) makeh strip_criterion 'time' \$(TEST).out
\t \$(PERL) makeh strip_criterion ' pod' \$(TEST).out
\t gvim -geom \$(GEOM) -d -font \$(FONT) `\$(PERL) -Mblib -MDevel::Cover::Test -e '\$\$t = Devel::Cover::Test->new("\$(TEST)", run_test_at_end => 0); print \$\$t->cover_gold'` \$(TEST).out

gold : pure_all
\t \$(PERL) create_gold \$(TEST)

all_test :
\t exec \$(PERL) all_versions make test

all_gold :
\t rm -rf test_output
\t exec \$(PERL) all_versions make gold

mani :
\t \$(PERL) -i.bak -ne '\$\$d = "test_output/cover"; print unless /^\$\$d/; END { open M, ">>MANIFEST"; opendir D, \$\$d; print M "\$\$d/\$\$_\\n" for sort grep -f "\$\$d/\$\$_", readdir D }' MANIFEST

ok :
\t \@$Perlbug -okay  || echo "Please send your report manually to $Author"

nok :
\t \@$Perlbug -nokay || echo "Please send your report manually to $Author"
    ]
}
