package Bencher::Scenario::LevenshteinModules;

our $DATE = '2016-01-10'; # DATE
our $VERSION = '0.04'; # VERSION

use 5.010001;
use strict;
use utf8;
use warnings;

our $scenario = {
    summary => 'Benchmark various modules calculating the Levenshtein edit distance',
    participants => [
        {
            fcall_template => "PERLANCAR::Text::Levenshtein::editdist(<word1>, <word2>)",
        },
        {
            fcall_template => "Text::Levenshtein::fastdistance(<word1>, <word2>)",
        },
        {
            fcall_template => "Text::Levenshtein::XS::distance(<word1>, <word2>)",
        },
        {
            fcall_template => "Text::LevenshteinXS::distance(<word1>, <word2>)",
        },
        {
            fcall_template => "Text::Levenshtein::Damerau::PP::pp_edistance(<word1>, <word2>)",
        },
        {
            fcall_template => "Text::Levenshtein::Damerau::XS::xs_edistance(<word1>, <word2>)",
        },
    ],
    datasets => [
        { name=>"a",       args => {word1=>"a"      , word2=>"aa"},      result => 1 },
        { name=>"foo",     args => {word1=>"foo"    , word2=>"bar"},     result => 3 },
        { name=>"program", args => {word1=>"program", word2=>"porgram"}, result => 2 },
        { name=>"reve"   , args => {word1=>"reve"   , word2=>"rêves"},   result => 3, tags=>['unicode'] },
        { name=>"euro"   , args => {word1=>"Euro"   , word2=>"€uro"},    result => 2, tags=>['unicode'] },
    ],
};

1;
# ABSTRACT: Benchmark various modules calculating the Levenshtein edit distance

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::LevenshteinModules - Benchmark various modules calculating the Levenshtein edit distance

=head1 VERSION

This document describes version 0.04 of Bencher::Scenario::LevenshteinModules (from Perl distribution Bencher-Scenario-LevenshteinModules), released on 2016-01-10.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m LevenshteinModules

To run module startup overhead benchmark:

 % bencher --module-startup -m LevenshteinModules

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 BENCHMARKED MODULES

L<PERLANCAR::Text::Levenshtein> 0.02

L<Text::Levenshtein> 0.13

L<Text::Levenshtein::XS> 0.502

L<Text::LevenshteinXS> 0.03

L<Text::Levenshtein::Damerau::PP> 0.25

L<Text::Levenshtein::Damerau::XS> 3.0

=head1 BENCHMARK PARTICIPANTS

=over

=item * PERLANCAR::Text::Levenshtein::editdist (perl_code)

Function call template:

 PERLANCAR::Text::Levenshtein::editdist(<word1>, <word2>)



=item * Text::Levenshtein::fastdistance (perl_code)

Function call template:

 Text::Levenshtein::fastdistance(<word1>, <word2>)



=item * Text::Levenshtein::XS::distance (perl_code)

Function call template:

 Text::Levenshtein::XS::distance(<word1>, <word2>)



=item * Text::LevenshteinXS::distance (perl_code)

Function call template:

 Text::LevenshteinXS::distance(<word1>, <word2>)



=item * Text::Levenshtein::Damerau::PP::pp_edistance (perl_code)

Function call template:

 Text::Levenshtein::Damerau::PP::pp_edistance(<word1>, <word2>)



=item * Text::Levenshtein::Damerau::XS::xs_edistance (perl_code)

Function call template:

 Text::Levenshtein::Damerau::XS::xs_edistance(<word1>, <word2>)



=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.22.0 >>, CPU: I<< Intel(R) Core(TM) i5-2557M CPU @ 1.70GHz (2 cores) >>, OS: I<< GNU/Linux LinuxMint version 17 >>, OS kernel: I<< Linux version 3.13.0-32-generic >>.

Benchmark with default option:

 +-----+----------------------------------------------------------------------------------+-------------+------------+---------+---------+
 | seq | name                                                                             | rate        | time       | errors  | samples |
 +-----+----------------------------------------------------------------------------------+-------------+------------+---------+---------+
 | 7   | {dataset=>"program",participant=>"Text::Levenshtein::fastdistance"}              | 9.12e+03    | 110μs      | 4.6e-08 | 27      |
 | 22  | {dataset=>"program",participant=>"Text::Levenshtein::Damerau::PP::pp_edistance"} | 1.016e+04   | 98.4μs     | 2.7e-08 | 20      |
 | 2   | {dataset=>"program",participant=>"PERLANCAR::Text::Levenshtein::editdist"}       | 15655       | 63.8773μs  | 0       | 35      |
 | 8   | {dataset=>"reve",participant=>"Text::Levenshtein::fastdistance"}                 | 2.11e+04    | 47.3μs     | 5.3e-08 | 20      |
 | 23  | {dataset=>"reve",participant=>"Text::Levenshtein::Damerau::PP::pp_edistance"}    | 21791.9     | 45.8886μs  | 0       | 20      |
 | 24  | {dataset=>"euro",participant=>"Text::Levenshtein::Damerau::PP::pp_edistance"}    | 2.28e+04    | 43.9μs     | 5.3e-08 | 20      |
 | 9   | {dataset=>"euro",participant=>"Text::Levenshtein::fastdistance"}                 | 2.422e+04   | 41.28μs    | 1.3e-08 | 21      |
 | 3   | {dataset=>"reve",participant=>"PERLANCAR::Text::Levenshtein::editdist"}          | 3.49e+04    | 28.7μs     | 5.3e-08 | 20      |
 | 21  | {dataset=>"foo",participant=>"Text::Levenshtein::Damerau::PP::pp_edistance"}     | 4.137e+04   | 24.17μs    | 5.7e-09 | 27      |
 | 6   | {dataset=>"foo",participant=>"Text::Levenshtein::fastdistance"}                  | 4.17e+04    | 24μs       | 2e-08   | 20      |
 | 4   | {dataset=>"euro",participant=>"PERLANCAR::Text::Levenshtein::editdist"}          | 42193.6     | 23.7003μs  | 0       | 24      |
 | 1   | {dataset=>"foo",participant=>"PERLANCAR::Text::Levenshtein::editdist"}           | 61872.5     | 16.1623μs  | 0       | 20      |
 | 20  | {dataset=>"a",participant=>"Text::Levenshtein::Damerau::PP::pp_edistance"}       | 120851      | 8.27466μs  | 0       | 33      |
 | 5   | {dataset=>"a",participant=>"Text::Levenshtein::fastdistance"}                    | 1.27e+05    | 7.87μs     | 1.3e-08 | 20      |
 | 0   | {dataset=>"a",participant=>"PERLANCAR::Text::Levenshtein::editdist"}             | 2.48e+05    | 4.03μs     | 5e-09   | 20      |
 | 27  | {dataset=>"program",participant=>"Text::Levenshtein::Damerau::XS::xs_edistance"} | 3.37e+05    | 2.96μs     | 3.3e-09 | 20      |
 | 12  | {dataset=>"program",participant=>"Text::Levenshtein::XS::distance"}              | 3.61e+05    | 2.77μs     | 3.3e-09 | 20      |
 | 28  | {dataset=>"reve",participant=>"Text::Levenshtein::Damerau::XS::xs_edistance"}    | 4.36e+05    | 2.29μs     | 3.1e-09 | 23      |
 | 13  | {dataset=>"reve",participant=>"Text::Levenshtein::XS::distance"}                 | 4.546e+05   | 2.2μs      | 6.8e-10 | 30      |
 | 29  | {dataset=>"euro",participant=>"Text::Levenshtein::Damerau::XS::xs_edistance"}    | 4.81e+05    | 2.08μs     | 7.9e-10 | 22      |
 | 14  | {dataset=>"euro",participant=>"Text::Levenshtein::XS::distance"}                 | 499024      | 2.00391μs  | 0       | 29      |
 | 11  | {dataset=>"foo",participant=>"Text::Levenshtein::XS::distance"}                  | 5.34e+05    | 1.87μs     | 6.7e-10 | 31      |
 | 26  | {dataset=>"foo",participant=>"Text::Levenshtein::Damerau::XS::xs_edistance"}     | 5.36e+05    | 1.87μs     | 7e-10   | 28      |
 | 10  | {dataset=>"a",participant=>"Text::Levenshtein::XS::distance"}                    | 6.9047e+05  | 1.4483μs   | 1.9e-11 | 64      |
 | 25  | {dataset=>"a",participant=>"Text::Levenshtein::Damerau::XS::xs_edistance"}       | 7.24e+05    | 1.38μs     | 1.7e-09 | 20      |
 | 17  | {dataset=>"program",participant=>"Text::LevenshteinXS::distance"}                | 2.55802e+06 | 0.390927μs | 0       | 23      |
 | 19  | {dataset=>"euro",participant=>"Text::LevenshteinXS::distance"}                   | 3.46426e+06 | 0.288662μs | 0       | 25      |
 | 18  | {dataset=>"reve",participant=>"Text::LevenshteinXS::distance"}                   | 4.11262e+06 | 0.243154μs | 0       | 20      |
 | 16  | {dataset=>"foo",participant=>"Text::LevenshteinXS::distance"}                    | 5.09501e+06 | 0.196271μs | 0       | 24      |
 | 15  | {dataset=>"a",participant=>"Text::LevenshteinXS::distance"}                      | 6.39672e+06 | 0.15633μs  | 0       | 68      |
 +-----+----------------------------------------------------------------------------------+-------------+------------+---------+---------+


Benchmark module startup overhead:

 +-----+--------------------------------+---------+-------------------+---------+---------+
 | seq | name                           | time    | mod_overhead_time | errors  | samples |
 +-----+--------------------------------+---------+-------------------+---------+---------+
 | 4   | Text::LevenshteinXS            | 3e+01ms | 2e+01ms           | 0.0011  | 21      |
 | 2   | Text::Levenshtein              | 24ms    | 12ms              | 0.00053 | 20      |
 | 3   | Text::Levenshtein::XS          | 2e+01ms | 7ms               | 0.00064 | 21      |
 | 5   | Text::Levenshtein::Damerau::PP | 19ms    | 6.3ms             | 0.00031 | 20      |
 | 6   | Text::Levenshtein::Damerau::XS | 16ms    | 4.1ms             | 0.00028 | 20      |
 | 1   | PERLANCAR::Text::Levenshtein   | 1e+01ms | 0.5ms             | 0.00061 | 20      |
 | 0   | perl -e1 (baseline)            | 12ms    | 0ms               | 0.00036 | 21      |
 +-----+--------------------------------+---------+-------------------+---------+---------+

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenario-LevenshteinModules>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenario-LevenshteinModules>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenario-LevenshteinModules>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
