# NAME

Passwd::Keyring::PWSafe3 - Password storage based on Password Safe encrypted files

# VERSION

Version 1.0000

# SYNOPSIS

Password Safe implementation of [Passwd::Keyring](https://metacpan.org/pod/Passwd%3A%3AKeyring). Passwords are
stored in the [Password Safe](http://passwordsafe.sourceforge.net)
encrypted file.

    use Passwd::Keyring::PWSafe3;

    my $keyring = Passwd::Keyring::PWSafe3->new(
         app=>"blahblah scraper",
         group=>"Johnny web scrapers",
         file=>"/home/joe/secrets.pwsafe3",        # ~/passwd-keyring.pwsafe3 by default
         master_password=>"very secret password",  # Or callback. See below
    );

    my $username = "John";  # or get from .ini, or from .argv...

    my $password = $keyring->get_password($username, "blahblah.com");
    unless( $password ) {
        $password = <somehow interactively prompt for password>;

        # securely save password for future use
        $keyring->set_password($username, $password, "blahblah.com");
    }

    login_somewhere_using($username, $password);
    if( password_was_wrong ) {
        $keyring->clear_password($username, "blahblah.com");
    }

# DESCRIPTION

This module does not require Password Safe to be installed, and can be
used as generic "store many passwords in file encrypted with single
master password" storage. Password Safe GUI, if installed, may help
the user to review, modify, or delete saved passwords.

> Official GUIs can be freely downloaded from [the official
> site](http://passwordsafe.sourceforge.net) - both Windows and (beta)
> Linux versions are available. Apart from them there exist
> various [compatible tools](http://passwordsafe.sourceforge.net/relatedprojects.shtml),
> for example <Pasaffe for Gnome|https://launchpad.net/pasaffe>  or
> <PwSafe for Mac|http://passwordsafe.sourceforge.net/relatedprojects.shtml>.

Actual handling of Password Safe format is based on [Crypt::PWSafe3](https://metacpan.org/pod/Crypt%3A%3APWSafe3)
module. Passwd::Keyring::PWSafe3 just wraps it into the interface
compatible with other Passwd::Keyring backends.

See [Passwd::Keyring::Auto::KeyringAPI](https://metacpan.org/pod/Passwd%3A%3AKeyring%3A%3AAuto%3A%3AKeyringAPI) for detailed comments
on keyring methods (this document is installed with
`Passwd::Keyring::Auto` package).

# CAVEATS

Underlying module ([Crypt::PWSafe3](https://metacpan.org/pod/Crypt%3A%3APWSafe3)) in fact rewrites the whole file
on every save and keeps all passwords cached in memory while active.
This means, that any attempts to use the file paralelly from a few
programs, or from a few objects within one program, are doomed to
cause lost updates. Also, all passwords from the file are kept in
(unprotected) memory while keyring object is active. Therefore, it is
recommended to use separate .psafe3 file for Passwd::Keyring::PWSafe3,
not mixing it with normal Password Safe database, and to keep keyring
object for a short time only, especially if modifications happen.

There are some limitations in [Crypt::PWSafe3](https://metacpan.org/pod/Crypt%3A%3APWSafe3) handling of Password
Safe format. Passwords are read and saved properly and it is possible
to alternate using them from perl, and via Password Safe GUI, but some
less important aspects of the format, like password expiraton policy,
may be ignored. Refer to [Crypt::PWSafe3](https://metacpan.org/pod/Crypt%3A%3APWSafe3) docs for more details.

# DATA MAPPING

Group name is mapped to Password Safe folder.

Realm is mapped as password title.

Username and password are ... well, used as username and password.

# SUBROUTINES/METHODS

## new(app=>'app name', group=>'passwords folder', file=>'pwsafe3 file', master\_password=>'secret or callback', lazy\_save=>1)

Initializes the processing. Croaks if Crypt::PWSafe3 is not installed or
master password is invalid. May create password file if it is missing.

Handled parameters:

- app

    Symbolic application name (used in password notes)

- group

    Name for the password group (used as folder name)

- file

    Location of .pwsafe3 file. If not given, `passwd-keyring.pwsafe3` in
    user home directory is used. Will be created if does not exist. Note:
    absolute path is required, relative paths are very error prone.

- master\_password 

    Password required to unlock the file. Can be given as string, or
    as callback returning a string (usually some way of interactively
    asking user for the password).  The callback gets two parameters: app
    and file.

    If this param is missing, module will prompt interactively for this
    password using console prompt.

- lazy\_save

    if given, asks not to save the file after every change (saving is
    fairly time consuming), but only when $keyring->save is called or when
    keyring is destroyed.

Note: it of course does not make much sense to keep app passwords in
encrypted storage if master password is saved in plain text. The
module most natural usage is to interactively ask for master password
(and use it to protect noticeable number of application-specific
passwords).

Ideas of how to workaround this obstacle are welcome. I loosely
consider either caching master password per desktop session
(implementing sht. similar to ssh-agent/gpg-agent or using one of
those somehow), or integrating the tool with PAM to use actual system
password, or both - but while it seems doable on Linux, cross platform
solution is not so easy.

## set\_password(username, password, realm)

Sets (stores) password identified by given realm for given user 

## get\_password($user\_name, $realm)

Reads previously stored password for given user in given app.
If such password can not be found, returns undef.

## clear\_password($user\_name, $realm)

Removes given password (if present)

## save

Saves unsaved changes, if any are present.

Important only when lazy\_save was given in constructor.

## is\_persistent

Returns info, whether this keyring actually saves passwords persistently.

(true in this case)

# AUTHOR

Marcin Kasperski

# BUGS

Please report any bugs or feature requests to
issue tracker at [https://helixteamhub.cloud/mekk/projects/perl/issues](https://helixteamhub.cloud/mekk/projects/perl/issues).

# SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Passwd::Keyring::PWSafe3

You can also look for information at:

[http://search.cpan.org/~mekk/Passwd-Keyring-PWSafe3/](http://search.cpan.org/~mekk/Passwd-Keyring-PWSafe3/)

Source code is tracked at:

[https://helixteamhub.cloud/mekk/projects/perl/repositories/keyring-pwsafe3](https://helixteamhub.cloud/mekk/projects/perl/repositories/keyring-pwsafe3)

# LICENSE AND COPYRIGHT

Copyright 2012 Marcin Kasperski.

This program is free software; you can redistribute it and/or modify it
under the terms of either: the GNU General Public License as published
by the Free Software Foundation; or the Artistic License.

See http://dev.perl.org/licenses/ for more information.
