package # hide from PAUSE
App::YTDL::Videos;

use warnings;
use strict;
use 5.010000;

use Exporter qw( import );
our @EXPORT_OK = qw( from_arguments_to_choices );

use JSON                   qw( decode_json );
use List::MoreUtils        qw( any none );
use Term::ANSIScreen       qw( :cursor :screen );
use Term::Choose           qw( choose );
use Term::ReadLine::Simple qw();
use URI                    qw();
use URI::Escape            qw( uri_unescape );

use App::YTDL::Data         qw( wrapper_get get_download_info_as_json );
use App::YTDL::Data_Extract qw( add_entry_to_info_hash json_to_hash );
use App::YTDL::Helper       qw( insert_sep );
use App::YTDL::Config       qw( sort_order_list );


sub from_arguments_to_choices {
    my ( $opt, @ids ) = @_;
    my $info = {};
    my $invalid_char = $opt->{invalid_char};
    my $more = 0;
    for my $id ( @ids ) {
        $opt->{view_count_length} = 0;
        if ( my $channel_id = _user_id( $opt, $id ) ) {
            my $tmp = _id_to_tmp_info_hash( $opt, 'CL', $channel_id );
            my $is_youtube = 1;
            _choose_videos_and_add_to_info_hash( $opt, $info, $tmp, $is_youtube );
        }
        elsif ( my $playlist_id = _playlist_id( $opt, $id ) ) {
            my $tmp = _id_to_tmp_info_hash( $opt, 'PL', $playlist_id );
            my $is_youtube = 1;
            _choose_videos_and_add_to_info_hash( $opt, $info, $tmp, $is_youtube );
        }
        elsif ( my $more_ids = _more_ids( $opt, $id ) ) {
            my $tmp = _more_url_to_tmp_info_hash( $opt, $more_ids );
            my $is_youtube = 1;
            _choose_videos_and_add_to_info_hash( $opt, $info, $tmp, $is_youtube );
        }
        elsif ( my $video_id = _video_id( $opt, $id )  ) {
            $info->{$video_id}{extractor}     = 'youtube';
            $info->{$video_id}{extractor_key} = 'Youtube';
        }
        else {
            my $tmp = _non_yt_id_to_tmp_info_hash( $opt, $id );
            my $is_youtube = 0;
            my @keys = keys %$tmp;
            if ( @keys == 1 ) {
                my $video_id = $keys[0];
                $info->{$video_id} = $tmp->{$video_id};
            }
            else {
                _choose_videos_and_add_to_info_hash( $opt, $info, $tmp, $is_youtube );
            }
        }
    }
    return $info;
}


sub _video_id {
    my ( $opt, $id ) = @_;
    my $invalid_char = $opt->{invalid_char};
    if ( ! $id ) {
        return;
    }
    if ( $id =~ m{^[\p{PerlWord}-]{11}\z} ) {
        return $id;
    }
    if ( $id !~ $opt->{yt_regexp} ) {
        return;
    }
    elsif ( $id =~ m{/.*?[?&;!](?:v|video_id)=([^$invalid_char]+)} ) {
        return $1;
    }
    elsif ( $id =~ m{/(?:e|v|embed)/([^$invalid_char]+)} ) {
        return $1;
    }
    elsif ( $id =~ m{#p/(?:u|search)/\d+/([^&?/]+)} ) {
        return $1;
    }
    elsif ( $id =~ m{youtu.be/([^$invalid_char]+)} ) {
        return $1;
    }
    return;
}

sub _playlist_id {
    my ( $opt, $id ) = @_;
    my $invalid_char = $opt->{invalid_char};
    if ( ! $id )                                        {
        return;
    }
    if ( $id =~ m{^p#(?:[FP]L)?([^$invalid_char]+)\z} ) {
        return $1;
    }
    if ( $id !~ $opt->{yt_regexp} ) {
        return;
    }
    elsif ( $id =~ m{/.*?[?&;!]list=([^$invalid_char]+)} ) {
        return $1;
    }
    elsif ( $id =~ m{^\s*([FP]L[\w\-]+)\s*\z} ) {
        return $1;
    }
    return;
}

sub _user_id {
    my ( $opt, $id ) = @_;
    my $invalid_char = $opt->{invalid_char};
    if ( ! $id ) {
        return;
    }
    if ( $id =~ m{^c#([^$invalid_char]+)\z} ) {
        return $1;
    }
    if ( $id !~ $opt->{yt_regexp} ) {
        return;
    }
    elsif ( $id =~ m{/user/([^$invalid_char]+)} ) {
        return $1;
    }
    elsif ( $id =~ m{/channel/([^$invalid_char]+)} ) { # ?
        return $1;
    }
    return;
}

sub _more_ids {
    my ( $opt, $id ) = @_;
    my $invalid_char = $opt->{invalid_char};
    if ( ! $id ) {
        return;
    }
    elsif ( $id !~ $opt->{yt_regexp} ) {
        return;
    }
    elsif ( uri_unescape( $id ) =~ m{youtu\.?be.*video_ids=([^$invalid_char]+(?:,[^$invalid_char]+)*)} ) {
        return $1;
    }
    return;
}


sub _id_to_tmp_info_hash {
    my( $opt, $type, $list_id ) = @_;
    printf "Fetching %s info ... \n", $type eq 'PL' ? 'playlist' : 'channel';
    my $url = URI->new( $type eq 'PL'
        ? 'https://gdata.youtube.com/feeds/api/playlists/' . $list_id
        : 'https://gdata.youtube.com/feeds/api/users/'     . $list_id . '/uploads'
    );
    my $tmp = {};
    my $start_index = 1;
    my $max_results = 50;
    my $count_entries = $max_results;
    while ( $count_entries == $max_results ) {  # or <link rel='next'>
        $url->query_form( 'start-index' => $start_index, 'max-results' => $max_results, 'v' => $opt->{yt_api_v}, 'alt' => 'json' );
        $start_index += $max_results;
        my $res = wrapper_get( $opt, $url->as_string );
        if ( ! defined $res ) {
            my $err_msg = $type . ': ' . $list_id . '   ' . ( $start_index - $max_results ) . '-' . $start_index;
            push @{$opt->{error_get_download_infos}}, $err_msg;
            next;
        }
        my $json = $res->decoded_content;
        my $h_ref = decode_json( $json ); #
        my $entries = $h_ref->{feed}{entry};
        $count_entries = @$entries;
        if ( ! $count_entries ) {
            last;
        }
        for my $entry ( @$entries ) {
            add_entry_to_info_hash( $opt, $tmp, $entry, $type, $list_id ); #
        }
        if ( $opt->{nr_of_latest_videos} && $start_index > $opt->{nr_of_latest_videos} ) {
            last;
        }
    }
    if ( ! keys %$tmp ) {
        my $prompt = "No videos found: $type - $url";
        choose( [ 'Print ENTER' ], { prompt => $prompt } );
    }
    my $up = keys %$tmp;
    print up( $up + 2 ), cldown;
    return $tmp;
}


sub _more_url_to_tmp_info_hash {
    my ( $opt, $more_ids ) = @_;
    my $tmp = {};
    for my $video_id ( split /,/, $more_ids ) {
        my $url = URI->new( 'https://gdata.youtube.com/feeds/api/videos/' . $video_id );
        $url->query_form( 'v' => $opt->{yt_api_v}, 'alt' => 'json' );
        my $res = wrapper_get( $opt, $url );
        if ( ! defined $res ) {
            my $err_msg = 'Video group: ' . $more_ids . ' - ' . $video_id . '   ' . $url;
            push @{$opt->{error_get_download_infos}}, $err_msg;
            next;
        }
        my $json = $res->decoded_content;
        my $h_ref = decode_json( $json );
        my $entry = $h_ref->{feed}{entry};
        add_entry_to_info_hash( $opt, $tmp, $entry );
    }
    return $tmp;
}


sub _non_yt_id_to_tmp_info_hash {
    my ( $opt, $id ) = @_;
    my $message = "Fetching download info: ";
    my $json_all = get_download_info_as_json( $opt, $id, $message );
    my $tmp = {};
    return $tmp if ! $json_all;
    for my $json ( split /\n+/, $json_all ) {
        json_to_hash( $opt, $tmp, $json );
    }
    return $tmp;
}


sub my_sort {
    my ( $opt, $h_ref, $a, $b ) = @_;
    my $item = $opt->{sort_item};
    my @s = $opt->{sort_order} eq 'Asc'
                ? ( $h_ref->{$a}{$item}, $h_ref->{$b}{$item} )
                : ( $h_ref->{$b}{$item}, $h_ref->{$a}{$item} );
    if ( $item eq 'view_count_raw' ) {
       ( $s[0] //  0 ) <=> ( $s[1] //  0 ) || ( $h_ref->{$a}{title} // '' ) cmp ( $h_ref->{$b}{title} // '' );
    }
    else {
       ( $s[0] // '' ) cmp ( $s[1] // '' ) || ( $h_ref->{$a}{title} // '' ) cmp ( $h_ref->{$b}{title} // '' );
   }
}


sub _choose_videos_and_add_to_info_hash {
    my ( $opt, $info, $tmp, $is_youtube ) = @_;
    my $regexp;
    my $back   = '       BACK';
    my $filter = '     FILTER';
    my $sort   = '       SORT';
    my $enter  = '      ENTER';
    my %chosen_video_ids;
    my @last_chosen_video_ids = ();

    FILTER: while ( 1 ) {
        my @pre  = ( length $regexp ? ( undef, $enter ) : ( undef, $filter, $sort, $enter ) );
        my @videos;
        my @tmp_video_ids;
        my $index = $#pre;
        my $mark = [];
        my @video_ids = sort { my_sort( $opt, $tmp, $a, $b ) }  keys %$tmp;

        VIDEO_ID: for my $video_id ( @video_ids ) {
            my $title = $tmp->{$video_id}{title};
            $title =~ s/\s+/ /g;
            $title =~ s/^\s+|\s+\z//g;
            next VIDEO_ID if length $regexp && $title !~ /$regexp/i;
            $tmp->{$video_id}{from_list} = 1;
            if ( $is_youtube ) {
                $tmp->{$video_id}{extractor}     = 'youtube';
                $tmp->{$video_id}{extractor_key} = 'Youtube';
            }
            if ( $opt->{sort_item} eq 'view_count_raw' || $opt->{show_view_count} ) {
                push @videos, sprintf "%11s | %7s  %10s  %$opt->{view_count_length}s  %s", $video_id,
                    $tmp->{$video_id}{duration}, $tmp->{$video_id}{upload_date}, $tmp->{$video_id}{view_count}, $title;
            }
            else {
                push @videos, sprintf "%11s | %7s  %10s  %s", $video_id, $tmp->{$video_id}{duration}, $tmp->{$video_id}{upload_date}, $title;
            }
            push @tmp_video_ids, $video_id;
            $index++;
            push @$mark, $index if any { $video_id eq $_ } keys %chosen_video_ids;
        }
        my $choices = [ @pre, @videos ];
        # Choose
        my @idx = choose(
            $choices,
            { prompt => '', layout => 3, index => 1, clear_screen => 1, mark => $mark,
              undef => $back, no_spacebar => [ 0 .. $#pre ] }
        );
        if ( ! defined $idx[0] || ! defined $choices->[$idx[0]] ) {
            if ( length $regexp ) {
                delete @{$info}{ @last_chosen_video_ids };
                $regexp = '';
                next FILTER;
            }
            delete @{$info}{ keys %chosen_video_ids };
            return;
        }
        my $choice = $choices->[$idx[0]];
        if ( $choice eq $filter ) {
            shift @idx;
            @last_chosen_video_ids = ();
            for my $i ( @idx ) {
                my $video_id = $tmp_video_ids[$i - @pre];
                $info->{$video_id} = $tmp->{$video_id};
                $chosen_video_ids{$video_id}++;
                push @last_chosen_video_ids, $video_id;
            }
            for my $m ( @$mark ) {
                if ( none { $m == $_ } @idx ) {
                    my $video_id = $tmp_video_ids[$m - @pre];
                    delete $chosen_video_ids{$video_id};
                    delete $info->{$video_id};
                }
            }
            my $trs = Term::ReadLine::Simple->new();
            $regexp = $trs->readline( "Regexp: " );
            next FILTER;
        }
        elsif ( $choice eq $sort ) {
            sort_order_list( $opt );
        }
        else {
            if ( $choice eq $enter ) {
                shift @idx;
            }
            @last_chosen_video_ids = ();
            for my $i ( @idx ) {
                my $video_id = $tmp_video_ids[$i - @pre];
                $info->{$video_id} = $tmp->{$video_id};
                $chosen_video_ids{$video_id}++;
                push @last_chosen_video_ids, $video_id;
            }
            for my $m ( @$mark ) {
                if ( none { $m == $_ } @idx ) {
                    my $video_id = $tmp_video_ids[$m - @pre];
                    delete $chosen_video_ids{$video_id};
                    delete $info->{$video_id};
                }
            }
            if ( length $regexp ) {
                $regexp = '';
                next FILTER;
            }
            else {
                last FILTER;
            }
        }
    }
}



1;


__END__
