use strict;
use warnings;

package MojoX::Log::Any;
# ABSTRACT: Use the current Log::Any adapter from Mojolicious

our $VERSION = '0.002'; #VERSION

use Log::Any;

sub import {
  my $class = shift;
  my $caller = caller;

  return if $caller->app->log->isa('Log::Any::Proxy');

  $caller->app->log(
    Log::Any->get_logger( default_adapter => 'MojoLog', @_ )
  );

  # Manually inflate nulls, since we are not doing automatic assignment
  $caller->app->log->inflate_nulls;
}

1;

=encoding utf8

=head1 NAME

MojoX::Log::Any - Use the current Log::Any adapter from Mojolicious

=head1 SYNOPSIS

  use Mojolicious::Lite;

  # Use Mojo::Log by default when importing
  use MojoX::Log::Any;

  # Or you can specify a different default adapter
  use MojoX::Log::Any default_adapter => 'Stderr';

  get '/' => sub {
    my $c = shift;

    app->log->debug('Using Log::Any::Adapter::MojoLog');

    # They can be redefined
    use Log::Any::Adapter;
    Log::Any::Adapter->set('Stderr');
    app->log->warning('Using Log::Any::Adapter::Stderr')
      if app->log->is_warning;

    # Or use whatever adapter you've set
    use Log::Log4perl qw(:easy);
    Log::Log4perl->easy_init($ERROR);

    Log::Any::Adapter->set('Log4perl');
    app->log->fatalf('Formatting with %s', 'Log::Any::Adapter::Log4perl');

    $c->render(text => 'OK!');
  };

  app->start;

=head1 DESCRIPTION

L<MojoX::Log::Any> makes it easy to use a L<Log::Any::Adapter> from within
L<Mojolicious> without getting in the way of the user.

When imported from within a Mojolicious application (of from within a
package into which Mojolicious' app function has been exported), it sets
that application's log attribute to a L<Log::Any::Proxy> connected to
whatever adapter is currently available.

When imported, the logger defaults to using L<Log::Any::Adapter::MojoLog>,
which seems to be the currently maintained adapter for L<Mojo::Log>. Any
parameters passed to the module's C<import> function are passed I<as is>
to the C<get_logger> function from L<Log::Any>, to allow for user
customisation and to maintain a coherent interface with that package.

=head1 MOTIVATION

There are numerous packages in the "MojoX::Log" namespace providing an
interface with the various different logging mechanisms on CPAN; except
Log::Any.

There is also a Log::Any adapter for Mojo::Log, which makes it
possible to use that logger from any application using Log::Any; but
not Mojolicious apps.

This package attempts to fill that void by offering Mojolicious
applications an easy way to plug into the current Log::Any::Adapter
(whatever it may be).

=head1 SEE ALSO

=over 4

=item *

L<Log::Any>

=item *

L<Log::Any::Adapter::MojoLog>

=item *

L<Log::Any::Adapter::Mojo>

=back

=head1 AUTHOR

=over 4

=item *

José Joaquín Atria <jjatria@cpan.org>

=back

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by José Joaquín Atria.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
