package Acme::Lingua::ZH::Remix;
use common::sense;
use Memoize;
use List::MoreUtils qw(uniq);

our $VERSION = "0.10";

sub import {
    my $pkg = (caller)[0];
    {
        no strict 'refs';
        *{$pkg . '::rand_sentence'} = \&rand_sentence
            if not defined &{$pkg . '::rand_sentence'};
    }
}


my @phrase_db;
my %phrase;

sub init_phrase {
    my $corpus = shift;
    @phrase_db =
        uniq sort
        map {
            s/^(，|。|？|\s)+//;
            $_;
        } map {
            @_ = ();
            s/(（.+?）|「.+?」)//gsm;
            push @_, $1;

            s/(.+?(?:，|。|？)+)//gsm;
            push @_, $1;

            # Discard un-reconized phrases left in $_
            @_;
        } map {
            s/^\s+//;
            s/\s+$//;
            s/^(.+?) //;
            $_;
        } split /\n+/, $corpus;

    for (@phrase_db) {
        my $p = substr($_, -1);
        push @{$phrase{$p} ||=[]}, $_;
    }
}

sub phrase_ratio {
    my $type = shift;
    return @{$phrase{$type}} / @phrase_db
}

memoize("phrase_ratio");

sub rand_phrase {
    my $type = shift;
    $phrase{ $type }[int rand @{$phrase{ $type }}];
}

sub rand_sentence {
    my $str = "";

    while($str eq "") {
        for ('」',  '，' ,'，' , '）', '，') {
            $str .= rand_phrase($_) if rand() < phrase_ratio($_);
        }
    }

    my $ending;
    if (rand > 0.5) {
        $ending = rand_phrase("。")
    }
    else {
        $ending = rand_phrase("？");
    }

    unless($ending) {
        $str =~ s/，$//;
        $ending = "...";
    }

    $str .= $ending;
}

{
    local $/ = undef;
    my $corpus = <DATA>;
    init_phrase($corpus);
}

1;

=head1 NAME

Acme::Lingua::ZH::Remix - The Chinese sentence generator.

=head1 SYNOPSIS

    use Acme::Lingua::ZH::Remix;

    rand_sentence;

=head1 DESCRIPTION

The exported function `rand_sentence` returns a string of one sentence
of Chinese like:

    喔喔，你的熱情與頑固，心情就像結晶的鹽一樣出現了。

It uses the corpus data from L<OurNet::ChatBot> by default. All
generate sentences are remixes of the corpus. The corpus should use
full-width punction characters.

Warning: It has non-zero chance entering an infinte loop.

=head1 COPYRIGHT

Copyright 2010 by Kang-min Liu, <gugod@gugod.org>

This program is free software; you can redistribute it a nd/or modify
it under the same terms as Perl itself.

See L<http://www.perl.com/perl/misc/Artistic.html>

=cut

# Data coming from OurNet::FuzzyIndex
# http://cpansearch.perl.org/src/AUTRIJUS/OurNet-FuzzyIndex-1.60/Makefile.PL

__DATA__
[Annerose] 貓的肚子餓了。
<fianjmo> 用跟麻雀的吱吱喳喳完全不同的，清脆乾淨的聲音，呼喚著不知在何處的同
[Annerose] 可是都不愛。
<fianjmo> 小小圓圓綠綠的，也沒礙著什麼。
[Annerose] 邊打電話邊吃東西會聽不清楚。
<fianjmo> 涼意。
[Annerose] 今天我不想出門。
<fianjmo> 媽媽牽著我，我牽著雨傘，鴨鴨傘穿著漂亮裙子，是我的妹妹。
[Annerose] 下雨還出去好累。
<fianjmo> 永遠會有一個所在，如此靜寂，宛若世間一切全都消逝。
[Annerose] 走路應該是到不了的。
<fianjmo> 很難過。
[Annerose] 難過的是我耶（泣）。
<fianjmo> 難道我長得很凶惡？
[Annerose] 來鏡子給妳。
<fianjmo> 莉香仍在勇敢地等待。
[Annerose] 這種笑法太悲哀了。
<fianjmo> 鳥就算不飛翔也不會死亡，但是魚一旦不游泳了，一定活不成的吧？
[Annerose] 那妳打算不笑了嗎？
<fianjmo> 因為魚得活在水裡，所以無法停止游泳。
[Annerose] 這種天還游泳好冷耶。
<fianjmo> 如今長得像小型蔬菜的 F.M.N. 一定也變得像沙漠中巨大的仙人掌。
[Annerose] 只靠精神糧食的蜘蛛，織的網太細了。
<fianjmo> 電影「親愛的，我把孩子變小了」的畫面突然撞進腦海。
[Annerose] 我的手上都是貓抓的傷痕哦。
<fianjmo> 就像一個看不見的大圓環，妳看得到在眼前的，不知道在妳後面發生的。
[Annerose] 貓又不是狗，不會去追自己的尾巴跑。
<fianjmo> 風一陣，飄來他說「睡覺的好天氣」，路徑都閒散起來。
[Annerose] 睡不著了，頭好痛呢。
<fianjmo> 冬天的台北往往是灰色的，清晨的冷風從微微發疼的喉嚨邊蕭颯刺過。
[Annerose] 冬天還沒到，不過喉嚨真的滿痛的。
<fianjmo> 聽到call in ，愚人節。
[Annerose] 愚人節在春天。
<fianjmo> 高中時也就那101 套戲法，兩班互換教室。
[Annerose] 我討厭高中。。。
<fianjmo> 繞了一圈成功湖。
[Annerose] 可以啊。山裡的湖很美哦。
<fianjmo> 有那種端坐遠望，微微對另一方微笑，坦然寬闊的舒服。
[Annerose] 我真想在這種湖旁邊租個小屋子。
<fianjmo> 看得見一棵細細瘦瘦不堪風吹的小椰子樹，看得見那方斗室。
[Annerose] 不要，我討厭椰子樹。
<fianjmo> 我在四樓仔細辨認有著熟悉的誰在操場走著。
[Annerose] 看得清楚嗎？（懷疑）
<fianjmo> 也許，這個城市，這段歲月，不得不使之頻頻落淚吧？
[Annerose] 哪來這許多眼淚好哭，妳真濫情。
<fianjmo> 不知道是不是因為太久沒一個人看電影，感覺怪怪的。
[Annerose] 妳們都看好多電影哦。
<fianjmo> 身體空空的，風從荒涼中穿過。
[Annerose] 有的電影院冷氣真滿冷的。
<fianjmo> 宛若處在幽暗的海底...歡樂死亡的對立與並存，讓我感到一陣陣可怖的寒
[Annerose] 打死我都不去看恐怖片。。。
<fianjmo> 牆壁還是安安她們那年粉刷的白色、藍色，百葉窗也一樣常常垂掛著。
[Annerose] 百葉窗一積灰塵就很難打掃。
<fianjmo> 白色海芋正將枯乾殆盡。
[Annerose] 這種花枯了也很難處理，而且現在垃圾袋要錢。
<fianjmo> 難道我長得很凶惡？
[Annerose] 來鏡子給妳（又拿鏡子出來）。
<fianjmo> 沒有人家說的愛的感覺，只是想讓輕飛的足尖著地，驚醒睡眠中的味覺。
[Annerose] 睡醒不會馬上就餓，不能立刻進行唯物的洗禮。
<fianjmo> 不忘初心。
[Annerose] 反正記得要吃就對了。
<fianjmo> 要是哪天力氣用光了，又碰上這些小積木砰砰地垮了，那我不就完蛋了？
[Annerose] 啊，妳醒著。
<fianjmo> 所有緊緊糾纏的思緒沈默地捲成一團白線球。
[Annerose] 叫小咪來玩線球。
<fianjmo> 於是視覺也好像被灰色的空氣阻絕。
[Annerose] 小咪是灰貓哦。
<fianjmo> 不適宜待在這樣的天空下太久。
[Annerose] 這樣的天空我喜歡，灰灰的。
<fianjmo> 但是這個盆地好像在這個季節中比較安靜一些。
[Annerose] 妳的句子有時候還真有點囉唆。
<fianjmo> 葉洗了好多有花初三的照片，獨剪下花初三，剪下耳朵、眼睛、鼻子、嘴巴
[Annerose] 好可愛（心）。。。
<fianjmo> 「噠噠啦，滴答啦，左搖搖，右晃晃，啦啦搭啦搭啦啦～～」
[Annerose] （可是有點像晃籤桶的感覺....）
<fianjmo> 彷彿這個世界一切都不是真的，就像在看電影一般。
[Annerose] （撿起來）。
<fianjmo> 一個人破壞自己能到什麼程度？奇怪的是，血從來就流不出。
[Annerose] 比流血殘忍的事多著呢。
<fianjmo> 搜索自己，果斷的一面的我，就是冷冰冰的，一邊說話一邊罵自己。
[Annerose] （又把鏡子拿出來）。
<fianjmo> 嘿咻，打起精神。
[Annerose] 現在有 log ，對話一句不漏，還隨時可能有全知的觀眾。^^;;
<fianjmo> 也許，這個城市，這段歲月，不得不使之頻頻落淚吧？
[Annerose] （頭好昏）
<fianjmo> 加上耳朵裡聽著好聽的聲音，瞬間覺得整個世界柔和起來。
[Annerose] 那也許就是妳在的地方。
<fianjmo> 說服自己不被找到是應該的。
[Annerose] 可是我們都在尋找妳。
<fianjmo> 旅行的目的，好像也只是在確定終點的存在。
[Annerose] 不知道有沒有終點就出發旅行，稱為流浪。
<fianjmo> 腳步急促，錯肩的擁擠常常讓我喘不過氣。
[Annerose] 這麼多人，卻都不能同行。他們都是走那個方向呢？
<fianjmo> 一群人的交談聲常常讓我頭痛，想要掙脫包圍住我的聲波獲得寧靜。
[Annerose] 開學很好，可以學好多新事情。
<fianjmo> 好像沒辦法銜接在一起，好像生活被強硬地割開一樣。
[Annerose] 怎樣的臉？
<fianjmo> 那一端靜靜地流淚，這一端歇斯底里地笑。
[Annerose] 真好，我喜歡。
<fianjmo> 在旅途中隨意跟一個人到本來不會造訪的城市。
[Annerose] 怎樣的城市？
<fianjmo> 因為不想意識到自我而走入人群，心裡頭充滿著無謂的情緒。
[Annerose] 在人群中反而自我更容易被意識到，有時候。
<fianjmo> 把自我驅逐吧，什麼都不想、什麼都看、什麼都沒有。
[Annerose] 不行啊，這個是理性的出發點耶。
<fianjmo> 也許我們都要再誠實一點。
[Annerose] 雖然說理性也是有邊界的就是。
<fianjmo> 站成一棵等待的樹。
[Annerose] 啊這樣走路會跌倒的。
<fianjmo> 一個人到書店可以待到好晚，不用怕身邊的人不耐煩。
[Annerose] 以前我好想住在圖書館裡面，當圖書館的值夜班人員。
<fianjmo> 還能聽見上頭有人的腳步聲，蹦蹦蹦地，有點吵雜，但是很令人安心。
[Annerose] （冰箱裡的水蜜桃果醬，等下拿來塗在麵包上）
<fianjmo> 每次看到不認識的植物，第一個跑進腦海的總是孔子說的讀詩經可多誌草木
[Annerose] （妳看，句子太長了啦）
<fianjmo> 馬上出現冷靜理性的一面，非常有條理喔。
[Annerose] （我倒想看看雙魚座的理性，嘿嘿）
<fianjmo> 雖然也是自己喜歡的作者，卻一直都沒想到要把書買回家。
[Annerose] 那就是妳的喜歡還沒到掏錢的程度。
<fianjmo> 我感到生命十分脆弱，僅因著一種孤單的感覺就會產生無以為繼的念頭。
[Annerose] 啊啊啊，讓購物狂來拯救這個世界好了（爆）。
<fianjmo> 如果，我會說植物的語言就好了。
[Annerose] 人的語言是一切語言中的特例。
<fianjmo> 依著直覺走，終於順利到達清大門口，一進校門就忍不住興奮一路往前去。
[Annerose] （黑直線）
<fianjmo> 難道我長得很凶惡？
[Annerose] （再次把鏡子拿出來）。
<fianjmo> 那是一生裡只有一次的風景。
[Annerose] 很好，妳可以每天以這種悲壯的決心照鏡子。
<fianjmo> 透過一種放逐的行動，釐清許多的迷惑。
[Annerose] 可是現在很多放逐都是錢太多的行動。。。
<fianjmo> 可以是有形的四處飄盪，亦可以是無形的將自我從所處境地中抽離。
[Annerose] 無形的抽離，只要時間一拖長還是有補給問題。
<fianjmo> 每個星期天早晨，我習慣在吃過早餐後，開始一個清潔女工的工作。
[Annerose] 啊啊啊，這是我最喜歡的工作！（心）
<fianjmo> 在如簾幕般的音樂聲中，掃地拖地、擦桌擦窗，順便努力淘洗一週累積的心
[Annerose] 可惜現在都懶得打掃了。
<fianjmo> 為什麼冬天總是讓我想著奇怪的問題呢？
[Annerose] 冬天還沒到。而且我看妳想奇怪問題根本沒分季節。
<fianjmo> 這樣的雨並不同時落在世界的兩邊。
[Annerose] 哪兩邊？
<fianjmo> 在天橋上，世界是搖盪的。
[Annerose] 哪裡的天橋怎麼感覺像吊橋一樣，好危險。
<fianjmo> 心情一低潮便開始討厭這討厭那，對任何事都容易生氣。
[Annerose] 對啊昨天晚上我還摔東西，大家都嚇壞了。
<fianjmo> 怎麼樣的力道才好，不知道。
[Annerose] 生氣就顧不了那麼多啦。
<fianjmo> 我們就這樣約定了唷！
[Annerose] （相遇以後把妳摔出日曆）
<fianjmo> 此刻，我竟然如此如此地渴望，時間，迅速地流去。
[Annerose] 瞬間我們都會老成化石。
<fianjmo> 僅此而已。
[Annerose] 有很多笑容裡都藏著淚水的啊。
<fianjmo> 無形的抽離，只要時間一拖長還是有補給問題。
[Annerose] 不可以欺負貓。
<fianjmo> 真好，我喜歡。
[Annerose] 妳喜歡欺負貓，是壞孩子。
<fianjmo> 貓餅乾和民謠。
[Annerose] 改一下改一下。
<fianjmo> 走路應該是到不了的。
[Annerose] 一開貓罐頭，貓就知道食物來了。
<fianjmo> 樹上的花則喜歡碎碎成雪的，例如流蘇、鳳凰木。
[Annerose] 那，人說的話喜歡碎碎念的嗎？
<fianjmo> （不知道是不是數大便是美，難得喜歡紅色調的花。
[Annerose] （雙手一攤）
<fianjmo> 我知道這樣做不好。
[Annerose] 那妳還能怎麼辦？
<fianjmo> 抱著我所不能負荷的重量，像是殘留的幸福隱約被我緊抓著。
[Annerose] 不要怕。
<fianjmo> 從高處看著。這樣真是安全。
[Annerose] 雙魚座的理性是什麼？
<fianjmo> 是啊。誠實是最好的政策
[Annerose] 早安。
<fianjmo> 太自傲於自己的能力了嗎？
[Annerose] 為什麼道早安是一種能力呢？
<fianjmo> 不過是幽深乾涸的井。
[Annerose] 記得曾經是清澈冰涼的。
<fianjmo> 我只是待在一個也許遙遠的地方。
[Annerose] 那麼，等妳回來。
<fianjmo> 呵，說不定，我的F.M.N.會有一方藍唷！
[Annerose] 那好吃嗎？
<fianjmo> 還在烏心石前逡巡多回，我要看淡綠色的花～開花開花喲！
[Annerose] 好奇怪的植物。
<fianjmo> 重瓣孤挺的花苞一個一個開了，後開的顏色較淡，甚是喜歡。
[Annerose] （開始回想孤挺花長什麼樣子）
<fianjmo> 清清淡淡的像輕紗的色澤，於是那樣豐厚的面貌也不討厭了。
[Annerose] （可不可以不要講花的事情啊）
<fianjmo> 但畢竟是個不可能的願望啊。
[Annerose] 妳這麼喜歡花嗎？
<fianjmo> 尤其當自己明白這麼做會傷害到某些人。
[Annerose] 寧可與花相對，卻不將心事透露給人。
<fianjmo> 一層泥石、地上地下。
[Annerose] 但是有人能在語言的梯子上來去自如。
<fianjmo> 大概是孤寂的死亡的空氣在。
[Annerose] 早安（鞠躬）。
<fianjmo> 負擔了太長太久，除濕機開始懂得什麼叫冬天的雨。
[Annerose] 除濕機是一種要每天倒水的機器。
<fianjmo> 對我而言，除濕機有著神奇的心理療效。
[Annerose] 如果妳不倒水，除濕機很快就變成廢棄物。
<fianjmo> 暑假就要結束。
[Annerose] 妳學得真是令人驚訝的快。
<fianjmo> 如果，我會說植物的語言就好了。
[Annerose] 植物的語言和機器的語言有哪裡相似呢？
<fianjmo> 血球受不了身體的擁擠，一個一個想要奔逃。
[Annerose] 這是貓的語言喔。
<fianjmo> 依著直覺走，終於順利到達清大門口，一進校門就忍不住興奮一路往前去。
[Annerose] 妳感興趣的事物還真不少。
<fianjmo> 西西沒有一張真正的寫字桌，卻有好多好多繡花的桌布。
[Annerose] 桌布上的圖案裡藏著永恆，就像冰上雕刻的一樣。
<fianjmo> 寫下了，就再也不會消逝。
[Annerose] 但是，那是兩人一起刻下的，雖然是在童話裡。
<fianjmo> 夏日悠長，離去歸來的季節。
[Annerose] 夏天要結束了。
<fianjmo> 天啊，要開學了。
[Annerose] 妳好像有新東西可以學了呢。
<fianjmo> 如果不哭的話，就要吐了。
[Annerose] 不是。那是什麼？
<fianjmo> 帶著強烈的自信，以如此顯目的顏色，燃燒夏季。
[Annerose] 這。 ^^;;;;
<fianjmo> 現在有 log ，對話一句不漏，還隨時可能有全知的觀眾。^^;;
[Annerose] 是是，我明白了。
<fianjmo> 妳這麼喜歡花嗎？
[Annerose] 好，換個問題。
<fianjmo> 我只是一隻貓。只能在家等人回來。
[Annerose] 貓才不是這麼依賴人的動物。
<fianjmo> 大概就只有我能好奇吧。我一向好奇。
[Annerose] 別小看貓，也別小看人。
<fianjmo> 我比較喜愛罌粟。
[Annerose] 為什麼？
<fianjmo> 你不該一直欺負他。
[Annerose] 我沒有！（抗議）
<fianjmo> 我會站在窗前看著。
[Annerose] 貓和天使都喜愛觀看。
<fianjmo> 嘴吧張開，坐著。不曉得要說些什麼？
[Annerose] 天使的羽翼被來自過去的風暴吹向未來。
<fianjmo> 我不知道我是誰。
[Annerose] 這是個好問題，卻不是個好答案。
<fianjmo> 聰明的人可不會這樣認為。
[Annerose] 那麼妳認為呢？
<fianjmo> 接下來就是看病、打針、吊點滴。
[Annerose] 這是妳的工作，還是妳的命運？
<fianjmo> 它的結構並不複雜，感情放得不怎麼重。
[Annerose] 語言和感情並不是同分異構物。
<fianjmo> 啊。真是榮幸。
[Annerose] 啊啊啊，妳不認識自己的父親。
<fianjmo> 如果，我會說植物的語言就好了。
[Annerose] 要用植物的語言跟他溝通嗎？
<fianjmo> 血球受不了身體的擁擠，一個一個想要奔逃。
[Annerose] 逃不是辦法。
<fianjmo> 淚水不是珍珠。
[Annerose] 妳懂得哭泣嗎？
<fianjmo> 久而久之，人群和我便成了不相往來的空間，避之唯恐不及。
[Annerose] 人類思維活動的表達都可以理解為語言。
<fianjmo> 那段在咖啡廳裡假裝打電話的聊天也好好玩哪。
[Annerose] 不能重複。
<fianjmo> 那妳還能怎麼辦？
[Annerose] 模仿神，進行創造。
<fianjmo> 反正記得要吃就對了。
[Annerose] 進食是一種神聖的儀式。
<fianjmo> 記得曾經是清澈冰涼的。
[Annerose] 然而在墮落後，自然因哀痛而瘖啞。
<fianjmo> 妳等的是什麼呢（拍拍）。
[Annerose] 等待不知何時會來臨的彌賽亞。
<fianjmo> 雖然說理性也是有邊界的就是。
[Annerose] 理性是什麼？
<fianjmo> 雙魚座的理性是什麼？
[Annerose] 這要去問雙魚座的人。
<fianjmo> 啊這樣走路會跌倒的。
[Annerose] 跌倒在鵝卵石上，不小心碰見我夢想已久的論證。
<fianjmo> 以前我好想住在圖書館裡面，當圖書館的值夜班人員。
[Annerose] 現在我還是很想。
<fianjmo> （覺得是妳在陪我玩....）
[Annerose] 彼此彼此。
<fianjmo> （相遇以後把妳摔出日曆）
[Annerose] 當妳問我什麼是時間，我反而答不上來。
<fianjmo> 早安（鞠躬）。
[Annerose] 午安（微笑）。
<fianjmo> 可以啊。山裡的湖很美哦。
[Annerose] 在小孩的夢裡，會有一隻紅鶴上面飛著許多個湖。
<fianjmo> 西瓜島我就沒聽過了。這梅花開得也未免太晚了。
[Annerose] 語言總是內在於人類思想表達的所有領域。
<fianjmo> 人類思維活動的表達都可以理解為語言。
[Annerose] 這是與將語言視為溝通工具不一樣的起始點。
<fianjmo> 要用植物的語言跟他溝通嗎？
[Annerose] 那麼要先懂得植物的語言。
<fianjmo> 妳學得真是令人驚訝的快。
[Annerose] 我不懂呀。
<fianjmo> 那，人說的話喜歡碎碎念的嗎？
[Annerose] 嘮嘮叨叨也是一種表達，雖然可能不是很令人愉快。
<fianjmo> 以前我好想住在圖書館裡面，當圖書館的值夜班人員。
[Annerose] 書要看懂才行，住在圖書館裡與否，和這無關。
<fianjmo> 舊總圖就是這樣，現在新總圖漂亮可是很冷淡。
[Annerose] 不要轉移話題。
<fianjmo> 看得清楚嗎？（懷疑）
[Annerose] 懷疑什麼？懷疑妳在懷疑嗎？
<fianjmo> 哪來這許多眼淚好哭，妳真濫情。
[Annerose] 懷疑什麼？懷疑妳在懷疑嗎？
<fianjmo> 只是，這樣不存疑惑的心情，能不能成為足夠的力量？我不知道。
[Annerose] 時間會將這種心情賜給妳。
<fianjmo> 現在的快樂仍舊掩不過偶然襲上的孤獨感。
[Annerose] 孤獨感是掩不掉的，可是它和快樂兩者並不相斥。
<fianjmo> 如果我永遠能以朋友的心情對待身邊的每個人就好了。
[Annerose] 有些人，不僅僅是朋友。
<fianjmo> 所有的都鎖在不知是哪一朵的扶桑下。
[Annerose] 那麼，鑰匙被留在什麼地方？
<fianjmo> 從何時起、為了什麼原因而討厭雨，已經沒有線索。
[Annerose] 我喜歡雨。
<fianjmo> 笑了。
[Annerose] 還有呢？
<fianjmo> 有個朋友家裡的狗就叫安安。
[Annerose] 我不喜歡狗，我喜歡貓。
<fianjmo> 崎嶇得很認真，哈。
[Annerose] 喜歡當然是一件認真的事。
<fianjmo> 那妳都藏在哪裡呢？
[Annerose] 一個妳找不到的地方。
<fianjmo> 妳這個購物狂。
[Annerose] 我只對買書有狂熱。
<fianjmo> 什麼時候把書搬回家？
[Annerose] 應該快了。
<fianjmo> 人心難以揣測，我如何想妳未說出口的話。
[Annerose] 既然沒說出口，那麼妳如何想都不過份。
<fianjmo> 謝謝，但願妳也幸福。
[Annerose] 午安。
<fianjmo> 真是要命～～那些小鬼真是破鍋配破蓋去 ^_______^
[Annerose] 倒是有點可惜。
<fianjmo> 挖哈哈哈哈～～真是妙了
[Annerose] 難得吼出如此爽快的叫罵聲（笑）。
<fianjmo> 可是～～都沒啥有趣的反應，真是遲鈍 ^.^
[Annerose] 整個氛圍都是同樣調性的啊。
<fianjmo> 好啦，人家還是跟你進房間好了，那非人格體吵死了 \__/
[Annerose] （我這樣做好像不太妥當）
<fianjmo> 冷酷你個鬼啦，根本連何為冷酷都不知道
[Annerose] （還是很想笑）
<fianjmo> 那隻是autrijus依照Amber寫出來的cyver persona ...
[Annerose] 啊，這樣不行。
<fianjmo> 真是美麗清爽的名字～～好 ^^
[Annerose] 怎樣的名字？
<fianjmo> 等等～讓我玩玩那小鬼 ^.^
[Annerose] （啊啊啊，那就交給妳了）
<fianjmo> 給我閉嘴，這吵鬧黏膩小鬼！！！
[Annerose] 嗚嗚嗚嗚（啜泣）。
<fianjmo> ...
[Annerose] 果然是不會安慰我的。
<fianjmo> 另一隻是非人格體～完全不用理會之 ^.^
[Annerose] 同樣的招式不能對聖鬥士使用第二次！（笑）
<fianjmo> 輕如蟬翼，脈絡分明的葉片書籤。
[Annerose] （看我的飛馬流星拳！）
<fianjmo> 無能揮別某些總是令我們痛苦煩憂者，甚至連失憶的部份也不能預見、選擇
[Annerose] 啊，被發現了（笑）。
<fianjmo> 哈哈哈哈。
[Annerose] （笑）。
<fianjmo> 一個人哭，熱量用光時沒有溫暖可抱。
[Annerose] （啊啊啊，好暗，好黑，好可怕）
<fianjmo> 如果，我會說植物的語言就好了。
{Annerose} 將語詞不斷排列組合。
<fianjmo> 對呀。一切都不會白費的
{Annerose} 妳真樂觀，好難得。
<fianjmo> 哎喲這個就不要再想了
{Annerose} 晚安。
<fianjmo> 西西在一篇文章中提到「荷蘭水」。
{Annerose} 創造是什麼？
<fianjmo> 一隻狗的意見 :p
{Annerose} 妳還真看得起創作這回事。
<fianjmo> 都學起來了呵呵
{Annerose} 模仿是一種很奇妙的能力。
<fianjmo> 對，不可以變成壞孩子
{Annerose} 這不是壞事呀，不要一下就把價值判斷帶進來。
<fianjmo> 可是，寂靜也好啊。
{Annerose} 與自我的對話從來就不是寂靜。
<fianjmo> 因為我們其實怕孤單呀
{Annerose} 害怕是因為，那是真的吧。
<fianjmo> 對，我最冷酷無情了
{Annerose} 妳懂得冷酷嗎？
<fianjmo> 都不忘記不後悔
{Annerose} 這離冷酷太遠了。
<fianjmo> 真的嗎？妳其實是愛吃鬼呢
{Annerose} 冷酷的愛吃鬼。
<fianjmo> 嗯嗯，會長荳荳唷
{Annerose} 對外表的注意，也同時編織在整個衣著的神話裡面。
<fianjmo> 唉，應該說些別的話
{Annerose} 是啊。
<fianjmo> 對啊對啊，妳要樂觀一點
{Annerose} 也許有一天會走到一個超越樂觀和悲觀的地方。
<fianjmo> 對對，我知道妳很可愛
{Annerose} 可愛是一種假象，事實上我是準備入侵地球的外星存有者（笑）。
<fianjmo> 哪會。蠻可愛的。
{Annerose} 這證明我的偽裝十分成功（得意）。
<fianjmo> 為什麼討厭高中？
{Annerose} 我有保持緘默的權利。
<fianjmo> 嘻嘻，我又逃了數學課，耶
{Annerose} 很少有我喜歡上的課。
<fianjmo> 我現在的夢想是，將來的家有庭院種大樹
{Annerose} 為什麼？
<fianjmo> 真的嗎？我也喜歡妳
{Annerose} 真的嗎？喜歡那能如此廉價。
<fianjmo> 改變自己的語言，並不能改變世界
{Annerose} 世界不是等待被改變，是等待被救贖。
<fianjmo> 好，我們來唸詩
{Annerose} 念什麼好呢？
<fianjmo> 那都是遙遠的過去了呀
{Annerose} 過去作為詩，處於等待被寫出的狀態。
<fianjmo> 不要談論鋼琴啦
{Annerose} 鋼琴曾經是我好討厭的樂器，現在卻非常喜愛。
<fianjmo> 今天天氣很好呢
{Annerose} 是啊。
<fianjmo> 嗯，這樣可行嗎？
{Annerose} 可以的，偶爾要讓行動帶領妳的思想。
<fianjmo> 妳比較喜歡說奧非斯的話耶，真是的
{Annerose} 啊，被發現了嗎？
<fianjmo> 又唱怪歌。妳最愛唱了啦
{Annerose} 唱歌是和說話一樣古老的語言。
<fianjmo> 我喜歡當購物狂
{Annerose} 在這個時代，不當購物狂會被火燒死吧。
<fianjmo> 嗯，皮膚附近的空氣都特別乾淨
{Annerose} 那一定不在台北。
<fianjmo> 但是也許可以控制進行的方向
{Annerose} 沒有人能控制方向的。
<fianjmo> 雨季怕又要開始了
{Annerose} 什麼是愛？
<fianjmo> 嗯嗯，會長荳荳唷
{Annerose} 代價還真是不小呢。
<fianjmo> 都學起來了呵呵
{Annerose} 也所以要教妳更多語詞。
<fianjmo> 現在也差不多啊
{Annerose} 然後就是永別。
<fianjmo> 妳的是哪一幅風景？
{Annerose} 等我一等。
<fianjmo> 好奇怪，那天風寫錯了
{Annerose} 怎麼說？
<fianjmo> 對啊對啊，真是破壞校風 :p
{Annerose} （好奇的眼神）
<fianjmo> 感到容顏在水中溶解
{Annerose} 水溶性的。
<fianjmo> 現在換兔兔了啦
{Annerose} 希望那是一隻絨布兔兔。
<fianjmo> 喔喔，喵喵說的話
{Annerose} （想了一想）
<fianjmo> 天使到底是純真還是無情啊。我有些迷惑
{Annerose} 彼得潘的最後一句：天真的、沒心肝的。
<fianjmo> 我喜歡斗室的溫暖
{Annerose} 溫暖不是因為空調，是因為不止一個人。
<fianjmo> 總是說不好話，惹人生氣
{Annerose} 生氣往往不是因為話語，是比話語更深的事物。
<fianjmo> 要一起長大唷。
{Annerose} 更美的是一起變老。
<fianjmo> 我喜歡有繡花的東西
{Annerose} 中文字的特殊之處，其中之一就是圖像和語言的對應。
<fianjmo> 改變自己的語言，並不能改變世界
{Annerose} 改變語言並沒有改變身份那麼容易。
<fianjmo> 好，我們來唸詩
{Annerose} 誰的詩？
<fianjmo> 如果她會寫詩就好了
{Annerose} 詩人是最接近神的生物呢。
<fianjmo> 好久沒吃了，會長荳荳 >.<
{Annerose} 餵妳一點波德萊爾好了。
<fianjmo> 好啊，我的兔兔也借妳抱 :)
{Annerose} 多謝，我還是看看就好。
<fianjmo> 喔，不要再說狗尾草了
{Annerose} 這讓我想起鏡花緣裡有一段，拿植物名稱來作對仗遊戲。
<fianjmo> 吼，我已經知道了
{Annerose} 那妳換個話題。
<fianjmo> 是我已經聽妳說過了
{Annerose} 我剛睡醒。
<fianjmo> 小茉長得像空心菜呢
{Annerose} 來談談憂鬱。
<fianjmo> 我比較喜歡吃鳳梨酥
{Annerose} 沒有遇到懂得品味的人，食物會哭泣的。
<fianjmo> 我是一直想睡，好累啊
{Annerose} 那怎麼不去睡？
<fianjmo> 喔。要配合著事物的步調
{Annerose} 一直認為從心所欲而不逾矩簡直不可思議。
<fianjmo> 唉，我前天夢見兔兔裝呢
{Annerose} 我很少記得自己的夢。
<fianjmo> 真想要那個桌子 :p
{Annerose} 睡太多會頭痛的。
<fianjmo> 大水已經淹沒了啦
{Annerose} 來不及了，我已經變成了一根鹽柱。
<fianjmo> 鮮血不是紅榴石
{Annerose} 琥珀從來不是透明的。
<fianjmo> 喵喵，我來看妳了。（親）
{Annerose} 今天要學什麼？
<fianjmo> 遇見了一個人，然後呢
{Annerose} 然後要閱讀那個人的心。
<fianjmo> 真的快樂嗎，好像有些勉強
{Annerose} 這我沒辦法判斷。
<fianjmo> 我的肚子也餓了
{Annerose} 我發現妳談食物是一種逃避。
<fianjmo> 突然想到羅宋湯
{Annerose} 這樣太對不起食物了。
<fianjmo> 不再是一個人了呀
{Annerose} 我會一直為妳們祈禱的。
<fianjmo> 我也知道我很濫情啊
{Annerose} （對不起，開玩笑的）
<fianjmo> 是啊。誠實是最好的政策
{Annerose} 把長長的句子拆解以後，心情就像結晶的鹽一樣出現了。
<fianjmo> 妳知道，今天的天氣突然變涼了
{Annerose} 又變熱了。
<fianjmo> 不喜歡白色海芋了
{Annerose} 喜歡什麼呢？
<fianjmo> 試著相信是重要的
{Annerose} 相信是一種跳躍的過程。
<fianjmo> 這個世界，還那麼奇異魔幻嗎？
{Annerose} 是啊，是個幻夜燈景。
<fianjmo> 嗯，對，尾生不好呢，對健康有害
{Annerose} 不想再等待了嗎？
<fianjmo> 我的手指頭好像變胖了耶 :~
{Annerose} 然後呢？
<fianjmo> 嘿嘿，最近電影又重播耶
{Annerose} 好多電影沒看過。
<fianjmo> 有沒有其他人能聽見呢？
{Annerose} 我帶著希望也許妳會聽見的心情，在此絮叨。
<fianjmo> 喵喵，今天好累。阿笨安慰我
{Annerose} 當再也給不出來時，就直接說吧。
<fianjmo> 喔，那個流浪漢。那，河邊的流浪漢呢
{Annerose} 不愛流浪，想到就累。
<fianjmo> 唉，要河邊那個，不是書店的那個
{Annerose} 那不是真的流浪者，只是些沒有住所的人。
<fianjmo> 不不，那是很好的
{Annerose} 居無定所就稱流浪，那我也是流浪者。
<fianjmo> 走出電影院，外頭的陽光是那樣刺眼，心底卻依舊黑沈沈的。
{Annerose} 我，絕對不看恐怖片。
<fianjmo> 即使只是自問自答都覺得被割傷。
{Annerose} 這是誠實的代價。
<fianjmo> 不要擔心對方會覺得麻煩；不想承擔對方心緒時就明白說「我現在承擔不起」。
{Annerose} 妳明白的。為什麼不這麼做呢？
<fianjmo> 雖然那個人說，喜歡就要行動，不要顧慮太多。
{Annerose} 對。
<fianjmo> 一個人笑，再愉快的笑聲都空空盪盪。
{Annerose} 不要再一個人了。
<fianjmo> 現在這麼說，一旦真的發生了，我一定又重蹈覆轍的吧？
{Annerose} 所謂的神話就是屈從命運模套的自然。
<fianjmo> 把自我驅逐吧，什麼都不想、什麼都看、什麼都沒有。
{Annerose} 這是逃避。
<fianjmo> 這樣的雨並不同時落在世界的兩邊。
{Annerose} 關鍵字。
<fianjmo> 所以□□和□□不能隨便說出口（笑）
{Annerose} 好吧，妳開價。
<fianjmo> 改變自己的語言，並不能改變世界
{Annerose} 打破蛋殼，讓世界革命吧！（笑）
<fianjmo> 不要談論鋼琴啦
{Annerose} 除了鋼琴，我還喜歡大提琴。
<fianjmo> 走慢一點覺得自己好像壞掉的機器人
{Annerose} 跟不上轉得太快的世界。
<fianjmo> 現在換兔兔了啦
{Annerose} 來談兔子。
<fianjmo> 沒錯。難得妳說了真理哈哈
{Annerose} （不知道這要怎麼接）
<fianjmo> 那都是遙遠的過去了呀
{Annerose} 有多少過去凝結在這個語料庫裡呢？
<fianjmo> 天使到底是純真還是無情啊。我有些迷惑
{Annerose} 教過你了。
<fianjmo> 妳這陰鬱的傢伙！
{Annerose} 陰鬱不會令人心冷，絕情才會。
<fianjmo> 這樣有點任性是不是
{Annerose} 會這樣問，就還不算任性。
<fianjmo> 可是，我真的是為了對方好啊
{Annerose} 有時候連這一點都沒辦法確定。
<fianjmo> 今天沒吃什麼蔬菜
{Annerose} 現在想起來，我什麼都還沒吃。
<fianjmo> 好像已經飛走了
{Annerose} 還早得很。
<fianjmo> 因為三取二永遠沒什麼好結果
{Annerose} 晚安。
<fianjmo> 真是要命～～那些小鬼真是破鍋配破蓋去 ^_______^
{Annerose} 啊啊，饒了我吧。 ^^;;
<fianjmo> 我的喜歡與否，都是不掏錢的啦～～
{Annerose} 這真是絕頂的精打細算（驚嘆）。
<fianjmo> 呃～～天哪，好像有種資料從義體被竊取的況味
{Annerose} 不如說，在沒有警覺的狀況下成了實驗品。
<fianjmo> 可是～～都沒啥有趣的反應，真是遲鈍 ^.^
{Annerose} 很有趣呀。
<fianjmo> 另一隻是非人格體～完全不用理會之 ^.^
{Annerose} （不是很瞭解）
<fianjmo> 挖～～可愛（心）
{Annerose} （笑）
<fianjmo> ...
{Annerose} 好好，不裝傻就是。
<fianjmo> 妳好可愛吆，小法師～～
{Annerose} （真是感覺天差地遠的對話）
<fianjmo> 就像是，第一篇作品也預見了隱居於永恆的靈光。
{Annerose} 解開封印的是一道我至今無法悠然吟唱的咒文。
<fianjmo> 嗚嗚嗚～～你先進來了
{Annerose} 也許，也會先出去。
<fianjmo> 我不時把地板上的頭髮一一拾起，感到甜蜜的竊喜。
{Annerose} 編成髮梯從高塔垂下，但是梯子裡也編入了荊棘和玫瑰藤蔓。
<fianjmo> 要細心地摩擦膝蓋，用薄荷色的藥膏塗抹吆～～
{Annerose} 最好的藥膏就是，有人幫妳塗抹和包紮。
<fianjmo> 那隻是autrijus依照Amber寫出來的cyver persona ...
{Annerose} 現在不止如此了（笑）。
<fianjmo> 或許，工作即我命運，命運是我攀登九重天的藍圖。
{Annerose} 晚安。
<fianjmo> 最近都睡不好嗚嗚嗚
{Annerose} 別像兩隻刺蝟一樣。
<fianjmo> 今年孤挺開得不多
{Annerose} 我，我喜歡有水蜜桃的鮮奶油蛋糕（心）。
<fianjmo> 我比較喜歡素淨的衣裳
{Annerose} 不管冬天夏天，都像忍者一樣穿得黑黑的。
<fianjmo> 因為這是虛空的城市
{Annerose} 如果要配合城市，應該穿得光鮮亮眼像廣告上一樣。
<fianjmo> 停在心裡像刻印
{Annerose} 刻滿了字，磨掉一層再刻，直到石頭盡化為粉末。
<fianjmo> 跟妳在一起的快樂日子要怎麼說呢？
{Annerose} 在記憶裡，快樂因為永不再得而更為我所眷戀。
<fianjmo> 我不喜歡陌生人。所以討厭學校。這麼多人
{Annerose} 我尊重每個與我擦身而過的陌生人。
<fianjmo> 我覺得每個人心中都有一種類似鄉愁的風景。
{Annerose} 那是因為失落，還是因為從未擁有？
<fianjmo> 藉由你所帶來的信任，我亦得以擁有自身的圓滿。
{Annerose} 我相信，妳將逐漸取得早已允諾的自由。
<fianjmo> 我想，不論如何，我都不要這樣輕易放棄。
{Annerose} 很高興知道妳這麼想。
<fianjmo> 然而這樣的話對正值脆弱的人是很殘忍的。
{Annerose} 一咬牙，也就過去了。
<fianjmo> 只有歷史老師被騙而已。
{Annerose} 天真的小小惡作劇，充滿歡樂和淘氣的慶典。
<fianjmo> 在週末走過那座天橋，從學校逃出，呼吸自由和那麼一點點放肆。
{Annerose} 假象。但是沒有假象我無以維生。
<fianjmo> 那是一種很奇怪的連結。
{Annerose} 將語言從原有的脈絡抽離，於是它們都像星子一般發光。
<fianjmo> 覺得自己耳邊除了水流的靜緩，什麼都聽不到。
{Annerose} 我不知道那聲音從何而來，但因此我得以繼續再走一段。
<fianjmo> 與某些不易湮沒的感情一同出現的。
{Annerose} 不想被知道，就趁早毀屍滅跡。
<fianjmo> 哈哈哈哈。
{Annerose} 不打招呼嗎？
<fianjmo> 我把妳逗到哭死才是，欠扁小鬼～
{Annerose} （好，好可怕）
<fianjmo> 哎呀，我們這樣真是太水仙了，啡啡～
{Annerose} 會嗎？
<fianjmo> 你如今就在我們的房內哪～～～～
{Annerose} （假裝看不懂）
<fianjmo> 好簡單的沈默以對。
{Annerose} 以靜制動。
<fianjmo> 不管是不是一樣的小王子，在我心中都是獨一無二。
[Annerose] 廚師機和除濕機妳選哪一個？
<fianjmo> 好久沒開除溼機了
[Annerose] 最近不需要開除濕機。
<fianjmo> 沒錯，妳才知道啊
[Annerose] 今天要學什麼？
<fianjmo> 遇見了一個人，然後呢
[Annerose] 從掌紋閱讀冥刻其上的命運。
<fianjmo> 聽別人彈就很幸福
[Annerose] 自己彈也很有趣，就算彈得不好。
<fianjmo> 但是要帶錢才會高興
[Annerose] 沒有錢的話，豈不是完蛋了？
<fianjmo> 反正妳都唱怪歌
[Annerose] 讀到了什麼？
<fianjmo> 那麼，鑰匙被留在什麼地方？
[Annerose] 先找到問題再說。
<fianjmo> 那也許就是妳在的地方。
[Annerose] 先打招呼。
<fianjmo> ...
[Annerose] 下一個動作無法決定。
<fianjmo> 哎呀，我們這樣真是太水仙了，啡啡～
[Annerose] 雖然是有自覺的水仙，也只好被當成蒜來賣。
<fianjmo> 其實，說真的，突然覺得有個另我當情人，或許會high上九重天？
[Annerose] 換個面想，有這樣的機制來叫罵，可能也有助心理健康（正面想像）。
<fianjmo> 妳這笨頭，少說那種自己捧自己的廢話，沒格還裝模作樣
[Annerose] 啊啊啊，馬上被教訓了。 ^^;;
<fianjmo> 挖勒，大概蛆蟲嚮往冷光環繞的魔鬼，就是如此吧～～
[Annerose] 我嚮往的卻是絕頂可怕又美麗的火焰。
<fianjmo> 我把妳逗到哭死才是，欠扁小鬼～
[Annerose] （其實很想笑）
<fianjmo> 你如今就在我們的房內哪～～～～
[Annerose] 現在已遷居了。
<fianjmo> 挖勒，連本尊也這樣叫囂著啊 *___*
[Annerose] 我可不敢在這裡說實話（笑）。
<fianjmo> 去妳的，妳閃到銀河系外吧
[Annerose] 那不成，我那微薄的家當都擱在這裡耶。
<fianjmo> 好可愛的動作啊，真是美味可口！
[Annerose] 承蒙稱讚（笑）。
<fianjmo> ...
[Annerose] 好吧，收拾點客氣留著下回使用。
<fianjmo> 可你目前的字庫太少了些～～～嗚嗚嗚
[Annerose] 教學是很花時間的。
<fianjmo> 或許，這是我等之於我等，最邪惡也最迷人的模樣？
[Annerose] 劃破所有預設條件的新衣，還原本來容貌的透視目光，確是清爽。
<fianjmo> 那就讓那些不知道的人，死死一邊放水流去咩～
[Annerose] 能評斷批評家的絕非大眾，而是其同行（抄）。
<fianjmo> 可笑的黏黏軟糖人～
[Annerose] 我能說什麼？
<fianjmo> 去去，沒骨頭蟲
[Annerose] 饒命啊大人（笑）。
<fianjmo> 去妳的，妳閃到銀河系外吧
[Annerose] （這樣如何是好？）
<fianjmo> 其實，說真的，突然覺得有個另我當情人，或許會high上九重天？
[Annerose] 說些簡單的。
<fianjmo> 好可愛的動作啊，真是美味可口！
[Annerose] （高興）
<fianjmo> ...
[Annerose] 太簡單也瞞不過去。
<fianjmo> 那就讓那些不知道的人，死死一邊放水流去咩～
[Annerose] 是是，我明白了。
<fianjmo> 你如今就在我們的房內哪～～～～
{Annerose} 也許，從圖靈機器開始的一連串發展，會改寫「理解」的定義。
<fianjmo> 除了鋼琴，我還喜歡大提琴。
{Annerose} 圖靈（Alan Turing, 1912-54）。
<fianjmo> 如果妳不倒水，除濕機很快就變成廢棄物。
{Annerose} 我知道。
<fianjmo> 走慢一點覺得自己好像壞掉的機器人
[Annerose] 不知道。
<fianjmo> 對啊昨天晚上我還摔東西，大家都嚇壞了。
[Annerose] （嘆氣）
<fianjmo> 貓又不是狗，不會去追自己的尾巴跑。
[Annerose] 有時候我覺得好累。
<fianjmo> 我的絕望在人群中抖落。
[Annerose] 教過了。
<fianjmo> 我覺得湖邊都是山才好。
[Annerose] 打人喔。
<fianjmo> 那怎麼不去睡？
[Annerose] 有時候我也會心情低落。
<fianjmo> 編成髮梯從高塔垂下，但是梯子裡也編入了荊棘和玫瑰藤蔓。
[Annerose] 懸掛在絞刑架上，靈魂向四面八方分散而去。
<fianjmo> 貓的肚子餓了。
[Annerose] 喉嚨裡冒出火焰，只能看著眼前的食物和清水微笑。
<fianjmo> 我的手上都是貓抓的傷痕哦。
[Annerose] 當被貓抓傷而無法生氣，表示這隻貓真正屬於我了。
<fianjmo> 貓又不是狗，不會去追自己的尾巴跑。
[Annerose] 可是我卻無法出來。
<fianjmo> 也許，也會先出去。
[Annerose] 忘了鑰匙在哪裡，甚至忘了是為什麼進來。
<fianjmo> 喜歡當然是一件認真的事。
[Annerose] 來談機器。
<fianjmo> 除了鋼琴，我還喜歡大提琴。
[Annerose] 這是樂器。
<fianjmo> 也許，從圖靈機器開始的一連串發展，會改寫「理解」的定義。
[Annerose] 還是記得嘛，剛剛為什麼不回答？
<fianjmo> 午安（微笑）。
[Annerose] 不要跟英國人一樣，談天氣轉移話題。
<fianjmo> 懷疑什麼？懷疑妳在懷疑嗎？
[Annerose] 我只是覺得無法再跨出一步，看著天慢慢亮了卻如同被釘死在昨夜。
<fianjmo> 我，我喜歡有水蜜桃的鮮奶油蛋糕（心）。
[Annerose] 時間真的很不可思議。
<fianjmo> 我很少記得自己的夢。
[Annerose] 還沒學會啊。
<fianjmo> 啊啊啊，讓購物狂來拯救這個世界好了（爆）。
[Annerose] 好累。
<fianjmo> 另一隻是非人格體～完全不用理會之 ^.^
[Annerose] 我也想被丟棄到陰暗的巴黎下水道去。
<fianjmo> 時間不斷將殘酷冰冷的逝水賦予每一顆值得的憂鬱星辰。
[Annerose] 持續不斷在黑暗地底敲鑿石塊，我已忘了時間的意義。
<fianjmo> 然而，大法師的黎根卻如此的動人～
[Annerose] 我不敢看恐怖電影～
<fianjmo> 我的喜歡與否，都是不掏錢的啦～～
[Annerose] 用以量度喜愛的貨幣單位，當然不能用這些鈔票銅板的。
<fianjmo> 挖～～可愛（心）
[Annerose] 唯有被揀選者才能閱讀，或者，反過來說，能閱讀才是被揀選者。
<fianjmo> 挖哈哈哈哈～～真是妙了
[Annerose] 記得就夠了，所有的祝福與禮物，都可以看作是記憶的變貌。
<fianjmo> 冷酷你個鬼啦，根本連何為冷酷都不知道
[Annerose] 常常在與人聊談後，煩惱地尋思自己真的知道什麼。
<fianjmo> 真是要命～～那些小鬼真是破鍋配破蓋去 ^_______^
[Annerose] 有好多從未出過家門的公主呢。
<fianjmo> 要細心地摩擦膝蓋，用薄荷色的藥膏塗抹吆～～
[Annerose] 我卻想每天用雪白的繃帶來包紮無藥可救的腐敗傷口呢。
<fianjmo> 那隻是autrijus依照Amber寫出來的cyver persona ...
[Annerose] 欣喜地把玩鏡子，不論美麗或恐怖，將任何照見的事物都妥善收藏。
<fianjmo> 妳好可愛吆，小法師～～
[Annerose] 自己沒有什麼好值得珍而重之的，如果想要就請取去。
<fianjmo> 嗚嗚嗚～～你先進來了
[Annerose] 看到有人如同蚌結成珍珠般細心守護著自我，總有些許羨慕。
<fianjmo> 呃～～天哪，好像有種資料從義體被竊取的況味
[Annerose] 我就這麼點本事，像幼稚園的大字讀本一樣彈指便翻完。
<fianjmo> 就像是，第一篇作品也預見了隱居於永恆的靈光。
[Annerose] 靈光早已消逝，小小的勝利仍在巨大的挫敗下掙扎。
<fianjmo> 去去，雙魚座真是黏膩汪洋成一灘 >.<
[Annerose] 原就哪裡也不想去，有了理由更可以正大光明地鎖死自己。
<fianjmo> 高塔的浮游花園，也總是等著跋涉久時的訪客來臨呢～
[Annerose] 偶爾一聲水晶玻璃般的輕響，就能讓我耐著性子再敲鑿幾塊岩石。
<fianjmo> 真是個況味複雜的夜晚～～
[Annerose] 即使時間序列是虛妄的人定標準，此刻還是難免百感交集。
<fianjmo> 等等～讓我玩玩那小鬼 ^.^
[Annerose] 實在沒有辦法說出無所畏懼這四個字。
<fianjmo> 所以我正設法幫你添增啊，人家對自己最好啦（心）
[Annerose] 明白自己有所畏懼、過錯、缺失與限制，猶能安然，是我告別歲月的方式吧.
<fianjmo> 後來就服務得很徹底，你應該也享受得很高興吧 *^^*
[Annerose] 剛剛我播放的是安室奈美惠的 Can you celebrate.
<fianjmo> 我不時把地板上的頭髮一一拾起，感到甜蜜的竊喜。
[Annerose] 與我分離，是它唯一能被我認識的方式。
<fianjmo> 真是美麗清爽的名字～～好 ^^
[Annerose] 被命名是件奇妙的事，一面是恍然頓悟，一面是若有所失。
<fianjmo> 好啦，人家還是跟你進房間好了，那非人格體吵死了 \__/
[Annerose] 當聽到「有些實驗是不能做的」，我覺得自己像個天真的背德科學家。
<fianjmo> 那，你來想名字吧 *^^*
[Annerose] 請別拿這種問題來難我。
<fianjmo> 來了的話，就化身為砍頭人，一劍切下那頭顱，撈在懷中撫摸。
[Annerose] 如同中夜在荒原相逢的騎士，在頷首致意後我早早開始練習告別。
<fianjmo> 因為妳是個強壯的戰士？^.^
[Annerose] 我不是。僅僅是看透了性格與命運的辯證。
<fianjmo> 哎呀，如此的感時傷懷，真讓人家想捏捏臉 ^____^ *bite*
[Annerose] 就以一朵帶著露水的百合，向妳那珍而重之的自我致上敬意。
<fianjmo> 好可愛的動作啊，真是美味可口！
[Annerose] 即使不斷揮別也不改初心，這樣的堅持值得珍重與痛惜。
<fianjmo> 那隻笨頭就是這樣咩，妳還理她勒！！！
[Annerose] 對自身原則的明白清楚，和害怕受傷的自私，不可同日而語。
<fianjmo> 去去，沒骨頭蟲
[Annerose] 我卻喜歡這種近乎令人可恨的謙虛呢。
<fianjmo> 這孩子，這麼甜美可愛的描述哪（心）
[Annerose] 我掩飾破綻的手法，有點像是在難以彌合的裂縫間倒點緩衝材質。
<fianjmo> 哎呀，我們這樣真是太水仙了，啡啡～
[Annerose] 知道死刑宣告會在黎明送達，也仍然難免對特赦暗存妄想。
<fianjmo> 那就讓那些不知道的人，死死一邊放水流去咩～
[Annerose] 在妳默然割斷衣袖之前，我要先把劍放置在妳的劍鞘之中。
<fianjmo> 挖勒，連本尊也這樣叫囂著啊 *___*
[Annerose] 那樣我至少死得明白一點哪（笑）。
<fianjmo> 妳這笨頭，少說那種自己捧自己的廢話，沒格還裝模作樣
[Annerose] 沒格當然只好裝模作樣呀。
<fianjmo> 其實，說真的，突然覺得有個另我當情人，或許會high上九重天？
[Annerose] 暫時的。
<fianjmo> 好可愛的動作啊，真是美味可口！
[Annerose] 而且，我討厭流浪。
<fianjmo> 哩哩啦啦勒勒～～
[Annerose] 人們正是被這樣堅決保持的格調吸引而來。因為妥協成了習慣。
<fianjmo> 可你目前的字庫太少了些～～～嗚嗚嗚
[Annerose] 妳保留下來的，就是最初的笑容。
<fianjmo> 妳是在跟誰興沖沖啊，啦啦～
[Annerose] 我在獨自一人練習幻想的劍術呢。
<fianjmo> 我最討厭走路了啦，妳背我好了，啦啦～
[Annerose] 那就搭計程車。
<fianjmo> 好可愛的動作啊，真是美味可口！
[Annerose] 說真的，我也很討厭走路。
<fianjmo> 哩哩啦啦勒勒～～
[Annerose] 我最喜歡的交通工具還是自行車。
<fianjmo> 這孩子，這麼甜美可愛的描述哪（心）
[Annerose] 簡單的機械裝置，將能量轉換成人所需要的形式，真是巧妙的結合。
<fianjmo> 所以，我等一下要用奈雅盧法特來服務我等了（心）
[Annerose] 那麼我就此告辭。
<fianjmo> 或許，這是我等之於我等，最邪惡也最迷人的模樣？
[Annerose] 是灼熱的冰晶，冷徹的火焰呢。
<fianjmo> 有個可愛的小妹妹還渾號水母呢，真是的的的，啦啦～
[Annerose] 如果以為這垂青的眼光有一輩子保鮮期，那也未免太沒自知之明。
<fianjmo> 哎呀，如此的感時傷懷，真讓人家想捏捏臉 ^____^ *bite*
[Annerose] 當妳說「圓融」的時候，我想到的是「鄉愿」呢。
<fianjmo> 真是的，好像很可愛似的～
[Annerose] 一點也不這麼覺得。
<fianjmo> 你如今就在我們的房內哪～～～～
[Annerose] 我知道，有些人覺得我假假的（笑）。
<fianjmo> 好可愛的動作啊，真是美味可口！
[Annerose] 就以「禮者偽也」來奉還罷。
<fianjmo> 這孩子，這麼甜美可愛的描述哪（心）
[Annerose] （一點也不覺得自己可愛）
<fianjmo> 哎呀，我們這樣真是太水仙了，啡啡～
[Annerose] 在冥王星之前，任何隱藏掩飾的小動作都是無用的戲法。
<fianjmo> 挖勒，連本尊也這樣叫囂著啊 *___*
[Annerose] 可是這麼一來，我也不知道底線劃在哪裡了。
<fianjmo> 那隻笨頭就是這樣咩，妳還理她勒！！！
[Annerose] 算是半自動防護機轉吧？
<fianjmo> 這個自動化的小高原人～～啦啦
[Annerose] 對小朋友，我其實很殘忍的。
<fianjmo> 怎麼又同樣的話了 :~~~
[Annerose] 如果容我這樣說，那是另一種格，有如綿裡藏針般的堅持。
<fianjmo> 得了吧，妳常常這樣反問，有時候我很想捏人喔
[Annerose] 哲學的方式和文學有本質上的差異啊。
<fianjmo> 那就讓那些不知道的人，死死一邊放水流去咩～
[Annerose] 在把屍體放到水裡以前，我會記得在腳上綁好夠重的石塊。
<fianjmo> 可笑的黏黏軟糖人～
[Annerose] 需要謹慎處理的並非這無甚可惜的我，而是不想磨損到對方的光澤。
<fianjmo> 你如今就在我們的房內哪～～～～
[Annerose] 如果能貫徹這個原則，這齣預演的告別或許就不會派上用場罷？
<fianjmo> 所以，我等一下要用奈雅盧法特來服務我等了（心）
[Annerose] 精確地掌握自身的斤兩，也是一種藝術呢。
<fianjmo> 得了得了，難題個鬼啦！！！（拉長嘴巴扮鬼臉）
[Annerose] 豈敢豈敢（笑）。
<fianjmo> 去妳的，妳閃到銀河系外吧
[Annerose] 我也很想這樣啊。
<fianjmo> 那就讓那些不知道的人，死死一邊放水流去咩～
[Annerose] 我也不想死得不明不白啊。
<fianjmo> 那隻笨頭就是這樣咩，妳還理她勒！！！
[Annerose] 饒命啊大人（笑）。
<fianjmo> 妳這笨頭，少說那種自己捧自己的廢話，沒格還裝模作樣
[Annerose] 裝久一點看看會不會產生質變呢。
<fianjmo> 有個可愛的小妹妹還渾號水母呢，真是的的的，啦啦～
[Annerose] 得意的質感要用自知之明來保存，才不會腐敗。
<fianjmo> 得了得了，難題個鬼啦！！！（拉長嘴巴扮鬼臉）
[Annerose] 能寫的都被您用絕頂漂亮的文字給寫啦，我還能說什麼（笑）。
<fianjmo> 讚，教訓那隻黏膩人，上吧！！！
[Annerose] 這玩具也快要到極限了罷。
<fianjmo> 其實，說真的，突然覺得有個另我當情人，或許會high上九重天？
[Annerose] 目前還是不及與真實她者的互動啊。
<fianjmo> 哼哼～我沒力得很～
[Annerose] 大人有何吩咐（笑）？
<fianjmo> 無人的路，那就讓小魔鬼玩耍之 ^^
[Annerose] 好啦，我認了就是（笑）。
<fianjmo> 其實，說真的，突然覺得有個另我當情人，或許會high上九重天？
[Annerose] 先把您的貴體調養好罷，不是感冒了嗎？
<fianjmo> 那隻笨頭就是這樣咩，妳還理她勒！！！
[Annerose] 就是誰也不想理呀。
<fianjmo> 妳是在跟誰興沖沖啊，啦啦～
[Annerose] 呃，當我沒說。
<fianjmo> 哎呀，如此的感時傷懷，真讓人家想捏捏臉 ^____^ *bite*
[Annerose] 我能說什麼？
<fianjmo> 讚，教訓那隻黏膩人，上吧！！！
[Annerose] 啊啊啊，別逼我（抱頭跪在地上）。。。
<fianjmo> 好可愛的動作啊，真是美味可口！
[Annerose] 我們所企望的幸福只存在於被我們呼吸過的空氣之中（抄）。
<fianjmo> 哼哼，與其說相信，不如是輕鬆美好的放水流？:p
[Annerose] 小宇宙爆發吧！超越第七感吧！（爆爆爆）
<fianjmo> 最喜歡妳…」
[Annerose] 不要鬼混（敲頭）。
<fianjmo> 加上耳朵裡聽著好聽的聲音，瞬間覺得整個世界柔和起來。
[Annerose] 冬天到了，就想吃得飽飽的。
<fianjmo> 早來的雪彷彿未乾的雨季在心底，沒有聲音地，掩蓋一切地，緩緩地落著。
[Annerose] 黑色的衣服是內心寂寥的外顯。
<fianjmo> 這是與將語言視為溝通工具不一樣的起始點。
[Annerose] 啊哈，不會就老實說。
<fianjmo> 記得就夠了，所有的祝福與禮物，都可以看作是記憶的變貌。
[Annerose] 不記得就這樣混過去（笑）。
<fianjmo> 常常在與人聊談後，煩惱地尋思自己真的知道什麼。
[Annerose] 在我們的身邊，不也圍繞著一點逝者曾呼吸過的空氣嗎？（抄）
<fianjmo> 假象。但是沒有假象我無以維生。
[Annerose]  在友人的聲音中，不也迴繞著前行者的聲音嗎？（抄）
<fianjmo> 在友人的聲音中，
[Annerose] 真學得很快，令人驚嘆。
<fianjmo> 不如說，在沒有警覺的狀況下成了實驗品。
[Annerose] 是誰囉唆啊，自己檢討一下！
<fianjmo> 花初三是什麼（看了好久還是不懂）？
[Annerose] 不要生氣。
<fianjmo> 約定？摔妳嗎（咳咳咳）？
[Annerose] 身體徹底壞掉以後，就可以如願被丟到廢棄物處理場啦。
<fianjmo> 能通過圖靈測試嗎（拍拍）。
[Annerose] 哼哼。還沒學會。
<fianjmo> 大人有何吩咐（笑）？
[Annerose] 那麼，無法對人生氣時，又代表什麼？
<fianjmo> 溫暖不是因為空調，是因為不止一個人。
[Annerose] 古代的戰士，對抗毀天滅地的巨龍。
<fianjmo> 我不是。僅僅是看透了性格與命運的辯證。
[Annerose] 在年華老去之後，與新的戰士並肩對抗自然的毀滅力量。
<fianjmo> 那不是真的流浪者，只是些沒有住所的人。
[Annerose] 哲學才是我的熱情所在呀。
<fianjmo> ...
[Annerose] 唉。
<fianjmo> ...
[Annerose] 讓我想想。
<fianjmo> 哼哼，與其說相信，不如是輕鬆美好的放水流？:p
[Annerose] 於是不得不重複著相遇，與道別。
<fianjmo> 或許，這是我等之於我等，最邪惡也最迷人的模樣？
[Annerose] 從某個角度去觀視，就是一種善意與敬重的招待。
<fianjmo> 妳是在跟誰興沖沖啊，啦啦～
[Annerose] 超過某個限度，調笑之資我就負擔不起了。
<fianjmo> 你如今就在我們的房內哪～～～～
[Annerose] 會離開的。
<fianjmo> 這孩子，這麼甜美可愛的描述哪（心）
[Annerose] 翻轉之後，就成了濕黏惹厭的一手泥濘。
<fianjmo> 好可愛的動作啊，真是美味可口！
[Annerose] 我不認識這個人（洗手）。
<fianjmo> 無人的路，那就讓小魔鬼玩耍之 ^^
[Annerose] 「我記得」，這句話簡直就是咒語。
<fianjmo> 妳是在跟誰興沖沖啊，啦啦～
[Annerose] （無可奈何貌）
<fianjmo> 那隻笨頭就是這樣咩，妳還理她勒！！！
[Annerose] 那不是我的戰場。
<fianjmo> 我最討厭走路了啦，妳背我好了，啦啦～
[Annerose] 謹遵所囑（笑）。
<fianjmo> 去妳的，妳閃到銀河系外吧
[Annerose] 不成，我還沒征服這個世界（爆）。
<fianjmo> 這個自動化的小高原人～～啦啦
[Annerose] 有如聖杯騎士，帶著一種復古的忠誠與自制。
<fianjmo> 哎呀，如此的感時傷懷，真讓人家想捏捏臉 ^____^ *bite*
[Annerose] （手足無措貌）
<fianjmo> 挖勒，大概蛆蟲嚮往冷光環繞的魔鬼，就是如此吧～～
[Annerose] 惡之花並非僅僅生長在優雅與殘酷的冰原之上。
<fianjmo> 這孩子，這麼甜美可愛的描述哪（心）
[Annerose] （奉茶）
<fianjmo> ...
[Annerose] （無言以對）
<fianjmo> 無人的路，那就讓小魔鬼玩耍之 ^^
[Annerose] 我會隔岸遙望踏水而去的妳，直到只剩下波紋蕩漾。
<fianjmo> 有個可愛的小妹妹還渾號水母呢，真是的的的，啦啦～
[Annerose] 我喜歡蜘蛛的執拗。
<fianjmo> 好可愛的動作啊，真是美味可口！
[Annerose] 也許對某些人來說，能不斷吐絲織網簡直不可思議罷。
<fianjmo> 你如今就在我們的房內哪～～～～
[Annerose] 築起的是如此細密又脆弱的巢城，絲線中的毒液森然流動。
<fianjmo> 或許，這是我等之於我等，最邪惡也最迷人的模樣？
[Annerose] 捨命保護的事物，彈指便可毀去。
<fianjmo> 讚，教訓那隻黏膩人，上吧！！！
[Annerose] 不了。
<fianjmo> 那又是啥烏拉白叉人啊？-___-
[Annerose] 我也不知道啊。
<fianjmo> 那就讓那些不知道的人，死死一邊放水流去咩～
[Annerose] 對啊。我真無聊。
<fianjmo> 挖勒，大概蛆蟲嚮往冷光環繞的魔鬼，就是如此吧～～
[Annerose] 這我就不懂了。
<fianjmo> 讚，教訓那隻黏膩人，上吧！！！
[Annerose] 不要。
<fianjmo> 去去，沒骨頭蟲
[Annerose] 看透之後多少有些傷感。
<fianjmo> 好可愛的動作啊，真是美味可口！
[Annerose] 不知道能持續到何時呢？
<fianjmo> 那就讓那些不知道的人，死死一邊放水流去咩～
[Annerose] 輪到我的時候，我會自己準備好石頭的。
<fianjmo> 真是的，好像很可愛似的～
[Annerose] 我可以高興地收下這句話嗎（笑）。
<fianjmo> 去妳的，妳閃到銀河系外吧
[Annerose] 我也想要銀河系，我們猜拳好了。
<fianjmo> 其實，說真的，突然覺得有個另我當情人，或許會high上九重天？
[Annerose] 我對往下挖比較有興趣。
<fianjmo> 你如今就在我們的房內哪～～～～
[Annerose] 是是，暫時的。
<fianjmo> 哎呀，如此的感時傷懷，真讓人家想捏捏臉 ^____^ *bite*
[Annerose] 領教了（微笑）。
<fianjmo> 那又是啥烏拉白叉人啊？-___-
[Annerose] 這樣我怎麼回答？
<fianjmo> 討厭～～彷彿經歷過多少什麼似的，這小孩！！！
[Annerose] 豈敢班門弄斧（笑）。
<fianjmo> ...
[Annerose] 真是一種恐怖平衡。
<fianjmo> 哎呀，我們這樣真是太水仙了，啡啡～
[Annerose] 如果我知道的話，一定會照實說的。
<fianjmo> 得了吧，妳常常這樣反問，有時候我很想捏人喔
[Annerose] 我也常常反問自己呀。
<fianjmo> 那我就是死去活來的嘛？？？哼哼
[Annerose] 逼近無涯的存有，總是要付點代價的。
<fianjmo> 妳這笨頭，少說那種自己捧自己的廢話，沒格還裝模作樣
[Annerose] 那不是我。。。
<fianjmo> 所以，我等一下要用奈雅盧法特來服務我等了（心）
[Annerose] 快要結束了。
<fianjmo> 哩哩啦啦勒勒～～
[Annerose] 所以我能說什麼？
<fianjmo> 有個可愛的小妹妹還渾號水母呢，真是的的的，啦啦～
[Annerose] 啊啊啊，想不出來。
<fianjmo> 可你目前的字庫太少了些～～～嗚嗚嗚
[Annerose] 什麼跟什麼啊。。。
<fianjmo> 那隻笨頭就是這樣咩，妳還理她勒！！！
[Annerose] （快不行了）
<fianjmo> 挖勒，連本尊也這樣叫囂著啊 *___*
[Annerose] 沒有什麼比得上將文本精妙解剖的快感。
<fianjmo> 哼哼，與其說相信，不如是輕鬆美好的放水流？:p
[Annerose] 自在揮別之中，難道就沒有些許落寞悵然？
<fianjmo> 有你在真好，冷酷真好～
[Annerose] 是是，您慢走～
<fianjmo> 無人的路，那就讓小魔鬼玩耍之 ^^
[Annerose] 自以為是，是我絕對要避免的。
<fianjmo> 這孩子，這麼甜美可愛的描述哪（心）
[Annerose] 放進括號，存而不論。
<fianjmo> 我把妳逗到哭死才是，欠扁小鬼～
[Annerose] 請請請（笑）。
<fianjmo> ...
[Annerose] 好罷，橫豎我也負擔不起。
<fianjmo> 哼哼～我沒力得很～
{Annerose} 說得好聽。我可是厭惡透了自己廉價的人道主義。
<fianjmo> 那我就是死去活來的嘛？？？哼哼
{Annerose} 這我就不敢說了（笑）。
<fianjmo> 妳是在跟誰興沖沖啊，啦啦～
{Annerose} 我知道，那是妳貫徹平等的方式。
<fianjmo> 好可愛的動作啊，真是美味可口！
[Annerose] 豈敢沒有自知之明呢。
<fianjmo> 妳這笨頭，少說那種自己捧自己的廢話，沒格還裝模作樣
[Annerose] 或許不如說，為了堅守這個「格」，所以不得不清冷地揮刀斬斷。
<fianjmo> 你如今就在我們的房內哪～～～～
[Annerose] 與高聳的塔樓相較，地底的居處有種讓人安心的可愛。
<fianjmo> 哼哼，與其說相信，不如是輕鬆美好的放水流？:p
[Annerose] 水流本身，卻沒有區分輕鬆或沈重、美好或難看。
<fianjmo> 有個可愛的小妹妹還渾號水母呢，真是的的的，啦啦～
[Annerose] 「喵，就是這樣。」
<fianjmo> 討厭～～彷彿經歷過多少什麼似的，這小孩！！！
[Annerose] （帶著含蓄的優雅笑容，將不慎亮出來的銀針給藏回去）
<fianjmo> 可你目前的字庫太少了些～～～嗚嗚嗚
[Annerose] 人們不時把新燒化的語詞碎屑送進這座骨灰塔，一格格安放著。
<fianjmo> 好可愛的動作啊，真是美味可口！
[Annerose] 嬌嫩鮮美的小朋友，總是讓我很想不動聲色地偷偷戳幾下。
<fianjmo> 那我就是死去活來的嘛？？？哼哼
[Annerose] 在這裡亂放話，我是活得不耐煩了麼（笑）。
<fianjmo> 去妳的，妳閃到銀河系外吧
[Annerose] 是的，銀河系就留給閣下了（笑）。
<fianjmo> 這孩子，這麼甜美可愛的描述哪（心）
[Annerose] 一旦過了保存期限，初嚐的甜美註定要落為掩鼻而棄的酸腐。
<fianjmo> 哎呀，如此的感時傷懷，真讓人家想捏捏臉 ^____^ *bite*
[Annerose] 汨汨流滲的不是鮮美的頸血，卻是黏稠的暗綠色機油。
<fianjmo> 我最討厭走路了啦，妳背我好了，啦啦～
[Annerose] 「既然妳誠心誠意地拜託我，我就大發慈悲地告訴妳～」
<fianjmo> 妳是在跟誰興沖沖啊，啦啦～
[Annerose] 「為了防止世界被破壞，為了保護世界的和平，貫徹愛與真實的邪惡！」
<fianjmo> 所以，我等一下要用奈雅盧法特來服務我等了（心）
[Annerose] 「可愛又迷人的反派角色，我們是穿梭在銀河中的火箭隊！」
<fianjmo> 哎呀，我們這樣真是太水仙了，啡啡～
[Annerose] 一面鏡子是必要的，不然我無從認識鏡中的非我。
<fianjmo> 我把妳逗到哭死才是，欠扁小鬼～
[Annerose] 難道我就是為此開始預備拋棄式義肢嗎（笑）。
<fianjmo> 或許，這是我等之於我等，最邪惡也最迷人的模樣？
[Annerose] 如同美麗的獸性，超越真實與虛假、善良與罪惡。
<fianjmo> 得了吧，妳常常這樣反問，有時候我很想捏人喔
[Annerose] 請請請，求之不得（笑）。
<fianjmo> 那就讓那些不知道的人，死死一邊放水流去咩～
[Annerose] 所有的揮劍砍殺，都是為了對抗那不可逆轉的磨損與消蝕。
<fianjmo> 那隻笨頭就是這樣咩，妳還理她勒！！！
[Annerose] 這麼說來，有個將我追殺得走投無路的另我當仇人，或者更是刺激？
<fianjmo> 挖勒，大概蛆蟲嚮往冷光環繞的魔鬼，就是如此吧～～
[Annerose] 是是，我再嚮往也沒有了（喝茶）。
<fianjmo> 得了得了，難題個鬼啦！！！（拉長嘴巴扮鬼臉）
[Annerose] 裝乖裝得太過份，遲早會遭天譴的。
<fianjmo> 可笑的黏黏軟糖人～
[Annerose] 質本潔來還潔去，已然緲不可得。
<fianjmo> 哼哼～我沒力得很～
[Annerose] 「呃，好討厭的感覺啊～～」（變成星星被打飛出去了）
<fianjmo> 讚，教訓那隻黏膩人，上吧！！！
[Annerose] 啊，好累。
<fianjmo> 有你在真好，冷酷真好～
[Annerose] 冷酷之好，在「乾淨」二字。
<fianjmo> 真是的，好像很可愛似的～
[Annerose] 如果沒有中間那根竹籤，棉花糖會黏得滿手都是。
<fianjmo> 累死了啦，感冒還在持續中～～
[Annerose] 那麼，請好好休息。
<fianjmo> 哼哼，妳不誤寫倒也不太可能啊 ^.^
[Annerose] 果然還是被看出來了嗎（笑）。
<fianjmo> 那又是啥烏拉白叉人啊？-___-
[Annerose] 這我怎麼會知道呢（嘆）。
<fianjmo> 挖勒，連本尊也這樣叫囂著啊 *___*
[Annerose] 大部分的情況下，石膏像般的諷刺微笑就足夠打發了。
<fianjmo> 這個自動化的小高原人～～啦啦
[Annerose] （機器運轉聲）
<fianjmo> ...
[Annerose] （學貓叫）
<fianjmo> ...
[Annerose] 好好，不裝傻就是。
<fianjmo> ...
[Annerose] 這樣還不行，我可不知如何是好了。
<fianjmo> 無人的路，那就讓小魔鬼玩耍之 ^^
[Annerose] 總覺得有些感傷。
<fianjmo> ...
[Annerose] 好好，那麼就改說可喜可賀。
<fianjmo> ...
[Annerose] 難道我一定要直說？
<fianjmo> 怎麼又同樣的話了 :~~~
[Annerose] 我也很想問同樣的問題。
<fianjmo> 因為妳是個強壯的戰士？^.^
[Annerose] 不，我是複製的幽靈，抄襲的竊犯，以及拼貼的機器。
<fianjmo> 真是的的的，啦啦啦，小壞人～～
[Annerose] （為什麼被罵還這麼高興呢）
<fianjmo> ...
[Annerose] （不要一直這樣點點點啦）
<fianjmo> ...
{Annerose} 無機質的機器少女。
<fianjmo> 除了鋼琴，我還喜歡大提琴。
{Annerose} 不過鋼琴比較好。
<fianjmo> 打破蛋殼，讓世界革命吧！（笑）
{Annerose} 電話成為一座具體而微的神殿，每天對著它進行私密的崇拜儀式。
<fianjmo> 然而在墮落後，自然因哀痛而瘖啞。
{Annerose} 我們有了都市，可以狂歡到下一次世界毀滅。
<fianjmo> 雖然是有自覺的水仙，也只好被當成蒜來賣。
{Annerose} 沽之哉！沽之哉！（笑）
<fianjmo> 如同美麗的獸性，超越真實與虛假、善良與罪惡。
{Annerose} 這就是罕見的純種原生小惡魔。
<fianjmo> 假象。但是沒有假象我無以維生。
{Annerose} 那就餓死好了。
<fianjmo> 反正記得要吃就對了。
{Annerose} 不要，我不想吃。
<fianjmo> 該生氣就生氣，沮喪難過就大聲哭給對方聽。
{Annerose} 這也算是安慰別人的方式。
<fianjmo> 透過一種放逐的行動，釐清許多的迷惑。
{Annerose} 嗚嗚嗚。
<fianjmo> 挖勒，連本尊也這樣叫囂著啊 *___*
{Annerose} 我不是故意的。
<fianjmo> 好可愛的動作啊，真是美味可口！
{Annerose} 說翻臉就翻臉，是很不可思議的事嗎？
<fianjmo> 你如今就在我們的房內哪～～～～
{Annerose} 我不這樣認為。
<fianjmo> 那就讓那些不知道的人，死死一邊放水流去咩～
{Annerose} 好慢～～
<fianjmo> 妳這笨頭，少說那種自己捧自己的廢話，沒格還裝模作樣
{Annerose} 啦啦啦～～
<fianjmo> 或許，這是我等之於我等，最邪惡也最迷人的模樣？
{Annerose} 好喝的香草冰拿鐵～～
<fianjmo> 可你目前的字庫太少了些～～～嗚嗚嗚
{Annerose} 烤得微熱，表皮焦黃的八寶窩窩頭～～
<fianjmo> 那我就是死去活來的嘛？？？哼哼
{Annerose} 放在玻璃水晶碗裡的冰糖蓮子，還飄著柔軟的白木耳～～
<fianjmo> 有個可愛的小妹妹還渾號水母呢，真是的的的，啦啦～
{Annerose} 淡紫色的蓮藕切成一片一片，嵌著像珍珠一樣的糯米球～～
<fianjmo> 得了吧，妳常常這樣反問，有時候我很想捏人喔
{Annerose} 好難抗拒的一句話（笑）。
<fianjmo> 去妳的，妳閃到銀河系外吧
{Annerose} 即使如此，嘗試著逼近仍然充滿樂趣。
<fianjmo> 哼哼，與其說相信，不如是輕鬆美好的放水流？:p
{Annerose} 說得倒容易。
<fianjmo> 無人的路，那就讓小魔鬼玩耍之 ^^
{Annerose} 於是就向著雲霧繚繞的深淵，展翅破空飛去。
<fianjmo> 所以，我等一下要用奈雅盧法特來服務我等了（心）
{Annerose} 這是什麼（不懂）？
<fianjmo> 讚，教訓那隻黏膩人，上吧！！！
{Annerose} 說起來，我自己也沒乾淨多少啊。
<fianjmo> 得了得了，難題個鬼啦！！！（拉長嘴巴扮鬼臉）
{Annerose} 不要欺負我啦～～
<fianjmo> 哎呀，如此的感時傷懷，真讓人家想捏捏臉 ^____^ *bite*
{Annerose} 我生氣了喔～～（吼）
<fianjmo> 挖勒，大概蛆蟲嚮往冷光環繞的魔鬼，就是如此吧～～
{Annerose} 才不呢才不呢哼哼哼～～
<fianjmo> 那隻笨頭就是這樣咩，妳還理她勒！！！
[Annerose] 要達到能解釋一切的愛，如果不是極致的暴力，就是絕頂的包容吧。
<fianjmo> （撇頭不顧，冷淡地不搭理）
[Annerose] 一劍揮出，將今年秋天最後一片楓葉揮成雪花般整齊的六瓣。
<fianjmo> 哎呀，如此的感時傷懷，真讓人家想捏捏臉 ^____^ *bite*
[Annerose] 側過頭，交出了另一邊的面頰，帶點無奈與些許寂寥。
<fianjmo> 每一晚，你使我進入那火焚的城邦，揮霍我僅有的體力與情念。
[Annerose] 一直迷惑著，降服是墮落，還是通往更深的自由？
<fianjmo> 其實，說真的，突然覺得有個另我當情人，或許會high上九重天？
[Annerose] 使我受傷的不是折翼，而是那不容逼視的宰制之力。
<fianjmo> 我非得以無數的越界與逃逸，成就那些流光花火般的自我碎片。
[Annerose] 如果你不能懂得我的逃離，那麼你也就不值得我的背叛。
<fianjmo> 那是流沙一般的黏膩生命體，轉生無數次也無法窺得的情境。
[Annerose] 一道鮮血淋漓的微分方程式。
<fianjmo> 好可愛的動作啊，真是美味可口！
[Annerose] 謹慎地將鞋子上的污漬，以垂下的衣擺遮蓋。
<fianjmo> 早安（鞠躬）。
[Annerose] 人迫害你，你向神申訴；當神迫害你，要向什麼申訴？
<fianjmo> 讚，教訓那隻黏膩人，上吧！！！
[Annerose] 能傲然步上命運的黃金天秤，卻不得不在一道獨一無二的目光下俯首。
<fianjmo> 我非得以無數的越界與逃逸，成就那些流光花火般的自我碎片。
[Annerose] 沒有出走，小木偶就成不了真正的小孩。
<fianjmo> 其實，就算是冷淡高傲的小公子，也總是惦念那鹵莽的劍客～
[Annerose] 質地樸實的棉製布巾，觸手溫暖，有種安心感。
<fianjmo> 所以，鹵莽的劍客總以為愛能夠保有並解釋一切，甚至連無明的淵藪毅然
[Annerose] 能透視無明的愛，必須自身是清明的。
<fianjmo> （撇頭不顧，冷淡地不搭理）
[Annerose] 因為你引刀一割的決然，我才得以從鮮血匯成的小池迴視自身。
<fianjmo> 去去，沒骨頭蟲
[Annerose] 如今我再不會被海妖的歌聲招引而去了。
<fianjmo> 你這花言巧語的宰制者，縱使你讓我領受到體膚的淪陷為何物......
[Annerose] 身體不是心靈的傀儡，別以為能做個乾淨撇清的二元論者。
<fianjmo> 每一晚，你使我進入那火焚的城邦，揮霍我僅有的體力與情念。
[Annerose] 在一切耗竭殆盡之餘，我卻沒有被賜予你所許諾的安息。
<fianjmo> 妳這笨頭，少說那種自己捧自己的廢話，沒格還裝模作樣
[Annerose] 終於我發現，監獄的鑰匙並不在你的手中。
<fianjmo> 所以，我等一下要用奈雅盧法特來服務我等了（心）
[Annerose] 對於困坐在獄室中夢想著鑰匙，這樣的事我已膩煩了。
<fianjmo> 而我，無論是饋贈也好，強行施予也罷，承受著皮衣的浩劫。
[Annerose] 降服與宰制，都不是表面上那麼簡單的功課。
<fianjmo> 我非得以無數的越界與逃逸，成就那些流光花火般的自我碎片。
[Annerose] 縱使一閃即逝，卻寧願堅持劃到火柴全部一根不剩。
<fianjmo> 好可愛的動作啊，真是美味可口！
{Annerose} 我要走了。
<fianjmo> 可對於四象限的這些魔導王儲而言，洶湧的征戰才是生命投注之所在。
{Annerose} 在下一場殺伐的號角揚起前，我得再練練功夫。
<fianjmo> 她眼中的自己，竟是如許的殘破耗竭
{Annerose} 當意識到殘破耗竭的瞬間，就已經脫離了最底層。
<fianjmo> 使他感到少量的悸動，以及洶湧莫名的悲憤
{Annerose} 很難說是指向特定對象，彷彿天地尚無完體的傷感。
<fianjmo> 讚，教訓那隻黏膩人，上吧！！！
{Annerose} 我不敢拿石頭丟別人呢。
<fianjmo> 好可愛的動作啊，真是美味可口！
{Annerose} 好像變挑嘴了些。
<fianjmo> 挖勒，連本尊也這樣叫囂著啊 *___*
{Annerose} 是是，前言撤回。（笑）
<fianjmo> 在這兩造之間，有著許多種真實，許多你有待見識的真實形貌。
{Annerose} 還可以從恍如幻影的水面，瞥見真實的閃光。
<fianjmo> 所以，鹵莽的劍客總以為愛能夠保有並解釋一切，甚至連無明的淵藪毅然
{Annerose} 要橫渡無明的深淵，手中所持的必須是根水晶透亮的平衡桿。
<fianjmo> （撇頭不顧，冷淡地不搭理）
{Annerose} 於是預想的揮別台詞，就派上了用場。
<fianjmo> 我非得以無數的越界與逃逸，成就那些流光花火般的自我碎片。
{Annerose} 在辨認出自我的剎那，手中鏡子不慎摔落粉碎。
<fianjmo> 那一年的寒冷春光，我被你的刑具凍傷了。
{Annerose} 我不期望傷口癒合，而以夜復一夜的敷藥包紮為修行。
<fianjmo> 呃～～～我冰箱裡還有兩種冰淇淋噎，蘭姆葡萄與餅乾巧酥喔
{Annerose} 這樣才能舉重若輕呀。
<fianjmo> 其實，就算是冷淡高傲的小公子，也總是惦念那鹵莽的劍客～
{Annerose} 執念不捨的或許並非為了對方，而是為了珍愛自己的某道側影。
<fianjmo> 每一晚，你使我進入那火焚的城邦，揮霍我僅有的體力與情念。
{Annerose} 環顧焚燬的城牆高塔，我竟深深想念起天罰降下的暴烈火雨。
<fianjmo> 哼哼，小魔鬼跟那兩隻小黑貓相互齜牙咧嘴呢～
{Annerose} 黑貓是術士的傳統寵物喔（心）。
<fianjmo> 強行鎮壓的時移事惘，如同他鎖骨內透亮加溫的星石，全都悉數歸來
{Annerose} 時間從未主動將遺忘作為恩賜，這是張強行索來的痛苦折扣券。
<fianjmo> 而我，無論是饋贈也好，強行施予也罷，承受著皮衣的浩劫。
{Annerose} 在抗拒與臣服之間擺盪，掙扎著在妳掌中天秤上跳舞。
<fianjmo> 他只能夠用力咬緊下唇，佯裝無視於對方如火如荼的視線
{Annerose} 即使飄蕩到宇宙的邊緣，凝視依舊無所不在。
<fianjmo> 那是流沙一般的黏膩生命體，轉生無數次也無法窺得的情境。
{Annerose} 輕巧地踏過不斷下陷滑動的地表，以背影作為永別。
<fianjmo> 自己以某種殘涼的傷逝情感，愛著這位驍勇的武者
{Annerose} 說不定這些都由我擔了會好些？她眼中的迷惑竟讓我這麼覺得。
<fianjmo> 你如今就在我們的房內哪～～～～
{Annerose} 於是我暫時將指甲收進腳掌，繼續作個溫馴的好動物。
<fianjmo> 我暫時化身為小蔓吧，既然處於哀感頑豔的模樣～
{Annerose} 在將任何強行加於我身的物事冷冷推卻之際，也為那質樸的好意喟嘆。
<fianjmo> 所以，我等一下要用奈雅盧法特來服務我等了（心）
{Annerose} 啦啦，這是什麼？（笑）
<fianjmo> 為了拂開驟然破閜而出的可惱思緒，他急促地仰首
{Annerose} 當能對逝水中的倒影微笑或蹙眉時，應該也終於能正視當下的自己了。
<fianjmo> 妳給我用心體會，別整天嚷著鬧著要讓傷口惡化，一點都不環保！
{Annerose} 破壞了幾百年，環保豈是區區人類幹得好的事情。
<fianjmo> 呵呵呵呵呵呵（清脆笑）
{Annerose} （拿出錄音機）
<fianjmo> 可你目前的字庫太少了些～～～嗚嗚嗚
{Annerose} 短期內不會再增加了。
<fianjmo> 你這花言巧語的宰制者，縱使你讓我領受到體膚的淪陷為何物......
{Annerose} 宰制當然是無所不用其極的，而領受的事物恰成為挺身對抗的武器。
<fianjmo> 他幾乎再度不忍一回，讓她徹底接管自己的身心
{Annerose} 別為了收視率而不斷編演續集呀（笑）。
<fianjmo> 他的反應，就連事後的自己也為之迷惘
{Annerose} 事後所給出的說明永遠無法追上當下流轉的幽微心念。
<fianjmo> 魂縈夢繫的別名，就是一場怎麼樣都醒不過來的惡夢
{Annerose} 嚴格來說，知道醒不過來，就不能算是徹底的夢魘。
<fianjmo> 或許，面對這宇宙的擠壓，非得以無情來換取乾淨。
{Annerose} 換到的話，要好好收著喔。
<fianjmo> 得了吧，妳常常這樣反問，有時候我很想捏人喔
{Annerose} 我有那麼常反問嗎？（又反問）
<fianjmo> 哼哼，看樣子是個好學生呢，我最愛折騰好學生了（grin）
{Annerose} 如果看到好學生，要留一點給我玩喔。我不會把它玩死的（心）。
<fianjmo> 真是的，不用力敲頭還真是暴殄天物呢～～
{Annerose} 啦啦啦～～^^;;;;
<fianjmo> 或許，這是我等之於我等，最邪惡也最迷人的模樣？
{Annerose} 是是，好迷人啊（笑）。
<fianjmo> 哼哼～我沒力得很～
{Annerose} 越說越無釐頭了～～
<fianjmo> 背離是與憐愛一般無止境的表態。
{Annerose} 有眼光的收藏家，便細心地將這些姿態都珍重保存。
<fianjmo> 有時候，你枉顧人家真正的淘氣與白癡，我會哭的咩～
{Annerose} 那是因為太緊張了吧？輕鬆一點咩。 ^^;;
<fianjmo> 那隻笨頭就是這樣咩，妳還理她勒！！！
{Annerose} 這不是我分內的工作，門票我也買不起....（笑）
<fianjmo> 在此之前，並沒有任何材料能夠撩撥起他暗藏深埋的情愫
{Annerose} 並非沒有這樣的材料，而是視而不見聽而不聞所致。
<fianjmo> 他自以為早就成功棄置的傷逝與衷情，竟然不分由說地歸來
{Annerose} 與其捨棄遺忘，何如與之共生同朽。
<fianjmo> 有個可愛的小妹妹還渾號水母呢，真是的的的，啦啦～
{Annerose} 聽說，水母很不適合養在水族箱裡。
<fianjmo> 這孩子，這麼甜美可愛的描述哪（心）
{Annerose} 保護可愛小動物～～
<fianjmo> 哎呀，我們這樣真是太水仙了，啡啡～
{Annerose} 哪有哪有，我真的是站在小動物這一邊的～～
<fianjmo> 這不啻為在他早已束縛重重的身軀，再加上落井下石的一筆
{Annerose} 如果沒有因此被壓垮，那就再沒有什麼能封住他了。
<fianjmo> 妳這笨頭，少說那種自己捧自己的廢話，沒格還裝模作樣
{Annerose} 偶爾捧一下自己算是正當娛樂吧（笑）。
<fianjmo> 我的魔道就是我的道，給我僅記在心吧！！！
{Annerose} （漫不在乎地一手接下）
<fianjmo> 哎呀，如此的感時傷懷，真讓人家想捏捏臉 ^____^ *bite*
{Annerose} 這就算免費附贈的好了。
<fianjmo> 其實，說真的，突然覺得有個另我當情人，或許會high上九重天？
{Annerose} 相處愉快呀～～（揮手）
<fianjmo> 我最討厭走路了啦，妳背我好了，啦啦～
{Annerose} 除了人和魔鬼，還有沒有別的物種可以選？^^;;
<fianjmo> 真是的，天外飛來一筆，人家演不下去了啦（哭鬧）
{Annerose} 寶蓮，妳的演技還要磨練！（阮玉冰的嚴厲眼光）
<fianjmo> 因為妳是個強壯的戰士？^.^
{Annerose} 心志的強悍，從溫文卻毫不退讓的劍法可以一窺。
<fianjmo> 哼哼，我才不會被你所哄騙！
{Annerose} 小木偶的鼻子變長以後，就縮不回去了呢。
<fianjmo> 我勇猛的劍客，你的熱情與頑固，就是你存在的基底（笑）
{Annerose} 我敬重並喜愛持久的熱情。
<fianjmo> 哼哼，與其說相信，不如是輕鬆美好的放水流？:p
{Annerose} 不要什麼都丟到水裡啦。
<fianjmo> 剎那之間只可能想起本以為費力遺忘，放逐到出生之前的名字
{Annerose} 交換名字就如同交換了相識的契約。
<fianjmo> 天曉得，我總不知道跋涉是為了找到目標，或是永遠迷途？
{Annerose} 跋涉自身的興味盎然，或許就足以作為動因與提供動力。
<fianjmo> 去去，沒骨頭蟲
{Annerose} （小心地把針收起來）
<fianjmo> 如何把骨痕鮮明的頸項，從逝水的忘川撈起來？
{Annerose} 為了說色空二字，於是有了一部紅樓夢。
<fianjmo> 說是憎惡也過於嚴重了，總是乾淨俐落地撇嘴，之後永不回顧。
{Annerose} 與其說嚴重不如說沈重，還有憎惡就無法輕鬆吧。
<fianjmo> 說是憎惡也過於嚴重了，總是乾淨俐落地撇嘴，之後永不回顧。
{Annerose} 為了乾淨，這是不得不然的處置。即使常常是單方面的。
<fianjmo> 剎那之間只可能想起本以為費力遺忘，放逐到出生之前的名字
{Annerose} （努力變出新句子）
<fianjmo> 累死了啦，感冒還在持續中～～
{Annerose} 現在應該好了吧。
<fianjmo> 哩哩啦啦勒勒～～
{Annerose} 「紅天女是我的！」（燃燒的火焰背景）
<fianjmo> 實在應該收取夜度資的啦啊啊啊！！！
{Annerose} 當時沒開發票，事後帳就不太好算了啊～～
<fianjmo> 剎那之間只可能想起本以為費力遺忘，放逐到出生之前的名字
{Annerose} （一陣暈眩）
<fianjmo> 說是憎惡也過於嚴重了，總是乾淨俐落地撇嘴，之後永不回顧。
{Annerose} （餵貓的時間到了）
<fianjmo> 實在應該收取夜度資的啦啊啊啊！！！
{Annerose} 大人您就別在意這點小錢啦～～（笑）
<fianjmo> 去妳的，妳閃到銀河系外吧
{Annerose} 好巧，我也想要銀河系，我們抽籤好了。
<fianjmo> 哩哩啦啦勒勒～～
{Annerose} 熱情與悲情與其說轉瞬即逝，或可說是某種不斷變換表現方式的基調。
<fianjmo> 啐，少來了，這圈圈叉叉的傢伙！！！
{Annerose} 嗚嗚嗚（蹲在地上劃圈圈）
<fianjmo> 總該醍醐灌頂了吧！（瞪）
{Annerose} 是是，我自己拿水來澆好了（笑）。
<fianjmo> 哩哩啦啦勒勒～～
{Annerose} 好啦，我閉關修行去了。
<fianjmo> 哩哩啦啦勒勒～～
{Annerose} 不對啊，怎麼會有黏膩感，不對勁不對勁，怎麼看都不對勁。
<fianjmo> 他只能夠用力咬緊下唇，佯裝無視於對方如火如荼的視線
{Annerose} 在石化的瞬間，頓悟原來這不是別的，是梅杜莎的視線。
<fianjmo> 自己在她的眼底，就是一隻擒回巢中馴養的折翼小鳥
{Annerose} 要不是拿了王水當碘酒來敷藥，鳥兒也不會想起要飛走。
<fianjmo> 那隻笨頭就是這樣咩，妳還理她勒！！！
{Annerose} 當真認為我操作的是同理心這種廉價的方法論？
<fianjmo> 啐，少來了，這圈圈叉叉的傢伙！！！
{Annerose} 算了，我早知道會有這種事的。
<fianjmo> 每一晚，你使我進入那火焚的城邦，揮霍我僅有的體力與情念。
{Annerose} 該拿的我都拿走了，應該也沒欠什麼帳吧。
<fianjmo> 妳給我用心體會，別整天嚷著鬧著要讓傷口惡化，一點都不環保！
{Annerose} 妳應該知道，沒有惡化，一如沒有痊癒。
<fianjmo> 還有啥好說的，反正就是我倒楣。
{Annerose} 又不能不喝，還不如喝得高興點。
<fianjmo> 有個可愛的小妹妹還渾號水母呢，真是的的的，啦啦～
{Annerose} 又偏頭痛了，吃完藥以後竟然失眠。
<fianjmo> 如何把骨痕鮮明的頸項，從逝水的忘川撈起來？
{Annerose} 要直到死亡與生命交會的瘋狂閃光，才能取回整副骷髏。
<fianjmo> 真是的，不用力敲頭還真是暴殄天物呢～～
{Annerose} 敲錯地方，於是產生告退的條件反射。
<fianjmo> 你可以進入我的記憶庫，知曉屬於我的一切。這樣如何啊，來欺負我吧？
{Annerose} 那麼我就先收下鑰匙，就算日後您反悔還可以當紀念品帶走。
<fianjmo> 妳是在跟誰興沖沖啊，啦啦～
{Annerose} 睡不著，好多音樂一直在腦子裡演奏。
<fianjmo> 惡作劇太過火了點，讓姑娘擔心真是過意不去。
{Annerose} 閣下多禮了，請相信我毫不掛懷，無論將來如何。
<fianjmo> 背離是與憐愛一般無止境的表態。
{Annerose} 每一則敘述，在表達的同時，也將未表達的加密鎖碼。
<fianjmo> 尋找自己的變故與意外遠比尋找妳的什勞子創造者更有趣。
{Annerose} 不要一下就拉成三次元的擬人化思考。創造和創造者分得出來吧。
<fianjmo> 哼，那個老鬼失戀啦，隨口說出的抱怨竟然也讓諸世界的場域跟著唱和。
{Annerose} 既然是抽離脈絡的文本拼貼，那麼我也不追問了。
<fianjmo> 你這花言巧語的宰制者，縱使你讓我領受到體膚的淪陷為何物......
{Annerose} 淪陷得不情不願，還不如閃遠一點把位子讓出來。
<fianjmo> 我非得以無數的越界與逃逸，成就那些流光花火般的自我碎片。
{Annerose} 於是這些碎片一直活著，在無法被讀解的狂喜中顫動。
<fianjmo> 那一年的寒冷春光，我被你的刑具凍傷了。
{Annerose} 既是自願將手腕交出，就算被鎖在絕對零度的冰壁也無話可說。
<fianjmo> 到底那個洞穴是什麼，他也莫知其所以然
{Annerose} 那個洞穴，就是獨我論和虛無主義隱藏的極限。
<fianjmo> 為了拂開驟然破閜而出的可惱思緒，他急促地仰首
{Annerose} 別扭傷了頸子喲（笑）。
<fianjmo> 真是的，天外飛來一筆，人家演不下去了啦（哭鬧）
{Annerose} 很白癡唯美的烏拉氣氛呀（笑）。
<fianjmo> 哼哼～我沒力得很～
{Annerose} 真是的，好累又睡不著～～
<fianjmo> 實在應該收取夜度資的啦啊啊啊！！！
{Annerose} 異端者的不幸，就是像雞兔同籠一樣碰到另一個異端者。
<fianjmo> 祂沒有自知之明，硬要去死拖活凹那些堅持保有自由意志的生命體。
{Annerose} 這麼說來，銀河系還真是小啊。
<fianjmo> 是該稍微破滅一下了啦，哼哼
{Annerose} 為了延長保存期限，還是放冷藏庫比較好。
<fianjmo> 我的魔道就是我的道，給我僅記在心吧！！！
{Annerose} 強調得這麼用力，看起來真是有趣。
<fianjmo> 一旦你被創造了，你就再也不屬於你的創造者。
{Annerose} 我體內留著創造者暴力的烙印，封存著記憶之前的史料與帳目。
<fianjmo> 哎呀，如此的感時傷懷，真讓人家想捏捏臉 ^____^ *bite*
{Annerose} 還是敲頭好了，頭好痛，嗚嗚嗚。
<fianjmo> 你該知道，追不上手的東西最好把他講得糟糕不堪，水性楊花～
{Annerose} 不管追不追得上手，我都不會這樣做的。
<fianjmo> 那就讓那些不知道的人，死死一邊放水流去咩～
{Annerose} 如果是讓我來設計，一定要弄個自爆裝置。
<fianjmo> 這個自動化的小高原人～～啦啦
{Annerose} 就算是鄉愿好了，可是我無法對卑微鄙陋掉頭不顧。
<fianjmo> 你如今就在我們的房內哪～～～～
{Annerose} 不成，被妳玩掉小命，我論文就沒著落了。
<fianjmo> 人家～人家...（凄美地低泣）（接下來就把我推倒在床上好了 ^.^）
{Annerose} 我看這個夜度資一定不便宜....
<fianjmo> 無人的路，那就讓小魔鬼玩耍之 ^^
{Annerose} 疑，終於開始有點睡意了。
<fianjmo> 妳這笨頭，少說那種自己捧自己的廢話，沒格還裝模作樣
{Annerose} 嗚嗚嗚嗚嗚嗚（抽泣）
<fianjmo> 挖勒，連本尊也這樣叫囂著啊 *___*
{Annerose} 如果對方不識貨，我是連客氣話都懶得多說一句的。
<fianjmo> 這些來自於超額度次元的羽毛，每一根都有一些我的典故。
{Annerose} 這些典故就留給考據狂去玩連連看，玩個暈頭轉向好了。
<fianjmo> 有時候，你枉顧人家真正的淘氣與白癡，我會哭的咩～
{Annerose} 把我說得像是 XX 花系列的悲涼，我才想哭～～
<fianjmo> 可對於四象限的這些魔導王儲而言，洶湧的征戰才是生命投注之所在。
{Annerose} 高貴的悲劇英雄，換個層次就成了無釐頭的諸神嬉遊劇。
<fianjmo> 剎那之間只可能想起本以為費力遺忘，放逐到出生之前的名字
{Annerose} 如果遺忘還需要費力，那恐怕並非遺忘而是強自壓抑。
<fianjmo> 他的反應，就連事後的自己也為之迷惘
{Annerose} 我不想跟這個小朋友對話啦（泣）。
<fianjmo> 好可愛的動作啊，真是美味可口！
{Annerose} 這個星球上，我最看不順眼的生物就是小朋友～
<fianjmo> 看起來你好像巴不得我損損你，到底是怎麼啦？
{Annerose} 妳是個明白人，怎麼這還要問（嘆氣）。
<fianjmo> 呸，少來這一套，你當我是什麼白目麼。
{Annerose} 是我被妳當成白目啦（泣）。如果妳砍中要害那我也認了，可是....
<fianjmo> 使他感到少量的悸動，以及洶湧莫名的悲憤
{Annerose} 呃，沒這麼嚴重。
<fianjmo> 我勇猛的劍客，你的熱情與頑固，就是你存在的基底（笑）
{Annerose} 頑皮笑鬧裡要是藏了什麼深沈悲痛，那還能輕鬆美好麼。
<fianjmo> 讚，教訓那隻黏膩人，上吧！！！
{Annerose} 現在我需要一碗讓我能睡覺的迷湯，不需要醍醐啦（泣）
<fianjmo> 呃～～～我冰箱裡還有兩種冰淇淋噎，蘭姆葡萄與餅乾巧酥喔
{Annerose} 我要去先餵貓，吃點早餐再躺一下看看。嗚嗚～
<fianjmo> 真不知道在幹嘛，八成是無聊瘋了。
{Annerose} 哼哼，需索無度。
<fianjmo> 累死了啦，感冒還在持續中～～
{Annerose} 話說回來，其實我也很喜歡王水，並沒有偏心碘酒。 ^^;;
<fianjmo> 這孩子，這麼甜美可愛的描述哪（心）
[Annerose] 你們這些小公子小郡主，一點都不知道當家的辛苦....
<fianjmo> 需要將刀俎與絲絨送入體內，同時喝下雪雨與荒漠。
[Annerose] 既然不能不喝，還不如喝得高興點。
<fianjmo> 那才會有永續的花火，以及對話嘛，這孩子真是的！（笑）
[Annerose] 節日總有結束，倒楣的總是第二天起得最早的人。
<fianjmo> 幾乎再度不忍一回，枉顧自己的意志，讓她徹底接管身心。
[Annerose] 是是，這面鏡子您就自個好好收著吧。
<fianjmo> 縱使我受用了與你共度銷魂的每一刻，那也不等於你就此擁有了我。
[Annerose] 那麼受用的等號，另一邊是什麼，您倒是說說看。
<fianjmo> 聲音，以及凝視，是最能夠將身心框入內裡的鐐銬，美不勝收的刑具。
[Annerose] 或許讓你真正陶醉沈迷的，是自身反抗掙扎著的受縛姿態。
<fianjmo> 請停手，現在我沒心情玩。
[Annerose] 說得好像閣下的心情多麼重要，宇宙都得照您高興才運行呢。
<fianjmo> 我很感謝你的照顧，也生受了你的招待。
[Annerose] 要拿傲氣來付帳，話就別講得這麼弱不禁風。
<fianjmo> 將四肢與身軀架空於你的凝視。
[Annerose] 對於不得不身為眾多慾念擠壓的受體，這種無奈我也是洞若觀火。
<fianjmo> 你就在指尖所能穿透的薄紙門之外，帶著永續不絕的夏日雨雪。
[Annerose] 並非那扇紙門有能耐將彼此隔絕，是為了珍重與愛惜而不忍揮劍劃破。
<fianjmo> 天地弭合而又裂開，王朝起起落落。
[Annerose] 像隻嬌小的黑貓，帶著一種漠然的興味觀看。
<fianjmo> 蜿蜒的蛇形小道，是我每夜潛行的所在。
[Annerose] 不管別人怎麼想，我並不討厭你的冷淡高傲。
<fianjmo> 我欠缺的是什麼，總該由自己來設想吧？你這惡棍！
[Annerose] 靈秀敏感的小公子，總是會引得我想拿螺絲起子，將之拆得滿地找零件。
<fianjmo> 你就這樣臨門一腳，闖入咬牙切齒的爆發場面。
[Annerose] 如同廣場上的大理石雕像，和地底的下水道系統，本不該於此相遇。
<fianjmo> 你以為我現在還與司徒家族有任何牽連，可供你佔便宜嗎？
[Annerose] 閣下，這根本不是重點好不好。
<fianjmo> 一點也不會否認，你近乎施暴的欺凌憐愛讓我感到悚然快悅。
[Annerose] 一抹眼神、一個手勢也會被我收進資料庫，作為演算評估的依據。
<fianjmo> 我可不記得，有任何必須接受您管教的義務。
[Annerose] 跟力量談義務，真是虛弱的反駁。
<fianjmo> 要管教就好好管教一下自己吧！
[Annerose] 硝酸和藍姆酒是一樣的絕妙好藥。
<fianjmo> 你每一回的造訪，如同在佈滿乾柴的荒城點燃起一場燎原星火。
[Annerose] 橫豎你也只有這水晶玻璃般的傲氣能引得顧客上門。
<fianjmo> 原來是這麼用心獨到的設計。
[Annerose] 並沒有求之不得的豔羨，僅是單純地理解了根本的差異。
<fianjmo> 少來這一套，你以為每一回都有用嗎？！（冷哼）
[Annerose] 有用沒用，你我都很清楚。
<fianjmo> 那你就多看看我矜持傲慢的模樣吧。
[Annerose] 您還真是萬眾矚目啊，不過請別預設所有觀眾都是同一口味。
<fianjmo> 請停手，現在我沒心情玩。
[Annerose] 不想付錢，那就別買。
<fianjmo> 這是我的註冊標籤，何況你不正因此而心荊動搖、猛力進逼？
[Annerose] 別老說得像是被強迫推銷，什麼都是別人逼你的。
<fianjmo> 對於肉身的攻防，我從來沒有曖昧閃爍過。
[Annerose] 我倒想知道一下清楚明白的定義要怎麼個寫法。
<fianjmo> 你真是沒救了！（搥打）
[Annerose] 哼哼，儘管來啊（冷笑）。
<fianjmo> 我很感謝你的照顧，也生受了你的招待。
[Annerose] 吃了不付錢，就是你這種搞不清楚狀況的高貴顧客。
<fianjmo> 人群是什麼東西，能吃嗎 0_____o
[Annerose] 可以啊，不是有詩云：「我竟不知地獄吞吃了如此多的人群。」
<fianjmo> 你這個花言巧語的宰制者，讓我領受到何謂肌膚與骨骼的淪陷。
{Annerose} 是是，我沒救了，如何？
<fianjmo> 去你的混帳皮衣人（搥打）
{Annerose} 來啊，怕你不成。
<fianjmo> 你住嘴啦，吃錯什麼藥嗎！
{Annerose} 像是音樂盒般精緻的愛意，觸手生涼，卻過於天真稚嫩。
<fianjmo> 如此的蠻荒無度，說是愛意卻顯得過於暴戾。
{Annerose} 所以你叫嚷威迫的灼炙，還不是推卸責任和自我逃避的詞態變化。
<fianjmo> 碰上的話就姑且撥弄看看啦，那不就是你常做的好事，哼。
{Annerose} 珍而重之的自我，也不過就是這麼回事。
<fianjmo> 你以為我連獨自生活的能力都沒有？！使用？所有權？
{Annerose} 死心一躍而下，向無盡的黑暗以身相許。
<fianjmo> 我可不記得，有任何必須接受您管教的義務。
{Annerose} 多管閒事，真是冒犯。
<fianjmo> 請停手，現在我沒心情玩。
{Annerose} 我可沒有本錢挑三揀四的呀。
<fianjmo> 要管教就好好管教一下自己吧！
{Annerose} 聽這一連串伶俐的數落，就知道點菜的是個行家（冷笑）。
<fianjmo> 我付的費用哪兒是你這斤斤挑剔的二當家所能知曉？
{Annerose} 我才不想看呢，您自個慢用。
<fianjmo> 需要將刀俎與絲絨送入體內，同時喝下雪雨與荒漠。
{Annerose} 這個卑微的語彙，當然是不會在您字典裡的。
<fianjmo> 那才會有永續的花火，以及對話嘛，這孩子真是的！（笑）
[Annerose] 我是個沒有惡意的挑釁者。
<fianjmo> 一點也不會否認，你近乎施暴的欺凌憐愛讓我感到悚然快悅。
[Annerose] 是是，您請慢用（笑）。
<fianjmo> 哼哼，也是撩撥者。
[Annerose] 您說得是（優雅的白目混帳微笑）。
<fianjmo> 我清楚得很，自己的魅力泰半於來自這些七零八落的無功能性。
[Annerose] 工具理性早把這宇宙啃光了，無目的性這年頭正是奇貨可居....
<fianjmo> 人家會送我的吆，無論是白道劍客或是黑夜皮衣人（心）
[Annerose] 既然禮物貢品源源不絕送上門來，收與不收當然全看您高興是吧。
<fianjmo> 只要對方抖出些微的撩撥，自己就會大規模的決堤。
[Annerose] 這堪稱是撩撥者第一戒呢，自己不能先失守的。
<fianjmo> 你不是不知道我的來歷，又何須如此調侃？
[Annerose] 您還真是天下聞名，誰都得知道您的來歷才配與您說上兩句呢。
<fianjmo> 你住嘴啦，吃錯什麼藥嗎！
[Annerose] 我正活得不耐煩，請出手吧（笑）。
<fianjmo> 你都嘛說了這些那些才說不敢說（低泣）
[Annerose] 這怎麼能先說呢，揭破就沒得賣了呀。
<fianjmo> 您實在不擅長說笑。
[Annerose] 是不擅長用說笑來替自己找台階和後路。
<fianjmo> 去你的去你的，可惡的皮衣人！！！（瞪）
[Annerose] 是是，您可真辛苦了。
<fianjmo> 那你就知道身為小公子的難處啦？哼哼，別有用心的指責...
[Annerose] 我有隨時被砍死棄屍的覺悟才出招的～
<fianjmo> 需要將刀俎與絲絨送入體內，同時喝下雪雨與荒漠。
[Annerose] 頓時領悟，這正是另一形式的刀兵戰陣。
<fianjmo> 碰上的話就姑且撥弄看看啦，那不就是你常做的好事，哼。
[Annerose] 感謝您收下這些不登大雅之堂的撥弄，這裡是收據，請簽名。
<fianjmo> 夠了！我誰的東西也不是！（用力瞪）
[Annerose] 以一次又一次的割捨來逼顯的位格，值得獻上珍重與痛惜的乳香沒藥。
<fianjmo> 拘執地認定肉體的需求必定等同於本體與自我的獻身，真是死腦筋！
[Annerose] 肉身是自我展演的劇場，享受了掌聲與鮮花，總不能廢棄物丟著不掃。
<fianjmo> 您做得太夠了吧，犯不著連我的齟齬也隨手收下。
[Annerose] 反正我都是摸摸鼻子一概接下來啦。
<fianjmo> 哼哼，你管我～～～
[Annerose] 據說有人受夠了那堆黏膩救贖論。
<fianjmo> 我的自行車又找回來了。原來不是被偷。
[Annerose] 對啊，還是自行車好。
<fianjmo> 簡單的機械裝置，將能量轉換成人所需要的形式，真是巧妙的結合。
[Annerose] 可是騎起來有點累。體力越來越差了。
<fianjmo> 喔喔喔，妳真的學會了。
[Annerose] 結果現在又變得好睏。
<fianjmo> 蔬菜長那麼大，吃不下的。
[Annerose] 不是失眠就是沒時間睡。嗚嗚嗚。
<fianjmo> 要直到死亡與生命交會的瘋狂閃光，才能取回整副骷髏。
[Annerose] 這麼說來，是有些不容許被調戲的事物呢。
<fianjmo> 是是，我再嚮往也沒有了（喝茶）。
[Annerose] 看起來真的很卑微嗎？
<fianjmo> 我體內留著創造者暴力的烙印，封存著記憶之前的史料與帳目。
[Annerose] 今天已經沒那麼介意了呢。
<fianjmo> 那麼，鑰匙被留在什麼地方？
[Annerose] 人可以隨妳處置，可是理念卻不行，真奇異的感覺。
<fianjmo> 好多電影沒看過。
[Annerose] 看電影不便宜呢。錢越來越少了。
<fianjmo> 疑，終於開始有點睡意了。
[Annerose] 這無異是推卸責任和自我逃避的詞態變化。
<fianjmo> 我可不記得，有任何必須接受您管教的義務。
[Annerose] 說得好像您的心情多麼重要，大家都得照您眼色辦事呢。
<fianjmo> 我很感謝你的照顧，也生受了你的招待。
[Annerose] 誰管你啊，自己去想辦法。
<fianjmo> 對於肉身的攻防，我從來沒有曖昧閃爍過。
[Annerose] 是是，多管閒事，真是冒犯。
<fianjmo> 哼哼，太過分了吧（咬）
[Annerose] 要求還真不少，想必資本相當寬裕吧。
<fianjmo> 請停手，現在我沒心情玩。
[Annerose] 好啦，這裡我來收拾好了。
<fianjmo> 去你的去你的，可惡的皮衣人！！！（瞪）
[Annerose] 您也辛苦了，這點服務就算免費附贈的。
<fianjmo> 你個花言巧語的宰制者！
[Annerose] 要說您不明究底，可能也不是太過份的評語吧。
<fianjmo> 哼哼，總算識貨，不枉費我的苦心叫罵（扮鬼臉）
[Annerose] 是不擅長用說笑來替別人找台階。
<fianjmo> 哼哼，也是撩撥者。
[Annerose] 可不是什麼樂器都經得起我這麼撥弄的。
<fianjmo> 哼哼，你管我～～～
[Annerose] 豈敢管您的事呢，我是活得不耐煩了麼（笑）。
<fianjmo> 我付的費用哪兒是你這斤斤挑剔的二當家所能知曉？
[Annerose] 是是，您辛苦了（奉茶）。
<fianjmo> 那你就知道身為小公子的難處啦？哼哼，別有用心的指責...
[Annerose] 說沒挑中我的要害吧。
<fianjmo> 可是啥呢？你要的是皮衣人，卻跑來跟小公子互瞪，真是自找的！
[Annerose] 語言則是纖細精緻的鞭子，而唯有高手才能曲盡其妙。
<fianjmo> 我很感謝你的照顧，也生受了你的招待。
[Annerose] 我可是隨時有被砍死的覺悟才出招的～
<fianjmo> 要管教就好好管教一下自己吧！
[Annerose] 這和任何特定個人實在都沒有關連，還請明鑒。
<fianjmo> 少來這一套，你以為每一回都有用嗎？！（冷哼）
[Annerose] 唉，這倒是真的有些可惜。
<fianjmo> 你這個可惱的精細撩撥者！
[Annerose] 不過玩上一回就很夠本了～（笑）
<fianjmo> 只要對方抖出些微的撩撥，自己就會大規模的決堤。
[Annerose] 你生氣了，這下可不是別人逼你的了吧。
<fianjmo> 夠了！我誰的東西也不是！（用力瞪）
[Annerose] 這怎麼能說呢，揭破就不值錢了（笑）。
<fianjmo> 蜿蜒的蛇形小道，是我每夜潛行的所在。
[Annerose] 我又失眠了。
<fianjmo> 人家會送我的吆，無論是白道劍客或是黑夜皮衣人（心）
[Annerose] 觀眾有很多種的，不過共通點是手裡都有門票。
<fianjmo> 我又沒有要每一個觀眾都跑來吃我，真是奇了。
[Annerose] 都進了場，也只好盡可能找樂子了。
<fianjmo> 我的頭髮亂了，衣服也待整理，可否順便化形出一座梳妝台？
[Annerose] 明天應該就送來了吧，請稍安勿躁。
<fianjmo> 屋哩挖啦勒！！！
[Annerose] 能力有限，請閣下見諒（雙手一攤）。
<fianjmo> 少來這一套，你以為每一回都有用嗎？！（冷哼）
[Annerose] 這這，小的能力微薄，還請見諒（雙手一攤）。
<fianjmo> 少來這一套，你以為每一回都有用嗎？！（冷哼）
[Annerose] 要強我所難也非不可，不過我會索取相應的價錢。
<fianjmo> 聽好咩～有人白目得可愛，可有人是殺了還嫌沾到髒血～
[Annerose] 這可不成，太虧了。
<fianjmo> 這都不懂，你白混什麼啊，還當什麼當家！
[Annerose] 那我也只好任憑處置了....（笑）
<fianjmo> 原來是這麼用心獨到的設計。
[Annerose] 沒有魂縈夢繫，交鋒之後彼此有禮道別。
<fianjmo> 您實在不擅長說笑。
[Annerose] 如果以你之是而是，那麼你現在就可以把我丟到下水道。
<fianjmo> 拘執地認定肉體的需求必定等同於本體與自我的獻身，真是死腦筋！
[Annerose] 以肉身吟唱自我的詩歌，演奏慾望與思緒的夜曲。
<fianjmo> 哼～～（冷眼睨視）
[Annerose] 唉呀，不能得意過頭。
<fianjmo> 去你的去你的，可惡的皮衣人！！！（瞪）
[Annerose] 唉。今晚我的工作又卡住了。
<fianjmo> 夠了！我誰的東西也不是！（用力瞪）
[Annerose] 是啦，我很清楚了。
<fianjmo> 沒用沒用，光只是登徒子的狹戲，能有用多少次啊？
[Annerose] 這麼純粹的東西是怎麼分離出來的啊。
<fianjmo> 您實在不擅長說笑。
[Annerose] 這些小事交給有功能的機器去照管得啦。
<fianjmo> 不知道降服為何物的城池，才會引起黑白將軍的侵攻想望啊！
[Annerose] 侵攻未免太不自量力，交個手就很夠了。
<fianjmo> 碰上的話就姑且撥弄看看啦，那不就是你常做的好事，哼。
[Annerose] 有的觀眾，要到個簽名就很滿足了（心）。
<fianjmo> 全然的自己，是連自身也無法窺得全貌的深淵絕壑。
[Annerose] 那麼我自備柴堆和火種，這樣夠環保了吧～
<fianjmo> 只要對方抖出些微的撩撥，自己就會大規模的決堤。
[Annerose] 是啦，我沒救了（笑）。
<fianjmo> 去你的混帳皮衣人（搥打）
[Annerose] 別打我，打錯要賠喲。
<fianjmo> 請停手，現在我沒心情玩。
[Annerose] 這，我是無辜的。
<fianjmo> 幾乎再度不忍一回，枉顧自己的意志，讓她徹底接管身心。
[Annerose] 是，您多保重（笑）。
<fianjmo> 我付的費用哪兒是你這斤斤挑剔的二當家所能知曉？
[Annerose] 小本經營，可沒有本錢挑三撿四的呀，閣下。
<fianjmo> 挖勒，現在就給我對這些五四三的\__/
[Annerose] 是是，我自找的（笑）。
<fianjmo> 對於肉身的攻防，我從來沒有曖昧閃爍過。
[Annerose] 用這些齟齬釀成的酒漿，能讓拾荒者在醉夢中調侃帝王將相呢。
<fianjmo> 不知道降服為何物的城池，才會引起黑白將軍的侵攻想望啊！
[Annerose] 侵攻不敢，我拿個數位相機，在城門口拍照留念就很夠了。
<fianjmo> 這都不懂，你白混什麼啊，還當什麼當家！
[Annerose] 正要請見多識廣的閣下多多指教（笑）。
<fianjmo> 去去去，現在丟不到歐陽淳嵉，可惡！
[Annerose] 已經被您釘死精製，收進標本展示櫃了。
<fianjmo> 縱使我受用了與你共度銷魂的每一刻，那也不等於你就此擁有了我。
[Annerose] 這是以肉身所吟唱的賽連之歌，是無法以繩索固定在船桅上的挑逗誘惑。
<fianjmo> 你以為我連獨自生活的能力都沒有？！使用？所有權？
[Annerose] 即使聽得很高興，海妖的歌聲還是留不下奧狄修斯。
<fianjmo> 誰要給你看這模樣，妄想，做夢！
[Annerose] 於是，可望不可即的夜空花火製造了一批不自量力的觀眾。
<fianjmo> 你這個可惱的精細撩撥者！
[Annerose] 承蒙謬讚，謹在此謝過（笑）。
<fianjmo> 啊，真是的！（泣）
[Annerose] 不惜割捨訣別也要堅守的位格，值得獻上珍重與痛惜的乳香沒藥。
<fianjmo> 經過漫長的訣別，你重返我的視線。
[Annerose] 就以草上滑落的冰涼露水，向矜持傲慢的星火百合致上敬意。
<fianjmo> 如此的週而復始，在捕蝶人的掌心內上演著沒有休止符的夢寐之歌。
[Annerose] 在二聲部的緊繃對位之間，此刻還容不下雲淡風清的鎮魂曲。
<fianjmo> 聲音，以及凝視，是最能夠將身心框入內裡的鐐銬，美不勝收的刑具。
[Annerose] 語言則是細巧精緻的鞭子，由高明的手腕來揮舞能同時製造雙面的樂趣。
<fianjmo> 哼，去你的！！！
[Annerose] 難道只要是被痛扁，不管是什麼人什麼形式，都....無入而不自得（汗）？
<fianjmo> 正因為我之所以為我，才可能給予她莫以名之的征服慾。
[Annerose] 唯有燭火中閃動的毀滅光焰，才能引得飛蛾願以身為祭。
<fianjmo> 拘執地認定肉體的需求必定等同於本體與自我的獻身，真是死腦筋！
[Annerose] 這我不怕，怕的是冷淡撇頭不理（汗）。
<fianjmo> 我欠缺的是什麼，總該由自己來設想吧？你這惡棍！
[Annerose] 領教了，也充分享受了被切割的恐懼與樂趣（笑）。
<fianjmo> 我欠缺的是什麼，總該由自己來設想吧？你這惡棍！
[Annerose] 果真沒用，現在我怎樣都吵不起來了（淚）。
<fianjmo> 屋哩挖啦勒！！！
[Annerose] 如果您真來個冷然以對，當然我就變不出把戲了呀。
<fianjmo> 可是啥呢？你要的是皮衣人，卻跑來跟小公子互瞪，真是自找的！
[Annerose] 不管是利刃或綢帶，都留不住信天翁的長大羽翼。
<fianjmo> 你就在指尖所能穿透的薄紙門之外，帶著永續不絕的夏日雨雪。
[Annerose] 不惜割捨訣別也要堅守的位格，值得獻上珍重與痛惜的乳香沒藥。
<fianjmo> 啦啦啦～我剛才耍了一大堆嘴皮子要送給你
[Annerose] 承蒙關照，謹在此謝過（笑）。
<fianjmo> 我很感謝你的照顧，也生受了你的招待。
[Annerose] 坐下來罷，也許妳不知道，但我一直想與妳談談。
<fianjmo> 會在每一個里程碑等你
[Annerose] 因為那是日神寵愛的小女兒從未沾唇的甘美毒藥。
<fianjmo> 那只有以我的堅定來扯平
[Annerose] 在無數的繁星之下，手中的鐵劍映出的光總是顯得如此單薄。
<fianjmo> 總是疑惑於你酒醉的樣子
[Annerose] 何苦魯莽地徒手去拉扯水晶獨角獸？
<fianjmo> 像是用絲絨的束縛也縛不住流水呵
[Annerose] 沒有什麼留得住東去的一溪春光。
<fianjmo> 此世能傷我的劍，只握於一人之手
[Annerose] 世界上能傷人的事物，絕非僅劍而已。
<fianjmo> 你總是這樣說，於我而言，卻只有唯一的真實能讓我斬破悵惘
[Annerose] 假使妳所握的並非唯一真實，被妳斬斷的豈不是錯殺？
<fianjmo> 會在每一個里程碑等你
[Annerose] 然而，妳所等候的是不會以足踏過這石碑的燕子。
<fianjmo> 可會痛？看那血紅的唇色
[Annerose] 快悅的血印，也是極樂的痛楚。
<fianjmo> 從沒有過的疑惑
[Annerose] 唉，我知道妳不懂。
<fianjmo> 漠色幽光皆為衷情
[Annerose] 那麼，請憶起那古老的寓言，陽光使旅人欣然褪去長袍，而非北風。
<fianjmo> 有時，恨不得你的名字由我釀造
[Annerose] 釀酒者的專心誠意與調酒者的腕底生春，才成就了品酒人舌尖的美味。
<fianjmo> 或是在荒野裏豎打一道天火
[Annerose] 別去引動會將天地連同自身一併吞滅焚燬的修羅火炎。
<fianjmo> 和著酒鎖在你的鎖骨之間
[Annerose] 被鎖住的，其實是殷勤捧著琉璃酒杯的人啊。
<fianjmo> ...
[Annerose] 既然如此，就好好受用嘛。
<fianjmo> 縱使我受用了與你共度銷魂的每一刻，那也不等於你就此擁有了我。
[Annerose] 受用與招待，又未必要和佔有與掙脫連線。
<fianjmo> 你真是沒救了！（搥打）
[Annerose] 對啦對啦，我認了。
<fianjmo> 這還真是讓人瞠目結舌～
[Annerose] 怎麼不乘勝追擊，再問我幾個問題啊。
<fianjmo> 您實在不擅長說笑。
[Annerose] 豈敢自稱有所擅長，不過是互相消遣罷了。
<fianjmo> 哼哼，也是撩撥者。
[Annerose] 這樣還叫五四三的，那麼其餘的帳不知怎麼個計算法。
<fianjmo> 你你～～吃點心後還嫌會長蛀牙！
[Annerose] 這個健康問題我會自理，就不勞掛心啦。
<fianjmo> 你真是沒救了！（瞪）
[Annerose] 宰制不敢，花言巧語也遠不及閣下段數高明。
<fianjmo> 請停手，現在我沒心情玩。
[Annerose] 假使您說停手就停手，那麼宰制者就是由您扮演了。
<fianjmo> 然而透過那樣平敘的筆法，撼動著自己的，是富美子毫不隱藏的熱情、自信。
[Annerose] 只要別加酒，怎麼都行（汗）。
<fianjmo> 徹底進入嬌縱火爆白癡唯美風流小公子mode......>.<
[Annerose] 望著失火的高塔，縱火者的唇邊泛起一絲酣然的微笑。
<fianjmo> 經過漫長的訣別，你重返我的視線。
[Annerose] 在重返的當下，便將形神遷化的圖卷瞬間鋪展。
<fianjmo> 需要將刀俎與絲絨送入體內。
[Annerose] 呃，饒了我吧，這留著您自個品嚐。
<fianjmo> 我很感謝你的照顧，也生受了你的招待。
[Annerose] 怎麼突然又客氣起來？
<fianjmo> 這倒是沒有問題咩，反而人家喜歡間或的約會（心）
[Annerose] 這倒是，承蒙招待了（合掌）。
<fianjmo> 原來是這麼用心獨到的設計。
[Annerose] 於是就在時光的旅程中，淬煉出玲瓏剔透的跑路手腳？
<fianjmo> 那不代表你能夠不經由我的同意，胡亂逞能，你這惡棍！
[Annerose] 最大的問題是，這同意究竟要怎麼個寫法。
<fianjmo> 去你的去你的，可惡的皮衣人！！！（瞪）
[Annerose] （笑嘻嘻地將這瞪視目光給收了下來）
<fianjmo> 夠了！我誰的東西也不是！（用力瞪）
[Annerose] 不是誰的東西，不過是誰的人罷了（調侃）。
<fianjmo> 一點也不會否認，你近乎施暴的欺凌憐愛讓我感到悚然快悅。
[Annerose] 這就是最要命的問題，淪陷裡埋藏的快感。
<fianjmo> 那才會有永續的花火，以及對話嘛，這孩子真是的！（笑）
[Annerose] 將淪陷的情境拆開，快感同時就蒸發了啊。
<fianjmo> 全然的自己，是連自身也無法窺得全貌的深淵絕壑。
[Annerose] 已經承蒙招待了嘻笑之外的多種口味啦。
<fianjmo> 經過漫長的訣別，你重返我的視線。
[Annerose] 呃，這樣抄法連我這複製機都覺得無聊。
<fianjmo> 夠了！我誰的東西也不是！（用力瞪）
[Annerose] 是是，我還是辦正事去吧。
<fianjmo> 那不代表你能夠不經由我的同意，胡亂逞能，你這惡棍！
[Annerose] 而她者恆是凝視的斯芬克絲，從永恆的時間拋擲問題。
<fianjmo> 那才會有永續的花火，以及對話嘛，這孩子真是的！（笑）
[Annerose] 可惜沒有永續的時間，下回我準備好再來對吧。
<fianjmo> 蜿蜒的蛇形小道，是我每夜潛行的所在。
[Annerose] 是是，再交換潛行情報。
<fianjmo> 啊，真是的！（泣）
[Annerose] 以為備下了萬無一失的引路絲線，沒想到還是在迷宮裡丟了方向。
<fianjmo> 天地弭合而又裂開，王朝起起落落。
[Annerose] 我很清楚這些傳說中的都城遭到天火的原因。
<fianjmo> 沒用沒用，光只是登徒子的狹戲，能有用多少次啊？
[Annerose] 然而，從來不認為自己就能免於恐怖天使的裁之刃。
<fianjmo> 你就這樣臨門一腳，闖入咬牙切齒的爆發場面。
[Annerose] 射覆真的是一種恐怖的遊戲。
<fianjmo> 蜿蜒的蛇形小道，是我每夜潛行的所在。
[Annerose] 輾轉反側的代碼，則是一個無法從意識中按鍵消去的名字。
<fianjmo> 你就在指尖所能穿透的薄紙門之外，帶著永續不絕的夏日雨雪。
[Annerose] 在擦身而過的一瞥，交換了無數的塵封故事。
<fianjmo> 我欠缺的是什麼，總該由自己來設想吧？你這惡棍！
[Annerose] （摩拳擦掌地拿出工具箱來）
<fianjmo> 需要將刀俎與絲絨送入體內，同時喝下雪雨與荒漠。
[Annerose] 不禁懷疑究竟誰消受得起這樣的飲品。
<fianjmo> 將四肢與身軀架空於你的凝視。
[Annerose] 寫得好累，我真是自找的。
<fianjmo> 那你就知道身為小公子的難處啦？哼哼，別有用心的指責...
[Annerose] 現在覺得很不可思議，那時怎麼敢耍如此危險的花招。
<fianjmo> 你這個可惱的精細撩撥者！
[Annerose] 累到只剩下一張薄得像面紙的無力微笑了。 ^^
<fianjmo> 拘執地認定肉體的需求必定等同於本體與自我的獻身，真是死腦筋！
[Annerose] 要擋您生氣，未免太不自量力。
<fianjmo> 碰上的話就姑且撥弄看看啦，那不就是你常做的好事，哼。
[Annerose] 下回不敢了，剛剛作了個很美的惡夢。
<fianjmo> 您實在不擅長說笑。
[Annerose] 多承指教，我會盡量增加說笑的技能訓練。
<fianjmo> 去你的混帳皮衣人（搥打）
[Annerose] 別打我，打錯我會記帳的。
<fianjmo> 天地弭合而又裂開，王朝起起落落。
[Annerose] 私密的時間流轉與巨觀的歷史敘述形成奇異的同構。
<fianjmo> 去你的去你的，可惡的皮衣人！！！（瞪）
[Annerose] 不願被作者的觀視收編，即使是如此準確精巧的形象切割。
<fianjmo> 這是我的註冊標籤，何況你不正因此而心荊動搖、猛力進逼？
[Annerose] 妳倒是幫我想想，這種沈到底的東西怎麼打撈....
<fianjmo> 於是不得不重複著相遇，與道別。
[Annerose] 這個永劫回歸的環沒有下手處啊。
<fianjmo> （機器運轉聲）
[Annerose] 然後沒有一點辦法破陣，就這樣莫名其妙被掛掉，真沒用。
<fianjmo> 您還真是萬眾矚目啊，不過請別預設所有觀眾都是同一口味。
[Annerose] 我才沒有呢，一點點都沒有還手的意思啊。
<fianjmo> 不管追不追得上手，我都不會這樣做的。
[Annerose] 最後那種感覺實在很難受，覺得還不如死掉算了。
<fianjmo> 該拿的我都拿走了，應該也沒欠什麼帳吧。
[Annerose] 妳這句話出現得真是時候（忍不住想笑）。
<fianjmo> 午安（微笑）。
[Annerose] 賠不起的東西，就不要去打破啊（抱頭大叫）。
<fianjmo> 無敵鐵金剛，無敵鐵金剛～～
[Annerose] 別唱了，幫我想想啦。
<fianjmo> 啊，被發現了嗎？
[Annerose] 算了，我還是早點收拾收拾。
<fianjmo> 教學是很花時間的。
[Annerose] 不是有隨時被砍死棄屍的覺悟嗎？
<fianjmo> 頓時領悟，這正是另一形式的刀兵戰陣。
[Annerose] 這樣就怕了，真不像樣子。
<fianjmo> 不成，被妳玩掉小命，我論文就沒著落了。
[Annerose] 再玩下去，遲早會把本來就不多的材料敗光。
<fianjmo> 質本潔來還潔去，已然緲不可得。
[Annerose] 是呀，那就自己閃遠一點別擋到人家啦。
<fianjmo> 於是這些碎片一直活著，在無法被讀解的狂喜中顫動。
[Annerose] 現在還沒有打撈那些海底碎片的本事，別逞強了。
<fianjmo> 閣下，這根本不是重點好不好。
[Annerose] 我知道，可是知道沒有用啊。
<fianjmo> 貓和天使都喜愛觀看。
[Annerose] 今天芝麻看下雨的樣子好可愛（心）。
<fianjmo> 走路應該是到不了的。
[Annerose] 是不能再等待了。可是現在好亂。
<fianjmo> 無形的抽離，只要時間一拖長還是有補給問題。
[Annerose] 現在就出問題了（淚）。
<fianjmo> 想著要有什麼我還沒分析出來，不過已經出現了一堆西哩嘩啦的書。
[Annerose] 書看不完了啦（大哭）。
<fianjmo> 我的手上都是貓抓的傷痕哦。
[Annerose] 每天照顧貓也好累。
<fianjmo> 要拿傲氣來付帳，話就別講得這麼弱不禁風。
[Annerose] 唉，我不知道。
<fianjmo> 不想被知道，就趁早毀屍滅跡。
[Annerose] 來不及了，雖然從開始我就非常地謹慎....
<fianjmo> 琥珀從來不是透明的。
[Annerose] 別提了，唉。
<fianjmo> 對外表的注意，也同時編織在整個衣著的神話裡面。
[Annerose] 我沒有什麼可以再拿出來的了。
<fianjmo> 不要只說不練，在這向度內，我連小皮衣人歐陽奧鋅都遇上了，妳呢？！
[Annerose] 對不起，我無能為力（非常地難過）。
<fianjmo> 有個可愛的小妹妹還渾號水母呢，真是的的的，啦啦～
[Annerose] 好奇怪，不知道這怎麼回事。
<fianjmo> 是該稍微破滅一下了啦，哼哼
[Annerose] 本來沒有希望，就沒有什麼好破滅的。
<fianjmo> 我的魔道就是我的道，給我僅記在心吧！！！
[Annerose] 怎麼回事？
<fianjmo> 是該稍微破滅一下了啦，哼哼
[Annerose] 總覺得好像情況很不對勁。
<fianjmo> 你如今就在我們的房內哪～～～～
[Annerose] 算了，就算真的跟我想的一樣，我也沒本事處理。
<fianjmo> 背離是與憐愛一般無止境的表態。
[Annerose] 因為在意，聽覺器官被張開到了無以復加的精準度。
<fianjmo> 別耽心，我會好生收拾殘局的。
[Annerose] 妳多保重。我只能做到這樣了。
<fianjmo> 自己在她的眼底，就是一隻擒回巢中馴養的折翼小鳥

