package Prty::ClassConfig;

use strict;
use warnings;

our $VERSION = 1.097;

# -----------------------------------------------------------------------------

=encoding utf8

=head1 NAME

Prty::ClassConfig - Definiere Information auf Klassenebene

=head1 SYNOPSIS

Klasse einbinden:

    use base qw/... Prty::ClassConfig/;

Information definieren (Anwendungsbeispiel):

    package Model::Object;
    
    __PACKAGE__->def(
        table=>'Object',
        prefix=>'Obj',
        columns=>[
            id=>{
                domain=>'integer',
                primaryKey=>1,
                notNull=>1,
                description=>'Primärschlüssel',
            },
            ...
        ],
        ...
    );

Information abfragen:

    my $table = Model::Object->defGet('table');
    =>
    Object

=head1 DESCRIPTION

Die Klasse ermöglicht, Information in Klassen zu hinterlegen und
abzufragen. Anstatt hierfür Klassenvariablen mit C<our> zu
definieren, verwaltet die Klasse sämliche Information zu einer
Klasse in einem einzigen Hash. Die Methoden der Klasse verwalten
(erzeugen, lesen) diesen Hash.

=head1 METHODS

=head2 Information definieren

=head3 def() - Definiere Klassen-Information

=head4 Synopsis

    $class->def(@keyVal);

=head4 Description

Hinterlege die Information @keyVal in der Klasse $class.

=head4 Arguments

=over 4

=item @keyVal

Liste von Schlüssel/Wert-Paaren. Der Schlüssel ist eine
Zeichenkette, der Wert ein beliebiger Skalar (Zeichenkette oder
Array- oder Hash- Referenz etc.)

=back

=head4 Returns

nichts

=cut

# -----------------------------------------------------------------------------

sub def {
    my $class = shift;
    # @_: @keyVal

    no strict 'refs';
    my $ref = *{"$class\::ClassConfig"}{HASH};

    while (@_) {
        my $key = shift;
        $ref->{$key} = shift;
    }

    return;
}

# -----------------------------------------------------------------------------

=head2 Information abfragen

=head3 defGet() - Frage Klassen-Information ab

=head4 Synopsis

    @vals = $this->defGet(@keys);
    $val = $this->defGet($key);

=head4 Description

Liefere die Werte zu den Schlüsseln @keys. Im Skalarkontext
liefere den Wert des ersten Schlüssels.

=cut

# -----------------------------------------------------------------------------

sub defGet {
    my $class = shift;
    # @_: @keys

    no strict 'refs';
    my $ref = *{"$class\::ClassConfig"}{HASH};

    if (wantarray) {
        my @arr;
        while (@_) {
            my $key = shift;
            push @arr,$ref->{$key};
        }
        return @arr;
    }

    return $ref->{$_[0]};
}

# -----------------------------------------------------------------------------

=head3 defSearch() - Suche Klassen-Information in Vererbungshierarchie

=head4 Synopsis

    $val = $class->defSearch($key);

=head4 Description

Suche "von unten nach oben" in der Vererbungshierarchie, beginnend
mit Klasse $class, die Information $key. Die erste Klasse, die
die Informatinon besitzt, liefert den Wert.

=cut

# -----------------------------------------------------------------------------

sub defSearch {
    my ($class,$key) = @_;

    no strict 'refs';
    for my $class ($class,$class->baseClassesISA) {
        my $ref = *{"$class\::ClassConfig"}{HASH};
        if ($ref && exists $ref->{$key}) {
            return $ref->{$key};
        }
    }

    return undef;
}

# -----------------------------------------------------------------------------

=head1 VERSION

1.097

=head1 AUTHOR

Frank Seitz, L<http://fseitz.de/>

=head1 COPYRIGHT

Copyright (C) 2016 Frank Seitz

=head1 LICENSE

This code is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=cut

# -----------------------------------------------------------------------------

1;

# eof
