package Prty::Pod;
use base qw/Prty::Hash/;

use strict;
use warnings;

our $VERSION = 1.07;

# -----------------------------------------------------------------------------

=encoding utf8

=head1 NAME

Prty::Pod - POD-Generator

=head1 BASE CLASS

L<Prty::Hash>

=head1 DESCRIPTION

Ein Objekt der Klasse repräsentiert einen POD-Generator. Die
Methoden der Klasse erzeugen die Konstrukte, aus denen ein
POD-Dokument aufgebaut ist.

=head1 ATTRIBUTES

=over 4

=item indentation => $n (Default: 4)

Einrücktiefe bei Code-Abschnitten und Listen.

=back

=head1 METHODS

=head2 Konstruktor

=head3 new() - Konstruktor

=head4 Synopsis

    $pg = $class->new(@keyVal);

=head4 Description

Instanziiere einen POD-Generator und liefere eine Referenz auf
dieses Objekt zurück.

=head4 Example

Generiere POD mit Einrückung 2:

    $pg = Prty::Pod->new(
        indentation=>2,
    );

=cut

# -----------------------------------------------------------------------------

sub new {
    my $class = shift;
    # @_: @keyVal

    my $self = $class->SUPER::new(
        indentation=>4,
    );
    $self->set(@_);

    return $self;
}

# -----------------------------------------------------------------------------

=head2 Abschnitt-Kommandos

Anmerkung: Die Methoden ergänzen den generierten POD-Code um eine
Leerzeile am Ende, so dass das nächste Konstrukt direkt angefügt
werden kann. Diese Leezeile ist in den Beispielen nicht
wiedergegeben.

=head3 encoding() - Deklaration des Encodings

=head4 Synopsis

    $pod = $pg->encoding($encoding);

=head4 Description

Erzeuge eine Deklaration des Encodings $encoding und liefere
den resultierenden POD-Code zurück.

=head4 Example

    $pg->encoding('utf-8');

erzeugt

    =encoding utf-8

=cut

# -----------------------------------------------------------------------------

sub encoding {
    my ($self,$encoding) = @_;
    return "=encoding $encoding\n\n";
}

# -----------------------------------------------------------------------------

=head3 section() - Abschnitt

=head4 Synopsis

    $pod = $pg->section($level,$title,$body);

=head4 Description

Erzeuge einen Abschnitt der Tiefe $level mit dem Titel $title und
dem Abschnitts-Körper $body und liefere den resultierenden
POD-Code zurück.

=head4 Example

    $pg->section(1,'Test',"Dies ist\nein Test.");

erzeugt

    =head1 Test
    
    Dies ist
    ein Test.

=cut

# -----------------------------------------------------------------------------

sub section {
    my ($self,$level,$title,$body) = @_;
    $body =~ s/\s+$//;
    return "=head$level $title\n\n$body\n\n";
}

# -----------------------------------------------------------------------------

=head3 code() - Code-Abschnitt

=head4 Synopsis

    $pod = $pg->code($text);

=head4 Description

Erzeuge einen Code-Abschnitt mit Text $text und liefere den
resultierenden POD-Code zurück.

=head4 Example

    $pg->code("sub f {\n    return 1;\n}");

erzeugt

    $n Leerzeichen
    ----
        sub f {
            return 1;
        }

Der Code ist um $n Leerzeichen (den Wert des Objekt-Attributs
"indentation") eingerückt.

=cut

# -----------------------------------------------------------------------------

sub code {
    my ($self,$text) = @_;

    $text =~ s/^\n+//;
    $text =~ s/\s+$//;

    my $indent = ' ' x $self->indentation;
    $text =~ s/^/$indent/mg;

    return "$text\n\n";
}

# -----------------------------------------------------------------------------

=head3 bulletList() - Punkte-Liste

=head4 Synopsis

    $pod = $pg->bulletList(\@items);

=head4 Description

Erzeuge eine Punkte-Liste mit den Elementen @items und liefere
den resultierenden POD-Code zurück.

=head4 Example

    $pg->bulletList(['Eins','Zwei']);

erzeugt

    =over 4
    
    =item *
    
    Eins
    
    =item *
    
    Zwei
    
    =back

=cut

# -----------------------------------------------------------------------------

sub bulletList {
    my ($self,$itemA) = @_;

    my $indent = $self->indentation;

    my $pod = '';
    for (@$itemA) {
        my $item = $_;
        $item =~ s/^\n+//;
        $item =~ s/\s+$//;
        $pod .= "=item *\n\n$item\n\n";
    }
    if ($pod) {
        $pod = "=over $indent\n\n$pod=back\n\n";
    }

    return $pod;
}

# -----------------------------------------------------------------------------

=head3 orderedList() - Aufzählungs-Liste

=head4 Synopsis

    $pod = $pg->orderedList(\@items);

=head4 Description

Erzeuge eine Aufzählungs-Liste mit den Elementen @items und liefere
den resultierenden POD-Code zurück.

=head4 Example

    $pg->orderedList(['Eins','Zwei']);

erzeugt

    =over 4
    
    =item 1.
    
    Eins
    
    =item 2.
    
    Zwei
    
    =back

=cut

# -----------------------------------------------------------------------------

sub orderedList {
    my ($self,$itemA) = @_;

    my $indent = $self->indentation;

    my $pod = '';
    my $i = 1;
    for (@$itemA) {
        my $item = $_;
        $item =~ s/^\n+//;
        $item =~ s/\s+$//;
        $pod .= sprintf "=item %d.\n\n$item\n\n",$i++;
    }
    if ($pod) {
        $pod = "=over $indent\n\n$pod=back\n\n";
    }

    return $pod;
}

# -----------------------------------------------------------------------------

=head3 definitionList() - Definitions-Liste

=head4 Synopsis

    $pod = $pg->definitionList(\@items);

=head4 Description

Erzeuge eine Definitions-Liste mit den Elementen @items und liefere
den resultierenden POD-Code zurück.

=head4 Example

Die Aufrufe

    $pg->definitionList([A=>'Eins',B=>'Zwei']);

oder

    $pg->definitionList([['A','Eins'],['B','Zwei']]);

erzeugt

    =over 4
    
    =item A
    
    Eins
    
    =item B
    
    Zwei
    
    =back

=cut

# -----------------------------------------------------------------------------

sub definitionList {
    my ($self,$itemA) = @_;

    my $step = 2;
    if (ref $itemA->[0]) {
        $step = 1; # zweielementige Listen
    }

    my $pod = '';
    for (my $i = 0; $i < @$itemA; $i += $step) {
        my ($key,$val) = $step == 1? @{$itemA->[$i]}: @$itemA[$i,$i+1];
        $val =~ s/^\n+//;
        $val =~ s/\s+$//;
        $pod .= "=item $key\n\n$val\n\n";
    }
    if ($pod) {
        my $indent = $self->indentation;
        $pod = "=over $indent\n\n$pod=back\n\n";
    }

    return $pod;
}

# -----------------------------------------------------------------------------

=head3 cut() - Beende POD-Abschnitt

=head4 Synopsis

    $pod = $pg->cut;

=head4 Description

Erzeuge die Markierung, die einen POD-Abschnitt beendet, wenn
danach Perl-Code folgen soll.

=head4 Example

    $pg->cut;

erzeugt

    =cut

=cut

# -----------------------------------------------------------------------------

sub cut {
    my ($self,$encoding) = @_;
    return "=cut\n\n";
}

# -----------------------------------------------------------------------------

=head2 Format-Codes

=head3 fmt() - Liefere Formatting Code (a.k.a. "Inline-Segment")

=head4 Synopsis

    $str = $this->fmt($type,$text);

=head4 Description

Erzeuge Inline-Segment vom Typ $type (B, I, C usw.)
und liefere den resultierenden POD-Code dieses zurück.

Die Methode sorgt dafür, dass das Segment korrekt generiert wird,
wenn in $text die Zeichen '<' oder '>' vorkommen.

=head4 Examples

Nomal:

    $pg->fmt('C','$x');
    =>
    C<$x>

Eingebettet >:

    $pg->fmt('C','$class->new()');
    =>
    C<< $class->new() >>

2x Eingebettet >:

    $pg->fmt('C','$x >> $y');
    =>
    C<<< $x >> $y >>>

=cut

# -----------------------------------------------------------------------------

sub fmt {
    my ($this,$type,$text) = @_;

    my $maxL = 0;
    while ($text =~ /(>+|<+)/g) {
        my $l = length($1);
        if ($l > $maxL) {
            $maxL = $l;
        }
    }
    if ($maxL or $text =~ /^</ or $text =~ />$/) {
        $text = " $text ";
    }
    $maxL++;

    return sprintf '%s%s%s%s',$type,'<'x$maxL,$text,'>'x$maxL;
}

# -----------------------------------------------------------------------------

=head1 VERSION

1.07

=head1 AUTHOR

Frank Seitz, L<http://fseitz.de/>

=head1 COPYRIGHT

Copyright (C) 2016 Frank Seitz

=head1 LICENSE

This code is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=cut

# -----------------------------------------------------------------------------

1;

# eof
