#!/usr/bin/perl

use strict;
use warnings;
{
    use Carp;
    use XUL::Gui;
    use Roku::DVP::Remote;
    use Env qw( %ENV );
}
{
    my $Ip = $ENV{ROKU_DEV_TARGET};

    if ( not $Ip ) {

        my $ip_ra = Roku::DVP::Remote::sniff();

        if ( @{ $ip_ra || [] } ) {

            if ( @{$ip_ra} == 1 ) {

                $Ip = $ip_ra->[0];
            }
            else {

                # FIXME Prompt user to choose ...
                $Ip = $ip_ra->[0];
            }
        }
    }

    my $Roku = Roku::DVP::Remote->new($Ip);

    die "couldn't connect to a Roku"
        if not $Roku;

    print "connected at: $Ip\n";

    sub _sources {
        my ($image) = @_;
        my $id    = $image->getAttribute('id');
        my $class = $image->getAttribute('class');
        my $src_on  = sprintf '%s_ON.gif', $id;
        my $src_off = sprintf '%s.gif', $id;
        return ( $src_on, $src_off, $id );
    }

    sub _remote_action {
        my ($self) = @_;
        my $id = $self->getAttribute('id');
        my $response = $Roku->keypress($id);
        my ($ok) = $response =~ m{ \s ( \w+ ) \r?\n }xms;
        print "clicked: $id => $ok\n";
        return;
    }

    sub _get_image {
        my ($name) = @_;

        my $empty_quare_gif = <<'END_GIF';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END_GIF

        my $back_gif = <<'END_GIF';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==
END_GIF

        my $background_gif = <<'END_GIF';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_GIF

        my $back_on_gif = <<'END_GIF';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END_GIF

        my $blank_horizontal_gif = <<'END_GIF';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=
END_GIF

        my $blank_square_gif = <<'END_GIF';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END_GIF

        my $blank_vertical_gif = <<'END_GIF';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END_GIF

        my $down_gif = <<'END_GIF';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_GIF

        my $down_on_gif = <<'END_GIF';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END_GIF

        my $empty_square_gif = <<'END_GIF';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END_GIF

        my $fwd_gif = <<'END_GIF';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END_GIF

        my $fwd_on_gif = <<'END_GIF';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END_GIF

        my $home_gif = <<'END_GIF';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END_GIF

        my $home_on_gif = <<'END_GIF';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END_GIF

        my $info_gif = <<'END_GIF';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END_GIF

        my $info_on_gif = <<'END_GIF';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=
END_GIF

        my $instantreplay_gif = <<'END_GIF';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_GIF

        my $instantreplay_on_gif = <<'END_GIF';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END_GIF

        my $left_gif = <<'END_GIF';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END_GIF

        my $left_on_gif = <<'END_GIF';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END_GIF

        my $play_gif = <<'END_GIF';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END_GIF

        my $play_on_gif = <<'END_GIF';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END_GIF

        my $rev_gif = <<'END_GIF';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END_GIF

        my $rev_on_gif = <<'END_GIF';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END_GIF

        my $right_gif = <<'END_GIF';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==
END_GIF

        my $right_on_gif = <<'END_GIF';
R0lGODlhNQBWAOenAAAAAAICAwQFBgYHBwgICQcJCgwMDAwODw0PEA8RERERERASExIUFRQUFBMV
FhMWFxUWFhUXGBcXFxcXGBYYGRcYGRgZGRkZGRcaGhkaGRoaGhkbGxobGxsbGxocHBscHRwcHBsd
HRsdHhwdHR0dHR0eHR0eHx4eHh4eHx8eHh8eHx0fHx0fIB4fHx8fHx8fICAfHyAfICEfHx8gHx8g
IB8gISAgHyAgIB8hISAhISEhISAiIiEiISEiIiIiIiIiIyEjIyIjIyMjIyIkIyIkJCMkJCQkJCQk
JiUlJSUlJiQmJSUmJiUnJycnJycoJygoKCcpKikpKSgqKSkqKSkqKioqKioqKysqKiorKysrKyws
LC0tLS4uLiwvLi8vLzAvLy8wMDAwMDExMS8yNDIzMzMzMzI0NDQ0NDY2NjY3Nzc3Nzg4ODo6Ojs7
Ozw7PD09PUBAQENDREZGRkdHR0pLS1FRUVJSUlVVVVlZWVxcXF5eXmJiYmVlZmdnZ25tbW9vb3Jz
dEKPOkKQO0OQO0SRPHt7fEWSPkuRREaUP0eVQH+AgFaSUEqXQkuYQ0uZQ0yZRIODg02aRWKSXk6b
RmaSYk+cR4WGhlCdSGuSaFGeSYaIiYWJhVOgS4GNgFShTFWiTVWjTVWjTlajTlakT1ekT1elT1il
UP//////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////ywAAAAANQBWAAAI/gC5aMni
xAsYKlGwYMlihcrCKlmiMKmCpUrBgwkXWqkysAoVKgKzZPGokKHDkSJHVqmyRUuUKFWUfFggAECA
mzhz6tx502YAAxF0UOSyZWMWLVuyUNHipQoXL1q6dBgAQIABBhhEaP0gwgNXElo9mBDBwYNWDhtE
dKiwoEDNARF2aNGicC5HKimdMkEQgEAEKmTMkPnipbAXLk/DGD6MuDDiLoiT+lhAtYCHJSuzbElq
BalTIgQAMOiyRkwTHzpSq75xQ8eNGrBvuDiRosaNFC5cyHbhI0qWIQhsYjCyEIsWLlWaNMlCZEAA
E2vKICFBW0bu3NZlWL8+e/aJ77S9/lO/wYSJgwACLhS5cvLjSwMAWLjZ0l03a+0urN+QwXo/d+//
neACCSP4kIQDVXHwUBZSYMFFBABQwAYXuM3gWn/Z1TDDfvzJUEOAAuYG3nUn3NDDAQAccMNRW2BB
hRACDDCGF6+xtmF/NuZXg3a21fcddj6KONsNHggQQAVzHSUGAwHUkAYQJOAoJX+s5VcfgD9eCd6P
J9QQ3AE1bHaUW2JsIdt2trV2wwwz2LYjf28CmNtu/HUnYJYSoKeBb1nsEAADZiCRAn9sYqgfdvdR
SaKAKdRW5513zpYBVQ64mMUGAbhAhg8naHcfa6vNkF1+2lkHXgqyjRipjx0UAAAC/sZVgUEAQnCh
g26eSqkrazCwNqKcdoYorAfwGfAEchASoYUOtnWY65r9pYmosMJyBymXHcBHAEdVQAAAEFr0sCGV
+6k5gwtpWkeDbtZW+6uqJ4iA4gBa4IWgEFrsMG6uLqS2Jmw12KBbDLa94MK6crrLZbzziqQFAwAU
kQWzGLogan9s5kbDDTbA8MJuVWapMKQezDtFRRATMbF9O7YZ22ts9tofDDDgtmjCIXKXLQADLJQF
xENoIdtr6aL7b4fl7pYzltT+58IH8xr0MwBCbOEDfxe+6aELLcymAoeperewkNSCVzLPT02N79XR
3mgqpLnBwPSiqoq45dkDcHRF/gJUa8H2ff4GK6TgVw64pdh37kwASFgkQOuyFjfbX7DwIr7w4WJ7
Jy/PLD0MwBFb6GDCjXRmaTjmA85NYubfbd7zXAgSEbp9rNkGA9yH5zy2lqyfgLdDWyxAaxbllirD
7XBzZ7Pp1I1t+nU75y0S30JM/OnxyQvYa79zXlt54blBzTMWCS3wrdC48teC94ze4IMQsqWQvOGC
b+n7vBVl0cDwQzfL/JZFqEEFIEACHXDJctkz27xWUgWIVW9o55KB6Q7nAyzowQ9lwIABD2i3aq3g
R+KjF0XM5wO/oasGrrnZf/hwCVEcAg4RoEHu7PcjFICwYVuoQrKWJbnjzQlR/nOyhClMUQpE1MEC
/drd6UQWQi6IxFvgElfSbuOCRnVNQPCTAyJKMcRH9EEHLODBAWlopw7MKwp0uVe+9qWoufngAm8Q
hCiGyIlCFAEDN/heB12ANzROTWIUu4/Fbtc7HUhADJLwxBA/0QkteIBTImOXnfCGFCykbGW1k0Gb
1mc3ABnSB4rIxBBNcYg4fAAJefwRqnwUQix0oYEACFr/0jXGagkBBBr4QyVGmQg6ZACVtfTOCOal
hZXskGLOkqDqTAeCOzRijqZwhB6MgIRlnmCY4zMmAJSFTA6ZKmFM0wEE2jAILpoiEnbgQA/I2LoF
UgSK4RpXxdg3NwlAQBOh/hiiJ/ZwAQRqzp1VUKO+vOkr9sErBErohChNQQpMnAF+7Lwfz45iyYhN
LE3+saZ3bqABNSxiFPqERBM26DTowYdeR7nkrTJJuKWhhgJxMAQ0LwEIHZAAkqviTggzAzT0ES1s
9ruODj5AgTw0YpSMwAMLXnArzNkPbxCBZdXYdqH/gScIInABIHY5RELMwQIyvBbdvgNVLVjBgX4j
Vw0mOCIdYGETnBiiKCSBBgnkYH4kYsENOUeFJ1AvrRhS2vwAEVdTgKITUjFCS+2kVwHtVApVcBy+
btWmcnUySzeAxBAnAYgaZOAHv1ri804QQiug7HOhG93kdicgHMSBEoHQ/gMJNCCE+aEOW/MybRZi
NzscfYieNwBCF8qgptEurWx8nNcWXim8ByZTgjlr3mx0MIIM4HRpdBIstrRVF+pZr3jQHSuWMted
Ru1Iu//k2UeqYD5wDc1TNZAbsLynJeSKtZ3j+8j+HqibZrlAvrshpBJ/mJ9UXc47eLPCRhxIPN1E
MDe4Ma/8LLc689aJvDo76Uu0QEITwuZCnkLcdUigQvKi7jo7LSZa/+afRO2HrQR+VIA6SFbl5vCv
bLtRujpkMHBaGKjVshzeDhPZx1HWxYYCMBAxnDsRNfZpxPQIxEAnuos9K6MQdlTYcGbiJzdRJLxd
adKM1ywYvI2DkfTR/nFDOJDg8c9ZOLLOeakUg42JV2Srwi3nWuLdC60pO6XiFc2uA4OutTSoe8Rb
FojSXp8mE0ODdtqEfdQ83ZGNkiLZb4Mr5uILy49RM1YzokkbtYEwGIK16/S6bktjEDENmz1zSgT4
R7srH0++Y30XsHAGaydwBAK0Bi+OYBADXOO5cDOkMQfOyIQtMEll0BpzqWhWM9wcm8uX21L06sUF
JvUACyu1tW7ky85OBmmPs+EAfAqgECwAmwWY/DOptEPtRoVWxNc+XAo0EBoD0KUKHghABLDgA1Q9
WtA1G5zCaIy775DAAlRBAANZ8BMmFGFOOyrwfnplbBUKq9K7dsEI/oQHgAUMJAtIKMBzlPChpGlc
NnKz2XxHTGLwzUkDrhLABDbzESYdAAg+0JBljfYxcufbtiR2+HcYUBMDbKAjVKgBVSIABB3kyjqi
Gje6jz3BpJ+AOhYITcmpwJEsKARBAqiAa8i1JnbF/LhNM3GWMAAfABgAAx6BiEKMgCIBPKBfcfbU
29plbhreCQMqr0oEotqSpGihBq4KQAI8YB9/4aikmOeObUTAAFfZhAElKOZRyp6FJ3wgNAEogAIs
UAIWwGY7AvqQ5nPDAr1G2AQsGIEGHHCAmlRlASfQTEuMY/aRMEFeAQAAzwyQAAUs4PnQj770pa+A
BCTgAIlXPgEekgACinSBLgs5uYu0QAQIECD5yld+8nuyfpu4v/3oR7/6B4AACxylIRQZfZuJbxwW
MMAAA2AkPDGABCgABZAAFBAFUmB2UAASSeEUKXEsXHAsGFESDbEgEkERDAQTUeURR1EvWMAeDgEF
SCESDNSBdTEXHOggRQERSFEvnvEUxXRyZLcSLwgRW0AFUDASM1hMnRMQADs=
END_GIF

        my $search_gif = <<'END_GIF';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END_GIF

        my $search_on_gif = <<'END_GIF';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END_GIF

        my $select_gif = <<'END_GIF';
R0lGODlhVgBWAOf/AAABAAADBgAEBwAFCQAHCgIIDAUIBAcJBQMKDQULDggLBwYMDwoMCAsNCgcO
EAkPEQwPCwoQEg4QDAwREw8RDQ0SFBASDw4TFRETEA8UFhIUERMUEhAVFxEVFxQVExIWGBUWFBMX
GRUXFBQYGhYYFRgXGhkXGhcYFhUZGhYZGxoYGxgZFxYaHBkaGB0ZGBcbHRkbGBsaHRwaHhgcHhoc
GRkcHh0bHhscGh8bGhkdHx4cHxwdGxweHCAdHB8dICEdHBsfIR0fHB4fHSAeIiIeHRwgIh0gIh8g
HiMfHiAhHyAiICEjICIkISUjJiMkIiQlIyQmJCclKCklJCUnJCcmKSYoJScoJiUpKygpJykqKCcr
LSkrKC4qKSosKSstKiwtKy0uLC4vLS4wLTMvLi8xLi4yNDAyLzEzMDIzMTA0NjM0MjI2ODQ2MzU3
NDY4NTc5Njk6ODo7OT07Pjs9Ojs/QT0/PD5APT9APj5CRUFDQUNFQkdJRkhJR0lKSEpMSU1LT0pO
UE5QTU9RTlBST1JUUVJXWVZYVVlbWFxeW11fXGFjYF9kZmBlZ2NlYmRlY2doZmRpa2psaWhtb2pu
cW9xbm1xdHByb29zdnN1cnR5fHd5dnh6d3d8fnt9en1/fH6AfYKEgYOFgoSGg4eJhoiKh4qMiYyO
i4qPko2PjI6SlZGTkJKXmZWXlJOYmpSZm5qbmJqfoZugop6gnZyho6GjoKGmqKSmo6WnpKaopaSp
rKarrqyqrqesr6musayuq6uwsqyxs7CyrrK0sbO1srG2uLS2s7O4u7e5tri6t7i+wLy+u77BvbzB
w8HDv7/ExsHGycjFysXHxMbIxcnHy8fJxsrIzMjKx8nLyMfNz8nO0M7QzNLU0dDV2NPV0tTW09Xa
3Nnb19rc2dvd2tze297g3d3j5d/k5uDl6OHm6eTm4+Xn5OXq7ejq5+rs6evu6u3v6+7w7e/x7vDy
7/Hz8PL08fP18vT38/b49Pf59vj69/n7+P/6+fr8+fv9+vz/+/7//CwAAAAAVgBWAAAI/gDDfPFi
JQyZLFm6dPHSJaGXLA+rJMxS8CDCLBIbVrwYceLGhAsbDrwo0AvDLCMRlnwI8uRCiCyzgPlycUqN
CwIACNjJs6fPn0CDCh2acyeDDkmsJAwDpqGXL2BQhskSRmAYGgd0Hk1Ro2vXG113fIXRlaxXr2Br
GKlxI61ZGl3Xpq2xYy2NtDTI0khxIUHOAx2WbPnyRSFhhCYfgpkSQUCCDlnQqEEzpmpVMGAMWr7M
lGnmzZs/Y8YMGvToy5kxF15yIWuCG1OeIMbcBapMJwkAdAjzxkyVJUmCCz9yJMkRI8iFIzdyPPhy
5saRHD8iREhx68yJTxcS5AfzH9Wp/gtZgqWLkwg6Uzgpz5CmFSvmDwgw8gbNkx3dkVSvrh+J/v0A
BhggeAISGESAB3LHXRAMdrcgfkdMMUUHO63gBHwTIYQFAwAYIccXClpHnH9C6HeEdCOieKKJK4Z3
ooAwLigggwgKscMOS0TRgU40TPFSQmDsmAIcYIBXnHYvIpGdfycyF550/AHInIn7HTgidQkmCCCN
1TVY5RFLPADAA0c48VRDTAhwQBlhTIfkmyUa4Z+TcrpInYolhrffiHoimGWC+mXZJXdH3LCTeoR9
YQaFRrixxA5vIolnoAGiKGKNxOl5IJSDVnkgl1o2SGMQRqD3gBFMzOSFX2Z8cWd1/tkZRxxzckpX
q4pXGpEnlP+1qCCVMgZrHX9YKsglCTvBkARESwjQgRpP/IBrisOWmKJ2KHJaaZ4K8rlnjDT+4B2U
n37KXQtZdWDEQjAIIAQaSwTB5InEDZdkif71h6+Sme55J3XF3inqlgz+gOWoIWZJQ24RrOtFCgKk
moR180YaqYjSVcytiyECCOyCov5ZZbBZ3sAhAzVgkcWOZiYx5YrTztqvpfhmSuKlCJqoZcIid2ys
l0HQwGECMCAEAgBLeLHElVfKKsSST+NML5WCdgr0yD2DCnKXowZRg5gHwEDTjkwonaKlwc2KHMD8
1nyz1gPD7aW2upI8KNBeg52C/kksd+EythTD+TSfx9UMK9edRm03d//5u3i55d6gN0p9TzyrrWu7
KaLNPl+qNcefOyisjKDuXJ3QAByQQkMsewFw5vplJ+fUmZ5oJJeID+wpyT3n3vt+X6eeQhhekP3F
EtLJeiutI9Zau38+4E66l7uXjvj0d3PdoOTCE0+20ipm9yK+w774b6AiW4+9sbtPfzXIDHKvOkQW
AFA28timTd2SsxMrHXjWw5vv3hc3LeGHeqSiUfDm1wULRMwLlnuZdp5DL/MJrID7OWDXMqg7DGoP
VEZQINhWkJAdNeELf9MOsQCHq4wdIWTCgiHuYLi1UJlOenlLHQxqwzIUbm46/i5s0ryEAMAfHOhG
8AuCBhm0xJCJ6kYdLBfPuEMDsBXtCzhhQhemli8VmchgRBTXp36AHygq8YBQNKMay1jGJ2owcTY8
kPxICIb6aVFWGcvWvIy4IDIyyAg3ulFbBBnIQA5yBzcoJCEFeYMzvhGBd5vjQlr3tzzGjGs3wgEh
NbnIQyYSkYm8AQ5COUhPfjKRTNwS/Kg4tB124WhJW1oF6eU0XXVHLKW8gQ1s0BZS9vKXbaGBMIEJ
zGH2MpBRLNcCd1g8+5kNZpnSH3KUGEy20CAGLkjBDYRpTWGSBQbghEEKxkkDG8AgL3nRyw20CZe7
DFJuypzcygDgBL+Jz0R4/rJRKL0pzhaogC8UCGgFPNCCFKhgBSlA6Dj92YELdKAFBR2nRFewgQ2w
8y4HvFvX5JeCwlRORLNbzv6400twpqAFNOhAAg5QgRTsBQIFKIAGxnkCiY6zAhBIwGMimoKadgAC
EJBAB2CwTRoE8noAQp3qHtK610GNTtQUJkRToNIOsOEQm0DFKk6xiUSkAQOPkSgJOoCBKjRiE3ro
QAdEQAISUDUBaagEJdjQgRSc86ijq84Ch9dMJhwveZeb09OoWQMYQFSlX8gEM9qBj8Y29h3M6EQY
EmCBFag1Ah3ohDzw0Yw6HIUEK7gAAdxgjHrIwxVOgIA7CSgqjpbQmfiT/pm3xCVIcXYgAn0gBj92
y9ve8mMZfniABcYpgDA4Y7fz8MQBLsCXBNhhGfrQRz24EYcEuPST5kqia71gR/Dlr5bcwQFbYNAB
BvDBG/34h3r/EY91yGO96gXHIAADsTVMQ73yEMVRGmOH+/ZDH/aQxRMogFFUglCEwiuMA8sWwVke
R1eAbAsMIHAGZvBjvebIhSYkwQlhqAO+0ojDAUBwADQ8Q73xAMUDxAQHaayXH8FgAwSui90DH2iB
K2AdAE6Ywpn16we97MAOToGPC/+jGX9gAE8sQIhrGPkfsEjBStVw4n/MAxTyuQM11nuPXkhBABsg
yzvb16Vljo2ePkTS/sv+8wO2qAADeQBHkf+xjTUI4Chq5dAevLHecfRhJ1RW7zw0AQA7XAO+tZiC
s8bJlrawlopWbEgWt2jJihkxCDrYgQdSgInN8kMekBDAcOuK0AcIQBPzUC8/sCyAQP8DHpcwQzXW
O49Z1CABIFioO/EDRxrNMSF29NuV5uUfAI4SA0xghae7YQX6ujUFbm11NtZ7i6OtocrwAEYz9qHe
fJQCBs4m5yFDR9IRIuQCSHMdxWzVxeoA+QYU6AIuNqsPXljgATalagoeQIJerLcYSQBAG6oMX/W+
gw8AqGwH3NqCRiMxiWXWW2E2ELEtYidJcjLQjSbwhV1sNh+0WPE4/i278BRE4AK1WC8ypgAAVxf8
H/1IxhcWPU4V7HoHvnvQCAlDNotzzlpHEBcgdzCBKdDC08fYNwgQqtZxMsAIxVhvMGoAgGsXvB5P
zkUNGDBRh2eUdzuYnBfQHcunGQGPKDIiGXdAgR2kwtPlcIOzENpWywrgDnxWLyzQgwYX99kSwFgv
PkBhAa4n1Jg4R+Cnqig8hFAoll7EmH6MmEgPYOARnvY2A6xrUzWZIh/qrUclAE3wcRhCAG74xnrf
wQgCTEDcBn4fpHWYEBBUHGATpJZ3EpkCCLThGnNmhyLkg1mcAEAR5ljvNewsABOH3hQ6FUAi4NFn
PBSgrkRFpSpD/hQ8omUBCxTSIu5XhCRxkbEtG6gAJrCuXnWQwgwXwGwZPvHh0GsiAY1xPns54azN
i0IfqqYMYXAADRdKODdFs5cAJDRP93M20TQxpOIVUhYFvsAP6fUP9KANuhALuoAN77VetrAEXNdq
0YBfoJAADvV0v2CB0tUKSUAB2fdwkLMge6Vj9dRj+JQpTBRkBwAGxeBbQLhbvDBzddVq0LBb8RAK
KEgCtqcG1BBd9ZAOhMB1h6RRkaM3k0RPfgNSmOMk3IFIe0FVBeAFqYAOjnWGjUUOo2AFBMBcK3AA
VzAMjcUOk/AYboUTgnAOpsUOiHAUVZhXQqBUKRAVTYUdTyUe/vGzF0xHNIWwCssADu8gD+4ADspw
CoFgAkehUBPQAYtgDeDwCmQwAW01ThOQAJJADd6gClikTXglPXqFhV9gPPijPPkCYUxUAz2lUDB1
AmoQCI4QCYsACGfQAQUwARBVUz1lUXPgB0+gASLQUxJVAR1AB33wBBZ1TAijXfL0PbHlJi/SHWsH
Fv8kUSDgADrlABRgjglAAXWVUCM3Thtwjh2gUBLVAn2RABJAAjFwTLz2J9uDhVnQXd2IBPpTLW02
SidVUCuAUnSXbw75kBBZjyhlSG/EHSF0YwC5YBB0cbjyHLdUA9x0UuIkTteVby1AklxhUyM5Tt9k
VyiJXUvU/iUXKQQ4Rjk7hkLiIx54onZB4AM7wEu9VE7CdBe7FJQ24E265E66tEvGJJTbZAOFtDM4
tEyOh2aWk3tB5B1ixEQU2RZsYUinBBafBEqKlEtRqURSlD6MFzZRMWlclC+AE3TcYX4H4gM+6ZNj
iUZsxEZc6ZNtxJVolF0d4yVzNBXBhkfkB03mA44JcpFcyUSQ+ZhoOZmTiUzUoz1WU26pswI0QXbq
Rn63ki2/QhwEcmnqs0qjo3gEQzoflCV7RTwUJ34cGScVIymFI0XVc5nY44+/o5pdE3abCRU9h3vc
okezwzZQIh5IFUNSGUCYyTNZo0STg0XptgRmh3Z8ki0V/mQrpVM1/mg13dmc7pM7a7k3QSIAkBeX
5JecgbMry9k1OJSW2Tgy0CmYQWNFA2F7sqlm16IkP9Q/iZOZNwSeM/RBMFI13QcD3nN7m1ObcNk4
scMflIKaBGo6rSk34NkljKeAXRAG+ulzeSQpeqKcHoM+8NRBvJM7BLNB8FOeVhCLAlBPtgkz/JIv
l8I8syOYENecLKo+G+REQcMhYSMTj7cFluOgTxNENQp0tnQ9PZo+MuSkvaYwrfQFHjofW8hCcLme
XHqgVbOa8BmfT2qgPwADucEANKAQhtIBW7AEBmNJ2EJsiUmiMDKhDaIt3qlRrek+O/CGABABO0AT
RnBn/lPgBOFRJ9o5PhOUqOPzQyvqLV8aOo9kOjtgfBcgBAjxBH5hBFOgKxUEdN5CLUGkg5uiQq6Y
MGRmQAdoodZhpjphAkIwEhTyAEuwBEsiHvf0Ol20nXTqbhYKpdEZQ4k3mRQCAAwAA0sQFV5gBOkC
HMQWOFxEK1vaJwDSqxmqmuvzhQ8SBCuQGwBwATWwBDSRBVuwIwKQAtEBJ8OmJMjRH/0SLM+KVL0Z
nd8pBCnAIcaaAkaAGCbhBGLiLEIgKw64pTkJqZKnkwf6nvQaN1JWFB1AHIihEMuaG6J2AyKiPxaT
PGjHIsURHHaiQv8SIyILK0dQAyoFADrRAUFwBBCB/hJgwBBWcGs6QVkrQCrtCivLoTj7QSvVGiCF
s7ODUjdAa3YO4h2ARF6mhrICcAErK64tOxAEYRNJmzoMYAEWcAFYm7UXsAFa27Ve+7VgG7Zea7X3
5hcoCwCPYbFloxoD8RIS4gQgYLZnqxN0WxR1e7c5gbd6q7d227d3S7dnqyYRsAIhxLIdWhhPYRIi
YQVVYATlJR9EEbmSO7lCQVn6WhxnghAvQRhNQRi18R6PQl5Za7UXcLVka7URcLqqawERkLqni7Vk
m7que7qu27qwm7V15QN/E7EqwxReQDwmgQVVIbwWIbEN8RId0RAXgRgccRgKsT+aGxIO0bIZsbzM
DosYzqu8XYAFUGESFxEQADs=
END_GIF

        my $select_on_gif = <<'END_GIF';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_GIF

        my $up_gif = <<'END_GIF';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END_GIF

        my $up_on_gif = <<'END_GIF';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END_GIF

        my %b64_for = (
            'background.gif'       => $background_gif,
            'blank-horizontal.gif' => $blank_horizontal_gif,
            'blank-square.gif'     => $blank_square_gif,
            'blank-vertical.gif'   => $blank_vertical_gif,
            'empty-square.gif'     => $empty_quare_gif,
            'empty-square.gif'     => $empty_square_gif,

            # buttons
            'Back.gif'             => $back_gif,
            'Back_ON.gif'          => $back_on_gif,
            'Down.gif'             => $down_gif,
            'Down_ON.gif'          => $down_on_gif,
            'Fwd.gif'              => $fwd_gif,
            'Fwd_ON.gif'           => $fwd_on_gif,
            'Home.gif'             => $home_gif,
            'Home_ON.gif'          => $home_on_gif,
            'Info.gif'             => $info_gif,
            'Info_ON.gif'          => $info_on_gif,
            'InstantReplay.gif'    => $instantreplay_gif,
            'InstantReplay_ON.gif' => $instantreplay_on_gif,
            'Left.gif'             => $left_gif,
            'Left_ON.gif'          => $left_on_gif,
            'Play.gif'             => $play_gif,
            'Play_ON.gif'          => $play_on_gif,
            'Rev.gif'              => $rev_gif,
            'Rev_ON.gif'           => $rev_on_gif,
            'Right.gif'            => $right_gif,
            'Right_ON.gif'         => $right_on_gif,
            'Search.gif'           => $search_gif,
            'Search_ON.gif'        => $search_on_gif,
            'Select.gif'           => $select_gif,
            'Select_ON.gif'        => $select_on_gif,
            'Up.gif'               => $up_gif,
            'Up_ON.gif'            => $up_on_gif,
        );

        croak "no image defined for '$name'"
            if not exists $b64_for{$name};

        return sprintf 'data:image/gif;base64,%s', $b64_for{$name};
    }

    sub _get_keycommand_js {
        my ($id) = @_;
        my $js = _get_mouseup_js($id);
        $js = <<"END_JS";
document.getElementById('$id').click();
$js
END_JS
        return $js;
    }

    sub _get_mouseup_js {
        my ($id) = @_;
        my $src_off = _get_image( sprintf '%s.gif', $id );
        my $src_on  = _get_image( sprintf '%s_ON.gif', $id );
        $src_off =~ s{\n}{\\n}xmsg;
        $src_on  =~ s{\n}{\\n}xmsg;
        my $js = <<"END_JS";
document.getElementById('$id').setAttribute( 'src', '$src_on' );
var reset = function() { document.getElementById('$id').setAttribute( 'src', '$src_off' ); };
window.setTimeout( reset, 300 );
return;
END_JS
        return $js;
    }
}

display(
    Window(
        title      => 'Roku Control',
        height     => 390,
        width      => 235,
        style      => 'padding:0px;',
        Keyset(
            Key(
                id        => 'back_key',
                keycode   => 'VK_BACK',
                oncommand => _get_keycommand_js('Back'),
            ),
            Key(
                id        => 'home_key',
                keycode   => 'VK_HOME',
                oncommand => _get_keycommand_js('Home'),
            ),
            Key(
                id        => 'up_key',
                keycode   => 'VK_UP',
                oncommand => _get_keycommand_js('Up'),
            ),
            Key(
                id        => 'right_key',
                keycode   => 'VK_RIGHT',
                oncommand => _get_keycommand_js('Right'),
            ),
            Key(
                id        => 'down_key',
                keycode   => 'VK_DOWN',
                oncommand => _get_keycommand_js('Down'),
            ),
            Key(
                id        => 'left_key',
                keycode   => 'VK_LEFT',
                oncommand => _get_keycommand_js('Left'),
            ),
            Key(
                id        => 'space_key',
                key       => ' ',
                oncommand => _get_keycommand_js('Play'),
            ),
            Key(
                id        => 'ff_key',
                key       => '<',
                modifiers => 'shift',
                oncommand => _get_keycommand_js('Rev'),
            ),
            Key(
                id        => 'search_key',
                key       => 'S',
                modifiers => 'control',
                oncommand => _get_keycommand_js('Search'),
            ),
            Key(
                id        => 'rw_key',
                key       => '>',
                modifiers => 'shift',
                oncommand => _get_keycommand_js('Fwd'),
            ),
            Key(
                id        => 'info_key',
                key       => '*',
                modifiers => 'shift',
                oncommand => _get_keycommand_js('Info'),
            ),
            Key(
                id        => 'reload_key',
                keycode   => 'VK_F5',
                oncommand => _get_keycommand_js('InstantReplay'),
            ),
            Key(
                id        => 'return_key',
                keycode   => 'VK_RETURN',
                oncommand => _get_keycommand_js('Select'),
            ),
        ),
        Box(
            orient => 'vertical',
            style  => ( sprintf "background:url('%s');", _get_image('background.gif') ),
            Grid(
                Columns(
                    Column( flex => 0 ),
                    Column( flex => 0 ),
                    Column( flex => 0 ),
                    Column( flex => 0 ),
                    Column( flex => 0 ),
                ),
                Rows(
                    Row(    # height 22
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:22px; height:22px;',
                        ),
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:85px; height:22px;',
                        ),
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:22px; height:22px;',
                        ),
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:85px; height:22px;',
                        ),
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:22px; height:22px;',
                        ),
                    ),
                    Row(    # height 54
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:22px; height:54px;',
                        ),
                        Image(
                            onmouseup => _get_mouseup_js('Back'),
                            onclick   => \&_remote_action,
                            id        => 'Back',
                            src       => _get_image('Back.gif'),
                            style => 'cursor:pointer; width:85px; height:54px;',
                        ),
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:22px; height:54px;',
                        ),
                        Image(
                            onmouseup => _get_mouseup_js('Home'),
                            onclick   => \&_remote_action,
                            id        => 'Home',
                            src       => _get_image('Home.gif'),
                            style => 'cursor:pointer; width:85px; height:54px;',
                        ),
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:22px; height:54px;',
                        ),
                    ),
                ),
            ),
            Grid(
                Columns(
                    Column( flex => 0 ),
                    Column( flex => 0 ),
                    Column( flex => 0 ),
                    Column( flex => 0 ),
                    Column( flex => 0 ),
                ),
                Rows(
                    Row(    # height 53
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:22px; height:53px;',
                        ),
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:53px; height:53px;',
                        ),
                        Image(
                            onmouseup => _get_mouseup_js('Up'),
                            onclick   => \&_remote_action,
                            id        => 'Up',
                            src       => _get_image('Up.gif'),
                            style => 'cursor:pointer; width:86px; height:53px;',
                        ),
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:53px; height:53px;',
                        ),
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:22px; height:53px;',
                        ),
                    ),
                    Row(    # height 86
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:22px; height:86px;',
                        ),
                        Image(
                            onmouseup => _get_mouseup_js('Left'),
                            onclick   => \&_remote_action,
                            id        => 'Left',
                            src       => _get_image('Left.gif'),
                            style => 'cursor:pointer; width:53px; height:86px;',
                        ),
                        Image(
                            onmouseup => _get_mouseup_js('Select'),
                            onclick   => \&_remote_action,
                            id        => 'Select',
                            src       => _get_image('Select.gif'),
                            style => 'cursor:pointer; width:86px; height:86px;',
                        ),
                        Image(
                            onmouseup => _get_mouseup_js('Right'),
                            onclick   => \&_remote_action,
                            id        => 'Right',
                            src       => _get_image('Right.gif'),
                            style => 'cursor:pointer; width:53px; height:86px;',
                        ),
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:22px; height:86px;',
                        ),
                    ),
                    Row(    # height 53
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:22px; height:53px;',
                        ),
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:53px; height:53px;',
                        ),
                        Image(
                            onmouseup => _get_mouseup_js('Down'),
                            onclick   => \&_remote_action,
                            id        => 'Down',
                            src       => _get_image('Down.gif'),
                            style => 'cursor:pointer; width:86px; height:53px;',
                        ),
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:53px; height:53px;',
                        ),
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:22px; height:53px;',
                        ),
                    ),
                ),
            ),
            Grid(
                Columns(
                    Column( flex => 0 ),
                    Column( flex => 0 ),
                    Column( flex => 0 ),
                    Column( flex => 0 ),
                    Column( flex => 0 ),
                ),
                Rows(
                    Row(    # height 52
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:22px; height:52px;',
                        ),
                        Image(
                            onmouseup => _get_mouseup_js('Rev'),
                            onclick   => \&_remote_action,
                            id        => 'Rev',
                            src       => _get_image('Rev.gif'),
                            style => 'cursor:pointer; width:52px; height:52px;',
                        ),
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:17px; height:52px;',
                        ),
                        Image(
                            onmouseup => _get_mouseup_js('Play'),
                            onclick   => \&_remote_action,
                            id        => 'Play',
                            src       => _get_image('Play.gif'),
                            style => 'cursor:pointer; width:52px; height:52px;',
                        ),
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:16px; height:52px;',
                        ),
                        Image(
                            onmouseup => _get_mouseup_js('Fwd'),
                            onclick   => \&_remote_action,
                            id        => 'Fwd',
                            src       => _get_image('Fwd.gif'),
                            style => 'cursor:pointer; width:53px; height:52px;',
                        ),
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:22px; height:52px;',
                        ),
                    ),
                    Row(    # height 52
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:22px; height:52px;',
                        ),
                        Image(
                            onmouseup => _get_mouseup_js('InstantReplay'),
                            onclick   => \&_remote_action,
                            id        => 'InstantReplay',
                            src       => _get_image('InstantReplay.gif'),
                            style => 'cursor:pointer; width:53px; height:52px;',
                        ),
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:17px; height:52px;',
                        ),
                        Image(
                            onmouseup => _get_mouseup_js('Info'),
                            onclick   => \&_remote_action,
                            id        => 'Info',
                            src       => _get_image('Info.gif'),
                            style => 'cursor:pointer; width:53px; height:52px;',
                        ),
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:16px; height:52px;',
                        ),
                        Image(
                            onmouseup => _get_mouseup_js('Search'),
                            onclick   => \&_remote_action,
                            id        => 'Search',
                            src       => _get_image('Search.gif'),
                            style => 'cursor:pointer; width:53px; height:52px;',
                        ),
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:22px; height:52px;',
                        ),
                    ),
                    Row(    # height 22
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:22px; height:22px;',
                        ),
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:53px; height:22px;',
                        ),
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:17px; height:22px;',
                        ),
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:53px; height:22px;',
                        ),
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:16px; height:22px;',
                        ),
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:53px; height:22px;',
                        ),
                        Image(
                            src   => _get_image('empty-square.gif'),
                            style => 'width:22px; height:22px;',
                        ),
                    ),
                ),
            ),
        ),
    ),
);

__END__
