package PDK::Concern::Netdisco::Role;

use utf8;
use v5.30;
use Moose::Role;
use Carp qw(croak);
use namespace::autoclean;
use File::Path qw(make_path);

# 定义必须实现的方法
requires 'gen_iface_desc';
requires 'commands';

# 设备属性，只读，类型为 PDK::Device::Base，必填
has device => (
  is       => 'ro',
  does     => 'PDK::Device::Role',
  required => 1,
);

# 月份属性，只读，延迟加载
has month => (
  is      => 'ro',
  default => sub {
    my $month = `date +%Y-%m`;
    chomp($month);
    return $month;
  },
);

# 日期属性，只读，延迟加载
has date => (
  is      => 'ro',
  default => sub {
    my $date = `date +%Y-%m-%d`;
    chomp($date);
    return $date;
  },
);

# 工作目录属性，可读写
has workdir => (
  is      => 'rw',
  default => sub { $ENV{PDK_CONCERN_NETDISCO_HOME} // glob("~") },
);

# 调试开关，可读写
has debug => (
  is      => 'rw',
  isa     => 'Int',
  default => sub { $ENV{PDK_CONCERN_NETDISCO_DEBUG} // 0 },
);

# 获取当前时间
sub now {
  my $now = `date "+%Y-%m-%d %H:%M:%S"`;
  chomp($now);
  return $now;
}

# 打印调试信息
sub dump {
  my ($self, $msg) = @_;

  # 跳过空白行
  $msg .= ';' unless $msg =~ /^\s*$/ || $msg =~ /[,，！!。.]$/;

  # 打印时间戳
  my $text = $self->now() . " - [debug] $msg";
  if ($self->debug == 1) {
    say $text;
  }
  elsif ($self->debug > 1) {
    my $workdir = "$self->{workdir}/$self->{month}/$self->{date}";
    make_path($workdir) unless -d $workdir;

    my $filename = "$workdir/$self->{device}{host}.txt";
    open(my $fh, '>>:encoding(UTF-8)', $filename) or croak "无法打开文件 $filename 进行写入: $!";
    print $fh "$text\n" or croak "写入文件 $filename 失败: $!";
    close($fh) or croak "关闭文件句柄 $filename 失败: $!";
  }
}

# 将配置写入指定文件
sub write_file {
  my ($self, $config, $name) = @_;

  croak("必须提供非空配置信息") unless !!$config;

  $name //= $self->{device}{host} . ".txt";
  my $workdir = "$self->{workdir}/$self->{month}/$self->{date}";
  make_path($workdir) unless -d $workdir;

  $self->dump("准备将配置文件写入工作目录: ($workdir)");
  my $filename = "$workdir/$name";
  open(my $fh, '>>encoding(UTF-8)', $filename) or croak "无法打开文件 $filename 进行写入: $!";
  print $fh $config or croak "写入文件 $filename 失败: $!";
  close($fh) or croak "关闭文件句柄 $filename 失败: $!";

  $self->dump("已将配置文件写入文本文件: $filename");

  return { success => 1 };
}

# 探索网络拓扑
sub explore_topology {
  my ($self) = @_;

  # 获取设备信息
  my $device = $self->device;

  # 执行命令并获取结果
  my $result = $device->execCommands($self->commands);
  if ($result->{success}) {
    my @topology = split(/\n/, $result->{result});
    my @commands = $self->gen_iface_desc(\@topology);
    return $device->execCommands(\@commands);
  }
  else {
    return { success => 0, reason => $result->{reason} };
  }
}

# 规范化接口名称
sub refine_if {
  my ($self, $name) = @_;

  # 替换接口名称，精简描述邻居关系
  $name =~ s/Ten-GigabitEthernet/TE/gi;
  $name =~ s/GigabitEthernet/GE/gi;
  $name =~ s/Smartrate-Ethernet/SE/gi;
  $name =~ s/Ethernet/Eth/gi;
  $name =~ s/xethernet/XE/gi;
  $name =~ s/ethernet/E/gi;
  $name =~ s/xge/TE/gi;
  $name =~ s/Twe/TW/gi;
  $name =~ s/eth/Eth/gi;
  # 老的 H3C 交换机需要修改为这种格式
  $name =~ s/ge/G/gi;

  return $name;
}

1;

# ABSTRACT: Based Moose for network device discovery and management