#============================================================= -*-perl-*-
#
# Template::Plugin::Latex
#
# DESCRIPTION
#   Template Toolkit plugin for Latex
#
# AUTHOR
#   Andy Wardley   <abw@wardley.org>
#
# COPYRIGHT
#   Copyright (C) 1996-2006 Andy Wardley.  All Rights Reserved.
#
#   This module is free software; you can redistribute it and/or
#   modify it under the same terms as Perl itself.
#
# HISTORY
#   * Originally written by Craig Barratt, Apr 28 2001.
#   * Win32 additions by Richard Tietjen.
#   * Extracted into a separate Template::Plugin::Latex module by 
#     Andy Wardley, 27 May 2006
#
#========================================================================
 
package Template::Plugin::Latex;

use strict;
use warnings;
use base 'Template::Plugin';
use Template::Latex;
use File::Path;
use File::Spec;
use Cwd;

our $VERSION = sprintf("%d.%02d", q$Revision: 2.70 $ =~ /(\d+)\.(\d+)/);
our $DEBUG   = 0 unless defined $DEBUG;
our $ERROR   = '';


sub new {
    my $class   = shift;
    my $context = shift;
    my $config  = @_ && ref $_[-1] eq 'HASH' ? pop : { };
    Template::Latex->define_filter($context, $config);
    return '';
}

1;

__END__

=head1 NAME

Template::Plugin::Latex - Template Toolkit plugin for Latex

=head1 SYNOPSIS

    [% USE Latex %]

    [% FILTER latex("example.pdf") -%]
    \documentclass{article}
    \begin{document}
    This is a PDF document generated by 
    LaTeX and the Template Toolkit.
    \end{document}
    [% END -%]

=head1 DESCRIPTION

This plugin allows you to use LaTeX to generate PDF, PostScript and
DVI output files from the Template Toolkit.  

The C<latex> filter was distributed as part of the core Template
Toolkit until version 2.15 released in May 2006 when it was moved into
the separate Template-Latex distribution.  It should now be loaded as
a plugin to enable the C<latex> filter:

    [% USE Latex -%]
    [% FILTER latex %]
    ...LaTeX document...
    [% END %] 

You can specify a different filter name using the C<filter> parameter.

    [% USE Latex(filter='pdf') -%]
    [% FILTER pdf %]
    ...LaTeX document...
    [% END %] 

The paths to the F<latex>, F<pdflatex> and F<dvips> should be
pre-defined as part of the installation process (i.e. when you 
run C<perl Makefile.PL>).  You can specify alternate values when 
you load the plugin.

    [% USE Latex( latex    = '/usr/bin/latex'
                  pdflatex = '/usr/bin/pdflatex'
                  dvips    = '/usr/bin/dvips' )
    %]

You can also specify the default output format.  This value can be
C<latex>, C<pdf> or C<dvi>.

    [% USE Latex(format='pdf') %]

With the plugin loaded and a default format defined, you can now use
the C<latex> filter.

    [% FILTER latex -%]
    \documentclass{article}
    \begin{document}
    This is a PDF document generated by 
    Latex and the Template Toolkit.
    \end{document}
    [% END %]

You can pass additional arguments when you invoke the filter to set
program paths or specify the output format.

    [% FILTER latex(format='pdf') -%]
       ...LaTeX document...
    [% END %]

    [% FILTER latex(latex='/path/to/latex') -%]
       ...LaTeX document...
    [% END %]

The template content between the C<FILTER> and C<END> directives will
be piped to the appropriate program(s) to generate the document
output.  This is fine if you're generating a document directly from a
template.  For example:

F<example.pdf>:

    [% USE Latex(format='pdf') -%]
    [% FILTER latex %]
    ...LaTeX document...
    [% END -%]

The output will be a binary format PDF, PostScript or DVI file.  You
should be careful not to prepend or append any extraneous characters
or text outside the FILTER block as this text will be included in the
file output.  Notice in the above example how we use the post-chomp
flags ('-') at the end of the C<USE> and C<END> directives to remove
the trailing newline characters.

If you're redirecting the output to a file via the third argument of
the Template module's C<process()> method then you should also pass
the C<binmode> parameter set to a true value to indicate that it is a
binary file.

    use Template;
    
    my $tt = Template->new({
        INCLUDE_PATH => '/path/to/templates',
        OUTPUT_PATH  => '/path/to/pdf/output',
    });
    my $vars = {
        title => 'Hello World',
    }
    $tt->process('example.tt2', $vars, 'example.pdf', binmode => 1)
        || die $tt->error();

If you want to capture the output to a template variable, you can do
so like this:

    [% output = FILTER latex %]
    ...LaTeX document...
    [% END %]

If you want to write the output to a file then you can specify an 
C<output> parameter.

    [% FILTER latex(output='example.pdf') %]
    ...LaTeX document...
    [% END %]

If you don't explicity specify an output format then the filename
extension (e.g. 'pdf' in the above example) will be used to determine
the correct format.

=head1 AUTHOR

Andy Wardley E<lt>abw@wardley.orgE<gt> L<http://wardley.org/>

The original Latex plugin on which this is based was written by Craig
Barratt with additions for Win32 by Richard Tietjen.

=head1 COPYRIGHT

Copyright (C) 1996-2006 Andy Wardley.  All Rights Reserved.

This module is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

L<Template::Latex>

=cut

# Local Variables:
# mode: perl
# perl-indent-level: 4
# indent-tabs-mode: nil
# End:
#
# vim: expandtab shiftwidth=4:
