#include "src/ecg.h"

// it's the fastest way to hanlde arguments as perls (my %hash = @_)
// return an offset to modified array from witch agruments are uniq
int args_to_uniq(SV *args[], int len_args) {
  if (len_args % 2) croak("Not even list");

  int end = len_args - 1;
  int offset = end;
  for (int i = end; i >= 0; i -= 2) {

    SV *curkey = args[i - 1];
    SV *curval = args[i];
    int j;
    for (j = end; j > offset; j -= 2)
      if (!strcmp(SvPV_nolen(args[j - 1]), SvPV_nolen(curkey))) break;

    // if j == offset, we've reached the end without a match +
    if (j == offset) {
      // if i!= j, replace value (because if i=j all array may be already
      // uniq and we don't need to overwrite anything)
      if (j != i) {
        args[offset - 1] = curkey;
        args[offset] = curval;
      };
      offset -= 2;
    }
  };

  return offset + 1;
}

static void hv_store_or_croak(HV *hv, const char *key, SV *val) {
  if (hv_store(hv, key, strlen(key), val, 0)) return;
  SvREFCNT_dec(val);
  croak("Can't store value");
}

static void av_store_or_croak(AV *hv, int key, SV *val) {
  if (av_store(hv, key, val)) return;
  SvREFCNT_dec(val);
  croak("Can't store value");
}

// for tests - return a shape of registered attrs
static SV *ecg_to_array(ClassGen *gen) {
  AV *attrs = newAV();

  for (int i = 0; i < gen->size; i++) {
    HV *slot = newHV();
    ClassGenAttr *attr = gen->attrs[i];
    char *key = attr->name;
    hv_store_or_croak(slot, "key", newSVpv(key, 0));
    if (attr->ro) hv_store_or_croak(slot, "ro", &PL_sv_yes);
    if (attr->check) hv_store_or_croak(slot, "check", newSVsv(attr->check));

    // rtype
    char *rtype = "";

    switch (attr->rtype) {
    case ECG_RELAXED:
      rtype = "relaxed";
      break;
    case ECG_DEFAULT:
      rtype = "default";
      break;
    case ECG_DEFAULT_CODE:
      rtype = "default_code";
      break;
    case ECG_REQUIRED:
      rtype = "required";
      break;
    case ECG_LAZY:
      rtype = "lazy";
      break;
    }; // let compiler notify us about unused, don't use default

    hv_store_or_croak(slot, "rtype", newSVpv(rtype, 0));
    if (attr->check) hv_store_or_croak(slot, "check", newSVsv(attr->check));
    if (attr->rvalue) hv_store_or_croak(slot, "rvalue", newSVsv(attr->rvalue));
    av_push(attrs, newRV_noinc((SV *)slot));
  }

  return newRV_noinc((SV *)attrs);
}

static void ecg_free(ClassGen *gen) {
  for (int i = 0; i < gen->size; i++) {
    ClassGenAttr *attr = gen->attrs[i];
    if (attr->rvalue) SvREFCNT_dec(attr->rvalue);
    if (attr->check) SvREFCNT_dec(attr->check);
    free(gen->attrs[i]);
  };
  free(gen->attrs);
  free(gen);
}

// find attr pointer by name or push a name to the gen
// return pointer to attr
ClassGenAttr *ecg_attr_slot(ClassGen *gen, char *name) {

  ClassGenAttr *attr = NULL;

  // try to find. because attr with same name should have the same index
  for (int i = 0; i < gen->size; i++) {
    if (!strcmp(gen->attrs[i]->name, name)) {
      attr = gen->attrs[i];
      break;
    };
  }

  if (attr) {
    if (attr->check) SvREFCNT_dec(attr->check);
    if (attr->rvalue) SvREFCNT_dec(attr->rvalue);
    attr->ro = 0;
    attr->rtype = 0;
    attr->check = NULL;
    attr->rvalue = NULL;
  } else { // not found - push
    int index = gen->size;

    // realloc attrs array
    size_t attrs_size = ((gen->size + 1) * sizeof(gen->attrs[0]));
    // warn("realloc: %zd\n", attrs_size);
    ClassGenAttr **attrs = realloc(gen->attrs, attrs_size);
    if (!attrs) croak("No memory"); // error
    gen->attrs = attrs;
    gen->size++;

    // alloc attr
    int attr_size = sizeof(ClassGenAttr) + strlen(name) + 1;
    // warn("#allocating attr size: %d for attr: %s;\n", attr_size, name);
    attr = calloc(1, attr_size);
    if (!attr) croak("No memory");
    strcpy(attr->name, name);
    attrs[index] = attr;
  };

  return attr;
}

static ClassGen *find_gen(SV *cv) {
  MAGIC *mg = mg_findext(cv, PERL_MAGIC_ext, &GEN_TBL);
  if (!SvROK(mg->mg_obj)) {
    if (!PL_dirty) croak("Can't find storage");
    return NULL;
  }
  return INT2PTR(ClassGen *, SvIV((SV *)SvRV(mg->mg_obj)));
}

AV *init_storage(SV *obj) {
  if (!SvROK(obj)) croak("Not a ref");
  AV *array = newAV();
  sv_magicext(SvRV(obj), (SV *)array, PERL_MAGIC_ext, &STORAGE_TBL, NULL, 0);
  SvREFCNT_dec(array);
  return array;
}

AV *get_storage(SV *obj) {
  if (!SvROK(obj)) croak("Not a ref");
  MAGIC *mg = mg_findext(SvRV(obj), PERL_MAGIC_ext, &STORAGE_TBL);
  if (!mg) croak("No magic, forgot to init object?");
  return (AV *)mg->mg_obj;
}

static void reg_attr(ClassGen *gen, char *name, SV *attrrv) {
  if (!(SvROK(attrrv) && SvTYPE(SvRV(attrrv)) == SVt_PVHV)) croak("Not a HASH");
  HV *hash = (HV *)SvRV(attrrv);

  SV **tmp;
  if (!(tmp = hv_fetchs(hash, "rtype", 0))) croak("no rtype");
  char *rtype = SvPV_nolen(*tmp);

  ClassGenAttr *attr = ecg_attr_slot(gen, name);
  if (!strcmp(rtype, "relaxed")) {
    attr->rtype = ECG_RELAXED;
  } else if (!strcmp(rtype, "default")) {
    attr->rtype = ECG_DEFAULT;
  } else if (!strcmp(rtype, "default_code")) {
    attr->rtype = ECG_DEFAULT_CODE;
  } else if (!strcmp(rtype, "required")) {
    attr->rtype = ECG_REQUIRED;
  } else if (!strcmp(rtype, "lazy")) {
    attr->rtype = ECG_LAZY;
  } else {
    croak("Bad rtype %s", rtype);
  }

  if ((tmp = hv_fetchs(hash, "ro", 0))) attr->ro = 1;
  if ((tmp = hv_fetchs(hash, "check", 0))) attr->check = newSVsv(*tmp);
  if ((tmp = hv_fetchs(hash, "rvalue", 0))) attr->rvalue = newSVsv(*tmp);
};

static inline int name2index(ClassGen *gen, char *name) {
  for (int i = 0; i < gen->size; i++) {
    if (!strcmp(name, gen->attrs[i]->name)) return i;
  }
  croak("Unknown attribute \"%s\"", name);
}

void invoke_check(SV *cv, SV *value, char *name) {
  SV *ok = &PL_sv_undef;
  SV *msg = &PL_sv_undef;

  dSP;
  PUSHMARK(SP);
  PUSHs(value);
  PUTBACK;
  int count = call_sv(cv, G_ARRAY);
  // could return 0 or 1 or 2 or more
  SPAGAIN;

  if (count == 1) {
    ok = POPs;
  } else if (count > 1) {
    msg = POPs;
    ok = POPs;
  }
  PUTBACK;

  if (!SvTRUE(ok))
    croak("Bad value \"%s\" for attribute \"%s\": %s", SvPV_nolen(value), name,
          SvTRUE(msg) ? SvPV_nolen(msg) : "");
};

SV *invoke_lazy(SV *self, SV *cv) {
  dSP;

  PUSHMARK(SP);
  PUSHs(self);
  PUTBACK;
  int count = call_sv(cv, G_SCALAR);
  if (count != 1) croak("bad count");
  SPAGAIN;
  SV *result = newSVsv(POPs);
  PUTBACK;

  return result;
}

static void do_init(SV *cv, SV *class, SV *obj, SV *args[], int count) {
  int offset = args_to_uniq(args, count);

  AV *storage = init_storage(obj);

  ClassGen *gen = find_gen(cv);
  if (!gen) goto IN_GD; // global destruction?
  // // iterate passed opts
  for (int i = offset; i < count; i += 2) {
    int index = name2index(gen, SvPV_nolen(args[i]));
    SV *val = args[i + 1];
    char *key = SvPV_nolen(args[i]);
    SV *check = gen->attrs[index]->check;
    if (check) invoke_check(check, val, key);
    av_store_or_croak(storage, index, SvREFCNT_inc(val));
  }

  // iterate known attributes
  for (int i = 0; i < gen->size; i++) {
    if (av_exists(storage, i)) continue;
    ClassGenAttr *attr = gen->attrs[i];
    if (attr->rtype == ECG_REQUIRED) {
      croak("Attribute \"%s\" is required", attr->name);
    } else if (attr->rtype == ECG_DEFAULT) {
      av_store_or_croak(storage, i, newSVsv(attr->rvalue));
    } else if (attr->rtype == ECG_DEFAULT_CODE) {
      SV *val = invoke_lazy(class, attr->rvalue);
      av_store_or_croak(storage, i, val);
    }
  }

  sv_bless(obj, gv_stashsv(class, GV_ADD));
  return;

IN_GD:
  croak("trying to bless object during global destruction");
  return;
}

static void xs_attr(SV *cv) {
  dXSARGS;
  if (!(items == 1 || items == 2)) croak("Bad usage");

  SV *self = ST(0);
  SV *result;
  AV *storage = get_storage(self);
  int index = CvXSUBANY(cv).any_iv;

  if (items == 1) {
    SV **val = av_fetch(storage, index, 0);
    if (val) {
      result = *val;
    } else {

      ClassGen *gen = find_gen(cv);
      if (!gen) goto IN_GD_GET;
      ClassGenAttr *attr = gen->attrs[index];

      if (attr->rtype == ECG_LAZY) {
        result = invoke_lazy(self, attr->rvalue);
        av_store_or_croak(storage, index, result);
      } else {
        result = &PL_sv_undef;
      }
    }
  } else {
    ClassGen *gen = find_gen(cv);
    if (!gen) goto IN_GD_SET;
    ClassGenAttr *attr = gen->attrs[index];

    if (attr->ro) croak("Attribute \"%s\" is readonly", attr->name);
    SV *value = ST(1);
    if (attr->check) invoke_check(attr->check, value, attr->name);
    av_store_or_croak(storage, index, value);
    SvREFCNT_inc(value);
    result = self;
  }

  ST(0) = result;
  XSRETURN(1);

// global destruction?
IN_GD_GET:
  ST(0) = &PL_sv_undef;
  XSRETURN(1);

IN_GD_SET:
  ST(0) = self;
  XSRETURN(1);
}

static void xs_new(SV *cv) {
  dXSARGS;
  if (items < 1) croak("Usage: class, ref");
  SV *class = ST(0);
  SV *obj = sv_2mortal(newRV_noinc((SV *)newHV()));
  do_init(cv, class, obj, &ST(1), items - 1);
  ST(0) = obj;
  XSRETURN(1);
};

static void xs_init(SV *cv) {
  dXSARGS;
  if (items < 2) croak("Usage: class, ref");
  SV *class = ST(0);
  SV *obj = ST(1);
  do_init(cv, class, obj, &ST(2), items - 2);
  ST(0) = obj;
  XSRETURN(1);
};

static void xs_attrs_map(SV *cv) {
  dXSARGS;
  if (items != 1) croak("Usage: $obj->attrs()");
  SV *obj = ST(0);
  AV *storage = get_storage(obj);
  ClassGen *gen = find_gen(cv);

  SP--;
  for (int i = 0; i < gen->size; i++) {
    char *name = gen->attrs[i]->name;
    mPUSHp(name, strlen(name));
    SV **val = av_fetch(storage, i, 0);
    mPUSHs(val ? newSVsv(*val) : &PL_sv_undef);
  }
  XSRETURN(gen->size * 2);
}

void xs_attr_exists(SV *cv) {
  dXSARGS;
  if (items != 2) croak("Usage: $obj->attr_exists(\"name\")");
  SV *obj = ST(0);
  AV *storage = get_storage(obj);
  ClassGen *gen = find_gen(cv);
  if (!gen) goto IN_GD;
  int index = name2index(gen, SvPV_nolen(ST(1)));

  ST(0) = av_exists(storage, index) ? &PL_sv_yes : &PL_sv_undef;
  XSRETURN(1);

IN_GD:
  XSRETURN_UNDEF;
};

void xs_attr_delete(SV *cv) {
  dXSARGS;
  if (items != 2) croak("Usage: $obj->attr_delete(\"name\")");
  SV *obj = ST(0);
  AV *storage = get_storage(obj);
  ClassGen *gen = find_gen(cv);
  if (!gen) goto IN_GD;
  int index = name2index(gen, SvPV_nolen(ST(1)));

  ST(0) = av_delete(storage, index, 0);
  XSRETURN(1);

IN_GD:
  XSRETURN_UNDEF;
};

CV *gen_init_cv(SV *self) {
  CV *xsub = newXS(NULL, (XSUBADDR_t)xs_init, __FILE__);
  sv_magicext((SV *)xsub, self, PERL_MAGIC_ext, &GEN_TBL, NULL, 0);
  sv_2mortal((SV *)xsub);
  return xsub;
}

CV *gen_new_cv(SV *self) {
  CV *xsub = newXS(NULL, (XSUBADDR_t)xs_new, __FILE__);
  sv_magicext((SV *)xsub, self, PERL_MAGIC_ext, &GEN_TBL, NULL, 0);
  sv_2mortal((SV *)xsub);
  return xsub;
}

CV *gen_attrs_map(SV *self) {
  CV *xsub = newXS(NULL, (XSUBADDR_t)xs_attrs_map, __FILE__);
  sv_magicext((SV *)xsub, self, PERL_MAGIC_ext, &GEN_TBL, NULL, 0);
  sv_2mortal((SV *)xsub);
  return xsub;
}

CV *gen_attr_exists(SV *self) {
  CV *xsub = newXS(NULL, (XSUBADDR_t)xs_attr_exists, __FILE__);
  sv_magicext((SV *)xsub, self, PERL_MAGIC_ext, &GEN_TBL, NULL, 0);
  sv_2mortal((SV *)xsub);
  return xsub;
}

CV *gen_attr_delete(SV *self) {
  CV *xsub = newXS(NULL, (XSUBADDR_t)xs_attr_delete, __FILE__);
  sv_magicext((SV *)xsub, self, PERL_MAGIC_ext, &GEN_TBL, NULL, 0);
  sv_2mortal((SV *)xsub);
  return xsub;
}

CV *gen_attr(SV *self, char *name) {
  assert(SvROK(self));
  ClassGen *gen = INT2PTR(ClassGen *, SvIV((SV *)SvRV(self)));
  CV *xsub = newXS(NULL, (XSUBADDR_t)xs_attr, __FILE__);
  CvXSUBANY(xsub).any_iv = name2index(gen, name);
  sv_magicext((SV *)xsub, self, PERL_MAGIC_ext, &GEN_TBL, NULL, 0);
  sv_2mortal((SV *)xsub);
  return xsub;
}
