/*
 * This may look like C code, but it is really -*- C++ -*-
 *	(well, it's a mix of C and C++)
 * 
 * Copyright (C) 1988 University of Illinois, Urbana, Illinois
 * Copyright (C) 1989 University of Colorado, Boulder, Colorado
 * Copyright (C) 1990 University of Colorado, Boulder, Colorado
 *
 * written by Dirk Grunwald (grunwald@foobar.colorado.edu)
 *
 */

#ifndef SpinLock_h
#define SpinLock_h
#pragma once

/*
 *
 *	Implement a UNIX process spin-lock. Suitable for Threads,
 *	(if you know blocking thread isn't switched) and for Unix
 *	processes.
 */

#if defined(__GNUG__)
#  include <assert.h>
#endif

#include <AwesimeConfig.h>

/*
 *	C Version (always included)
 */


/*
 *	Specialize per architecture
 */

#if	defined(encore)
#  define SpinLockFree '\x00'
   typedef unsigned char SpinLockType;
#else
#  define SpinLockFree '\x00'
   typedef unsigned char SpinLockType;
#endif


/*
 *	Machines that don't actually use shared memory and spinlocks
 *	just check the current state to make certain things wouldn't
 *	deadlock on a true shared memory machine.
 */

inline void SpinLockRelease(SpinLockType *lock)
{
#if defined(__GNUG__)
    assert(*lock != SpinLockFree);
#endif
    *lock = SpinLockFree;
}

#if !defined(USE_SHARED_MEMORY)

static inline void SpinLockReserve(SpinLockType *lock)
{
#if defined(__GNUG__)
    assert(*lock == SpinLockFree);
#endif
    *lock = ~SpinLockFree;
}

static inline int SpinLockReserveNoBlock(SpinLockType *lock)
{
#if defined(__GNUG__)
    assert(*lock == SpinLockFree);
#endif
    *lock = ~SpinLockFree;
    return(1);
}

/*
 *	The following are all SHARED MEMORY spinlocks.
 */

#elif	defined(encore)
static inline void SpinLockReserve(SpinLockType *lock)
{
    for( ;; ) {
	volatile SpinLockType * x = lock;
	SpinLockType oldState = 1;

	/* coded this way due to error in G++/Gcc 1.35 */

	asm volatile( "sbitib $0,%0" : "=g" (*x) );
	asm volatile( "sfsb %0" : "=g" (oldState) );
	if( oldState ) {
	    while( *x != SpinLockFree );
	}
	else {
	    return;
	}
    }
}

static inline void SpinLockReserveNoBlock(SpinLockType *lock)
{
    volatile SpinLockType *x = lock;
    SpinLockType oldState = 1;
    asm volatile( "sbitib $0,%0" : "=g" (*x) );
    asm volatile( "sfsb %0" : "=g" (oldState) );
    /* 
     *	oldState == 0 -> we now own the lock
     *  oldState == 1 -> lock was owned, we don't own it.
     */
    return( ~ oldState );
}
#endif

/*
 *	G++ INTERFACE
 */

#if defined(__GNUG__)

class SpinLock {
    SpinLockType state;
public:
    SpinLock();
    ~SpinLock();
    void reserve();
    int reserveNoBlock();
    void release();
};

static inline SpinLock::SpinLock()
{
	state = SpinLockFree;
}

static inline SpinLock::~SpinLock()
{
	assert(state == SpinLockFree);
}

static inline void SpinLock::reserve()
{
    SpinLockReserve(&state);
}

static inline int SpinLock::reserveNoBlock()
{
    return( SpinLockReserveNoBlock(&state) );
}

static inline void SpinLock::release()
{
    SpinLockRelease(&state);
}

#endif

#endif
