// This may look like C code, but it is really -*- C++ -*-
// 
// Copyright (C) 1988 University of Illinois, Urbana, Illinois
// Copyright (C) 1989 University of Colorado, Boulder, Colorado
// Copyright (C) 1990 University of Colorado, Boulder, Colorado
//
// written by Dirk Grunwald (grunwald@foobar.colorado.edu)
//
#include "CpuMultiplexor.h"
#include "CpuMultiplexorP.h"
#include "Semaphore.h"
#include "FifoScheduler.h"
#include "Thread.h"

Semaphore::Semaphore(int count, ThreadContainer *scheduler)
{
    if (scheduler == 0) {
	pScheduler = new FifoScheduler;
	iDidAlloc = 1;
    } else {
	pScheduler = scheduler;
	iDidAlloc = 0;
    }
    pCount = count;
};

Semaphore::~Semaphore()
{
    lock.reserve();

    if (pScheduler != 0) {
	if (! pScheduler  -> isEmpty() ) {
	    cerr << "[Semaphore] Attempted to delete non-empty semaphore\n";
	    exit(1);
	}
	if (iDidAlloc) delete pScheduler;
	pScheduler = 0;
	iDidAlloc = 0;
    }

    lock.release();
}

void Semaphore::reserve()
{
    lock.reserve();
    if (pCount < 1) {
	lock.release();
	ThisCpu -> reserveByException( this );
    }
    else {
	pCount--;
	lock.release();
    }
}

//
//	This is only executed in the context of a CpuMultiplexor, never
//	a thread.
//
int
Semaphore::reserveByException(Thread *byWho)
{
    bool blocked = 0;
    lock.reserve();
    if (pCount < 1) {
	pScheduler -> add( byWho );
	blocked = 1;
    }
    pCount--;
    lock.release();
    return( blocked );
}

bool Semaphore::reserveNoBlock()
{
    bool gotIt = 1;

    lock.reserve();
    if (pCount < 1) {
	gotIt = 0;
    }
    else {
	pCount--;
    }
    lock.release();
    return( gotIt );
}
    
void Semaphore::release()
{
    Thread *p = 0;
    lock.reserve();
    if (pCount < 0) {
	assert( !pScheduler -> isEmpty() );
	p = pScheduler -> remove();
    }
    pCount++;
    lock.release();

    if (p != 0) {
	ThisCpu -> add(p);
    }
}

//
//	Need to lock these
//

unsigned Semaphore::size()
{
    return(pScheduler -> size());
}

int Semaphore::count(int xcount)
{
    lock.reserve();
    int tmp = pCount;
    pCount = xcount;
    lock.release();
    return(tmp);
}

int Semaphore::count()
{
    lock.reserve();
    int tmp = pCount;
    lock.release();
    return(tmp);
}

void Semaphore::incrCount(int increment)
{
    lock.reserve();
    pCount += increment;
    lock.release();
}

#ifdef UNDEF
void Semaphore::classPrintOn(ostream& out)
{
    int size = pScheduler -> size();

    out << " Semaphore with pCount =" << pCount << " and "
	<< size << " enqueued tasks\n";
    if (size > 0) {
	out << "Tasks are:\n" << *pScheduler << "\n";
    }
}
#endif
