/*******************************************************************************
*
* University of Western Australia
* Department of Computer Science
* Copyright (c) University of Western Australia
*
* SYSTEM :              VIP
* RELEASE:		4
* SUBSYSTEM:            LIB
* MODULE:		vipfreq.c - Functions to perform Fourier Transform on
*                              	    complex images.  All the functions in this
*                                   file are imported from an older version of
*                                   V.I.P. written in October 1988.
* AUTHOR:               DH PK
* CREATION DATE:        02 Oct 1991
* REVISION DATE:	25/5/94
*
******************************************************************************
*
* REVISION LOG
*
* REVISION:	       
* REVISION DATE:	
* COMMENT:		
* BY:			

****************************************************************************/

#ifndef lint
static char *sccs_id = "@(#)vipfreq.c	3.12 4/26/94";

#endif




#include <stdlib.h>
#include <stdio.h>
#include <math.h>
/*
#include <malloc.h>
#include <sys/types.h>
*/
#include <string.h>

#include "vip.h"
#include "vipiofn.h"
#include "vipspatfn.h"
#include "vipfreqfn.h"
#include "vipcomplexfn.h"
#include "misc.h"


/*--   Next_Power_Of_2  -------------------------------------------

Function returns next power of two >= input number

------------------------------------------------------------------*/

int     Next_Power_Of_2(x)
int     x;

{
    int     y = 1;

    while (y < x) {
	y *= 2;
    }

    return (y);
}

/*-- PadImage -------------------------------------------------*/

IMAGE  *PadImage(im)
IMAGE  *im;

{
    IMAGE  *outim;
    int     rows, cols;
    int     i;

    if (!Image_OK(im)) {
	VIP_Error_Msg("PadImage: received invalid image");
	return (NULL);
    }

    rows = Next_Power_Of_2((int) im->rows);
    cols = Next_Power_Of_2((int) im->cols);

    if (rows != im->rows || cols != im->cols) {	/* We need to pad */
	outim = (IMAGE *) Allocate_Image(im->umin, im->vmin, rows, cols, im->type);

	switch (im->type) {

	case BYTETYPE:
	    for (i = 0; i < im->rows; i++)
		memcpy(outim->i.c[i], im->i.c[i], im->cols * sizeof(char));
	    break;

	case COMPLEXTYPE:
	    for (i = 0; i < im->rows; i++)
		memcpy(outim->i.cx[i], im->i.cx[i], im->cols * sizeof(COMPLEX));
	    break;

	default:
	    VIP_Error_Msg("PadImage: cannot handle images other than BYTETYPE or COMPLEXTYPE");
	    exit(1);
	}

    }
    else {			/* no padding needed ; just copy to new
				 * pointer */
	outim = Copy_Image(im);
    }
    return (outim);
}


/*- Is_2_Power ------------------------------------------------------

Return 1 if the first argument, n, is an integer power of 2, 0 otherwise.
Also return a number, logn, to the second argument where 2^logn = n.

--------------------------------------------------------------------*/

int     Is_2_Power(n, logn)
int     n, *logn;
{
    int     i = 0, j = 1;

    if (n < 0)
	return (0);
    for (*logn = -1; j <= n; i++, j += j) {
	if (j == n) {
	    *logn = i;
	    break;
	}
    }

    if (*logn == -1) {
	/* n is not an integer power of 2 */
	VIP_Error_Msg("Dimension of the image should be an integer power of 2");
	return (0);
    }
    return (1);
}


/*- Shift_Image_Origin ----------------------------------------------

shift the origin of the image (default image origin is at the top-left
corner) to the image's centre.  Given an input image divided into four
quadrants, these four quadrants will be swapped as shown in the diagram
below.
	+---+---+		+---+---+
	|   |   |		|   |   |
	| 1 | 2 |		| 4 | 3 |
	|   |   |		|   |   |
	+---+---+	==>	+---+---+
	|   |   |		|   |   |
	| 3 | 4 |		| 2 | 1 |
	|   |   |		|   |   |
	+---+---+		+---+---+
	(input)			(output)

--------------------------------------------------------------------*/

int     Shift_Image_Origin(cin, cout)
IMAGE  *cin, *cout;
{
    register int rr, cc, nr, nc, nr1, nc1, cenr, cenc, i, j;
    COMPLEX *pc, *outp;

    if (!Image_OK(cin) || !Image_OK(cout)) {
	VIP_Error_Msg("Shift_Image_Origin: received invalid image");
	return (0);
    }


    if (cin->type != cout->type) {
	VIP_Error_Msg("Shift_Image_Origin: Input and output images should be of the same type");
	return (0);
    }
    if (cin->type != COMPLEXTYPE) {
	VIP_Error_Msg("Shift_Image_Origin: Cannot handle images other than COMPLEX type");
	return (0);
    }
    else if (cin->rows != cout->rows || cin->cols != cout->cols) {
	VIP_Error_Msg("Shift_Image_Origin: Input and output images have different dimensions");
	return (0);
    }

    nr = cin->rows;
    nc = cin->cols;
    nr1 = nr - 1;
    nc1 = nc - 1;
    cenr = nr >> 1;
    cenc = nc >> 1;
    for (i = 0, rr = cenr; i < nr; rr = (rr == nr1 ? 0 : rr + 1), i++) {
	pc = &(cin->i.cx[rr][cenc]);
	outp = cout->i.cx[i];
	for (j = 0, cc = cenc; j < nc; outp++, j++) {
	    memcpy(outp, pc, sizeof(COMPLEX));
	    pc++;
	    if (cc == nc1) {
		pc = cin->i.cx[rr];
		cc = 0;
	    }
	    else
		cc++;
	}
    }
    return (1);
}

/*- Image_Complex_Conjugate -----------------------------------------

Reverse the sign of the imaginary component of a COMPLEX image.
*** Note that Image_Complex_Conjugate is one of the functions that
does not preserve the content of images passing to it.
Image_Complex_Conjugate can be modified in the future if necessary.

--------------------------------------------------------------------*/

void    Image_Complex_Conjugate(im)
IMAGE  *im;
{
    register int rr, cc;
    cx_array cx;

    if (!Image_OK(im)) {
	VIP_Error_Msg("Image_Complex_Conjugate: received invalid image");
	return;
    }

    for (rr = im->rows - 1; rr >= 0; rr--)
	for (cc = im->cols - 1, cx = &im->i.cx[rr][im->cols - 1];
	    cc >= 0; cc--, cx--)
	    cx->i = -cx->i;
}

/*- FOURN -------------------------------------------------------------

   N dimensional FFT from Numerical Recipes in C p468
   Original code from N.M. Brenner of Lincoln Labs


   void fourn(data, nn, ndim, isign)

Function replaces data by its ndim dimensional discrete Fourier
transform, if isign = 1 nn[1..ndim] is an integer array containing
the lengths of each dimension (No of complex values), which must
be powers of 2.

data is a real array of length twice the product of these lengths,
in which the data are stored as in multidimensional complex array:
real and imaginary parts of each element in consecutive locations,
and the rightmost index of the array increases most rapidly as one
proceeds along data.  For a 2D array this is equivalent to storing
the data by rows.

If isign = -1, data is replaced by its inverse transform times the
product of the lengths of all dimensions.

Peter Kovesi  October 1990

------------------------------------------------------------------*/


#define SWAP(a,b) tempr = (a); (a)=(b); (b) = tempr;

void    fourn(float data[], int nn[], int ndim, int isign)
{

    int     i1, i2, i3, i2rev, i3rev, ip1, ip2, ip3, ifp1, ifp2;
    int     ibit, idim, k1, k2, n, nprev, nrem, ntot;
    float   tempi, tempr;
    double  theta, wi, wpi, wpr, wr, wtemp;	/* for trig functions */

/* adjust pointers to arrays so that when we index them from '1' we
   are actually starting from the beginning of the array.
*/
    data = data - 1;
    nn = nn - 1;

    ntot = 1;
    for (idim = 1; idim <= ndim; idim++) {	/* Get total No of complex
						 * values */
	ntot *= nn[idim];
    }

    nprev = 1;
    for (idim = ndim; idim >= 1; idim--) {	/* Main loop over dimensions */
	n = nn[idim];
	nrem = ntot / (n * nprev);
	ip1 = nprev << 1;
	ip2 = ip1 * n;
	ip3 = ip2 * nrem;
	i2rev = 1;

	for (i2 = 1; i2 <= ip2; i2 += ip1) {	/* This is the bit reversal
						 * section */
	    if (i2 < i2rev) {
		for (i1 = i2; i1 <= i2 + ip1 - 2; i1 += 2) {
		    for (i3 = i1; i3 <= ip3; i3 += ip2) {
			i3rev = i2rev + i3 - i2;
			SWAP(data[i3], data[i3rev]);
			SWAP(data[i3 + 1], data[i3rev + 1]);
		    }
		}
	    }

	    ibit = ip2 >> 1;
	    while (ibit >= ip1 && i2rev > ibit) {
		i2rev -= ibit;
		ibit >>= 1;
	    }
	    i2rev += ibit;
	}

/* Danielson - Lanczos section */

	ifp1 = ip1;
	while (ifp1 < ip2) {
	    ifp2 = ifp1 << 1;
	    theta = isign * 6.28318530717959 / (ifp2 / ip1);	/* Initialise for trig */
	    /* recurrence.  */
	    wtemp = sin(0.5 * theta);
	    wpr = -2.0 * wtemp * wtemp;
	    wpi = sin(theta);
	    wr = 1.0;
	    wi = 0.0;

	    for (i3 = 1; i3 <= ifp1; i3 += ip1) {
		for (i1 = i3; i1 <= i3 + ip1 - 2; i1 += 2) {
		    for (i2 = i1; i2 <= ip3; i2 += ifp2) {
			k1 = i2;/* Recurrence formula */
			k2 = k1 + ifp1;
			tempr = wr * data[k2] - wi * data[k2 + 1];
			tempi = wr * data[k2 + 1] + wi * data[k2];
			data[k2] = data[k1] - tempr;
			data[k2 + 1] = data[k1 + 1] - tempi;
			data[k1] += tempr;
			data[k1 + 1] += tempi;
		    }
		}
		wr = (wtemp = wr) * wpr - wi * wpi + wr;	/* Trig recurrence */
		wi = wi * wpr + wtemp * wpi + wi;
	    }
	    ifp1 = ifp2;
	}
	nprev *= n;

    }				/* main loop */

    if (isign == -1) {		/* rescale data back to proper level by
				 * dividing by ntot */
        i2 = 2 * ntot;
	for (i1 = 1; i1 <= i2; i1++) {
	    data[i1] /= ntot;
	}
    }

}

/*- FT_Image ------------------------------------------------------

Function to generate the Fast Fourier Transform of an image.

input:  inimage - pointer to image to be transformed
        shift   - if non-zero the origin of the image is
                  shifted so that zero frequencies are
                  placed in the middle of the image.
        forward_inverse - "forward" or "inverse".

returns: pointer to transformed image.

-------------------------------------------------------------------*/

IMAGE  *FT_Image(IMAGE * inimage, int shift, char *forward_inverse)
{
    IMAGE  *outimage = NULL, *tmp = NULL;
    int     nn[2];
    int     h, w, row, col;

    if (!Image_OK(inimage)) {
	VIP_Error_Msg("FT_Image: invalid image ");
	return (NULL);
    }

    /* set up data for FFT */

    /* find size parameters that are the closest powers of 2 >= actual size */

    h = Power_Of_2_Greater_Or_Equal((int) inimage->rows);
    w = Power_Of_2_Greater_Or_Equal((int) inimage->cols);

    nn[0] = w;
    nn[1] = h;

    outimage = Allocate_Image(0, 0, h, w, COMPLEXTYPE);

    if (!outimage) {
	VIP_Error_Msg("FT_Image: could not allocate space for data");
	return (NULL);
    }

    /*
     * Copy image into outimage. Note that outimage is one that may have been
     * padded out to size that is a power of 2
     */

    switch (inimage->type) {
    case BYTETYPE:
	for (row = 0; row < inimage->rows; row++)
	    for (col = 0; col < inimage->cols; col++)
		outimage->i.cx[row][col].r = inimage->i.c[row][col];

	break;

    case SHORTTYPE:
	for (row = 0; row < inimage->rows; row++)
	    for (col = 0; col < inimage->cols; col++)
		outimage->i.cx[row][col].r = inimage->i.s[row][col];

	break;

    case LONGTYPE:
	for (row = 0; row < inimage->rows; row++)
	    for (col = 0; col < inimage->cols; col++)
		outimage->i.cx[row][col].r = inimage->i.l[row][col];

	break;

    case FLOATTYPE:
	for (row = 0; row < inimage->rows; row++)
	    for (col = 0; col < inimage->cols; col++)
		outimage->i.cx[row][col].r = inimage->i.f[row][col];

	break;

    case DOUBLETYPE:
	for (row = 0; row < inimage->rows; row++)
	    for (col = 0; col < inimage->cols; col++)
		outimage->i.cx[row][col].r = inimage->i.d[row][col];

	break;

    case COMPLEXTYPE:
	for (row = 0; row < inimage->rows; row++)
	    memcpy(outimage->i.cx[row], inimage->i.cx[row], inimage->cols * sizeof(COMPLEX));

	break;

    case RGBTYPE:
    case HSITYPE:
	VIP_Error_Msg("FT_Image: not for RGB or HSI!!");
	Free_Image(outimage);
	return (NULL);
	break;

    default:
	VIP_Error_Msg("FT_Image: Illegal image type");
	Free_Image(outimage);
	return (NULL);

    }				/* switch */


    /* Check if we need to shift origin of input image to centre */

    if (strncmp(forward_inverse, "inverse", 7) == 0 && shift) {
	tmp = Copy_Image(outimage);
	Shift_Image_Origin(tmp, outimage);
	Free_Image(tmp);
    }

/* Now do the transform */

    if (strncmp(forward_inverse, "forward", 7) == 0)
	fourn((float *) &outimage->i.cx[0][0].r, nn, 2, 1);	/* forward transform */

    else if (strncmp(forward_inverse, "inverse", 7) == 0)
	fourn((float *) &outimage->i.cx[0][0].r, nn, 2, -1);	/* inverse transform */

    else {
	VIP_Error_Msg("FT_Image: undefined transform direction");
	Free_Image(outimage);
	return (NULL);
    }

    /* Check if we need to shift the output image origin to centre */

    if (strncmp(forward_inverse, "forward", 7) == 0 && shift) {
	tmp = Copy_Image(outimage);
	Shift_Image_Origin(tmp, outimage);
	Free_Image(tmp);
    }

    return (outimage);

}


/*- Homomorphic ------------------------------------------------

Function to perform homomorphic filtering on an image. An image
enhancing process that attempts to reduce the effects of low frequency
lighting variations across an image, yet at the same time accentuating
high frequency intensity variations to highlight the detail.
Reference: Gonzalez and Woods "Digital Image Processing" page 213.

Peter Kovesi  January 1994

input: image    - input image.
       LowGain  - gain factor for low frequencies (suggest 0.5).
       HighGain - gain factor for high frequencies (suggest 2.0).
       CutOff   - frequency at which transition from LowGain to
                  HighGain occurs (suggest 0.4).
       order    - order of Butterworth type filter (suggest 1)
       lHistogramCut - percentage of low end of histogram to
                       truncate (suggest 1).
       uHistogramCut - percentage of high end of histogram to
                       truncate (suggest 5).

Function returns filtered image.

------------------------------------------------------------------*/


IMAGE  *Homomorphic(IMAGE * image, double LowGain, double HighGain,
            double CutOff, int order, int lHistogramCut, int uHistogramCut)
{
    IMAGE  *t1 = (IMAGE *) NULL, *t2 = (IMAGE *) NULL, *t3 = (IMAGE *) NULL, *t4 = (IMAGE *) NULL,
           *t5 = (IMAGE *) NULL, *t6 = (IMAGE *) NULL, *t7 = (IMAGE *) NULL;
    IMAGE  *outimage = (IMAGE *) NULL, *filter = (IMAGE *) NULL;
    int     r, c;

    if (!Image_OK(image)) {
	VIP_Error_Msg("Homomorphic: received invalid image");
	return (NULL);
    }


    t1 = Log_Image(image);
    t2 = FT_Image(t1, 0, "forward");
    Free_Image(t1);
    filter = Make_Homomorphic_Filter(t2->rows, t2->cols, CutOff, LowGain,
	HighGain, order);
    t3 = Multiply_Image(t2, filter);
    Free_Image(t2);
    Free_Image(filter);

    t4 = FT_Image(t3, 0, "inverse");
    Free_Image(t3);
    if (!t4) {
	VIP_Error_Msg("Homomorphic: An error occurred during a function call");
	return (NULL);
    }


/* Extract real part of image */

    t5 = (IMAGE *) Allocate_Image(0, 0, t4->rows, t4->cols, FLOATTYPE);

    for (r = 0; r < t4->rows; r++) {
	for (c = 0; c < t4->cols; c++) {
	    t5->i.f[r][c] = t4->i.cx[r][c].r;
	}
    }

    Free_Image(t4);

    t6 = Exp_Image(t5);
    Free_Image(t5);

/* Remove outlying values of image by truncating top and lower ends of
the image histogram by HistogramCut percent */

    t7 = Truncate_Image_Histogram(t6, lHistogramCut, uHistogramCut);
    Free_Image(t6);

    outimage = Convert2Byte_Image(t7);
    Free_Image(t7);

    if (!outimage)
	VIP_Error_Msg("Homomorphic: An error occurred during a function call");

    return (outimage);
}




















