/*******************************************************************************
*
* University of Western Australia
* Department of Computer Science
* Copyright (c) University of Western Australia
*
* SYSTEM :              VIP
* RELEASE:		3
* SUBSYSTEM:            LIB
* MODULE:		vipcomplex.im - Routines to :
 *                              	(1) handle simple mathematical
 *                                      functions on complex numbers.
 *                                      (2) allocate/deallocate complex vectors.		
* REVISION:             3.1
* AUTHOR:               DH
* CREATION DATE:        02 Oct 1991
* REVISION DATE:	7/10/92        
*
********************************************************************************
*
* REVISION LOG
*
* REVISION:		3.1
* REVISION DATE:	9 July 1992
* COMMENT:		ANSIfied and SCCS'd
* BY:			CFF
*
* REVISION:
* REVISION DATE:	08 Oct 1991
* COMMENT:		
* BY:			DH
*
*******************************************************************************/

#ifndef lint
static char *sccs_id = "@(#)vipcomplex.c	3.1 7/10/92";
#endif


#include <stdio.h>
#include <stdlib.h>
#include <sys/file.h>
#include <math.h>
#include <malloc.h>
#include "vip.h"
#include "vipcomplexfn.h"


/*- Complex_Mult ----------------------------------------------------

Multiplying two complex numbers.  On exit, the function returns
	z3 = z1 * z2.
(It is faster to pass the pointers of a structure than copying the
structure itself.)

--------------------------------------------------------------------*/

void    Complex_Mult(z1, z2, z3)
COMPLEX *z1, *z2, *z3;
{
    float   a, b, c, d;

    a = z1->r;
    b = z1->i;
    c = z2->r;
    d = z2->i;

    z3->r = a * c - b * d;
    z3->i = a * d + b * c;
}


/*- Complex_Div -----------------------------------------------------

Dividing a complex number by another complex number.  On exit, the
function returns
	z3 = z1 / z2.

--------------------------------------------------------------------*/

void    Complex_Div(z1, z2, z3)
COMPLEX *z1, *z2, *z3;
{
    float   a, b, c, d, scale;

    a = z1->r;
    b = z1->i;
    c = z2->r;
    d = z2->i;
    scale = c * c + d * d;
    if (scale != 0.0) {
	z3->r = (a * c + b * d) / scale;
	z3->i = (b * c - a * d) / scale;
    }
    else {
	z3->r = 0.0;
	z3->i = 0.0;
    }
}


/*- Complex_Sub -----------------------------------------------------

Subtracting a complex number from another complex number.  On exit, the
function returns
	z3 = z1 - z2.

--------------------------------------------------------------------*/

void    Complex_Sub(z1, z2, z3)
COMPLEX *z1, *z2, *z3;
{
    z3->r = z1->r - z2->r;
    z3->i = z1->i - z2->i;
}


/*- Complex_Add -----------------------------------------------------

Summing two complex numbers.  On exit, the function returns
	z3 = z1 + z2.

--------------------------------------------------------------------*/

void    Complex_Add(z1, z2, z3)
COMPLEX *z1, *z2, *z3;
{
    z3->r = z1->r + z2->r;
    z3->i = z1->i + z2->i;
}


/*- Complex_Magnitude -----------------------------------------------

Return the magnitude of a complex number.
For example, given z = a + bi, the function returns sqrt(a*a + b*b).

--------------------------------------------------------------------*/

float   Complex_Magnitude(z)
COMPLEX *z;
{
    float tmp1 = 0.0, tmp2 = 0.0, tmp3 = 0.0;

    tmp1 = z->r * z->r;
    tmp2 = z->i * z->i;
    tmp3 = (float) sqrt(tmp1 + tmp2);
    return (tmp3);
}


/*- Init_Complex_Vector ---------------------------------------------

Set all components of the complex vector v to zero.
n is the dimension of the vector.

--------------------------------------------------------------------*/

void    Init_Complex_Vector(v, n)
COMPLEX *v;
int     n;
{
    if (!v)
	return;

    bzero(v, sizeof(COMPLEX) * n);
}


/*- Allocate_Complex_Vector -----------------------------------------

Return an allocated space for a complex vector of dimension nc.

--------------------------------------------------------------------*/

COMPLEX *Allocate_Complex_Vector(nc)
int     nc;
{
    COMPLEX *v;

    if (nc < 1)
	return (NULL);

    if (!(v = (COMPLEX *) malloc(nc * sizeof(COMPLEX))))
	return (NULL);

    Init_Complex_Vector(v, nc);
    return (v);
}


/*- Free_Complex_Vector ---------------------------------------------

Deallocate storage for complex vector v.

--------------------------------------------------------------------*/

int     Free_Complex_Vector(v)
COMPLEX *v;
{
    if (!v)
	return (0);

    free((char *) v);

    return (1);
}

