/*
  $Header: tools1.c,v 1.14 93/08/30 11:01:38 hmallat Exp $
 */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include "pixel.h"
#include "block.h"
#include "convert.h"
#include "draw.h"
#include "list.h"
#include "misc.h"
#include "tools1.h"
#include "tools2.h"
#include "undo.h"
#include "X_misc.h"
#include "filter.h"
#include "error.h"

/* specify routines to use in plotting, filling and rubberbanding */
PLOT put_data = replace_pixel1;
PLOT fill_data = replace_pixel1;
PLOT xor_data = xor_pixel1;
PLOT rubber_data = xor_pixel1;

/* routine to put data on screen */
BLOCK block_to_screen = block_to_24bit;

color 
  cur_color = WHITE; /* current color */
const int 
  skip_value = 16;
int 
  ab_max_r = 1, ab_flow = 1, /* airbrush radius and flow */
  b_w = 1, b_h = 1, /* brush width & height */
  cancelled = FALSE,
  counter = 1,
  cur_block_type = POINT_B,
  cur_fill_type = SOLID_F,
  cur_hotspot = CENTER_HS,
  cur_tool = POINT,
  fill_tolerance = 0, 
  filled = FALSE, /* fill objects */
  first_x = -1, first_y = -1, 
  grid = FALSE,
  grid_x = 1, grid_y = 1,
  last = LEFT,
  left = FALSE,
  limit = 1, 
  perimeter = TRUE, /* draw object perimeters */
  prev_x = -1, prev_y = -1, 
  rgb_mode = RGB_REPLACE,
  scope = SCOPE_EVERYWHERE,
  select_tolerance = 0,
  show_edges = TRUE, /* show mask edges */
  snap = FALSE,
  x_off = 0, /* where the brush origin lies measured from upper left corner */
  y_off = 0,
  xx = -1, yy = -1;
pixel 
  copy_bits = MASK_BIT | CMASK;
struct picture 
  *brush = NULL, 
  *cur_block = NULL, 
  *cur_pic = NULL, 
  *fill_pattern = NULL,
  *user1_b = NULL, 
  *user2_b = NULL, 
  *user3_b = NULL;
struct plist 
  *pixel_p = NULL;
struct view 
  *cur_view = NULL,
  *head_p = NULL; /* head of the view list */

/* FUNCTION:       list_pixel(x, y)
   DESCRIPTION:    feed pixels to add_coords at specific intervals
   PARAMETERS:
   x, y            coords
   RETURNS:        Nothing

   Written by Hannu Mallat, last modifications 15th July 1993.
*/   
void list_pixel(int x, int y)
{
  extern int counter, limit;
  if(--counter <limit)
    {
      add_coords(x, y);
      if(counter<1)
	counter += skip_value*limit;
    }
}

/* FUNCTION:       add_coords(int x, int y)
   DESCRIPTION:    add x, y to coord list used for optimizations
   PARAMETERS:
   x, y            coords
   RETURNS:        Nothing

   Written by Hannu Mallat, last modifications 15th July 1993.
*/   
void add_coords(int x, int y)
{
  extern struct plist *pixel_p;
  struct plist *p_p = NULL;
  
  p_p = (struct plist *) malloc(sizeof(struct plist));
  if(p_p != NULL)
    {
      p_p->x_c = x;
      p_p->y_c = y;
      p_p->next = pixel_p;
      pixel_p = p_p;
    }
  else
    panic("Cannot allocate memory! (add_coords)", TRUE);
}

/* FUNCTION:       del_coords()
   DESCRIPTION:    Destroy coord list used in optimizations
   PARAMETERS:
   none
   RETURNS:        Nothing

   Written by Hannu Mallat, last modifications 15th July 1993.
*/   
void del_coords()
{
  extern struct plist *pixel_p;
  struct plist *p_p = pixel_p;
  
  while(pixel_p != NULL)
    {
      p_p = pixel_p->next;
      free(pixel_p);
      pixel_p = p_p;
    }
  pixel_p = NULL;
}

/* FUNCTION:       toolbranch(int x, int y, int button, int kind)
   DESCRIPTION:    Branch to the appropriate drawing tool
   PARAMETERS:
   int x           Mouse x coordinate
   int y           Mouse y coordinate
   int button      Mouse button (NONE, LEFT or CENTER)
   int kind        What sort of event? (PRESS, MOVE_W, MOVE_WO, 
                                        CANCEL)
   RETURNS:        Nothing.

   Written by Hannu Mallat, last modifications 5th August 1993.
*/
void toolbranch(struct view *v_p, int x, int y, int button, int kind)
{
  extern struct picture *cur_block, *fill_pattern;
  extern color cur_color;
  extern int cur_tool, cur_block_type, cur_fill_type, last, fill_tolerance,
  grid_x, grid_y, snap;

  if(v_p != NULL)
    {
      if(snap == TRUE)
	{
	  x = ((x/v_p->scale)/grid_x)*grid_x;
	  y = ((y/v_p->scale)/grid_y)*grid_y;
	}
      else
	{
	  x = x/v_p->scale;
	  y = y/v_p->scale;
	}

      /* if needed, redraw single-colored brush and/or fill with the new
         current color */
      if(button == LEFT && cur_color != v_p->color1)
	{
	  last = LEFT;
	  cur_color = v_p->color1;
	  if(cur_block_type < USER1_B && cur_block_type > POINT_B)
	    create_brush(cur_block_type, cur_block->width);
	  if(cur_fill_type < USER1_F && cur_fill_type > SOLID_F)
	    create_fill(cur_fill_type, fill_pattern->width, fill_tolerance);
	}
      else if(button == CENTER && cur_color != v_p->color2)
	{
	  last = CENTER;
	  cur_color = v_p->color2;
	  if(cur_block_type < USER1_B && cur_block_type > POINT_B)
	    create_brush(cur_block_type, cur_block->width);
	  SWAP(v_p->color1, v_p->color2);
	  if(cur_fill_type < USER1_F && cur_fill_type > SOLID_F)
	    create_fill(cur_fill_type, fill_pattern->width, fill_tolerance);
	  SWAP(v_p->color1, v_p->color2);
	}
      else if(cur_color != v_p->color1 && cur_color != v_p->color2)
	{
	  if(cur_block_type < USER1_B && cur_block_type > POINT_B)
	    {
	      if(last == LEFT)
		{
		  cur_color = v_p->color1;
		  create_brush(cur_block_type, cur_block->width);
		}
	      else
		{
		  cur_color = v_p->color2;
		  create_brush(cur_block_type, cur_block->width);
		}
	    }
	  if(cur_fill_type < USER1_F && cur_fill_type > SOLID_F)
	    if(last == LEFT)
	      {
		cur_color = v_p->color1;
		create_fill(cur_fill_type, fill_pattern->width, 
			    fill_tolerance);
	      }
	    else
	      {
		cur_color = v_p->color2;
		SWAP(v_p->color1, v_p->color2);
		create_fill(cur_fill_type, fill_pattern->width, 
			    fill_tolerance);
		SWAP(v_p->color1, v_p->color2);
	      }
	}
      if(cur_tool <= ELLIPSE)
	draw_brush(v_p, x, y, FALSE);
      cursor_appearance();
      switch(cur_tool)
	{
	case POINT:
	  point(x, y, kind);
	  break;
	case DRAW:
	  draw(x, y, kind);
	  break;
	case AIRBRUSH:
	  airbrush(x, y, kind);
	  break;
	case LINE:
	  rubber_line(x, y, kind, LINE);
	  break;
	case POLYLINE:
	  rubber_line(x, y, kind, POLYLINE);
	  break;
	case RECTANGLE:
	  rubber_rect(x, y, kind);
	  break;
	case CIRCLE:
	  rubber_circle(x, y, kind);
	  break;
	case ELLIPSE:
	  rubber_ellipse(x, y, kind);
	  break;
	case COLOR_EXT:
	  color_ext(x, y, kind, button);
	  break;
	case RECT_S:
	  select_rect(x, y, kind, button);
	  break;
	case ELLIPSE_S:
	  select_ellipse(x, y, kind, button);
	  break;
	case POLY_S:
	  select_poly(x, y, kind, (kind != CANCEL) ? button : last);
	  break;
	case LASSO_S:
	  select_lasso(x, y, kind, (kind != RELEASE) ? button : last);
	  break;
	case COLOR_S:
	  select_n_colors(x, y, kind, button);
	  break;
	case FLOOD_FILL:
	  flood(x, y, kind);
	  break;
	case COLOR_REP:
	default:
	  break;
	}
      if(cur_tool <= ELLIPSE)
	if(kind != LEAVE && kind != CANCEL)
	  draw_brush(v_p, x, y, TRUE);
    }
}

/* FUNCTION:       cursor_appearance()
   DESCRIPTION:    Change the mouse pointer's appearance according to the
                   tool being used
   PARAMETERS:
   none
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 24th August 1993.
*/   
void cursor_appearance()
{
  extern struct view *head_p;
  extern int cur_tool;
  struct view *v_p = head_p;

  while(v_p != NULL)
    {
      switch(cur_tool)
	{
	case POINT:
	case DRAW:
	case LINE:
	case POLYLINE:
	case RECTANGLE:
	case CIRCLE:
	case ELLIPSE:
	  change_cursor(v_p, DRAW_C);
	  break;
	case AIRBRUSH:
	  change_cursor(v_p, AIRBRUSH_C);
	  break;
	case COLOR_EXT:
	  change_cursor(v_p, EXTRACT_C);
	  break;
	case RECT_S:
	case ELLIPSE_S:
	case POLY_S:
	case LASSO_S:
	case COLOR_S:
	  change_cursor(v_p, SELECT_C);
	  break;
	case FLOOD_FILL:
	  change_cursor(v_p, FILL_C);
	  break;
	}
      v_p = v_p->next;
    }
}

/* FUNCTION:       draw_brush(struct view *v_p, int x, int y, int flag)
   DESCRIPTION:    Draw or erase the current brush on the view under the
                   mouse. The background is saved/restored. A sort of
		   'sprite' routine.
   PARAMETERS:
   v_p             View pointer
   x, y            mouse coords
   flag            TRUE = draw, FALSE = erase
   RETURNS:        Nothing

   Written by Hannu Mallat, last modifications 5th August 1993.
*/   
void draw_brush(struct view *v_p, int x, int y, int flag)
{
  extern struct picture *cur_pic, *cur_block, *brush;
  extern struct view *cur_view;
  extern int b_w, b_h, x_off, y_off, xx, yy, rgb_mode, scope;
  struct view *help_v = NULL;
  struct picture *help_p = NULL;

  help_v = cur_view;
  help_p = cur_pic;
  cur_view = v_p;
  cur_pic = v_p->pic_p;

  if(flag == TRUE && v_p != NULL)
    {
      brush = copy_block(cur_pic, x-MIN(x, x_off), y-MIN(y, y_off), 
			 x+b_w-x_off, y+b_h-y_off);
      if(cur_block == NULL)
	{
	  if(rgb_mode == RGB_REPLACE)
	    {
	      if(scope == SCOPE_INSIDE)
		replace_pixel2(x, y);
	      else if(scope == SCOPE_OUTSIDE)
		replace_pixel3(x, y);
	      else
		replace_pixel1(x, y);
	    }
	  else if(rgb_mode == RGB_ADD)
	    {
	      if(scope == SCOPE_INSIDE)
		add_pixel2(x, y);
	      else if(scope == SCOPE_OUTSIDE)
		add_pixel3(x, y);
	      else
		add_pixel1(x, y);
	    }
	  else
	    {
	      if(scope == SCOPE_INSIDE)
		sub_pixel2(x, y);
	      else if(scope == SCOPE_OUTSIDE)
		sub_pixel3(x, y);
	      else
		sub_pixel1(x, y);
	    }
	}
      else
	{
	  if(rgb_mode == RGB_REPLACE)
	    {
	      if(scope == SCOPE_INSIDE)
		replace_block2(x, y);
	      else if(scope == SCOPE_OUTSIDE)
		replace_block3(x, y);
	      else
		replace_block1(x, y);
	    }
	  else if(rgb_mode == RGB_ADD)
	    {
	      if(scope == SCOPE_INSIDE)
		add_block2(x, y);
	      else if(scope == SCOPE_OUTSIDE)
		add_block3(x, y);
	      else
		add_block1(x, y);
	    }
	  else
	    {
	      if(scope == SCOPE_INSIDE)
		sub_block2(x, y);
	      else if(scope == SCOPE_OUTSIDE)
		sub_block3(x, y);
	      else
		sub_block1(x, y);
	    }
	}
      blockade(x-MIN(x, x_off), y-MIN(y, y_off), x+b_w-x_off, y+b_h-y_off, 
	       FALSE);
      xx = x;
      yy = y;
    }
  else
    {
      if(brush != NULL && v_p != NULL)
	{
	  paste_block(brush, xx-MIN(xx, x_off), yy-MIN(yy, y_off));
	  delete_block(brush);
	  brush = NULL;
	  blockade(xx-MIN(xx, x_off), yy-MIN(yy, y_off), 
		   xx+b_w-x_off, yy+b_h-y_off, FALSE);
	}
    }
  cur_view = help_v;
  cur_pic = help_p;
}

/* FUNCTION:       flagbranch(int which)
   DESCRIPTION:    Set respective global flags when a togglebutton is
                   toggled.
   PARAMETERS:     
   which           Which toggle (tools.h)
   RETURNS:        Nothing
   
   Written by Hannu Mallat, last modifications 21st July 1993.
*/   
void flagbranch(int which)
{
  extern struct view *head_p, *cur_view;
  extern struct picture *cur_pic;
  extern int filled, perimeter;
  extern int scope, rgb_mode, show_edges;
  struct view *v_p = head_p, *help_v = cur_view;
  struct picture *help_p = cur_pic;

  switch(which)
    {
      case PERIMETER:
	perimeter = !(perimeter);
	break;
      case FILLED:
	filled = !(filled);
	break;
      case REPLACE:
	rgb_mode = RGB_REPLACE;
	break;
      case ADD:
	rgb_mode = RGB_ADD;
	break;
      case SUB:
	rgb_mode = RGB_SUB;
	break;
      case EVERYWHERE:
	scope = SCOPE_EVERYWHERE;
	break;
      case INSIDE:
	scope = SCOPE_INSIDE;
	break;
      case OUTSIDE:
	scope = SCOPE_OUTSIDE;
	break;
      case SHOW_EDGE:
	while(v_p != NULL)
	  {
	    v_p->pic_p->flags = v_p->pic_p->flags & TOUCHED;
	    v_p = v_p->next;
	  }
	v_p = head_p;
	while(v_p != NULL)
	  {
	    if(v_p->pic_p->mask_edge != NULL && 
	       (v_p->pic_p->flags & SMUDGE) == 0)
	      {
		cur_pic = v_p->pic_p;
		cur_view = v_p;
		plot_mask_edge(xor_data);
		update_edge();
		v_p->pic_p->flags = v_p->pic_p->flags | SMUDGE;
	      }
	    v_p = v_p->next;
	  }
	show_edges = !show_edges;
	cur_view = help_v;
	cur_pic = help_p;
	break;
      default:
	break;
      }
  set_mode();
  update_menu();
  print_tool();
}

/* FUNCTION:       set_mode()
   DESCRIPTION:    Set the pointers for plotting and filling according to
                   the global flags
   PARAMETERS:
   none
   RETURNS:        Nothing

   Written by Hannu Mallat, last modifications 11th August 1993.
*/   
void set_mode()
{
  extern PLOT put_data , fill_data, rubber_data;
  extern int rgb_mode, scope;
  extern struct picture *cur_block, *fill_pattern;

  if(cur_block == NULL)
    {
      if(rgb_mode == RGB_REPLACE)
	if(scope == SCOPE_INSIDE) 
	  {
	    put_data = replace_pixel2;
	    fill_data = (fill_pattern == NULL)?replace_pixel2:replace_pattern2;
	  }
	else if(scope == SCOPE_OUTSIDE) 
	  {
	    put_data = replace_pixel3;
	    fill_data = (fill_pattern == NULL)?replace_pixel3:replace_pattern3;
	  }
	else 
	  {
	    put_data = replace_pixel1;
	    fill_data = (fill_pattern == NULL)?replace_pixel1:replace_pattern1;
	  }
      if(rgb_mode == RGB_ADD)
	if(scope == SCOPE_INSIDE) 
	  {
	    put_data = add_pixel2;
	    fill_data = (fill_pattern == NULL)?add_pixel2:add_pattern2;
	  }
	else if(scope == SCOPE_OUTSIDE) 
	  {
	    put_data = add_pixel3;
	    fill_data = (fill_pattern == NULL)?add_pixel3:add_pattern3;
	  }
	else 
	  {
	    put_data = add_pixel1;
	    fill_data = (fill_pattern == NULL)?add_pixel1:add_pattern1;
	  }
      if(rgb_mode == RGB_SUB)
	if(scope == SCOPE_INSIDE) 
	  {
	    put_data = sub_pixel2;
	    fill_data = (fill_pattern == NULL)?sub_pixel2:sub_pattern2;
	  }
	else if(scope == SCOPE_OUTSIDE) 
	  {
	    put_data = sub_pixel3;
	    fill_data = (fill_pattern == NULL)?sub_pixel3:sub_pattern3;
	  }
	else 
	  {
	    put_data = sub_pixel1;
	    fill_data = (fill_pattern == NULL)?sub_pixel1:sub_pattern1;
	  }
    }
  else
    {
      if(rgb_mode == RGB_REPLACE)
	if(scope == SCOPE_INSIDE) 
	  {
	    put_data = replace_block2;
	    fill_data = (fill_pattern == NULL)?replace_pixel2:replace_pattern2;
	  }
	else if(scope == SCOPE_OUTSIDE) 
	  {
	    put_data = replace_block3;
	    fill_data = (fill_pattern == NULL)?replace_pixel3:replace_pattern3;
	  }
	else 
	  {
	    put_data = replace_block1;
	    fill_data = (fill_pattern == NULL)?replace_pixel1:replace_pattern1;
	  }
      if(rgb_mode == RGB_ADD)
	if(scope == SCOPE_INSIDE) 
	  {
	    put_data = add_block2;
	    fill_data = (fill_pattern == NULL)?add_pixel2:add_pattern2;
	  }
	else if(scope == SCOPE_OUTSIDE) 
	  {
	    put_data = add_block3;
	    fill_data = (fill_pattern == NULL)?add_pixel3:add_pattern3;
	  }
	else 
	  {
	    put_data = add_block1;
	    fill_data = (fill_pattern == NULL)?add_pixel1:add_pattern1;
	  }
      if(rgb_mode == RGB_SUB)
	if(scope == SCOPE_INSIDE) 
	  {
	    put_data = sub_block2;
	    fill_data = (fill_pattern == NULL)?sub_pixel2:sub_pattern2;
	  }
	else if(scope == SCOPE_OUTSIDE) 
	  {
	    put_data = sub_block3;
	    fill_data = (fill_pattern == NULL)?sub_pixel3:sub_pattern3;
	  }
	else 
	  {
	    put_data = sub_block1;
	    fill_data = (fill_pattern == NULL)?sub_pixel1:sub_pattern1;
	  }
    }
  if(scope == SCOPE_INSIDE) rubber_data = xor_pixel2;
  else if (scope == SCOPE_OUTSIDE) rubber_data = xor_pixel3;
  else rubber_data = xor_pixel1;
}

/* FUNCTION:       set_block_to_screen(BLOCK function)
   DESCRIPTION:    set the pointer for the routine which feeds bitmaps
                   to X
   PARAMETERS:
   function
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 22th August 1993.
*/   
void set_block_to_screen(BLOCK function)
{
  extern BLOCK block_to_screen;

  block_to_screen = function;
}

/* FUNCTION:       set_grid(s, g, gx, gy)
   DESCRIPTION:    Set grid & snap values
   PARAMETERS:
   s               TRUE == on
   g               TRUE == on
   gx, gy          grid density
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 31th August 1993.
*/   
void set_grid(int s, int g, int gx, int gy)
{
  extern struct view *head_p, *cur_view;
  extern struct picture *cur_pic;
  extern int grid, grid_x, grid_y, snap;
  struct view *v_p = head_p, *help_p = cur_view;
  struct picture *pic_p = cur_pic;

  all_busy();
  grid = g;
  snap = s;
  grid_x = (gx>0)?gx:1;
  grid_y = (gy>0)?gy:1;

  while(v_p != NULL)
    {
      cur_view = v_p;
      cur_pic = v_p->pic_p;
      blockade(0, 0, v_p->pic_p->width, v_p->pic_p->height, FALSE);
      v_p = v_p->next;
    }
  cur_view = help_p;
  cur_pic = pic_p;
  all_busy_nomore();
}

/* FUNCTION:       set_tool(int tool)
   DESCRIPTION:    set current tool
   PARAMETERS:
   tool            
   RETURNS:        nothing
   
   Written by Hannu Mallat, last modifications 2nd August 1993.
*/   
void set_tool(int tool)
{
  extern int cur_tool;

  cur_tool = tool;
  print_tool();
}

/* FUNCTION:       set_airbrush
   DESCRIPTION:    Set airbrush flow and size
   PARAMETERS:
   size
   flow
   RETURNS:        Nothing

   Written by Hannu Mallat, last modifications 21st July 1993.
*/   
void set_airbrush(int size, int flow)
{
  extern int ab_max_r, ab_flow;

  ab_max_r = size;
  ab_flow = flow;
}

/* FUNCTION:       set_copy_bits(long bits)
   DESCRIPTION:    Set the bits which will be copied by clipboard and
                   duplicate operations
   PARAMETERS:
   bits
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 24th August 1993.
*/   
void set_copy_bits(long bits)
{
  extern pixel copy_bits;

  copy_bits = MASK_BIT | bits;
}

/* FUNCTION:       set_tolerance(int n)
   DESCRIPTION:    Set color range selection tolerance
   PARAMETERS:
   n
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 26th August 1993.
*/   
void set_select_tolerance(int n)
{
  extern int select_tolerance;

  select_tolerance = n;
}

/* FUNCTION:       print_tool()
   DESCRIPTION:    Print description of tool to toolbox
   PARAMETERS:
   none
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 4th August 1993.
*/   
void print_tool()
{
  extern int cur_tool, scope, rgb_mode;
  char s[100] = "";

  switch(cur_tool)
    {
    case POINT:
      strcpy(s, "Tool: Point, ");
      break;
    case DRAW:
      strcpy(s, "Tool: Draw, ");
      break;
    case AIRBRUSH:
      strcpy(s, "Tool: Airbrush, ");
      break;
    case LINE:
      strcpy(s, "Tool: Line, ");
      break;
    case POLYLINE:
      strcpy(s, "Tool: Polyline, ");
      break;
    case RECTANGLE:
      strcpy(s, "Tool: Rectangle (");
      break;
    case CIRCLE:
      strcpy(s, "Tool: Circle (");
      break;
    case ELLIPSE:
      strcpy(s, "Tool: Ellipse (");
      break;
    case COLOR_EXT:
      strcpy(s, "Tool: Color extract ");
      break;
    case COLOR_REP:
      strcpy(s, "Tool: Color rep., ");
      break;
    case RECT_S:
      strcpy(s, "Tool: Select rectangle, ");
      break;
    case ELLIPSE_S:
      strcpy(s, "Tool: Select ellipse, ");
      break;
    case POLY_S:
      strcpy(s, "Tool: Select polygon, ");
      break;
    case LASSO_S:
      strcpy(s, "Tool: Select lasso, ");
      break;
    case COLOR_S:
      strcpy(s, "Tool: Select color range, ");
      break;
    case FLOOD_FILL:
      strcpy(s, "Tool: Flood fill, ");
      break;
    }
  if(cur_tool == RECTANGLE || cur_tool == CIRCLE || cur_tool == ELLIPSE)
    {
      if(perimeter)
	strcat(s, "Perimeter");
      if(filled)
	if(perimeter)
	  strcat(s, ", Fill");
	else
	  strcat(s, "Fill");
      strcat(s, "), ");
    }
  if(cur_tool != COLOR_EXT)
    {
      if(scope == SCOPE_INSIDE)
	strcat(s, "Scope: inside mask, ");
      else if(scope == SCOPE_OUTSIDE)
	strcat(s, "Scope: outside mask, ");
      else
	strcat(s, "Scope: everywhere, ");
      if(rgb_mode == RGB_ADD)
	strcat(s, "Mode: add ");
      else if(rgb_mode == RGB_SUB)
	strcat(s, "Mode: substract ");
      else
	strcat(s, "Mode: replace ");
    }
  show_state(s);
}

/* FUNCTION:       update_menu()
   DESCRIPTION:    activate/deactivate menu items
   PARAMETERS:
   none
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 5th August 1993.
*/   
void update_menu()
{
  extern struct view *cur_view;
  extern struct undo_buffer *ud;
  extern struct picture *clipboard;
  extern int undid, scope;

  if(cur_view == NULL)
    {
      activate_menu_entry("New...");
      activate_menu_entry("Open...");
      deactivate_menu_entry("Save");
      deactivate_menu_entry("Save as...");
      deactivate_menu_entry("Abandon");
      activate_menu_entry("Exit...");
      deactivate_menu_entry("Edit");
      deactivate_menu_entry("Image");
      deactivate_menu_entry("Tool");
      deactivate_menu_entry("Color");
      activate_menu_entry("Options");
      activate_menu_entry("Help");
    }
  else
    {
      activate_menu_entry("Save");
      activate_menu_entry("Save as...");
      activate_menu_entry("Abandon");
      activate_menu_entry("Edit");
      deactivate_menu_entry("Can't undo");
      change_menu_entry("Can't undo", "Can't undo");
      if(ud != NULL)
	if(cur_view->pic_p == ud->pic_p)
	  if(undid == 0)
	    {
	      activate_menu_entry("Can't undo");
	      ud->name_p[0] = 'U';
	      ud->name_p[1] = 'n';
	      change_menu_entry("Can't undo", ud->name_p);
	    }
	  else
	    {
	      activate_menu_entry("Can't undo");
	      ud->name_p[0] = 'R';
	      ud->name_p[1] = 'e';
	      change_menu_entry("Can't undo", ud->name_p);
	    }
      if(cur_view->pic_p->mask_edge != NULL)
	{
	  activate_menu_entry("Cut");
	  activate_menu_entry("Delete");
	  activate_menu_entry("Copy");
	}
      else
	{
	  deactivate_menu_entry("Cut");
	  deactivate_menu_entry("Delete");
	  deactivate_menu_entry("Copy");
	}
      if(clipboard != NULL)
	{
	  activate_menu_entry("Paste");
	  activate_menu_entry("Paste to toolbox...");
	}
      else
	{
	  deactivate_menu_entry("Paste");
	  deactivate_menu_entry("Paste to toolbox...");
	}
      activate_menu_entry("Select all");
      activate_menu_entry("Deselect all");
      activate_menu_entry("Image");
      if(scope == SCOPE_INSIDE && cur_pic->mask_edge == NULL)
	{
	  deactivate_menu_entry("Filter");
	  deactivate_menu_entry("Duplicate");
	}
      else
	{
	  activate_menu_entry("Filter");
	  activate_menu_entry("Duplicate");
	}
      activate_menu_entry("Transform");
      activate_menu_entry("Add view...");
      activate_menu_entry("Zoom...");
      activate_menu_entry("Tool");
      activate_menu_entry("Color");
    }
}

/* FUNCTION:       need_grab()
   DESCRIPTION:    Tell X if we need grabbing of mouse parameters on
                   whole screen or not
   PARAMETERS:

   RETURNS:        NEED_GRAB
                   DONT_NEED_GRAB
		   END_GRAB

   Written by Hannu Mallat, last modifications 15th July 1993.
*/   
int need_grab()
{
  extern int cur_tool, first_x;
  switch(cur_tool)
    {
    case LINE:
    case RECTANGLE:
    case CIRCLE:
    case ELLIPSE:
    case RECT_S: 
    case ELLIPSE_S:
      if(first_x == -1)
	return NEED_GRAB;
      return END_GRAB;
      break;
    case POLYLINE:
/*    case POLY_S:*/ 
      return NEED_GRAB;
      break;
    default:
      return DONT_NEED_GRAB;
      break;
    }
}

/* FUNCTION:       create_brush(int which, int size)
   DESCRIPTION:    Create a new single-coloured brush or copy a multicoloured 
                   one as the current brush.
   PARAMETERS:
   which           type of the brush
   size            size of the brush
   RETURNS:        

   Written by Hannu Mallat, last modifications 22nd July 1993.
*/   
void create_brush(int which, int size)
{
  extern struct picture *cur_block, *clipboard, *user1_b, *user2_b, *user3_b;
  extern int cur_block_type, b_w, b_h, cur_hotspot;
  struct picture *pic_p = cur_pic;
  static int old_size;

  if(size == N_A)
    size = old_size;
  
  if(cur_block != NULL)
    delete_block(cur_block);
  cur_block = NULL;

  switch(which)
    {
    case POINT_B:
      old_size = 1;
      break;
    case DISC_B:
      cur_block = create_block(NULL, size/2*2+1, size/2*2+1);
      if(cur_block != NULL)
	{
	  cur_pic = cur_block;
	  fill_circle(size/2, size/2, size/2, set_pixel_and_mask);
	  old_size = size;
	}
      break;
    case BOX_B:
      cur_block = create_block(NULL, size, size);
      if(cur_block != NULL)
	{
	  cur_pic = cur_block;
	  fill_box(0, 0, size-1, size-1, set_pixel_and_mask);
	  old_size = size;
	}
      break;
    case HLINE_B:
      cur_block = create_block(NULL, size, size);
      if(cur_block != NULL)
	{
	  cur_pic = cur_block;
	  line(0, size/2, size-1, size/2, set_pixel_and_mask);
	  old_size = size;
	}
      break;
    case VLINE_B:
      cur_block = create_block(NULL, size, size);
      if(cur_block != NULL)
	{
	  cur_pic = cur_block;
	  line(size/2, 0, size/2, size-1, set_pixel_and_mask);
	  old_size = size;
	}
      break;
    case LLINE_B:
      cur_block = create_block(NULL, size, size);
      if(cur_block != NULL)
	{
	  cur_pic = cur_block;
	  line(0, 0, size-1, size-1, set_pixel_and_mask);
	  old_size = size;
	}
      break;
    case RLINE_B:
      cur_block = create_block(NULL, size, size);
      if(cur_block != NULL)
	{
	  cur_pic = cur_block;
	  line(0, size-1, size-1, 0, set_pixel_and_mask);
	  old_size = size;
	}
      break;
    case CLIP:
      if(clipboard != NULL)
	cur_block = duplicate_block(clipboard);
      break;
    case USER1_B:
      if(user1_b != NULL)
	cur_block = duplicate_block(user1_b);
      break;
    case USER2_B:
      if(user2_b != NULL)
	cur_block = duplicate_block(user2_b);
      break;
    case USER3_B:
      if(user3_b != NULL)
	cur_block = duplicate_block(user3_b);
      break;
    default:
      break;
    }
  cur_pic = pic_p;
  set_mode();
  if(cur_block == NULL)
    {
      cur_block_type = POINT_B;
      b_w = b_h = 1;
      set_hotspot(cur_hotspot);
    }
  else
    {
      cur_block_type = which;
      b_w = cur_block->width;
      b_h = cur_block->height;
      set_hotspot(cur_hotspot);
    }
}

void set_hotspot(int which)
{
  extern struct picture *cur_block;
  extern int x_off, y_off, cur_hotspot;

  cur_hotspot = which;
  if(cur_block == NULL)
    {
      x_off = 0;
      y_off = 0;
    }
  else
    switch(which)
      {
      case CENTER_HS:
	x_off = cur_block->width/2;
	y_off = cur_block->height/2;
	break;
      case LEFT_HS:
	x_off = 0;
	y_off = cur_block->height/2;
	break;
      case RIGHT_HS:
	x_off = cur_block->width-1;
	y_off = cur_block->height/2;
	break;
      case TOP_HS:
	x_off = cur_block->width/2;
	y_off = 0;
	break;
      case BOTTOM_HS:
	x_off = cur_block->width/2;
	y_off = cur_block->height-1;
	break;
      case TOPLEFT_HS:
	x_off = 0;
	y_off = 0;
	break;
      case TOPRIGHT_HS:
	x_off = cur_block->width-1;
	y_off = 0;
	break;
      case BOTTOMLEFT_HS:
	x_off = 0;
	y_off = cur_block->height-1;
	break;
      case BOTTOMRIGHT_HS:
	x_off = cur_block->width-1;
	y_off = cur_block->height-1;
	break;
      }
}

/* FUNCTION:       char *give_brush(int which, int maxsiz, int *w, int *h)
   DESCRIPTION:    Give the X module a 1-bit representation of a user
                   selected block in order to display it on the dialogue
   PARAMETERS:
   which
   maxsiz          maximum size
   *w              pointer to width where the X mod can read it
   *h              pointer for height
   RETURNS:        the bitmap

   Written by Hannu Mallat, last modifications 22th August 1993.
*/   
char *give_brush(int which, int maxsiz, int *w, int *h)
{
  extern struct picture *user1_b, *user2_b, *user3_b;
  struct picture *pic_p = NULL;
  char *c = NULL;

  switch(which)
    {
    case USER1_B:
      if(user1_b != NULL)
	if(user1_b->width <= maxsiz && user1_b->height <= maxsiz)
	  {
	    c = mask_to_1bit(user1_b);
	    *w = user1_b->width;
	    *h = user1_b->height;
	  }
	else
	  if(user1_b->width > user1_b->height)
	    {
	      pic_p = resize_block(user1_b, maxsiz, 
				   user1_b->height*maxsiz/user1_b->width);
	      if(pic_p != NULL)
		{
		  c = mask_to_1bit(pic_p);
		  *w = pic_p->width;
		  *h = pic_p->height;
		  delete_block(pic_p);
		}
	    }
	  else
	    {
	      pic_p = resize_block(user1_b, 
				   user1_b->width*maxsiz/user1_b->height,
				   maxsiz);
	      if(pic_p != NULL)
		{
		  c = mask_to_1bit(pic_p);
		  *w = pic_p->width;
		  *h = pic_p->height;
		  delete_block(pic_p);
		}
	    }
      break;
    case USER2_B:
      if(user2_b != NULL)
	if(user2_b->width <= maxsiz && user2_b->height <= maxsiz)
	  {
	    c = mask_to_1bit(user2_b);
	    *w = user2_b->width;
	    *h = user2_b->height;
	  }
	else
	  if(user2_b->width > user2_b->height)
	    {
	      pic_p = resize_block(user2_b, maxsiz, 
				   user2_b->height*maxsiz/user2_b->width);
	      if(pic_p != NULL)
		{
		  c = mask_to_1bit(pic_p);
		  *w = pic_p->width;
		  *h = pic_p->height;
		  delete_block(pic_p);
		}
	    }
	  else
	    {
	      pic_p = resize_block(user2_b, 
				   user2_b->width*maxsiz/user2_b->height,
				   maxsiz);
	      if(pic_p != NULL)
		{
		  c = mask_to_1bit(pic_p);
		  *w = pic_p->width;
		  *h = pic_p->height;
		  delete_block(pic_p);
		}
	    }
      break;
    case USER3_B:
      if(user3_b != NULL)
	if(user3_b->width <= maxsiz && user3_b->height <= maxsiz)
	  {
	    c = mask_to_1bit(user3_b);
	    *w = user3_b->width;
	    *h = user3_b->height;
	  }
	else
	  if(user3_b->width > user3_b->height)
	    {
	      pic_p = resize_block(user3_b, maxsiz, 
				   user3_b->height*maxsiz/user3_b->width);
	      if(pic_p != NULL)
		{
		  c = mask_to_1bit(pic_p);
		  *w = pic_p->width;
		  *h = pic_p->height;
		  delete_block(pic_p);
		}
	    }
	  else
	    {
	      pic_p = resize_block(user3_b, 
				   user3_b->width*maxsiz/user3_b->height,
				   maxsiz);
	      if(pic_p != NULL)
		{
		  c = mask_to_1bit(pic_p);
		  *w = pic_p->width;
		  *h = pic_p->height;
		  delete_block(pic_p);
		}
	    }
      break;
    }
  return c;
}

/* FUNCTION:       create_fill(int which, int size, int tolerance)
   DESCRIPTION:    create a fill pattern
   PARAMETERS:
   which
   size
   tolerance       fill tolerance
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 26th August 1993.
*/   
void create_fill(int which, int size, int tolerance)
{
  extern struct picture *fill_pattern, *clipboard, 
  *user1_b, *user2_b, *user3_b;
  extern int cur_fill_type, fill_tolerance;
  struct picture *pic_p = cur_pic, *help_block = cur_block;

  all_busy();
  if(fill_pattern != NULL)
    delete_block(fill_pattern);
  fill_pattern = NULL;
  help_block = cur_block;
  if(size < 2 && which <= LLINE_F)
    which = SOLID_F;

  switch(which)
    {
    case SOLID_F:
      cur_block = NULL;
      break;
    case DISC_F:
      cur_block = create_block(NULL, size/2*2+1, size/2*2+1);
      if(cur_block != NULL)
	{
	  cur_pic = cur_block;
	  fill_circle(size/2, size/2, size/2, set_pixel_and_mask);
	}
      break;
    case BOX_F:
      cur_block = create_block(NULL, size, size);
      if(cur_block != NULL)
	{
	  cur_pic = cur_block;
	  fill_box(0, 0, size/2-1, size/2-1, set_pixel_and_mask);
	  fill_box(size/2, size/2, size-1, size-1, set_pixel_and_mask);
	}
      break;
    case HLINE_F:
      cur_block = create_block(NULL, size, size);
      if(cur_block != NULL)
	{
	  cur_pic = cur_block;
	  line(0, 0, size-1, 0, set_pixel_and_mask);
	}
      break;
    case VLINE_F:
      cur_block = create_block(NULL, size, size);
      if(cur_block != NULL)
	{
	  cur_pic = cur_block;
	  line(0, 0, 0, size-1, set_pixel_and_mask);
	}
      break;
    case LLINE_F:
      cur_block = create_block(NULL, size, size);
      if(cur_block != NULL)
	{
	  cur_pic = cur_block;
	  line(0, 0, size-1, size-1, set_pixel_and_mask);
	}
      break;
    case RLINE_B:
      cur_block = create_block(NULL, size, size);
      if(cur_block != NULL)
	{
	  cur_pic = cur_block;
	  line(0, size-1, size-1, 0, set_pixel_and_mask);
	}
      break;
    case CLIP_F:
      if(clipboard != NULL)
	cur_block = duplicate_block(clipboard);
      break;
    case USER1_F:
      if(user1_b != NULL)
	cur_block = duplicate_block(user1_b);
      break;
    case USER2_F:
      if(user2_b != NULL)
	cur_block = duplicate_block(user2_b);
      break;
    case USER3_F:
      if(user3_b != NULL)
	cur_block = duplicate_block(user3_b);
      break;
    default:
      break;
    }
  cur_pic = pic_p;
  fill_pattern = cur_block;
  cur_block = help_block;
  fill_tolerance = tolerance;
  set_mode();
  if(fill_pattern == NULL)
    cur_fill_type = SOLID_F;
  else
    cur_fill_type = which;
  all_busy_nomore();
}

/* FUNCTION:       void give_color(struct view *v_p, 
                                   int *r_p, int *g_p, int *b_p, int which)
   DESCRIPTION:    Give the red, green and blue components of a color to X
                   module
   PARAMETERS:
   v_p             Pointer to a view
   r_p, g_p, b_p   Pointers to the components
   which           Primary or secondary
   RETURNS:        

   Written by Hannu Mallat, last modifications 22nd July 1993.
*/   
void give_color(struct view *v_p, int *r_p, int *g_p, int *b_p, int which)
{
  if(which == PRIMARY)
    {
      *r_p = GET_RED(v_p->color1);
      *g_p = GET_GREEN(v_p->color1);
      *b_p = GET_BLUE(v_p->color1);
    }
  else if (which == SECONDARY)
    {
      *r_p = GET_RED(v_p->color2);
      *g_p = GET_GREEN(v_p->color2);
      *b_p = GET_BLUE(v_p->color2);
    }
}

/* FUNCTION:       void take_color(struct view *v_p, 
                                   int r, int g, int b, int which)
   DESCRIPTION:    Set primary or secondary colour
   PARAMETERS:
   v_p             Pointer to the view
   r, g, b         Components
   which           prim or sec
   RETURNS:        

   Written by Hannu Mallat, last modifications 22nd July 1993.
*/   
void take_color(struct view *v_p, int r, int g, int b, int which)
{
  if(which == PRIMARY)
    v_p->color1 = PUT_RED(r)|PUT_GREEN(g)|PUT_BLUE(b);
  else if (which == SECONDARY)
    v_p->color2 = PUT_RED(r)|PUT_GREEN(g)|PUT_BLUE(b);
  show_colors(v_p);
}

/* FUNCTION:       feed_block(struct picture *pic_p, int x1, int y1, int x2,
                              int y2, int scale)
   DESCRIPTION:    Get a block through block_to_screen and feed it to X module
   PARAMETERS:
   pic_p          Pointer to the picture
   x1, y1         Top left corner
   x2, y2         Bottom right corner
   scale          Zoom scale
   RETURNS:        

   Written by Hannu Mallat, last modifications 13th July 1993.
*/   
char *feed_block(struct picture *pic_p, int x1, int y1, int x2, int y2, 
		 int scale)
{
  extern BLOCK block_to_screen;
  char *c = NULL;

  if(grid == TRUE)
    draw_grid(pic_p, x1, y1, x2, y2);
  c = block_to_screen(pic_p, x1, y1, x2, y2, scale);
  if(grid == TRUE)
    draw_grid(pic_p, x1, y1, x2, y2);
  return c;
}

/* FUNCTION:       colored_block(int r, g, b, width, height)
   DESCRIPTION:    Return a bitmap of a specified color to X
   PARAMETERS:
   r, g, b         color
   width, height   dimensions
   RETURNS:        pointer to bitmap data        
   
   Written by Hannu Mallat, last modifications 3rd August 1993.
*/   
char *colored_block(int r, int g, int b, int width, int height)
{
  extern BLOCK block_to_screen;
  extern struct picture *cur_pic;
  extern color cur_color;
  color c = cur_color;
  struct picture *pic_p = cur_pic, *pic_p2;

  cur_pic = create_block(NULL, width, height);
  cur_color = PUT_RED(r)|PUT_GREEN(g)|PUT_BLUE(b);
  if(cur_pic != NULL)
    fill_box(0, 0, width-1, height-1, replace_pixel1);
  cur_color = c;
  pic_p2 = cur_pic;
  cur_pic = pic_p;
  return block_to_screen(pic_p2, 0, 0, width-1, height-1, 1);
}

/* FUNCTION:       void blockade(int x1, int y1, int x2, int y2, int all)
   DESCRIPTION:    Update a block of bitmap to X
   PARAMETERS:
   x1, y1, x2, y2  coordinates
   all             Update all views to the image or just the current?
   RETURNS:        

   Written by Hannu Mallat, last modifications 5th August 1993.
*/   
void blockade(int x1, int y1, int x2, int y2, int all)
{
  extern BLOCK block_to_screen;
  extern struct view *cur_view, *head_p;
  extern struct picture *cur_pic;
  extern int grid;
  struct view *v_p = head_p;
  char *data;
  int xx1, xx2, yy1, yy2;

  if (x2<x1) {SWAP (x1, x2);}
  if (y2<y1) {SWAP (y1, y2);}

  if ((x1<0 && x2<0) || (x1>cur_pic->width && x2>cur_pic->width) ||
      (y1<0 && y2<0) || (y1>cur_pic->height && y2>cur_pic->height))
    return;
  if (x1<0) x1 = 0;
  if (y1<0) y1 = 0;
  if (x2>=cur_pic->width) x2 = cur_pic->width-1;
  if (y2>=cur_pic->height) y2 = cur_pic->height-1;

  xx1 = x1; xx2 = x2; yy1 = y1; yy2 = y2;

  if(all == FALSE)
    {
      if ((x1<cur_view->x1 && x2<cur_view->x1) || 
	  (x1>cur_view->x2 && x2>cur_view->x2) ||
	  (y1<cur_view->y1 && y2<cur_view->y1) || 
	  (y1>cur_view->y2 && y2>cur_view->y2))
	return;
      if (x1<cur_view->x1) x1 = cur_view->x1;
      if (y1<cur_view->y1) y1 = cur_view->y1;
      if (x2>cur_view->x2) x2 = cur_view->x2;
      if (y2>cur_view->y2) y2 = cur_view->y2;

      if(grid == TRUE)
	draw_grid(cur_pic, x1, y1, x2, y2);
      data = block_to_screen(cur_pic, x1, y1, x2, y2, cur_view->scale);
      if(grid == TRUE)
	draw_grid(cur_pic, x1, y1, x2, y2);
      if(data != NULL)
	{
	  putimage(x1*cur_view->scale, y1*cur_view->scale, 
		   (x2-x1+1)*cur_view->scale, (y2-y1+1)*cur_view->scale,
		   cur_view, data);
	  free(data);
	}
    }
  else
    while(v_p != NULL)
      {
	if (v_p->pic_p == cur_pic) /* Added by wet to make this make sense */
	  {
	    x1 = xx1; y1 = yy1; x2 = xx2; y2 = yy2;

	    if ((x1<v_p->x1 && x2<v_p->x1) || 
		(x1>v_p->x2 && x2>v_p->x2) ||
		(y1<v_p->y1 && y2<v_p->y1) || 
		(y1>v_p->y2 && y2>v_p->y2))
	      {}
	    else
	      {
		if (x1<v_p->x1) x1 = v_p->x1;
		if (y1<v_p->y1) y1 = v_p->y1;
		if (x2>v_p->x2) x2 = v_p->x2;
		if (y2>v_p->y2) y2 = v_p->y2;
		
		if(grid == TRUE)
		  draw_grid(cur_pic, x1, y1, x2, y2);
		data = block_to_screen(cur_pic, x1, y1, x2, y2, v_p->scale);
		if(grid == TRUE)
		  draw_grid(cur_pic, x1, y1, x2, y2);
		if(data != NULL)
		  {
		    putimage(x1*v_p->scale, y1*v_p->scale, 
			     (x2-x1+1)*v_p->scale, (y2-y1+1)*v_p->scale,
			     v_p, data);
		    free(data);
		  }
		else
		  panic("Unable to update view! (out of memory)", FALSE);
	      }
	  }
	v_p = v_p->next;
      }
}

/* FUNCTION:       invert()
   DESCRIPTION:    Invert an image or a part of it
   PARAMETERS:
   none
   RETURNS:        nothing
   
   Written by Hannu Mallat, last modifications 5th August 1993.
*/   
void invert()
{
  extern PLOT xor_data;
  extern struct picture *cur_pic;
  extern int show_edges, scope;

  all_busy();
  if(show_edges == TRUE) plot_mask_edge(xor_data);
  if(scope == SCOPE_EVERYWHERE)
    {
      memorize(cur_pic, 0, 0,
	       cur_pic->width-1, cur_pic->height-1, NULL, "invert");
      invert_block1(cur_pic);
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      blockade(0, 0, cur_pic->width-1, cur_pic->height-1, TRUE);
    }
  else if(scope == SCOPE_INSIDE && cur_pic->mask_edge != NULL)
    {
      memorize(cur_pic, cur_pic->min_x, cur_pic->min_y,
	       cur_pic->max_x, cur_pic->max_y, NULL, "invert");
      invert_block2(cur_pic);
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      blockade(cur_pic->min_x, cur_pic->min_y, 
	       cur_pic->max_x, cur_pic->max_y, TRUE);
    }
  else if(scope == SCOPE_OUTSIDE)
    {
      memorize(cur_pic, 0, 0,
	       cur_pic->width-1, cur_pic->height-1, NULL, "invert");
      invert_block3(cur_pic);
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      blockade(0, 0, cur_pic->width-1, cur_pic->height-1, TRUE);
    }
  touch(cur_pic);
  all_busy_nomore();
}

/* FUNCTION:       void darken(int p)
   DESCRIPTION:    Darken the image or specified are by a specified percent
   PARAMETERS:
   p
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 26th August 1993.
*/   
void darken(int p)
{
  extern PLOT xor_data;
  extern struct picture *cur_pic;
  extern int show_edges, scope;
  double pros = p/100.0;

  all_busy();
  if(show_edges == TRUE) plot_mask_edge(xor_data);
  if(scope == SCOPE_EVERYWHERE)
    {
      memorize(cur_pic, 0, 0,
	       cur_pic->width-1, cur_pic->height-1, NULL, "darken");
      rgb_adjust_block1(cur_pic, 1-pros, 1-pros, 1-pros, 0, 0, 0);
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      blockade(0, 0, cur_pic->width-1, cur_pic->height-1, TRUE);
    }
  else if(scope == SCOPE_INSIDE && cur_pic->mask_edge != NULL)
    {
      memorize(cur_pic, cur_pic->min_x, cur_pic->min_y,
	       cur_pic->max_x, cur_pic->max_y, NULL, "darken");
      rgb_adjust_block2(cur_pic, 1-pros, 1-pros, 1-pros, 0, 0, 0);
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      blockade(cur_pic->min_x, cur_pic->min_y, 
	       cur_pic->max_x, cur_pic->max_y, TRUE);
    }
  else if(scope == SCOPE_OUTSIDE)
    {
      memorize(cur_pic, 0, 0,
	       cur_pic->width-1, cur_pic->height-1, NULL, "darken");
      rgb_adjust_block3(cur_pic, 1-pros, 1-pros, 1-pros, 0, 0, 0);
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      blockade(0, 0, cur_pic->width-1, cur_pic->height-1, TRUE);
    }
  touch(cur_pic);
  all_busy_nomore();
}

/* FUNCTION:       void brighten(int p)
   DESCRIPTION:    Brighten the image or specified are by a specified percent
   PARAMETERS:
   p
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 26th August 1993.
*/   
void brighten(int p)
{
  extern PLOT xor_data;
  extern struct picture *cur_pic;
  extern int show_edges, scope;
  double pros = p/100.0;

  all_busy();
  if(show_edges == TRUE) plot_mask_edge(xor_data);
  if(scope == SCOPE_EVERYWHERE)
    {
      memorize(cur_pic, 0, 0,
	       cur_pic->width-1, cur_pic->height-1, NULL, "brighten");
      rgb_adjust_block1(cur_pic, 1+pros, 1+pros, 1+pros, 0, 0, 0);
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      blockade(0, 0, cur_pic->width-1, cur_pic->height-1, TRUE);
    }
  else if(scope == SCOPE_INSIDE && cur_pic->mask_edge != NULL)
    {
      memorize(cur_pic, cur_pic->min_x, cur_pic->min_y,
	       cur_pic->max_x, cur_pic->max_y, NULL, "brighten");
      rgb_adjust_block2(cur_pic, 1+pros, 1+pros, 1+pros, 0, 0, 0);
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      blockade(cur_pic->min_x, cur_pic->min_y, 
	       cur_pic->max_x, cur_pic->max_y, TRUE);
    }
  else if(scope == SCOPE_OUTSIDE)
    {
      memorize(cur_pic, 0, 0,
	       cur_pic->width-1, cur_pic->height-1, NULL, "brighten");
      rgb_adjust_block3(cur_pic, 1+pros, 1+pros, 1+pros, 0, 0, 0);
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      blockade(0, 0, cur_pic->width-1, cur_pic->height-1, TRUE);
    }
  touch(cur_pic);
  all_busy_nomore();
}

/* FUNCTION:       void rgb_agjust(double pr, pg, pb, int ir, ig, ib)
   DESCRIPTION:    do rgb value adjustments to image or a part of it
   PARAMETERS:
   pr, pg, pb      percentages with which to multiply the respective comp.
   ir, ig, ib      values to add to respective components
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 26th August 1993.
*/   
void rgb_adjust(double pr, double pg, double pb, int ir, int ig, int ib)
{
  extern PLOT xor_data;
  extern struct picture *cur_pic;
  extern int show_edges, scope;

  all_busy();
  if(show_edges == TRUE) plot_mask_edge(xor_data);
  if(scope == SCOPE_EVERYWHERE)
    {
      memorize(cur_pic, 0, 0,
	       cur_pic->width-1, cur_pic->height-1, NULL, "RGB adjust");
      rgb_adjust_block1(cur_pic, pr, pg, pb, ir, ig, ib);
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      blockade(0, 0, cur_pic->width-1, cur_pic->height-1, TRUE);
    }
  else if(scope == SCOPE_INSIDE && cur_pic->mask_edge != NULL)
    {
      memorize(cur_pic, cur_pic->min_x, cur_pic->min_y,
	       cur_pic->max_x, cur_pic->max_y, NULL, "RGB adjust");
      rgb_adjust_block2(cur_pic, pr, pg, pb, ir, ig, ib);
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      blockade(cur_pic->min_x, cur_pic->min_y, 
	       cur_pic->max_x, cur_pic->max_y, TRUE);
    }
  else if(scope == SCOPE_OUTSIDE)
    {
      memorize(cur_pic, 0, 0,
	       cur_pic->width-1, cur_pic->height-1, NULL, "RGB adjust");
      rgb_adjust_block3(cur_pic, pr, pg, pb, ir, ig, ib);
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      blockade(0, 0, cur_pic->width-1, cur_pic->height-1, TRUE);
    }
  touch(cur_pic);
  all_busy_nomore();
}

/* FUNCTION:       hsv_adjust(double h, s, v)
   DESCRIPTION:    Do HSV-adjust to image or a part of it
   PARAMETERS:
   h, s, v         Values to multiply hue, saturation and value with
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 26th August 1993.
*/   
void hsv_adjust(double h, double s, double v)
{
  extern PLOT xor_data;
  extern struct picture *cur_pic;
  extern int show_edges, scope;

  all_busy();
  if(show_edges == TRUE) plot_mask_edge(xor_data);
  if(scope == SCOPE_EVERYWHERE)
    {
      memorize(cur_pic, 0, 0,
	       cur_pic->width-1, cur_pic->height-1, NULL, "HSV adjust");
      hsv_adjust_block1(cur_pic, h, s, v);
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      blockade(0, 0, cur_pic->width-1, cur_pic->height-1, TRUE);
    }
  else if(scope == SCOPE_INSIDE && cur_pic->mask_edge != NULL)
    {
      memorize(cur_pic, cur_pic->min_x, cur_pic->min_y,
	       cur_pic->max_x, cur_pic->max_y, NULL, "HSV adjust");
      hsv_adjust_block2(cur_pic, h, s, v);
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      blockade(cur_pic->min_x, cur_pic->min_y, 
	       cur_pic->max_x, cur_pic->max_y, TRUE);
    }
  else if(scope == SCOPE_OUTSIDE)
    {
      memorize(cur_pic, 0, 0,
	       cur_pic->width-1, cur_pic->height-1, NULL, "HSV adjust");
      hsv_adjust_block3(cur_pic, h, s, v);
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      blockade(0, 0, cur_pic->width-1, cur_pic->height-1, TRUE);
    }
  touch(cur_pic);
  all_busy_nomore();
}

/* FUNCTION:       matrix(double mul, div, *mat)
   DESCRIPTION:    Do a matrix operation to image or a part of it
   PARAMETERS:
   mul, div        the parts of the fractional to multiply the result with
   *mat            array of matrix cell values
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 26th August 1993.
*/   
void matrix(double mul, double div, double *mat)
{
  extern PLOT xor_data;
  extern struct picture *cur_pic;
  extern int show_edges, scope;

  if(div != 0.0)
    {
      all_busy();
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      if(scope == SCOPE_EVERYWHERE)
	{
	  memorize(cur_pic, 0, 0,
		   cur_pic->width-1, cur_pic->height-1, NULL, 
		   "martix operation");
	  do_5x5_matrix_block1(cur_pic, mul, div, mat);
	  if(show_edges == TRUE) plot_mask_edge(xor_data);
	  blockade(0, 0, cur_pic->width-1, cur_pic->height-1, TRUE);
	}
      else if(scope == SCOPE_INSIDE && cur_pic->mask_edge != NULL)
	{
	  memorize(cur_pic, cur_pic->min_x, cur_pic->min_y,
		   cur_pic->max_x, cur_pic->max_y, NULL, "matrix operation");
	  do_5x5_matrix_block2(cur_pic, mul, div, mat);
	  if(show_edges == TRUE) plot_mask_edge(xor_data);
	  blockade(cur_pic->min_x, cur_pic->min_y, 
		   cur_pic->max_x, cur_pic->max_y, TRUE);
	}
      else if(scope == SCOPE_OUTSIDE)
	{
	  memorize(cur_pic, 0, 0,
		   cur_pic->width-1, cur_pic->height-1, NULL, 
		   "matrix operation");
	  do_5x5_matrix_block3(cur_pic, mul, div, mat);
	  if(show_edges == TRUE) plot_mask_edge(xor_data);
	  blockade(0, 0, cur_pic->width-1, cur_pic->height-1, TRUE);
	}
      touch(cur_pic);
      all_busy_nomore();
    }
  else
    show_errormsg("Divider cannot be zero!", "Bummer!", NULL);
}

/* FUNCTION:       add_noise(int range)
   DESCRIPTION:    Add random noise to image or a part of it
   PARAMETERS:
   range           maximum noise value
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 26th August 1993.
*/   
void add_noise(int range)
{
  extern PLOT xor_data;
  extern struct picture *cur_pic;
  extern int show_edges, scope;

  all_busy();
  if(range != 0)
    {
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      if(scope == SCOPE_EVERYWHERE)
	{
	  memorize(cur_pic, 0, 0,
		   cur_pic->width-1, cur_pic->height-1, NULL, "add noise");
	  add_noise_block1(cur_pic, range);
	  if(show_edges == TRUE) plot_mask_edge(xor_data);
	  blockade(0, 0, cur_pic->width-1, cur_pic->height-1, TRUE);
	}
      else if(scope == SCOPE_INSIDE && cur_pic->mask_edge != NULL)
	{
	  memorize(cur_pic, cur_pic->min_x, cur_pic->min_y,
		   cur_pic->max_x, cur_pic->max_y, NULL, "add noise");
	  add_noise_block2(cur_pic, range);
	  if(show_edges == TRUE) plot_mask_edge(xor_data);
	  blockade(cur_pic->min_x, cur_pic->min_y, 
		   cur_pic->max_x, cur_pic->max_y, TRUE);
	}
      else if(scope == SCOPE_OUTSIDE)
	{
	  memorize(cur_pic, 0, 0,
		   cur_pic->width-1, cur_pic->height-1, NULL, "add noise");
	  add_noise_block3(cur_pic, range);
	  if(show_edges == TRUE) plot_mask_edge(xor_data);
	  blockade(0, 0, cur_pic->width-1, cur_pic->height-1, TRUE);
	}
      touch(cur_pic);
    }
  all_busy_nomore();
}

/* FUNCTION:       vflip()
   DESCRIPTION:    Flip image vertically (up <-> down)
   PARAMETERS:
   none
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 4th August 1993.
*/   
void vflip()
{
  extern PLOT xor_data;
  extern struct picture *cur_pic;
  extern int show_edges;

  all_busy();
  if(show_edges == TRUE) plot_mask_edge(xor_data);
  memorize(cur_pic, 0, 0,
	   cur_pic->width-1, cur_pic->height-1, NULL, "vertical flip");
  vflip_block(cur_pic);
  if(show_edges == TRUE) plot_mask_edge(xor_data);
  blockade(0, 0, cur_pic->width-1, cur_pic->height-1, TRUE);
  touch(cur_pic);
  all_busy_nomore();
}

/* FUNCTION:       hflip()
   DESCRIPTION:    Flip image horizontally
   PARAMETERS:
   none
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 4th August 1993.
*/   
void hflip()
{
  extern PLOT xor_data;
  extern struct picture *cur_pic;
  extern int show_edges;

  all_busy();
  if(show_edges == TRUE) plot_mask_edge(xor_data);
  memorize(cur_pic, 0, 0,
	   cur_pic->width-1, cur_pic->height-1, NULL, "horizontal flip");
  hflip_block(cur_pic);
  if(show_edges == TRUE) plot_mask_edge(xor_data);
  blockade(0, 0, cur_pic->width-1, cur_pic->height-1, TRUE);
  touch(cur_pic);
  all_busy_nomore();
}

/* FUNCTION:       resize(int width, int height)
   DESCRIPTION:    Resize a block to new dimensions
   PARAMETERS:
   width, height
   RETURNS:     nothing   

   Written by Hannu Mallat, last modifications 6th August 1993.
*/   
void resize(int width, int height)
{
  extern PLOT xor_data;
  extern struct view *head_p;
  extern struct picture *cur_pic;
  extern int show_edges;
  struct picture *pic_p, *old_pic = cur_pic;
  struct view *v_p = head_p;

  if(show_edges == TRUE) plot_mask_edge(xor_data);
  pic_p = resize_block(cur_pic, width, height);
  if(pic_p != NULL)
    {
      memorize(cur_pic, 0, 0, cur_pic->width-1, cur_pic->height-1, pic_p, 
	       "resize");
      cur_pic = pic_p;
      
      minimize_mask_edge();
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      while(v_p != NULL)
	{
	  if(v_p->pic_p == old_pic)
	    {
	      v_p->pic_p = cur_pic;
	      handle_resize(v_p);
	    }
	  v_p = v_p->next;
	}
      delete_block(old_pic);
      touch(cur_pic);
    }
  else
    {
      all_busy_nomore();
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      panic("Not enough memory for resize!", FALSE);
    }
}

/* FUNCTION:       rotate(double angle, int cw)
   DESCRIPTION:    Rotate a block
   PARAMETERS:
   angle
   cw              TRUE = clockwise, FALSE = counterclockwise
   RETURNS:        nothing   

   Written by Hannu Mallat, last modifications 10th August 1993.
*/   
void rotate(double angle, int cw)
{
  extern PLOT xor_data;
  extern struct view *head_p;
  extern struct picture *cur_pic;
  extern int show_edges;
  struct picture *pic_p, *old_pic = cur_pic;
  struct view *v_p = head_p;

  if(show_edges == TRUE) plot_mask_edge(xor_data);
  pic_p = rotate_block(cur_pic, (cw == FALSE) ? angle/180.0*PI : 
		       (360.0-angle)/180.0*PI);
  if(pic_p != NULL)
    {
      memorize(cur_pic, 0, 0, cur_pic->width-1, cur_pic->height-1, pic_p,
	       "rotate");
      cur_pic = pic_p;
      
      minimize_mask_edge();
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      while(v_p != NULL)
	{
	  if(v_p->pic_p == old_pic)
	    {
	      v_p->pic_p = cur_pic;
	      handle_resize(v_p);
	    }
	  v_p = v_p->next;
	}
      delete_block(old_pic);
      touch(cur_pic);
    }
  else
    {
      all_busy_nomore();
      if(show_edges == TRUE) plot_mask_edge(xor_data);
      panic("Not enough memory for rotate!", FALSE);
    }
}

/* FUNCTION:       update_edge()
   DESCRIPTION:    Draw mask edge on screen
   PARAMETERS:

   RETURNS:        

   Written by Hannu Mallat, last modifications 2nd August 1993.
*/   
void update_edge()
{
  blockade(cur_pic->min_x, cur_pic->min_y, 
	   cur_pic->max_x, cur_pic->max_y, TRUE);
}

/* FUNCTION:       deselect_all()
   DESCRIPTION:    Remove mask & edges
   PARAMETERS:

   RETURNS:        

   Written by Hannu Mallat, last modifications 2nd August 1993.
*/   
void deselect_all()
{
  extern PLOT xor_data;
  extern int show_edges;

  all_busy();
  if(show_edges == TRUE) plot_mask_edge(xor_data); update_edge();
  memorize(cur_pic, cur_pic->min_x, cur_pic->min_y,
	   cur_pic->max_x, cur_pic->max_y, NULL, "deselect all");
  if(cur_pic->mask_edge != NULL)
    {
      fill_box(cur_pic->min_x, cur_pic->min_y,
	       cur_pic->max_x, cur_pic->max_y, clear_mask);
      delete_mask_edge();
    }
  all_busy_nomore();
  update_menu();
}

/* FUNCTION:       select_all()
   DESCRIPTION:    Select all of an image
   PARAMETERS:

   RETURNS:        

   Written by Hannu Mallat, last modifications 3rd August 1993.
*/   
void select_all()
{
  extern PLOT xor_data;
  extern int show_edges;

  all_busy();
  if(show_edges == TRUE) plot_mask_edge(xor_data);
  memorize(cur_pic, 0, 0,
	   cur_pic->width-1, cur_pic->height-1, NULL, "select all");
  fill_box(0, 0, cur_pic->width-1, cur_pic->height-1, set_mask);
  cur_pic->min_x = 0;
  cur_pic->min_y = 0;
  cur_pic->max_x = cur_pic->width-1;
  cur_pic->max_y = cur_pic->height-1;
  
  minimize_mask_edge();
  if(show_edges == TRUE) {plot_mask_edge(xor_data);update_edge();}
  all_busy_nomore();
  update_menu();
}

/* FUNCTION:       all_busy()
   DESCRIPTION:    Display hourglass pointer in toolbox and every window
   PARAMETERS:
   none
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 23th August 1993.
*/   
void all_busy()
{
  extern struct view *head_p;
  struct view *v_p = head_p;

  while(v_p != NULL)
    {
      view_busy(v_p);
      v_p = v_p->next;
    }
  busy();
}

/* FUNCTION:       all_busy_nomore()
   DESCRIPTION:    Remove hourglass
   PARAMETERS:
   none
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 23th August 1993.
*/   
void all_busy_nomore()
{
  extern struct view *head_p;
  struct view *v_p = head_p;

  while(v_p != NULL)
    {
      view_busy_nomore(v_p);
      v_p = v_p->next;
    }
  busy_nomore();
}

/* FUNCTION:       draw_grid(*pic_p, x1, y1, x2, y2)
   DESCRIPTION:    Draw a grid of horizontal and vertical lines to a specified
                   part of an image if grid == TRUE. If there should be a grid
		   line on every row or column, don't draw.
   PARAMETERS:
   *pic_p          picture
   x1, y1, x2, y2  rectangle dimensions
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 30th August 1993.
*/   
void draw_grid(struct picture *pic_p, int x1, int y1, int x2, int y2)
{
  extern struct picture *cur_pic;
  extern PLOT xor_data;
  extern color cur_color;
  extern int grid_x, grid_y;
  struct picture *help_p = cur_pic;
  int x, y;
  color help_c = cur_color;

  cur_pic = pic_p;
  cur_color = WHITE;
  if(grid_x > 1)
    {
      x = (x1 % grid_x == 0)?x1:(x1/grid_x+1)*grid_x;
      while(x <= x2)
	{
	  vline(x, y1, y2, xor_data);
	  x += grid_x;
	}
    }
  if(grid_y > 1)
    {
      y = (y1 % grid_y == 0)?y1:(y1/grid_y+1)*grid_y;
      while(y <= y2)
	{
	  hline(x1, x2, y, xor_data);
	  y += grid_y;
	}
    }
  cur_pic = help_p;
  cur_color = help_c;
}
