/* FILE:   pixel.c
 
   Routines for drawing blocks and pixels are here.

   Written by Petri Kuittinen, last modifications 2nd August 1993.
*/

#include <stdlib.h>
#include "misc.h"
#include "pixel.h"
#include "draw.h"

/* is point (X,Y) in screen? */
#define IN_SCREEN(X,Y) ((X)>=0 && (Y)>=0 && (X)<cur_pic->width && \
			(Y)<cur_pic->height)
/* calcute position in pixel array */
#define CALC_P(X,Y) (cur_pic->buf+(Y)*cur_pic->width+(X)) 


/* FUNCTION:     void replace_pixel# (int x, int y)
   DESCRIPTION:  write pixel to screen in location (x, y), replace the
   old value. Uses global variables cur_color and cur_pic (must be
   initialized before calling this).
   #=1 replace everywhere (no mask check)
   #=2 when mask is set
   #=3 when mask is not set
   PARAMETERS:
   int x         x-coordinate [0, width-1]
   int y         y-coordinate [0, height-1]
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 28th June 1993
*/
void replace_pixel1(int x , int y)
{
  if (IN_SCREEN(x, y))
    {
      register pixel *p;
      p = CALC_P(x, y);
      *p = GET_UP_BITS(*p)|cur_color;
    }
}

void replace_pixel2(int x, int y)
{
  register pixel *p;
  p = CALC_P(x, y);

  if (IN_SCREEN(x, y) && IS_MASK(*p))
    *p = GET_UP_BITS(*p)|cur_color;
}

void replace_pixel3(int x, int y)
{
  register pixel *p;
  p = CALC_P(x, y);

  if (IN_SCREEN(x, y) && !(IS_MASK(*p)))
    *p = GET_UP_BITS(*p)|cur_color;
}


/* FUNCTION:     void add_pixel# (int x, int y)
   DESCRIPTION:  write pixel to screen in location (x, y), addition
   operation is perfomed with each color component (adds old value to new
   value). Uses global variables cur_pic and cur_color (must be
   initialized before calling this).
   #=1 do everywhere (no mask check)
   #=2 when mask is set
   #=3 when mask is not set
   PARAMETERS:
   int x         x-coordinate [0, width-1]
   int y         y-coordinate [0, height-1]
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 28th June 1993
*/
void add_pixel1(int x , int y)
{
  if (IN_SCREEN(x, y))
    {
      register pixel *p;
      register color r, g, b;
      p = CALC_P(x, y);
      r = GET_RED(*p)+GET_RED(cur_color); LIMIT_255(r);
      g = GET_GREEN(*p)+GET_GREEN(cur_color); LIMIT_255(g);
      b = GET_BLUE(*p)+GET_BLUE(cur_color); LIMIT_255(b);
      *p = GET_UP_BITS(*p)|PUT_RED(r)|PUT_GREEN(g)|PUT_BLUE(b);
    }
}

void add_pixel2(int x , int y)
{
  register pixel *p;
  p = CALC_P(x, y);
  if (IN_SCREEN(x, y) && IS_MASK(*p))
    {
      register color r, g, b;
      r = GET_RED(*p)+GET_RED(cur_color); LIMIT_255(r);
      g = GET_GREEN(*p)+GET_GREEN(cur_color); LIMIT_255(g);
      b = GET_BLUE(*p)+GET_BLUE(cur_color); LIMIT_255(b);
      *p = GET_UP_BITS(*p)|PUT_RED(r)|PUT_GREEN(g)|PUT_BLUE(b);
    }
}

void add_pixel3(int x , int y)
{
  register pixel *p;
  p = CALC_P(x, y);

  if (IN_SCREEN(x, y) && !(IS_MASK(*p)))
    {
      register color r, g, b;
      r = GET_RED(*p)+GET_RED(cur_color); LIMIT_255(r);
      g = GET_GREEN(*p)+GET_GREEN(cur_color); LIMIT_255(g);
      b = GET_BLUE(*p)+GET_BLUE(cur_color); LIMIT_255(b);
      *p = GET_UP_BITS(*p)|PUT_RED(r)|PUT_GREEN(g)|PUT_BLUE(b);
    }
}

/* FUNCTION:     void sub_pixel# (int x, int y)
   DESCRIPTION:  write pixel to screen in location (x, y), substraction
   operation is perfomed with each color component (substracts new value
   from old value). Uses global variables cur_pic and cur_color (must be
   initialized before calling this).
   #=1 do everywhere (no mask check)
   #=2 when mask is set
   #=3 when mask is not set
   PARAMETERS:
   int x         x-coordinate [0, width-1]
   int y         y-coordinate [0, height-1]
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 28th June 1993
*/
void sub_pixel1(int x , int y)
{
  if (IN_SCREEN(x, y))
    {
      register pixel *p;
      register color r, g, b;
      p = CALC_P(x, y);
      r = GET_RED(*p)-GET_RED(cur_color); LIMIT_0(r);
      g = GET_GREEN(*p)-GET_GREEN(cur_color); LIMIT_0(g);
      b = GET_BLUE(*p)-GET_BLUE(cur_color); LIMIT_0(b);
      *p = GET_UP_BITS(*p)|PUT_RED(r)|PUT_GREEN(g)|PUT_BLUE(b);
    }
}

void sub_pixel2(int x , int y)
{
  register pixel *p;
  p = CALC_P(x, y);
  if (IN_SCREEN(x, y) && IS_MASK(*p))
    {
      register color r, g, b;
      r = GET_RED(*p)-GET_RED(cur_color); LIMIT_0(r);
      g = GET_GREEN(*p)-GET_GREEN(cur_color); LIMIT_0(g);
      b = GET_BLUE(*p)-GET_BLUE(cur_color); LIMIT_0(b);
      *p = GET_UP_BITS(*p)|PUT_RED(r)|PUT_GREEN(g)|PUT_BLUE(b);
    }
}

void sub_pixel3(int x , int y)
{
  register pixel *p;
  p = CALC_P(x, y);

  if (IN_SCREEN(x, y) && !(IS_MASK(*p)))
    {
      register color r, g, b;
      r = GET_RED(*p)-GET_RED(cur_color); LIMIT_0(r);
      g = GET_GREEN(*p)-GET_GREEN(cur_color); LIMIT_0(g);
      b = GET_BLUE(*p)-GET_BLUE(cur_color); LIMIT_0(b);
      *p = GET_UP_BITS(*p)|PUT_RED(r)|PUT_GREEN(g)|PUT_BLUE(b);
    }
}

/* FUNCTION:     void xor_pixel# (int x, int y)
   DESCRIPTION:  write pixel to screen in location (x, y), do logical not
   operation to old value. Uses global variable cur_pic (must be initialized
   before calling this).
   #=1 replace everywhere (no mask check)
   #=2 when mask is set
   #=3 when mask is not set
   PARAMETERS:
   int x         x-coordinate [0, width-1]
   int y         y-coordinate [0, height-1]
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 28th June 1993
*/
void xor_pixel1(int x , int y)
{
  if (IN_SCREEN(x, y))
    {
      register pixel *p;
      p = CALC_P(x, y);
      *p = *p^CMASK;
    }
}

void xor_pixel2(int x, int y)
{
  register pixel *p;
  p = CALC_P(x, y);

  if (IN_SCREEN(x, y) && IS_MASK(*p))
    *p = *p^CMASK;
}

void xor_pixel3(int x, int y)
{
  register pixel *p;
  p = CALC_P(x, y);

  if (IN_SCREEN(x, y) && !(IS_MASK(*p)))
    *p = *p^CMASK;
}


/* FUNCTION:     void set_mask (int x, int y)
   DESCRIPTION:  set mask bit in location (x, y).
   PARAMETERS:
   int x         x-coordinate [0, width-1]
   int y         y-coordinate [0, height-1]
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 14th July 1993
*/
void set_mask (int x, int y)
{
  if (IN_SCREEN(x, y))
    {
      register pixel *p;
      p = CALC_P(x, y);
      *p |= MASK_BIT;
    }
}


/* FUNCTION:     void clear_mask (int x, int y)
   DESCRIPTION:  clear mask bit in location (x, y).
   PARAMETERS:
   int x         x-coordinate [0, width-1]
   int y         y-coordinate [0, height-1]
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 14th July 1993
*/
void clear_mask (int x, int y)
{
  if (IN_SCREEN(x, y))
    {
      register pixel *p;
      p = CALC_P(x, y);
      *p &= NO_MASK;
    }
}


/* FUNCTION:     int get_mask (int x, int y)
   DESCRIPTION:  get mask bit in location (x, y).
   PARAMETERS:
   int x         x-coordinate [0, width-1]
   int y         y-coordinate [0, height-1]
   RETURNS:      1 if mask bit is set in that location, otherwise zero.
   Returns -1 if out of screen.
   
   Written by Petri Kuittinen, last modifications 14th July 1993
*/
int get_mask (int x, int y)
{
  if (IN_SCREEN(x, y))
    {
      register pixel *p;
      p = CALC_P(x, y);
      return (GET_UP_BITS(*p)!=0);
    }
  else return -1;
}


/* FUNCTION:     void set_fill_bit (int x, int y)
   DESCRIPTION:  set fill bit in location (x, y).
   PARAMETERS:
   int x         x-coordinate [0, width-1]
   int y         y-coordinate [0, height-1]
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 20th August 1993
*/
void set_fill_bit (int x, int y)
{
  if (IN_SCREEN(x, y))
    {
      register pixel *p;
      p = CALC_P(x, y);
      *p |= FB;
    }
}


/* FUNCTION:     void clear_fill_bit (int x, int y)
   DESCRIPTION:  clear fill bit in location (x, y).
   PARAMETERS:
   int x         x-coordinate [0, width-1]
   int y         y-coordinate [0, height-1]
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 20th August 1993
*/
void clear_fill_bit (int x, int y)
{
  if (IN_SCREEN(x, y))
    {
      register pixel *p;
      p = CALC_P(x, y);
      *p &= NO_FB;
    }
}


/* FUNCTION:     int get_fill_bit (int x, int y)
   DESCRIPTION:  get fill bit in location (x, y).
   PARAMETERS:
   int x         x-coordinate [0, width-1]
   int y         y-coordinate [0, height-1]
   RETURNS:      1 if fill bit is set in that location, otherwise zero.
   Returns -1 if out of screen.
   
   Written by Petri Kuittinen, last modifications 20th August 1993
*/
int get_fill_bit (int x, int y)
{
  if (IN_SCREEN(x, y))
    {
      register pixel *p;
      p = CALC_P(x, y);
      return (GET_FB(*p));
    }
  else return -1;
}


/* FUNCTION:     color get_color (int x, int y)
   DESCRIPTION:  get color value in location (x, y).
   PARAMETERS:
   int x         x-coordinate [0, width-1]
   int y         y-coordinate [0, height-1]
   RETURNS:      24-bit color value in that location, otherwise zero.
   Returns -1 if out of screen.
   
   Written by Petri Kuittinen, last modifications 14th July 1993
*/
color get_color (int x, int y)
{
  if (IN_SCREEN(x,y))
    {
      register pixel *p;
      p = CALC_P(x, y);
      return (*p&CMASK);
    }
  else return -1;
}


/* FUNCTION:     void set_pixel_and_mask (int x, int y)
   DESCRIPTION:  write pixel and set mask bit in location (x, y).
   PARAMETERS:
   int x         x-coordinate [0, width-1]
   int y         y-coordinate [0, height-1]
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 14th July 1993
*/
void set_pixel_and_mask (int x, int y)
{
  if (IN_SCREEN(x,y))
    {
      register pixel *p;
      p = CALC_P(x, y);
      *p = MASK_BIT|cur_color;
    }
}


/* FUNCTION:     void replace_block# (int x, int y)
   DESCRIPTION:  write block to screen in location (xc, yc), overwrites the
   area below the block. Uses global variable cur_block and cur_pic (must
   initialized before calling this).
   #=1 replace everywhere (no mask check)
   #=2 when mask is set
   #=3 when mask is not set
   PARAMETERS:
   int x         center x-coordinate 
   int y         center y-coordinate [0, height-1]
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 20th July 1993
*/
void replace_block1(int x, int y)
{
  register int w, h, xc, yc, x_offset;
  register pixel *src, *tar;

  src = cur_block->buf;
  w = cur_block->width;
  h = cur_block->height;
  xc = x-x_off;
  yc = y-y_off;
  tar= CALC_P (xc, yc);
  x_offset = cur_pic->width-cur_block->width;

  while (h--)
    {
      while (w--)
	{
	  if (IS_MASK(*src) && IN_SCREEN(xc, yc))
	    {
	      *tar = GET_UP_BITS(*tar)|((*src)&CMASK);
	    }
	  tar++; src++; xc++;
	}
      w = cur_block->width; tar += x_offset; xc -= w; yc++;
    }
}


void replace_block2(int x, int y)
{
  register int w, h, xc, yc, x_offset;
  register pixel *src, *tar;

  src = cur_block->buf;
  w = cur_block->width;
  h = cur_block->height;
  xc = x-x_off;
  yc = y-y_off;
  tar= CALC_P (xc, yc);
  x_offset = cur_pic->width-cur_block->width;

  while (h--)
    {
      while (w--)
	{
	  if (IS_MASK(*src) && IN_SCREEN(xc, yc) && IS_MASK(*tar))
	    {
	      *tar = GET_UP_BITS(*tar)|((*src)&CMASK);
	    }
	  tar++; src++; xc++;
	}
      w = cur_block->width; tar += x_offset; xc -= w; yc++;
    }
}


void replace_block3(int x, int y)
{
  register int w, h, xc, yc, x_offset;
  register pixel *src, *tar;

  src = cur_block->buf;
  w = cur_block->width;
  h = cur_block->height;
  xc = x-x_off;
  yc = y-y_off;
  tar= CALC_P (xc, yc);
  x_offset = cur_pic->width-cur_block->width;

  while (h--)
    {
      while (w--)
	{
	  if (IS_MASK(*src) && IN_SCREEN(xc, yc) && !(IS_MASK(*tar)))
	    {
	      *tar = GET_UP_BITS(*tar)|((*src)&CMASK);
	    }
	  tar++; src++; xc++;
	}
      w = cur_block->width; tar += x_offset; xc -= w; yc++;
    }
}


/* FUNCTION:     void add_block# (int x, int y)
   DESCRIPTION:  write block to screen in location (xc, yc), performs
   and addition operation with below the block. Uses global variable
   cur_block and cur_pic (must initialized before calling this).
   #=1 do everywhere (no mask check)
   #=2 when mask is set
   #=3 when mask is not set
   PARAMETERS:
   int x         center x-coordinate 
   int y         center y-coordinate [0, height-1]
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 20th July 1993
*/
void add_block1(int x, int y)
{
  register int w, h, xc, yc, x_offset;
  register color r, g, b;
  register pixel *src, *tar;

  src = cur_block->buf;
  w = cur_block->width;
  h = cur_block->height;
  xc = x-x_off;
  yc = y-y_off;
  tar= CALC_P (xc, yc);
  x_offset = cur_pic->width-cur_block->width;

  while (h--)
    {
      while (w--)
	{
	  if (IS_MASK(*src) && IN_SCREEN(xc, yc))
	    {
	      r = GET_RED(*src)+GET_RED(*tar); LIMIT_255(r);
	      g = GET_GREEN(*src)+GET_GREEN(*tar); LIMIT_255(g);
	      b = GET_BLUE(*src)+GET_BLUE(*tar); LIMIT_255(b);
	      *tar = GET_UP_BITS(*tar)|PUT_RED(r)|PUT_GREEN(g)|PUT_BLUE(b);
	    }
	  tar++; src++; xc++;
	}
      w = cur_block->width; tar += x_offset; xc -= w; yc++;
    }
}


void add_block2(int x, int y)
{
  register int w, h, xc, yc, x_offset;
  register color r, g, b;
  register pixel *src, *tar;

  src = cur_block->buf;
  w = cur_block->width;
  h = cur_block->height;
  xc = x-x_off;
  yc = y-y_off;
  tar= CALC_P (xc, yc);
  x_offset = cur_pic->width-cur_block->width;

  while (h--)
    {
      while (w--)
	{
	  if (IS_MASK(*src) && IN_SCREEN(xc, yc) && IS_MASK(*tar))
	    {
	      r = GET_RED(*src)+GET_RED(*tar); LIMIT_255(r);
	      g = GET_GREEN(*src)+GET_GREEN(*tar); LIMIT_255(g);
	      b = GET_BLUE(*src)+GET_BLUE(*tar); LIMIT_255(b);
	      *tar = GET_UP_BITS(*tar)|PUT_RED(r)|PUT_GREEN(g)|PUT_BLUE(b);
	    }
	  tar++; src++; xc++;
	}
      w = cur_block->width; tar += x_offset; xc -= w; yc++;
    }
}


void add_block3(int x, int y)
{
  register int w, h, xc, yc, x_offset;
  register color r, g, b;
  register pixel *src, *tar;

  src = cur_block->buf;
  w = cur_block->width;
  h = cur_block->height;
  xc = x-x_off;
  yc = y-y_off;
  tar= CALC_P (xc, yc);
  x_offset = cur_pic->width-cur_block->width;

  while (h--)
    {
      while (w--)
	{
	  if (IS_MASK(*src) && IN_SCREEN(xc, yc) && !(IS_MASK(*tar)))
	    {
	      r = GET_RED(*src)+GET_RED(*tar); LIMIT_255(r);
	      g = GET_GREEN(*src)+GET_GREEN(*tar); LIMIT_255(g);
	      b = GET_BLUE(*src)+GET_BLUE(*tar); LIMIT_255(b);
	      *tar = GET_UP_BITS(*tar)|PUT_RED(r)|PUT_GREEN(g)|PUT_BLUE(b);
	    }
	  tar++; src++; xc++;
	}
      w = cur_block->width; tar += x_offset; xc -= w; yc++;
    }
}


/* FUNCTION:     void sub_block# (int x, int y)
   DESCRIPTION:  write block to screen in location (xc, yc), performs
   and substraction operation with below the block. Uses global variable
   cur_block and cur_pic (must initialized before calling this).
   #=1 do everywhere (no mask check)
   #=2 when mask is set
   #=3 when mask is not set
   PARAMETERS:
   int x         center x-coordinate 
   int y         center y-coordinate [0, height-1]
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 20th July 1993
*/
void sub_block1(int x, int y)
{
  register int w, h, xc, yc, x_offset;
  register color r, g, b;
  register pixel *src, *tar;

  src = cur_block->buf;
  w = cur_block->width;
  h = cur_block->height;
  xc = x-x_off;
  yc = y-y_off;
  tar= CALC_P (xc, yc);
  x_offset = cur_pic->width-cur_block->width;

  while (h--)
    {
      while (w--)
	{
	  if (IS_MASK(*src) && IN_SCREEN(xc, yc))
	    {
	      r = GET_RED(*tar)-GET_RED(*src); LIMIT_0(r);
	      g = GET_GREEN(*tar)-GET_GREEN(*src); LIMIT_0(g);
	      b = GET_BLUE(*tar)-GET_BLUE(*src); LIMIT_0(b);
	      *tar = GET_UP_BITS(*tar)|PUT_RED(r)|PUT_GREEN(g)|PUT_BLUE(b);
	    }
	  tar++; src++; xc++;
	}
      w = cur_block->width; tar += x_offset; xc -= w; yc++;
    }
}


void sub_block2(int x, int y)
{
  register int w, h, xc, yc, x_offset;
  register color r, g, b;
  register pixel *src, *tar;

  src = cur_block->buf;
  w = cur_block->width;
  h = cur_block->height;
  xc = x-x_off;
  yc = y-y_off;
  tar= CALC_P (xc, yc);
  x_offset = cur_pic->width-cur_block->width;

  while (h--)
    {
      while (w--)
	{
	  if (IS_MASK(*src) && IN_SCREEN(xc, yc) && IS_MASK(*tar))
	    {
	      r = GET_RED(*tar)-GET_RED(*src); LIMIT_0(r);
	      g = GET_GREEN(*tar)-GET_GREEN(*src); LIMIT_0(g);
	      b = GET_BLUE(*tar)-GET_BLUE(*src); LIMIT_0(b);
	      *tar = GET_UP_BITS(*tar)|PUT_RED(r)|PUT_GREEN(g)|PUT_BLUE(b);
	    }
	  tar++; src++; xc++;
	}
      w = cur_block->width; tar += x_offset; xc -= w; yc++;
    }
}


void sub_block3(int x, int y)
{
  register int w, h, xc, yc, x_offset;
  register color r, g, b;
  register pixel *src, *tar;

  src = cur_block->buf;
  w = cur_block->width;
  h = cur_block->height;
  xc = x-x_off;
  yc = y-y_off;
  tar= CALC_P (xc, yc);
  x_offset = cur_pic->width-cur_block->width;

  while (h--)
    {
      while (w--)
	{
	  if (IS_MASK(*src) && IN_SCREEN(xc, yc) && !(IS_MASK(*tar)))
	    {
	      r = GET_RED(*tar)-GET_RED(*src); LIMIT_0(r);
	      g = GET_GREEN(*tar)-GET_GREEN(*src); LIMIT_0(g);
	      b = GET_BLUE(*tar)-GET_BLUE(*src); LIMIT_0(b);
	      *tar = GET_UP_BITS(*tar)|PUT_RED(r)|PUT_GREEN(g)|PUT_BLUE(b);
	    }
	  tar++; src++; xc++;
	}
      w = cur_block->width; tar += x_offset; xc -= w; yc++;
    }
}


/* FUNCTION:     void xor_block# (int x, int y)
   DESCRIPTION:  invert a block in location (xc, yc). Uses global variable
   cur_block and cur_pic (must initialized before calling this).
   #=1 do everywhere (no mask check)
   #=2 when mask is set
   #=3 when mask is not set
   PARAMETERS:
   int x         center x-coordinate 
   int y         center y-coordinate [0, height-1]
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 20th July 1993
*/
void xor_block1(int x, int y)
{
  register int w, h, xc, yc, x_offset;
  register pixel *src, *tar;

  src = cur_block->buf;
  w = cur_block->width;
  h = cur_block->height;
  xc = x-x_off;
  yc = y-y_off;
  tar= CALC_P (xc, yc);
  x_offset = cur_pic->width-cur_block->width;

  while (h--)
    {
      while (w--)
	{
	  if (IS_MASK(*src) && IN_SCREEN(xc, yc))
	    {
	      *tar = (*tar)&CMASK;
	    }
	  tar++; src++; xc++;
	}
      w = cur_block->width; tar += x_offset; xc -= w; yc++;
    }
}


void xor_block2(int x, int y)
{
  register int w, h, xc, yc, x_offset;
  register pixel *src, *tar;

  src = cur_block->buf;
  w = cur_block->width;
  h = cur_block->height;
  xc = x-x_off;
  yc = y-y_off;
  tar= CALC_P (xc, yc);
  x_offset = cur_pic->width-cur_block->width;

  while (h--)
    {
      while (w--)
	{
	  if (IS_MASK(*src) && IN_SCREEN(xc, yc) && IS_MASK(*tar))
	    {
	      *tar = (*tar)&CMASK;
	    }
	  tar++; src++; xc++;
	}
      w = cur_block->width; tar += x_offset; xc -= w; yc++;
    }
}


void xor_block3(int x, int y)
{
  register int w, h, xc, yc, x_offset;
  register pixel *src, *tar;

  src = cur_block->buf;
  w = cur_block->width;
  h = cur_block->height;
  xc = x-x_off;
  yc = y-y_off;
  tar= CALC_P (xc, yc);
  x_offset = cur_pic->width-cur_block->width;

  while (h--)
    {
      while (w--)
	{
	  if (IS_MASK(*src) && IN_SCREEN(xc, yc) && !(IS_MASK(*tar)))
	    {
	      *tar = (*tar)&CMASK;
	    }
	  tar++; src++; xc++;
	}
      w = cur_block->width; tar += x_offset; xc -= w; yc++;
    }
}


/* FUNCTION:     void replace_pattern# (int x, int y)
   DESCRIPTION:  write pattern pixel to screen in location (x, y), replace
   the old value. The pixel to be written (taken from cur_block) is defined
   using modulo arithemetic with screen coordinates. Uses global variables
   cur_block and cur_pic (must be initialized before calling this). 
   #=1 replace everywhere (no mask check)
   #=2 when mask is set
   #=3 when mask is not set
   PARAMETERS:
   int x         x-coordinate [0, width-1]
   int y         y-coordinate [0, height-1]
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 3rd August 1993
*/
void replace_pattern1(int x , int y)
{
  if (IN_SCREEN(x, y))
    {
      register pixel *src, *tar;
      src = cur_block->buf+((x+x_off)%cur_block->width)+
	cur_block->width*((y+y_off)%cur_block->height);
      tar = CALC_P(x, y);
      if (IS_MASK(*src))
	*tar = GET_UP_BITS(*tar)|(*src&CMASK);
    }
}

void replace_pattern2(int x , int y)
{
  register pixel *src, *tar;
  tar = CALC_P(x, y);
  if (IN_SCREEN(x, y) && IS_MASK(*tar))
    {
      src = cur_block->buf+((x+x_off)%cur_block->width)+
	cur_block->width*((y+y_off)%cur_block->height);

      if (IS_MASK(*src))
	*tar = GET_UP_BITS(*tar)|(*src&CMASK);
    }
}

void replace_pattern3(int x , int y)
{
  register pixel *src, *tar;
  tar = CALC_P(x, y);
  if (IN_SCREEN(x, y) && !(IS_MASK(*tar)))
    {
      src = cur_block->buf+((x+x_off)%cur_block->width)+
	cur_block->width*((y+y_off)%cur_block->height);

      if (IS_MASK(*src))
	*tar = GET_UP_BITS(*tar)|(*src&CMASK);
    }
}


/* FUNCTION:     void add_pattern# (int x, int y)
   DESCRIPTION:  write pattern pixel to screen in location (x, y), perform
   an addition operation (with each color component) with the old value.
   The pixel to be written (taken from cur_block) is defined
   using modulo arithemetic with screen coordinates. Uses global variables
   cur_block and cur_pic (must be initialized before calling this). 
   #=1 replace everywhere (no mask check)
   #=2 when mask is set
   #=3 when mask is not set
   PARAMETERS:
   int x         x-coordinate [0, width-1]
   int y         y-coordinate [0, height-1]
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 3rd August 1993
*/
void add_pattern1(int x , int y)
{
  if (IN_SCREEN(x, y))
    {
      register pixel *src, *tar, r, g, b;

      src = cur_block->buf+((x+x_off)%cur_block->width)+
	cur_block->width*((y+y_off)%cur_block->height);
      tar = CALC_P(x, y);
      if (IS_MASK(*src))
	{
	  r = GET_RED(*tar)+GET_RED(*src); LIMIT_255(r);
	  g = GET_GREEN(*tar)+GET_GREEN(*src); LIMIT_255(g);
	  b = GET_BLUE(*tar)+GET_BLUE(*src); LIMIT_255(b);
	  *tar = GET_UP_BITS(*tar)|PUT_RED(r)|PUT_GREEN(g)|PUT_BLUE(b);
	}
    }
}

void add_pattern2(int x , int y)
{
  register pixel *src, *tar, r, g, b;

  tar = CALC_P(x, y);
  if (IN_SCREEN(x, y) && IS_MASK(*tar))
    {
      src = cur_block->buf+((x+x_off)%cur_block->width)+
	cur_block->width*((y+y_off)%cur_block->height);

      if (IS_MASK(*src))
	{
	  r = GET_RED(*tar)+GET_RED(*src); LIMIT_255(r);
	  g = GET_GREEN(*tar)+GET_GREEN(*src); LIMIT_255(g);
	  b = GET_BLUE(*tar)+GET_BLUE(*src); LIMIT_255(b);
	  *tar = GET_UP_BITS(*tar)|PUT_RED(r)|PUT_GREEN(g)|PUT_BLUE(b);
	}
    }
}

void add_pattern3(int x , int y)
{
  register pixel *src, *tar, r, g, b;
  
  tar = CALC_P(x, y);
  if (IN_SCREEN(x, y) && !(IS_MASK(*tar)))
    {
      src = cur_block->buf+((x+x_off)%cur_block->width)+
	cur_block->width*((y+y_off)%cur_block->height);

      if (IS_MASK(*src))
	{
	  r = GET_RED(*tar)+GET_RED(*src); LIMIT_255(r);
	  g = GET_GREEN(*tar)+GET_GREEN(*src); LIMIT_255(g);
	  b = GET_BLUE(*tar)+GET_BLUE(*src); LIMIT_255(b);
	  *tar = GET_UP_BITS(*tar)|PUT_RED(r)|PUT_GREEN(g)|PUT_BLUE(b);
	}
    }
}


/* FUNCTION:     void sub_pattern# (int x, int y)
   DESCRIPTION:  write pattern pixel to screen in location (x, y), perform
   an substraction operation (with each color component) with the old value.
   The pixel to be written (taken from cur_block) is defined
   using modulo arithemetic with screen coordinates. Uses global variables
   cur_block and cur_pic (must be initialized before calling this). 
   #=1 replace everywhere (no mask check)
   #=2 when mask is set
   #=3 when mask is not set
   PARAMETERS:
   int x         x-coordinate [0, width-1]
   int y         y-coordinate [0, height-1]
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 3rd August 1993
*/
void sub_pattern1(int x , int y)
{
  if (IN_SCREEN(x, y))
    {
      register pixel *src, *tar, r, g, b;

      src = cur_block->buf+((x+x_off)%cur_block->width)+
	cur_block->width*((y+y_off)%cur_block->height);
      tar = CALC_P(x, y);
      if (IS_MASK(*src))
	{
	  r = GET_RED(*tar)-GET_RED(*src); LIMIT_0(r);
	  g = GET_GREEN(*tar)-GET_GREEN(*src); LIMIT_0(g);
	  b = GET_BLUE(*tar)-GET_BLUE(*src); LIMIT_0(b);
	  *tar = GET_UP_BITS(*tar)|PUT_RED(r)|PUT_GREEN(g)|PUT_BLUE(b);
	}
    }
}

void sub_pattern2(int x , int y)
{
  register pixel *src, *tar, r, g, b;

  tar = CALC_P(x, y);
  if (IN_SCREEN(x, y) && IS_MASK(*tar))
    {
      src = cur_block->buf+((x+x_off)%cur_block->width)+
	cur_block->width*((y+y_off)%cur_block->height);

      if (IS_MASK(*src))
	{
	  r = GET_RED(*tar)-GET_RED(*src); LIMIT_0(r);
	  g = GET_GREEN(*tar)-GET_GREEN(*src); LIMIT_0(g);
	  b = GET_BLUE(*tar)-GET_BLUE(*src); LIMIT_0(b);
	  *tar = GET_UP_BITS(*tar)|PUT_RED(r)|PUT_GREEN(g)|PUT_BLUE(b);
	}
    }
}

void sub_pattern3(int x , int y)
{
  register pixel *src, *tar, r, g, b;
  
  tar = CALC_P(x, y);
  if (IN_SCREEN(x, y) && !(IS_MASK(*tar)))
    {
      src = cur_block->buf+((x+x_off)%cur_block->width)+
	cur_block->width*((y+y_off)%cur_block->height);

      if (IS_MASK(*src))
	{
	  r = GET_RED(*tar)-GET_RED(*src); LIMIT_0(r);
	  g = GET_GREEN(*tar)-GET_GREEN(*src); LIMIT_0(g);
	  b = GET_BLUE(*tar)-GET_BLUE(*src); LIMIT_0(b);
	  *tar = GET_UP_BITS(*tar)|PUT_RED(r)|PUT_GREEN(g)|PUT_BLUE(b);
	}
    }
}


/* FUNCTION:     void xor_pattern# (int x, int y)
   DESCRIPTION:  write pattern pixel to screen in location (x, y), inverts
   the old value. The pixel to be written (taken from cur_block) is defined
   using modulo arithemetic with screen coordinates. Uses global variables
   cur_block and cur_pic (must be initialized before calling this). 
   #=1 replace everywhere (no mask check)
   #=2 when mask is set
   #=3 when mask is not set
   PARAMETERS:
   int x         x-coordinate [0, width-1]
   int y         y-coordinate [0, height-1]
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 3rd August 1993
*/
void xor_pattern1(int x , int y)
{
  if (IN_SCREEN(x, y))
    {
      register pixel *src, *tar;
      src = cur_block->buf+((x+x_off)%cur_block->width)+
	cur_block->width*((y+y_off)%cur_block->height);
      tar = CALC_P(x, y);
      if (IS_MASK(*src))
	*tar ^= CMASK;
    }
}

void xor_pattern2(int x , int y)
{
  register pixel *src, *tar;
  tar = CALC_P(x, y);
  if (IN_SCREEN(x, y) && IS_MASK(*tar))
    {
      src = cur_block->buf+((x+x_off)%cur_block->width)+
	cur_block->width*((y+y_off)%cur_block->height);

      if (IS_MASK(*src))
	*tar ^= CMASK;
    }
}

void xor_pattern3(int x , int y)
{
  register pixel *src, *tar;
  tar = CALC_P(x, y);
  if (IN_SCREEN(x, y) && !(IS_MASK(*tar)))
    {
      src = cur_block->buf+((x+x_off)%cur_block->width)+
	cur_block->width*((y+y_off)%cur_block->height);

      if (IS_MASK(*src))
	*tar ^= CMASK;
    }
}


/* FUNCTION:     void paste_block (struct picture *pic_p, int x, int y)
   DESCRIPTION:  paste a rectangle shaped aread to the picture in
   location (x,y).
   PARAMETERS:
   struct picture *pic_p  pointer to a block to be pasted struct
   int x         upper left hand edge coordinates where to paste block
   int y
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 5th August 1993
*/
void paste_block (struct picture *pic_p, int x, int y)
{
  register int w, h, x_offset;
  register pixel *src, *tar;

  src = pic_p->buf;
  w = pic_p->width;
  h = pic_p->height;
  tar= CALC_P (x, y);
  x_offset = cur_pic->width-pic_p->width;

  while (h--)
    {
      while (w--)
        {
          if (IN_SCREEN(x, y))
            {
              *tar = GET_UP_BITS(*tar)|((*src)&CMASK);
            }
          tar++; src++; x++;
        }
      w = pic_p->width; tar += x_offset; x -= w; y++;
    }
}


/* FUNCTION:     void add_mask_edge (int x, int y)
   DESCRIPTION:  add point (x, y) mask edge if possible.
   PARAMETERS:
   int x
   int y
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 20th August 1993
*/
void add_mask_edge (int x, int y)
{
  register int r, l, i;
  register struct coord *p;

  if (!(IN_SCREEN(x, y))) return;

  l = 0 ;
  r = cur_pic->n_edges-1;
  p = cur_pic->mask_edge;
  
  if (cur_pic->n_edges==0)
    {
      i = 0;
      cur_pic->min_x = cur_pic->max_x = x;
      cur_pic->min_y = cur_pic->max_y = y;
    }
  else
    {
      /* Search the right y-coord position using binary search */
      do
	{
	  i = (l+r)/2;
	  if (y<p[i].y) r = i-1; else l = i+1;
	}
      while (y!=p[i].y && l<=r);
 
      if (y>p[i].y && i<cur_pic->n_edges)
        i++;
      else if (y==p[i].y)
	{	  
	  /* Then search for right x-coord position */
	  if (x<p[i].x)
	    {
	      while (i>0 && y==p[i-1].y && x<=p[i-1].x)
		i--;
	    }
	  else if (x>p[i].x)
	    {
	      do
		i++;
	      while (i<cur_pic->n_edges && y==p[i].y && x>p[i].x);
	    }
	  if (i<cur_pic->n_edges && y==p[i].y && x==p[i].x)
	    return; /* Point already in egde list */
	}
    }
  
  /* Allocate more space every 4096th mask edge point */
  if (((cur_pic->n_edges)&4095)==0)
    {
      cur_pic->mask_edge =
	realloc (cur_pic->mask_edge,
		 (cur_pic->n_edges+4096)*sizeof(struct coord));
      if (cur_pic->mask_edge==NULL)
	{
	  cur_pic->n_edges = 0;
	  return;
	}
    }
  
  /* move mask edge data list one entry forward (space for new edge point) */
  p = cur_pic->mask_edge+cur_pic->n_edges-1;
  r = cur_pic->n_edges-i;
  
  while (r--)
    {
      *(p+1) = *p;
      p--;
    }

  /* insert mask edge point */
  cur_pic->mask_edge[i].x = x;
  cur_pic->mask_edge[i].y = y;
  cur_pic->n_edges ++;

  if (x<cur_pic->min_x) cur_pic->min_x = x;
  if (x>cur_pic->max_x) cur_pic->max_x = x;
  if (y<cur_pic->min_y) cur_pic->min_y = y;
  if (y>cur_pic->max_y) cur_pic->max_y = y;
}


/* FUNCTION:     void delete_mask_edge()
   DESCRIPTION:  Delete mask edge.
   PARAMETERS:
   none
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 14th August 1993
*/
void delete_mask_edge()
{
  if (cur_pic->n_edges)
    {
      free (cur_pic->mask_edge);
      cur_pic->mask_edge = NULL;
      cur_pic->n_edges = 0;
    }
}


/* FUNCTION:     void plot_mask_edge (void (*put_pixel) (int x, int y))
   DESCRIPTION:  Plot all points in mask edge.
   PARAMETERS:
   void (*put_pixel) (int x, int y) pointer to the pixel plottin routine
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 14th August 1993
*/
void plot_mask_edge (void (*put_pixel) (int x, int y))
{
  register int i;
  register struct coord *p;

  i = cur_pic->n_edges;
  p = cur_pic->mask_edge;

  while (i--)
    {
      put_pixel (p->x, p->y);
      p++;
    }
}

/* FUNCTION:     void minimize_mask_edge()
   DESCRIPTION:  First this routine deletes the old mask edge, then it
   creates the minimal (=optimal) mask edge from current picture using
   an algorithm by the idiot himself. The routine only goes through
   pixels from rectangle (min_x, min_y)-(max_x, max_y) so cur_pic->min_?
   and cur_pic->max_? must be set before calling this routine!
   PARAMETERS:
   none
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 14th August 1993
*/
void minimize_mask_edge()
{
  register int x, y, x1, x2, y2, x_offset;
  register pixel *p;

  delete_mask_edge();
  
  x1 = cur_pic->min_x;
  x2 = cur_pic->max_x;
  y2 = cur_pic->max_y;

  x_offset = cur_pic->width-(x2-x1+1);
  p = cur_pic->buf+x1+cur_pic->width*cur_pic->min_y;

  /* This algorithm is actually pretty simple. Pixels which are masked,
     but have neighbouring pixel(s) which are not masked or are in the edge
     of screen form the mask edge. */
  for (y = cur_pic->min_y; y<=y2; y++)
    {
      for (x = x1; x<=x2; x++)
        {
	  if (IS_MASK(*p) &&
	      (x<1 || x>=cur_pic->width-1 || y<1 || y>=cur_pic->height-1 ||
	       !(IS_MASK(*(p-1))) ||
	       !(IS_MASK(*(p+1))) ||
	       !(IS_MASK(*(p-cur_pic->width))) || 
	       !(IS_MASK(*(p+cur_pic->width)))))
	    add_mask_edge(x, y);
          p++;
        }
      p += x_offset;
    }
}


#define STACK_SIZE 10000 /* This should be enough for the even most
			    complicated fills. */
/* Macros used in flood fill routine */
#define PUSH(Y, XL, XR, DY)     /* push new segment on stack */ \
  if (sp<stack+STACK_SIZE && (Y)+(DY)>=0 && (Y)+(DY)<cur_pic->height) \
{sp->y = (Y); sp->xl = (XL); sp->xr = (XR); sp->dy = (DY); sp++;}

#define POP(Y, XL, XR, DY)      /* pop segment off stack */ \
{sp--; (Y) = sp->y+((DY) = sp->dy); (XL) = sp->xl; (XR) = sp->xr;}

/* FUNCTION:     int fill_this(int x, int y, int val, int range)
   DESCRIPTION:  A routine which determines if we need to fill this pixel
   or not.
   PARAMETRES:
   int x         coordinated of the pixel 
   int y
   int val       color value.
   int range     tolarated error range of color component versus val.
   RETURNS:      1 if we need to fill, otherwise zero.
   
   Written by Petri Kuittinen, last modification 20th August 1993
*/
int fill_this(int x, int y, int val, int range)
{
  register pixel *p, data, r, g, b;
  
  p = CALC_P(x, y); data = *p;
  r = GET_RED(data)-GET_RED(val); r = ABS(r);
  g = GET_GREEN(data)-GET_GREEN(val); g = ABS(g);
  b = GET_BLUE(data)-GET_BLUE(val); b = ABS(b);
  
  return (!(IS_FB(data)) && r<=range && g<=range && b<=range);
}


/* FUNCTION:     void flood_fill (int xc, int yc,
   void (*put_pixel) (int x, int y))
   DESCRIPTION:  Flood fill an area using a seed fill algorithm found in
   Graphics Gems I.
   PARAMETRES:
   int xc        start-up coordinates of filling 
   int yc
   int range     tolarated error range of color component versus original
   starting point pixel value (range=0 in 'normal' fill).
   void (*put_pixel) (x, y) pointer to the pixel drawing routine
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modification 20th August 1993
*/
void flood_fill (int x, int y, int range, void (*put_pixel) (int x, int y))
{
  register int l, x1, x2, dy;
  register pixel *p;
  register color old_val;
  struct segment stack[STACK_SIZE], *sp; /* stack of filled segments */

  sp = stack;
  old_val = get_color (x, y);
  
  if (old_val==cur_color || x<0 || x>=cur_pic->width || y<0 ||
      y>=cur_pic->height)
    return;

  PUSH (y, x, x, 1);    /* needed in some cases */
  PUSH (y+1, x, x, -1); /* seed segment (popped 1st) */

  while (sp>stack)
    {
      /* pop segment off stack and fill a neighboring scan line */
      POP(y, x1, x2, dy);

      /* segment of scan line y-dy for x1<=x<=x2 was previously filled,
	 now explore adjacent pixels in scan line y. */

      for (x=x1; x>=0 && fill_this(x, y, old_val, range); x--)
	{
	  put_pixel (x, y);
	  set_fill_bit (x, y);
	}
	  
      if (x>=x1) goto skip;
      l = x+1;
      if (l<x1) PUSH(y, l, x1-1, -dy); /* leak on left? */
      x = x1+1;
      do
	{
	  for (; x<cur_pic->width && fill_this(x, y, old_val, range); x++)
	    {
	      put_pixel (x, y);
	      set_fill_bit (x, y);
	    }
	  PUSH(y, l, x-1, dy);
	  if (x>x2+1) PUSH(y, x2+1, x-1, -dy); /* leak on right? */
	skip:       
	  for (x++; x<=x2 && !(fill_this(x, y, old_val, range)); x++);
	  l = x;
	}
      while (x<=x2);
    }

  /* clear all fill bits in picture */
  l = cur_pic->width*cur_pic->height;
  p = cur_pic->buf;
  while (l--)
    *p++ &= NO_FB;
}


/* FUNCTION:     void fill_by_color (color col, int range,
   void (*put_pixel) (int x, int y))
   DESCRIPTION:  Fill all pixels in the picture which are near enough
   given color.
   PARAMETRES:
   color col     given color
   int range     tolarated error range of color component versus col.
   void (*put_pixel) (int x, int y) pointer to the actual pixel plotting
   routine.
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modification 20th August 1993
*/
void fill_by_color (color col, int range, void (*put_pixel) (int x, int y))
{
  register int x, y;
  register pixel *p, data, r, g, b;
  
  p = cur_pic->buf;

  for (y=0; y<cur_pic->height; y++)
    for (x=0; x<cur_pic->width; x++)
      {
	data = *p++;
	r = GET_RED(data)-GET_RED(col); r = ABS(r);
	g = GET_GREEN(data)-GET_GREEN(col); g = ABS(g);
	b = GET_BLUE(data)-GET_BLUE(col); b = ABS(b);
	if (r<=range && g<=range && b<=range)
	  put_pixel(x, y);
      }
}
