 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for minvert
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lminvert
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lminvert - compute inverse of matrix
* 
*       Purpose: lminvert computes the inverse of matrix. If diagonal only is
*		 requested, compute 1/A(i,i); else matrix must be
*                square and LU decomposition with backsolve is used  to
*		 obtain the matrix inverse.
*
*         Input: kobject in_obj - input matrix object
*                int diag - request inversion of diagonal only if non-zero
*
*        Output: kobject out_obj - output matrix object for inverse
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Scott Wilson 13 Aug 1994
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lminvert(
kobject in_obj,
int diag,
kobject out_obj)
/* -library_def_end */

/* -library_code */
{
        char *lib = "kmatrix", *rtn = "lminvert";
        double *A=NULL,*L=NULL,*WORK=NULL;
        kdcomplex *AZ=NULL,*LZ=NULL,*WORKZ=NULL,dcone;
        int i,j,num,nplanes,*ipiv,info;
        int type, otype;
        int wc,hc,dc,tc,ec;
        int pi_len,work_len;
        klist *objlist=NULL;

        /* Make sure we have valid objects */
        if (in_obj == KOBJECT_INVALID)
          {
            kerror(lib, rtn, "Bogus input object");
            return(FALSE);
          }
        if (out_obj == KOBJECT_INVALID)
          {
            kerror(lib, rtn, "Bogus ouput object");
            return(FALSE);
          }

        /* Reference the input object to avoid side effects, then add the
           reference to the list of goodies to be autmatically free'd on
           error. */
        KCALL((in_obj = kpds_reference_object(in_obj)));
        objlist = klist_add(objlist,in_obj,"KOBJECT");

        /* See if the data type of the input object is real or complex */
        KCALL(kpds_get_attribute(in_obj,KPDS_VALUE_DATA_TYPE,&type));
        if (type == KCOMPLEX || type == KDCOMPLEX) otype = KDCOMPLEX;
        else otype = KDOUBLE;

        /* Constrain the input data to double real or double complex for read */
        KCALL(kpds_set_attribute(in_obj,KPDS_VALUE_DATA_TYPE, otype));

        /* See how big the data is */
        KCALL(kpds_get_attribute(in_obj,KPDS_VALUE_SIZE,&wc,&hc,&dc,&tc,&ec));
        /* Set up to process input by planes */
        KCALL(kpds_set_attribute(in_obj,KPDS_VALUE_REGION_SIZE,wc,hc,1,1,1));
        nplanes = dc*tc*ec;
        pi_len = kmin(wc,hc);
        work_len = hc*8;

        if (!diag)
          {
            if (hc != wc)
              {
                kinfo(KSTANDARD,"lminvert: Matrix must be square\n");
                (void)klist_free(objlist, (kfunc_void)lkcall_free);
                return(0);
              }
          }

        if (!kpds_query_value(out_obj)) KCALL(kpds_create_value(out_obj));

        /* Set the output object size and data type */
        KCALL(kpds_set_attributes(out_obj,
                             KPDS_VALUE_SIZE,wc,hc,dc,tc,ec,
                             KPDS_VALUE_DATA_TYPE,otype,NULL));
        KCALL((out_obj = kpds_reference_object(out_obj)));
        objlist = klist_add(objlist,out_obj,"KOBJECT");
        KCALL(kpds_set_attribute(out_obj,KPDS_VALUE_REGION_SIZE,
                                            wc,hc,1,1,1));

        /* Allocate plane buffers */
        if (otype == KDOUBLE)
          {
            KCALL(!((A=(double *)kmalloc(wc*hc*sizeof(double))) == NULL));
            objlist = klist_add(objlist,A,"KMALLOC");
            if (!diag)
              {
                KCALL(!((L=(double *)kmalloc(wc*hc*sizeof(double))) == NULL));
                objlist = klist_add(objlist,L,"KMALLOC");
                KCALL(!((WORK=(double *)kmalloc(work_len*sizeof(double))) == NULL));
                objlist = klist_add(objlist,WORK,"KMALLOC");
              }
          }
        else
          {
            KCALL(!((AZ=(kdcomplex *)kmalloc(wc*hc*sizeof(kdcomplex)))== NULL));
            objlist = klist_add(objlist,AZ,"KMALLOC");
            if (!diag)
              {
                KCALL(!((LZ=(kdcomplex *)kmalloc(wc*hc*sizeof(kdcomplex)))== NULL));
                objlist = klist_add(objlist,LZ,"KMALLOC");
                KCALL(!((WORKZ=(kdcomplex *)kmalloc(work_len*sizeof(kdcomplex))) == NULL));
                objlist = klist_add(objlist,WORKZ,"KMALLOC");
            }
          }

        /* If doing only diag, get it done and bail out */
        if (diag)
          {
            dcone.r = 1.0; dcone.i = 0.0;
            for (num=0; num<nplanes; num++)
              {
                if (otype == KDOUBLE)
                  {
                    A=(double *)kpds_get_data(in_obj,KPDS_VALUE_REGION,(kaddr)A);
                    for (i=0; i<kmin(wc,hc); i++) A[i*wc+i] = 1.0/A[i*wc+i];
                    KCALL(kpds_put_data(out_obj,KPDS_VALUE_REGION,(kaddr)A));
                  }
                else
                  {
                    AZ=(kdcomplex *)kpds_get_data(in_obj,KPDS_VALUE_REGION,(kaddr)AZ);
                    for (i=0; i<kmin(wc,hc); i++)
                      AZ[i*wc+i] = kdcdiv(dcone,AZ[i*wc+i]);
                    KCALL(kpds_put_data(out_obj,KPDS_VALUE_REGION,(kaddr)AZ));
                  }
              }
            (void)klist_free(objlist, (kfunc_void)lkcall_free);
	    return TRUE;
          }

        /* Allocate IPVT array */
        if (!diag)
          {
            KCALL(!((ipiv=(int *)kmalloc(pi_len*sizeof(int))) == NULL));
            objlist = klist_add(objlist,ipiv,"KMALLOC");
          }

        for (num=0; num<nplanes; num++)
          {
            /* Get a plane from the input object */
            if (otype == KDOUBLE)
              A = (double *)kpds_get_data(in_obj,KPDS_VALUE_REGION,(kaddr)A);
            else
              AZ=(kdcomplex *)kpds_get_data(in_obj,KPDS_VALUE_REGION,(kaddr)AZ);

            if (otype == KDOUBLE) kbzero(L,wc*hc*sizeof(double));
            else kbzero(LZ,wc*hc*sizeof(kdcomplex));

            kbzero(ipiv,pi_len*sizeof(int));

            /* Transpose so that F77 will like it */
            if (otype == KDOUBLE)
              {
                for (i=0; i<hc; i++)
                  for (j=0; j<wc; j++)
                    L[j*hc+i] = A[i*wc+j];
              }
            else
              {
                for (i=0; i<hc; i++)
                  for (j=0; j<wc; j++)
                    LZ[j*hc+i] = AZ[i*wc+j];
              }

            /* Obtain LU factorization */
            if (otype == KDOUBLE) DGETRF(&hc,&wc,L,&hc,ipiv,&info);
            else ZGETRF(&hc,&wc,LZ,&hc,ipiv,&info);

            if (info != 0)
              {
                kinfo(KSTANDARD,"lminvert: LAPACK {dz}getrf() returned with INFO=%d \n\
where INFO < 0 -> for INFO = -i, the i'th argument had an illegal value and\n\
      INFO > 0 -> for INFO = i, U(i,i) is exactly zero. Matrix is singular\n\
                  and the inverse cannot be computed.",info);
                (void)klist_free(objlist, (kfunc_void)lkcall_free);
                return(0);
              }
   
             /* Obtain inverse matrix from LU factorization */
            if (otype == KDOUBLE) DGETRI(&hc,L,&hc,ipiv,WORK,&work_len,&info);
            else ZGETRI(&hc,LZ,&hc,ipiv,WORKZ,&work_len,&info);

            if (info != 0)
              {
                kinfo(KSTANDARD,"lminvert: LAPACK {dz}getri() returned with INFO=%d \n\
where INFO < 0 -> for INFO = -i, the i'th argument had an illegal value and\n\
      INFO > 0 -> for INFO = i, U(i,i) is exactly zero. Matrix is singular\n\
                  and the inverse cannot be computed.",info);
                (void)klist_free(objlist, (kfunc_void)lkcall_free);
                return(0);
              }

            /* Untranspose so that C will like it! */
            if (otype == KDOUBLE)
              {
                for (i=0; i<hc; i++)
                  for (j=0; j<wc; j++)
                    A[i*wc+j] = L[j*hc+i];
              }
            else
              {
                for (i=0; i<hc; i++)
                  for (j=0; j<wc; j++)
                    AZ[i*wc+j] = LZ[j*hc+i];
              }

            if (otype == KDOUBLE)
              {
                  KCALL(kpds_put_data(out_obj,KPDS_VALUE_REGION,(kaddr)A));
              }
            else
              {
                  KCALL(kpds_put_data(out_obj,KPDS_VALUE_REGION,(kaddr)AZ));
              }

          }

        (void)klist_free(objlist, (kfunc_void)lkcall_free);

	return TRUE;
}
/* -library_code_end */
