 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for meigen
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lmeigen
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lmeigen - compute eigenvectors and eigenvalues
* 
*       Purpose: lmeigen computes the eigenvectors and eigenvalues of a square 
*                matrix. The matrix is NOT required to be symmetric.
*
*		 All output is in double complex form for generality.
*
*                If the eigenvalues are complex, then they will occur in
*                complex conjugate pairs. 
*
*		 The eignenvectors are normalized to have a unit Euclidian
*                norm and a purely real largest component.
*
*		 lmeigen uses the ZGEEV routine from LAPACK to actually
*                obtain the decomposition.
*
*         Input: kobject in_obj
*
*        Output: kobject evec_obj - matrix of eigenvectors
*                kobject eval_obj - diagonal matrix of eigenvalues
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: All output is of type double complex.
*    Written By: Scott Wilson
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: *
****************************************************************/
/* -library_def */
int lmeigen(
kobject in_obj,
kobject evec_obj,
kobject eval_obj)
/* -library_def_end */

/* -library_code */
{
        char *lib = "kmatrix", *rtn = "lmeigen";
        double *RWORK=NULL;
        kdcomplex *A=NULL,*W=NULL,*VR=NULL,*LWORK=NULL;
        int i,j,m,n,num,nplanes,info;
        int wc,hc,dc,tc,ec;
        int lwork_len,rwork_len;
        klist *objlist=NULL;
        char jobvl,jobvr;
        kdcomplex tmpz;

        /* Make sure we have valid objects */
        if (in_obj == KOBJECT_INVALID)
          {
            kerror(lib, rtn, "Bogus input object");
            return(FALSE);
          }
        if (evec_obj == KOBJECT_INVALID && eval_obj == KOBJECT_INVALID)
          {
            kerror(lib, rtn, "Must have at least one output object");
            return(FALSE);
          }

        /* Reference the input object to avoid side effects, then add the
           reference to the list of goodies to be autmatically free'd on
           error. */
        KCALL((in_obj = kpds_reference_object(in_obj)));
        objlist = klist_add(objlist,in_obj,"KOBJECT");

        /* Constrain the input data to double complex for read */
        KCALL(kpds_set_attribute(in_obj,KPDS_VALUE_DATA_TYPE, KDCOMPLEX));

        /* See how big the data is */
        KCALL(kpds_get_attribute(in_obj,KPDS_VALUE_SIZE,&wc,&hc,&dc,&tc,&ec));
 
        if (wc != hc)
          {
            kinfo(KSTANDARD,"lmeigen: Matrix must be square.\n");
            (void)klist_free(objlist, (kfunc_void)lkcall_free);
	    return FALSE;
          }

        /* Set up to process input by planes */
        KCALL(kpds_set_attribute(in_obj,KPDS_VALUE_REGION_SIZE,wc,hc,1,1,1));
        nplanes = dc*tc*ec;

        /* Set the output object sizes and data types */
        if (evec_obj != KOBJECT_INVALID)
          {
            if (!kpds_query_value(evec_obj)) KCALL(kpds_create_value(evec_obj));
            KCALL(kpds_set_attributes(evec_obj,
                                 KPDS_VALUE_SIZE,wc,hc,dc,tc,ec,
                                 KPDS_VALUE_DATA_TYPE,KDCOMPLEX,NULL));
            KCALL((evec_obj = kpds_reference_object(evec_obj)));
            objlist = klist_add(objlist,evec_obj,"KOBJECT");
            KCALL(kpds_set_attribute(evec_obj,KPDS_VALUE_REGION_SIZE,
                                            wc,hc,1,1,1));
          }
        if (eval_obj != KOBJECT_INVALID)
          {
            if (!kpds_query_value(eval_obj)) KCALL(kpds_create_value(eval_obj));
            KCALL(kpds_set_attributes(eval_obj,
                                 KPDS_VALUE_SIZE,wc,hc,dc,tc,ec,
                                 KPDS_VALUE_DATA_TYPE,KDCOMPLEX,NULL));
            KCALL((eval_obj = kpds_reference_object(eval_obj)));
            objlist = klist_add(objlist,eval_obj,"KOBJECT");
            KCALL(kpds_set_attribute(eval_obj,KPDS_VALUE_REGION_SIZE,
                                            wc,hc,1,1,1));
          }

        /* Allocate plane buffers */
        KCALL(!((A=(kdcomplex *)kmalloc(wc*hc*sizeof(kdcomplex)))== NULL));
        objlist = klist_add(objlist,A,"KMALLOC");
        KCALL(!((VR=(kdcomplex *)kmalloc(wc*hc*sizeof(kdcomplex)))== NULL));
        objlist = klist_add(objlist,VR,"KMALLOC");
        KCALL(!((W=(kdcomplex *)kmalloc(wc*sizeof(kdcomplex)))== NULL));
        objlist = klist_add(objlist,W,"KMALLOC");

        /* Allocate work arrays */
        lwork_len = 4*(2*kmax(wc,hc));
        KCALL(!((LWORK=(kdcomplex *)kmalloc(lwork_len*sizeof(kdcomplex)))== NULL));
        objlist = klist_add(objlist,LWORK,"KMALLOC");
        rwork_len = 2*wc;
        KCALL(!((RWORK=(double *)kmalloc(rwork_len*sizeof(double)))== NULL));
        objlist = klist_add(objlist,RWORK,"KMALLOC");

        jobvl = 'N';
        jobvr = 'V';

        for (num=0; num<nplanes; num++)
          {
            /* Get a plane from the input object */
            A=(kdcomplex *)kpds_get_data(in_obj,KPDS_VALUE_REGION,(kaddr)A);

             /* Transpose so that F77 likes it */
             for (i=0; i<hc; i++)
               for (j=0; j<wc; j++)
                {
                  tmpz = A[j*wc+i];
                  A[j*wc+i] = A[i*wc+j];
                  A[i*wc+j] = tmpz;
                }

            /* Obtain eigendecomposition */
            ZGEEV(&jobvl,&jobvr,&wc,A,&wc,W,VR,&wc,VR,&wc,LWORK,&lwork_len,
                  RWORK, &info);

            if (info != 0)
              {
                kinfo(KSTANDARD,"lmeigen: LAPACK zgeev() returned with INFO=%d\n\
where INFO < 0 -> for INFO = -i, the i'th argument had an illegal value and\n\
      INFO > 0 -> for INFO = i, QR algorithm failed to converge.\n",info);
              }

            /* Untranspose VR so that C likes it */
            for (i=0; i<wc; i++)
              for (j=0; j<wc; j++)
                {
                  tmpz = VR[j*wc+i];
                  VR[j*wc+i] = VR[i*wc+j];
                  VR[i*wc+j] = tmpz;
                }

            /* Read the values out of W and plant'em in A */
            kbzero(A,wc*hc*sizeof(kdcomplex));
            for (i=0; i<wc; i++) A[i*wc+i] = W[i];

            /* Write the data to the output objects */
            if (evec_obj != KOBJECT_INVALID)
              KCALL(kpds_put_data(evec_obj,KPDS_VALUE_REGION,(kaddr)VR));
            if (eval_obj != KOBJECT_INVALID)
              KCALL(kpds_put_data(eval_obj,KPDS_VALUE_REGION,(kaddr)A));
          }

        (void)klist_free(objlist, (kfunc_void)lkcall_free);
	return TRUE;
}
/* -library_code_end */
