 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for gtubeness
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lgtubeness
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
int build_scaled_unit_circle PROTO((float *, float *, int , float *));
int build_tube PROTO((vertex_3d *, vertex_3d *, int, vertex_3d *, vertex_3d *,
		      float *, float *, float *_colors, int, int, float *));

void lprim_copy PROTO((kobject, kobject, int, vertex_3d *, vertex_3d *));
int transform_yaxis_to_abc PROTO((vertex_3d *, matrix4x4 *));
int make_strip PROTO((vertex_3d *, vertex_3d *, vertex_3d *, int));
int load_colors PROTO((float *, float *, int, int, int));
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lgtubeness - replace lines with tubes.
* 
*       Purpose: A Geometry filter which will replaces all line segments
*                in a geometry object with tubes.  Non-line-segment objects
*                will be passed to the output object unchanged.
*
*         Input: kobject input - the input geometry object.
*                float scale - radius of all output tubes.
*                int subs - a value controlling the number of sides for
*                    each tube.  the range of this value is 2<=N<=6, and
*                    the number of sides on each tube is computed as:
*                    2<<(N-1).
*                int cap_style - indicates how the tubes should be capped.
*	             0 - no cap, a tube without ends
*                    1 - flat caps, no mitering
*
*        Output: kobject output - the new geometry object.
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: Restrictions on data or input as applicable
*    Written By: wes
*          Date: Apr 13, 1995
*      Verified: 
*  Side Effects: note the relationship between the cap_style variable
*                and the list of styles in the .pane file.
* Modifications: 
****************************************************************/
/* -library_def */
int lgtubeness(kobject input,
	       kobject output,
	       float scale,
	       int subs,
	       int cap_style)
/* -library_def_end */

/* -library_code */
{
    int *prim_list=NULL;
    int status,num_prims,output_num_prims=0;
    int i;
    int nsides_per_tube,ntriangles_per_tube,nverts_per_tube;
    vertex_3d bmin,bmax,center,lbmin,lbmax;
    vertex_3d *work_verts,*work_norms;
    float *ctable,*stable;
    int has_alpha,layout;
    float *work_colors;
    
    status = kgeom_query_primitive_list(input);
    if (status)
    {
        status = kgeom_get_attribute(input,KGEOM_OBJECT,KGEOM_NUMBER_PRIMITIVES,&num_prims);
    }
    else
    {
	kerror("GEOMETRY","lktubeness", "Can't get KGEOM_NUMBER_PRIMITVES attributes from the input object.  Maybe it's not a GEOM file?");
	return(FALSE);
    }

    /* grab the list of primitives */
    status = kgeom_get_attribute(input,KGEOM_OBJECT,KGEOM_PRIMITIVE_LIST,&prim_list);
    if (status == FALSE)
    {
	kfprintf(output,"\tCan't get PRIM_LIST attribute.  Maybe not a GEOM file?\n");
	return (FALSE);
    }
    
    kgeom_copy_attributes(input,output);
    kgeom_create_primitive_list(output);

    for (i=0;i<3;i++)
    {
	bmin.v[i] = KGEOM_BIGNUM;
	bmax.v[i] = -1. * KGEOM_BIGNUM;
    }

    status = kgeom_get_attribute(input,KGEOM_OBJECT,KGEOM_HAS_ALPHA,&has_alpha);
    if (!status)
	kerror("GEOMETRY","lktubeness","Can't get the has_alpha attribute from the input object.");
    status = kgeom_set_attribute(output,KGEOM_OBJECT,KGEOM_HAS_ALPHA,has_alpha);
    
    nsides_per_tube = 2<<(subs-1);
    ntriangles_per_tube = nsides_per_tube * 2;
    nverts_per_tube = ntriangles_per_tube + 2;
    work_verts = (vertex_3d *)kmalloc(sizeof(vertex_3d)*nverts_per_tube);
    work_norms = (vertex_3d *)kmalloc(sizeof(vertex_3d)*nverts_per_tube);

    if (has_alpha)
	work_colors = (float *)kmalloc(sizeof(float)*nverts_per_tube*4);
    else
	work_colors = (float *)kmalloc(sizeof(float)*nverts_per_tube*3);

    status = kgeom_get_attribute(input,KGEOM_OBJECT,KGEOM_LAYOUT,&layout);
    status = kgeom_set_attribute(output,KGEOM_OBJECT,KGEOM_LAYOUT,KPER_VERTEX);
    
    ctable = (float *)kmalloc(sizeof(float)*nsides_per_tube);
    stable = (float *)kmalloc(sizeof(float)*nsides_per_tube);

    build_scaled_unit_circle(ctable,stable,nsides_per_tube,&scale);
    
    for (i=0;i<num_prims;i++)
    {
        kgeom_set_attribute(input,KGEOM_OBJECT,KGEOM_PRIMITIVE_POSITION,i);
	

	switch (prim_list[i])
	{

	case KGEOM_POLYLINE_DISJOINT:
	    {
		int in_nverts,status,in_segs;
		int i,has_color,next_color_step;
		float *c1;
		int color_stride;
		int ntriangles; /* per output prim, of which there will be
				   one per input segment. */
		float *verts=NULL,*colors=NULL;
		vertex_3d *v1,*v2,*s;

		status = kgeom_get_attribute(input,KGEOM_POLYLINE_DISJOINT,KGEOM_NUMBER_VERTICES,&in_nverts);
		kgeom_get_data(input,KGEOM_POLYLINE_DISJOINT,&verts,&colors);

		in_segs = in_nverts/2;

		s = (vertex_3d *)verts;
		v1 = s;
		v2 = v1+1;

		if (colors == NULL)
		{
		    has_color = 0;
		    color_stride = 0;
		    c1 = NULL;
		}
		else
		{
		    has_color = 1;
		    c1 = colors;
		    if (has_alpha)
			color_stride = 4;
		    else
			color_stride = 3;
		}
		next_color_step = (layout == KPER_VERTEX) ?
		    color_stride * 2 : color_stride;
		
		for (i=0;i<in_segs;i++,v1+=2,v2+=2,c1+=next_color_step)
		{
		    status = build_tube(v1,v2,nsides_per_tube,work_verts,
					work_norms,ctable,stable,c1,
					color_stride,layout,work_colors);
		    
		    if (status == CHILL)
		    {
			output_num_prims++;
			kgeom_set_attribute(output,KGEOM_OBJECT,KGEOM_NUMBER_PRIMITIVES,output_num_prims);
			kgeom_set_attribute(output,KGEOM_TRIANGLES_CONNECTED,KGEOM_NUMBER_VERTICES,nverts_per_tube);

			kgeom_put_data(output,KGEOM_TRIANGLES_CONNECTED,
				       work_verts,
				       ((colors == NULL) ? NULL : work_colors),
				       work_norms,NULL);

			lcompute_bbox((vertex_3d *)work_verts,nverts_per_tube,&lbmin,&lbmax);
			lunion_boxes(&lbmin,&lbmax,&bmin,&bmax,&bmin,&bmax);
		    }
		}
	    }
	    break;
	case KGEOM_POLYLINE_CONNECTED:
	    {
		int in_nverts,status,in_segs;
		int i,has_color;
		int ntriangles; /* per output prim, of which there will be
				   one per input segment. */
		float *verts=NULL,*colors=NULL;
		vertex_3d *v1,*v2,*s;
		float *c1,*c2;
		int color_stride;

		
		status = kgeom_get_attribute(input,KGEOM_POLYLINE_CONNECTED,KGEOM_NUMBER_VERTICES,&in_nverts);
		kgeom_get_data(input,KGEOM_POLYLINE_CONNECTED,&verts,&colors);

		if (!status)
		    kerror("GEOMETRY","lktubeness","Can't get the has_alpha attribute from the input object.");

		in_segs = in_nverts-1;

		s = (vertex_3d *)verts;
		v1 = s;
		v2 = v1+1;

		if (colors == NULL)
		{
		    c1 = NULL;
		    has_color = 0;
		    color_stride = 0;
		}
		else
		{
		    has_color = 1;
		    if (has_alpha)
			color_stride = 4;
		    else
			color_stride = 3;
		    c1 = colors;
		}
		
		for (i=0;i<in_segs;i++,v1=v2,v2+=1, c1+=color_stride)
		{
		    status = build_tube(v1,v2,nsides_per_tube,work_verts,
					work_norms,ctable,stable,c1,
					color_stride,layout,work_colors);
		    
		    if (status == CHILL)
		    {
			output_num_prims++;
			kgeom_set_attribute(output,KGEOM_OBJECT,KGEOM_NUMBER_PRIMITIVES,output_num_prims);
			/**
			  * other geom code doesn't do the following.  i
			  * treid it out but it doesn't really seem to
			  * make any difference.  1/31/95 wes
			**/
			kgeom_set_attribute(output,KGEOM_OBJECT,KGEOM_PRIMITIVE_POSITION,output_num_prims-1);
			kgeom_set_attribute(output,KGEOM_TRIANGLES_CONNECTED,KGEOM_NUMBER_VERTICES,nverts_per_tube);
			kgeom_put_data(output,KGEOM_TRIANGLES_CONNECTED,
				       work_verts,
				       (has_color ? work_colors : NULL),
				       work_norms,NULL);

			lcompute_bbox((vertex_3d *)work_verts,nverts_per_tube,&lbmin,&lbmax);
			lunion_boxes(&lbmin,&lbmax,&bmin,&bmax,&bmin,&bmax);
		    }
		}
	    }

	    break;
	default:
	    output_num_prims++;
	    kgeom_set_attribute(output,KGEOM_OBJECT,KGEOM_NUMBER_PRIMITIVES,output_num_prims);
/*	    kgeom_set_attribute(output,KGEOM_OBJECT,KGEOM_PRIMITIVE_POSITION,output_num_prims); */
	    lprim_copy(input,output,prim_list[i],&bmin,&bmax);
	    break;
	}
    }

    kgeom_set_attribute(output,KGEOM_OBJECT,KGEOM_BOUNDING_BOX,&(bmin.v[0]),&(bmax.v[0]));

    for (i=0;i<3;i++)
	center.v[i] = bmin.v[i] + 0.5 * (bmax.v[i] - bmin.v[i]);
    kgeom_set_attribute(output,KGEOM_OBJECT,KGEOM_CENTER,&(center.v[0]));
    
    kfree(work_verts);
    kfree(work_norms);
    kfree(ctable);
    kfree(stable);
	
    return TRUE;
}

void
lprim_copy(kobject input,
	   kobject output,
	   int prim_type,
	   vertex_3d *bmin,
	   vertex_3d *bmax)
{
    /**
      * this routine will copy the primitive from the input object to
      * the output object.  the calling routine is repsonsible for all
      * the bookeeping of prim_list pointers and that crap.  additionally,
      * this routine will do a bounding box union of the copied primitive
      * with that box passed in.
    **/
    int i;
    vertex_3d lbmin2,lbmax2;

    for (i=0;i<3;i++)
    {
	lbmin2.v[i] = KGEOM_BIGNUM;
	lbmax2.v[i] = -1. * KGEOM_BIGNUM;
    }
    
    switch (prim_type)
    {
    case KGEOM_POLYLINE_DISJOINT:
	{
	    int nverts,status;
	    float *verts=NULL,*colors=NULL;

	    status = kgeom_get_attribute(input,KGEOM_POLYLINE_DISJOINT,KGEOM_NUMBER_VERTICES,&nverts);
	    kgeom_set_attribute(output,KGEOM_POLYLINE_DISJOINT,KGEOM_NUMBER_VERTICES,nverts);
	    kgeom_get_data(input,KGEOM_POLYLINE_DISJOINT,&verts,&colors);
	    kgeom_put_data(output,KGEOM_POLYLINE_DISJOINT,verts,colors);

	    lcompute_bbox((vertex_3d *)verts,nverts,&lbmin2,&lbmax2);

	}
	break;
    case KGEOM_POLYLINE_CONNECTED:
	{
	    int nverts,status;
	    float *verts=NULL,*colors=NULL;
	    
	    status = kgeom_get_attribute(input,KGEOM_POLYLINE_CONNECTED,KGEOM_NUMBER_VERTICES,&nverts);
	    kgeom_set_attribute(output,KGEOM_POLYLINE_CONNECTED,KGEOM_NUMBER_VERTICES,nverts);
	    kgeom_get_data(input,KGEOM_POLYLINE_CONNECTED,&verts,&colors);
	    kgeom_put_data(output,KGEOM_POLYLINE_CONNECTED,verts,colors);
	    lcompute_bbox((vertex_3d *)verts,nverts,&lbmin2,&lbmax2);
	}
	break;
    case KGEOM_TRIANGLES_DISJOINT:
	{
	    float *v=NULL,*norms=NULL,*c=NULL,*tc=NULL;
	    int nverts;
	    kgeom_get_attribute(input,KGEOM_TRIANGLES_DISJOINT,KGEOM_NUMBER_VERTICES,&nverts);
	    kgeom_get_data(input,KGEOM_TRIANGLES_DISJOINT,&v,&c,&norms,&tc);
	    kgeom_set_attribute(output,KGEOM_TRIANGLES_DISJOINT,KGEOM_NUMBER_VERTICES,&nverts);
	    kgeom_put_data(input,KGEOM_TRIANGLES_DISJOINT,v,c,norms,tc);
	    lcompute_bbox((vertex_3d *)v,nverts,&lbmin2,&lbmax2);
	    
	}
	break;
    case KGEOM_TRIANGLES_CONNECTED:
	{
	    float *v=NULL,*c=NULL,*norm=NULL,*tc=NULL;
	    int nverts;
	    kgeom_get_attribute(input,KGEOM_TRIANGLES_CONNECTED,KGEOM_NUMBER_VERTICES,&nverts);
	    kgeom_get_data(output,KGEOM_TRIANGLES_CONNECTED,&v,&c,&norm,&tc);
	    
	    kgeom_set_attribute(output,KGEOM_TRIANGLES_CONNECTED,KGEOM_NUMBER_VERTICES,nverts);
	    kgeom_put_data(output,KGEOM_TRIANGLES_CONNECTED,v,c,norm,tc);
	    lcompute_bbox((vertex_3d *)v,nverts,&lbmin2,&lbmax2);
	}
	break;
    case KGEOM_SPHERES:
	{
	    int nspheres;
	    float *v=NULL,*r=NULL,*c=NULL;
	    kgeom_get_attribute(input,KGEOM_SPHERES,KGEOM_NUMBER_VERTICES,&nspheres);
	    kgeom_get_data(input,KGEOM_SPHERES,&v,&c,&r);
	    
	    kgeom_get_attribute(output,KGEOM_SPHERES,KGEOM_NUMBER_VERTICES,&nspheres);
	    kgeom_get_data(output,KGEOM_SPHERES,v,c,r);
	    /**
	      * this isn't 100% correct because it doesn't take into account
	      * the radius values.
	    **/
	    lcompute_bbox((vertex_3d *)v,nspheres,&lbmin2,&lbmax2);
	}
	
	break;
    case KGEOM_QUADMESH:
	{
	    float *v=NULL,*c=NULL,*norm=NULL,*tc=NULL;
	    int has_loc = FALSE,has_color=FALSE,has_norms=FALSE,
	        has_tc=FALSE,tc_size;
	    int status,usize,vsize;

	    status = kgeom_get_attribute(input,KGEOM_QUADMESH,KGEOM_QUADMESH_SIZE,&usize,&vsize);
	    kgeom_set_attribute(output,KGEOM_QUADMESH,KGEOM_QUADMESH_SIZE,usize,vsize);

	    has_loc =  kgeom_get_data(input,KGEOM_QUADMESH_LOCATION_ALL,&v);
	    if (has_loc)
	    {
		kgeom_put_data(output,KGEOM_QUADMESH_LOCATION_ALL,v);
		/**
		 * this doesn't necessarily work...what about uniform meshes?
		 **/
		lcompute_bbox((vertex_3d *)v,usize*vsize,&lbmin2,&lbmax2);
	    }

	    has_color = kgeom_get_data(input,KGEOM_QUADMESH_COLOR_ALL,&c);
	    if (has_color)
		kgeom_put_data(output,KGEOM_QUADMESH_COLOR_ALL,c);
	    
	    has_norms = kgeom_get_data(input,KGEOM_QUADMESH_NORMAL_ALL,&norm);
	    if (has_norms)
		 kgeom_put_data(output,KGEOM_QUADMESH_NORMAL_ALL,norm);
	    
	    has_tc = kgeom_get_data(input,KGEOM_QUADMESH_TEXTURE_COORD_ALL,&tc);
	    if (has_tc)
		kgeom_put_data(output,KGEOM_QUADMESH_TEXTURE_COORD_ALL,tc);

	}
	break;
    case KGEOM_OCTMESH:
	{
	    int usize,vsize,wsize,status;
	    int has_loc,has_color,has_norms,has_tc;
	    float *v=NULL,*c=NULL,*norm=NULL,*tc=NULL;
	    
	    status = kgeom_get_attribute(input,KGEOM_OCTMESH,KGEOM_OCTMESH_SIZE,&usize,&vsize,&wsize);
	    status = kgeom_set_attribute(output,KGEOM_OCTMESH,KGEOM_OCTMESH_SIZE,&usize,&vsize,&wsize);
	    
	    has_loc =  kgeom_get_data(input,KGEOM_OCTMESH_LOCATION_ALL,&v);
	    if (has_loc)
	    {
		kgeom_put_data(output,KGEOM_OCTMESH_LOCATION_ALL,v);
		/**
		 * this doesn't necessarily work...what about uniform meshes?
		 **/
		lcompute_bbox((vertex_3d *)v,usize*vsize*wsize,&lbmin2,&lbmax2);
	    }

	    has_color = kgeom_get_data(input,KGEOM_OCTMESH_COLOR_ALL,&c);
	    if (has_color)
		kgeom_put_data(output,KGEOM_OCTMESH_COLOR_ALL,c);

	    has_norms = kgeom_get_data(input,KGEOM_OCTMESH_NORMAL_ALL,&norm);
	    if (has_norms)
		kgeom_put_data(output,KGEOM_OCTMESH_NORMAL_ALL,norm);
	    
	    has_tc = kgeom_get_data(input,KGEOM_QUADMESH_TEXTURE_COORD_ALL,&tc);
	    if (has_tc)
		kgeom_put_data(output,KGEOM_QUADMESH_TEXTURE_COORD_ALL,tc);

	}
	break;
    }
    lunion_boxes(bmin,bmax,&lbmin2,&lbmax2,bmin,bmax);
}

int
build_tube(vertex_3d *v1,
	   vertex_3d *v2,
	   int nsides_per_tube,
	   vertex_3d *work_verts,
	   vertex_3d *work_norms,
	   float *ctable,
	   float *stable,
	   float *in_colors,
	   int color_stride,
	   int layout,
	   float *work_colors)
{
    vertex_3d *e1,*e2,*n1,*n2;
    matrix4x4 m;
    vertex_3d unitvec,dirvec;
    double mag;
    int i;
    int status;
    vertex_3d *c1=NULL,*c2=NULL,*colortable=NULL;

    e1 = (vertex_3d *)kmalloc(sizeof(vertex_3d)*nsides_per_tube);
    e2 = (vertex_3d *)kmalloc(sizeof(vertex_3d)*nsides_per_tube);
    n1 = (vertex_3d *)kmalloc(sizeof(vertex_3d)*nsides_per_tube);
    n2 = (vertex_3d *)kmalloc(sizeof(vertex_3d)*nsides_per_tube);
    
    /* compute the unit direction vector of the tube. */
    for (i=0;i<3;i++)
	dirvec.v[i] = v2->v[i] - v1->v[i];

    mag = 0.;
    for (i=0;i<3;i++)
	mag += dirvec.v[i]*dirvec.v[i];
    mag = ksqrt(mag);

    /* if the height of the tube is zero, bail out */
    if (mag != 0.)
    {
	mag = 1./mag;
	for (i=0;i<3;i++)
	    unitvec.v[i] = dirvec.v[i]* mag;

	/* compute the transformation that will allow us to take points
	   which lie in the x-z plane and transform them into a plane
	   with a normal that corresponds to the direction of the tube. */
	
	transform_yaxis_to_abc(&unitvec,&m);

	/* load up some points, the scaled unit circle from the precomputed
	   sin/cos tables, into a vertex array.  we'll have a circle that
	   lies in the x-z plane. */
	 
	for (i=0;i<nsides_per_tube;i++)
	{
	    e1[i].v[0] = ctable[i];
	    e1[i].v[1] = 0.;
	    e1[i].v[2] = stable[i];
	}

	/* rotate the circle so that it lies in a plane whose normal is
	   the direction of the tube. */
	
	npoint_xfrm((KGEOM_VERTEX_TYPE *)&(e1[0].v[0]),
		    &m,
		    (KGEOM_VERTEX_TYPE *)&(e2[0].v[0]),
		    nsides_per_tube);

	kmemcpy(n1,e2,sizeof(vertex_3d)*nsides_per_tube);
	kmemcpy(n2,e2,sizeof(vertex_3d)*nsides_per_tube);

	/* e2 now has the rotated circle.  translate this circle to
	   the point in space of one end of the tube; v1. put the
	   results into e1. */
	
	npoint_add((KGEOM_VERTEX_TYPE *)&(e2[0].v[0]),
		   (KGEOM_VERTEX_TYPE *)&(v1->v[0]),
		   (KGEOM_VERTEX_TYPE *)&(e1[0].v[0]),
		   nsides_per_tube);

	/* translate e2 to the other end of the tube. */
	npoint_add((KGEOM_VERTEX_TYPE *)&(e2[0].v[0]),
		   (KGEOM_VERTEX_TYPE *)&(v2->v[0]),
		   (KGEOM_VERTEX_TYPE *)&(e2[0].v[0]),
		   nsides_per_tube);

	/* ok, now we have two circle's worth of points which
	   form the ends of the tube.  now, load up the destination
	   arrays with points in the right order so as to correspond
	   to t-strips. */

	make_strip(e1,e2,work_verts,nsides_per_tube);
	make_strip(n1,n2,work_norms,nsides_per_tube);

	if (in_colors != NULL)
	    load_colors(work_colors,in_colors,nsides_per_tube,layout,
			color_stride);
	
/*	build_cylinder_surface(e1,e2,nsides_per_tube,work_verts,c1,c2,
			       colortable,n1,n2,work_norms); */
	
	status = CHILL;
    }
    else
    {
	/* a degenerate condition */
	status = WHACKED;
    }

    kfree(e1);
    kfree(e2);
    kfree(n1);
    kfree(n2);
    return(status);
}

#define MY_PI 3.1415927
int
build_scaled_unit_circle(float *ctable,
			 float *stable,
			 int n,
			 float *scale)
{
    int i;
    double dscale,t,dt,t2;

    dscale = *scale;
    
    dt = (2. * MY_PI)/n;
    t = 0;
    for (i=0;i<n;i++,t+=dt)
    {
	t2 = kcos(t);
	ctable[i] = t2 * dscale;
	t2 = ksin(t);
	stable[i] = t2 * dscale;
    }
    return(CHILL);
}

int
transform_yaxis_to_abc(vertex_3d *abc,matrix4x4 *m)
{
    /**
      * this routine will compute a transformation matrix M which will allow
      * one to transform points which lie in the x-z plane into a plane
      * with the normal "abc".
    **/

       /**
      * the basic idea is that we have a unit circle (scaled to the
      * appropriate radius) which lies in the XZ plane.  this plane
      * has a normal of [0,1,0].  we want to figure out how to rotate
      * the plane containing this circle into the plane which has a
      * normal, which is the same as the direction formed by the
      * input vector.
      *
      * the derivation for the rotations goes something like this:
      *
      * we compute the unit vector from the input vector, and call
      * it (A,B,C).  take the cross product of this vector and [0,1,0]
      * and we end up with (C,0,-A).  the cross product forms the
      * axis of rotation.  the angle of rotation about this axis is
      * the dot product of [A,B,C] and [0,1,0]
      *
      * after some hand waving, we end up with the following transformation
      * matrix (thanks go to Terry Ligocki of LBL for his help):
 
      |                                              |
      |  A^2 * B + C^2                 A*B*C - A*C   |
      |  -------------        A        -----------   |
      |    A^2 + C^2                    A^2 + C^2    |
      |                                              |
T =   |       -A              B            -C        |
      |                                              |
      |   A*B*C - A*C                 B * C^2 + A^2  |
      |   -----------         C       -------------  |
      |    A^2 + C^2                    A^2 + C^2    |
      |                                              |
 
      This is assuming that a point, p, is a column vector and the transformed
      point is T*p (i.e. the matrix multiples on the left).
 
      *
    **/

    double a2c2;
    double m11,m12,m13,m31,m32,m33;
    
    identity_4x4(m);

    a2c2 = abc->v[0]*abc->v[0] + abc->v[2]*abc->v[2];

    if (a2c2 == 0.)  /* a degenerate condition; make an identity matrix */
    {
        m11 = 1.;
        m12 = 0.;
        m13 = 0.;
 
        m31 = 0.;
        m32 = 0.;
        m33 = 1.;
 
    }
    else
    {
        a2c2 = 1./a2c2;
 
/*        m11 = tp.x*tp.x*tp.y + tp.z*tp.z; */
	m11 = abc->v[0]*abc->v[0]*abc->v[1] + abc->v[2]*abc->v[2];
        m11 *= a2c2;
 
/*        m12 = -1. * tp.x; */
	m12 = -1. * abc->v[0];
 
/*        m13 = tp.x * tp.y * tp.z - tp.x * tp.z; */
	m13 = abc->v[0] * abc->v[1] * abc->v[2] - abc->v[2]*abc->v[2];
	
        m13 *= a2c2;
 
        m31 = m13;
 
/*        m32 = -1. * tp.z; */
	m32 = -1. * abc->v[2];
 
/*        m33 = tp.y * tp.z * tp.z + tp.x * tp.x; */
	m33 = abc->v[1] * abc->v[2] * abc->v[2] + abc->v[0] * abc->v[0];
        m33 *= a2c2;
    }

    m->m[0][0] = m11;
    m->m[0][1] = m12;
    m->m[0][2] = m13;
    m->m[2][0] = m31;
    m->m[2][1] = m32;
    m->m[2][2] = m33;
   
    return(CHILL);
}

int
make_strip(vertex_3d *s1,
	   vertex_3d *s2,
	   vertex_3d *d,
	   int n)
{
    int offset,i;
    /**
      * we're given two arrays of things, s1 and s2, each of which is
      * N long, and we want to scramble them into a new array D, which
      * better be N*2+2 long, so as to make a triangle strip from s1 and s2.
    **/
    offset = 0;
    for (i=0;i<n;i++)
    {
	kmemcpy(d+offset,s2+i,sizeof(vertex_3d));
	offset++;
	kmemcpy(d+offset,s1+i,sizeof(vertex_3d));
	offset++;
    }
    kmemcpy(d+offset,s2,sizeof(vertex_3d));
    offset++;
    kmemcpy(d+offset,s1,sizeof(vertex_3d));

    return(CHILL);
}
	
int
load_colors(float *d,
	    float *s,
	    int n,
	    int layout,
	    int stride)
{
    int offset,i;
    float even[4],odd[4];
    int tsize;

    tsize = sizeof(float)*stride;

    offset = 0;

    kmemcpy(even,s,tsize);
    if (layout == KPER_VERTEX)
	kmemcpy(odd,s+stride,tsize);
    else
	kmemcpy(odd,even,tsize);
    
    for (i=0;i<n;i++)
    {
	kmemcpy(d+offset,even,tsize);
	offset += stride;
	kmemcpy(d+offset,odd,tsize);
	offset += stride;
    }
    kmemcpy(d+offset,even,tsize);
    offset += stride;
    kmemcpy(d+offset,odd,tsize);

    return(CHILL);
}
/* -library_code_end */
