
/*********************************************************************

Copyright (C) 1993, 1994, 1995 Lawrence Berkeley Laboratory.  All Rights
Reserved.  Permission to copy and modify this software and its
documentation (if any) is hereby granted, provided that this notice
is retained thereon and on all copies.  

This software is provided as a professional academic contribution
for joint exchange.   Thus is is experimental and scientific
in nature, undergoing development, and is provided "as is" with
no warranties of any kind whatsoever, no support, promise of
updates or printed documentation.

This work is supported by the U. S. Department of Energy under 
contract number DE-AC03-76SF00098 between the U. S. Department 
of Energy and the University of California.


	Author: Wes Bethel
		Lawrence Berkeley Laboratory

  "this software is 100% hand-crafted by a human being in the USA"

*********************************************************************/
 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


#include <geometry.h>
#include "gprgeom.h"

#define NUM_PRIMS 7

static void
    print_polyline_dis PROTO((kobject, int, int, int,int,int,int,int,kfile *)),
    print_polyline_con PROTO((kobject, int, int,int,int,int,int,int,kfile *)),
    print_triangles_dis PROTO((kobject, int, int,int,int,int,int,int,kfile *)),
    print_triangles_con PROTO((kobject, int, int,int,int,int,int,int,kfile *)),
    print_spheres PROTO((kobject, int, int,int,int,int,int,int,kfile *)),
    print_qmesh PROTO((kobject, int, int,int,int,int,int,int,kfile *)),
    print_omesh PROTO((kobject, int, int,int,int,int,int,int,kfile *)),
    print_unknown PROTO((kobject, int, int,int,int,int,int,int,kfile *));

static void (*prtfuncs[NUM_PRIMS+1])()=
{
    print_polyline_dis,
    print_polyline_con,
    print_triangles_dis,
    print_triangles_con,
    print_spheres,
    print_qmesh,
    print_omesh,
    print_unknown
};

static void print_vectors PROTO((float *, int, int, char *, char *, kfile *));
static void print_att PROTO((kobject, kfile *));

/*-----------------------------------------------------------
| 
|  Routine Name: kprgeom_util()
| 
|       Purpose: routine to print out information contained
|                in an object containing geometry information.
| 
|         Input: kobject "k" - the object in question
| 
|        Output: bunch of text 
|    Written By: wes
|          Date: May 20, 1993
| Modifications: 
| 
------------------------------------------------------------*/
void
kprgeom_util(kobject k,
	     kfile *output,
	     int do_att,
	     int do_loc,
	     int do_color,
	     int do_norm_radii,
	     int do_texture)
{
    int status;
    int num_prims,i,prim_type;
    int func_index;
    int *prim_list=NULL;
    int has_alpha;


    if (do_att)
	print_att(k,output);
    
    kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_HAS_ALPHA,&has_alpha);
	
    /* ask for floating point presentation of location data */
    kgeom_set_attribute(k,KGEOM_OBJECT,KGEOM_LOCATION_DATA_TYPE,KFLOAT);

    /* check to see how many primitives are in the object */
    status = kgeom_query_primitive_list(k);
    if (status)
        status = kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_NUMBER_PRIMITIVES,&num_prims);

    if (status == FALSE)
	kfprintf(output,"\tCan't get PRIM_NUM attribute.  Maybe not a GEOM file?\n");
    else
	kfprintf(output,"\tNumber of primitives in this object is %d.\n",num_prims);

    /* grab the list of primitives */
    status = kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_PRIMITIVE_LIST,&prim_list);
    if (status == FALSE)
    {
	kfprintf(output,"\tCan't get PRIM_LIST attribute.  Maybe not a GEOM file?\n");
	return;
    }

    for (i=0;i<num_prims;i++)
    {
        /* TEMPORARY KLUDGE... */
        kgeom_set_attribute(k,KGEOM_OBJECT,KGEOM_PRIMITIVE_POSITION,i);
	switch (prim_list[i])
	{
	case KGEOM_POLYLINE_DISJOINT:
	    func_index = 0;
	    break;
	case KGEOM_POLYLINE_CONNECTED:
	    func_index = 1;
	    break;
	case KGEOM_TRIANGLES_DISJOINT:
	    func_index = 2;
	    break;
	case KGEOM_TRIANGLES_CONNECTED:
	    func_index = 3;
	    break;
        case KGEOM_SPHERES:
            func_index = 4;
	    break;
	case KGEOM_QUADMESH:
	    func_index = 5;
	    break;
	case KGEOM_OCTMESH:
	    func_index = 6;
	    break;
	default:
	    func_index = NUM_PRIMS;
	    break;
	}
	(*prtfuncs[func_index])(k,i,prim_list[i],do_loc,do_color,do_norm_radii,do_texture,has_alpha,output);
    }
}

static void
print_polyline_dis(kobject k,
		   int n,
		   int prim_type,
		   int do_loc,
		   int do_color,
		   int do_norm_radii,
		   int do_texture,
		   int has_alpha,
		   kfile *o)
{
    int nverts,status,i,layout;
    float *verts=NULL,*colors=NULL;

    status = kgeom_get_attribute(k,KGEOM_POLYLINE_DISJOINT,KGEOM_NUMBER_VERTICES,&nverts);
    if (status == FALSE)
    {
	kfprintf(o,"\t\tError obtaining KGEOM_NUMBER from this primitive. Skipping it an continuing...\n");
	return;
    }
    
    kfprintf(o,"\n------- PRIMITIVE %d is KPOLYLINE_DIS <%d>.\n",n,prim_type);
    kfprintf(o,"\tand has %d vertices (%d lines).\n",nverts, nverts/2);

    status = kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_LAYOUT,&layout);
    if (status == FALSE)
    {
       kfprintf(o,"\t\tError obtaining the KGEOM_LAYOUT information.\n");
       layout = -1;
    }
    
    /* assert that we want floating point, RGB data */
    kgeom_set_attribute(k,KGEOM_OBJECT,KGEOM_COLOR_DATA_TYPE,KFLOAT);
    
    kgeom_get_data(k,KGEOM_POLYLINE_DISJOINT,&verts,&colors);
    
    if (colors == NULL)
       kfprintf(o,"\t\tcolor data is NOT present.\n");
    else
       kfprintf(o,"\t\tcolor data is present on a %s basis.\n",(layout == KPER_VERTEX ? "PER VERTEX" : "PER LINE"));
    
    if (do_loc)
       print_vectors(verts,nverts,3,"Vertices","index\tX\tY\tZ",o);
    
    if (do_color && colors)
       print_vectors(colors,(layout == KPER_VERTEX ? nverts : nverts/2),(has_alpha == TRUE ? 4 : 3),"Colors",(has_alpha == TRUE ? "index\tR\tG\tB\tA" : "index\tR\tG\tB"),o);
}

static void
print_spheres(kobject k,
	      int n,
	      int prim_type,
	      int do_loc,
	      int do_color,
	      int do_norm_radii,
	      int do_texture,
	      int has_alpha,
	      kfile *o)
{
    int nspheres,i;
    float *v=NULL,*r=NULL,*c=NULL;

    kgeom_get_attribute(k,KGEOM_SPHERES,KGEOM_NUMBER_VERTICES,&nspheres);

    kfprintf(o,"\n------- PRIMITIVE %d is KSPHERES <%d>\n",n,prim_type);
    kfprintf(o,"\tand has %d spheres.\n",nspheres);

    /* ask for floating point presentation of the radius data */
    kgeom_set_attribute(k,KGEOM_SPHERES,KGEOM_RADIUS_DATA_TYPE,KFLOAT);
    kgeom_set_attribute(k,KGEOM_SPHERES,KGEOM_COLOR_DATA_TYPE,KFLOAT);

    kgeom_get_data(k,KGEOM_SPHERES,&v,&c,&r);

    if (c == NULL)
	kfprintf(o,"\t\tcolor data is NOT present.\n");
    else
    {
	kfprintf(o,"\t\tcolor data is present.\n");
    }

    if (r == NULL)
	kfprintf(o,"\t\tradius data is NOT present.\n");
    else
    {
	kfprintf(o,"\t\tradius data is present.\n");
    }

    if (do_loc)
       print_vectors(v,nspheres,3,"Centers","index\tX\tY\tZ",o);

    if (do_color && c)
	print_vectors(c,nspheres,(has_alpha == TRUE ? 4 : 3),"Colors",
		      (has_alpha == TRUE ? "index\tR\tG\tB\tA" : "index\tR\tG\tB"),o);
    
    if (do_norm_radii && r)
	print_vectors(r,nspheres,1,"Radii","index\tRadius",o);
}

static void
print_polyline_con(kobject k, 
		   int n,
		   int prim_type,
		   int do_loc,
		   int do_color,
		   int do_norm_radii,
		   int do_texture,
		   int has_alpha,
		   kfile *o)
{
    int nverts,status,i,layout;
    float *verts=NULL,*colors=NULL;

    status = kgeom_get_attribute(k,KGEOM_POLYLINE_CONNECTED,KGEOM_NUMBER_VERTICES,&nverts);
    if (status == FALSE)
    {
	kfprintf(o,"\t\tError obtaining KGEOM_NUMBER_VERTICES from this primitive. Skipping it and continuing...\n");
	return;
    }

    kfprintf(o,"\n------- PRIMITIVE %d is KGEOM_POLYLINE_CONNECTED <%d>\n",n,prim_type);
    kfprintf(o,"\tand has %d vertices (%d lines).\n",nverts, nverts-1);

    status = kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_LAYOUT,&layout);
    if (status == FALSE)
    {
       kfprintf(o,"\t\tError obtaining the KGEOM_LAYOUT information.\n");
       layout = -1;
    }
    
    /* assert that we want floating point, RGB data */
    kgeom_set_attribute(k,KGEOM_OBJECT,KGEOM_COLOR_DATA_TYPE,KFLOAT);
    
    kgeom_get_data(k,KGEOM_POLYLINE_CONNECTED,&verts,&colors);
    if (colors == NULL)
       kfprintf(o,"\t\tcolor data is NOT present.\n");
    else
       kfprintf(o,"\t\tcolor data is present on a %s basis.\n",(layout == KPER_VERTEX ? "PER VERTEX" : "PER LINE"));
    
    if (do_loc)
       print_vectors(verts,nverts,3,"Vertices","index\tX\tY\tZ",o);
    
    if (do_color && colors)
       print_vectors(colors,(layout == KPER_VERTEX ? nverts : nverts-1),(has_alpha == TRUE ? 4 : 3),"Colors",(has_alpha == TRUE ? "index\tR\tG\tB\tA" : "index\tR\tG\tB"),o);
}

static void
print_triangles_dis(kobject k,
		    int n,
		    int prim_type,
		    int do_loc,
		    int do_color,
		    int do_norm_radii,
		    int do_texture,
		    int has_alpha,
		    kfile *o)
{
    int nverts;
    int status;
    int layout;
    int i,j;
    float *v=NULL,*norms=NULL,*c=NULL,*tc=NULL;

    kfprintf(o,"\n------- PRIMITIVE %d is KGEOM_TRIANGLES_DISJOINT <%d>\n",n,prim_type);
    kgeom_get_attribute(k,KGEOM_TRIANGLES_DISJOINT,KGEOM_NUMBER_VERTICES,&nverts);
    kfprintf(o,"\tand has %d vertices (%d triangles).\n",nverts, nverts/3);

    kgeom_get_data(k,KGEOM_TRIANGLES_DISJOINT,&v,&c,&norms,&tc);
    status = kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_LAYOUT,&layout);

    if (norms == NULL)
	kfprintf(o,"\t\tnormal data is NOT present.\n");
    else
	kfprintf(o,"\t\tnormal data is present on a %s basis.\n",(layout == KPER_VERTEX ? "PER VERTEX" : "PER FACE"));

    if (c == NULL)
	kfprintf(o,"\t\tcolor data is NOT present.\n");
    else
	kfprintf(o,"\t\tcolor data is present on a %s basis.\n",(layout == KPER_VERTEX ? "PER VERTEX" : "PER FACE"));

    if (tc == NULL)
	kfprintf(o,"\t\ttexture coordinate data is NOT present.\n");
    else
	kfprintf(o,"\ttexture coordinate data is present.\n");

    if ((do_loc) && (v != NULL))
	print_vectors(v,nverts,3,"Vertices","index\tX\tY\tZ",o);

    if ((do_norm_radii) && (norms != NULL))
	print_vectors(norms,(layout == KPER_VERTEX ? nverts : nverts/3),3,"Normals","index\tX\tY\tZ",o);

    if ((do_color) && (c != NULL))
	print_vectors(c,(layout == KPER_VERTEX ? nverts : nverts/3),(has_alpha == TRUE ? 4 : 3),"Colors",(has_alpha == TRUE ? "index\tR\tG\tB\tA" : "index\tR\tG\tB"),o);

    if (do_texture)
	kfprintf(o,"\tTexture Coordinates are not yet supported.\n");
	
}


static void
print_triangles_con(kobject k,
		    int n,
		    int prim_type,
		    int do_loc,
		    int do_color,
		    int do_norm_radii,
		    int do_texture,
		    int has_alpha,
		    kfile *o)
{ 
   int nverts,i,j;
    int layout;
    int offset;
    
    float *v=NULL,*c=NULL,*norm=NULL,*tc=NULL;

    kfprintf(o,"\n------- PRIMITIVE %d is KGEOM_TRIANGLES_CONNECTED <%d>\n",n,prim_type);
    kgeom_get_attribute(k,KGEOM_TRIANGLES_CONNECTED,KGEOM_NUMBER_VERTICES,&nverts);
    kfprintf(o,"\tand has %d vertices (%d triangles).\n",nverts,nverts-2);
    kgeom_get_data(k,KGEOM_TRIANGLES_CONNECTED,&v,&c,&norm,&tc);
    kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_LAYOUT,&layout);

    if (c==NULL)
        kfprintf(o,"\t\tcolor data is NOT present.\n");
    else
	kfprintf(o,"\t\tcolor data is present on a %s basis.\n",(layout == KPER_VERTEX ? "PER VERTEX" : "PER FACE"));

    if (norm==NULL)
	kfprintf(o,"\t\tnormal data is NOT present.\n");
    else
	kfprintf(o,"\t\tnormal data is present.\n");

    if (tc==NULL)
	kfprintf(o,"\t\ttexture coordinate data is NOT present.\n");
    else
	kfprintf(o,"\t\ttexture coordinate data is present.\n");

    /* print the vertices */
    if (do_loc)
	print_vectors(v,nverts,3,"Vertices","index\tX\tY\tZ",o);

    if (do_color)
    {
	if (c != NULL)	/* print the colors */
	    print_vectors(c,(layout == KPER_FACE ? nverts-2 : nverts),(has_alpha == TRUE ? 4 : 3),"Colors",(has_alpha == TRUE ? "index\tR\tG\tB\tA" : "index\tR\tG\tB"),o);
    }
}

static void
print_qmesh(kobject k, 
	    int n,
	    int prim_type,
	    int do_loc,
	    int do_color,
	    int do_norm_radii,
	    int do_texture,
	    int has_alpha,
	    kfile *o)
{
    int nverts,i,j;
    int usize,vsize;
    int layout;
    int offset;
    int status;
    
    float *v=NULL,*c=NULL,*norm=NULL,*tc=NULL;
    int has_loc = FALSE,has_color=FALSE,has_norms=FALSE,has_tc=FALSE,tc_size;
    
    kfprintf(o,"\n------- PRIMITIVE %d is KGEOM_QUADMESH <%d>\n",n,prim_type);
    status = kgeom_get_attribute(k,KGEOM_QUADMESH,KGEOM_QUADMESH_SIZE,&usize,&vsize);
    if (status)
	kfprintf(o,"\twhich has %d by %d vertices in U and V.\n",usize,vsize);
    else
	kfprintf(o,"\t\tError: can't get quadmesh size.\n");

    has_loc =  kgeom_get_data(k,KGEOM_QUADMESH_LOCATION_ALL,&v);
    has_color = kgeom_get_data(k,KGEOM_QUADMESH_COLOR_ALL,&c);
    has_norms = kgeom_get_data(k,KGEOM_QUADMESH_NORMAL_ALL,&norm);
    has_tc = kgeom_get_data(k,KGEOM_QUADMESH_TEXTURE_COORD_ALL,&tc);
    kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_TEXTURE_COORD_SIZE,&tc_size);
    
    kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_LAYOUT,&layout);

    kfprintf(o,"\n");

    if (!has_loc)
       kfprintf(o,"\t\tlocaton data is NOT present.\n");
    else
       kfprintf(o,"\t\tlocaton data is present.\n");

    if (!has_color)
       kfprintf(o,"\t\tcolor data is NOT present.\n");
    else
       kfprintf(o,"\t\tcolor data is present.\n");

    if (!has_norms)
       kfprintf(o,"\t\tnormal data is NOT present.\n");
    else
       kfprintf(o,"\t\tnormal data is present.\n");

    if (!has_tc)
       kfprintf(o,"\t\ttexture coordinate data is NOT present.\n");
    else
       kfprintf(o,"\t\ttexture coordinate data is present.\n");
       
    if (has_tc == 1)
	kfprintf(o,"\t\ttexture coord size is %d.\n",tc_size);
    
    if (has_loc)
    {
	if (do_loc)
	    print_vectors(v,usize*vsize,3,"Vertices","index\tX\tY\tZ",o);
    }
    if (has_color)
    {
	if (do_color)  /* assume for now per-vertex coloring */
	{
	    if (layout == KPER_VERTEX)
		print_vectors(c,usize*vsize,((has_alpha == 1) ? 4 : 3),"Colors",
			      ((has_alpha == 1) ? ("index\tR\tG\tB\tA") :
				("Colors",("index\tR\tG\tB"))),o);
	}
    }
    if (has_norms)
	if (do_norm_radii)
	    print_vectors(norm,usize*vsize,3,"Normals","index\tX\tY\tZ",o);
    
    if (has_tc)
    {
	if (do_texture)
	{
	    if (tc_size == 2)
		print_vectors(tc,usize*vsize,2,"Texture Coords","index\tU\tV",o);
	    else /* assume tc == 3 */
		print_vectors(tc,usize*vsize,3,"Texture Coords","index\tU\tV\tW",o);
	}
    }
}

static void
print_omesh(kobject k, 
	    int n,
	    int prim_type,
	    int do_loc,
	    int do_color,
	    int do_norm_radii,
	    int do_texture,
	    int has_alpha,
	    kfile *o)
{
    int nverts,i,j;
    int usize,vsize,wsize;
    int layout;
    int offset;
    int status;
    
    float *v=NULL,*c=NULL,*norm=NULL,*tc=NULL;
    int has_loc = FALSE,has_color=FALSE,has_norms=FALSE;
    
    kfprintf(o,"\n------- PRIMITIVE %d is KGEOM_OCTMESH <%d>\n",n,prim_type);
    status = kgeom_get_attribute(k,KGEOM_OCTMESH,KGEOM_OCTMESH_SIZE,&usize,&vsize,&wsize);
    if (status)
	kfprintf(o,"\twhich has %d, %d, and %d vertices in U,V and W.\n",usize,vsize,wsize);
    else
	kfprintf(o,"\t\tError: can't get octmesh size.\n");

    has_loc =  kgeom_get_data(k,KGEOM_OCTMESH_LOCATION_ALL,&v);
    has_color = kgeom_get_data(k,KGEOM_OCTMESH_COLOR_ALL,&c);
    has_norms = kgeom_get_data(k,KGEOM_OCTMESH_NORMAL_ALL,&norm);
    
    kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_LAYOUT,&layout);

    kfprintf(o,"\n");

    if (!has_loc)
       kfprintf(o,"\t\tlocaton data is NOT present.\n");
    else
       kfprintf(o,"\t\tlocaton data is present.\n");

    if (!has_color)
       kfprintf(o,"\t\tcolor data is NOT present.\n");
    else
       kfprintf(o,"\t\tcolor data is present.\n");

    if (!has_norms)
       kfprintf(o,"\t\tnormal data is NOT present.\n");
    else
       kfprintf(o,"\t\tnormal data is present.\n");
    
    if (has_loc)
    {
	if (do_loc)
	    print_vectors(v,usize*vsize*wsize,3,"Vertices","index\tX\tY\tZ",o);
    }
    if (has_color)
    {
	if (do_color)  /* assume for now per-vertex coloring */
	{
	    if (layout == KPER_VERTEX)
		print_vectors(c,usize*vsize*wsize,((has_alpha == 1) ? 4 : 3),"Colors",
			      ((has_alpha == 1) ? ("index\tR\tG\tB\tA") :
				("Colors",("index\tR\tG\tB"))),o);
	}
    }
    if (has_norms)
	if (do_norm_radii)
	    print_vectors(norm,usize*vsize*wsize,3,"Normals","index\tX\tY\tZ",o);
}

static void
print_unknown(kobject k, 
	      int n,
	      int prim_type,
	      int do_loc,
	      int do_color,
	      int do_norm_radii,
	      int do_texture,
	      int has_alpha,
	      kfile *o)
{
    kfprintf(o,"\tUnrecognized primitive in position #%d is of type %d.\n",n,prim_type);
}

static void
print_vectors(float *v,
	      int n,
	      int length,
	      char *string,
	      char *hdr_string,
	      kfile *f)
{
    int i,j,offset;
    
    kfprintf(f,"\n\t%s\n",string);
    kfprintf(f,"%s\n",hdr_string);
    offset = 0;
    for (i=0;i<n;i++)
    {
	kfprintf(f,"%d",i);
	for (j=0;j<length;j++)
	    kfprintf(f,"\t%g ",v[offset++]);
	kfprintf(f,"\n");
    }

}

static void
print_att(kobject k,
	  kfile *f)
{
    /**
      * this routine prints all the object-level attributes.
    **/
    
    int status,layout;
    int i,j;
    char *name=NULL;
/*    float *min=NULL,*max=NULL,*center=NULL,*rotvec=NULL,*scalevec=NULL; */
    float *min=NULL,*max=NULL,*center=NULL,*mx=NULL,*mx2;
    
    double ambient,diffuse,spec,spec_exp,metal;

    lset_common_geom_attributes(k);
    
    status = kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_NAME,&name);
    if (status == FALSE)
	kfprintf(f,"\tCan't get NAME attribute.  Maybe not a GEOM file?\n");
    else
       if (name == NULL)
	  kfprintf(f,"\tObject name is : <NULL>\n",name);
       else
	  kfprintf(f,"\tObject name is : <%s>\n",name);

    status = kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_BOUNDING_BOX,&min,&max);
    if (status == TRUE)
        kfprintf(f,"\tBounding box: %g,%g,%g <-> %g,%g,%g \n",min[0],min[1],min[2],max[0],max[1],max[2]);
    else
        kfprintf(f,"\tThis object has NO bounding box.\n");

    status = kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_CENTER,&center);
    if (status == TRUE)
	kfprintf(f,"\tCenter point: %g,%g,%g \n",center[0],center[1],center[2]);
    else
	kfprintf(f,"\tThis object has NO center point.\n");
#if 0
    /* obsolete, 1/27/95. wes */
    status = kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_ROTATE,&rotvec);
    if (status == TRUE)
	kfprintf(f,"\tRotate angles:\n\t\tX Axis: %g\n\t\tY Axis: %g\n\t\tZ Axis: %g \n",rotvec[0],rotvec[1],rotvec[2]);
    else
	kfprintf(f,"\tNo rotate attribute is present for this object.\n");

    status = kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_SCALE,&scalevec);
    if (status == TRUE)
	kfprintf(f,"\tScaling factors:\n\t\tX Axis: %g\n\t\tY Axis: %g\n\t\tZ Axis: %g \n",scalevec[0],scalevec[1],scalevec[2]);
    else
	kfprintf(f,"\tNo scale attribute is present for this object.\n");
#endif
    status = kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_MATRIX,&mx);
    if (status == TRUE)
    {
	kfprintf(f,"\tObject Transformation Matrix:\n");
	mx2 = mx;
	for (j=0;j<4;j++)
	{
	    kfprintf(f,"\t");
	    for (i=0;i<4;i++)
		kfprintf(f,"\t%g",*mx2++);
	    kfprintf(f,"\n");
	}
    }
    else
	kfprintf(f,"\tThis object has NO transformation matrix.\n");

    /* deal with colors, etc.*/
    status = kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_AMBIENT,&ambient);
    if (!status)
       kfprintf(f,"\tAmbient attribute is not defined.\n");
    else
       kfprintf(f,"\tAmbient:\t%g\n",ambient);

    status = kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_DIFFUSE,&diffuse);
    if (!status)
       kfprintf(f,"\tDiffuse attribute is not defined.\n");
    else
       kfprintf(f,"\tDiffuse:\t%g\n",diffuse);


    status = kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_SPECULAR_COEFFICIENT,&spec);
    if (!status)
       kfprintf(f,"\tSpecular Coefficient attribute is not defined.\n");
    else
       kfprintf(f,"\tSpecularC:\t%g\n",spec);

    status = kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_SPECULAR_EXPONENT,&spec_exp);
    if (!status)
       kfprintf(f,"\tSpecular Exponent attribute is not defined.\n");
    else
       kfprintf(f,"\tSpecularE:\t%g\n",spec_exp);

    status = kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_METAL,&metal);
    if (!status)
       kfprintf(f,"\tMetalness attribute is not defined.\n");
    else
      kfprintf(f,"\tMetalness:\t%g\n",metal);


    status = kgeom_get_attribute(k,KGEOM_OBJECT,KGEOM_LAYOUT,&layout);
    if (status)
	kfprintf(f,"\tLayout is <%s>\n",layout == KPER_VERTEX ? "PER_VERTEX" : "PER_FACE, or PER_SPHERE, etc.");
    else
	kfprintf(f,"\tLayout attribute is not defined.\n");
}
