 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for kgridder_2d
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lkgridder_2d
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "geometry.h"


/* -library_includes */
#include "ggridder2d.h"
static void compute_grid_extents_2d PROTO((float *,int,float *));
static void idsfft_wrapper PROTO((float *,int,int,float *,int,int,float *));
static void bivar_wrapper PROTO((float *,int,float *,float *,float *,float *,int,int));

/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lkgridder_2d
* 
*       Purpose: This should be a complete description that anyone
*                could understand;  it should have acceptable grammar
*                and correct spelling.
*
*         Input: inobj - kobject containing input scatter data.  the
*                   data will be interpreted as 1d scatter data, even if
*                   it's not 1d.  the only restrictions on the data are as
*                   follows:
*
*                   1. LOCATION_SIZE == 2 (x and y coords)  &&
*                      e == 1 (scalar value data).  in this case the
*                      output data will consist of w=xres, h=yres,
*                      d = 1, e = 1, t = 1, LOCATTION_SIZE = 2.
*
*                   2. LOCATION_SIZE == 3 (x,y,z coords) &&
*                      NO VALUE DATA!!.  the output data will consist
*                      of LOCATION_SIZE == 3, and the dimensions of
*                      the location segment will be w=xres,h=yres,
*                      d=1,t=1.
*
*                numerical method - this integer selects one of many
*                      methods for achieving 2d gridding.  at present,
*                      only inverse-distance and fourier analysis are
*                      supported.  there are some #define's in "geometry.h"
*
*                float grid_parms[] - an array of floating point values
*                      which hold parms specific to each of the
*                      numeric routines.  there is some coding dependancy
*                      between this routine and the calling routine.
*
*                int autocompute_grid_extents - a flag indicating whether or
*                      not this routine should compute the corners of the
*                      interpolation grid by finding the min/max in each of
*                      X and Y of the input.  if set to TRUE, the extents
*                      are computed for you, if FALSE, the following parameter
*                      holds the corners of the grid.
*
*                int xres,yres - integers indicating the number of samples
*                      in the output interpolation grid.  note that underlying
*                      numerical routines do support computing z=f(x,y) for
*                      arbitrary (x,y).  constructing a grid is a courtesy
*                      to the user.
*
*        Output: outobj - kobject containing gridded data.
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: Restrictions on data or input as applicable
*    Written By: Wes Bethel
*          Date: Aug 22, 1994
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */

int lggridder2d(kobject inobj,
		 kobject outobj,
		 int numerical_method,
		 float *grid_parms,
		 int autocompute_grid_extents,
		 float *grid_bounds,
		 int xres,
		 int yres)

/* -library_def_end */

/* -library_code */
{
    float *x,*y,*z,*in_loc=NULL;
    float loc_bounds[4];
    int ldims[4];
    float X,dX,Y,dY;
    int ix,iy,offset;
    int i,ndp;
    int has_loc;
    
    kpds_get_attribute(inobj,KPDS_LOCATION_SIZE,ldims,ldims+1,ldims+2,
		       ldims+3);

    for (i=0,ndp=1;i<3;i++)
	ndp *= ldims[i];

    kpds_set_attribute(inobj,KPDS_LOCATION_DATA_TYPE,KFLOAT);
    in_loc = (float *)kmalloc(sizeof(float)*ndp*ldims[3]);
    
    kpds_get_data(inobj,KPDS_LOCATION_ALL,in_loc);

    /**
      * get the output grid extents.
    **/

    if (autocompute_grid_extents)
	compute_grid_extents_2d(in_loc,ndp,loc_bounds);
    else
	kmemcpy(loc_bounds,grid_bounds,sizeof(float)*4);
    
    /**
      * construct the output grid.  we're going to cheat here by mallocing
      * enough space for contiguous storage of x,y, and z coords.  the reason
      * for doing this is so that later on when it's time to put_data(),
      * the location data, if 3d, can be done with a single call.  the same
      * memory area could also be put_data()`d to the value segment, if
      * necessary.
    **/
    x = (float *)kmalloc(sizeof(float)*xres*yres*3);
    y = x + xres*yres;
    z = y + xres*yres;

    X = loc_bounds[0];
    Y = loc_bounds[1];
    dX = (loc_bounds[2]-loc_bounds[0])/(xres-1);
    dY = (loc_bounds[3]-loc_bounds[1])/(yres-1);
    offset = 0;
    
    for (iy=0;iy<yres;iy++)
    {
	X = loc_bounds[0];
	for (ix=0;ix<xres;ix++)
	{
	    x[offset] = X;
	    y[offset] = Y;
	    offset++;
	    X += dX;
	}
	Y += dY;
    }

    /**
      * ok, now call the appropriate numerical routine.
    **/

    switch (numerical_method)
    {
    case INVERSE_DISTANCE_2D:
	bivar_wrapper(in_loc,ndp,grid_parms,grid_parms+1,grid_parms+2,x,xres,yres);
	break;
    case FOURIER_ANALYSIS_2D:
	i = (int)grid_parms[0];
	idsfft_wrapper(in_loc,ndp,i,x,xres,yres,loc_bounds);
	break;
    default:
	break;
    }

    kpds_create_location(outobj);
    kpds_create_value(outobj);
    
    kpds_set_attribute(outobj,KPDS_VALUE_DATA_TYPE,KFLOAT);
    kpds_set_attribute(outobj,KPDS_VALUE_SIZE,xres,yres,1,1,1);

    kpds_put_data(outobj,KPDS_VALUE_ALL,z); 
    
    kpds_set_attribute(outobj,KPDS_LOCATION_DATA_TYPE,KFLOAT);
    if (ldims[3] == 3) /* input data had 3 coord dimensions */
    {
	kpds_set_attribute(outobj,KPDS_LOCATION_SIZE,xres,yres,1,3);
	kpds_put_data(outobj,KPDS_LOCATION_ALL,x);
    }
    else /* input data had 2 coord dimensions */
    {
	kpds_set_attribute(outobj,KPDS_LOCATION_SIZE,xres,yres,1,2);
	kpds_put_data(outobj,KPDS_LOCATION_ALL,x);
	
    }
    kfree(x);
    
    return TRUE;
}


static void
compute_grid_extents_2d(float *in_loc,
			int ndp,
			float *extents)
{
    /**
      * extents[] will hold the min/max of the location points in in_loc[].
      * is is assumed that the points are in XXYYZZ order in in_loc[], and that
      * there are ndp of them.
      *
      * extents[0] = xmin
      * extents[1] = ymin
      * extenst[2] = xmax
      * extents[3] = ymax
    **/
    
    register float *f1;
    register float min,max;
    register int i;


    min = max = *in_loc;
    f1 = in_loc+1;
    for (i=1;i<ndp;i++,f1++)
    {
	if (*f1 < min)
	    min = *f1;
	else if (*f1 > max)
	    max = *f1;
    }
    
    extents[0] = min;  /* x min */
    extents[2] = max;  /* x max */
    
    min = max = *f1++;
    for (i=1;i<ndp;i++,f1++)
    {
	if (*f1 < min)
	    min = *f1;
	else if (*f1 > max)
	    max = *f1;
    }
    
    extents[1] = min;  /* x min */
    extents[3] = max;  /* x max */
}

static void
bivar_wrapper(float *in_loc,
	      int n_in_points,
	      float *radius,
	      float *exponent,
	      float *undefined_value,
	      float *out_loc,
	      int xres,
	      int yres)
{
    float *in_x,*in_y,*in_z;
    float *out_x,*out_y,*out_z;

    in_x = in_loc;
    in_y = in_x + n_in_points;
    in_z = in_y + n_in_points;

    out_x = out_loc;
    out_y = out_loc + xres*yres;
    out_z = out_y + xres*yres;
    
    bivar_work(in_x,in_y,in_z,n_in_points,out_x,out_y,out_z,xres,yres,
	       undefined_value,exponent,radius);
}

static void
idsfft_wrapper(float *in_loc,
	       int ndp,
	       int ncp,
	       float *out_loc,
	       int out_xsize,
	       int out_ysize,
	       float *bounds)
{
    float *xptr,*yptr,*zptr;
    float *out_xptr,*out_yptr,*out_zptr;
    int *iwork,i,nx,ny;
    float *work,*ty=NULL;
    int md=1;
    float Y,dY;

    /**
      * recompute the Y portion of the output grid.
    **/
    Y = bounds[1];
    dY = (bounds[3]-bounds[1])/(out_ysize-1);
    ty= (float *)kmalloc(sizeof(float)*out_ysize);
    for (i=0;i<out_ysize;i++,Y+=dY)
	ty[i] = Y;

    /**
      * setup consists of mallocing some work arrays.
    **/
    i = (31 > 27 + ncp) ? 31 : 27 + ncp;
    i = i * ndp + out_xsize * out_ysize;
    iwork = (int *)kmalloc(sizeof(int)*i*sizeof(int));
    kmemset(iwork,0,i);

    i = ndp * 5;
    work = (float *)kmalloc(sizeof(float)*i);
    kmemset(work,0,i*sizeof(float));

    xptr = in_loc;
    yptr = xptr + ndp;
    zptr = yptr + ndp;

    out_xptr = out_loc;
    out_yptr = out_xptr + out_xsize*out_ysize;
    out_zptr = out_yptr + out_xsize*out_ysize;

    nx = out_xsize;
    ny = out_ysize;

    IDSFFT(&md,&ncp,&ndp,xptr,yptr,zptr,&nx,&ny,
	   out_xptr,ty,out_zptr,iwork,work);
}

/* -library_code_end */
