 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */


/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>          Legend (Class) List Maintenance Routines
   >>>>
   >>>>  Private: spc_add_legend_node                                
   >>>>           spc_delete_legend_node                                
   >>>>           spc_add_legend_entry                                
   >>>>           spc_delete_legend_entry                                
   >>>>           spc_delete_entire_legend                                
   >>>>           spc_paint_from_img 
   >>>>           spc_polygon_from_img 
   >>>>           spc_initialize_legend_list                                
   >>>>   Static:
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/

#include "spectrum.h"

/*-----------------------------------------------------------
|
|  Routine Name: spc_add_legend_node
|
|       Purpose: Adds one node to the spc_legend_list.
|
|        Input: class_id   - id of class
|               red        - red value of class color
|               green      - green value of class color
|               blue       - blue value of class color
|               name       - name  assoc. w/ legend info
|               clusters   - array of cluster numbers
|               clusternum - # of cluster numbers (size of clusters array)
|               orig_type  - indicates if legend entry created directly
|                            by user, or created from legend file
|        Output: None
|       Returns: pointer to new node in the spc_legend_list.
|    Written By: Danielle Argiro
|          Date: May 12, 1993
| Modifications:
|
------------------------------------------------------------*/
LegendEntry *spc_add_legend_node(
    int   class_id,
    int   red,
    int   green,
    int   blue,
    char  *name,
    int   *clusters,
    int   clusternum,
    int   orig_type)
{
	char        *classname = NULL;
	LegendEntry *legend_ptr; 
	int         accepted;

	/*
	 * get a classname for the new legend entry from the user
	 */
        if (kstrcmp(name, "New Category") == 0)
        {
            while (classname == NULL)
	    {
		accepted = spc_get_new_classname(&classname);
	        if (!accepted) return(NULL);
	    }
        }
        else classname = kstrdup(name);

	legend_ptr = spc_create_legend_node(class_id, red, green, blue,
				            classname, clusters, clusternum, 
					    orig_type, TRUE);
 	kfree(classname);
        return(legend_ptr);
}

/*-----------------------------------------------------------
|
|  Routine Name: spc_create_legend_node
|
|       Purpose: Creates one node in the spc_legend_list.
|
|         Input: class_id   - id of class
|                red        - red value of class color
|                green      - green value of class color
|                blue       - blue value of class color
|                classname  - class name
|                clusters   - array of cluster numbers
|                clusternum - # of cluster numbers (size of clusters array)
|                orig_type  - indicates if legend entry created directly
|                             by user, or created from legend file
|                create_entry - pass TRUE if a legend entry in GUI needs
|                               to be created.
|        Output: 
|       Returns: The pointer to the legend entry created
|    Written By: Danielle Argiro
|          Date: May 12, 1993
| Modifications:
|
------------------------------------------------------------*/
LegendEntry *spc_create_legend_node(
    int   class_id,
    int   red,
    int   green,
    int   blue,
    char  *classname,
    int   *clusters,
    int   clusternum,
    int   orig_type,
    int   create_entry)
{
	int         i;
	LegendEntry *legend_ptr, *last;
	xvobject    entry_offset    = NULL;
	xvobject    contents_offset = NULL;

	/* 
	 * find the last legend pointer in the list, to use its legend
         *  entry backplane as the offset for the new legend entry backplane
	 */
	legend_ptr = spc_legend_list;
	if (legend_ptr != NULL)
	{
	    while (legend_ptr->next != NULL)
	        legend_ptr = legend_ptr->next;
	    entry_offset = legend_ptr->back;
	    contents_offset = legend_ptr->contents_obj;
	}

        /*
         * create the first node in the legend list
         */
        if (spc_legend_list == NULL)
        {
            spc_legend_list =  (LegendEntry *) kcalloc (1, sizeof(LegendEntry));
            spc_legend_list->class      = class_id;
            spc_legend_list->classname  = kstrdup(classname);
	    spc_legend_list->classname_token = kstring_to_token(classname);
            spc_legend_list->hidden     = FALSE;
            spc_legend_list->next       = NULL;
            spc_legend_list->redval     = red;
            spc_legend_list->greenval   = green;
            spc_legend_list->blueval    = blue;
            spc_legend_list->orig_type  = orig_type;
            spc_legend_classnum++;

	    if (create_entry)
	    {
                spc_create_entry_display(class_id, spc_legend_list, 
				         entry_offset);
                spc_create_classcontents_label(spc_legend_list, 
					       contents_offset);
	    }

            /* add associated clusters to legend node */
            for (i = 0; i < clusternum; i++)
                spc_add_cluster_to_class(spc_legend_list, clusters[i]);

            return(spc_legend_list);
        }

        /* create later nodes in the legend list */
        else
        {
            legend_ptr = (LegendEntry *) kcalloc (1, sizeof(LegendEntry));

            legend_ptr->class      = class_id;
            legend_ptr->classname       = kstrdup(classname);
            legend_ptr->classname_token = kstring_to_token(classname);
            legend_ptr->hidden     = FALSE;
            legend_ptr->next       = NULL;
            legend_ptr->redval     = red;
            legend_ptr->greenval   = green;
            legend_ptr->blueval    = blue;
            legend_ptr->orig_type  = orig_type;
            spc_legend_classnum++;

	    if (create_entry)
	    {
                spc_create_entry_display(class_id, legend_ptr, 
				         entry_offset);
                spc_create_classcontents_label(legend_ptr, contents_offset);
	    }

            /* add associated clusters to legend node */
            for (i = 0; i < clusternum; i++)
                spc_add_cluster_to_class(legend_ptr, clusters[i]);

            /* link new legend pointer into list */
            last = spc_legend_list;
            while (last->next != NULL)
                 last = last->next;
            last->next = legend_ptr;

            return(legend_ptr);
        }
}

/*-----------------------------------------------------------
|
|  Routine Name: spc_delete_legend_node
|
|       Purpose: Deletes one node from the spc_legend_list.
|
|         Input: legend_ptr - pointer to legend entry node to be deleted
|        Output: None
|       Returns: None
|    Written By: Danielle Argiro
|          Date: May 12, 1993
| Modifications:
|
------------------------------------------------------------*/
void spc_delete_legend_node(
    LegendEntry *legend_ptr)
{
        int         i;
        LegendEntry *last, *save;

        /* nothing to delete */
        if (spc_legend_list == NULL)
        {
            kerror(NULL, "spc_delete_legend_node",
		   "No legend entries (current classes) to delete!");
	    kinfo(KHOSTILE, "Pretty obvious, eh?  Or maybe not, to you!");
            return;
        }

        if (legend_ptr == spc_legend_list)
        {
            /* delete pointers from lookup list assoc. w/ each
               cluster in this class */
            for (i = 0; i < legend_ptr->clusternum; i++)
	    {
                spc_legend_lookup[legend_ptr->clusters[i]] = NULL;
		spc_classes[legend_ptr->clusters[i]] = -1;
	    }

            /* delete the node */
            save = spc_legend_list->next;
            if (spc_legend_list == spc_current_class)
                spc_set_current_class(NULL);
            kfree(spc_legend_list->clusters);
            kfree(spc_legend_list->classname);
            kfree(spc_legend_list);
            spc_legend_list = save;
        }
        else
        {
            /* delete pointers from lookup list assoc. w/ each
               cluster in this class */
            for (i = 0; i < legend_ptr->clusternum; i++)
	    {
                spc_legend_lookup[legend_ptr->clusters[i]] = NULL;
		spc_classes[legend_ptr->clusters[i]] = -1;
	    }

            /* find the node to be deleted */
            last = spc_legend_list;
            while (last->next != legend_ptr)
                last = last->next;

            /* delete the node */
            last->next = legend_ptr->next;
            if (legend_ptr == spc_current_class)
                spc_set_current_class(NULL);
            kfree(legend_ptr->clusters);
            kfree(legend_ptr->classname);
            kfree(legend_ptr);
        }
        spc_legend_classnum--;
}

/*-----------------------------------------------------------
|
|  Routine Name: spc_add_legend_entry
|
|       Purpose: Starts a new class by adding one node to the 
|                spc_legend_list.
|
|        Input: class_id   - id of class
|               red        - red value of class color
|               green      - green value of class color
|               blue       - blue value of class color
|               classname  - class name
|               clusters   - array of cluster numbers
|               clusternum - # of cluster numbers (size of clusters array)
|               orig_type  - indicates if legend entry created directly
|                            by user, or created from legend file
|        Output: None
|       Returns: pointer to new node in the spc_legend_list.
|    Written By: Danielle Argiro
|          Date: May 12, 1993
| Modifications:
|
------------------------------------------------------------*/
LegendEntry *spc_add_legend_entry(
    int   class_id,
    int   red,
    int   green,
    int   blue,
    char  *classname,
    int   *clusters,
    int   clusternum,
    int   orig_type)
{
	char        temp[KLENGTH];
        LegendEntry *legend_ptr;

	if (class_id < 0)
	    class_id = spc_get_unassigned_class_id();

	/*
	 * add the new legend node
	 */
        legend_ptr = spc_add_legend_node(class_id, red, green, blue, classname, 
					 clusters, clusternum, orig_type);

	if (legend_ptr == NULL) return(NULL);

	xvw_set_attribute(legend_ptr->colorbox, 
			  XVW_COLOR_COLOROBJ, spc_image);

        /*
         *  legend entry being created fresh ...
         *  default colorcell to color of first cluster
         */
        if (kstrncmp(legend_ptr->classname, "Un-named class_id category", 26) >= 0)
        {
            xvw_get_attributes(legend_ptr->colorbox,
                               XVW_COLORCELL_REDVAL,   &red,
                               XVW_COLORCELL_GREENVAL, &green,
                               XVW_COLORCELL_BLUEVAL,  &blue,
                               NULL);
            ksprintf(temp, " RGB values:  %d %d %d", red, green, blue);
        }
        /*
         *  legend entry being created from previously classed image,
         *  with perhaps a legend ...  set color to specified color
         */
        else
        {
            xvw_set_attributes(legend_ptr->colorbox,
                               XVW_COLORCELL_REDVAL,   legend_ptr->redval,
                               XVW_COLORCELL_GREENVAL, legend_ptr->greenval,
                               XVW_COLORCELL_BLUEVAL,  legend_ptr->blueval,
                               NULL);
            ksprintf(temp, " RGB values:  %d %d %d", legend_ptr->redval,
                     legend_ptr->greenval, legend_ptr->blueval);

        }
        xvw_set_attribute(legend_ptr->RGBobj, XVW_LABEL, temp);

	/* book-keeping */
        spc_set_current_class(legend_ptr);

	return(legend_ptr);
}


/*-----------------------------------------------------------
|
|  Routine Name: spc_delete_legend_entry
|
|       Purpose: Deletes an old class by deleting its node from the
|                spc_legend_list.
|
|         Input: legend_ptr - pointer representing class to be deleted
|        Output: None
|       Returns: None
|    Written By: Danielle Argiro
|          Date: June 12, 1993
| Modifications: Updated from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

void spc_delete_legend_entry(
    LegendEntry *legend_ptr)
{
	LegendEntry *tmp_legend_ptr;
	xvobject entry_offset, contents_offset;

        /*
         * need to re-layout all the legend entries to fill hole
         * about to be made in column of entries;  also need to
	 * re-layout all class contents labelstrings in the same way.
         * do them both at the same time.
         */
        tmp_legend_ptr = spc_legend_list;
        entry_offset = contents_offset = NULL;
        while (tmp_legend_ptr != NULL)
        {
            if (tmp_legend_ptr != legend_ptr)
            {
                xvw_set_attribute(tmp_legend_ptr->back, XVW_BELOW, 
				  entry_offset);
                xvw_set_attribute(tmp_legend_ptr->contents_obj, XVW_BELOW, 
				  contents_offset);
                entry_offset = tmp_legend_ptr->back;
                contents_offset = tmp_legend_ptr->contents_obj;
            }
            tmp_legend_ptr = tmp_legend_ptr->next;
        }

	/* restore the old colors of the pixels in the image */
	xvw_set_attribute(legend_ptr->colorbox, XVW_COLORCELL_RESTORE, TRUE);

        /* delete the legend entry widget set from legend display */
        xvw_destroy(legend_ptr->back);

        /* delete the class contents labelstring from statistics/info display */
        xvw_destroy(legend_ptr->contents_obj);

        /* delete the legend node set from global list */
        spc_delete_legend_node(legend_ptr);

}


/*-----------------------------------------------------------
|
|  Routine Name: spc_delete_entire_legend
|
|       Purpose: Deletes all classes from the legend list;
|                updates displayed image accordingly
|
|         Input: None
|        Output: None
|       Returns: None
|    Written By: Danielle Argiro
|          Date: Nov 4, 1993
| Modifications:
|
------------------------------------------------------------*/
void spc_delete_entire_legend(void)
{
        LegendEntry *legend_ptr, *save_next;

        /* delete legend list */
        legend_ptr = spc_legend_list;
        while (legend_ptr != NULL)
        {
	    xvw_set_attribute(legend_ptr->colorbox, 
			      XVW_COLORCELL_RESTORE, TRUE);
            xvw_destroy(legend_ptr->back);
            save_next = legend_ptr->next;
            spc_delete_legend_node(legend_ptr);
            legend_ptr = save_next;
        }

        /* no current entry */
        spc_set_current_class(NULL);

}


/*-----------------------------------------------------------
|
|  Routine Name: spc_paint_from_img
|
|       Purpose: This is the event handler to add/delete clusters
|		 to a class using the "paint" method.  It is only
|		 used when the [-method] command line arg is specified
|		 as "Paint". It works for painting in the main spectrum
|		 workspace, or painting in the zoom window, only
|		 (another event handler takes care of the same 
|                thing in the scatterplot).
|
|         Input: xvobject    - main spectrum workspace or zoom workspace object
|                client_data - unused
|                event       - the ButtonMotion or Doubleclick event
|        Output: dispatch    - whether to allow event to propogate; unused
|       Returns: none
|    Written By: Danielle Argiro
|          Date: Feb 20, 1995
| Modifications: 
|
------------------------------------------------------------*/

/* ARGSUSED */
void spc_paint_from_img(
    xvobject object, 
    kaddr    client_data, 
    XEvent   *event, 
    int      *dispatch)
{

	int             cluster; 
	double          tmp_double;

	/* no current class - return */
        if (spc_current_class == NULL) return;

	/*
	 *  see what cluster they have selected
	 */
	xvw_get_attribute(object, XVW_IMAGE_VALUE, &tmp_double);
	cluster = (int) tmp_double;

	/*
	 *  add or delete cluster from the class
	 */
	spc_add_or_delete_cluster(cluster);
}


/*-----------------------------------------------------------
|
|  Routine Name: spc_polygon_from_img
|
|       Purpose: This is the event handler to add/delete clusters
|		 to a class using the "polygon" method.  It is only
|		 used when the [-method] command line arg is specified
|		 as "Polygon". It works for polygon specification in the 
|		 main spectrum workspace, or painting in the zoom window, 
|		 only (another event handler takes care of the same 
|		 thing in the scatterplot).
|
|         Input: xvobject    - image workspace object
|                client_data - unused
|                event       - the ButtonMotion event
|        Output: dispatch    - whether to allow event to propogate
|       Returns: none
|    Written By: Danielle Argiro
|          Date: June 12, 1993
| Modifications: Updated from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

/* ARGSUSED */
void spc_polygon_from_img(
    xvobject object,
    kaddr    client_data,
    XEvent   *event,
    int      *dispatch)
{
	int       *unique_values;
	int       *dblclick = (int *) client_data;
	kobject   tmp_roi;
	int       i, unique_cluster_number;

	/*
	 * get polygon ROI from image
	 */
	xvw_get_attribute(object, XVW_IMAGE_ROI, &tmp_roi);
	if (tmp_roi == NULL) return;

	/*
         * from the ROI, get an integer array of unique clusters
         */
        unique_values = spc_find_unique_values_from_polygon(tmp_roi,
                                                        &unique_cluster_number);

        /*
         *  add or delete unique clusters from the polygon
         */
	for (i = 0; i < unique_cluster_number; i++)
            spc_add_or_delete_cluster(unique_values[i]);
}

/*-----------------------------------------------------------
|
|  Routine Name: spc_initialize_legend_list
|
|       Purpose: Initializes the spc_legend_list according to the entries 
|                in the spc_class column, when a new image is input.
|
|         Input: None
|        Output: None
|       Returns: None
|    Written By: Danielle Argiro
|          Date: Nov 6, 1993
| Modifications:
|
------------------------------------------------------------*/

void spc_initialize_legend_list(void)
{
	char        *classname;
	int         *clusters;
	int         i, j, clusternum, tmp_int;
	int         red, green, blue;
	LegendEntry *legend_ptr;
	xvobject    entry_offset;
	int         *class_array = NULL;
	int         num = 0;

	/*
	 * Go through the class column; for each non-minus-1 class id,
	 * add the class id to an array that can be sorted.
	 */
	for (i = 0; i < spc_map_colsize; i++)
        {
            if (spc_classes[i] != -1)
            {   
		if (karray_locate((char **) class_array, 
				  (kaddr) spc_classes[i], num) == -1)
		{
		    class_array = (int *) karray_add((char **) class_array, 
						   (kaddr) spc_classes[i], num);
		    num++;
		}
	    }
	}

	/*
	 *  sort the array of class ID's for which classes need to be created
	 */
        for (i = 0; i < num; i++)
        {
           for (j = 1; j < num - i; j++)
           {
              if (class_array[j-1] > class_array[j])
              {
                 tmp_int = class_array[j-1];
                 class_array[j-1] = class_array[j];
                 class_array[j] = tmp_int;
              }
           }
        }

	/*
	 *  for each class ID, create a legend entry.
	 */
	for (i = 0; i < num; i++)
	{
             legend_ptr = spc_find_class_from_id(class_array[i]);
	     if (legend_ptr == NULL)
	     {
	         clusters = spc_construct_legend_info(class_array[i],
                                   &clusternum, &red, &green, &blue,
                                   &classname);
                 (void) spc_add_legend_entry(class_array[i], red, green,
                                   blue, classname, clusters, clusternum,
                                   SPC_LGD_CREATED);
                 kfree(classname);
	     }
	}

	/*
	 * go through class column once again, so as to add in the
	 * clusters for each class.
	 */
	for (i = 0; i < spc_map_colsize; i++)
	{
	    if (spc_classes[i] != -1)
	    {
                legend_ptr = spc_find_class_from_id(spc_classes[i]);
		if (legend_ptr != NULL)
	            spc_add_cluster_to_class(legend_ptr, i);
	    }
        }

	spc_update_all_classcontents_labels();
}
