 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>      Utility Routines for Shuffling Classes
   >>>>
   >>>>  Private:
   >>>>           spc_delete_or_empty
   >>>>           spc_hide_or_show
   >>>>           spc_catch_all
   >>>>           spc_transfer_classes
   >>>>           spc_refresh_class_colors
   >>>>           spc_empty_class
   >>>>   Static:
   >>>>           hide_or_show_class
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "spectrum.h"

static void hide_or_show_class PROTO((LegendEntry *, int));

/*-----------------------------------------------------------
|
|  Routine Name: spc_delete_or_empty
|
|       Purpose: Prompts the user for the category(s) to be
|                either deleted or emptied, and either deletes
|                or empties them, depending on flag.
|
|         Input: flag - one of SPC_DELETE or SPC_EMPTY
|        Output: none
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: June 13, 1993
| Modifications: Updated from Khoros 1.0 (DA)
|
------------------------------------------------------------*/
int spc_delete_or_empty(
     int flag)
{
        char            **classnames; 
        char            *error_mesg_none, *prompt, *label;
	char            temp[KLENGTH], temp2[KLENGTH];
        int             i, size, num, cat_flag;
        LegendEntry     *legend_ptr;
        xvw_list_struct **list_return;

        char *error_mesgND = "No class categories to delete",
             *error_mesgNE = "No class categories to empty",
             *promptD1     = "Select classes to delete",
             *promptE1     = "Select classes to empty",
             *promptD2     = "Delete class category(s): ",
             *promptE2     = "Empty class category(s): ",
             *labelD       = "Current classes",
             *labelE       = "Current Classes w/ Clusters";

        /* delete */
        if (flag == SPC_DELETE)
        {
            error_mesg_none = error_mesgND;
            prompt = promptD1;
            label  = labelD;
            cat_flag = SPC_ALL;
        }

        /* empty */
        else if (flag == SPC_EMPTY)
        {
            error_mesg_none = error_mesgNE;
            prompt = promptE1;
            label  = labelE;
            cat_flag = SPC_NOTEMPTY;
        }

	/* mistake */
	else 
	{
	    errno = KINTERNAL;
	    kerror(NULL, "spc_delete_or_empty",
		   "Call to spc_delete_or_empty() with bogus flag");
	    kinfo(KHOSTILE, "The authors of this code were brain damaged.");
	    return(FALSE);
	}

        if (spc_legend_list == NULL)
        {
            kerror(NULL, "spc_delete_or_empty", error_mesg_none);
	    kinfo(KHOSTILE, "Use your brain (if you have one).");
            return(FALSE);
        }
        classnames = spc_create_strings_category_list(NULL, cat_flag, 
					          &size, NULL, 0);

	if (size == 0)
	{
	    kerror(NULL, "spc_delete_or_empty", error_mesg_none);
	    return(FALSE);
	}
        else if (size == 1)
        {
            legend_ptr = spc_find_legend_from_classname(classnames[0]);
            if (flag == SPC_DELETE)
	    {
		if (legend_ptr == NULL)
                {
                    kerror(NULL, "spc_delete_legend_entry", "No legend entry selected to delete!  Please select an entry for deletion by clicking the mouse in the color box of the desired item in the legend display.");
	            kinfo(KHOSTILE, "Try to think about what you're doing.  We know you're not used to it, but try anyway.");
                    return(FALSE);
                }

                ksprintf(temp, "Delete class category %s?", legend_ptr->classname);
                if (!(kprompt(KSTANDARD, "Ok", "Cancel", 0, temp)))
                    return(FALSE);
                spc_delete_legend_entry(legend_ptr);
	    }
            else if (flag == SPC_EMPTY)
                spc_empty_class(legend_ptr);
        }
        else
        {
            list_return = xvu_run_list_multsel_wait(classnames, size, prompt,
					            label, FALSE, FALSE, &num);
            if (list_return == NULL) return(FALSE);

            if (flag == SPC_DELETE)     
		ksprintf(temp2, promptD2);
            else if (flag == SPC_EMPTY) 
		ksprintf(temp2, promptE2);

            for (i = 0; i < num; i++)
            {
                ksprintf(temp, "%s", list_return[i]->string);
                if (i < num-1) 
		    kstrcat(temp, ", ");
                kstrcat(temp2, temp);
            }
            kstrcat(temp2, "?");
            if (!(kprompt(KSTANDARD, "Ok", "Cancel", 1, temp2)))
                return(FALSE);

            for (i = 0; i < num; i++)
            {
                legend_ptr = spc_find_legend_from_classname(list_return[i]->string);
                if (flag == SPC_DELETE)
                    spc_delete_legend_entry(legend_ptr);
                else if (flag == SPC_EMPTY)
                    spc_empty_class(legend_ptr);
            }
            kfree(list_return);
        }

        kfree(classnames);
	return(TRUE);
}



/*-----------------------------------------------------------
|
|  Routine Name: spc_hide_or_show
|
|       Purpose: Prompts the user for the category(s) to be
|                either hidden or shown, and either hides
|                or shows them, depending on flag.
|
|         Input: flag - one of SPC_HIDDEN or SPC_DISPLAYED
|        Output: none
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: June 13, 1993
| Modifications: Updated from Khoros 1.0 (DA)
|
------------------------------------------------------------*/
int spc_hide_or_show(
    int flag)
{
        int  i, size, num, cat_flag;
        LegendEntry *legend_ptr;
        xvw_list_struct **list_return;
        char **classnames;
        char *error_mesg_none, *error_mesg_all, *prompt, *label;
        char *error_mesgNH = "No class categories to hide.",
             *error_mesgNS = "No class categories to show.",
             *error_mesgAH = "All class categories already hidden.",
             *error_mesgAS = "All class categories already shown.",
             *promptH = "Select classes to hide",
             *promptS = "Select classes to display",
             *labelH  = "Currently Displayed Classes",
             *labelS  = "Currently Hidden Classes";

	/* hiding the class */
        if (flag == SPC_HIDDEN)
        {
            error_mesg_none = error_mesgNH;
            error_mesg_all  = error_mesgAH;
            prompt = promptH;
            label  = labelH;
            cat_flag = SPC_DISPLAYED;
        }

	/* showing the class */
        else if (flag == SPC_DISPLAYED)
        {
            error_mesg_none = error_mesgNS;
            error_mesg_all  = error_mesgAS;
            prompt = promptS;
            label  = labelS;
            cat_flag = SPC_HIDDEN;
        }

	/* internal error: bogus flag */
        else
        {
	    errno = KINTERNAL;
            kerror(NULL, "spc_delete_or_empty",
                   "Call to spc_delete_or_empty() with bogus flag");
	    kinfo(KHOSTILE, "The authors of this code were terminally confused.");
            return(FALSE);
        }

	/* user error: bogus attempt to hide or show */
        if (spc_legend_list == NULL)
        {
            kerror(NULL, "spc_hide_or_show", error_mesg_none);
	    kinfo(KHOSTILE, "You obviously don't have a clue what's going on.");
            return(FALSE);
        }

	/* create list of classes that can be hidden or shown */
        classnames = spc_create_strings_category_list("ALL Classes", cat_flag, 
						      &size, NULL, 0);
	/* user error */
        if (size == 0)
        {
            kerror(NULL, "spc_hide_or_show", error_mesg_all);
	    kinfo(KHOSTILE, "Perhaps you should go home and try again when you're not so stupid.");
            return(FALSE);
        }

	/* error - should'nt ever be only one since we included "All Classes" */
	else if (size == 1)
	{
	    kerror(NULL, "spc_hide_or_show", "internal error in Hide or Show");
	    return(FALSE);
	}

	/* 
	 * only one class; hide or show the one immediately 
	 */
        else if (size == 2)
        {
            legend_ptr = spc_find_legend_from_classname(classnames[1]);
	    hide_or_show_class(legend_ptr, flag);
        }

	/*
	 * more than one class to potentially be hidden or shown
	 */
        else
        {
	    /*
	     * prompt for class(es) including option to do them all
	     */
            list_return = xvu_run_list_multsel_wait(classnames, size, prompt, 
						    label, FALSE, FALSE, &num);
            if (list_return == NULL) return(FALSE);

	    /*
	     * user wants to hide/show all at once
	     */
	    if (kstrcmp(list_return[0]->string, "ALL Classes") == 0)
	    {
                for (i = 1; i < size; i++)
                {
                    legend_ptr = spc_find_legend_from_classname(classnames[i]);
	            hide_or_show_class(legend_ptr, flag);
                }
	    }

	    /*
	     * user wants to hide/show specific classes
	     */
	    else
	    {
                for (i = 0; i < num; i++)
                {
                    legend_ptr = spc_find_legend_from_classname(list_return[i]->string);
	            hide_or_show_class(legend_ptr, flag);
                }
                kfree(list_return);
            }
	}
        kfree(classnames);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: spc_catch_all
|
|       Purpose: dumps all unassigned clusters into a "catch-all" category.
|
|         Input: none
|        Output: none
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: June 13, 1993
| Modifications: Updated from Khoros 1.0 (DA)
|
------------------------------------------------------------*/


int spc_catch_all(void)
{
        int  i, *clusters;
        LegendEntry *legend_ptr;
        char *prompt = "Are you sure you want to dump all remaining clusters into a catch-all category?" ;

	/*
	 *  make sure they really want to do this 
	 */
        if (!(kprompt(KSTANDARD, "Yes", "Cancel", 1, prompt)))
            return(FALSE);

	/*
	 *  create the catch-all category with the first cluster to go in it
	 */
	clusters = (int *) kmalloc(sizeof(int));
	clusters[0] = 0;
        while (spc_legend_lookup[clusters[0]] != NULL)
           (clusters[0])++;

	legend_ptr = spc_find_legend_from_classname("Catch-All Category");
        if (legend_ptr == NULL)
            legend_ptr = spc_add_legend_entry(-1, 0, 0, 0, 
					      "Catch-All Category",
					      clusters, 1, SPC_LGD_CREATED);
	/*
	 *  dump all unassigned clusters into catch-all category
	 */
        for (i = 0; i < spc_map_rownum; i++)
        {
            if (spc_legend_lookup[i] == NULL)
            {
                spc_add_cluster_to_class(legend_ptr, i);
            }
        }

	/*
	 *  set the current class to the new catch-all category
	 */
	spc_set_current_class(legend_ptr);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: spc_transfer_classes
|
|       Purpose: Transfers the chosen category(s) into another
|                category.
|         Input: none
|        Output: none
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: June 13, 1993
| Modifications: Updated from Khoros 1.0 (DA)
|
------------------------------------------------------------*/
int spc_transfer_classes(void)
{
        int             i, j, k, src_size, dest_size, src_num, transfer_size;
        int             *transfer_clusters, total_clusters;
        char            **classnames, **destinations, **exclude_list;
        LegendEntry     *legend_ptr;
        xvw_list_struct **source_list_return;
        xvw_list_struct *dest_list_return;

        char *prompt1 = "Select Source Class(s)";
        char *label1  = "Current Class Categories";
        char *prompt2 = "Select Destination Class";
        char *label2  = "Possible Destination Classes";

        if (spc_legend_list == NULL)
        {
            kerror(NULL, "spc_transfer_classes",
		   "No class categories to transfer");
	    kinfo(KHOSTILE, "Only intelligent users should really attempt to use this application.");
            return(FALSE);
        }

        classnames = spc_create_strings_category_list(NULL, SPC_NOTEMPTY, 
						      &src_size, NULL, 0);
        if (src_size == 1)
        {
            kerror(NULL, "spc_transfer_classes",
		   "Cannot perform transfer with only one class category");
	    kinfo(KHOSTILE, "Are you familiar with the word, 'transfer'?  Can you pronounce it?.");
            return(FALSE);
        }

        /*
         *  create list of source classes to choose from (as many)
         */
        source_list_return = 
		xvu_run_list_multsel_wait(classnames, src_size, prompt1, 
					  label1, FALSE, FALSE, &src_num);
        if (source_list_return == NULL) return(FALSE);

        /*
         *  exclude source classes from choices of destination classes
         */
        exclude_list = (char **) kcalloc(1, sizeof(char *));
        for (i = 0; i < src_num; i++)
        {
            exclude_list[i] = kstring_copy(source_list_return[i]->string, NULL);
            exclude_list = (caddr_t *) 
			   krealloc(exclude_list, (i+2)*sizeof(char *));
        }

        /*
         *  create list of destination classes to choose from
         *  (only one, excluding source classes chosen)
         */
        destinations = spc_create_strings_category_list(NULL, SPC_ALL,
						        &dest_size,
                                                        exclude_list, src_num);
        dest_list_return = xvu_run_list_wait(destinations, dest_size, prompt2, 
					     label2, 0, FALSE);
        if (dest_list_return == NULL) return(FALSE);

        /*
         *  collect clusters to be transferred from list of source classes
         *  save clusters in an integer array
         */
        k = 0;
        transfer_clusters = (int *) kmalloc(sizeof(int));  
	transfer_size = 1;
        for (i = 0; i < src_num; i++)
        {
            legend_ptr = 
		spc_find_legend_from_classname(source_list_return[i]->string);

            transfer_clusters =  (int *) krealloc(transfer_clusters,
                                 (transfer_size+legend_ptr->clusternum) *
                                 sizeof(int));
            transfer_size += legend_ptr->clusternum;

            total_clusters = legend_ptr->clusternum;
            for (j = 0; j < total_clusters; j++)
            {
               transfer_clusters[k] = legend_ptr->clusters[0];
               spc_delete_cluster_from_class(legend_ptr, transfer_clusters[k++]);
            }
        }

        /*
         *  get clusters from integer array, and add them into chosen class
         */
        legend_ptr = spc_find_legend_from_classname(dest_list_return->string);
        for (i = 0; i < k; i++)
            spc_add_cluster_to_class(legend_ptr, transfer_clusters[i]);

	/*
	 *  set the current class to the destination class
	 */
	spc_set_current_class(legend_ptr);

	/*
	 *  update the "Class Contents" display
	 */
	spc_update_all_classcontents_labels();

        kfree(classnames);
        kfree(exclude_list);
        kfree(destinations);
        kfree(source_list_return);
        kfree(dest_list_return);

	return(TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: spc_empty_class
|
|       Purpose: Empties the class represented by the legend_ptr passed in
|
|         Input: legend_ptr - node representing class to be emptied
|        Output: none
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: June 13, 1993
| Modifications: Updated from Khoros 1.0 (DA)
|
------------------------------------------------------------*/
int spc_empty_class(
    LegendEntry *legend_ptr)
{
        int i, total_clusters;

        total_clusters = legend_ptr->clusternum;
        for (i = total_clusters-1; i >= 0; i--)
	{
	    if (!(spc_delete_cluster_from_class(legend_ptr, 
				legend_ptr->clusters[i])))
		return(FALSE);
	}
	legend_ptr->orig_type = SPC_LGD_CREATED;
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: spc_refresh_class_colors
|
|       Purpose: After the colormap of the image has changed because
|                map columns were changed, a function of map columns
|                was applied, or whatever, this routine refreshes the
|                colors of the currently assigned classes.
|
|         Input: none
|        Output: none
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: Nov 3, 1993
| Modifications:
|
------------------------------------------------------------*/

void spc_refresh_class_colors(void)
{
        LegendEntry *legend_ptr;

        legend_ptr = spc_legend_list;
        while (legend_ptr != NULL)
        {
            xvw_set_attribute(legend_ptr->colorbox,
                              XVW_COLORCELL_UPDATE,  TRUE);
            legend_ptr = legend_ptr->next;
        }
}

/*-----------------------------------------------------------
|
|  Routine Name: hide_or_show_class
|
|       Purpose: Given a legend pointer, hides or shows the class.
|         Input: legend_ptr - pointer to desired class
|                flag       - SPC_HIDDEN to hide, SPC_DISPLAYED to show.
|        Output: none
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: Feb 16, 1994
| Modifications:
|
------------------------------------------------------------*/
static void hide_or_show_class(
    LegendEntry *legend_ptr,
    int         flag)
{
	int i;

	/* sanity check */
	if (legend_ptr == NULL)
	    return;

	/*
	 * hide class by restoring original color
	 */
        if (flag == SPC_HIDDEN)
        {
	    legend_ptr->hidden = TRUE;
	    xvw_set_attribute(legend_ptr->colorbox,
                              XVW_COLORCELL_RESTORE, TRUE);
        }

	/*
	 * show class by activating class colors
	 */
        else
        {
	    legend_ptr->hidden = FALSE;
            xvw_set_attributes(legend_ptr->colorbox,
                               XVW_COLORCELL_REDVAL,   legend_ptr->redval,
                               XVW_COLORCELL_GREENVAL, legend_ptr->greenval,
                               XVW_COLORCELL_BLUEVAL,  legend_ptr->blueval,
                               NULL);
        }
}
