/*
 * Khoros: $Id$
 */
 
#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */
 

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            File Title: classifyClusters.c
   >>>>
   >>>>  Static:
   >>>>             base()
   >>>>             getCoocurranceCounts1()
   >>>>             getMahalanobisDist()
   >>>>             mergeTwoClusters()
   >>>>             updateCoocurranceCounts1()
   >>>>  Public:
   >>>>             classifyRemainingClusters()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "spectrum.h"

#define VALID_CLUSTER -1
#define VERY_INVALID_CLUSTER 100000

#define NOTASSIGNED -1

typedef struct {
  int first;
  int second;
  double score;
} CLUSTSCORE;

static int base PROTO(( int *, int ));
static int getCoocurranceCounts1 PROTO(( int, int, int, kobject, int **, int * ));
static double ** getMahalanobisDist PROTO(( int, int, double **, int *, double ***, double **** ));
static int mergeTwoClusters PROTO(( int, int, int, int, double **, int *, double **, double ***, double *** ));
static int updateCoocurranceCounts1 PROTO(( int, int, int, int **, int * ));


/*-----------------------------------------------------------
|
|  Routine Name: getMahalanobisDist - 
|
|       Purpose: This should be a complete description that anyone
|                could understand;  it should have acceptable grammar
|                and correct spelling.
|
|         Input: argument1 - explanation
|                argument2 - explanation
|                argument3 - explanation
|
|        Output: argument4 - explanation
|                argument5 - explanation
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Julio Barros
|          Date: Summer 94
| Modifications: Dave Modl, upgraded to Khoros 2.0
|
------------------------------------------------------------*/


static double **
getMahalanobisDist (
  int              dim,
  int              num_entries,
  double            **means,
  int              *count,
  double            ***cov_upper,
  double            ****cov_cfactors
  )
{

/* Arguments are */
/* Problem dimension */
/* Number of entries in the table */
/* Mean vectors */
/* Number of elements */
/* Upper triangular covariance matrix */
/* Mapping for the cluster reduction */
/* Flag to delete clusters with non-PD matrices */
/* Cholesky factorization of cov. matrices */

  kstring          routine = "getMahalanobisDist()";

  int i, j, k;			/* Loop control */
  double **sq_dists;		/* Squared distances */
  double max;			/* Maximum starting distance */

  
  double k1, k2;			/* Mid-calculation values */
  double alpha_1;			/* Mid-calculation values */

  double *tmp_vec1;		/* Temporary */
  double *tmp_vec2;		/* Temporary */
  double tmp_val;			/* Temporary */


  /*  FACTOR THE COVARIANCE MATRICES (CHOLESKY FACTORIZATION)  */
  /*  INITIALIZE ALL CLUSTERS TO VALID OR INVALID  */

  sq_dists = SM_get_matrix ( num_entries );

  *cov_cfactors = (double ***) kcalloc ( num_entries, sizeof (double **) );
  if ( *cov_cfactors == NULL ) {
    kerror(NULL, routine, "MEMORY ALLOCATION FAILURE\n");
    return NULL;
  }

  for ( i = 0 ; i < num_entries ; i ++ ) {
    (*cov_cfactors) [i] = SM_get_matrix ( dim );
    SM_copy_matrix ( dim, cov_upper [i], (*cov_cfactors) [i] );
    if ( ! SM_cfactor ( dim, (*cov_cfactors) [i] ) ) {
      if (i != 0)
          kinfo(KVERBOSE, "WARNING: matrix not pd %d\n",i);
      sq_dists [i][i] = VERY_INVALID_CLUSTER;
    } else {
      sq_dists [i][i] = VALID_CLUSTER;
    }
  }
  
  tmp_vec1 = (double *) kcalloc ( dim, sizeof(double) );
  if ( tmp_vec1 == NULL ) {
    kerror(NULL, routine, "MEMORY ALLOCATION FAILURE\n");
    return NULL;
  }
  
  tmp_vec2 = (double *) kcalloc ( dim , sizeof(double) );
  if ( tmp_vec2 == NULL ) {
    kerror(NULL, routine, "MEMORY ALLOCATION FAILURE\n");
    return NULL;
  }
  
  /*  INITIALIZE TABLE OF MERGING CRITERION VALUES  */
  
  max = 0.0;
  
  for ( i = 0 ; i < num_entries ; i ++ ) {
    for ( j = i+1 ; j < num_entries ; j ++ ) {
      if ( (sq_dists [j][j] == VALID_CLUSTER) && 
		(sq_dists [i][i] == VALID_CLUSTER) ) {

		for ( k = 0 ; k < dim ; k ++ ) 
		  tmp_vec1 [k] = tmp_vec2 [k] = means [i][k] - means [j][k];
	
      /*+++
      kprintf( "------ cov_cfactors[j=%d] ---------\n", j );
      for( jj = 0; jj < dim; jj ++ ) {
        for( kk = 0; kk < dim; kk++ ) {
          if (kk >= jj)
            kprintf("%9.5f ",(*cov_cfactors)[j][jj][kk]);
          else
            kprintf("          ");
        }
        kprintf("\n");
      }
      kprintf("------\n");
      +++*/

		SM_csolve ( dim, (*cov_cfactors) [j], tmp_vec1 );

        /*+++
        kprintf("------- i:%d  j:%d ------\n", i, j );
        kprintf("----- tmp_vec1: " );
        for( kk = 0; kk < dim; kk++ ) {
            kprintf("%9.5f ",tmp_vec1[kk]);
        }
        kprintf("\n");
        kprintf("----- tmp_vec2: " );
        for( kk = 0; kk < dim; kk++ ) {
            kprintf("%9.5f ",tmp_vec2[kk]);
        }
        kprintf("\n\n");
        +++*/

		k1 = dot_product ( tmp_vec1, tmp_vec2, dim );
	
		for ( k = 0 ; k < dim ; k ++ ) 
		  tmp_vec1 [k] = tmp_vec2 [k] ;
	
      /*+++
      kprintf( "------ cov_cfactors[i=%d] ---------\n", i );
      for( jj = 0; jj < dim; jj ++ ) {
        for( kk = 0; kk < dim; kk++ ) {
          if (kk >= jj)
            kprintf("%9.5f ",(*cov_cfactors)[i][jj][kk]);
          else
            kprintf("          ");
        }
        kprintf("\n");
      }
      kprintf("------\n\n\n");
      +++*/

		SM_csolve ( dim, (*cov_cfactors) [i], tmp_vec1 );

        /*+++
        kprintf("----- tmp_vec1: " );
        for( kk = 0; kk < dim; kk++ ) {
            kprintf("%9.5f ",tmp_vec1[kk]);
        }
        kprintf("\n");
        kprintf("----- tmp_vec2: " );
        for( kk = 0; kk < dim; kk++ ) {
            kprintf("%9.5f ",tmp_vec2[kk]);
        }
        kprintf("\n\n");
        +++*/

		k2 = dot_product ( tmp_vec1, tmp_vec2, dim );
	
		tmp_val = k1 * k2;
		if ( tmp_val < 0.0 ) {
   		   kerror(NULL, routine,
             "ERROR:  sqrt domain error\ni = %d   j = %d\nk1 = %f   k2 = %f\n",
             i, j, k1, k2 );
		  return NULL;
		}
		tmp_val = (double) sqrt ( (double) tmp_val );
		alpha_1 = ( tmp_val - k2 ) / ( k1 - k2 );
		sq_dists [i][j] = alpha_1 * alpha_1 * k1;
		if ( sq_dists [i][j] > max ) max = sq_dists [i][j] ;
	
      }
      else 
		sq_dists[i][j] = HUGE_VAL;
      
    }
  }
  return sq_dists;
}



/*-----------------------------------------------------------
|
|  Routine Name: getCoocurranceCounts1 - 
|
|       Purpose: This function implements the eight connected
|                neighbor tally count.
|
|         Input: numRows  - # of rows in the image to be processed;
|                numCols  - # of columns in the image to be processed;
|                numClust - # of clusters in the image to be processed;
|                image    - the image to be processed;
|                com      - (OUTPUT) co-occurrence matrix to be filled;
|                cc       - (OUTPUT) cluster count array to be filled;
|
|        Output: none;
|                
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Julio Barros
|          Date: Summer 94
| Modifications: Converted to Khoros 2.0 - Dave Modl, Aug 94
|
------------------------------------------------------------*/

static int
getCoocurranceCounts1 (
  int           numRows,
  int           numCols,
  int           numClust,
  kobject       image,
  int           **com,
  int           *cc
  )
{
  kstring       routine  = "getCoocurranceCounts1()";

  int            i,j,k;
  int         *  holdPrev;
  int         *  targetPrev;
  int         *  targetCurr;
  int         *  targetNext;
  int         *  line1;
  int         *  line2;
  int         *  line3;
  int            st, ot;
  int            invalidCluster;

    /*
     *  Make sure the function parameters are valid.
     */
  if( cc == NULL || com == NULL || image == NULL ) {
    kerror( NULL, routine, "Null function parameter(s) found." );
    return FALSE;
  }

    /*
     *  Setup to process the data found in the image.
     */
  /* Initialize the cooccurrence matrix */
  for (i=0;i<numClust;i++)
    for (j=0;j<numClust;j++)
      com[i][j] = 0; 

  /* Initialize the cluster count */
  for (k=0;k<numClust;k++){
    cc[k] = 0;
  }

    /*
     *  Create a set of three scan lines to use to
     *  parse through the data in the image.
     */
  invalidCluster = numClust;

    /*
     *  Allocate space to hold three lines with extra space
     *  on either side to use in the neighbor tracking.
     */
  line1 = (int *) kcalloc( numCols+2, sizeof(int) );
  line2 = (int *) kcalloc( numCols+2, sizeof(int) );
  line3 = (int *) kcalloc( numCols+2, sizeof(int) );

    /*
     *  Point the targets to the second location in for the
     *  lines created above.  This will allow us to compute
     *  with 'normal' indices within the targets but have the
     *  buffer spaces available to simplify the loop that does
     *  the cluster data analysis.
     */
  targetPrev = &line1[1];
  targetCurr = &line2[1];
  targetNext = &line3[1];

    /*
     *  Load the first line (also where the targetPrev points to)
     *  with invalid cluster values that will point to an unused
     *  location in the cluster count array and cooccurrence matrix.
     */
  for( j = 0; j < numCols+2; j++ ) {
    line1[j] = invalidCluster;
  }

    /*
     *  Go get the first line of data from the image and load it
     *  into the current target line.
     */
  kpds_get_data( image, KPDS_VALUE_LINE, targetCurr );
  targetCurr[-1] = targetCurr[numCols] = invalidCluster;

  /* go through the entire image */
    /*
     *  Use lines from the image value data three at a time
     *  to compute the neighbor coocurr numbers.
     */
  for( i=0; i<numRows-1; i++ ) {

    kpds_get_data( image, KPDS_VALUE_LINE, targetNext );
    targetNext[-1] = targetNext[numCols] = invalidCluster;

      /*
       *  Now, process cluster targets and neighbors to update
       *  the cluster count vector and the cooccurrence matrix.
       */
    for( j=0; j<numCols; j++ ) {

      st = targetCurr[j];	/* Set the cluster location. */
      cc[st]++;				/* increment the cluster count */

      /*
       *  For each neighbor (value) increment the corresponding entry
	   *  in the coocurr matrix.  *
       *      -1   0  +1
       *       ---------
       *  -1 | n | n | n |
       *     | --------- |    n = neighbor cell (8 of them);
       *   0 | n | T | n |    T = target cell;
       *     | --------- |
       *  +1 | n | n | n |
       */

      ot = targetPrev[j-1];  /*  Upper Left Neighbor   */
      com[st][ot]++;

      ot = targetPrev[j];    /*  Upper Center Neighbor */
      com[st][ot]++;

      ot = targetPrev[j+1];  /*  Upper Right Neighbor  */
      com[st][ot]++;

      ot = targetCurr[j-1];  /*  Left Neighbor         */
      com[st][ot]++;

      ot = targetCurr[j+1];  /*  Right Neighbor        */
      com[st][ot]++;

      ot = targetNext[j-1];  /*  Lower Left Neighbor   */
      com[st][ot]++;

      ot = targetNext[j];    /*  Lower Center Neighbor */
      com[st][ot]++;

      ot = targetNext[j+1];  /*  Lower Right Neighbor  */
      com[st][ot]++;
    }

    holdPrev   = targetPrev;
    targetPrev = targetCurr;
    targetCurr = targetNext;
    targetNext = holdPrev;
  }


    /*
     *  Now do the last line in the image.
     */
  for( j=0; j<numCols; j++ ) {

    st = targetCurr[j];    /* Set the cluster location.   */
    cc[st]++;              /* increment the cluster count */

    ot = targetPrev[j-1];  /*  Upper Left Neighbor   */
    com[st][ot]++;

    ot = targetPrev[j];    /*  Upper Center Neighbor */
    com[st][ot]++;

    ot = targetPrev[j+1];  /*  Upper Right Neighbor  */
    com[st][ot]++;

    ot = targetCurr[j-1];  /*  Left Neighbor         */
    com[st][ot]++;

    ot = targetCurr[j+1];  /*  Right Neighbor        */
    com[st][ot]++;

    /* +++ Don't have to do these neighbors +++
       ++++++++++++++++++++++++++++++++++++++++
    ot = targetNext[j-1];
    com[st][ot]++;

    ot = targetNext[j];
    com[st][ot]++;

    ot = targetNext[j+1];
    com[st][ot]++;
       ++++++++++++++++++++++++++++++++++++++++
    +++ */
  }

  kfree( line1 );
  kfree( line2 );
  kfree( line3 );

  return TRUE;
}



/*-----------------------------------------------------------
|
|  Routine Name: updateCoocurranceCounts1 - 
|
|       Purpose: For the tally method it is easy to update the com.
|                All that needs to be done is to add the rows and colums
|                of the merging clear one.
|
|         Input: argument1 - explanation
|                argument2 - explanation
|                argument3 - explanation
|
|        Output: argument4 - explanation
|                argument5 - explanation
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Julio Barros
| Modifications: Dave Modl, upgraded to Khoros 2.0
| Modifications:
|
------------------------------------------------------------*/

static int
updateCoocurranceCounts1(
  int new,
  int old,
  int numClust,
  int **com,
  int *cc
  )
{
  kstring        routine = "updateCoocurranceCounts1()";

  int i;
  
  /* Transfer the cluster counts */
  cc[new] += cc[old];
  cc[old] = 0;

  /* transfer the coocurrance counts (rows and cols)*/
  for( i=0; i<numClust; i++ ) {
    com[new][i] += com[old][i];
    com[old][i] = 0;
  }

  for( i=0; i<numClust; i++ ) {
    com[i][new] += com[i][old];
    com[i][old] = 0;
  }

  return TRUE;
}



/*-----------------------------------------------------------
|
|  Routine Name: base - 
|
|       Purpose: Used to find equivelant clusters.  The idea is
|                that if entry i has a value, j != i, it has been
|                merged with another cluster.  j is the cluster it
|                was merged with.  This function will follow the
|                chain and return the final self equivilant cluster.
|
|         Input: argument1 - explanation
|                argument2 - explanation
|                argument3 - explanation
|
|        Output: argument4 - explanation
|                argument5 - explanation
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Julio Barros
|          Date: Summer 94
| Modifications: Dave Modl, upgraded to Khoros 2.0
|
------------------------------------------------------------*/

static int
base(
  int          * ce,
  int            i
  )
{
  kstring        routine = "base()";
 
  while (i != ce[i])
    i = ce[i];
  return i;
}

/*-----------------------------------------------------------
|
|  Routine Name: mergeTwoClusters - 
|
|       Purpose: This should be a complete description that anyone
|                could understand;  it should have acceptable grammar
|                and correct spelling.
|
|         Input: argument1 - explanation
|                argument2 - explanation
|                argument3 - explanation
|
|        Output: argument4 - explanation
|                argument5 - explanation
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Julio Barros
|          Date: Summer 94
| Modifications: Dave Modl, upgraded to Khoros 2.0
|
------------------------------------------------------------*/


static int
mergeTwoClusters(
  int              row,
  int              col,
  int              numBands,
  int              numClust,
  double            **clasDist,
  int              *count,
  double            **means,
  double            ***cov_upper,
  double            ***cov_cfactors
  )
{
  kstring          routine = "mergeTwoClusters()";

  int i,k;
  double k1, k2;			/* Mid-calculation values */
  double alpha_1;			/* Mid-calculation values */
  double tmp_val;			/* Temporary */
  double *tmp_vec1;
  double *tmp_vec2;

  tmp_vec1 = (double *) kcalloc (numBands,sizeof(double) );
  tmp_vec2 = (double *) kcalloc ( numBands , sizeof(double) );

  if (tmp_vec1 == NULL || tmp_vec2 == NULL){
    kerror(NULL, routine, "Memory allocation failure. tmp_vecx\n");
    return FALSE;
  }

  /*  MERGE CLUSTERS  */
  SM_comb_stats ( numBands, count[row], count[col], &(count[row]),
		 means[row], means[col], means[row], cov_upper[row], 
		 cov_upper[col], cov_upper[row] );

  /*  DELETE CLUSTER BY SETTING DIAGONAL ELEMENT > 0  */
  clasDist [col][col] = (double) row;    /* annexed by "row" */

  /*  ANYTHING PREVIOUSLY ANNEXED BY "col" MUST BE CHANGED  */
  for ( k = 0 ; k < numClust ; k ++ ) {
    if ( clasDist [k][k] == (double) col ) {
      clasDist [k][k] = (double) row ;
    }
  }
    
  /*  RE-FACTOR THE NEW COVARIANCE MATRIX  */
    
  SM_copy_matrix ( numBands, cov_upper [row], cov_cfactors [row] );
  if ( ! SM_cfactor ( numBands, cov_cfactors [row] ) ) {
    /* kerror(NULL, routine, "ERROR matrix %d is not pd", row); */
    kwarn(NULL, routine, "Not all clusters could not be assigned to a class");
    return FALSE;
  }
    
  /*  CALCULATE NEW VALUES  */
    
  for ( i = 0 ; i < row ; i ++ ) {
    if ( clasDist [i][i] == VALID_CLUSTER ){
	
      for ( k = 0 ; k < numBands ; k ++ ) 
	tmp_vec1 [k] = tmp_vec2 [k] = 
	  means [i][k] - means [row][k];
	
      SM_csolve ( numBands, cov_cfactors [row], tmp_vec1 );
      k1 = dot_product ( tmp_vec1, tmp_vec2, numBands );
	
      for ( k = 0 ; k < numBands ; k ++ ) 
	tmp_vec1 [k] = tmp_vec2 [k] ;
	
      SM_csolve ( numBands, cov_cfactors [i], tmp_vec1 );
      k2 = dot_product ( tmp_vec1, tmp_vec2, numBands );
	
      tmp_val = k1 * k2;
      if ( tmp_val < 0.0 ) {
        kerror(NULL, routine,
             "ERROR:  sqrt domain error\ni = %d   j = %d\nk1 = %f   k2 = %f\n",
             i, row, k1, k2 );
	    return FALSE;
      }
      tmp_val = (double) sqrt ( (double) tmp_val );
      alpha_1 = ( - k2 + tmp_val ) / ( k1 - k2 );
      clasDist [i][row] = alpha_1 * alpha_1 * k1;
    }
  }
    
  for ( i = row+1 ; i < numClust ; i ++ ) {
    if ( clasDist [i][i] == VALID_CLUSTER ){
	
      for ( k = 0 ; k < numBands ; k ++ ) 
	tmp_vec1 [k] = tmp_vec2 [k] = 
	  means [i][k] - means [row][k];
	
      SM_csolve ( numBands, cov_cfactors [row], tmp_vec1 );
      k1 = dot_product ( tmp_vec1, tmp_vec2, numBands );
	
      for ( k = 0 ; k < numBands ; k ++ ) 
	tmp_vec1 [k] = tmp_vec2 [k] ;
	
      SM_csolve ( numBands, cov_cfactors [i], tmp_vec1 );
      k2 = dot_product ( tmp_vec1, tmp_vec2, numBands );
	
      tmp_val = k1 * k2;
      if ( tmp_val < 0.0 ) {
        kerror(NULL, routine,
             "ERROR:  sqrt domain error\ni = %d   j = %d\nk1 = %f   k2 = %f\n",
             i, row, k1, k2 );
	    return FALSE;
      }
      tmp_val = (double) sqrt ( (double) tmp_val );
      alpha_1 = ( - k2 + tmp_val ) / ( k1 - k2 );
      clasDist [row][i] = alpha_1 * alpha_1 * k1;
	
    }
  }

  kfree(tmp_vec1);
  kfree(tmp_vec2);

  return TRUE;
}


/*-----------------------------------------------------------
|
|  Routine Name: classifyRemainingClusters - 
|
|       Purpose: This should be a complete description that anyone
|                could understand;  it should have acceptable grammar
|                and correct spelling.
|
|         Input: argument1 - explanation
|                argument2 - explanation
|                argument3 - explanation
|
|        Output: argument4 - explanation
|                argument5 - explanation
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Julio Barros
|          Date: Summer 94
| Modifications: Dave Modl, upgraded to Khoros 2.0
|
------------------------------------------------------------*/

int *
classifyRemainingClusters(
  kobject src_image,
  int numRows,
  int numCols,
  int numBands,
  int numClust,
  int *count,
  double **means,
  double ***cov_upper,
  int *oldClass,
  int *newClass
  )
{
  kstring           routine = "classifyRemainingClusters()";

  CLUSTSCORE maxClustScore;
  int i,j;
  double ***cov_cfactors;	/* Cholesky factorization of cov. matrices */
  int *cc = NULL;			/* Class count */
  double **clasDist = NULL;	/* Distances between clusters (upper triag) */
  int **com;				/* Co-occurance matrix */
  int *tempClass = NULL;
  int mergesLeft;
  

  cc = (int *) kcalloc(numClust,sizeof(int));
  com = (int **) kcalloc(numClust,sizeof(int *));
  tempClass = (int *) kcalloc(numClust,sizeof(int));
  if( newClass == NULL ) {
    newClass = (int *) kcalloc(numClust,sizeof(int));
  }

  if (cc == NULL || com == NULL || tempClass == NULL || newClass == NULL){
    kerror(NULL, routine, "MEMORY ALLOCATION FAILURE\n");
    return NULL;
  }

  for (i=0;i<numClust;i++){
     /*
      *  Allocate memory for the cooccurrence matrix (with a twist)
      *  - need to add one to the length so that we have room for
      *    the dummy cluster value that will be generated as a
      *    'catchall' for the cooccurrence counts that are on the
      *    fringe of the image being processed.
      *  This is somewhat sloppy, but it allows us to 'motor'
      *  through the image without extraneuos testing inside
      *  the 'for' loops within the getCoocurranceCounts1() function.
      */
    com[i] = (int *) kcalloc(numClust+1,sizeof(int));
    if (com[i] == NULL){
      kerror(NULL, routine, "MEMORY ALLOCATION FAILURE\n");
      return NULL;
    }
  }

  /* Calculate the cluster distances */
  clasDist = getMahalanobisDist( numBands, numClust, means,
				                 count, cov_upper, &cov_cfactors);

  kpds_set_attribute( src_image, KPDS_VALUE_DATA_TYPE, KINT );
  kpds_set_attribute( src_image, KPDS_VALUE_POSITION, 0, 0, 0, 0, 0);

  getCoocurranceCounts1( numRows, numCols, numClust, src_image, com, cc);

  /* Play with the class lists and Merge initial seed points */
  for (i=0;i<numClust;i++)
    newClass[i] = i;

  /* After this loop is run the newclass list will look like a
     finished new class list.  That is, unmerged clusters will have
     their entry number in their entries and merged clusters will have
     the smaller entry number in their entries. */

  for (i=1;i<numClust;i++)	/* Start from the bottom */
    if (oldClass[i] != NOTASSIGNED && cc[i] > 0) /* if it is assigned */
      for (j=i+1;j<numClust;j++)	/* Loot at the rest */
	/* Find others assigned to the same label*/
	if (oldClass[j] == oldClass[i] && cc[j] > 0){ 
	  newClass[j] = i;	/* Change their entries in the new class list */
	  mergeTwoClusters( i, j, numBands, numClust, clasDist, count,
			            means, cov_upper, cov_cfactors );
	  updateCoocurranceCounts1( i, j, numClust, com, cc );
	}
  

  /* Merge the remaining clusters */
  mergesLeft = 1;
  while (mergesLeft){ 
    maxClustScore.score = -1;
    mergesLeft = 0;

    /* Calculate the cluster pair scores based on cooccurance and dist */
    for (i=1;i<numClust;i++)
      if (cc[i] > 0){
	for (j=i+1;j<numClust;j++)
	  if ((cc[j] > 0) && (i != j) && 
	      ((oldClass[i]==NOTASSIGNED) || (oldClass[j]==NOTASSIGNED)) &&
	      (com[i][j] > 0) && (clasDist[i][j] > 0)){
	    double temp = (double) com[i][j] / (double) cc[i] * 
	      (double) com[j][i] / (double) cc[j] *  1 /clasDist[i][j];

	    mergesLeft++;

	    if (temp > maxClustScore.score){
	      maxClustScore.first = i;
	      maxClustScore.second = j;
	      maxClustScore.score = temp;
	    }
	  }
      }

    /* Merge the two clusters and fix up the matrices */
    if (mergesLeft){
      /*+++
      kprintf("Picked %d and %d at %f\n",
	     maxClustScore.first,maxClustScore.second,maxClustScore.score); 
      kfflush(kstdout);
      +++*/

      /* watch for that mergining into an unsassinged bug */
      if (oldClass[maxClustScore.first] == NOTASSIGNED)
	oldClass[maxClustScore.first] = oldClass[maxClustScore.second];

      if (oldClass[maxClustScore.second] == NOTASSIGNED)
	oldClass[maxClustScore.second] = oldClass[maxClustScore.first];

      mergeTwoClusters( maxClustScore.first, maxClustScore.second,
                        numBands, numClust, clasDist, count, means,
                        cov_upper, cov_cfactors);

    /* update the coocurrance counts */
      updateCoocurranceCounts1( maxClustScore.first, maxClustScore.second,
			                    numClust, com, cc);

    /* First is smaller than second so change the entry for the second */
      newClass[maxClustScore.second] = base(newClass,maxClustScore.first);
    }
  }
  
  /* First make sure that our class list is correct */
  for (i=0;i<numClust;i++){
    /*+++
    kprintf("%d %d became ",i,newClass[i]);
    +++*/
    newClass[i] = base(newClass,i);
    tempClass[i] = NOTASSIGNED;
    /*+++
    kprintf("%d\n",newClass[i]);
    +++*/
  }

  /* make tempClass hold the translations between the old classes and
     the new classes */
  for (i=0;i<numClust;i++)
    if (oldClass[i] != NOTASSIGNED){

      if ((tempClass[newClass[i]] != NOTASSIGNED) &&  
      (tempClass[newClass[i]] != oldClass[i]))
        kerror(NULL, routine, "I've made a mistake reassiging classes.\n");

      tempClass[newClass[i]] = oldClass[i];
    }

  for (i=0;i<numClust;i++)
    newClass[i] = tempClass[newClass[i]];

  kfree(cc);
  kfree(clasDist[0]);
  kfree(clasDist);

  for (i=0;i<numClust;i++)
    kfree(com[i]);
  kfree(com);
  
  for (i=0;i<numClust;i++)
    kfree(cov_cfactors[0][0]);
  kfree(cov_cfactors[0]);
  kfree(cov_cfactors);
  kfree(tempClass);

  return newClass;
}

/*
 *  dot_product ()    Compute the dot product between two vectors.
 *
 *  Written by:  Patrick M. Kelly
 *  Date:        11/13/90
 *
 *  Modified:
 *    8/21/94   Dave Modl
 *              Converted to Khoros 2.0
 */

double
dot_product(
  double      * vec_1,
  double      * vec_2,
  int          len
  )
{
        double result;
        int i;

        result = 0.0;
        for ( i = 0 ; i < len ; i ++ ) result += vec_1 [i] * vec_2 [i] ;

        return ( result );
}
