 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>         Utility routines for extractor
   >>>>
   >>>>  Private: startup_from_clui
   >>>>           get_roi
   >>>>           adjust_gui_from_roi_shape
   >>>>           adjust_gui_from_roi_mode
   >>>>   Static:
   >>>>           display_extracted_roi
   >>>>           extract_roi
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "extractor.h"

static void display_extracted_roi    PROTO((kobject, int));
static kobject extract_roi           PROTO((subform_pane *));

/*-----------------------------------------------------------
|
|  Routine Name: startup_from_clui
|
|       Purpose: adjusts GUI according to command line arguments
|
|         Input:
|        Output:
|       Returns:
|    Written By: Danielle Argiro
|          Date: Dec 19, 1994
| Modifications:
|
------------------------------------------------------------*/

void startup_from_clui(void)
{
	xvobject help, quit, license;
        subform_pane *pane_info = gui_info->subform->pane;


	/*
	 *  Tack the help, quit, and license to the right edge of the form.
	 */
        quit = xvf_get_xvobject(gui_info->quit_struct, XVF_BACKPLANE, TRUE);
        help = xvf_get_xvobject(gui_info->help_struct, XVF_BACKPLANE, TRUE);
        license = xvf_get_xvobject(gui_info->license_struct,XVF_BACKPLANE,TRUE);
        xvw_set_attribute(quit, XVW_LEFT_OF, NULL);
        xvw_set_attribute(help, XVW_LEFT_OF, quit);
        xvw_set_attribute(license, XVW_LEFT_OF, help);

        xvf_set_attribute(pane_info->i_struct, XVF_FILE_NAME,
                          clui_info->i_file);
        xvf_set_attribute(pane_info->o_struct, XVF_FILE_NAME,
                          clui_info->o_file);

        xvf_set_attribute(pane_info->shape_struct, XVF_TOGGLE_NUM,
                          clui_info->shape_toggle);
        xvf_set_attribute(pane_info->policy_struct, XVF_LIST_VAL,
                          clui_info->policy_list);
        xvf_set_attribute(pane_info->mult_struct, XVF_LOGIC_VAL,
                          clui_info->mult_logic);
        xvf_set_attribute(pane_info->pres_struct, XVF_LIST_VAL,
                          clui_info->pres_list);
	kfree(pane_info->i);
	kfree(pane_info->o);

	pane_info->i = kstrdup(clui_info->i_file);
	pane_info->o = kstrdup(clui_info->o_file);

	pane_info->shape_val = clui_info->shape_toggle;
 	pane_info->policy    = clui_info->policy_list;
 	pane_info->mult      = clui_info->mult_logic;
 	pane_info->pres      = clui_info->pres_list;

        /*
         * adjust ROI pane according to initial ROI shape type.
         */
        xvf_add_extra_call(gui_info->subform_struct,
                           adjust_gui_from_roi_shape, NULL, XVF_CALL_LAST);

	/*
	 * add event handler
	 */
	xvw_add_event(image, ButtonPressMask, get_roi, NULL);

	xvw_set_attributes(image,
			   XVW_IMAGE_ROI_PRESENTATION, clui_info->pres_list, 
			   XVW_IMAGE_ROI_MULTIBAND,    clui_info->mult_logic,
	   	           XVW_IMAGE_ROI_POLICY,       clui_info->policy_list, 
			   XVW_IMAGE_ROI_SHAPE,        clui_info->shape_toggle,
			   NULL);
}

/*-----------------------------------------------------------
|
|  Routine Name: get_roi
|
|       Purpose: Extracts the region of interest
|
|         Input: object      - the image object
|                client_data - not used
|                event       - the ButtonPress event
|                dispatch    - whether or not to dispatch event
|        Output: None
|    Written By: Danielle Argiro
|          Date: Dec 19, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
void get_roi(
     xvobject object,
     kaddr    client_data,
     XEvent   *event,
     int      *dispatch)
{
        kobject         roi_object, out_object;
        subform_pane   *pane_info = gui_info->subform->pane;

        roi_object = extract_roi(pane_info);
        if (roi_object == NULL)
            return;

	if ((!pane_info->disp) && (pane_info->o == NULL)) 
	{
	    kerror(NULL, "get_roi", "No output file provided, and 'Display Extracted ROI' set to FALSE; therefore, the ROI extraction operation had no effect");
	    return;
	}

	/*
	 * displaying extracted ROI, but haven't created ROI display toplevel
	 */
	if ((pane_info->disp) && (disp_roi == NULL))
	{
	   /*
	    *  create manager to be the parent to display extracted ROI
	    */
	    disp_roi = xvw_create_manager(NULL, "extracted");

	   /*
	    *  for ROI shape of signal, want an area, 2D axis, & 2D plot object
	    */
	    disp_area     = xvw_create_area(disp_roi, "disp_area");
	    xvw_set_attributes(disp_area,
                XVW_GRAPHICS_VIEWPORT_MIN_X, 0.2,
                XVW_GRAPHICS_VIEWPORT_MIN_Y, 0.2,
                XVW_GRAPHICS_VIEWPORT_MAX_X, 0.9,
                XVW_GRAPHICS_VIEWPORT_MAX_Y, 0.9,
		XVW_MINIMUM_WIDTH,           350,
		XVW_MINIMUM_HEIGHT,          350,
	        XVW_TACK_EDGE,               KMANAGER_TACK_ALL,
		NULL);
	   /*
	    *  for ROI shape of image, want an image object
	    */
	    disp_image = xvw_create_image(disp_roi, "disp_roi");
	}

	/*
	 *  if we are displaying the extracted ROI...
	 */
	if (disp_roi != NULL)
	    display_extracted_roi(roi_object, pane_info->pres);

        /*
         *  output filename provided; write out ROI to file specified
         */
	if (pane_info->o != NULL)
	{
            /* open an output object for writing */
            out_object = kpds_open_object(pane_info->o, KOBJ_WRITE);
            if (out_object == NULL)
            {
                  kerror(NULL, "main", "Unable to open output file 'roi.xv'");
                  kexit(KEXIT_SUCCESS);
            }
    
            /*
             *  copy the attributes & the data of the ROI to the output object
             */
            kpds_copy_object(roi_object, out_object);
    
            /*
             *  that's all... close the output object & temp ROI object
             */
            kpds_close_object(out_object);
            kpds_close_object(roi_object);
	}
	*dispatch = FALSE;
}


/*-----------------------------------------------------------
|
|  Routine Name: display_extracted_roi
|
|       Purpose: Displays the extracted ROI depending on whether 
|                the ROI Presentation is set to signal, image, or surface.
|                REMEMBER that this routine is replicated both in extractor
|                and in editimage.
|
|         Input: roi_object - The kobject associated w/ extracted roi
|		 pres       - KIMAGE_ROI_SIGNAL, 
|                             KIMAGE_ROI_IMAGE or 
|                             KIMAGE_ROI_SURFACE
|        Output:
|       Returns:
|    Written By: Danielle Argiro
|          Date: Jan 06, 1995
| Modifications:
|
------------------------------------------------------------*/

static void display_extracted_roi(
    kobject roi_object,
    int     pres)
{
	xvobject xaxis, yaxis;

	/* display extracted ROI in 2D plot object */
        if (pres == KIMAGE_ROI_SIGNAL)
        {
	    xvw_unmanage(disp_image);
            xvw_manage(disp_area);
            if (disp_plot3d != NULL)
            {
		xvw_destroy(disp_plot3d);
                disp_plot3d = NULL;
            }
            if (disp_axis2d == NULL)
            {
	        xvw_set_attribute(disp_area, XVW_AREA_TITLE_STRING, "Signal");
                disp_axis2d = xvw_create_axis2d(disp_area, "disp_axis2d");
                xvw_set_attribute(disp_axis2d, XVW_AREA_ATTACH, disp_axis2d);
		xvw_get_attributes(disp_axis2d, 
				   XVW_AXIS2D_AXIS_X, &xaxis,
				   XVW_AXIS2D_AXIS_Y, &yaxis,
				   NULL);
		xvw_set_attribute(xaxis, XVW_AXIS_LABEL, "Number of Points");
		xvw_set_attribute(yaxis, XVW_AXIS_LABEL, "Pixel");
                disp_plot2d = xvw_create_plot2d(disp_area, "disp_plot2d");
                xvw_set_attribute(disp_plot2d, XVW_AREA_ATTACH, disp_axis2d);
            }
            xvw_set_attribute(disp_plot2d, XVW_PLOT2D_PLOTOBJ, roi_object);
        }

        /* display extracted ROI in image object */
        else if (pres == KIMAGE_ROI_IMAGE)
        {
            xvw_unmanage(disp_area);
            xvw_manage(disp_image);
            if (disp_axis2d != NULL)
            {
                xvw_destroy(disp_axis2d);
                xvw_destroy(disp_plot2d);
                disp_axis2d = NULL;
                disp_plot2d = NULL;
            }
            if (disp_plot3d != NULL)
            {
                xvw_destroy(disp_plot3d);
                disp_plot3d = NULL;
            }
            xvw_set_attribute(disp_image, XVW_IMAGE_IMAGEOBJ, roi_object);
	}

        /* display extracted ROI in 3D plot object */
        else if (pres == KIMAGE_ROI_SURFACE)
        {
	    xvw_unmanage(disp_image);
            xvw_manage(disp_area);
            if (disp_axis2d != NULL)
            {
		xvw_destroy(disp_axis2d);
                xvw_destroy(disp_plot2d);
                disp_axis2d = NULL;
                disp_plot2d = NULL;
            }
            if (disp_plot3d == NULL)
            {
	        xvw_set_attribute(disp_area, XVW_AREA_TITLE_STRING, "Surface");
                disp_plot3d = xvw_create_plot3d(disp_area, "disp_plot3d");
                xvw_set_attributes(disp_plot3d,
                                   XVW_PLOT3D_PLOTOBJ,  roi_object,
                                   XVW_PLOT3D_PLOTTYPE, KPLOT3D_MESH,
                                   NULL);
            }
            else xvw_set_attribute(disp_plot3d, XVW_PLOT3D_PLOTOBJ,
                                   roi_object);
        }

	else 
	{
	    kerror(NULL, "display_extracted_roi",
		   "Unexpected presentation type!");
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: extract_roi
|
|       Purpose: Does interactive (mouse) ROI extraction or
|                non-interactive (keyboard) ROI extraction.
|                When ROI extraction is done interactively with the 
|                mouse, xvw_get_attribute() is used; when ROI 
|                extraction is done non-interactively with the 
|                keyboard, xvw_getroi_xxx() is used. 
|
|         Input: roi_info - ptr to PaneInfo struct for roi pane
|
|        Output: None
|    Written By: Danielle Argiro
|          Date: Jan 05, 1995
| Modifications:
|
------------------------------------------------------------*/
static kobject extract_roi(
    subform_pane *roi_info)
{
	kobject roi_object;

	int policy;
	xvw_get_attribute(image, XVW_IMAGE_ROI_POLICY, &policy);
	
        /*
         * Insertion/Extraction Mode set to Mouse Specification (interactive)
         * simply set the XVW_IMAGE_ROI attribute on the image object
         */
        if (roi_info->mode_val == EXTRACT_W_MOUSE)
        {
            /*
             *  let image object do the interactive ROI extraction,
             *  it'll pass back a temp kobject.
             */
            xvw_get_attribute(image, XVW_IMAGE_ROI, &roi_object);
        }
        /*
         * Insertion/Extraction Mode set to Keybd Specification (noninteractive)
         * have to call the appropriate getroi() routine w/ specified dimensions
         */
        else
        {
            /* keyboard extraction of a rectangle */
            if (roi_info->shape_val == KIMAGE_ROI_RECTANGLE)
            {
                roi_object = xvw_getroi_rectangle(image, NULL,
                                roi_info->rect_x, roi_info->rect_y,
                                roi_info->rect_width, roi_info->rect_height);
            }

            else if ((roi_info->shape_val == KIMAGE_ROI_POLYLINE) ||
                    (roi_info->shape_val == KIMAGE_ROI_FREEHAND))
            {
                kerror(NULL, "roi_extract_roi",
                       "Sorry, but keyboard (noninteractive) extraction of polyline and freehand ROIs are not supported");
                return(NULL);
            }

            /* keyboard extraction of a line */
            else if (roi_info->shape_val == KIMAGE_ROI_LINE)
            {
                roi_object = xvw_getroi_line(image, NULL,
                                roi_info->line_x1, roi_info->line_y1,
                                roi_info->line_x2, roi_info->line_y2);
            }

            else if (roi_info->shape_val == KIMAGE_ROI_CIRCLE)
            {
                roi_object = xvw_getroi_circle(image, NULL,
                                roi_info->circle_x, roi_info->circle_y,
                                roi_info->circle_radius);
            }
            else if (roi_info->shape_val == KIMAGE_ROI_ELLIPSE)
            {
                roi_object = xvw_getroi_ellipse(image, NULL,
                                roi_info->ellipse_x, roi_info->ellipse_y,
                                roi_info->ellipse_a, roi_info->ellipse_b);
            }
        }
	return(roi_object);
}

/*-----------------------------------------------------------
|
|  Routine Name: adjust_gui_from_roi_shape
|
|       Purpose: This routine appropriately maps and unmaps the float
|                selections that are used for keyboard ROI extraction
|                depending on the ROI shape.  REMEMBER that this routine
|                is duplicated in editimage.
|                
|         Input: roi_info - ptr to PaneInfo struct for roi pane
|        Output: None
|    Written By: Danielle Argiro
|          Date: Jan 04, 1995
| Modifications:
|
------------------------------------------------------------*/
void adjust_gui_from_roi_shape(void)
{
	subform_pane *roi_info = gui_info->subform->pane;

	/*
	 *  note i am not doing this by the book, but the quick & dirty way
	 */
	if (roi_info->shape_val == KIMAGE_ROI_RECTANGLE)
	{
	   xvw_unmanage(roi_info->line_x1_struct->Selptr->back);
	   xvw_unmanage(roi_info->line_x2_struct->Selptr->back);
	   xvw_unmanage(roi_info->line_y1_struct->Selptr->back);
	   xvw_unmanage(roi_info->line_y2_struct->Selptr->back);
           xvw_unmanage(roi_info->circle_radius_struct->Selptr->back);
           xvw_unmanage(roi_info->circle_x_struct->Selptr->back);
           xvw_unmanage(roi_info->circle_y_struct->Selptr->back);
           xvw_unmanage(roi_info->ellipse_x_struct->Selptr->back);
           xvw_unmanage(roi_info->ellipse_y_struct->Selptr->back);
           xvw_unmanage(roi_info->ellipse_a_struct->Selptr->back);
           xvw_unmanage(roi_info->ellipse_b_struct->Selptr->back);

	   xvw_manage(roi_info->rect_width_struct->Selptr->back);
	   xvw_manage(roi_info->rect_height_struct->Selptr->back);
	   xvw_manage(roi_info->rect_x_struct->Selptr->back);
	   xvw_manage(roi_info->rect_y_struct->Selptr->back);
	}
	else if (roi_info->shape_val == KIMAGE_ROI_CIRCLE)
	{
	   xvw_unmanage(roi_info->rect_width_struct->Selptr->back);
	   xvw_unmanage(roi_info->rect_height_struct->Selptr->back);
	   xvw_unmanage(roi_info->rect_x_struct->Selptr->back);
	   xvw_unmanage(roi_info->rect_y_struct->Selptr->back);
           xvw_unmanage(roi_info->line_x1_struct->Selptr->back);
           xvw_unmanage(roi_info->line_x2_struct->Selptr->back);
           xvw_unmanage(roi_info->line_y1_struct->Selptr->back);
           xvw_unmanage(roi_info->line_y2_struct->Selptr->back);
           xvw_unmanage(roi_info->ellipse_x_struct->Selptr->back);
           xvw_unmanage(roi_info->ellipse_y_struct->Selptr->back);
           xvw_unmanage(roi_info->ellipse_a_struct->Selptr->back);
           xvw_unmanage(roi_info->ellipse_b_struct->Selptr->back);

	   xvw_manage(roi_info->circle_radius_struct->Selptr->back);
	   xvw_manage(roi_info->circle_x_struct->Selptr->back);
	   xvw_manage(roi_info->circle_y_struct->Selptr->back);
	}

        else if (roi_info->shape_val == KIMAGE_ROI_ELLIPSE)
        {
           xvw_unmanage(roi_info->rect_width_struct->Selptr->back);
           xvw_unmanage(roi_info->rect_height_struct->Selptr->back);
           xvw_unmanage(roi_info->rect_x_struct->Selptr->back);
           xvw_unmanage(roi_info->rect_y_struct->Selptr->back);
           xvw_unmanage(roi_info->line_x1_struct->Selptr->back);
           xvw_unmanage(roi_info->line_x2_struct->Selptr->back);
           xvw_unmanage(roi_info->line_y1_struct->Selptr->back);
           xvw_unmanage(roi_info->line_y2_struct->Selptr->back);
           xvw_unmanage(roi_info->circle_radius_struct->Selptr->back);
           xvw_unmanage(roi_info->circle_x_struct->Selptr->back);
           xvw_unmanage(roi_info->circle_y_struct->Selptr->back);

           xvw_manage(roi_info->ellipse_x_struct->Selptr->back);
           xvw_manage(roi_info->ellipse_y_struct->Selptr->back);
           xvw_manage(roi_info->ellipse_a_struct->Selptr->back);
           xvw_manage(roi_info->ellipse_b_struct->Selptr->back);
        }

        else if ((roi_info->shape_val == KIMAGE_ROI_POLYLINE) ||
		 (roi_info->shape_val == KIMAGE_ROI_FREEHAND))
        {
           xvw_unmanage(roi_info->line_x1_struct->Selptr->back);
           xvw_unmanage(roi_info->line_x2_struct->Selptr->back);
           xvw_unmanage(roi_info->line_y1_struct->Selptr->back);
           xvw_unmanage(roi_info->line_y2_struct->Selptr->back);
           xvw_unmanage(roi_info->circle_radius_struct->Selptr->back);
           xvw_unmanage(roi_info->circle_x_struct->Selptr->back);
           xvw_unmanage(roi_info->circle_y_struct->Selptr->back);
           xvw_unmanage(roi_info->ellipse_x_struct->Selptr->back);
           xvw_unmanage(roi_info->ellipse_y_struct->Selptr->back);
           xvw_unmanage(roi_info->ellipse_a_struct->Selptr->back);
           xvw_unmanage(roi_info->ellipse_b_struct->Selptr->back);
           xvw_unmanage(roi_info->rect_width_struct->Selptr->back);
           xvw_unmanage(roi_info->rect_height_struct->Selptr->back);
           xvw_unmanage(roi_info->rect_x_struct->Selptr->back);
           xvw_unmanage(roi_info->rect_y_struct->Selptr->back);
        }
        else if (roi_info->shape_val == KIMAGE_ROI_LINE) 
        {
           xvw_unmanage(roi_info->circle_radius_struct->Selptr->back);
           xvw_unmanage(roi_info->circle_x_struct->Selptr->back);
           xvw_unmanage(roi_info->circle_y_struct->Selptr->back);
           xvw_unmanage(roi_info->ellipse_x_struct->Selptr->back);
           xvw_unmanage(roi_info->ellipse_y_struct->Selptr->back);
           xvw_unmanage(roi_info->ellipse_a_struct->Selptr->back);
           xvw_unmanage(roi_info->ellipse_b_struct->Selptr->back);
           xvw_unmanage(roi_info->rect_width_struct->Selptr->back);
           xvw_unmanage(roi_info->rect_height_struct->Selptr->back);
           xvw_unmanage(roi_info->rect_x_struct->Selptr->back);
           xvw_unmanage(roi_info->rect_y_struct->Selptr->back);

           xvw_manage(roi_info->line_x1_struct->Selptr->back);
           xvw_manage(roi_info->line_x2_struct->Selptr->back);
           xvw_manage(roi_info->line_y1_struct->Selptr->back);
           xvw_manage(roi_info->line_y2_struct->Selptr->back);
        }

	/*
	 * trick to prevent user from using the keyboard for polygons or
	 * freehand drawing; allow them to use it for anything else
	 */
	if ((roi_info->shape_val == KIMAGE_ROI_POLYLINE) ||
	    (roi_info->shape_val == KIMAGE_ROI_FREEHAND))
	{
	   xvf_set_attribute(roi_info->mode_struct, XVF_TOGGLE_NUM, 1);
	   xvf_set_attribute(roi_info->mode_struct->Selptr->toggle_next->next->back_kformstruct,
			     XVF_ACTIVATE, FALSE);
	   roi_info->mode_val = 1;
	   roi_info->mode_num = 1;
	}
	else 
	{
	   xvf_set_attribute(roi_info->mode_struct->Selptr->toggle_next->next->back_kformstruct,
			     XVF_ACTIVATE, TRUE);
	}

	adjust_gui_from_roi_mode();
}

/*-----------------------------------------------------------
|
|  Routine Name: adjust_gui_from_roi_mode
|
|       Purpose: This routine appropriately sensitizes and un-sensitizes
|                the keyboard-ROI-extraction parameters based on whether
|                the "mode" parameter is set to mouse (interactive) or
|                keyboard (non-interactive) ROI extraction. REMEMBER
|                that this routine is duplicated in editimage.
|         Input:
|        Output:
|    Written By: Danielle Argiro
|          Date: Jan 04, 1995
| Modifications:
|
------------------------------------------------------------*/

void adjust_gui_from_roi_mode(void)
{
	int set, shape;
	subform_pane *roi_info = gui_info->subform->pane;

	if (roi_info->mode_val == EXTRACT_W_MOUSE)
	   set = FALSE;
	else set = TRUE;

	shape = roi_info->shape_val;

	if  (shape == KIMAGE_ROI_LINE) 
        {
            xvw_sensitive(roi_info->line_x1_struct->Selptr->back, set);
            xvw_sensitive(roi_info->line_x2_struct->Selptr->back, set);
            xvw_sensitive(roi_info->line_y1_struct->Selptr->back, set);
            xvw_sensitive(roi_info->line_y2_struct->Selptr->back, set);
            xvw_sensitive(roi_info->extract_struct->Selptr->back, set);
        }
	else if (shape == KIMAGE_ROI_ELLIPSE)
	{
	    xvw_sensitive(roi_info->ellipse_x_struct->Selptr->back, set);
            xvw_sensitive(roi_info->ellipse_y_struct->Selptr->back, set);
            xvw_sensitive(roi_info->ellipse_a_struct->Selptr->back, set);
            xvw_sensitive(roi_info->ellipse_b_struct->Selptr->back, set);
            xvw_sensitive(roi_info->extract_struct->Selptr->back, set);
	}
	else if (shape == KIMAGE_ROI_CIRCLE)
	{
	    xvw_sensitive(roi_info->circle_radius_struct->Selptr->back, set);
            xvw_sensitive(roi_info->circle_x_struct->Selptr->back, set);
            xvw_sensitive(roi_info->circle_y_struct->Selptr->back, set);
            xvw_sensitive(roi_info->extract_struct->Selptr->back, set);
	}
	else if (shape == KIMAGE_ROI_RECTANGLE)
	{
            xvw_sensitive(roi_info->rect_width_struct->Selptr->back, set);
            xvw_sensitive(roi_info->rect_height_struct->Selptr->back, set);
            xvw_sensitive(roi_info->rect_x_struct->Selptr->back, set);
            xvw_sensitive(roi_info->rect_y_struct->Selptr->back, set);
            xvw_sensitive(roi_info->extract_struct->Selptr->back, set);
        }
	else if ((shape == KIMAGE_ROI_POLYLINE) ||
		 (shape == KIMAGE_ROI_FREEHAND))
	{
            xvw_sensitive(roi_info->extract_struct->Selptr->back, set);
	}

}
