#include <xvinclude.h>

void autocolor_cb PROTO((xvobject, kaddr, kaddr));

/*
 *  This program displays an image to be autocolored. It then creates a list
 *  object to display all the currently supported colormaps.  Each time a
 *  colormap is selected from the list, that colormap is applied to the image.
 *  A read/write color allocation policy is used to speed up the installation
 *  of new colormaps.
 */
void main(
   int  argc,
   char *argv[],
   char *envp[])
{
	kobject  data_object;
	xvobject manager, image, list, actual_list;
	char     *filename = "image:ball";
	char     **autocolor_list;
	int      autocolor_num;

        /* initialize Khoros program */
        khoros_initialize(argc, argv, envp, "ENVISION");

	/* support other filenames on cmd line, as in "% example image:feath" */
	if (argc > 1)
	   filename = argv[1];

	/* create the data object associated with the input image */
	data_object = kpds_open_input_object(filename);

        /* initialize the xvwidgets lib */
        if (!xvw_initialize(XVW_MENUS_XVFORMS))
        {
           kerror(NULL, "main", "unable to open display");
           kexit(KEXIT_FAILURE);
        }

	/* 
	 * create the image object w/ input image.  note that since we
	 * will be changing the colors in the colormap often,  using a
         * color allocation policy of read/write speeds up the process
         * significantly; the tradeoff is suffering techno-flash.
	 */
	image = xvw_create_image(NULL, "image");
	xvw_set_attributes(image, 
			   XVW_IMAGE_IMAGEOBJ, data_object,
			   XVW_COLOR_ALLOC_POLICY, KCOLOR_ALLOC_READWRITE,
			   NULL);

	/*
	 *  get the list of strings representing the different autocolor
         *  options, and the size of that list.
	 */
	kcolor_get_attribute(data_object, KCOLOR_MAP_AUTOCOLOR_LIST, 
			     &autocolor_list, &autocolor_num);

	/* 
	 * create a resizable list object to display colormaps; it needs
	 * a manager object to act as its backplane
	 */
	manager = xvw_create_manager(NULL, "parent");
	list = xvw_create_list(manager, "Autocolor List");
	xvw_set_attributes(list,
		           XVW_RESIZABLE,     TRUE,
		           XVW_CHAR_HEIGHT,   10.0,
		           XVW_CHAR_WIDTH,    20.0,
		           NULL);
	/* 
	 * the list object is compound - need the inner actual list on which
	 *  to set the list contents and add the callback 
         */
	actual_list = xvw_retrieve_list(list);
	xvw_change_list(actual_list, autocolor_list, 
			autocolor_num, FALSE);
	xvw_add_callback(actual_list, XVW_LIST_ITEM_SELECT, 
			autocolor_cb, data_object);

	/* display and run */
	xvf_run_form();
}

/*
 *  The autocolor callback takes the colormap chosen from the list,
 *  and uses Color Services to set the KCOLOR_MAP_AUTOCOLOR attribute,
 *  which changes the colormap on the displayed image.  The data object
 *  associated with the input image is passed in as the client_data, so
 *  we can send it to kcolor_set_attribute().  The list object itself
 *  passes a xvw_list_struct into its callbacks as the call_data, so we
 *  can use the string and the index of the item selected from the list.
 */
void autocolor_cb (
    xvobject object,
    kaddr    client_data,
    kaddr    call_data)
{
	xvw_list_struct *list = (xvw_list_struct *) call_data;
        kobject data_object = (kobject) client_data;

	kfprintf(kstderr,"Applying Autocolor (%d) '%s' \n",
		list->list_index+1, list->string);
	kcolor_set_attribute(data_object, KCOLOR_MAP_AUTOCOLOR, 
			     list->list_index);
}
