 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>		    File Descriptor Change Detect Routines
   >>>>
   >>>>   These routines are used to detect whether a file
   >>>>   descriptor has new data.  When new data is detected
   >>>>   we call the programmer's callback routine.  The file
   >>>>   descriptor is detected using the X toolkit to listen on the
   >>>>   input port.
   >>>>
   >>>>   Static:
   >>>>             check_input_fid()
   >>>>  Private:
   >>>>
   >>>>   Public:
   >>>>             xvw_add_detectfid()
   >>>>             xvw_remove_detectfid()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"	


typedef struct _xvw_fid
{
	xvobject   object;
	XtInputId  id;
	int	   fid;
	kfunc_void routine;
	kaddr	   client_data;
} xvw_fid;

static klist *fid_list  = NULL;


/*-----------------------------------------------------------
|
|  Routine Name: check_input_fid - check to see if any new input data
|
|       Purpose: Action routine used to detect when new input has
|                been detected on a file id.  When data is detected
|                we call the user's callback routine with the
|		 id and client data.
|
|         Input: data - our xvw_fid structure
|                fid  - the file descriptor that has input
|                id   - the current input id
|
|        Output: none
|	Returns: none
|
|    Written By: Mark Young
|          Date: Jul 15, 1992 10:42
| Modifications: Converted from check_input_fid() in Khoros 1.0 (MY)
|
------------------------------------------------------------*/
/* ARGSUSED */
static void check_input_fid(
   XtPointer data,
   int       *fid,
   XtInputId *id)
{
	xvw_fid *entry = (xvw_fid *) data;


	/*
	 *  since we have detected input on the fid, call back the
	 *  user's callback routine and if they return TRUE we then
	 *  update the read time to the current time.
	 */
	(void) entry->routine(entry->object, entry->fid, entry->client_data);
}


/************************************************************
*
*  Routine Name: xvw_add_detectfid - add (fid) input handler to an object
*
*       Purpose: Causes a detection mechanism to be installed on the
*                specified file descriptor; the file descriptor will be
*		 polled intermittently for change (input or output).
*		 If a change in the file descriptor is detected, then the 
*                specified input handler is called.
*
*		 The input handler can be associated with an object, so that
*		 fid detection is automatically discontinued when the object is
*		 destroyed.  If NULL is passed for the object, then the input
*                handler is added to the global file detection list.
*
*                !The detect file callback must be declared in the following
*		 !form:\f(CW
*                !int input_handler(
*                !     xvobject object,
*                !     int      fid,
*                !     kaddr    client_data)\fP
*
*                !\fIobject -\fP
*                !    If \fIxvw_add_detectfid()\fP is called with a particular
*                !    xvobject, that object will be passed into the input handle
r.
*
*                !\fIfid -\fP
*                !    This is the file descriptor being monitored for change.
*
*
*                !\fIclient_data -\fP
*                !    The pointer to the client data, used to pass parameters
*                !    from the application to the input handler.
*
*         Input: object      - object on which to place the input handler.  
*                              Pass NULL if the input handler is to be invoked 
*                              in general, and not associated with any 
*                              particular object.
*		 fid         - the file descriptor which will be monitored
*                              for input (or output)
*                routine     - the input handler routine to be called when 
*                              change on the fid is detected
*                client_data - pointer to client data that
*                              will be passed to input handler
*        Output:
*	Returns: 
*  Restrictions: 
*    Written By: Mark Young
*          Date: Jul 15, 1992 1:23
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

void xvw_add_detectfid(
   xvobject   object,
   int        fid,
   kfunc_void routine,
   kaddr      client_data)
{
	xvw_fid *entry;


	if ((entry = (xvw_fid *) kcalloc(1, sizeof(xvw_fid))) == NULL)
	{
	   kerror("xvutils", "xvw_add_detectfid", 
		  "Cannot allocate memory for xvw_fid structure");
	   return;
	}

	/*
	 *  if the file is being readin from a fid then we want to "listen"
	 *  on the given fid.
	 */
	entry->object  = object;
	entry->fid     = fid;
	entry->routine = routine;
	entry->client_data = client_data;
	entry->id = XtAppAddInput(xvw_appcontext(NULL), fid,
		(XtPointer) XtInputReadMask, check_input_fid,(XtPointer) entry);

	/*
	 *  Add the entry to the detect input handler list.  If the object is
	 *  NULL then we add it to the global file list.
	 */
	if (!object)
	{
	   fid_list = klist_insert(fid_list, (kaddr) fid, entry,
				KLIST_HEAD, TRUE);
	}
	else
	{
	   object->detectfid = klist_insert(object->detectfid, (kaddr) fid,
				entry, KLIST_HEAD, TRUE);
	}
}


/************************************************************
*
*  Routine Name: xvw_remove_detectfid - remove (fid) input handler from 
*					an object 
*
*       Purpose: Causes the detection mechanism previously installed with
*		 \fIxvw_add_detectfid()\fP to be removed from the specified 
*                file descriptor. 
*
*		 If the input handler was associated with an object, the
*		 file detection will be automatically removed when the object 
*		 is destroyed.  Alternatively, it can be removed before the
*                object is destroyed by using this routine.  If NULL was 
*                passed to \fIxvw_add_detectfid()\fP for the object, then 
*                you must call this routine if you need to remove the installed 
*                input handler.
*
*         Input: object      - object for which to remove the input file
*			       handler, or NULL if the input handler was 
*                              being invoked in general, not associated with 
*                              a particular object.
*         	 fid         - the file descriptor on which input 
*                              (or output) was being detected.
*                routine     - the input handler that was being called when
*                              change on the fid was detected
*                client_data - pointer to client data which 
*                              was being passed to input handler
*        Output:
*       Returns: 
*  Restrictions: 
*    Written By: Mark Young
*          Date: Jul 15, 1992
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

void xvw_remove_detectfid(
   xvobject   object,
   int        fid,
   kfunc_void routine,
   kaddr      client_data)
{
	xvw_fid *entry;
        klist   *temp, **list = !object ? &fid_list : &object->detectfid;

	temp = *list;
	while ((temp = klist_locate(temp, (kaddr) fid)) != NULL)
	{
	   entry = (xvw_fid *) klist_clientdata(temp);
	   if (routine == entry->routine && client_data == entry->client_data)
	   {
	      XtRemoveInput(entry->id);
	      kfree(entry);
	      temp = klist_delete(temp, temp->identifier);
	      *list = klist_head(temp);
	      return;
	   }
	   temp = klist_next(temp);
	}
}
