 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>		    Gadget/xvobject Event Handler Routines
   >>>>
   >>>>   These routines are used to add an event handler
   >>>>   on a xvobject.  Since Xt cannot support event
   >>>>	  handling on gadgets, only widgets, we do this by
   >>>>   placing the handler on the parent and then direct
   >>>>   the dispatching of the event ourselves.
   >>>>
   >>>>   Static:
   >>>>		  event_remove()
   >>>>		  event_check()
   >>>>		  event_handler()
   >>>>  Private:
   >>>>
   >>>>   Public:  
   >>>>		  xvw_insert_event()
   >>>>		  xvw_add_event()
   >>>>		  xvw_remove_event()
   >>>>		  xvw_destroy_events()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"	
#include <xvobjects/ManagerP.h>


/*
 *  We need to keep track of all the gadget events handlers.  We would rather
 *  simply add it to the gadget's event table, but since there is no way
 *  for our dispatch routine to dig into the gadget's event table.  The
 *  definition of XtEventTable is hidden inside the Xt library.  This is
 *  unfortunate, since it will greatly complicate what should be an easy
 *  extension. ;-(
 */
typedef struct {
	EventMask   mask;
	int	    event_mask;

	xvobject    object;
	kaddr       client_data;
	kfunc_void  routine;
	
	Boolean destroy;
	Boolean dispatch;
	Boolean inside;
} HandlerEvent;

/*
 *  So what is this?  Well, in a nutshell it is a modified version of
 *  Xtoolkits mask structure.  It allows us to take the event type and
 *  find the corresponding mask and x & y offset.  We use this information
 *  to figure out which of the gadgets event handler to call.  The xoffset
 *  and yoffset are base event offset to the x and y values.  We use this
 *  to see about re-writing the x and y position to be relative to the gadget
 *  rather than the parent object.
 */
typedef struct {
	EventMask mask;
	unsigned int xoffset;
	unsigned int yoffset;
} MaskEvent;

#define PointerMotion (PointerMotionMask|PointerMotionHintMask)
#define ButtonMotion  (ButtonMotionMask|Button1MotionMask|Button2MotionMask|\
		       Button3MotionMask|Button4MotionMask|Button5MotionMask)
#define MotionMask    (ButtonMotion|PointerMotion)

static MaskEvent mask_events[] = {
#define offset(field)  XtOffsetOf(XEvent, field)
  {0, 0, 0},
  {0, 0, 0},
  {KeyPressMask, offset(xkey.x), offset(xkey.y)},
  {KeyReleaseMask, offset(xkey.x), offset(xkey.y)},
  {ButtonPressMask, offset(xbutton.x), offset(xbutton.y)},
  {ButtonReleaseMask, offset(xbutton.x), offset(xbutton.y)},
  {MotionMask, offset(xmotion.x), offset(xmotion.y)},
  {EnterWindowMask, offset(xcrossing.x), offset(xcrossing.y)},
  {LeaveWindowMask, offset(xcrossing.x), offset(xcrossing.y)},
  {FocusChangeMask, 0, 0},
  {FocusChangeMask, 0, 0},
  {KeymapStateMask, 0, 0},
  {ExposureMask, offset(xexpose.x), offset(xexpose.y)},
  {0, offset(xgraphicsexpose.x), offset(xgraphicsexpose.y)},
  {0, 0, 0},
  {VisibilityChangeMask, 0, 0},
  {SubstructureNotifyMask, offset(xcreatewindow.x), offset(xcreatewindow.y)},
  {StructureNotifyMask|SubstructureNotifyMask, 0, 0},
  {StructureNotifyMask|SubstructureNotifyMask, 0, 0},
  {StructureNotifyMask|SubstructureNotifyMask, 0, 0},
  {SubstructureRedirectMask, 0, 0},
  {StructureNotifyMask|SubstructureNotifyMask, 0, 0},
  {StructureNotifyMask|SubstructureNotifyMask, 0, 0},
  {SubstructureRedirectMask, 0, 0},
  {StructureNotifyMask|SubstructureNotifyMask, 0, 0},
  {ResizeRedirectMask, 0, 0},
  {StructureNotifyMask|SubstructureNotifyMask, 0, 0},
  {SubstructureRedirectMask, 0, 0},
  {PropertyChangeMask, 0, 0},
  {0, 0, 0},
  {0, 0, 0},
  {0, 0, 0},
  {ColormapChangeMask, 0, 0},
  {0, 0, 0},
  {0, 0, 0},
#undef offset
};


/*-----------------------------------------------------------
|
|  Routine Name: event_check - check to see if an event is valid
|
|       Purpose: Check to see if the event is valid for the given object.
|		 There are certain conditions in which we will want
|		 make sure that an event is valid.  For instance, if
|		 we add a ButtonMotion event, we will get PointerMotion
|		 events as well.  This is due to an oversight on Xt's part.
|		 Hence, this routine enforces that...
|
|         Input: object - the object to check the event for
|		 event  - the event in question
|		 mask_event - the mask event structure
|		 handler    - the event handler structure instance for this
|			      object.
|	Returns: returns TRUE if this handler should be called, otherwise
|		 FALSE.
|
|    Written By: Mark Young
|          Date: Oct 29, 1992
| Modifications:
|
------------------------------------------------------------*/
#define AnyButtonMask (Button1Mask|Button2Mask|Button3Mask|\
		       Button4Mask|Button5Mask)
int event_check(
   xvobject object,
   XEvent   *event,
   EventMask mask,
   HandlerEvent *handler)
{
	/*
	 *  Now that we have verified that the event is for us, we
	 *  really need to check a few other things.  For instance,
	 *  if we indicate that we want ButtonMotion events, we will
	 *  get them irregardless if a Button is selected or not.  So
	 *  we use the following series of checks to fix these anamolies.
	 */
	if (event->type == MotionNotify && !(handler->mask & PointerMotion))
	{
	   if (!(event->xmotion.state & AnyButtonMask))
	      return(FALSE);
	}
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: event_remove - remove a handler from the event list
|
|       Purpose: Race thru the event handler list removing all the
|		 event handlers that are marked for deletion, but are
|		 not in the middle of being dispatched.
|
|         Input: list - the gadget's event handler list
|		 override - destroy the event list no matter what the
|			    setting of the destroy and dispatch list.
|	Returns: returns the new event list
|
|    Written By: Mark Young
|          Date: Oct 29, 1992
| Modifications:
|
------------------------------------------------------------*/
static klist *event_remove(
   klist   *head,
   int     override)
{
	klist *list = head;


	while (list != NULL)
	{
	   HandlerEvent *handler = (HandlerEvent *) klist_clientdata(list);

	   if (override || (handler->destroy && !handler->dispatch))
	   {
	      if (klist_checkhead(list))
		 head = klist_next(list);

	      kfree(list->client_data);
	      list = klist_delete(list, list->identifier);
	   }
	   else
	      list = klist_next(list);
	}
	return(head);
}

/*-----------------------------------------------------------
|
|  Routine Name: event_handler - the event_handler handler for the
|				  gadget
|
|       Purpose: This routine will check to see if the event for
|		 the gadget matches the gadget's event handler list.
|		 If so then we call each of the gadget's event handler.
|
|         Input: object     - the parent which we installed the dispatch
|                client_data - the object to check the event for
|                event      - the event structure to be dispatched
|
|        Output: dispatch   - whether we should continuing with the
|			      dispatching of this event  
|       Returns: none
|
|    Written By: Mark Young
|          Date: Oct 30, 1992 04:34
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void event_handler(
   Widget    parent,
   XtPointer client_data,
   XEvent    *event,
   Boolean   *dispatch)
{
	HandlerEvent *handler = (HandlerEvent *) client_data;

	MaskEvent *mask_event;
	int i, idispatch = (int) *dispatch;
        xvobject object = handler->object;


	/*
	 *  Mask out the send event field since we need to index the
	 *  structure to get a variety of information.
	 */
	if ((i = (event->type & 0x7f)) > knumber(mask_events) ||
	   (mask_events[i].mask == 0))
	   return;
	else
	   mask_event = &mask_events[i];

	/*
	 *  If the object is a gadget then we better check to see if the
	 *  event is for this object.
	 */
	if (!XtIsWidget(xvw_widget(object)))
	{
	   /*
	    *  Now see if the event in question has an xpos, ypos so that we
	    *  can check to see if the event was for this object.
	    */
	   if (mask_event->xoffset > 0 && mask_event->yoffset > 0)
	   {
	      Position xpos, ypos;
	      kaddr addr; 
	      char  *base = (char *) event;

	      addr = base + mask_event->xoffset; xpos = *((int *) addr);
	      addr = base + mask_event->yoffset; ypos = *((int *) addr);

	      if (!ManagerCheckSelection(object, xpos, ypos))
	      {
		 if (event->type != MotionNotify || !handler->inside ||
		     !(handler->event_mask & LeaveWindowMask))
		 {
	            return;
		 }
		 handler->inside = FALSE;
		 event->type = LeaveNotify;
	      }
	      else if (event->type == MotionNotify &&
		       (handler->event_mask & EnterWindowMask))
	      {
		 if (handler->inside == TRUE)
		    return;

		 handler->inside = TRUE;
		 event->type = EnterNotify;
	      }
	   }
	}

	/*
	 *  Now that we know this event was intended for this object, but we
	 *  need to perform the infamous "Sanity check".  Make sure that an
	 *  event handler list exists for the desired object.  Normally this
	 *  would be the gadget's event table, but since we are unable to dive
	 *  into that structure, so we maintain such a list via the handerlist.
	 */
	if (!event_check(object, event, mask_event->mask, handler))
	   return;

	(*(handler->routine))(object, handler->client_data, event, &idispatch);
	*dispatch = idispatch;
}

/************************************************************
*
*  Routine Name: xvw_insert_event - insert an event handler for an object
*
*       Purpose: Insert an event handler for a visual object.  Since Xt 
*		 cannot support event handling of gadgets, we support them 
*		 by placing the event handler on the parent and then 
*	         directing the dispatch of the event handler directly.
*
*         Input: object      - object on which to install event handler
*                event_mask  - event mask representing the event for
*			       which the event handler is to be fired
*                routine     - the event handler to be called when event occurs
*                client_data - private data to be used by the event handler
*                position    - position of the event handler KLIST_HEAD or
*			       KLIST_TAIL
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Oct 29, 1992
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

void xvw_insert_event(
   xvobject   object,
   int        event_mask,
   kfunc_void routine,
   kaddr      client_data,
   int	      position)
{
	HandlerEvent *handler;
	Widget       widget  = xvw_widget(object);


	/*
	 *  Add it to our object's event list...
	 */
	if ((handler = (HandlerEvent *)kcalloc(1,sizeof(HandlerEvent))) == NULL)
	{
	   kerror("xvwidgets", "xvw_add_event",
		  "Unable to allocate internal event handler structure.");
	   return;
	}

	if (position != KLIST_TAIL && position != KLIST_HEAD)
	   position = KLIST_TAIL;

	handler->object      = object;
	handler->routine     = routine;
	handler->event_mask  = event_mask;
	handler->client_data = client_data;
	handler->mask = (EventMask) event_mask;
	object->events = (kaddr) klist_insert(object->events,
		(kaddr) handler->mask, handler, position, TRUE);

	/*
	 *  So how does this work.  So we will install the desired event
	 *  mask on the parent if it's a gadget.  Otherwise we install it
	 *  on the widget.  Then the event handler "event_handler" will
	 *  be called to dispatch the event to the appropriate object.
	 */
	if (!XtIsWidget(widget))
	{
	   widget = XtParent(widget);

	   /*
	    *  Yikes!  Gadgets don't have Enter and Leave Window capabilities.
	    *  How about if we do it.  It shouldn't be too hard (yeah, right!).
	    */
	   if (handler->mask & (EnterWindowMask | LeaveWindowMask))
	   {
	      handler->mask |= PointerMotionMask;
	      handler->mask &= ~(EnterWindowMask | LeaveWindowMask);
	   }
	}
	XtInsertEventHandler(widget, handler->mask, FALSE, event_handler,
		handler, position == KLIST_TAIL ? XtListTail : XtListHead);
}

/************************************************************
*
*  Routine Name: xvw_add_event - add an event handler to an object
*
*       Purpose: Add an event handler to a visual object.  When the
*                specified event(s) occur, the event handler will be called.
*
*                Since the X Toolkit cannot support event handling of gadgets, 
*                we support them by placing the event handler on the parent 
*                and then directing the dispatch of the event handler directly.
*
*		 The event handler \fImust\fP be associated with an object; only
*                when the specified event(s) occur on the specified object
*                will the event handler be called (ie, the same event in 
*                another object will be ignored).
*
*                An event handler must be declared in the following form:\f(CW
*                !void event_handler(
*     		 !   xvobject object,
*     		 !   kaddr  client_data,
*     		 !   XEvent *event,
*     		 !   int *dispatch)\fP
*
*		 !\fIobject -\fP
* 		 !    The object for which the event handler was invoked.
*                !    It will not be NULL.
*
*		 !\fIclient_data -\fP
*                !    The pointer to the client data, used to pass parameters
*                !    from the application to the event handler.
*                
*                !\fIevent -\fP
*                !    This is a pointer to the XEvent union which caused the 
*                !    event handler to be invoked.  For details on the XEvent
*                !    union, see Chapter 8 of the \fIXlib Programming Manual\fP,
*                !    by Adrian Nye;  the definition of the XEvent union is on 
*                !    page 232.
*
*		 !\fIdispatch -\fP
*                !    By default, the event that caused this event handler to be
*                !    invoked will continue to propagate to any other event 
*                !    handlers that might also be installed for the same event 
*                !    on the same visual object.  The \fIdispatch\fP integer 
*                !    pointer can be used to prevent the event from being 
*                !    dispatched, and to prevent any other such event handlers 
*                !    from being called.  To prevent dispatch of the event to 
*                !    any other event handlers, set this variable to FALSE, 
*                !    as in: \f(CW*dispatch = FALSE;\fP
*
*         Input: object      - object on which to install event handler
*                              (note that you may not pass NULL)
*                event_mask  - event mask representing the event for
*			       which the event handler is to be fired
*                routine     - the event handler to be called when event occurs
*                client_data - private data to be used by the event handler
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Oct 29, 1992
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

void xvw_add_event(
   xvobject   object,
   int        event_mask,
   kfunc_void routine,
   kaddr      client_data)
{
	xvw_insert_event(object, event_mask, routine, client_data, KLIST_TAIL);
}

/************************************************************
*
*  Routine Name: xvw_remove_event - remove an event handler from an object
*
*       Purpose: Removes an event handler from a visual object.
*
*         Input: object      - visual object from which to remove installed
*			       event handler
*                event_mask  - event mask representing the event for
*                              which the event handler was being fired
*                routine     - the event handler to remove
*                client_data - private data being used by the event handler
*
*    Written By: Mark Young
*          Date: Oct 29, 1992
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

void xvw_remove_event(
   xvobject   object,
   int        event_mask,
   kfunc_void routine,
   kaddr      client_data)
{
	klist     *list;
	Widget    widget = xvw_widget(object);


	/*
	 *  So how does this work.  Well we installed the event handler on
	 *  the parent if the object is a gadget.  We did this since the
	 *  gadget can have an event handler installed on it, but the toolkit
	 *  will simply ignore it.  So we will have to remove it from the
	 *  parent as well as the gadget.  If the object turns out to be
	 *  a widget, then simply remove it from the widget.
	 */
	if (!XtIsWidget(widget))
	   widget = XtParent(widget);

	/*
	 *  Go through our event handler list, find the event handler,
         *  and remove it from the list.
	 */
	list = object->events;
	while (list != NULL)
	{
	   HandlerEvent *handler = (HandlerEvent *) klist_clientdata(list);

	   if (routine == handler->routine && handler->event_mask & event_mask)
	   {
	      handler->destroy = TRUE;
	      XtRemoveEventHandler(widget, handler->mask, FALSE,
			event_handler, handler);
	   }
	   list = klist_next(list);
	}
	object->events = event_remove(object->events, FALSE);
}

/************************************************************
*
*  Routine Name: xvw_destroy_events - destroy all event handlers currently
*				      installed on an object
*
*       Purpose: Removes all event handlers currently installed on a 
*	         visual object.  
*
*		 This function is automatically called when a visual object
*		 is destroyed, but is publicly provided since it is
*		 sometimes necessary to explicitly remove all event
*		 handlers from an object.
*
*         Input: object - object/gadget to destroy all events for
*
*    Written By: Mark Young
*          Date: May 19, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

void xvw_destroy_events(
   xvobject   object)
{
	klist  *list;
	HandlerEvent *handler;
	Widget widget = xvw_widget(object);


	/*
	 *  So how does this work.  Well we installed the event handler on
	 *  the parent if the object is a gadget.  We did this since the
	 *  gadget can have an event handler installed on it, but the toolkit
	 *  will simply ignore it.  So we will have to remove it from the
	 *  parent as well as the gadget.  If the object turns out to be
	 *  a widget, then simply remove it from the widget.
	 */
	if (!XtIsWidget(widget))
	   widget = XtParent(widget);

	/*
	 *  Now that we deleted the real event handler we need to cleanup
	 *  our handler event list.
	 */
	list = object->events;
	while (list != NULL)
	{
	   handler = (HandlerEvent *) klist_clientdata(list);
	   XtRemoveEventHandler(widget, handler->mask, FALSE,
			event_handler, handler);

	   list = klist_next(list);
	}
	object->events = event_remove(object->events, TRUE);
}
