 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>		       Error Display			      <<<<
   >>>>                                                       <<<<
   >>>>  Private:      xvu_error()			      <<<<              
   >>>>                                                       <<<<
   >>>>   Static:      xvu_error_ok()		              <<<<              
   >>>>                                                       <<<<
   >>>>   Public:      xvu_error_wait()			      <<<<              
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"	

static int  xvu_error_done;

/*-----------------------------------------------------------
|
|  Routine Name: xvu_error_ok()
|
|       Purpose: Destroys the error object when the user clicks 
|                on the acknowledgement button.
|
|         Input: object      - the acknowledgement button
|                client_data - toplevel of error object
|		 call_data   - unused
|
|        Output: none
|    Written By: Danielle Argiro & Mark Young
|          Date: Jan 4, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void xvu_error_ok(
   xvobject object,
   kaddr  client_data,
   kaddr  call_data)
{
	xvu_error_done = TRUE;
}

/*-----------------------------------------------------------------
|
|  Routine Name: xvu_error
|
|       Purpose: xvu_error produces standardized error messages for
|                X-based Khoros library routines and X based Khoros 
|                applications.  It will be set up as the error
|		 handler for kerror in all X-based Khoros routines by
|                xvf_initialize().
|
|         Input: toolbox   - name of toolbox (if applicable)
|                program   - name of program (if applicable, NULL otherwise)
|                library   - name of library (if applicable, NULL otherwise)
|                routine   - name of routine
|                category  - name of error message category (~/verrors.h)
|                message   - grammatically correct, clear explanation of
|                            the error that occurred
|        Output: none
|    Written By: Danielle Argiro & Mark Young
|          Date: Jan 4, 1994
| Modifications:
|
--------------------------------------------------------------*/

int xvu_error(
   char *toolbox,
   char *program,
   char *library,
   char *routine,
   char *category,
   char *message)
{
	knotify handler;
	char buffer[25*KLENGTH], title[KLENGTH];

	buffer[0]  = '\0';
	title[0] = '\0';


	handler = kset_errorhandler(NULL);
	if (toolbox != NULL) 
	{
	    kstrcat(buffer, "Toolbox: ");
	    kstrcat(buffer, toolbox);
	}
	if (program != NULL) 
	{
	    kstrcat(buffer, "\nProgram: ");
	    kstrcat(buffer, program);
	}

	if (library != NULL) 
	{
	    kstrcat(buffer, "\nLibrary: ");
	    kstrcat(buffer, library);
	}

	if (routine != NULL)
	{
	    kstrcat(buffer, "\nRoutine: ");
	    kstrcat(buffer, routine);
	}

	kstrcat(buffer, "\n");

	if (message != NULL)
	{
	    kstrcat(buffer, "\n");
	    kstrcat(buffer, message);

	    kstrcat(title, "Error: ");

	    kstrcat(title, category);
	}
	else
	{
	    kstrcat(buffer, "\n");
	    kstrcat(buffer, category);

	    kstrcat(title, "Error:");
	}
	xvu_error_wait(buffer, title, NULL);
	(void) kset_errorhandler(handler);
	return(TRUE);
}

/**************************************************************
*
*  Routine Name: xvu_error_wait - pop up error object (1 button); wait for
*				  acknowledgement
*
*       Purpose: Creates a pop-up error object which must be acknowledged 
*                by the user before control is returned to the application 
*                program.  
*
*                IMPORTANT NOTE: to be consistent with the standards
*                of the Khoros Software development system, you should really
*                be calling kerror(), not xvu_error_wait().  
*
*         Input: error_mesg   - string describing error message
*                error_label  - short label for top of error object; passing 
*                               NULL will result in default of "Error".
*		 button_label - label for acknowledgment button; passing
*                               NULL will result in default of "Ok".
*        Output:
*	Returns: Returns FALSE if it failed to create the error object
*		 otherwise waits for user to acknowledge error message,
*    Written By: Danielle Argiro & Mark Young
*          Date: Jan 4, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
**************************************************************/

int xvu_error_wait(
   char *error_mesg,
   char *error_label,
   char *button_label)
{
	knotify  handler;
	char     name[KLENGTH];
	xvobject error, button = NULL;


        /*
         *  xvu_error_wait supports non-X displays
         */
	xvu_error_done = FALSE;
	if (xvw_display(NULL) == NULL)
	{
	    kinfo(KSTANDARD, "%s\n", error_mesg);
	    return(TRUE);
	}

	/*
	 *  xvw_create_error creates the error object pop-up
	 */
	xvw_busy(NULL, TRUE);
	handler = kset_errorhandler(NULL);

	(void) ksprintf(name, "Error for %s", kprog_get_program());
	if ((error = xvw_create_error(NULL, name)) == NULL)
	{
	   (void) kset_errorhandler(handler);
	    xvw_busy(NULL, FALSE);
	   return(FALSE);
	}
	xvw_set_attributes(error,
		XVW_ERROR_MESSAGE, error_mesg,
		XVW_ERROR_LABEL,   error_label ? error_label : "Error",
		XVW_ERROR_BUTTON_LABEL, button_label ? button_label : "Ok",
		NULL);
	xvw_add_callback(error, XVW_ERROR_CALLBACK, xvu_error_ok, NULL);
	xvw_get_attribute(error, XVW_ERROR_BUTTON_OBJECT, &button);

	(void) kset_errorhandler(handler);
	xvw_place(error, button);
	while (!xvu_error_done)
	    xvw_process_event();  

	xvw_busy(NULL, FALSE);
	return(TRUE);
}
