 /*
  * Khoros: $Id$
  */

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>                 Connection Gadget Routines
   >>>>
   >>>>	Private:
   >>>>	 Static:
   >>>>			ClassInitialize()
   >>>>			Initialize()
   >>>>			Redisplay()
   >>>>			SetValues()
   >>>>			Destroy()
   >>>>			PickGadget()
   >>>>			ConnectionCallback()
   >>>>  Public:
   >>>>			xvw_create_connection()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <xvobjects/ConnectionP.h>

static void ClassInitialize	 PROTO((void));
static void Initialize           PROTO((Widget, Widget, ArgList, Cardinal *));
static void Realize              PROTO((Widget));
static void Redisplay		 PROTO((Widget, XEvent *, Region));
static Boolean SetValues	 PROTO((Widget, Widget, Widget, ArgList,
					Cardinal *));
static void Destroy		 PROTO((Widget));
static int  PickGadget		 PROTO((Widget, int, int));
static void RecomputeSize	 PROTO((Widget, Boolean));
static void AllocateGC		 PROTO((Widget));
static int  ChangeUpdate	 PROTO((xvobject, String, kaddr));
static void UpdateConnection	 PROTO((XtPointer, XtIntervalId  *id));
static void ConnectionCallback	 PROTO((xvobject, kaddr, XEvent *, int *));

static int  dotted_len = 2;
static char dotted[]   = { 3, 1 };


/*-------------------------------------------------------
|
|   Full class attributes
|
--------------------------------------------------------*/

static xvattribute attributes[] = {
{XVW_CONNECTION_BEGIN,      NULL,   XtRObject,   NULL},
{XVW_CONNECTION_END,        NULL,   XtRObject,   NULL},
{XVW_CONNECTION_TYPE,       NULL,   XtRInt,	 NULL},
{XVW_CONNECTION_LINEWIDTH,  NULL,   XtRInt,	 NULL},
{XVW_CONNECTION_UPDATETIME, NULL,   XtRDouble,	 NULL},
{XVW_CONNECTION_CALLBACK,   NULL,   XtRCallback, NULL},
};

/*-------------------------------------------------------
|
|   Full class record constant
|
--------------------------------------------------------*/

#define offset(field) XtOffsetOf(XvwConnectionGadgetRec, connection.field)

static XtResource resources[] = { 
{XVW_CONNECTION_BEGIN, NULL, XtRObject, sizeof(xvobject), 
      offset(begin), XtRObject, (XtPointer) NULL},
{XVW_CONNECTION_END, NULL, XtRObject, sizeof(xvobject), 
      offset(end), XtRObject, (XtPointer) NULL},
{XVW_CONNECTION_TYPE, NULL, XtRInt, sizeof(int), 
      offset(type), XtRImmediate, (XtPointer) KCONNECTION_TYPE_MANHATTAN},
{XVW_CONNECTION_LINEWIDTH, NULL, XtRInt, sizeof(int), 
      offset(linewidth), XtRImmediate, (XtPointer) 0},
{XVW_CONNECTION_UPDATETIME, NULL, XtRDouble, sizeof(double),
      offset(update_time), XtRString, (XtPointer) "0.2"},
{XVW_CONNECTION_CALLBACK, NULL, XtRCallback, sizeof(XtPointer),
      offset(connection_callback), XtRCallback, (XtPointer) NULL},
};
#undef offset


#define SUPERCLASS (&xvwManagerGadgetClassRec)

XvwConnectionGadgetClassRec xvwConnectionGadgetClassRec =
{
  {
    (WidgetClass) SUPERCLASS,		/* superclass		  */	
    "Connection",			/* class_name		  */
    sizeof(XvwConnectionGadgetRec),	/* size			  */
    ClassInitialize,			/* class_initialize	  */
    NULL,				/* class_part_initialize  */
    FALSE,				/* class_inited		  */
    Initialize,				/* initialize		  */
    NULL,				/* initialize_hook	  */
    (XtProc) Realize,			/* realize		  */
    NULL,				/* actions		  */
    0,					/* num_actions		  */
    resources,				/* resources		  */
    XtNumber(resources),		/* resource_count	  */
    NULLQUARK,				/* xrm_class		  */
    FALSE,				/* compress_motion	  */
    XtExposeCompressMaximal,		/* compress_exposure	  */
    FALSE,				/* compress_enterleave    */
    FALSE,				/* visible_interest	  */
    Destroy,				/* destroy		  */
    XtInheritResize,			/* resize		  */
    Redisplay,				/* expose		  */
    SetValues,				/* set_values		  */
    NULL,				/* set_values_hook	  */
    XtInheritSetValuesAlmost,		/* set_values_almost	  */
    NULL,				/* get_values_hook	  */
    NULL,				/* accept_focus		  */
    XtVersion,				/* version		  */
    NULL,				/* callback_private	  */
    NULL,				/* tm_table		  */
    NULL,				/* query_geometry	  */
    NULL,				/* display_accelerator	  */
    NULL				/* extension		  */
  },  /* RectObjClass fields initialization */
  {
    PickGadget,				/* pick object proc       */
    XtInheritEraseSel,			/* erase selected proc    */
    XtInheritRefreshSel,		/* refresh selection proc */
    XtInheritChangeGeometry,		/* change geometry proc   */
  },  /* XvwManagerGadgetClass fields initialization */
  {
    NULL,				/* extension - not used   */
  },  /* XvwConnectionGadgetClass fields initialization */
};
#undef SUPERCLASS

/*
 * xvwConnectionGadgetClass for public consumption
 */
WidgetClass xvwConnectionGadgetClass = (WidgetClass) &xvwConnectionGadgetClassRec;


/*-----------------------------------------------------------
|
|  Routine Name: ClassInitialize
|
|       Purpose: This method is called the first time an  
|                instance of connection object class has been created.
|                It will initialize all the class attributes. 
|
|         Input: None 
|
|        Output: None
|
|    Written By: Mark Young
|          Date: Nov 23, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ClassInitialize(void)
{
	xvw_init_attributes(xvwConnectionGadgetClass, attributes,
		XtNumber(attributes), NULL, 0, 
		"$DESIGN/objects/library/xvobjects/uis/Connection.pane");
	xvw_load_resources("$DESIGN/objects/library/xvobjects/app-defaults/Connection");

        xvw_define_attributes(xvwConnectionGadgetClass,
          XVW_CONNECTION_START,  XtRInt,    ChangeUpdate, NULL,
          XVW_CONNECTION_STOP,   XtRInt,    ChangeUpdate, NULL,
          NULL);
}

/*-----------------------------------------------------------
|
|  Routine Name: Initialize
|
|       Purpose: This will initialize the gc for a new XvwManagerGadget.
|
|         Input: request - not used
|                new     - widget instance after initialization, with
|                          gc initialized
|
|        Output: None
|       Returns: None
|
|    Written By: Mark Young
|          Date: Aug 19, 1992 9:13
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Initialize(
   Widget   request,
   Widget   new,
   ArgList  args,
   Cardinal *num_args)
{
        XvwConnectionGadget xobj = (XvwConnectionGadget) new;
 
	xobj->connection.time_id = 0;
        xobj->connection.gc1 = NULL;
        xobj->connection.gc2 = NULL;
        xobj->connection.gc_update = NULL;
}

/*-----------------------------------------------------------
|
|  Routine Name: Realize
|
|       Purpose: This method will create the graphics context used
|		 in drawing.
|
|         Input: request - not used
|                new     - object instance after initialization, with
|                          text initialized
|
|        Output:
|
|    Written By: Mark Young
|          Date: Mar 09, 1993 12:02
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Realize(
   Widget   widget)
{
	xvobject object = xvw_object(widget);

	AllocateGC(widget);
	xvw_add_event(object, ButtonPressMask, ConnectionCallback, NULL);
}

/*-----------------------------------------------------------
|
|  Routine Name: Redisplay
|
|       Purpose: This routine will redraw the connection object in response
|                to an expose event. 
|
|         Input: widget - the connection object that was exposed 
|                event  - the event that caused the redraw
|                region - the region that was exposed 
|
|        Output: None
|
|    Written By: Mark Young
|          Date: Jun 29, 1992 10:35
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Redisplay(
   Widget widget,
   XEvent *event,
   Region region)
{
	XvwConnectionGadget xobj = (XvwConnectionGadget) widget;

	XPoint   pt;
	xvobject object = xvw_object(widget);
	xvobject end   = xobj->connection.end;
	xvobject begin = xobj->connection.begin;


	/*
	 *  If the begin and end points aren't set then don't draw anything
	 */
	if (!xvw_check_realized(begin) || !xvw_check_realized(end))
	   return;

	RecomputeSize(widget, FALSE);

	/*
	 *  Actually draw the line
	 */
	pt = xobj->connection.pt[0];

	xobj->connection.pt[0].y--;
	if (xobj->connection.type == KCONNECTION_TYPE_MANHATTAN)
	   xobj->connection.pt[0].x--;
	XDrawLines(xvw_display(object), xvw_window(object),xobj->connection.gc2,
		xobj->connection.pt, xobj->connection.num, CoordModePrevious);

	xobj->connection.pt[0].y += 2;
	if (xobj->connection.type == KCONNECTION_TYPE_MANHATTAN)
	   xobj->connection.pt[0].x += 2;
	XDrawLines(xvw_display(object), xvw_window(object),xobj->connection.gc2,
		xobj->connection.pt, xobj->connection.num, CoordModePrevious);

	xobj->connection.pt[0].y++;
	if (xobj->connection.type == KCONNECTION_TYPE_MANHATTAN)
	   xobj->connection.pt[0].x++;
	XDrawLines(xvw_display(object), xvw_window(object),xobj->connection.gc2,
		xobj->connection.pt, xobj->connection.num, CoordModePrevious);

	xobj->connection.pt[0] = pt;
	XDrawLines(xvw_display(object), xvw_window(object),xobj->connection.gc1,
		xobj->connection.pt, xobj->connection.num, CoordModePrevious);
}

/*-----------------------------------------------------------
|
|  Routine Name: SetValues
|
|       Purpose: If the the size or postion of the connection has changed, 
|		 recalculate the size and position of the connection.
|
|         Input: current - the object containing current settings 
|                request - the object containing requested settings
|                new     - the object processed through all set values methods 
|
|        Output: None directly 
|
|       Returns: TRUE (1) if redisplay is required, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Jun 29, 1992 10:35
| Modifications: 
|
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean SetValues(
   Widget   current,
   Widget   request,
   Widget   new,
   ArgList  args,
   Cardinal *num_args)
{
	XvwConnectionGadget cobj = (XvwConnectionGadget) current;
	XvwConnectionGadget nobj = (XvwConnectionGadget) new;

	Boolean  redisplay = FALSE;


	if (cobj->connection.begin     != nobj->connection.begin     ||
	    cobj->connection.end       != nobj->connection.end	     ||
	    cobj->connection.linewidth != nobj->connection.linewidth ||
	    cobj->connection.type      != nobj->connection.type)
	{
	   RecomputeSize(new, TRUE);
	   redisplay = TRUE;
	}

	if (cobj->manager.foreground   != nobj->manager.foreground  ||
	    cobj->manager.background   != nobj->manager.background  ||
	    cobj->connection.linewidth != nobj->connection.linewidth)
	{
           /*
            *  Create the gc to be used with the refresh mechanism
            */
	   XtReleaseGC(new, nobj->connection.gc1);
	   XtReleaseGC(new, nobj->connection.gc2);
	   XtReleaseGC(new, nobj->connection.gc_update);
	   AllocateGC(new);
	   redisplay = TRUE;
	}

	if (cobj->connection.update_time != nobj->connection.update_time)
	{
           if (nobj->connection.time_id != 0)
	      xvw_set_attribute(xvw_object(new), XVW_CONNECTION_START, TRUE);
	}
	return(redisplay);
}

/*-----------------------------------------------------------
|
|  Routine Name: Destroy
|
|       Purpose: This method will remove the connection timeout
|                from the connection gadget before it is destroyed. 
|
|         Input: widget - the widget being destroyed
|
|        Output: none
|
|    Written By: Mark Young
|          Date: Oct 22, 1992 12:24
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Destroy(
   Widget widget)
{
	XvwConnectionGadget xobj = (XvwConnectionGadget) widget;

	if (xobj->connection.time_id != 0)
	   XtRemoveTimeOut(xobj->connection.time_id);

	XtReleaseGC(widget, xobj->connection.gc1);
	XtReleaseGC(widget, xobj->connection.gc2);
	XtReleaseGC(widget, xobj->connection.gc_update);
}

/*-----------------------------------------------------------
|
|  Routine Name: PickGadget
|
|       Purpose: Given a connection object and an attempted picking location,
|                see if the connection has been picked. 
|
|         Input: widget - the connection object we are checking 
|                x      - the x device coordinate that the pick attempt was at 
|                y      - the y device coordinate that the pick attempt was at
|
|        Output: None 
|
|       Returns: TRUE (1) if object was successfully picked, FALSE (0) otherwise|
|    Written By: Mark Young
|          Date: Jun 29, 1992 10:35
| Modifications: 
|
------------------------------------------------------------*/
/* ARGSUSED */
static int PickGadget(
   Widget widget,
   int    x,
   int    y)
{
	XvwConnectionGadget xobj = (XvwConnectionGadget) widget;

	int	 i;
	XPoint	 *pt = xobj->connection.pt;
	Position run, rise, deltax, deltay, temp, xtmp, ytmp;


	if (!xobj->connection.begin || !xobj->connection.end)
	   return(FALSE);

	xtmp = pt[0].x;
	ytmp = pt[0].y;
	for (i = 1; i < xobj->connection.num; i++)
	{
	   run  = pt[i].x;
	   rise = pt[i].y;
	   deltax = x - xtmp;
	   deltay = y - ytmp;
	   if (kabs(run) <= PICK_TOLERANCE)
	   {
	      if (kabs(deltax) <= PICK_TOLERANCE && 
		  y+PICK_TOLERANCE > kmin(ytmp,ytmp+rise) &&
		  y-PICK_TOLERANCE < kmax(ytmp,ytmp+rise))
		 return(TRUE);
	   }
	   else if (kabs(rise) <= PICK_TOLERANCE)
	   {
	      if (kabs(deltay) <= PICK_TOLERANCE && 
		  x+PICK_TOLERANCE > kmin(xtmp,xtmp+run) &&
		  x-PICK_TOLERANCE < kmax(xtmp,xtmp+run))
		 return(TRUE);
	   }
	   else
	   {
	      temp = (rise * deltax) / run;
	      if (kabs(deltay - temp) <= PICK_TOLERANCE)
		 return(TRUE);
	   }
	   xtmp += pt[i].x;
	   ytmp += pt[i].y;
	}
	return(FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: RecomputeSize
|
|       Purpose: recomputes the size
|
|         Input: widget   - the line object whose selection we are refreshing
|		 override - whether to override the initial position check
|        Output:
|    Written By: Mark Young
|          Date: Nov 18, 1993
| Modifications: 
|
------------------------------------------------------------*/
/* ARGSUSED */
static void RecomputeSize(
   Widget  widget,
   Boolean override)
{
	XvwConnectionGadget xobj = (XvwConnectionGadget) widget;

	XtWidgetGeometry temp;
	XPoint   *pt, c[4];
	unsigned int w1, w2, h1, h2;
	xvobject object = xvw_object(widget);
	xvobject parent = xvw_parent(object);
	int i, x, y, xpos1, xpos2, ypos1, ypos2, tmpx, tmpy;

	static int num_steps = 32;
	double step, unit, t, t0, t1, t2, t3;

	/*
	 *  Initialize the points to the begin and end points
	 */
	xvw_translate_coords(parent, 0, 0, &x, &y);
	xvw_translate_coords(xobj->connection.begin, -x, -y, &xpos1, &ypos1);
	xvw_translate_coords(xobj->connection.end,   -x, -y, &xpos2, &ypos2);
	xvw_geometry(xobj->connection.begin, NULL, NULL, &w1, &h1, NULL);
	xvw_geometry(xobj->connection.end, NULL, NULL, &w2, &h2, NULL);

	xpos1 = xpos1+w1; ypos1 = ypos1+h1/2;
	xpos2 = xpos2;    ypos2 = ypos2+h2/2;
	if (!override &&
	    xobj->connection.x1 == xpos1 && xobj->connection.y1 == ypos1 &&
	    xobj->connection.x2 == xpos2 && xobj->connection.y2 == ypos2)
	{
	   return;
	}
	xobj->connection.x1 = xpos1; xobj->connection.y1 = ypos1;
	xobj->connection.x2 = xpos2; xobj->connection.y2 = ypos2;


	pt = xobj->connection.pt;
	if (xobj->connection.type == KCONNECTION_TYPE_LINEAR)
	{
	   xobj->connection.num = 4;
	   pt[0].x = xpos1;      pt[0].y = ypos1;
	   pt[1].x = xpos1 + 10; pt[1].y = ypos1;
	   pt[2].x = xpos2 - 10; pt[2].y = ypos2;
	   pt[3].x = xpos2;      pt[3].y = ypos2;
	}
	else if (xobj->connection.type == KCONNECTION_TYPE_MANHATTAN)
	{
	   if (xpos1+20 < xpos2)
	   {
	      xobj->connection.num = 4;
	      pt[0].x = xpos1;   	        pt[0].y = ypos1;
	      pt[1].x = xpos1 + (xpos2-xpos1)/2; pt[1].y = ypos1;
	      pt[2].x = xpos1 + (xpos2-xpos1)/2; pt[2].y = ypos2;
	      pt[3].x = xpos2;   	        pt[3].y = ypos2;
	   }
	   else
	   {
	      xobj->connection.num = 6;
	      pt[0].x = xpos1;      pt[0].y = ypos1;
	      pt[1].x = xpos1 + 10; pt[1].y = ypos1;
	      pt[2].x = xpos1 + 10; pt[2].y = ypos1 + (ypos2-ypos1)/2;
	      pt[3].x = xpos2 - 10; pt[3].y = ypos1 + (ypos2-ypos1)/2;
	      pt[4].x = xpos2 - 10; pt[4].y = ypos2;
	      pt[5].x = xpos2;      pt[5].y = ypos2;
	   }
	}
	else if (xobj->connection.type == KCONNECTION_TYPE_HEXAGON)
	{
	   if (xpos1+20 < xpos2)
	   {
	      xobj->connection.num = 4;
	      pt[0].x = xpos1;   	        pt[0].y = ypos1;
	      pt[1].x = xpos1+(xpos2-xpos1)/2; pt[1].y = ypos1+(ypos2-ypos1)/4;
	      pt[2].x = xpos1+(xpos2-xpos1)/2; pt[2].y = ypos2-(ypos2-ypos1)/4;
	      pt[3].x = xpos2;   	        pt[3].y = ypos2;
	   }
	   else
	   {
	      xobj->connection.num = 6;
	      pt[0].x = xpos1;      pt[0].y = ypos1;
	      pt[1].x = xpos1 + 10; pt[1].y = ypos1;
	      pt[2].x = xpos1 + 10; pt[2].y = ypos1 + (ypos2-ypos1)/2;
	      pt[3].x = xpos2 - 10; pt[3].y = ypos1 + (ypos2-ypos1)/2;
	      pt[4].x = xpos2 - 10; pt[4].y = ypos2;
	      pt[5].x = xpos2;      pt[5].y = ypos2;
	   }
	}
	else if (xobj->connection.type == KCONNECTION_TYPE_DIAMOND)
	{
	   if (xpos1+20 < xpos2)
	   {
	      xobj->connection.num = 4;
	      pt[0].x = xpos1;   	        pt[0].y = ypos1;
	      pt[1].x = xpos2-(xpos2-xpos1)/4; pt[1].y = ypos1+(ypos2-ypos1)/4;
	      pt[2].x = xpos1+(xpos2-xpos1)/4; pt[2].y = ypos2-(ypos2-ypos1)/4;
	      pt[3].x = xpos2;   	        pt[3].y = ypos2;
	   }
	   else
	   {
	      xobj->connection.num = 6;
	      pt[0].x = xpos1;      pt[0].y = ypos1;
	      pt[1].x = xpos1 + 10; pt[1].y = ypos1;
	      pt[2].x = xpos1 + 10; pt[2].y = ypos1 + (ypos2-ypos1)/2;
	      pt[3].x = xpos2 - 10; pt[3].y = ypos1 + (ypos2-ypos1)/2;
	      pt[4].x = xpos2 - 10; pt[4].y = ypos2;
	      pt[5].x = xpos2;      pt[5].y = ypos2;
	   }
	}
	else if (xobj->connection.type == KCONNECTION_TYPE_SPLINE)
	{

	   if (xpos1+20 < xpos2)
	   {
	      x = kmax(kabs((xpos2-xpos1)/2),20);
	      y = kmax(kabs((ypos2-ypos1)/4),10);
	      c[0].x = xpos1;       c[0].y = ypos1;
	      c[1].x = xpos1 + x;   c[1].y = ypos1;
	      c[2].x = xpos2 - x;   c[2].y = ypos2;
	      c[3].x = xpos2;       c[3].y = ypos2;
	   }
	   else
	   {
	      x = kmax(kabs((xpos2-xpos1)/2),20);
	      y = kmax(kabs((ypos2-ypos1)/2),20);
	      c[0].x = xpos1;     c[0].y = ypos1;
	      c[1].x = xpos1 + x; c[1].y = (ypos1 > ypos2) ? ypos1-y : ypos1+y;
	      c[2].x = xpos2 - x; c[2].y = (ypos1 > ypos2) ? ypos2+y : ypos2-y;
	      c[3].x = xpos2;     c[3].y = ypos2;
	   }

	   unit = 1.0/(num_steps-1);
	   xobj->connection.num = num_steps;
	   for (i = 0, step = 0.0; i < num_steps; i++, step += unit)
	   {
	      t  = (1.0 - step);
	      t0 = t*t*t;
	      t1 = 3*step*t*t;
	      t2 = 3*step*step*t;
	      t3 = step*step*step;
	      pt[i].x = c[0].x*t0 + c[1].x*t1 + c[2].x*t2 + c[3].x*t3 + 0.5;
	      pt[i].y = c[0].y*t0 + c[1].y*t1 + c[2].y*t2 + c[3].y*t3 + 0.5;
	   }
	}

	x = xpos1 = xpos2 = pt[0].x;
	y = ypos1 = ypos2 = pt[0].y;
	for (i = 1; i < xobj->connection.num; i++)
	{
	   xpos1 = kmin(xpos1, pt[i].x);
	   xpos2 = kmax(xpos2, pt[i].x);
	   ypos1 = kmin(ypos1, pt[i].y);
	   ypos2 = kmax(ypos2, pt[i].y);

	   tmpx = pt[i].x;
	   tmpy = pt[i].y;
	   pt[i].x = pt[i].x - x; x = tmpx;
	   pt[i].y = pt[i].y - y; y = tmpy;
	}
	temp.request_mode = CWWidth | CWHeight | CWX | CWY;
	temp.x = (Position) xpos1 - 2;
	temp.y = (Position) ypos1 - 2;
	temp.width  = (Dimension) (xpos2-xpos1) + 4;
	temp.height = (Dimension) (ypos2-ypos1) + 4;
	(void) XtMakeGeometryRequest(widget, &temp, &temp);
}

/*-----------------------------------------------------------
|
|  Routine Name: AllocateGC
|
|       Purpose: allocate the gc's used by the connection
|
|         Input: widget   - the line object whose selection we are refreshing
|		 override - whether to override the initial position check
|        Output:
|    Written By: Mark Young
|          Date: Nov 18, 1993
| Modifications: 
|
------------------------------------------------------------*/
/* ARGSUSED */
static void AllocateGC(
   Widget widget)
{
	XvwConnectionGadget xobj = (XvwConnectionGadget) widget;

	XGCValues values;
	unsigned long mask;


        /*
         *  Create the gc to be used with the refresh mechanism
         */
        values.line_width = xobj->connection.linewidth;
        values.foreground = xobj->manager.background;
        values.background = xobj->manager.foreground;
        mask = GCForeground | GCBackground | GCLineWidth;
	xobj->connection.gc2 = XtGetGC(widget, mask, &values);

        values.line_width = xobj->connection.linewidth;
        values.foreground = xobj->manager.foreground;
        values.background = xobj->manager.background;
        mask = GCForeground | GCBackground | GCLineWidth;
	xobj->connection.gc1 = XtGetGC(widget, mask, &values);

        mask = GCForeground | GCBackground | GCLineWidth | GCLineStyle;
        if (xobj->connection.linewidth > 0)
	   values.line_width = xobj->connection.linewidth;
	else
	   values.line_width = xobj->connection.linewidth;

        values.line_style = LineDoubleDash;
        values.foreground = xobj->manager.foreground;
        values.background = xobj->manager.background;
	xobj->connection.gc_update = XtGetGC(widget, mask, &values);
	XSetDashes(XtDisplay(widget), xobj->connection.gc_update, 0,
		dotted, dotted_len);
}

/*-----------------------------------------------------------
|
|  Routine Name: ChangeUpdate - changes the connection updating
|
|       Purpose: This routine will set the whether we should be refreshing
|		 the 
|
|         Input: object - the object in which we will be setting the
|			  update
|		 attribute - the attribute to be set
|		 calldata  - the calldata to be used with the attribute
|
|        Output:
|    Written By: Mark Young
|          Date: Nov 23, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static int ChangeUpdate(
   xvobject object,
   String   attribute,
   kaddr    calldata)
{
	XvwConnectionGadget xobj = (XvwConnectionGadget) xvw_widget(object);

	unsigned long speed;
	XtAppContext app_context;


        if (xobj->connection.time_id != 0)
           XtRemoveTimeOut(xobj->connection.time_id);

	if (kstrcmp(attribute, XVW_CONNECTION_START) == 0)
	{
	   /*
	    * Compute the update
	    */
	   if (xvw_check_realized(object) == TRUE &&
	       xvw_check_visible(object) == TRUE)
	   {
	      speed = xobj->connection.update_time * 1000;
	      app_context = xvw_appcontext(object);
	      xobj->connection.offset = 0;
              xobj->connection.time_id = XtAppAddTimeOut(app_context, speed,
			UpdateConnection, (XtPointer) object);
	   }
	}
	else if (kstrcmp(attribute, XVW_CONNECTION_STOP) == 0)
	{
	   xobj->connection.time_id = 0;
	   if (xvw_check_realized(object) == TRUE &&
	       xvw_check_visible(object) == TRUE)
	   {
	      XDrawLines(xvw_display(object), xvw_window(object),
		xobj->connection.gc1, xobj->connection.pt,
		xobj->connection.num, CoordModePrevious);
	   }
	}
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: UpdateConnection - changes the connection updating
|
|       Purpose: This routine will refresh the updated connection
|         Input: object - the object which update the connection
|        Output:
|    Written By: Mark Young
|          Date: Nov 23, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void UpdateConnection(
   XtPointer     client_data,
   XtIntervalId  *id)
{
	xvobject object = (xvobject) client_data;
	XvwConnectionGadget xobj = (XvwConnectionGadget) xvw_widget(object);

	unsigned long speed;
	XtAppContext app_context;


	if (xobj->connection.offset == 0)
	   xobj->connection.offset = 4;
	else xobj->connection.offset--;
	XSetDashes(xvw_display(object), xobj->connection.gc_update,
		xobj->connection.offset, dotted, dotted_len);
	XDrawLines(xvw_display(object), xvw_window(object),
		xobj->connection.gc_update, xobj->connection.pt,
		xobj->connection.num, CoordModePrevious);

	/*
	 * Compute the update
	 */
	speed = xobj->connection.update_time * 1000;
	app_context = xvw_appcontext(object);
        xobj->connection.time_id = XtAppAddTimeOut(app_context, speed,
			UpdateConnection, (XtPointer) object);
}

/*-----------------------------------------------------------
|
|  Routine Name: ConnectionCallback
|
|       Purpose: Connection callback, which is used to call any connections

|         Input: object - the selected object
|                client_data  - (not used)
|                event  - the event
|        Output: dispatch - whether to continue dispatching this event
|    Written By: Mark Young
|          Date: Dec 6, 1993
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ConnectionCallback(
   xvobject object,
   kaddr    client_data,
   XEvent   *event,
   int	    *dispatch)
{
	XvwConnectionGadget xobj = (XvwConnectionGadget) xvw_widget(object);

	if (event->type != ButtonPress || event->xbutton.button != Button1)
	   return;

	XtCallCallbacks((Widget) xobj, XVW_CONNECTION_CALLBACK, NULL);
}


/************************************************************
*
*  Routine Name: xvw_create_connection - create a connection object 
*
*       Purpose: The connection object provides a mechanism for visually
*		 linking two objects.  It is used in \fHcantata\fP to connect 
*                glyph objects together, but can be used to connect
*                any two objects.  A line will be drawn between the two
*		 objects; the line will resize and redraw if either of the
*		 two objects are moved.  Several styles of connections
*		 are provided, which can be used to change the appearance of
*		 the connection.
*
*         Input: parent - the parent object; NULL will cause a
*                         default toplevel to be created automatically
*                name   - the name with which to reference the object 
*
*        Output:
*       Returns: The connection object on success, NULL on failure.
*  Restrictions:
*    Written By: Mark Young 
*          Date: Jun 29, 1992 10:31
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

xvobject xvw_create_connection(
   xvobject parent,
   char   *name)
{
	xvobject  object;


	/*
	 *  Create the line object
	 */
	object = xvw_create(parent, FALSE, TRUE, name, ConnectionGadgetClass);
	if (object != NULL)
	{
	   xvw_set_attributes(object,
                XVW_MENUABLE,     TRUE,     /* menuable       */
                XVW_RESIZABLE,    FALSE,    /* resizable      */
                XVW_SELECTABLE,   FALSE,    /* selectable     */
		NULL);
	}
	return(object);
}
