 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>	              Khoros Workspace Widget Utilities
   >>>>   Static:
   >>>>			WorkspaceGUIInitialize()
   >>>>			WorkspaceGUIActions()
   >>>>			WorkspaceGUICallback()
   >>>>			WorkspaceCheckRunning()
   >>>>  Private:
   >>>>			WorkspaceInitClipboard()
   >>>>			WorkspaceChildren()
   >>>>			WorkspaceRun()
   >>>>			WorkspaceCallbacks()
   >>>>			WorkspaceGlyphRunCallback()
   >>>>			WorkspaceGlyphModifiedCallbacks()
   >>>>			WorkspaceActions()
   >>>>			WorkspaceAnnounce()
   >>>>			WorkspaceMap()
   >>>>			WorkspaceUnmap()
   >>>>   Public:
   >>>>	
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <xvlang/WorkspaceP.h>

static void WorkspaceGUIInitialize  PROTO((xvobject));
static void WorkspaceGUIActions	    PROTO((xvobject, kaddr, kaddr));
static void WorkspaceGUICallback    PROTO((xvobject, kaddr, kaddr));
static void WorkspaceCheckRunning   PROTO((xvobject, int *, int *));

static xvobject clipboard = NULL;

static char  *options[] = { "Craftsman", "Edit GUI", "Delete Selections",
			    "Delete All Selections", "Attributes" };


/*-----------------------------------------------------------
|
|  Routine Name: WorkspaceGUIInitialize
|
|       Purpose: Initialize the options to be used with the object's option
|                menu.
|
|         Input: workspace - the workspace which to initialize the options for.
|    Written By: Mark Young
|          Date: Apr 25, 1994
|
------------------------------------------------------------*/
/* ARGSUSED */
static void WorkspaceGUIInitialize(
   xvobject workspace)
{
        XvwWorkspaceWidget xwid = (XvwWorkspaceWidget) xvw_widget(workspace);
 
        int      i;
        kguide   *guide;
        xvobject button, menu;
        kform_struct **kformstructs;
        kselection *sel_list, *selection;

	/*
	 *  Check to see if the options have already been initialized...
	 */
	if (xwid->workspace.wkspgui_initialized == TRUE)
	   return;

	xwid->workspace.wkspgui->editable = XVF_FULL_EDIT;
	if (!kvf_check_for_single_pane(xwid->workspace.wkspgui,"Initialize"))
	   return;

	sel_list = xwid->workspace.wkspgui->subform->guide->pane->sel_list;
	selection = kvf_gui_named_item_exists(sel_list, KUIS_ROUTINE, -1);
	if (selection != NULL)
	{
	   xvf_add_extra_call(selection->back_kformstruct, WorkspaceRun,
			(kaddr) workspace, XVF_CALL_SUBSTITUTE);
	}

	/* find the selected pane */
	guide = kvf_search_sel_guide(xwid->workspace.wkspgui->subform);

	/* get the options button on the backplane for the selections */
	if ((kformstructs = kvf_get_selection_kformstructs(guide->pane,
			KUIS_OPTIONS, &i)) != NULL && i == 1)
	{
	   menu = kformstructs[0]->Selptr->back;
	   for (i = 0; i < knumber(options); i++)
	   {
	      button = xvw_create_button(menu, options[i]);
	      xvw_set_attribute(button, XVW_LABEL, options[i]);
	      xvw_insert_callback(button, XVW_BUTTON_SELECT, TRUE,
			WorkspaceGUIActions, workspace);

              if (kstrcmp(options[i], "Delete Selections") == 0)
              {
                 xvw_sensitive(button, FALSE);
                 xwid->workspace.delete_button = button;
              }
	   }
	   kfree(kformstructs);
	}

	/*
	 * get all the selections so that we can raise them when the form is
	 * put into edit mode.
	 */
	xvw_insert_callback(guide->pane->back, XVW_SELECT_CALLBACK, FALSE,
			WorkspaceGUICallback, xwid);
	xwid->workspace.wkspgui_initialized = TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: WorkspaceGUIActions
|
|       Purpose: This routine will be called when a wkspgui action is
|		 activated.
|
|         Input: object      - which button being manipulated
|                client_data - the glyph
|                call_data   - not used
|    Written By: Mark Young
|          Date: Dec 1, 1993
|
------------------------------------------------------------*/
/* ARGSUSED */
static void WorkspaceGUIActions(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	xvobject workspace = (xvobject) client_data;
	XvwWorkspaceWidget xwid = (XvwWorkspaceWidget) xvw_widget(workspace);

	int      i;
	kguide   *guide;
	char     *name = NULL;


	xvw_get_attribute(object, XVW_LABEL, &name);
	if (kstrcmp(name, "Craftsman") == 0)
	{
	   kspawn("craftsman");
	}
	else if (kstrcmp(name, "Edit GUI") == 0)
	{
	   int editing = TRUE;

	   guide = kvf_search_sel_guide(xwid->workspace.wkspgui->subform);
	   if (guide == NULL)
	      return;

	   xvw_get_attribute(guide->pane->back, XVW_EDIT_MODE_ON, &editing);
	   if (editing == FALSE)
	   {
	      /* 
	       *  Turn edit mode on for the pane, and also make sure that
	       *  the options button rides on top of the editing window...
	       */
	      xvw_set_attribute(guide->pane->back, XVW_EDIT_MODE_ON, TRUE);
	   }
	   else
	   {
	      /* Turn editing off...  */
	      xvw_set_attribute(guide->pane->back, XVW_EDIT_MODE_ON, FALSE);
	   }
	}
	else if (kstrcmp(name, "Delete Selections") == 0)
	{
	   xvobject     *selections;
	   kform_struct *kformstruct;
	   int          num_sels = 0;

	   guide = kvf_search_sel_guide(xwid->workspace.wkspgui->subform);
	   if (guide == NULL)
	      return;

	   xvw_get_attributes(guide->pane->back,
		XVW_SELECTIONS, &selections,
		XVW_NUM_SELECTIONS, &num_sels,
		NULL);

	   for (i = 0; i < num_sels; i++)
	   {
	      kformstruct = xvf_get_kformstruct(xwid->workspace.wkspgui,
				selections[i]);
	      if (kformstruct != NULL)
		 xvf_set_attribute(kformstruct, XVF_DELETE, TRUE);
	   }
	}
	else if (kstrcmp(name, "Delete All Selections") == 0)
	{
	   int          num = 0;
	   kform_struct **kformstructs;

	   guide = kvf_search_sel_guide(xwid->workspace.wkspgui->subform);
	   if (guide == NULL)
	      return;

           kformstructs = kvf_get_selection_kformstructs(guide->pane,NONE,&num);
	   for (i = 0; i < num; i++)
	   {
	      if (karray_locate((char **) xwid->workspace.selections,
			kformstructs[i], xwid->workspace.num_selections) == -1)
	      {
		 xvf_set_attribute(kformstructs[i], XVF_DELETE, TRUE);
	      }
	   }
	   kfree(kformstructs);
	}
	else if (kstrcmp(name, "Attributes") == 0)
	{
	   xvw_activate_menu(workspace);
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: WorkspaceGUICallback
|
|       Purpose: This routine will be called when the wkspgui is put into
|                edit mode.
|
|         Input: object      - which button being manipulated
|                client_data - the glyph
|                call_data   - not used
|    Written By: Mark Young
|          Date: Aug 28, 1994
|
------------------------------------------------------------*/
/* ARGSUSED */
static void WorkspaceGUICallback(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
        int     editing = *((int *) call_data);
        XvwWorkspaceWidget xwid = (XvwWorkspaceWidget) client_data;
 
	int	 i;
	xvobject button;
 
        if (xwid->workspace.selections == NULL)
           return;
 
        if (editing)
        {
           if (xwid->workspace.delete_button != NULL)
              xvw_sensitive(xwid->workspace.delete_button, TRUE);

           for (i = 0; i < xwid->workspace.num_selections; i++)
	   {
	      if ((button = xvf_get_xvobject(xwid->workspace.selections[i],
                        XVF_BUTTON_OBJ, TRUE)) != NULL)
	      {
		 xvw_raise(button);
	      }
	   }
        }
        else
        {
           if (xwid->workspace.delete_button != NULL)
              xvw_sensitive(xwid->workspace.delete_button, FALSE);
        }
}

/*-----------------------------------------------------------
|
|  Routine Name: WorkspaceCheckRunning
|
|       Purpose: This routine will check to see if the workspace is running
|		 or not.
|
|         Input: workspace   - whether the workspace is running or not
|    Written By: Mark Young
|          Date: Jan 28, 1995
|
------------------------------------------------------------*/
/* ARGSUSED */
static void WorkspaceCheckRunning(
   xvobject workspace,
   int	    *running,
   int	    *scheduling)
{
	xvobject *selections;
	int      i, status, type, num, run, schedule;
        XvwWorkspaceWidget xwid = (XvwWorkspaceWidget) xvw_widget(workspace);


	selections = WorkspaceChildren(workspace, FALSE, &num);
	for (i = 0, run = schedule = FALSE; i < num; i++)
	{
	   if (xvw_check_subclass(selections[i], NodeWidgetClass) == FALSE)
	      continue;

	   xvw_get_attributes(selections[i],
		XVW_NODE_TYPE,    &type,
		XVW_NODE_EXECUTE, &status,
		NULL);

	   if (status == KNODE_SCHEDULED || status == KNODE_RUNNING)
	   {
	       run = TRUE;
	       if (status == KNODE_SCHEDULED || type != KNODE_TYPE_SINK)
	       {
		  schedule = TRUE;
	          break;
	       }
	   }
	}
	if (running    != NULL) *running    = run;
	if (scheduling != NULL) *scheduling = schedule;
	kfree(selections);
}

/*-----------------------------------------------------------
|
|  Routine Name: WorkspaceInitClipboard - routine for initializing the
|				workspace clipboard
|
|       Purpose: This routine initializes the workspace clipboard shared
|		 by the different
|
|    Written By: Mark Young
|          Date: Jul 04, 1994
| Modifications:
|
------------------------------------------------------------*/

void WorkspaceInitClipboard(void)
{
	xvobject toplevel;

	if (clipboard != NULL)
	   return;

	toplevel  = xvw_create_application_shell("Clipboard", NULL, NULL);
	clipboard = xvw_create_workspace(toplevel, "Clipboard");
	xvw_set_attributes(clipboard,
		XVW_WIDTH, 750,
		XVW_HEIGHT, 512,
		NULL);
}

/*-----------------------------------------------------------
|
|  Routine Name: WorkspaceChildren - get the workspace children
|
|       Purpose: This routine gets the workspace children.
|         Input: parent   - 
|                selected - whether to get the selected children, or all
|        Output: num      - number of returned children
|       Returns: the children or NULL
|
|    Written By: Mark Young
|          Date: Aug 21, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
xvobject *WorkspaceChildren(
   xvobject workspace,
   int      selected,
   int	    *num)
{
	XvwWorkspaceWidget xwid = (XvwWorkspaceWidget) xvw_widget(workspace);
	xvobject *children = NULL, parent = xwid->viewport.plane;

	if (selected == TRUE)
	{
	   xvw_get_attributes(parent,
		XVW_SELECTIONS,     &children,
		XVW_NUM_SELECTIONS, num,
		NULL);
	   children = (xvobject *) kdupalloc(children, *num * sizeof(xvobject));
	}

	if (children == NULL)
	   children = xvw_children(parent, NULL, num);
	return(children);
}

/*-----------------------------------------------------------
|
|  Routine Name: WorkspaceRun - the run button on the wkspgui executed
|
|       Purpose: This performs a run of the workspace via the run button
|		 on the wkspgui.
|
|         Input: client_data - the workspace widget
|
|        Output:
|	Returns:
|    Written By: Mark Young
|          Date: Jul 23, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */ 
void WorkspaceRun(
   kaddr    client_data)
{
	xvobject workspace = (xvobject) client_data;

	xvw_set_attribute(workspace, XVW_WORKSPACE_RUN, TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: WorkspaceCallbacks - workspace button callbacks routine
|
|       Purpose: This routine is the handler for performing certain workspace
|		 button callbacks functions, such as mapping or unmapping the
|		 clipboard.
|
|         Input: object - object in which the object was invoked
|		 client_data - the workspace widget
|		 calldata    - not used
|
|        Output:
|	Returns:
|    Written By: Mark Young
|          Date: Jul 23, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */ 
void WorkspaceCallbacks(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	xvobject workspace = (xvobject) client_data;
	XvwWorkspaceWidget xwid = (XvwWorkspaceWidget) xvw_widget(workspace);

	kcontrol   *pane;
	kselection *selection;
	char	   line[KLENGTH];
	xvobject   parent, connector;


	if (object == xwid->workspace.titlebar)
	{
	   if (workspace == clipboard)
	      xvw_set_attribute(workspace, XVW_WORKSPACE_SHOW_CLIPBOARD, FALSE);
	   else
	      WorkspaceUnmap(workspace);
	}
	else if (object == xwid->workspace.input)
	{
	   kstrcpy(line,"-I 1 0 0 1 0 1 45x1+1+1 ' ' 'Input File' 'ifile' i1");
	   pane = xwid->workspace.wkspgui->subform->guide->pane;
	   selection = kvf_create_new_selection(pane, line);

	   parent = xvw_create_manager(workspace, "Port Parent");
	   xvw_set_attributes(parent,
		XVW_XPOSITION, xwid->viewport.xoffset + 100,
		XVW_YPOSITION, xwid->viewport.yoffset + 100,
		XVW_RESIZABLE, FALSE,
		NULL);
	   connector = xvw_create_port(parent, "Input Port");
	   xvw_set_attributes(connector,
		XVW_PORT_SELECTION, selection,
		XVW_PORT_TYPE, KPORT_TYPE_INPUT,
		XVW_PORT_CONNECTION_PARENT, workspace,
		XVW_PORT_MODIFIED, TRUE,
		NULL);
	}
	else if (object == xwid->workspace.output)
	{
	   kstrcpy(line,"-O 1 0 0 1 0 1 45x1+1+2 ' ' 'Output File' 'ofile' o1");
	   pane = xwid->workspace.wkspgui->subform->guide->pane;
	   selection = kvf_create_new_selection(pane, line);

	   parent = xvw_create_manager(workspace, "Port Parent");
	   xvw_set_attributes(parent,
		XVW_XPOSITION, xwid->viewport.xoffset + xwid->core.width  - 100,
		XVW_YPOSITION, xwid->viewport.yoffset + xwid->core.height - 100,
		XVW_RESIZABLE, FALSE,
		NULL);
	   connector = xvw_create_port(parent, "Output Port");
	   xvw_set_attributes(connector,
		XVW_PORT_SELECTION, selection,
		XVW_PORT_TYPE, KPORT_TYPE_OUTPUT,
		XVW_PORT_CONNECTION_PARENT, workspace,
		XVW_PORT_MODIFIED, TRUE,
		NULL);
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: WorkspaceGlyphRunCallbacks - glyph run callback routine
|
|       Purpose: This routine is the handler for performing certain workspace
|		 button callbacks functions, such as mapping or unmapping the
|		 clipboard.
|
|         Input: object - object in which the object was invoked
|		 client_data - the workspace widget
|		 calldata    - not used
|
|        Output:
|	Returns:
|    Written By: Mark Young
|          Date: Jul 23, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */ 
void WorkspaceGlyphRunCallback(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	xvobject workspace = (xvobject) client_data;
	XvwWorkspaceWidget xwid = (XvwWorkspaceWidget) xvw_widget(workspace);

	int scheduling;
	int running = *((int *) call_data);

	if ((running == KNODE_STOPPED && xwid->workspace.running == FALSE)    ||
	    (running == KNODE_RUNNING && xwid->workspace.scheduling == FALSE) ||
	    (running == KNODE_SCHEDULED && xwid->workspace.scheduling == TRUE))
	{
	   return;
	}
	WorkspaceCheckRunning(workspace, &running, &scheduling);

	if (running == xwid->workspace.running &&
	    scheduling == xwid->workspace.scheduling)
	{
	   return;
	}
	xwid->workspace.running = running;
	xwid->workspace.scheduling = scheduling;
	XtCallCallbacks((Widget) xwid, XVW_WORKSPACE_RUN_CALLBACK, &running);
}

/*-----------------------------------------------------------
|
|  Routine Name: WorkspaceGlyphModifiedCallbacks - glyph modified callback
|						   routine
|
|       Purpose: This routine is the handler for performing certain workspace
|		 button callbacks functions, such as mapping or unmapping the
|		 clipboard.
|
|         Input: object - object in which the object was invoked
|		 client_data - the workspace widget
|		 calldata    - not used
|
|        Output:
|	Returns:
|    Written By: Mark Young
|          Date: Jul 23, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */ 
void WorkspaceGlyphModifiedCallback(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	xvobject workspace = (xvobject) client_data;
	XvwWorkspaceWidget xwid = (XvwWorkspaceWidget) xvw_widget(workspace);

	int modified = *((int *) call_data);

	if (xwid->workspace.responsive && xwid->workspace.running && modified)
	   xvw_set_attribute(object, XVW_NODE_EXECUTE, TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: WorkspaceActions
|
|       Purpose: WorkspaceActions is used to set the certain actions associated
|		 with the current workspace.
|
|         Input: object    - the object in which we will be setting the
|			     attribute value for
|		 attribute - the desired attribute
|		 calldata  - the value to be set
|
|        Output:
|       Returns: TRUE (1) if redisplay is required, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Nov 27, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int WorkspaceActions(
   xvobject workspace,
   char     *attribute,
   kaddr    calldata)
{
	XvwWorkspaceWidget xwid = (XvwWorkspaceWidget) xvw_widget(workspace);

	kfile    *file;
	char     *filename;
	Widget   widget = (Widget) xwid;
	int      i, selected, run, num = 0;
	xvobject *selections, *exported, object, tmp, procedure;


	object = xwid->viewport.plane;
	if (kstrcmp(attribute, XVW_WORKSPACE_RUN) == 0)
	{
	   run = *((int *) calldata);
	   selections = WorkspaceChildren(workspace, FALSE, &num);
	   for (i = 0; i < num; i++)
	   {
	      if (xvw_check_subclass(selections[i], NodeWidgetClass) == TRUE)
		 xvw_set_attribute(selections[i], XVW_NODE_EXECUTE, run);
	   }
	   kfree(selections);
	   kannounce(XVLANG, "WorkspaceActions",
		"Running workspace '%s'... done", xvw_name(workspace));

	   /*
	    *  Check to see if we are running...  There are conditions that
	    *  if we aren't running, but requested that we should run, then
	    *  we better inform everyone.
	    */
	   if (run == TRUE)
	   {
	      WorkspaceCheckRunning(workspace, &run, NULL);
	      if (run == FALSE)
	      {
	         xwid->workspace.running = FALSE;
	         xwid->workspace.scheduling = FALSE;
	         XtCallCallbacks(widget, XVW_WORKSPACE_RUN_CALLBACK, &run);
	      }
	   }
	}
	else if (kstrcmp(attribute, XVW_WORKSPACE_STEP) == 0)
	{
	   selections = WorkspaceChildren(workspace, FALSE, &num);
	   if (num > 0)
	   {
	      for (i = 0; i < num; i++)
	      {
	         if (xvw_check_subclass(selections[i], NodeWidgetClass) == TRUE)
		 {
		    run = FALSE;
		    xvw_get_attribute(selections[i], XVW_NODE_RUNNABLE, &run);
		    if (run == TRUE)
		    {
		       xvw_set_attribute(selections[i], XVW_NODE_EXECUTE, run);
		       break;
		    }
		 }
	      }
	      kfree(selections);
	   }
	   kannounce(XVLANG, "WorkspaceActions",
		"Single step workspace '%s'... done", xvw_name(workspace));
	}
        else if (kstrcmp(attribute, XVW_WORKSPACE_SHOW_CLIPBOARD) == 0)
	{
	   WorkspaceInitClipboard();

	   if (xvw_check_mapped(xvw_toplevel(clipboard)) == FALSE)
	      xvw_realize(xvw_toplevel(clipboard));
	   else
	      xvw_unrealize(xvw_toplevel(clipboard));
	}
	else if (kstrcmp(attribute, XVW_WORKSPACE_RESET) == 0)
	{
	   selections = WorkspaceChildren(workspace, FALSE, &num);
	   for (i = 0; i < num; i++)
	   {
	      if (xvw_check_subclass(selections[i], NodeWidgetClass) == TRUE)
		 xvw_set_attribute(selections[i], XVW_NODE_RESET, TRUE);
	   }
	   kfree(selections);
	   kannounce(XVLANG, "WorkspaceActions",
		"Resetting workspace '%s'... done", xvw_name(workspace));
	}
	else if (kstrcmp(attribute, XVW_WORKSPACE_CLEAR) == 0 ||
		 kstrcmp(attribute, XVW_CANVAS_DELETE) == 0)
	{
	   if (kstrcmp(attribute, XVW_WORKSPACE_CLEAR) == 0)
	   {
	      selected = FALSE;
	      kfree(xwid->workspace.filename);
	   }
	   else
	      selected = TRUE;

	   if (object != clipboard)
	   {
	      file = ktmpfile();
	      WorkspaceSave(widget, file, FALSE, FALSE, selected, FALSE,
		   NULL, NULL);
	      krewind(file);
	      xwid->workspace.saved = (kfile **) karray_add((char **)
                   xwid->workspace.saved, file, xwid->workspace.saved_num++);
	   }
	   selections = WorkspaceChildren(workspace, selected, &num);

	   for (i = 0; i < num; i++)
	   {
	      if (xvw_check_subclass(selections[i], ConnectionGadgetClass))
	      {
	         xvw_destroy(selections[i]);
		 selections[i] = NULL;
	      }
	   }
	   for (i = 0; i < num; i++)
	   {
	      if (selections[i] == NULL)
		 continue;

	      if (xvw_check_subclass(selections[i], NodeWidgetClass))
	         xvw_destroy(selections[i]);
	   }
	   kfree(selections);

	   if (kstrcmp(attribute, XVW_WORKSPACE_CLEAR) == 0)
	   {
	      kfree(xwid->workspace.filename);
	      kexpr_variables_clear((long) workspace);
	   }

	   kannounce(XVLANG, "WorkspaceActions",
		   "Clearing workspace '%s'... done", xvw_name(workspace));

	   /*
	    *  Kludge:  To update the variable list in cantata
	   WorkspaceMap(workspace);
	    */
	}
	else if (kstrcmp(attribute, XVW_WORKSPACE_CREATE_PROCEDURE) == 0 ||
		 kstrcmp(attribute, XVW_WORKSPACE_CREATE_COUNTLOOP) == 0 ||
		 kstrcmp(attribute, XVW_WORKSPACE_CREATE_WHILELOOP) == 0)
	{
	   /*
	    *  Now save the contents of the current workspace and restore
	    *  it into the clipboard workspace (aka restore).
	    */
	   file = ktmpfile();
	   WorkspaceSave(widget, file, FALSE, TRUE, TRUE, TRUE,
				&exported, &num);
	   xvw_set_attribute(workspace, XVW_CANVAS_DELETE, TRUE);

	   /*
	    *  Create the procedure/whileloop/countloop
	    */
	   if (kstrcmp(attribute, XVW_WORKSPACE_CREATE_PROCEDURE) == 0)
	   {
	      procedure = xvw_create_procedure(workspace, "Procedure");
	      xvw_set_attribute(procedure, XVW_NODE_NAME, "Procedure");
	   }
	   else if (kstrcmp(attribute, XVW_WORKSPACE_CREATE_COUNTLOOP) == 0)
	   {
	      procedure = xvw_create_loop(workspace, "CountLoop");
	      xvw_set_attributes(procedure,
			XVW_NODE_NAME, "Count Loop",
			XVW_LOOP_TYPE, KLOOP_TYPE_COUNT,
			NULL);
	   }
	   else if (kstrcmp(attribute, XVW_WORKSPACE_CREATE_WHILELOOP) == 0)
	   {
	      procedure = xvw_create_loop(workspace, "WhileLoop");
	      xvw_set_attributes(procedure,
			XVW_NODE_NAME, "While Loop",
			XVW_LOOP_TYPE, KLOOP_TYPE_WHILE,
			NULL);
	   }
	   tmp = NULL;
	   xvw_get_attribute(procedure, XVW_PROCEDURE_WORKSPACE_OBJECT, &tmp);
	   exported = (xvobject *) karray_add((char **) exported,
			(kaddr) procedure, num++);

	   /*
	    *  Restore the workspace into procedure
	    */
	   krewind(file);
	   WorkspaceRestore(xvw_widget(tmp), file, 0, 0, exported, num);
	   /*GlyphUpdateForm(xvw_widget(procedure));*/
	   kfree(exported);
	   kfclose(file);

	   /*
            *  Kludge:  To update the variable list in cantata
           WorkspaceMap(workspace);
            */
	}
	else if (kstrcmp(attribute, XVW_CANVAS_DUPLICATE) == 0)
	{
	   /*
	    *  Now save the contents of the current workspace and restore
	    *  it, thus duplicating the currently selected glyphs. 
	    */
	   file = ktmpfile();
	   WorkspaceSave(widget, file, FALSE, FALSE, TRUE, FALSE, NULL, NULL);
	   krewind(file);
	   WorkspaceRestore(widget, file, 10, 10, NULL, 0);
	   kfclose(file);
	}
	else if (kstrcmp(attribute, XVW_WORKSPACE_SAVE)    == 0 ||
		 kstrcmp(attribute, XVW_WORKSPACE_SAVEALL) == 0)
	{
	   filename = *((String *) calldata);


	   if (filename == NULL)
	      filename = xwid->workspace.filename;
	    
	   if ((file = kfoutput(filename)) == NULL)
	   {
	      kerror(XVLANG, "WorkspaceSave", "Unable to open file '%s' for \
saving workspace", !filename ? "(NULL)" : filename);
	      return(FALSE);
	   }
	   else if (filename != xwid->workspace.filename &&
		    kstrcmp(attribute, XVW_WORKSPACE_SAVEALL) == 0)
	   {
	      kfree(xwid->workspace.filename);
	      xwid->workspace.filename = kstrdup(filename);
	   }

	   if (kstrcmp(attribute, XVW_WORKSPACE_SAVE) == 0)
	      WorkspaceSave(widget, file, TRUE, FALSE, TRUE, TRUE, NULL, NULL);
	   else
	      WorkspaceSave(widget, file, TRUE, FALSE, FALSE, FALSE, NULL,NULL);

	   kfclose(file);
	   kannounce(XVLANG, "WorkspaceActions", "Saving workspace '%s' to \
file '%s'... done", xvw_name(workspace), filename);
	}
	else if (kstrcmp(attribute, XVW_WORKSPACE_CHECK) == 0)
	{
	   selections = WorkspaceChildren(workspace, FALSE, &num);
	   if (num > 0)
	   {
	      for (i = 0; i < num; i++)
	      {
	         if (xvw_check_subclass(selections[i], NodeWidgetClass) == TRUE)
		    xvw_set_attribute(selections[i], XVW_NODE_TEST, TRUE);
	      }
	      kfree(selections);
	   }
	   else
	   {
	      run = FALSE;
	      WorkspaceGlyphRunCallback(NULL, (kaddr) workspace, (kaddr) &run);
	   }
	   kannounce(XVLANG, "WorkspaceActions",
		"Running workspace '%s'... done", xvw_name(workspace));
	}
	else if (kstrcmp(attribute, XVW_WORKSPACE_RESTORE) == 0)
	{
	   filename = *((String *) calldata);

	   if ((file = kfinput(filename)) == NULL)
	   {
	      kerror(XVLANG, "WorkspaceAction", "Unable to open file '%s' for \
restoring workspace", !filename ? "(NULL)" : filename);
	      return(FALSE);
	   }
	   else if (filename != xwid->workspace.filename)
	   {
	      kfree(xwid->workspace.filename);
	      xwid->workspace.filename = kstrdup(filename);
	   }

	   WorkspaceRestore(widget, file, 0, 0, NULL, 0);
	   kfclose(file);
	   kannounce(XVLANG, "WorkspaceActions", "Restoring workspace '%s' \
from file '%s'... done", xvw_name(workspace), filename);

	   /*
	    *  Kludge:  To update the variable list in cantata
	   WorkspaceMap(workspace);
	    */
	}
	else if (kstrcmp(attribute, XVW_WORKSPACE_SAVE_WKSPGUI) == 0)
	{
	   filename = *((String *) calldata);

	   kvf_print_UIS_file(xwid->workspace.wkspgui, filename, TRUE, TRUE);
	}
	else if (kstrcmp(attribute, XVW_WORKSPACE_SHOW_WKSPGUI) == 0)
	{
	   /*
	    *  If the workspace GUI is not mapped then map it, otherwise unmap
	    *  it.
	    */
	   if (!xvw_check_mapped(xwid->workspace.wkspgui->subform->toplevel))
              xvf_map_subform(xwid->workspace.wkspgui->subform);
           else
              xvf_unmap_subform(xwid->workspace.wkspgui->subform);

	   WorkspaceGUIInitialize(workspace);
	}
	else if (kstrcmp(attribute, XVW_WORKSPACE_EXPORT_SELECTIONS) == 0)
	{
	   kform *form;

	   selections = WorkspaceChildren(workspace, TRUE, &num);
	   for (i = 0; i < num; i++)
	   {
	      if (xvw_check_subclass(selections[i], GlyphWidgetClass) == TRUE)
	      {
		 form = NULL;
	         xvw_get_attribute(selections[i], XVW_GLYPH_FORM, &form);
                 if (form != NULL)
		 {
		    xvw_copy_pane(xwid->workspace.wkspgui,
			  xwid->workspace.procedure, selections[i], form);
		 }
	      }
	   }
	   kfree(selections);
	   kannounce(XVLANG, "WorkspaceActions", "Export Selections to \
workspace GUI '%s'... done", xvw_name(workspace));
	}
	else if (kstrcmp(attribute, XVW_WORKSPACE_REDRAW) == 0)
	{
	   xvw_refresh(object);
	   kannounce(XVLANG, "WorkspaceActions",
		"Redrawing workspace '%s'... done", xvw_name(workspace));
	}
        else if (kstrcmp(attribute, XVW_CANVAS_UNDO) == 0)
        {
	   /*
	    *  Check to see if the number saved is greater than 0, if so
	    *  then restore the last saved workspace and delete it from the
	    *  list.
	    */
	   if (xwid->workspace.saved_num > 0)
	   {
	      file = xwid->workspace.saved[xwid->workspace.saved_num-1];
	      WorkspaceRestore(widget, file, 0, 0, NULL, 0);
	      xwid->workspace.saved = (kfile **) karray_delete((char **)
		   xwid->workspace.saved, file, xwid->workspace.saved_num--);
	      kfclose(file);
	   }
        }
        else if (kstrcmp(attribute, XVW_CANVAS_COPY) == 0)
        {
	   WorkspaceInitClipboard();

	   /*
	    *  Go ahead and clear the clipboard, before restoring..
	    */
	   xvw_set_attribute(clipboard, XVW_CANVAS_DELETE, TRUE);

	   /*
	    *  Now save the contents of the current workspace and restore
	    *  it into the clipboard workspace (aka restore).
	    */
	   file = ktmpfile();
	   WorkspaceSave(widget, file, FALSE, FALSE, TRUE, FALSE, NULL, NULL);
	   krewind(file);
	   WorkspaceRestore(xvw_widget(clipboard), file, 0, 0, NULL, 0);
	   kfclose(file);
        }
        else if (kstrcmp(attribute, XVW_CANVAS_PASTE) == 0)
        {
	   WorkspaceInitClipboard();

	   file = ktmpfile();
	   WorkspaceSave(xvw_widget(clipboard), file, FALSE, FALSE, TRUE, FALSE,
			NULL, NULL);
	   krewind(file);
	   WorkspaceRestore(widget, file, 0, 0, NULL, 0);
	   kfclose(file);
        }
        else if (kstrcmp(attribute, XVW_CANVAS_CUT) == 0)
        {
	   WorkspaceInitClipboard();

	   xvw_set_attributes(workspace,
		XVW_CANVAS_COPY, TRUE,
		XVW_CANVAS_DELETE, TRUE,
		NULL);
        }
	else if (kstrcmp(attribute, XVW_WORKSPACE_INFO) == 0)
	{
	   int loops, conditionals, procedures, glyphs, connections, nodes;

	   selections = WorkspaceChildren(workspace, FALSE, &num);
	   loops = conditionals = procedures = glyphs = connections = nodes = 0;
	   for (i = 0; i < num; i++)
	   {
	      if (xvw_class(selections[i]) == LoopWidgetClass)
		 loops++;
	      else if (xvw_class(selections[i]) == ConditionalWidgetClass)
		 conditionals++;
	      else if (xvw_class(selections[i]) == ProcedureWidgetClass)
		 procedures++;
	      else if (xvw_class(selections[i]) == GlyphWidgetClass)
		 glyphs++;
	      else if (xvw_class(selections[i]) == ConnectionGadgetClass)
		 connections++;
	      else if (xvw_class(selections[i]) == NodeWidgetClass)
		 nodes++;
	   }
	   kfree(selections);
	   kinfo(KFORCE, "For Workspace '%s':\n\n\
  Glyphs       %d\n  Procedures   %d\n  Conditionals %d\n  Loops        %d\n\
  Connections  %d\n  Nodes        %d", xvw_name(workspace), glyphs,
		procedures, conditionals, loops, connections, nodes);
	}
	else
	{
	   kerror(XVLANG, "WorkspaceActions", "invalid attribute %s",
			attribute);
	   return(FALSE);
	}
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: WorkspaceAnnounce
|
|       Purpose: This routine will perform a variety of workspace announces
|		 for such things as clear, wkspgui, run, etc.
|         Input: object      - the button to announce for
|                client_data - the workspace widget
|                event       - event the activated us
|		 dispatch    - whether to continue dispatch the event
|    Written By: Mark Young
|          Date: Nov 23, 1993
|
------------------------------------------------------------*/
/* ARGSUSED */ 
void WorkspaceAnnounce(
   xvobject object,
   kaddr    client_data,
   XEvent   *event,
   int      *dispatch)
{
	xvobject workspace = (xvobject) client_data;
	XvwWorkspaceWidget xwid = (XvwWorkspaceWidget) xvw_widget(workspace);

	char *tmp = NULL;

	if (event->type == LeaveNotify)
	{
	   kannounce(XVLANG, "WorkspaceAnnounce", "");
	}
	else if (event->type == EnterNotify)
	{
	   kannounce(XVLANG, "WorkspaceAnnounce", "%s:  %s",
			xvw_name(workspace), tmp);
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: WorkspaceMap
|
|       Purpose: This routine will map the workspace canvas.  It is
|		 also used to issue a callback on the top most level
|		 of the workspace callback.
|         Input: object - the workspace to be mapped
|    Written By: Mark Young
|          Date: Dec 08, 1994
|
------------------------------------------------------------*/
/* ARGSUSED */ 
void WorkspaceMap(
   xvobject workspace)
{
	Widget   widget;
	xvobject object = workspace;


	xvw_map(workspace);
	while (xvw_check_subclass(xvw_parent(object), WorkspaceWidgetClass))
	    object = xvw_parent(object);

	widget = xvw_widget(object);
        XtCallCallbacks(widget, XVW_WORKSPACE_ACTIVE_CALLBACK, &workspace);
}

/*-----------------------------------------------------------
|
|  Routine Name: WorkspaceUnmap
|
|       Purpose: This routine will unmap the workspace canvas.  It is
|		 also used to issue a callback on the top most level
|		 of the workspace callback.
|         Input: object - the workspace to be unmapped
|    Written By: Mark Young
|          Date: Dec 08, 1994
|
------------------------------------------------------------*/
/* ARGSUSED */ 
void WorkspaceUnmap(
   xvobject workspace)
{
	Widget   widget;
	xvobject object = workspace;


	xvw_unmap(workspace);
	while (xvw_check_subclass(xvw_parent(object), WorkspaceWidgetClass))
	    object = xvw_parent(object);

	widget = xvw_widget(object);
	workspace = xvw_parent(workspace);
        XtCallCallbacks(widget, XVW_WORKSPACE_ACTIVE_CALLBACK, &workspace);
}
