 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>	              Khoros Workspace Widget Utilities
   >>>>   Static:
   >>>>			RestoreGlyph()
   >>>>			RestoreConditional()
   >>>>			RestoreProcedure()
   >>>>			RestoreLoop()
   >>>>			RestoreNodeConnections()
   >>>>			RestoreControlConnections()
   >>>>			RestoreExternalConnections()
   >>>>  Private:
   >>>>			WorkspaceRestore()
   >>>>   Public:
   >>>>	
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <xvlang/WorkspaceP.h>
#include <xvlang/GlyphP.h>

/*-----------------------------------------------------------
|
|  Routine Name: RestoreNode - Restore the node
|
|       Purpose: This routine does the actually restoring of a glyph or control
|		 from a workspace file.
|
|         Input: file  - the file the information is suppose to be read from
|		 line  - the information just read
|		 xoffset,yoffset - initial x & y offset
|		 beta_format     - whether we are restoring the glyph using
|				   the beta format
|        Output: ident - the glyph indentifier
|    Written By: Mark Young
|          Date: Aug 26, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static xvobject RestoreNode(
    kfile    *file,
    xvobject parent,
    char     *line,
    int	     xoffset,
    int	     yoffset,
    int	     beta_format,
    int      *ident,
    xvobject glyph)
{
        XvwWorkspaceWidget xwid = (XvwWorkspaceWidget) xvw_widget(parent);

	XvwGlyphWidget xglyph;
	kguide     *guide;
	kselection *selection;
	kform      *form = NULL;
	kobject    toolbox = NULL, program = NULL, paneobj = NULL;

	klist    *list, *tblist = NULL;
	int      i, xpos, ypos, tbname_token, oname_token;
	char     *name = NULL, *panefile = NULL, **info, *tmp, *var, *varname;
	char     tbname[KLENGTH],  oname[KLENGTH], command[KLENGTH],
		 variables[KLENGTH], newname[KLENGTH], identifier[KLENGTH];


	newname[0] = '\0';
	if (ksscanf(line, "%[^:]:%[^#:]:%[^:]:%d:%d:%d:%[^:]:%[^\n]",identifier,
		tbname, oname, ident, &xpos, &ypos, variables, command) != 8 &&
	    ksscanf(line, "%[^:]:%[^#:]:%[^:]:%[^:]:%d:%d:%d:%[^:]:%[^\n]",
		identifier, tbname, oname, newname, ident, &xpos, &ypos,
		variables, command) != 9 &&
	    ksscanf(line, "%[^#:]:%[^:]:%d:%d:%d:%[^\n]", tbname,
		oname, ident, &xpos, &ypos, command) != 6)
	{
	   return(NULL);
	}
	xvw_set_attributes(glyph,
		   XVW_XPOSITION, xpos + xoffset,
		   XVW_YPOSITION, ypos + yoffset,
		   XVW_NODE_NAME, oname,
		   NULL);

	xvw_get_contents(NULL, &tblist);
	tbname_token = kstring_to_token(tbname);
	oname_token  = kstring_to_token(oname);
	if ((list = klist_locate(tblist, (kaddr) tbname_token)) != NULL &&
	    (list = (klist *) klist_clientdata(list))           != NULL &&
	    (list = klist_locate(list, (kaddr) oname_token))    != NULL)
	{
	   info = (char **) klist_clientdata(list);
	   name = info[0];
	   panefile = info[5];
	}
	else
	{
	   kinfo(KXVLIB, "Swing and a miss... for toolbox '%s' oname '%s\n",
			tbname, oname);
	   if ((toolbox = kcms_open_toolbox(tbname)) == NULL ||
	       (program = kcms_open_cmobj(toolbox, oname)) == NULL)
	   {
	      if (toolbox == NULL)
	      {
	         kerror(XVLANG, "WorkspaceRestore", "Failed to open \
toolbox '%s'.  Failed to restore the object '%s' within toolbox '%s'.  Please \
check to see that toolbox '%s' exists and is accessible to '%s' before \
restoring this workspace.", tbname, oname, tbname, tbname, kprog_get_program());
	      }
	      else
	      {
	         kerror(XVLANG, "WorkspaceRestore", "Failed to open \
object '%s'.  Failed to restore the object '%s' within toolbox '%s'.  Please \
check to see that object '%s' exists within this toolbox and is accessible to \
'%s' before restoring this workspace.", oname, oname, tbname, oname,
				kprog_get_program());
	       }
	       return(NULL);
	   }
	   kcms_get_attribute(program, KCMS_CMOBJ_ICON_NAME, &name);
	   kcms_get_attribute(program, KCMS_CMOBJ_UIS_PANE, &paneobj);
	   kcms_get_attribute(paneobj, KCMS_PATH, &panefile);
	}

	if (kstrlen(newname) > 0)
	   name = newname;

	xvw_set_attributes(glyph,
		   XVW_NODE_NAME,      name,
		   XVW_GLYPH_FORMFILE, panefile,
		   XVW_GLYPH_TBNAME,   tbname,
		   XVW_GLYPH_ONAME,    oname,
		   NULL);

	if (xvw_get_attribute(glyph, XVW_GLYPH_FORM, &form) && form != NULL)
	{
	   /*
	    *  Need to make sure that the command doesn't contain new line
	    *  tokens.  If they do then we will want to substitute these with
	    *  real new line characters.
	    */
	   kstring_replace(command, "\\n", "\n", command);
	   if (command[0] == '-')
	   {
	      tmp = kstring_cat(" ", command, NULL);
	      xvf_modify_form_from_cmd(form, tmp); kfree(tmp);
	   }
	   else xvf_modify_form_from_cmd(form, command);

           /* find the selected pane */
           guide = kvf_search_sel_guide(form->subform);
           xvf_clear_modified(guide->pane->sel_list);
	}

	/*
	 *  Clear the temporary output connections
	 */
	xglyph = (XvwGlyphWidget) xvw_widget(glyph);
	for (i = 0; i < xglyph->glyph.osize && beta_format == TRUE; i++)
	{
	   tmp = NULL;
	   xvf_get_attribute(xglyph->glyph.osels[i], XVF_FILE_NAME, &tmp);
	   if (kstrstr(tmp, "/io") != NULL)
	      xvf_set_attribute(xglyph->glyph.osels[i], XVF_LITERAL, NULL);
	   kfree(tmp);
	}

	/*
	 *  Clear the temporary input connections
	 */
	for (i = 0; i < xglyph->glyph.isize; i++)
	{
	   tmp = NULL;
	   xvf_get_attribute(xglyph->glyph.isels[i], XVF_FILE_NAME, &tmp);
	   if (kstrstr(tmp, "/io") != NULL)
	      xvf_set_attribute(xglyph->glyph.isels[i], XVF_LITERAL, NULL);
	   xvw_set_attribute(xglyph->glyph.iports[i], XVW_PORT_MODIFIED, TRUE);
	   kfree(tmp);
	}

	/*
	 *  Export the desired selections, specified in the variables
	 *  array, to the workspace gui.
	 */
	if ((var = kstring_cleanup(variables, variables)) != NULL)
	{
	   while (var != NULL)
	   {
	      if ((tmp = kstrchr(var, ' ')) != NULL)
	         *tmp++ = '\0';

	      if ((varname = kstrchr(var, '!')) != NULL)
		 *varname++ = '\0';

	      if ((selection = kvf_variable_sel_search(form, var)) != NULL)
	      {
	         xvw_copy_selection(xwid->workspace.wkspgui,
			xwid->workspace.procedure, glyph,
			selection->back_kformstruct, varname);
		 for (i = 0; i < xglyph->glyph.isize; i++)
		 {
		    if (xglyph->glyph.isels[i] == selection->back_kformstruct)
		    {
		       xvw_set_attribute(xglyph->glyph.iports[i],
					XVW_PORT_DAV, TRUE);
		    }
		 }
	      }
	      var = tmp;
	   }
	}
	if (program) kcms_close(program);
	if (toolbox) kcms_close(toolbox);
	return(glyph);
}

/*-----------------------------------------------------------
|
|  Routine Name: RestoreGlyph - Restore the glyph
|
|       Purpose: This routine does the actually restoring of a glyph from a
|		 workspace file.
|
|         Input: file  - the file the information is suppose to be read from
|		 line  - the information just read
|		 xoffset,yoffset - initial x & y offset
|		 beta_format     - whether we are restoring the glyph using
|				   the beta format
|        Output: ident - the glyph indentifier
|    Written By: Mark Young
|          Date: Aug 26, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static xvobject RestoreGlyph(
    kfile    *file,
    xvobject parent,
    char     *line,
    int	     xoffset,
    int	     yoffset,
    int	     beta_format,
    int      *ident)
{
	xvobject glyph;

	glyph = xvw_create_glyph(parent, "glyph");
	if (!RestoreNode(file, parent, line, xoffset, yoffset,
			beta_format, ident, glyph))
	{
	   xvw_destroy(glyph);
	   return(NULL);
	}
	return(glyph);
}

/*-----------------------------------------------------------
|
|  Routine Name: RestoreConditional - Restore the conditional
|
|       Purpose: This routine does the actually restoring of a control from a
|		 workspace file.
|
|         Input: file  - the file the information is suppose to be read from
|		 line  - the information just read
|		 xoffset,yoffset - initial x & y offset
|		 beta_format     - whether we are restoring the control using
|				   the beta format
|        Output: ident - the control indentifier
|    Written By: Mark Young
|          Date: Aug 26, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static xvobject RestoreConditional(
    kfile    *file,
    xvobject parent,
    char     *line,
    int	     xoffset,
    int	     yoffset,
    int	     beta_format,
    int      *ident)
{
	xvobject conditional;

	conditional = xvw_create_conditional(parent, "conditional");
	if (!RestoreNode(file, parent, line, xoffset, yoffset,
			beta_format, ident, conditional))
	{
	   xvw_destroy(conditional);
	   return(NULL);
	}
	return(conditional);
}

/*-----------------------------------------------------------
|
|  Routine Name: RestoreProcedure - Restore the procedure
|
|       Purpose: This routine does the actually restoring of a glyph from a
|		 workspace file.
|
|         Input: file  - the file the information is suppose to be read from
|		 line  - the information just read
|		 xoffset,yoffset - initial x & y offset
|        Output: ident - the glyph indentifier
|    Written By: Mark Young
|          Date: Aug 26, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static xvobject RestoreProcedure(
    kfile    *file,
    xvobject parent,
    char     *line,
    int	     xoffset,
    int	     yoffset,
    int      *ident)
{
	int   xpos, ypos;
	kform *form = NULL;
	xvobject procedure, workspace = NULL;
	char  *tmp, oname[KLENGTH], command[KLENGTH];


	if (ksscanf(line, "ProcedureBegin:%[^:]:%d:%d:%d:%[^\n]", oname,
			ident, &xpos, &ypos, command) != 5)
	{
	   return(NULL);
	}
	procedure = xvw_create_procedure(parent, "procedure");
	xvw_set_attributes(procedure,
		XVW_NODE_NAME, oname,
		XVW_XPOSITION, xpos + xoffset,
		XVW_YPOSITION, ypos + yoffset,
		NULL);

	xvw_get_attributes(procedure,
		XVW_GLYPH_FORM, &form,
		XVW_PROCEDURE_WORKSPACE_OBJECT, &workspace,
		NULL);

	if (workspace != NULL)
	   WorkspaceRestore(xvw_widget(workspace), file, 0, 0, NULL, 0);

	if (form != NULL)
	{
	   if (command[0] == '-')
	   {
	      tmp = kstring_cat(" ", command, NULL);
	      xvf_modify_form_from_cmd(form, tmp); kfree(tmp);
	   }
	   else xvf_modify_form_from_cmd(form, command);
	}
	return(procedure);
}

/*-----------------------------------------------------------
|
|  Routine Name: RestoreLoop - Restore the loop
|
|       Purpose: This routine does the actually restoring of a loop from a
|		 workspace file.
|
|         Input: file  - the file the information is suppose to be read from
|		 line  - the information just read
|		 xoffset,yoffset - initial x & y offset
|        Output: ident - the glyph indentifier
|    Written By: Mark Young
|          Date: Aug 26, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static xvobject RestoreLoop(
    kfile    *file,
    xvobject parent,
    char     *line,
    int	     xoffset,
    int	     yoffset,
    int      *ident)
{
	int   xpos, ypos;
	kform *form = NULL;
	xvobject loop, workspace = NULL;
	char  *tmp, name[KLENGTH], oname[KLENGTH], command[KLENGTH];


	if (ksscanf(line, "LoopBegin:%[^:]:%[^:]:%d:%d:%d:%[^\n]", name, oname,
			ident, &xpos, &ypos, command) != 6)
	{
	   return(NULL);
	}
	loop = xvw_create_loop(parent, "loop");
	xvw_set_attributes(loop,
		XVW_NODE_NAME,   name,
		XVW_GLYPH_ONAME, oname,
		XVW_XPOSITION, xpos + xoffset,
		XVW_YPOSITION, ypos + yoffset,
		NULL);

	xvw_get_attributes(loop,
		XVW_GLYPH_FORM, &form,
		XVW_PROCEDURE_WORKSPACE_OBJECT, &workspace,
		NULL);

	if (workspace != NULL)
	   WorkspaceRestore(xvw_widget(workspace), file, 0, 0, NULL, 0);

	if (form != NULL)
	{
	   if (command[0] == '-')
	   {
	      tmp = kstring_cat(" ", command, NULL);
	      xvf_modify_form_from_cmd(form, tmp); kfree(tmp);
	   }
	   else xvf_modify_form_from_cmd(form, command);
	}
	return(loop);
}

/*-----------------------------------------------------------
|
|  Routine Name: RestoreNodeConnections - Restore the glyph connections
|
|       Purpose: This routine does the actually restoring of a glyph from a
|		 workspace file.
|
|         Input: file  - the file the information is suppose to be read from
|		 line  - the information just read
|		 xoffset,yoffset - initial x & y offset
|        Output: ident - the glyph indentifier
|    Written By: Mark Young
|          Date: Aug 26, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void RestoreNodeConnections(
    kfile    *file,
    xvobject parent,
    char     *line,
    int	     xoffset,
    int	     yoffset,
    int      *ident,
    klist    *glyphs)
{
	int      inum,   onum;
	klist    *ilist, *olist;
	xvobject iglyph, oglyph, iport, oport;
	char     ivar[KLENGTH], ovar[KLENGTH];


	if (ksscanf(line, "NodeConnection:%d:%[^:]:%d:%[^:]", &onum, ovar,
			&inum, ivar) != 4 &&
	    ksscanf(line, "Connection:%d:%[^:]:%d:%[^:]", &onum, ovar,
                        &inum, ivar) != 4 &&
	    ksscanf(line, "%d:%[^:]:%d:%[^:]", &onum, ovar, &inum, ivar) != 4)
	{
	   return;
	}

	if ((ilist = klist_locate(glyphs, (kaddr) inum)) != NULL)
	   iglyph = (xvobject) klist_clientdata(ilist);
	else iglyph = NULL;

	if ((olist = klist_locate(glyphs, (kaddr) onum)) != NULL)
	   oglyph = (xvobject) klist_clientdata(olist);
	else oglyph = NULL;

	iport = GlyphFindNode(iglyph, ivar, NULL, NULL);
	oport = GlyphFindNode(oglyph, NULL, ovar, NULL);
	if (iport && oport)
	   PortCreateConnection(iport, oport);
}

/*-----------------------------------------------------------
|
|  Routine Name: RestoreControlConnections - Restore the glyph control
|					     connections
|
|       Purpose: This routine does the actually restoring of the control
|		 connections between the different glyphs.
|
|         Input: file  - the file the information is suppose to be read from
|		 line  - the information just read
|		 xoffset,yoffset - initial x & y offset
|        Output: ident - the glyph indentifier
|    Written By: Mark Young
|          Date: Oct 20, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void RestoreControlConnections(
    kfile    *file,
    xvobject parent,
    char     *line,
    int	     xoffset,
    int	     yoffset,
    int      *ident,
    klist    *glyphs)
{
	int      inum,   onum;
	klist    *ilist, *olist;
	xvobject iglyph, oglyph;


	if (ksscanf(line, "ControlConnection:%d:%d:", &onum, &inum) != 2)
	   return;

	if ((ilist = klist_locate(glyphs, (kaddr) inum)) != NULL)
	   iglyph = (xvobject) klist_clientdata(ilist);
	else iglyph = NULL;

	if ((olist = klist_locate(glyphs, (kaddr) onum)) != NULL)
	   oglyph = (xvobject) klist_clientdata(olist);
	else oglyph = NULL;

	if (iglyph && oglyph)
	   GlyphCreateConnection(iglyph, oglyph);
}

/*-----------------------------------------------------------
|
|  Routine Name: RestoreExternalConnections - Restore the external glyph
|
|       Purpose: This routine does the actually restoring of the external
|		 node and control connections between the different glyphs.
|
|         Input: file  - the file the information is suppose to be read from
|		 line  - the information just read
|		 xoffset,yoffset - initial x & y offset
|		 
|        Output: ident - the glyph indentifier
|    Written By: Mark Young
|          Date: Oct 20, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void RestoreExternalConnections(
    kfile    *file,
    xvobject parent,
    char     *line,
    int	     xoffset,
    int	     yoffset,
    int      *ident,
    klist    *glyphs,
    xvobject *exported,
    int	     num_exported)
{
	kselection   *selection;
	klist        *ilist, *olist;
	kform_struct *kformstruct = NULL;
	char	     ivar[KLENGTH], ovar[KLENGTH];
	int          i, inum,  onum, external, node = FALSE, control = FALSE;

	XvwGlyphWidget xglyph;
	xvobject  iglyph, oglyph, iport, oport;
	static xvobject procedure = NULL;


	if (ksscanf(line, "ExternalConnectionControl:%d:%d:%d:",
			&external, &onum, &inum) == 3)
	{
	   control = TRUE;
	}
	else if (ksscanf(line, "ExternalConnectionNode:%d:%d:%[^:]:%d:%[^:]:",
                        &external, &onum, ovar, &inum, ivar) == 5)
	{
	   node = TRUE;
	}
	else
	   return;


	/*
	 *  Total kludge.  Just don't ask, because i'll be too embarassed
	 *  to answer.
	 */
	if (num_exported > 0 && procedure != exported[num_exported-1])
	{
	   procedure = exported[num_exported-1];
	}

	if (external == 0)
	{
	   if (inum < num_exported)
	      iglyph = exported[inum];
	   else iglyph = NULL;

 
           if ((olist = klist_locate(glyphs, (kaddr) onum)) != NULL)
              oglyph = (xvobject) klist_clientdata(olist);
           else oglyph = NULL;
	}
	else
	{
           if ((ilist = klist_locate(glyphs, (kaddr) inum)) != NULL)
              iglyph = (xvobject) klist_clientdata(ilist);
           else iglyph = NULL;

	   if (onum < num_exported)
	      oglyph = exported[onum];
	   else oglyph = NULL;
	}

	if (control == TRUE)
	{
	   if (!iglyph || !oglyph)
	      return;

	   if (external == 0)
	      GlyphCreateConnection(iglyph, procedure);
	   else
	      GlyphCreateConnection(procedure, oglyph);
	}
	else if (node == TRUE)
	{
	   xglyph = (XvwGlyphWidget) xvw_widget(procedure);

	   if (external == 0)
	   {
	      iport      = GlyphFindNode(iglyph, ivar, NULL, NULL);
	      (void) GlyphFindNode(oglyph, NULL, ovar, &kformstruct);

	      if (kformstruct != NULL && kformstruct->Selptr->extensions !=NULL)
	         selection = kformstruct->Selptr->extensions[0];
	      else
	         selection = NULL;

	      for (i = 0, oport = NULL; i < xglyph->glyph.osize; i++)
	      {
		 if (selection == xglyph->glyph.osels[i]->Selptr)
		 {
		    oport = xglyph->glyph.oports[i];
		    break;
		 }
	      }
	   }
	   else
	   {
	      iport = GlyphFindNode(iglyph, ivar, NULL, &kformstruct);
	      if (iport != NULL)
		 xvw_set_attribute(iport, XVW_PORT_DAV, TRUE);

	      oport = GlyphFindNode(oglyph, NULL, ovar, NULL);
	      if (kformstruct != NULL && kformstruct->Selptr->extensions !=NULL)
		 selection = kformstruct->Selptr->extensions[0];
	      else
		 selection = NULL;

	      for (i = 0, iport = NULL; i < xglyph->glyph.isize; i++)
	      {
		 if (selection == xglyph->glyph.isels[i]->Selptr)
		 {
		    iport = xglyph->glyph.iports[i];
		    break;
		 }
	      }
	   }

	   if (!iport || !oport)
	      return;

	   PortCreateConnection(iport, oport);
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: WorkspaceRestore
|
|       Purpose: WorkspaceRestore is used to restore the glyph's within a
|		 workspace.
|
|         Input: widget - the workspace widget to restore to
|		 file   - the open file descriptor in which to read the
|			  workspace from
|		 xoffset - the x offset to add to each glyph
|		 yoffset - the y offset to add to each glyph
|       Returns: TRUE (1) if redisplay is required, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Nov 30, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int WorkspaceRestore(
   Widget widget,
   kfile  *file,
   int    xoffset,
   int    yoffset,
   xvobject *exported,
   int	    num_exported)
{
	XvwWorkspaceWidget xwid = (XvwWorkspaceWidget) widget;

	klist    *list, *glyphs = NULL;
	xvobject object, glyph, workspace;
	char     line[KLENGTH], dummy[KLENGTH], error[KLENGTH];
	int      num, ident, add_selected = FALSE, idummy, i = 0;

	object = xwid->viewport.plane;
	workspace = xvw_object(widget);

	/*
	 *  If there are children in the workspace then go ahead and unselect
	 *  them and then set a flag so that we can restore the workspace
	 *  selected.
	 */
	xvw_get_attribute(object, XVW_NUM_CHILDREN, &num);
	if (num > 0)
	{
	   xvw_set_attribute(object, XVW_SELECT_DELETE_ALL, TRUE);
	   add_selected = TRUE;
	}

	while (kfgets(line, KLENGTH, file) != NULL)
	{
	   if (line[0] == '#' || line[0] == '\0' ||
	       kstring_cleanup(line, line) == NULL)
	      continue;

	   glyph = NULL; i++;
	   if (kstring_subcmp(line, "Glyph:") == 0)
	   {
	      glyph = RestoreGlyph(file, workspace, line, xoffset,
				yoffset, FALSE, &ident);
	   }
	   else if (kstring_subcmp(line, "NodeConnection:") == 0 ||
		    kstring_subcmp(line, "Connection:") == 0)
	   {
	      RestoreNodeConnections(file, workspace, line, xoffset,
				yoffset, &ident, glyphs);
	   }
	   else if (kstring_subcmp(line, "Conditional:") == 0)
	   {
	      glyph = RestoreConditional(file, workspace, line, xoffset,
				yoffset, FALSE, &ident);
	   }
	   else if (kstring_subcmp(line, "ControlConnection:") == 0)
	   {
	      RestoreControlConnections(file, workspace, line, xoffset,
				yoffset, &ident, glyphs);
	   }
	   else if (kstring_subcmp(line, "ProcedureBegin:") == 0)
	   {
	      glyph = RestoreProcedure(file, workspace, line, xoffset,
				yoffset, &ident);
	   }
	   else if (kstring_subcmp(line, "LoopBegin:") == 0)
	   {
	      glyph = RestoreLoop(file, workspace, line, xoffset,
				yoffset, &ident);
	   }
	   else if (kstring_subcmp(line, "ProcedureEnd:") == 0 ||
		    kstring_subcmp(line, "LoopEnd:") == 0)
	   {
	      break;
	   }
	   else if (kstring_subcmp(line, "ExternalConnection") == 0)
	   {
	      RestoreExternalConnections(file, workspace, line, xoffset,
			yoffset, &ident, glyphs, exported, num_exported);
	   }
	   else if (kstring_subcmp(line, "AnnotationsBegin") == 0)
	   {
	      xvw_restore_annotations(object, file);
	   }
	   else if (kstring_subcmp(line, "VariablesBegin:") == 0)
	   {
	      while (kfgets(line, KLENGTH, file) != NULL)
	      {
		  if (kstring_cleanup(line, line) == NULL || line[0] == '#')
		     continue;

		  if (kstring_subcmp(line, "VariablesEnd:") != 0)
		     kexpr_evaluate_generic((long) workspace, line,
				KSTRING, NULL, error);
		  else
		     break;
	      }
	   }
	   else if (ksscanf(line, "%[^#:]:%[^:]:%d:%d:%d:%[^:]", dummy,
			dummy, &idummy, &idummy, &idummy, dummy) == 6)
	   {
	      glyph = RestoreGlyph(file, workspace, line, xoffset,
				yoffset, TRUE, &ident);
	   }
	   else if (ksscanf(line,"%d:%[^:]:%d:%[^:]", &idummy, dummy,
			&idummy, dummy) == 4)
	   {
	      RestoreNodeConnections(file, workspace, line, xoffset,
				yoffset, &ident, glyphs);
	   }
	   else
	      i--;

	   if (glyph != NULL)
	      glyphs = klist_add(glyphs, (kaddr) ident, glyph);
 	}

	if (add_selected == TRUE)
	{
	   for (list = glyphs; list != NULL;  list = klist_next(list))
	   {
	       glyph = (xvobject) klist_clientdata(list);
	       xvw_set_attribute(object, XVW_SELECT_ADD, glyph);
	   }
	}
	klist_free(glyphs, NULL);

	if (i == 0)
	{
	   kerror(XVLANG, "WorkspaceRestore", "Invalid Workspace.  Sorry, but \
'%s' does not appear to be a valid workspace.  Please verify that this is a \
valid Khoros 2.0 workspace and try again.", kfile_filename(kfileno(file)));
	}
	return(TRUE);
}
