 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>	              Khoros Glyph Widget Utilities
   >>>>  Private:
   >>>> 		GlyphInitForm()
   >>>> 		GlyphUpdateForm()
   >>>>			SetFormfile()
   >>>>			GetFormfile()
   >>>> 		GlyphHandler()
   >>>> 		GlyphFormExpression()
   >>>> 		GlyphCreateConnection()
   >>>> 		GlyphFormCallback()
   >>>>   Static:
   >>>> 		OptionsInitialize()
   >>>> 		UpdateNetworkType()
   >>>> 		OptionsCallback()
   >>>> 		OptionsActions()
   >>>> 		ComposerCallback()
   >>>> 		InputFilenameCallback()
   >>>> 		InputDavCallback()
   >>>> 		OutputDavCallback()
   >>>> 		SelectedCallback()
   >>>> 		GlyphRunCallback()
   >>>> 		ConnectionDestroyed()
   >>>> 		ConnectionCallback()
   >>>>   Public:
   >>>>	
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <kutils/ksignal.h>
#include <xvlang/GlyphP.h>
#include <xvobjects/xvobjects.h>

static void OptionsInitialize PROTO((xvobject));
static void UpdateNetworkType PROTO((xvobject));
static void OptionsCallback   PROTO((xvobject, kaddr, kaddr));
static void OptionsActions    PROTO((xvobject, kaddr, kaddr));
static void ComposerCallback  PROTO((int, kstatus, kaddr, char *));
static void InputFilenameCallback PROTO((xvobject, kaddr, kaddr));
static void InputDavCallback  PROTO((xvobject, kaddr, kaddr));
static void OutputDavCallback PROTO((xvobject, kaddr, kaddr));
static void SelectedCallback  PROTO((xvobject, kaddr, kaddr));
static void ConnectionDestroyed   PROTO((xvobject, kaddr, kaddr));
static void ConnectionCallback	  PROTO((xvobject, kaddr, kaddr));

static char  *options[] = { "Bugs/Todo", "Composer", "Edit GUI", "Save GUI",
	    "Export to Workspace GUI", "Attributes", "Info", "Destroy" };


/*-----------------------------------------------------------
|
|  Routine Name: OptionsInitialize
|
|       Purpose: Initialize the options to be used with the object's option
|		 menu.
|
|         Input: object - the object in which to initialize the options for
|    Written By: Mark Young
|          Date: Apr 25, 1994
|
------------------------------------------------------------*/
/* ARGSUSED */ 
static void OptionsInitialize(
   xvobject glyph)
{
	XvwGlyphWidget xwid = (XvwGlyphWidget) xvw_widget(glyph);

	int      i;
	kguide   *guide;
	xvobject button, menu;
	kform_struct **kformstructs;


	/*
	 *  Check to see if the options have already been initialized...
	 */
	if (xwid->glyph.options_initialized == TRUE)
	   return;

	/* find the selected pane */
	guide = kvf_search_sel_guide(xwid->glyph.form->subform);

	/* get the options button on the backplane for the selections */
	if ((kformstructs = kvf_get_selection_kformstructs(guide->pane,
			KUIS_OPTIONS, &i)) != NULL && i == 1)
	{
	   menu = kformstructs[0]->Selptr->back;
	   for (i = 0; i < knumber(options); i++)
	   {
	      button = xvw_create_button(menu, options[i]);
	      xvw_set_attribute(button, XVW_LABEL, options[i]);
	      xvw_insert_callback(button, XVW_BUTTON_SELECT, TRUE,
			OptionsActions, glyph);

	      if (kstrcmp(options[i], "Export to Workspace GUI") == 0)
	      {
		 xvw_sensitive(button, FALSE);
		 xwid->glyph.export_button = button;
	      }
	   }
	   xwid->glyph.options_button = xvf_get_xvobject(kformstructs[0],
			XVF_BUTTON_OBJ, TRUE);
	   xvw_insert_callback(guide->pane->back, XVW_SELECT_CALLBACK, FALSE,
			OptionsCallback, xwid);
	   kfree(kformstructs);
	}
	xwid->glyph.options_initialized = TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: UpdateNetworkType
|
|       Purpose: Looks at the glyph to figure out the network type.  This is
|		 done by examining the glyph's input & output data and control
|		 conections.
|
|         Input: glyph - the object in which to update it's network type
|    Written By: Mark Young
|          Date: Jan 25, 1995
|
------------------------------------------------------------*/
/* ARGSUSED */ 
static void UpdateNetworkType(
   xvobject glyph)
{
	int i, type, test, inputs, outputs;
	XvwGlyphWidget xwid = (XvwGlyphWidget) xvw_widget(glyph);


	inputs = (xwid->glyph.inum > 0);
	for (i = 0; inputs == FALSE && i < xwid->glyph.isize; i++)
        {
	   xvw_get_attribute(xwid->glyph.iports[i], XVW_PORT_SELECTED, &test);
	   if (test == TRUE) inputs = TRUE;
	}

	outputs = (xwid->glyph.onum > 0);
	for (i = 0; outputs == FALSE && i < xwid->glyph.osize; i++)
        {
	   xvw_get_attribute(xwid->glyph.oports[i], XVW_PORT_SELECTED, &test);
	   if (test == TRUE) outputs = TRUE;
	}

	if (inputs && outputs)
	   type = KNODE_TYPE_TRANSFER;
	else if (inputs)
	   type = KNODE_TYPE_SINK;
	else
	   type = KNODE_TYPE_SOURCE;

	xvw_set_attribute(glyph, XVW_NODE_TYPE, type);
}

/*-----------------------------------------------------------
|
|  Routine Name: OptionsCallback
|
|       Purpose: This routine will be called when the pane is put into
|		 edit mode.
|
|         Input: object      - which button being manipulated
|                client_data - the glyph
|                call_data   - not used
|    Written By: Mark Young
|          Date: Aug 28, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */ 
static void OptionsCallback(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	int      editing = *((int *) call_data);
	XvwGlyphWidget xwid = (XvwGlyphWidget) client_data;


	if (xwid->glyph.options_button == NULL)
	   return;

	if (editing)
	{
	   xvw_raise(xwid->glyph.options_button);
	   if (xwid->glyph.export_button != NULL)
              xvw_sensitive(xwid->glyph.export_button, TRUE);
	}
	else
	{
	   if (xwid->glyph.export_button != NULL)
	      xvw_sensitive(xwid->glyph.export_button, FALSE);
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: OptionsActions
|
|       Purpose: This routine will be called when the help quit is
|                activated.
|
|         Input: object      - which button being manipulated
|                client_data - the glyph
|                call_data   - not used
|    Written By: Mark Young
|          Date: Dec 1, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */ 
static void OptionsActions(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	xvobject glyph = (xvobject) client_data;
	XvwGlyphWidget xwid = (XvwGlyphWidget) xvw_widget(glyph);

	int      i;
	kguide   *guide;
	xvobject procedure;
	kobject  toolbox = NULL, cmobj = NULL;
	char     temp[KLENGTH], filename[KLENGTH], *name = NULL;


	xvw_get_attribute(object, XVW_LABEL, &name);
	if (kstrcmp(name, "Bugs/Todo") == 0)
	{
	   if (!xwid->glyph.help)
	   {
	      xwid->glyph.help = xvw_create_help(NULL, "Glyph Help Output");
	      (void) ksprintf(temp, "Output For '%s'", xwid->node.name);
	      xvw_set_attributes(xwid->glyph.help,
			XVW_HELP_TITLE, temp,
			XVW_PREFERRED_HEIGHT, 350,
			XVW_HELP_QUITLABEL,   "Close",
			XVW_HELP_DESTROY_ON_QUIT, FALSE,
			NULL);
	   }
	   kdirname(xwid->glyph.panefile, temp);
	   kdirname(temp, temp);
	   ksprintf(filename, "%s/info", temp);
	   xvw_set_attributes(xwid->glyph.help,
		XVW_HELP_MORE_FILES, TRUE,
		XVW_HELP_FILENAME, filename,
		NULL);
	}
	else if (kstrcmp(name, "Composer") == 0)
	{
	   ksprintf(temp,"composer -tb %s -oname %s", xwid->glyph.tbname,
		xwid->glyph.oname);
	   xvw_fork(temp, ComposerCallback, object);
	   xvw_sensitive(object, FALSE);
	}
	else if (kstrcmp(name, "Edit GUI") == 0)
	{
	   int editing = TRUE;

           if ((guide=kvf_search_sel_guide(xwid->glyph.form->subform)) == NULL)
              return;

	   xvw_get_attribute(guide->pane->back, XVW_EDIT_MODE_ON, &editing);
	   if (editing == FALSE)
	   {
	      /*
	       *  Turn edit mode on for the pane, and also make sure that
	       *  the options button rides on top of the editing window...
	       */
	      xvw_set_attribute(guide->pane->back, XVW_EDIT_MODE_ON, TRUE);
	   }
	   else
	   {
	      /*
	       *  Turn editing off...
	       */
	      xvw_set_attribute(guide->pane->back, XVW_EDIT_MODE_ON, FALSE);
	   }
	}
	else if (kstrcmp(name, "Save GUI") == 0)
	{
	   char   *prompt = "Filename: ", *answer = temp;

	   /* prompt for file from user... */
	   kstrcpy(answer, xwid->glyph.panefile);
	   if (xvu_query_wait("Please enter KHOROS filename", &prompt, "Ok",
			      &answer, 1, 35) && kstrlen(answer) > 0)
	   {
	      kvf_print_UIS_file(xwid->glyph.form, answer, TRUE, TRUE);
	   }
	}
	else if (kstrcmp(name, "Export to Workspace GUI") == 0)
	{
	   xvobject     *selections;
	   kform_struct *kformstruct;
	   int          num_sels = 0;

	   /*
	    *  Make sure the workspace GUI is not NULL....
	    */
	   if (xwid->glyph.wkspgui == NULL || xwid->glyph.form == NULL)
	      return;

           if ((guide=kvf_search_sel_guide(xwid->glyph.form->subform)) == NULL)
              return;
 
	   xvw_get_attributes(guide->pane->back,
		XVW_SELECTIONS, &selections,
		XVW_NUM_SELECTIONS, &num_sels,
		NULL);

	   procedure = NULL;
	   if (xvw_check_subclass(xwid->glyph.connection_parent,
			WorkspaceWidgetClass))
	   {
	      xvw_get_attribute(xwid->glyph.connection_parent,
			XVW_WORKSPACE_PROCEDURE_OBJECT, &procedure);
	   }

	   for (i = 0; i < num_sels; i++)
	   {
	      kformstruct = xvf_get_kformstruct(xwid->glyph.form,selections[i]);
	      if (kformstruct != NULL)
	      {
		 xvw_copy_selection(xwid->glyph.wkspgui, procedure, glyph,
				kformstruct, NULL);
	      }
	   }
	   xvw_set_attribute(guide->pane->back, XVW_SELECT_DELETE_ALL, TRUE);
	}
	else if (kstrcmp(name, "Info") == 0)
	{
	   int  otype;
	   char *author, *email, *category, *subcategory, *iconname, *type;


           if ((toolbox = kcms_open_toolbox(xwid->glyph.tbname)) == NULL ||
               (cmobj   = kcms_open_cmobj(toolbox, xwid->glyph.oname)) == NULL)
           {
              (void) kerror(XVLANG, "OptionsActions", "Cannot open %s '%s'.",
                        (!toolbox) ? "toolbox" : "object",
			(!toolbox) ? xwid->glyph.tbname :
				     xwid->glyph.oname);
	      if (cmobj)  kcms_close(cmobj);
	      if (toolbox) kcms_close(toolbox);
	      return;
	   }
	   author = email = category = subcategory = iconname = type = NULL;
	   kcms_get_attribute(cmobj, KCMS_CMOBJ_AUTHOR,       &author);
	   kcms_get_attribute(cmobj, KCMS_CMOBJ_AUTHOR_EMAIL, &email);
	   kcms_get_attribute(cmobj, KCMS_CMOBJ_CATEGORY,     &category);
	   kcms_get_attribute(cmobj, KCMS_CMOBJ_SUBCATEGORY,  &subcategory);
	   kcms_get_attribute(cmobj, KCMS_CMOBJ_ICON_NAME,    &iconname);
	   kcms_get_attribute(cmobj, KCMS_CMOBJ_PROGTYPE,     &otype);
	   if      (otype == KCMS_XVROUTINE) type = "xvroutine";
	   else if (otype == KCMS_KROUTINE)  type = "kroutine";
	   else if (otype == KCMS_PANE)      type = "pane";
	   else if (otype == KCMS_SCRIPT)    type = "pane";
	   else type = "(unknown)";

	   if (!author) author = "(unknown)";
	   if (!email)  email  = "(unknown)";
	   if (!iconname)    iconname     = "(unknown)";
	   if (!category)    category     = "(unknown)";
	   if (!subcategory) subcategory  = "(unknown)";

	   kinfo(KFORCE,"\
\tObject Attributes\n\
Toolbox      : %s\n\
Object       : %s\n\
Object Type  : %s\n\n\
Category     : %s\n\
Subcategory  : %s\n\
Icon Name    : %s\n\n\
Author Name  : %s\n\
Author Email : %s", xwid->glyph.tbname, xwid->glyph.oname, type, category,
		    subcategory, iconname, author, email);
	   kcms_close(cmobj); kcms_close(toolbox);
	}
	else if (kstrcmp(name, "Attributes") == 0)
	{
	   char   *prompts[3], *answers[3];

           if ((toolbox = kcms_open_toolbox(xwid->glyph.tbname)) == NULL ||
               (cmobj   = kcms_open_cmobj(toolbox, xwid->glyph.oname)) == NULL)
           {
              (void) kerror(XVLANG, "OptionsActions", "Cannot open %s '%s'.",
                        (!toolbox) ? "toolbox" : "object",
			(!toolbox) ? xwid->glyph.tbname :
				     xwid->glyph.oname);
	      if (cmobj)  kcms_close(cmobj);
	      if (toolbox) kcms_close(toolbox);
	      return;
	   }
	   prompts[0] = "Category:   ";
	   prompts[1] = "SubCategory:";
	   prompts[2] = "Icon Name:  ";
	   kcms_get_attribute(cmobj, KCMS_CMOBJ_CATEGORY,     &answers[0]);
	   kcms_get_attribute(cmobj, KCMS_CMOBJ_SUBCATEGORY,  &answers[1]);
	   kcms_get_attribute(cmobj, KCMS_CMOBJ_ICON_NAME,    &answers[2]);

	   if (!answers[0]) answers[0] = "(unknown)";
	   if (!answers[1]) answers[1] = "(unknown)";
	   if (!answers[2]) answers[2] = "(unknown)";
	   if (xvu_query_wait("Object attributes", prompts, "Ok", answers,
			      3, 50))
	   {
	      kcms_set_attribute(cmobj, KCMS_CMOBJ_CATEGORY,     answers[0]);
	      kcms_set_attribute(cmobj, KCMS_CMOBJ_SUBCATEGORY,  answers[1]);
	      kcms_set_attribute(cmobj, KCMS_CMOBJ_ICON_NAME,    answers[2]);
	   }
	   kcms_close(cmobj); kcms_close(toolbox);
	}
	else if (kstrcmp(name, "Destroy") == 0)
	{
	   if (kprompt(KSTANDARD, "Destroy", "Cancel", 1, "Destroy the \
glyph '%s' from the current workspace?", xwid->node.name) == TRUE)
	   {
	      xvw_set_attribute(glyph, XVW_NODE_EXECUTE, FALSE);
	      xvw_destroy(glyph);
	   }
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: InputFilenameCallback - input filename change callback
|
|       Purpose: This routine is used to detect when a change to a glyph's
|		 input filename has ocurred. 
|		 
|         Input: glyph - glyph to be used to change to be sensitive
|        Output:
|	Returns:
|    Written By: Mark Young
|          Date: Nov 26, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */ 
static void InputFilenameCallback(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
        xvobject glyph = (xvobject) client_data;
	char     *filename   = *((char **) call_data); 

	xvw_set_attribute(glyph, XVW_NODE_MODIFIED, TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: InputDavCallback - dav change on our input detected
|
|       Purpose: This routine is used to detect when a change to a glyph's
|		 input data has ocurred. 
|		 
|
|         Input: glyph - glyph to be used to change to be sensitive
|        Output:
|	Returns:
|    Written By: Mark Young
|          Date: Nov 26, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */ 
static void InputDavCallback(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
        xvobject glyph = (xvobject) client_data;
	int      dav   = *((int *) call_data); 

	xvw_set_attribute(glyph, XVW_NODE_MODIFIED, TRUE);
	RequestDispatchNode();
}

/*-----------------------------------------------------------
|
|  Routine Name: OutputDavCallback - dav change on our output detected
|
|       Purpose: This routine is used to detect when a change to a glyph's
|		 input data has ocurred. 
|		 
|
|         Input: glyph - glyph to be used to change to be sensitive
|        Output:
|	Returns:
|    Written By: Mark Young
|          Date: Nov 26, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */ 
static void OutputDavCallback(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
        xvobject glyph = (xvobject) client_data;
	int      dav   = *((int *) call_data); 

}

/*-----------------------------------------------------------
|
|  Routine Name: SelectedCallback - selected change on our input/output
|				    detected
|
|       Purpose: This routine is used to detect when a change to a glyph's
|		 input/output port selection has ocurred.  The specific
|		 change reported is whether the port is selected or not.
|
|         Input: object - the port that changed
|		 glyph  - glyph that the port belongs to
|        Output:
|	Returns:
|    Written By: Mark Young
|          Date: Nov 26, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */ 
static void SelectedCallback(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
        xvobject glyph = (xvobject) client_data;

	UpdateNetworkType(glyph);
}

/*-----------------------------------------------------------
|
|  Routine Name: GlyphRunCallback - the run button on the glyph executed
|
|       Purpose: This performs a run of the glyph via the run button
|		 on the panefile.
|
|         Input: client_data - the glyph widget
|
|        Output:
|	Returns:
|    Written By: Mark Young
|          Date: Jul 23, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */ 
void GlyphRunCallback(
   kaddr    client_data)
{
	xvobject glyph = (xvobject) client_data;

	int i, selected;
	XvwGlyphWidget xwid = (XvwGlyphWidget) xvw_widget(glyph);


	for (i = 0; i < xwid->glyph.isize; i++)
	{
	   xvw_get_attribute(xwid->glyph.iports[i], XVW_PORT_SELECTED,
			&selected);
	   if (selected == TRUE)
	      xvw_set_attribute(xwid->glyph.iports[i], XVW_PORT_DAV, TRUE);
	}
	xvw_set_attribute(glyph, XVW_NODE_MODIFIED, TRUE);
	xvw_set_attribute(glyph, XVW_NODE_EXECUTE, TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: GlyphUpdateForm
|
|       Purpose: Updates the I/O ports & label of a glyph depending 
|                on the currently selected pane of the form associated
|                with a glyph.  This routine is called directly from 
|                the glyph widget's SetValues routine to initially create 
|                I/O ports and label when the form is first associated 
|                with the glyph using the XVW_GLYPH_FORM attribute; it
|                is also used as a callback installed with xvf_add_extra_call()
|                for when the pane on a form associated with a glyph is
|                changed.
|
|         Input: xwid  - the glyph widget
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro and Mark Young
|          Date: Oct 13, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
void GlyphUpdateForm(
    Widget widget)
{
	XvwGlyphWidget xwid = (XvwGlyphWidget) widget;

	kguide   *guide;
	kform_struct **sel;
        int      i, indx, num;
        xvobject offset, *port;
	xvobject glyph  = xvw_object((Widget) xwid);
	xvobject parent = xvw_parent(glyph);


	/* find the selected pane */
        guide = kvf_search_sel_guide(xwid->glyph.form->subform);
	xvf_clear_modified(guide->pane->sel_list);

        /*
	 *  get the array of backplanes for all output selections on new
	 *  pane and allocate new array of output ports
	 */
        sel = kvf_get_selection_kformstructs(guide->pane, KUIS_INPUTFILE, &num);
	port = (xvobject *) kmalloc(num * sizeof(xvobject));

	/* destroy all the old input ports */
        for (i = 0; i < xwid->glyph.isize; i++)
	{
	    if ((indx = karray_locate((char **) sel,
			(kaddr) xwid->glyph.isels[i], num)) == -1)
	    {
	       xvw_destroy(xwid->glyph.iports[i]);
	    }
	}

        offset = xwid->glyph.destroy;
        for (i = 0; i < num; offset = port[i++])
        {
	    if ((indx = karray_locate((char **) xwid->glyph.isels,
			(kaddr) sel[i], xwid->glyph.isize)) == -1)
	    {
               port[i] = xvw_create_port(glyph, "InputPort");
	       xvw_set_attributes(port[i],
		   XVW_PORT_SELECTION, sel[i],
		   XVW_PORT_TYPE,      KPORT_TYPE_INPUT,
		   XVW_PORT_MODIFIED,  TRUE,
		   XVW_BELOW,	       offset,
		   XVW_RIGHT_OF,       NULL,
		   XVW_PORT_CONNECTION_PARENT, xwid->glyph.connection_parent,
		   XVW_PORT_DAV_PIXMAP, xwid->glyph.dav_pixmap,
		   NULL);
	       xvw_add_event(port[i], EnterWindowMask | LeaveWindowMask,
			GlyphHandler, (kaddr) glyph);
	       xvw_insert_callback(port[i], XVW_PORT_DAV_CALLBACK,
			FALSE, InputDavCallback, glyph);
	       xvw_insert_callback(port[i], XVW_PORT_FILENAME_CALLBACK,
			FALSE, InputFilenameCallback, glyph);
	       xvw_insert_callback(port[i], XVW_PORT_SELECTED_CALLBACK,
			FALSE, SelectedCallback, glyph);
#if 0
/*
 *  Start of creating iconic glyphs...
 */
xvw_unmanage(port[i]);
#endif
	    }
	    else
	    {
	       port[i] = xwid->glyph.iports[indx];
	       xvw_set_attribute(port[i], XVW_BELOW, offset);
	    }
	}
	xwid->glyph.isize = num;
	kfree(xwid->glyph.isels);  xwid->glyph.isels  = sel;
	kfree(xwid->glyph.iports); xwid->glyph.iports = port;


        /*
	 *  get the array of backplanes for all output selections on new
	 *  pane and allocate new array of output ports
	 */
        sel = kvf_get_selection_kformstructs(guide->pane,KUIS_OUTPUTFILE,&num);
	port = (xvobject *) kmalloc(num * sizeof(xvobject));

	/* destroy all the old input ports */
        for (i = 0; i < xwid->glyph.osize; i++)
	{
	    if ((indx = karray_locate((char **) sel,
			(kaddr) xwid->glyph.osels[i], num)) == -1)
	    {
	       xvw_destroy(xwid->glyph.oports[i]);
	    }
	}

        /* create all the new output ports according to output selections */
        offset = xwid->glyph.destroy;
        for (i = 0; i < num; offset = port[i++])
        {
	    if ((indx = karray_locate((char **) xwid->glyph.osels,
			(kaddr) sel[i], xwid->glyph.osize)) == -1)
	    {
               port[i] = xvw_create_port(glyph, "OutputPort");
	       xvw_set_attributes(port[i],
		   XVW_PORT_SELECTION, sel[i],
		   XVW_PORT_TYPE,      KPORT_TYPE_OUTPUT,
		   XVW_BELOW,	       offset,
		   XVW_LEFT_OF,	       NULL,
		   XVW_PORT_CONNECTION_PARENT, xwid->glyph.connection_parent,
		   XVW_PORT_DAV_PIXMAP, xwid->glyph.dav_pixmap,
		   NULL);
	       xvw_add_event(port[i], EnterWindowMask | LeaveWindowMask,
			GlyphHandler, (kaddr) glyph);
	       xvw_insert_callback(port[i], XVW_PORT_DAV_CALLBACK,
			FALSE, OutputDavCallback, glyph);
	       xvw_insert_callback(port[i], XVW_PORT_SELECTED_CALLBACK,
			FALSE, SelectedCallback, glyph);
#if 0
/*
 *  Start of creating iconic glyphs...
 */
xvw_unmanage(port[i]);
#endif
	    }
	    else
	    {
	       port[i] = xwid->glyph.oports[indx];
	       xvw_set_attribute(port[i], XVW_BELOW, offset);
	    }
        }
	xwid->glyph.osize = num;
	kfree(xwid->glyph.osels);  xwid->glyph.osels  = sel;
	kfree(xwid->glyph.oports); xwid->glyph.oports = port;
	UpdateNetworkType(glyph);
}

/*-----------------------------------------------------------
|
|  Routine Name: GlyphInitForm
|
|       Purpose: This routine is called directly from the glyph widget's
|		 SetValues routine to initially create I/O ports and label
|		 when the form is first associated with the glyph using the
|		 XVW_GLYPH_FORM attribute.
|
|         Input: xwid  - the glyph widget
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro and Mark Young
|          Date: Oct 13, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
void GlyphInitForm(
    kaddr client_data)
{
	XvwGlyphWidget xwid = (XvwGlyphWidget) client_data;

        int      i;
	kguide   *guide;
	xvobject glyph  = xvw_object((Widget) xwid);
	xvobject parent = xvw_parent(glyph);


	/* find the selected pane */
        guide = kvf_search_sel_guide(xwid->glyph.form->subform);
	GlyphUpdateForm((kaddr) xwid);

        /* get the array of backplanes for all routine selections on new pane */
	kfree(xwid->glyph.rsels);
        if ((xwid->glyph.rsels = kvf_get_selection_kformstructs(guide->pane,
			KUIS_ROUTINE, &xwid->glyph.rsize)) != NULL)
	{
	   xvf_get_attribute(xwid->glyph.rsels[0], XVF_EXECTYPE,
		&xwid->glyph.exectype);
	}
	else
	{
	    xwid->glyph.exectype = 0;
	    xwid->node.runnable = FALSE;
	    xvw_set_attribute(xwid->glyph.cflow_in,  XVW_MAPPED, FALSE);
	    xvw_set_attribute(xwid->glyph.cflow_out, XVW_MAPPED, FALSE);
	}

	if (!xwid->glyph.exectype)
	{
	   xvw_unmap(xwid->glyph.execute);
	   xvw_set_attribute(glyph, XVW_NODE_SHOW_MODIFIED, FALSE);

	   for (i = 0; i < xwid->glyph.osize; i++)
	      xvw_set_attribute(xwid->glyph.oports[i], XVW_PORT_DAV, TRUE);
	}
	else
	{
	   xvw_map(xwid->glyph.execute);
	   xvw_set_attribute(glyph, XVW_NODE_MODIFIED, TRUE);
	}

#if 0
/*
 *  Start of Glyph Icon Support
 */
	if (kdirname(xwid->glyph.panefile, temp) != NULL)
	{
	   ksprintf(path, "%s/ObjectIcon", temp);
	   if (kaccess(path, R_OK) == 0)
	      xvw_set_attribute(glyph, XVW_NODE_PIXMAPFILE, path);
	}
#endif
}

/*-----------------------------------------------------------
|
|  Routine Name: GetFormfile
|
|       Purpose: GetFormfile is used to get the filename associated
|		 with the current form.
|
|         Input: widget    - the widget in which we will be getting the
|			     filename
|		 attribute - the old object to be closed
|
|        Output: calldata  - pointer to retrieve the object name
|       Returns: TRUE (1) if redisplay is required, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Nov 21, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */ 
int GetFormfile(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	XvwGlyphWidget xwid = (XvwGlyphWidget) xvw_widget(object);
	String *filename = (String *) calldata;


	if (kstrcmp(attribute, XVW_GLYPH_FORMFILE) != 0)
	{
	   kerror(XVLANG, "GetFormfile", "invalid attribute %s",
			attribute);
	   return(FALSE);
	}
	*filename = xwid->glyph.panefile;
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: SetFormfile
|
|       Purpose: SetFormfile is used to set the filename associated
|		 with the attribute.
|
|         Input: object    - the object in which we will be getting the
|			     filename
|		 attribute - the old object to be closed
|		 calldata  - the filename to be set
|
|        Output:
|       Returns: TRUE (1) if redisplay is required, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Nov 21, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */ 
int SetFormfile(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	XvwGlyphWidget xwid = (XvwGlyphWidget) xvw_widget(object);
	String *filename = (String *) calldata;

	kform  *form;
	kselection *sel_list, *selection;
	Widget parent = XtParent((Widget) xwid);


	if (kstrcmp(attribute, XVW_GLYPH_FORMFILE) != 0)
	{
	   kerror(XVLANG, "SetFormfile", "invalid attribute %s", attribute);
	   return(FALSE);
	}

	/*
	 *  So create the form...
	 */
	if ((form = kvf_create_form(*filename, NONE, NULL, NULL)) == NULL)
	   return(FALSE);

	/*
	 *  Go ahead and override the run button on the pane file (if one
	 *  exists).  We do this so that the run button will use our scheduler.
	 */
	kvf_change_entry(form, FALSE, GlyphFormCallback, object);
	if (kvf_check_for_single_pane(form, "SetFormfile"))
	{
	   sel_list = form->subform->guide->pane->sel_list;
	   selection = kvf_gui_named_item_exists(sel_list, KUIS_ROUTINE, -1);
	   if (selection != NULL)
	   {
	      xvf_add_extra_call(selection->back_kformstruct,
		    GlyphRunCallback, (kaddr) object, XVF_CALL_SUBSTITUTE);
	   }
	}
	kfree(xwid->glyph.panefile);
	xwid->glyph.panefile = kstrdup(*filename);
	xvw_set_attribute(object, XVW_GLYPH_FORM, form);
	xwid->glyph.created_form = TRUE;
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: GlyphHandler
|
|       Purpose: This routine will perform a variety of glyph callbacks,
|		 such as destroy, menu, execute, remote, etc.
|
|         Input: object      - the button being activated
|                client_data - the glyph widget
|                event       - event the activated us
|		 dispatch    - whether to continue dispatch the event
|
|    Written By: Mark Young
|          Date: Nov 23, 1993
|
------------------------------------------------------------*/
/* ARGSUSED */ 
void GlyphHandler(
   xvobject object,
   kaddr    client_data,
   XEvent   *event,
   int      *dispatch)
{
	xvobject glyph = (xvobject) client_data;
	XvwGlyphWidget xwid = (XvwGlyphWidget) xvw_widget(glyph);

	Widget widget;
	int    execute;
	static xvobject selected_object = NULL;


	*dispatch = FALSE;
	widget = xvw_widget(object);
	object = xvw_object(widget);
	if (event->type == LeaveNotify)
	{
	   if (xwid->glyph.show_status == TRUE)
	      xvw_set_attribute(xwid->glyph.status, XVW_LABEL, NULL);

	   kannounce(XVLANG, "GlyphHandler", " ");
	   return;
	}
	else if (event->type == EnterNotify)
	{
	   char     *tmp = NULL;
	   kform_struct *kformstruct = NULL;

	   if (object == xwid->glyph.menuform)
	      tmp = "Pane Access";
	   else if (object == xwid->glyph.destroy)
	      tmp = "Destroy";
	   else if (object == xwid->glyph.error)
	      tmp = "Display Error Message";
	   else if (object == xwid->glyph.execute)
	      tmp = "Run";
	   else if (object == xwid->glyph.cflow_in)
	      tmp = "Input Control Connection Node";
	   else if (object == xwid->glyph.cflow_out)
	      tmp = "Output Control Connection Node";
	   else if (object == xwid->glyph.remote)
	      tmp = "Host To Execute On";
	   else if (object == xwid->node.label)
	      tmp = "Operator's Icon Name";
	   else
	   {
	      if (!xvw_get_attribute(object, XVW_PORT_SELECTION,&kformstruct) ||
		  !kformstruct)
	      {
	         return;
	      }
	      xvf_get_attribute(kformstruct, XVF_TITLE, &tmp);
	   }
	   if (xwid->glyph.show_status == TRUE)
	      xvw_set_attribute(xwid->glyph.status, XVW_LABEL, tmp);

	   kannounce(XVLANG, "GlyphHandler", "%s:  %s", xwid->node.name,tmp);
	   return;
	}
	else if (event->type == ButtonPress)
	{
	   selected_object = object;
	   return;
	}
	else if (event->type != ButtonRelease || object != selected_object)
	{
	   return;
	}

	if (object == xwid->glyph.destroy)
	{
	   xvw_set_attribute(glyph, XVW_NODE_EXECUTE, FALSE);
	   xvw_destroy(xvw_object((Widget) xwid));
	}
	else if (object == xwid->glyph.error)
	{
	   xvw_unmap(xwid->glyph.error);
	   xvw_map(xvw_toplevel(xwid->glyph.help));
	}
	else if (object == xwid->glyph.menuform)
	{
	   if (xwid->glyph.form != NULL)
	   {
	      if (!xvw_check_mapped(xwid->glyph.form->subform->toplevel))
	      {
		 xvw_set_attribute(object, XVW_PIXMAP,
			xwid->glyph.closepane_pixmap);
	         xvf_map_subform(xwid->glyph.form->subform);
	      }
	      else
	      {
		 xvw_set_attribute(object, XVW_PIXMAP,
			xwid->glyph.openpane_pixmap);
	         xvf_unmap_subform(xwid->glyph.form->subform);
	      }
	      OptionsInitialize(glyph);
	   }
	   else
	      kerror(XVLANG, "GlyphHandler",
		     "No form associated with glyph");
	}
	else if (object == xwid->glyph.execute)
	{
	   xvw_get_attribute(glyph, XVW_NODE_EXECUTE, &execute);
	   if (execute)
	      xvw_set_attribute(glyph, XVW_NODE_EXECUTE, FALSE);
	   else
	   {
	      /*
	       *  Force the glyph to execute.  We do this by clearing it's
	       *  data availables and then running the glyph.
	       */
	      xvw_set_attribute(glyph, XVW_NODE_MODIFIED, TRUE);
	      xvw_set_attribute(glyph, XVW_NODE_EXECUTE, TRUE);
	   }
	}
#if 0
	else if (object == xwid->glyph.remote)
	{
	}
#endif
	else if (object == xwid->glyph.cflow_in ||
		 object == xwid->glyph.cflow_out)
	{
	   static xvobject cflow_in  = NULL;
	   static xvobject cflow_out = NULL;


	   /*
	    *  Check to see if the object is an input or output
	    */
	   if (object == xwid->glyph.cflow_in)
	      cflow_in = glyph;
	   else if (object == xwid->glyph.cflow_out)
	      cflow_out = glyph;

	   /*
	    *  If either both input or output cflow are set then create the
	    *  connection, with the exception that we can't make a control
	    *  dependence on the same glyph.
	    */
	   if (cflow_in && cflow_out && cflow_in != cflow_out)
	   {
	      GlyphCreateConnection(cflow_in, cflow_out);
	      cflow_in = cflow_out = NULL;
	   }
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: GlyphFormExpression
|
|       Purpose: This routine will evaluate an expression associated with
|		 a form.
|         Input: form - the form that the expression should be evaluated for
|                client-data - the client data associated with the form
|                string - the expression to be evaluated
|                type   - the type of expression to be evaluated
|	 Output: return_value - the return value to be evaluated
|		 error        - the error string to be returned
|    Written By: Mark Young
|          Date: Aug 14, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */ 
int GlyphFormExpression(
   kform *form,
   kaddr client_data,
   char  *string,
   int   type,
   kaddr return_value,
   char  *error)
{
	xvobject glyph = (xvobject) client_data;
        XvwGlyphWidget xwid = (XvwGlyphWidget) xvw_widget(glyph);

	long id = (long) xwid->glyph.expression_id;

	return(kexpr_evaluate_generic(id, string, type, return_value, error));
}

/*-----------------------------------------------------------
|
|  Routine Name: ComposerCallback - composer callback routine
|
|
|       Purpose: This routine is used to change the fact that
|		 a running composer is terminated.
|
|         Input: object - object to be used to change to be sensitive
|        Output:
|	Returns:
|    Written By: Mark Young
|          Date: Nov 26, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */ 
static void ComposerCallback(
   int     pid,
   kstatus status,
   kaddr   client_data,
   char    *errorfile)
{
        xvobject object = (xvobject) client_data;

	xvw_sensitive(object, TRUE);
	if (errorfile) kunlink(errorfile);
}

/*-----------------------------------------------------------
|
|  Routine Name: GlyphFormCallback - form callback routine
|
|       Purpose: This routine is used to process the user's form
|                changes.   Any changes in the form are noted
|                and the object/gadget/KHOROS-item is updated.  Update is
|                accomplished by using the xvw_set_attributes() to modify
|                the attributes of the object/gadget/KHOROS-item.
|
|         Input: object - object to be used to build the form
|    Written By: Mark Young
|          Date: Nov 26, 1993
|
------------------------------------------------------------*/
/* ARGSUSED */ 
void GlyphFormCallback(
   kform    *form,
   ksubform *subform,
   kaddr    client_data)
{
        xvobject glyph = (xvobject) client_data;
	XvwGlyphWidget xwid = (XvwGlyphWidget) xvw_widget(glyph);

	kguide *guide;
	int    i, modified;
	kselection *selection;


	if (form->quit)
	{
	   form->quit = subform->quit = FALSE;
	   xvw_set_attribute(xwid->glyph.menuform, XVW_PIXMAP,
		xwid->glyph.openpane_pixmap);
	}

	for (i = 0, modified = FALSE; i < xwid->glyph.isize; i++)
	{
	   xvf_get_attribute(xwid->glyph.isels[i], XVF_MODIFIED, &modified);
	   if (modified)
	   {
	      xvf_set_attribute(xwid->glyph.isels[i], XVF_MODIFIED, FALSE);
	      xvw_set_attribute(xwid->glyph.iports[i], XVW_PORT_MODIFIED,
				TRUE);
	      modified = FALSE;
	   }
	}

	for (i = 0, modified = FALSE; i < xwid->glyph.osize; i++)
	{
	   xvf_get_attribute(xwid->glyph.osels[i], XVF_MODIFIED, &modified);
	   if (modified)
	   {
	      xvf_set_attribute(xwid->glyph.osels[i], XVF_MODIFIED, FALSE);
	      xvw_set_attribute(xwid->glyph.oports[i], XVW_PORT_MODIFIED,
				TRUE);
	      modified = FALSE;
	   }
	}

	if ((guide = kvf_search_sel_guide(subform)) != NULL)
	{
	   selection = kvf_gui_named_item_exists(guide->pane->sel_list,
				KUIS_QUIT, -1);
	   if (selection != NULL)
	      selection->modified = FALSE;

	   if (xwid->glyph.exectype != 0 &&
	       xvf_check_modified(guide->pane->sel_list))
	   {
	      xvf_clear_modified(guide->pane->sel_list);
	      xvw_set_attribute(glyph, XVW_NODE_MODIFIED, TRUE);
	   }
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: GlyphFindNode - routine for finding the connection according
|			         to it's variable name
|
|       Purpose: This routine is called when we want to locate a port given
|		 it's variable name.
|
|         Input: glyph - the glyph we are searching
|		 ivar  - the input variable name
|		 ovar  - the output variable name
|        Output: selection - return the selection if not NULL
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: Dec 12, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */ 
xvobject GlyphFindNode(
   xvobject glyph,
   char     *ivar,
   char	    *ovar,
   kform_struct **selection)
{
	int i, size = 0;
	char *var = NULL, *varname = NULL;

	kform_struct   **sels = NULL;
	xvobject       *ports = NULL;
	XvwGlyphWidget xwid = (XvwGlyphWidget) xvw_widget(glyph);


	if (!glyph || (!ivar && !ovar))
	   return(NULL);

	if (ivar)
	{
	   var  = ivar;
	   size = xwid->glyph.isize;
	   sels = xwid->glyph.isels;
	   ports = xwid->glyph.iports;
	}
	else if (ovar)
	{
	   var  = ovar;
	   size = xwid->glyph.osize;
	   sels = xwid->glyph.osels;
	   ports = xwid->glyph.oports;
	}

	/*
	 *  Scan for the desired port...
	 */
	for (i = 0; i < size; i++)
	{
	    xvf_get_attribute(sels[i], XVF_VARIABLE, &varname);
	    if (kstrcmp(var, varname) == 0)
	    {
	       if (selection)
		  *selection = sels[i];
	       return(ports[i]);
	    }
	}

	if (selection)
	   *selection = NULL;
	return(NULL);
}

/*-----------------------------------------------------------
|
|  Routine Name: GlyphDestroyCallback - destroy routine for stopping a glyph
|				        and deleting it's expressions
|
|       Purpose: This routine is called when we are about to destroy the
|		 glyph.
|
|         Input: glyph - the glyph we are about destroy
|		 client_data - the client data (not used)
|		 call_data   - the call data (not used)
|    Written By: Mark Young
|          Date: Aug 24, 1994
| Modifications:
|
------------------------------------------------------------*/

extern klist *frontier;

/* ARGSUSED */ 
void GlyphDestroyCallback(
   xvobject glyph,
   kaddr    client_data,
   kaddr    call_data)
{
	XvwGlyphWidget xwid = (XvwGlyphWidget) xvw_widget(glyph);

	Widget parent;
	int    execute = FALSE;

	/*
	 *  Stop the sucker...
	 */
	frontier = klist_delete(frontier, (kaddr) glyph);
	if (xwid->glyph.pid != 0)
        {
           kill(xwid->glyph.pid, SIGHUP);
           xwid->glyph.pid = 0;
        }

	parent = XtParent((Widget) xwid);
	if (xwid->node.running != KNODE_STOPPED &&
	    parent->core.being_destroyed == FALSE)
	{
	   XtCallCallbacks((Widget) xwid, XVW_NODE_RUN_CALLBACK, &execute);
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: ConnectionDestroyed
|
|       Purpose: This routine is called when the connection is destroyed
|         Input: connection - the connection to be destroyed.
|
|    Written By: Mark Young
|          Date: Oct 23, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ConnectionDestroyed(
   xvobject connection,
   kaddr    client_data,
   kaddr    call_data)
{
	XvwGlyphWidget xwid = (XvwGlyphWidget) client_data;


	if (karray_locate((char **) xwid->glyph.iconnections,
		(kaddr) connection, xwid->glyph.inum) != -1)
	{
	   xwid->glyph.iconnections = (xvobject *) karray_delete((char **)
              xwid->glyph.iconnections, (kaddr) connection, xwid->glyph.inum--);
	}
	else if (karray_locate((char **) xwid->glyph.oconnections,
		(kaddr) connection, xwid->glyph.onum) != -1)
	{
	   xwid->glyph.oconnections = (xvobject *) karray_delete((char **)
              xwid->glyph.oconnections, (kaddr) connection, xwid->glyph.onum--);
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: GlyphCreateConnection - routine for creating connections
|					between two glyphs
|
|       Purpose: This routine is called when we want to create a connection
|         Input: iglyph - the input glyph
|                oglyph - the output glyph
|        Output: 
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: Oct 23, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int GlyphCreateConnection(
   xvobject iglyph,
   xvobject oglyph)
{
	XvwGlyphWidget ixwid = (XvwGlyphWidget) xvw_widget(iglyph);
	XvwGlyphWidget oxwid = (XvwGlyphWidget) xvw_widget(oglyph);

	xvobject connection, parent;


        if ((parent = oxwid->glyph.connection_parent) == NULL)
           parent = xvw_parent(oglyph);

	connection = xvw_create_connection(parent, "ControlConnection");
	xvw_set_attributes(connection,
		XVW_CONNECTION_END,   ixwid->glyph.cflow_in,
		XVW_CONNECTION_BEGIN, oxwid->glyph.cflow_out,
		NULL);

        xvw_insert_callback(connection, XVW_DESTROY, FALSE,
			ConnectionDestroyed, ixwid);
        xvw_insert_callback(connection, XVW_DESTROY, FALSE,
			ConnectionDestroyed, oxwid);
	xvw_add_callback(connection, XVW_CONNECTION_CALLBACK,
			ConnectionCallback, oxwid);

	ixwid->glyph.iconnections = (xvobject *) karray_add((char **)
	    ixwid->glyph.iconnections, (kaddr) connection, ixwid->glyph.inum++);
	oxwid->glyph.oconnections = (xvobject *) karray_add((char **)
	    oxwid->glyph.oconnections, (kaddr) connection, oxwid->glyph.onum++);
	UpdateNetworkType(iglyph); UpdateNetworkType(oglyph);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: ConnectionCallback
|
|       Purpose: This routine is called when the connection is activated
|         Input: object      - connection being destroyed
|                client_data - the glyph widget
|                call_data   - not used
|        Output:
|       Returns:
|
|    Written By: Mark Young
|          Date: Nov 05, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ConnectionCallback(
   xvobject connection,
   kaddr    client_data,
   kaddr    call_data)
{
	XvwGlyphWidget xwid = (XvwGlyphWidget) client_data;

	int    i;
	char *choice[] =
	{
		"Delete Connection",
		"Connection Options",
		"",
	};

	if (!kchoose(KFORCE, choice, knumber(choice), 0, NULL, &i,
			xwid->node.name))
	{
	   return;
	}
	else i--;

	kinfo(KDEBUG, "entry '%s' selected for filename '%s'\n", choice[i],
			xwid->node.name);
	if (i == 0)
	   xvw_destroy(connection);
	else if (i == 1)
	   xvw_activate_menu(connection);
}
