 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */



/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>                2D Plot Gadget Routines
   >>>>
   >>>>	 Private:
   >>>>	  Static:
   >>>>			ClassInitialize()
   >>>>			Initialize()
   >>>>			Destroy()
   >>>>			Redisplay()
   >>>>			SetValues()
   >>>>			ReloadCallback()
   >>>>			ColorCallback()
   >>>>
   >>>>			FileDetect()
   >>>>			GetObjectName()
   >>>>			SetObjectName()
   >>>>			FindMinMax()
   >>>>			ReloadPlot()
   >>>>			PickGadget()
   >>>>   Public:
   >>>>			xvw_create_plot2d()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <xvisual/Plot2DP.h>


static void ClassInitialize PROTO((void));
static void Initialize	    PROTO((Widget, Widget, ArgList, Cardinal *));
static void Redisplay	    PROTO((Widget, XEvent *, Region));
static int  FileDetect	    PROTO((xvobject, char *, kaddr));
static int  GetObjectName   PROTO((xvobject, char *, kaddr));
static int  SetObjectName   PROTO((xvobject, char *, kaddr));
static Boolean SetValues    PROTO((Widget, Widget, Widget, ArgList,
				   Cardinal *));
static void ReloadCallback  PROTO((kobject, char *, Widget, kdms_callback *));
static void ColorCallback   PROTO((xvobject, kaddr));

static void Destroy	    PROTO((Widget));
static void FindMinMax	    PROTO((XvwPlot2DGadget));
static void ReloadPlot	    PROTO((XvwPlot2DGadget));
static int PickGadget	    PROTO((Widget, int, int));


/*--------------------------------------------------------
|
|   Full class attributes
|
--------------------------------------------------------*/

static xvattribute attributes[] = {
{XVW_PLOT2D_PLOTOBJ,       NULL,    XtRInt,        NULL}, 
{XVW_PLOT2D_DATA_MAX_X,    NULL,    XtRDouble,     NULL}, 
{XVW_PLOT2D_DATA_MAX_Y,    NULL,    XtRDouble,     NULL}, 
{XVW_PLOT2D_DATA_MIN_X,    NULL,    XtRDouble,     NULL}, 
{XVW_PLOT2D_DATA_MIN_Y,    NULL,    XtRDouble,     NULL}, 
{XVW_PLOT2D_PLOTTYPE,      NULL,    XtRInt,        NULL}, 
{XVW_PLOT2D_PLOTSIZE,      NULL,    XtRInt,        NULL}, 
{XVW_PLOT2D_POINTS,        NULL,    XtRPointer,    NULL}, 
};

/*--------------------------------------------------------
|
|   Full class record constant
|
--------------------------------------------------------*/

#define offset(field) XtOffsetOf(XvwPlot2DGadgetRec, plot2d.field)

static double data_min     = 0.0,
              data_max     = 1.0;

static XtResource resources[] = {
{XVW_PLOT2D_DATA_MAX_X,  NULL, XtRDouble, sizeof(double),
      offset(data_max_x), XtRDouble, (XtPointer) &data_max},
{XVW_PLOT2D_DATA_MAX_Y,  NULL, XtRDouble, sizeof(double),
      offset(data_max_y), XtRDouble, (XtPointer) &data_max},
{XVW_PLOT2D_DATA_MIN_X,  NULL, XtRDouble, sizeof(double),
      offset(data_min_x), XtRDouble, (XtPointer) &data_min},
{XVW_PLOT2D_DATA_MIN_Y,  NULL, XtRDouble, sizeof(double),
      offset(data_min_y), XtRDouble, (XtPointer) &data_min},
{XVW_PLOT2D_PLOTTYPE,  NULL, XtRInt, sizeof(int),
      offset(plot_type), XtRImmediate, (XtPointer) KPLOT2D_LINEPLOT},
{XVW_PLOT2D_PLOTSIZE, NULL, XtRInt, sizeof(int),
      offset(plot_size), XtRImmediate, (XtPointer) 0},
{XVW_PLOT2D_POINTS, NULL, XtRPointer, sizeof(Coord *),
      offset(points), XtRPointer, (XtPointer) NULL},
{XVW_PLOT2D_PLOTOBJ, NULL, XtRPointer, sizeof(kobject),
      offset(pobject), XtRPointer, (XtPointer) NULL},
};
#undef offset


#define SUPERCLASS (&xvwColorGadgetClassRec)

XvwPlot2DGadgetClassRec xvwPlot2DGadgetClassRec =
{
  {
    (WidgetClass) SUPERCLASS,		/* superclass		  */	
    "Plot2D",				/* class_name		  */
    sizeof(XvwPlot2DGadgetRec),		/* size			  */
    ClassInitialize,			/* class_initialize	  */
    NULL,				/* class_part_initialize  */
    FALSE,				/* class_inited		  */
    Initialize,				/* initialize		  */
    NULL,				/* initialize_hook	  */
    NULL,				/* realize		  */
    NULL,				/* actions		  */
    0,					/* num_actions		  */
    resources,				/* resources		  */
    knumber(resources),		/* resource_count	  */
    NULLQUARK,				/* xrm_class		  */
    FALSE,				/* compress_motion	  */
    FALSE,				/* compress_exposure	  */
    FALSE,				/* compress_enterleave    */
    FALSE,				/* visible_interest	  */
    Destroy,				/* destroy		  */
    XtInheritResize,			/* resize		  */
    Redisplay,				/* expose		  */
    SetValues,				/* set_values		  */
    NULL,				/* set_values_hook	  */
    XtInheritSetValuesAlmost,		/* set_values_almost	  */
    NULL,				/* get_values_hook	  */
    NULL,				/* accept_focus		  */
    XtVersion,				/* version		  */
    NULL,				/* callback_private	  */
    NULL,				/* tm_table		  */
    NULL,				/* query_geometry	  */
    NULL,				/* display_accelerator	  */
    NULL				/* extension		  */
  },  /* RectObjClass fields initialization */
  {
    PickGadget,				/* pick object proc       */
    XtInheritEraseSel,			/* erase selected proc    */
    XtInheritRefreshSel,		/* refresh selection proc */
    XtInheritChangeGeometry,		/* change geometry proc   */
  },  /* XvwManagerGadgetClass fields initialization */
  {
    XtInheritRecomputePosition,		/* recompute position proc    */
  },  /* XvwGraphicsGadgetClass fields initialization */
  {
    ColorCallback,                      /* reload color proc    */
  },  /* XvwColorGadgetClass fields initialization */
  {
    NULL,				/* extension - not used   */
  },  /* XvwPlot2DGadgetClass fields initialization */
};
#undef SUPERCLASS

/*
 * xvwPlot2DGadgetClass for public consumption
 */
WidgetClass xvwPlot2DGadgetClass = (WidgetClass) &xvwPlot2DGadgetClassRec;


/*-----------------------------------------------------------
|
|  Routine Name: ClassInitialize
|
|       Purpose: This method is called the first time an  
|                instance of plot2d object class has been created.
|                It will initialize all the class attributes.
|
|         Input: None 
|
|        Output: None
|
|    Written By: Mark Young
|          Date: Aug 03, 1994
| Modifications:
|
------------------------------------------------------------*/

static void ClassInitialize(void)
{
        xvw_init_attributes(xvwPlot2DGadgetClass, attributes,
		knumber(attributes), NULL, 0, 
		"$DESIGN/objects/library/xvisual/uis/Plot2D.pane");
	xvw_load_resources("$DESIGN/objects/library/xvisual/app-defaults/Plot2D");

	xvw_define_attribute(xvwPlot2DGadgetClass, XVW_PLOT2D_PLOTFILE,
		XtRString, SetObjectName, GetObjectName);
}

/*-----------------------------------------------------------
|
|  Routine Name: Initialize
|
|       Purpose: This method will set up the initial settings 
|                for a 2D plot object instance. 
|
|         Input: request - not used
|                new     - object instance after initialization, 
|                          with attributes set
|    Written By: Mark Young
|          Date: Aug 03, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Initialize(
   Widget   request,
   Widget   new,
   ArgList  args,
   Cardinal *num_args)
{
	XvwPlot2DGadget xobj = (XvwPlot2DGadget) new;


	/*
	 *  Find the world coordinate minimum and maximum values (some of
	 *  the 3D plotting routines need them.
	 */
	new->core.border_width = 0;
	ReloadPlot(xobj);
	FindMinMax(xobj);
}

/*-----------------------------------------------------------
|
|  Routine Name: Redisplay 
|
|       Purpose: This method is used to initially draw the 
|                the 2D plot object and for redrawing the object 
|                whenever part of it becomes exposed.
|
|         Input: widget - the object being exposed
|                event  - the expose event with information
|                         about what is being exposed
|                region - the region being exposed
|
|    Written By: Mark Young
|          Date: Aug 03, 1994
| Modifications:
|
-----------------------------------------------------------*/
/* ARGSUSED */
static void Redisplay(
   Widget widget,
   XEvent *event,
   Region region)
{
	XvwPlot2DGadget xobj = (XvwPlot2DGadget) widget;

	register Coord *points;
	register int i, id, size, marker_type, ncolors  = 0;
	XColor   *bg, *plotcolor, *xcolors = NULL;


	/*
	 *  Check to see if we are currently visible and we have points
	 *  to be displayed
	 */
	if (!xobj->plot2d.points || !xvw_check_visible(xvw_object(widget)))
	   return;

	id	    = xobj->graphics.id;
	size	    = xobj->plot2d.plot_size;
	points	    = xobj->plot2d.points;
	bg	    = &xobj->graphics.bg;
	plotcolor   = &xobj->graphics.fg;
	marker_type = xobj->graphics.marker_type;

	if ((xobj->plot2d.plot_type == KPLOT2D_COLORMARKER) ||
	    (xobj->plot2d.plot_type == KPLOT2D_COLORLINE))
	{
	   unsigned long *histogram = ColorGetHistogram(widget);

	   if (xobj->color.object == NULL)
	   {
	       kerror("design", "Plot2D Redisplay",
		      "Data being plotted has no color information associated with it.  Cannot display colormarker or colorline plot; re-setting plot type to Line Marker");
	       xobj->plot2d.plot_type =  KPLOT2D_LINEPLOT;
	   }
	   else
	   {
	       if ((xcolors = ColorGetXColors(widget)) == NULL)
	          return;
    
	       ncolors = xobj->color.info->ncolors;
	       for (i = 0; i < ncolors; i++)
	       {
	          if (histogram[i] > 0)
	             (void) ColorAllocate(widget, i);
	       }
	   }
	}

	switch (xobj->plot2d.plot_type)
	{
	   case KPLOT2D_LINEPLOT:         
		X2D_draw_polyline(id, points, size, plotcolor);
		break;

	   case KPLOT2D_DISCRETE:   
		X2D_draw_discrete(id, points, size, plotcolor);
		break;

	   case KPLOT2D_BARGRAPH:  
		X2D_draw_bargraph(id, points, size, plotcolor);
		break;

	   case KPLOT2D_POLYMARKER: 
		X2D_draw_polymarker(id, points, size, marker_type, plotcolor);
		break;

	   case KPLOT2D_COLORMARKER: 
		X2D_draw_colormarker(id, points, size, marker_type, 
				     xcolors, ncolors, plotcolor);
		break;  

	   case KPLOT2D_LINEMARKER: 
		X2D_draw_linemarker(id, points, size, marker_type, plotcolor);
		break;
   
	   case KPLOT2D_COLORLINE: 
		X2D_draw_colorline(id, points, size, NULL, xcolors, ncolors);
		break;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: GetObjectName
|
|       Purpose: GetObjectName is used to get the filename associated
|		 with the attribute.
|
|         Input: object    - the object in which we will be getting the
|			     filename
|		 attribute - the old object to be closed
|
|        Output: calldata  - pointer to retrieve the object name
|       Returns: TRUE (1) if redisplay is required, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Aug 03, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static int GetObjectName(
   xvobject object,
   char   *attribute,
   kaddr  calldata)
{
	kobject plot;
	XvwPlot2DGadget xobj = (XvwPlot2DGadget) xvw_widget(object);
	String *filename = (String *) calldata;


	if (kstrcmp(attribute, XVW_PLOT2D_PLOTFILE) == 0)
	   plot = xobj->plot2d.pobject;
	else
	{
	   kerror(XVISUAL, "GetObjectName", "invalid attribute %s",
			attribute);
	   return(FALSE);
	}
	*filename = NULL;
	return(kpds_get_attribute(plot, KPDS_NAME, filename));
}

/*-----------------------------------------------------------
|
|  Routine Name: SetObjectName
|
|       Purpose: SetObjectName is used to set the filename associated
|		 with the attribute.
|
|         Input: widget    - the widget in which we will be getting the
|			     filename
|		 attribute - the old object to be closed
|		 calldata  - the filename to be set
|
|       Returns: TRUE (1) if redisplay is required, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: Aug 03, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static int SetObjectName(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	kobject plot = NULL;
	int	locate = TRUE;
	XvwPlot2DGadget xobj =  (XvwPlot2DGadget) xvw_widget(object);
	String *filename = (String *) calldata;


	if (*filename != NULL && (plot = kdms_locate(*filename)) == NULL)
	{
	   locate = FALSE;
	   if ((plot = kpds_open_object(*filename, KOBJ_READ)) == NULL)
	   {
	      kerror(XVISUAL, "SetObjectName", "Failed to load 2D Plot '%s'",
			*filename);
	      return(TRUE);
	   }
	}

	if (kstrcmp(attribute, XVW_PLOT2D_PLOTFILE) == 0)
	   xvw_set_attribute(object, XVW_PLOT2D_PLOTOBJ, plot);
	else
	{
	   kerror(XVISUAL, "SetObjectName", "invalid attribute %s",
			attribute);
	   if (!locate) kpds_close_object(plot);
	   return(FALSE);
	}
	if (!locate) kpds_close_object(plot);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: SetValues
|
|       Purpose: This method is used to set the public values
|                of a 2D plot object instance.  These values
|		 include the name and the plot object or points. 
|
|         Input: current - the object containing current settings
|                request - the object containing requested settings
|                new     - the object processed through all set values methods
|
|       Returns: FALSE (0) - XPlotRedraw will handle the redraw in any case  
|    Written By: Mark Young
|          Date: Aug 03, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean SetValues(
   Widget   current,
   Widget   request,
   Widget   new,
   ArgList  args,
   Cardinal *num_args)
{
	XvwPlot2DGadget cobj = (XvwPlot2DGadget) current;
	XvwPlot2DGadget robj = (XvwPlot2DGadget) request;
	XvwPlot2DGadget nobj = (XvwPlot2DGadget) new;

	xvobject object;
	Boolean  redisplay = FALSE;


	if (cobj->plot2d.pobject       != nobj->plot2d.pobject       ||
	    robj->graphics.reset_world != cobj->graphics.reset_world ||
	    cobj->plot2d.plot_size     != nobj->plot2d.plot_size     ||
	    cobj->plot2d.points	       != nobj->plot2d.points)
	{
	   if (cobj->plot2d.points != nobj->plot2d.points)
	   {
	      Coord *tmp;
	      int   size = (nobj->plot2d.plot_size * sizeof(Coord));

	      tmp = (Coord *) kmalloc(size);
	      kfree(cobj->plot2d.points);
	      kmemcpy(tmp, nobj->plot2d.points, size);
	      nobj->plot2d.points = tmp;  
	   }

	   if (cobj->plot2d.pobject != nobj->plot2d.pobject)
	   {
	      nobj->plot2d.pobject =kpds_reference_object(nobj->plot2d.pobject);
	      kpds_close_object(cobj->plot2d.pobject);
	      kpds_add_value_callback(nobj->plot2d.pobject,KPDS_CALLBACK_CHANGE,
			ReloadCallback, (kaddr) new);
	      object = xvw_object(new);
	      xvw_set_attribute(object,XVW_COLOR_COLOROBJ,nobj->plot2d.pobject);
	   }
	   ReloadPlot(nobj);
	   FindMinMax(nobj);
	   redisplay = TRUE;
	}

	if (cobj->plot2d.plot_type != nobj->plot2d.plot_type)
	{
	   redisplay = TRUE;
	}
	return(redisplay);
}

/*-----------------------------------------------------------
|
|  Routine Name: Destroy
|
|       Purpose: Close the image, shape, colormap, and overlay
|		 objects before the image widget is destroyed.
|
|         Input: widget - the image widget being destroyed 
|
|    Written By: Mark Young
|          Date: Aug 03, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Destroy(
   Widget widget)
{
	XvwPlot2DGadget xobj = (XvwPlot2DGadget) widget;

	/*
	 *  Close plot object
	 */
	kpds_close_object(xobj->plot2d.pobject);
	kfree(xobj->plot2d.points);
}

/*-----------------------------------------------------------
|
|  Routine Name: ReloadCallback
|
|       Purpose: Reload the plot2d data structure from the pobject
|		 data object.
|
|         Input:
|        Output: none
|    Written By: Mark Young
|          Date: Aug 03, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ReloadCallback(
   kobject object,
   char    *segment,
   Widget  widget,
   kdms_callback *cb)
{
	int x, y, w, h;

	ReloadPlot((XvwPlot2DGadget) widget);
	FindMinMax((XvwPlot2DGadget) widget);
	x = widget->core.x + 1; y = widget->core.y + 1;
	w = widget->core.width; h = widget->core.height;
	XClearArea(XtDisplay(widget), XtWindowOfObject(widget),x,y,w,h,FALSE);
	Redisplay(widget, NULL, NULL);
}

/*-----------------------------------------------------------
|
|  Routine Name: ColorCallback
|
|       Purpose: Refresh the palette display since our ColorClass
|		 has had a change to its map data, and we are required
|		 to change the colors in the palette.
|
|         Input: object - the object to be refreshed (the Plot2D)
|		 call_data   - (not used)
|        Output:
|    Written By: Mark Young & Danielle Argiro
|          Date: June 28, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ColorCallback(
   xvobject object,
   kaddr    call_data)
{
	XEvent  event;
	Widget  widget = xvw_widget(object);
	Region  region   = XCreateRegion();

	event.type = Expose;
	event.xexpose.x = event.xexpose.y = 0;
	event.xexpose.width  = widget->core.width;
	event.xexpose.height = widget->core.height;
	XtAddExposureToRegion(&event, region);
	Redisplay(widget, &event, region);
	XDestroyRegion(region);
}

/*-----------------------------------------------------------
|
|  Routine Name: FindMinMax
|
|       Purpose: This routine will find the min and max values
|		 from the data in a given 2D plot object
|
|         Input: xobj - the object to find the min max for 
|
|       Returns: TRUE (1) if redisplay is required, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: Aug 03, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void FindMinMax(
   XvwPlot2DGadget xobj)
{
	register int i;
	register Coord wcmin, wcmax, *points;


	if ((points = xobj->plot2d.points) == NULL)
	   return;

	wcmin.x = wcmax.x = points[0].x;
	wcmin.y = wcmax.y = points[0].y;
	wcmin.d = wcmax.d = points[0].d;

	for (i = xobj->plot2d.plot_size -1; i >= 0; i--)
	{
	   if (points[i].x > wcmax.x) wcmax.x = points[i].x;
	   else if (points[i].x < wcmin.x) wcmin.x = points[i].x;

	   if (points[i].y > wcmax.y) wcmax.y = points[i].y;
	   else if (points[i].y < wcmin.y) wcmin.y = points[i].y;

	   if (points[i].d > wcmax.d) wcmax.d = points[i].d;
	   else if (points[i].d < wcmin.d) wcmin.d = points[i].d;
	}

	xobj->plot2d.data_max_x = wcmax.x;
	xobj->plot2d.data_max_y = wcmax.y;
	xobj->plot2d.data_min_x = wcmin.x;
	xobj->plot2d.data_min_y = wcmin.y;

	if (xobj->graphics.reset_world)
	{
	   if ( wcmax.x < xobj->graphics.wcmin.x )
	   {
	      xvw_set_attributes(xvw_object((Widget) xobj),
			         XVW_GRAPHICS_WCMIN_X, wcmin.x,
			         XVW_GRAPHICS_WCMAX_X, wcmax.x,
				 NULL);
	   }
	   else
	   {
	      xvw_set_attributes(xvw_object((Widget) xobj),
			         XVW_GRAPHICS_WCMAX_X, wcmax.x,
			         XVW_GRAPHICS_WCMIN_X, wcmin.x,
				 NULL);
	   }

	   if ( wcmax.y < xobj->graphics.wcmin.y )
	   {
	      xvw_set_attributes(xvw_object((Widget) xobj),
			         XVW_GRAPHICS_WCMIN_Y, wcmin.y,
			         XVW_GRAPHICS_WCMAX_Y, wcmax.y,
				 NULL);
	   }
	   else
	   {
	      xvw_set_attributes(xvw_object((Widget) xobj),
			         XVW_GRAPHICS_WCMAX_Y, wcmax.y,
			         XVW_GRAPHICS_WCMIN_Y, wcmin.y,
				 NULL);
	   }
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: ReloadPlot
|
|       Purpose: This routine will subsample the data from
|		 a given 2D plot object.
|
|         Input: xobj - the object to subsample 
|
|    Written By: Mark Young
|          Date: Aug 03, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ReloadPlot(
   XvwPlot2DGadget xobj)
{
	kobject object;
	int i, w, h, d, t, e;

	register double *data = NULL;
	register Coord  *points;


	/*
	 *  Sanity check to make sure that the image object exists
	 */
	if ((object = xobj->plot2d.pobject) == NULL)
	   return;

	/*
	 *  Set the data segment data to be 
	 */
	kpds_set_attributes(xobj->plot2d.pobject,
			KPDS_VALUE_DATA_TYPE, KDOUBLE,
			KPDS_LOCATION_DATA_TYPE, KDOUBLE,
			NULL);

	/*
	 *  Get the value data demsionality
	 */
	kpds_get_attribute(xobj->plot2d.pobject, KPDS_VALUE_SIZE,
		&w, &h, &d, &t, &e);

	/*
	 *  Set the position to be the beginning of the value and location
	 *  data segments.
	 */
	kpds_set_attributes(xobj->plot2d.pobject,
			KPDS_VALUE_POSITION, 0, 0, 0, 0, 0,
			KPDS_LOCATION_POSITION, 0, 0, 0, 0, 0,
			NULL);

	/*
	 * Special case: pairs of ascii data points
	 */
	if (w == 2)
	{
	    /*
             * Go ahead and malloc room for the new data...
             */
            if (xobj->plot2d.plot_size != h)
            {
                xobj->plot2d.plot_size = h;
                xobj->plot2d.points = (Coord *) krealloc(xobj->plot2d.points,
                                      xobj->plot2d.plot_size*sizeof(Coord));
                if (xobj->plot2d.points == NULL)
                {
                    kerror(XVISUAL, "ReloadPlot",
                           "Unable to allocate plot coordinate data");
                    return;
                }
            }

	    /*
	     * get data points in pairs
	     */
	    
	    points = xobj->plot2d.points;
	    kpds_set_attribute(xobj->plot2d.pobject, KPDS_VALUE_REGION_SIZE,
			       w, 1, 1, 1, 1);
	    for (i = 0; i < h; i++)
	    {
		data = (double *) kpds_get_data(object, KPDS_VALUE_REGION,data);
		points[i].x = data[0];
		points[i].y = data[1];
		points[i].d = (unsigned long) data[1];
	    }
	    kfree(data);
	}

	/*
	 *  General case; Get the image plane
	 */
	else
	{
	    /* get a plane of image data */
	    data = (double *) kpds_get_data(object, KPDS_VALUE_PLANE, NULL);
	    if (data == NULL)
	    {
	       kerror(XVISUAL, "ReloadPlot", "Unable to get image plane");
	       return;
	    }

	    /* 
	     * Go ahead and malloc room for the new data...  
	     */
	    if (xobj->plot2d.plot_size != w*h)
	    {
	        xobj->plot2d.plot_size = w*h;
	        xobj->plot2d.points = (Coord *) krealloc(xobj->plot2d.points,
				      xobj->plot2d.plot_size*sizeof(Coord));
	        if (xobj->plot2d.points == NULL)
	        {
	            kerror(XVISUAL, "ReloadPlot", 
		           "Unable to allocate plot coordinate data");
	            return;
	        }
	    }

	    points = xobj->plot2d.points;

	    /* X implicit; Y in value data; D in value data */
	    for (i = 0; i < xobj->plot2d.plot_size; i++)
	    {
	       points[i].x = i;
	       points[i].y = data[i];
	       points[i].d = (unsigned long) data[i];
	     }

	    /* X in location data; Y in value data; D in value data */
	     if (kpds_get_data(object, KPDS_LOCATION_PLANE, data))
	     {
	       for (i = 0; i < xobj->plot2d.plot_size; i++)
	          points[i].x = data[i];
	     }
	     kfree(data);
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: PickGadget
|
|       Purpose: Given a plot object and an attempted picking location,
|                see if the plot has been picked. 
|
|         Input: widget - the line object we are checking 
|                x      - the x device coordinate that the pick attempt was at 
|                y      - the y device coordinate that the pick attempt was at
|
|       Returns: TRUE (1) if object was successfully picked, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: Aug 03, 1994
| Modifications: 
|
------------------------------------------------------------*/
/* ARGSUSED */
static int PickGadget(
   Widget widget,
   int    x,
   int    y)
{
	XvwPlot2DGadget xobj = (XvwPlot2DGadget) widget;

	Real  xpos1, xpos2, ypos1, ypos2;
	Coord c1, c2, c, *points;
	int i, id = xobj->graphics.id;


	if ((points = xobj->plot2d.points) == NULL)
	   return(FALSE);

	xpos1 = x - PICK_TOLERANCE; ypos1 = y - PICK_TOLERANCE;
	xpos2 = x + PICK_TOLERANCE; ypos2 = y + PICK_TOLERANCE;
	X2D_convert_point_dc_to_wc(id, xpos1, ypos1, &c1);
	X2D_convert_point_dc_to_wc(id, xpos2, ypos2, &c2);
	X2D_convert_point_dc_to_wc(id, (Real) x, (Real) y, &c);

	xpos1 = kabs(c1.x - c2.x);
	ypos1 = kabs(c1.y - c2.y);
	for (i = 0; i < xobj->plot2d.plot_size; i++)
	{
	   if (kabs(c.x-points[i].x) < xpos1 && kabs(c.y - points[i].y) < ypos1)
	      return(TRUE);
	}
	return(FALSE);
}


/************************************************************
*
*  Routine Name: xvw_create_plot2d - create a 2D plot object 
*
*       Purpose: A 2D plot object is used to display a 2D plot.
*                A variety of plot types are supported, including
*                line plots, discrete plots, bar graphs, polymarker
*                plots, linemarker plots, and colormarker plots.
*                The 2D plot object is generally used in conjunction 
*                with a 2D axis object, where both are created as children
*                of an area object so that a coordinated world coordinate
*                view is automatically supported.
*
*         Input: parent - the parent object; NULL will cause a
*                         default toplevel to be created automatically
*                name   - the name with which to reference the object 
*        Output: none
*       Returns: Returns the created 2D plot xvobject, or NULL upon failure
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Aug 03, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

xvobject xvw_create_plot2d(
   xvobject parent,
   char     *name)
{
	xvobject  object;


	/*
	 *  Create the main plot workspace
	 */
	object = xvw_create(parent, FALSE, TRUE, name, Plot2DGadgetClass);
	if (object != NULL)
	{
	   xvw_set_attributes(object,
                XVW_MENUABLE,     TRUE,    /* menuable       */
                XVW_RESIZABLE,    FALSE,   /* resizable      */
                XVW_SELECTABLE,   TRUE,   /* selectable     */
		NULL);
	}
	return(object);
}
