/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>              3D Utilities
   >>>>
   >>>>  Private:
   >>>>              X3D_add_coords()
   >>>>              X3D_subtract_coords()
   >>>>              X3D_multiply_coords()
   >>>>              X3D_divide_coords()
   >>>>              X3D_normalize()
   >>>>              X3D_cross_product()
   >>>>              X3D_dot_product()
   >>>>              X3D_distance()
   >>>>              _X3D_compute_normal()
   >>>>              _X3D_normal()
   >>>>              _X3D_vertex_normal()
   >>>>   Static:
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "graphics.h"


/*-----------------------------------------------------------
|
|  Routine Name: X3D_add_coords
|
|       Purpose: Adds two coords together.
|
|         Input: a - coord to be added
|		 b - coord to be added
|
|        Output: c - result of addition
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X3D_add_coords(
   Coord *a,
   Coord *b,
   Coord *c)
{
	c->x = a->x + b->x;
	c->y = a->y + b->y;
	c->z = a->z + b->z;
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_sub_coords
|
|       Purpose: Subtracts two coords together.
|
|         Input: a - coord to be subtracted
|		 b - coord to be subtracted
|
|        Output: c - result of subtraction
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X3D_subtract_coords(
   Coord *a,
   Coord *b,
   Coord *c)
{
	c->x = a->x - b->x;
	c->y = a->y - b->y;
	c->z = a->z - b->z;
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_multiply_coords
|
|       Purpose: Multiples two coords together.
|
|         Input: a - coord to be multipled
|		 b - coord to be multipled
|
|        Output: c - result of multiplication
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X3D_multiply_coords(
   Coord *a,
   Coord *b,
   Coord *c)
{
	c->x = a->x * b->x;
	c->y = a->y * b->y;
	c->z = a->z * b->z;
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_divide_coords
|
|       Purpose: Divides two coords together.
|
|         Input: a - coord to be added
|		 b - coord to be added
|
|        Output: c - result of addition
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X3D_divide_coords(
   Coord *a,
   Coord *b,
   Coord *c)
{
	if (b->x != 0.0) c->x = a->x / b->x;
	if (b->y != 0.0) c->y = a->y / b->y;
	if (b->z != 0.0) c->z = a->z / b->z;
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_normalize
|
|       Purpose: Normalizes coord.
|
|         Input: pt - coord to be normalized
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X3D_normalize(
   Coord *pt)
{
	Real	magnitude;


	magnitude = ksqrt(pt->x*pt->x + pt->y*pt->y + pt->z*pt->z);
	if (magnitude == 0.0)
	   return;

	pt->x /= magnitude;
	pt->y /= magnitude;
	pt->z /= magnitude;
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_cross_product
|
|       Purpose: Find the cross product of two coords.
|
|         Input: a - coord to find the cross product of
|		 b - coord to find the cross product with

|        Output: c - resulting cross product
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X3D_cross_product(
   Coord *a,
   Coord *b,
   Coord *c)
{
	c->x = (a->y*b->z) - (a->z*b->y);
	c->y = (a->z*b->x) - (a->x*b->z);
	c->z = (a->x*b->y) - (a->y*b->x);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_dot_product
|
|       Purpose: Perform the dot product on two coords.
|
|         Input: a - coord to perform dot product of
|		 b - coord to perform dot product with
|
|        Output: none
|
|       Returns: dot product
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

Real X3D_dot_product(
   Coord *a,
   Coord *b)
{
	Real value;

	value = a->x * b->x + a->y * b->y + a->z * b->z;
	return(value);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_distance
|
|       Purpose: Find the distance between to coords.
|
|         Input: a - first coord
|		 b - second coord
|
|        Output: none
|
|       Returns: distance between coords
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

Real X3D_distance(
   Coord *a,
   Coord *b)
{
	register Real x, y, z, value;

	x = a->x - b->x;
	y = a->y - b->y;
	z = a->z - b->z;
	value = ksqrt((double) (x*x + y*y + z*z));
	return(value);
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_compute_normal
|
|       Purpose: Computes the normal to a
|
|         Input: graphics -
|		 pt1	  -
|		 pt2	  -
|		 pt3	  -
|
|        Output: norm	  - normal
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void _X3D_compute_normal(
   X3DGraphics *graphics,
   Coord       *pt1,
   Coord       *pt2,
   Coord       *pt3,
   Coord       *norm)
{
	Coord	a, b;


	a.x = pt2->x - pt1->x; a.y = pt2->y - pt1->y; a.z = pt2->z - pt1->z;
	b.x = pt3->x - pt2->x; b.y = pt3->y - pt2->y; b.z = pt3->z - pt2->z;
	X3D_cross_product(&a, &b, norm);
	X3D_normalize(norm);
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_normal
|
|       Purpose:
|
|         Input: graphics -
|		 pt1	  -
|		 pt2	  -
|		 pt3	  -
|
|        Output: none
|
|       Returns: normal
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

Real _X3D_normal(
   X3DGraphics *graphics,
   Coord       *pt1,
   Coord       *pt2,
   Coord       *pt3)
{
	Coord    vpn, norm;
	register Real	dotprod;


	_X3D_compute_normal(graphics, pt1, pt2, pt3, &norm);

	vpn.x = graphics->focus.x - graphics->camera.x;
	vpn.y = graphics->focus.y - graphics->camera.y;
	vpn.z = graphics->focus.z - graphics->camera.z;
	X3D_normalize(&vpn);

	dotprod = X3D_dot_product(&vpn, &norm);
	if (graphics->ncode == NORM_ABSOLUTE)
	   dotprod = kabs(dotprod);
	else if (graphics->ncode == NORM_TRUNCATE && dotprod < 0.0)
	   dotprod = 0.0;

	return(dotprod);
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_vertex_normal
|
|       Purpose: 
|
|         Input: graphics -
|		 pts	  -
|		 width	  -
|		 height	  -
|		 x	  -
|		 y	  -
|
|        Output: none
|
|       Returns: normal
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

Real _X3D_vertex_normal(
   X3DGraphics  *graphics,
   Coord        *pts,
   register int width,
   register int height,
   register int x,
   register int y)
{
	Real	 dotprod;
	Coord	 vpn, norm, normal;
	register int i1, i2, i3;


	normal.x =
	normal.y =
	normal.z = 0.0;
	i2 = y*width + x;
	if (x > 0 && y < height-1)
	{
	   i1 = (y+1)*width + (x-1);
	   i3 = (y+1)*width + x;
	   _X3D_compute_normal(graphics, &pts[i1], &pts[i2], &pts[i3], &norm);
	   X3D_add_coords(&normal, &norm, &normal);
	}

	if (x < width-1 && y < height-1)
	{
	   i1 = (y+1)*width + x;
	   i3 = (y+1)*width + (x+1);
	   _X3D_compute_normal(graphics, &pts[i1], &pts[i2], &pts[i3], &norm);
	   X3D_add_coords(&normal, &norm, &normal);
	}

	if (y > 0 && x > 0)
	{
	   i1 = y*width + (x-1);
	   i3 = (y-1)*width + (x-1);
	   _X3D_compute_normal(graphics, &pts[i1], &pts[i2], &pts[i3], &norm);
	   X3D_add_coords(&normal, &norm, &normal);
	}

	if (y < height-1 && x > 0)
	{
	   i1 = (y+1)*width + (x-1);
	   i3 = y*width + (x-1);
	   _X3D_compute_normal(graphics, &pts[i1], &pts[i2], &pts[i3], &norm);
	   X3D_add_coords(&normal, &norm, &normal);
	}

	if (x > 0 && y > 0)
	{
	   i1 = (y-1)*width + (x-1);
	   i3 = (y-1)*width + x;
	   _X3D_compute_normal(graphics, &pts[i1], &pts[i2], &pts[i3], &norm);
	   X3D_add_coords(&normal, &norm, &normal);
	}

	if (x < width-1 && y > 0)
	{
	   i1 = (y-1)*width + x;
	   i3 = (y-1)*width + (x+1);
	   _X3D_compute_normal(graphics, &pts[i1], &pts[i2], &pts[i3], &norm);
	   X3D_add_coords(&normal, &norm, &normal);
	}

	if (y > 0 && x < width-1)
	{
	   i1 = (y-1)*width + (x+1);
	   i3 = y*width + (x+1);
	   _X3D_compute_normal(graphics, &pts[i1], &pts[i2], &pts[i3], &norm);
	   X3D_add_coords(&normal, &norm, &normal);
	}

	if (y < height-1 && x < width-1)
	{
	   i1 = y*width + (x+1);
	   i3 = (y+1)*width + (x+1);
	   _X3D_compute_normal(graphics, &pts[i1], &pts[i2], &pts[i3], &norm);
	   X3D_add_coords(&normal, &norm, &normal);
	}
	X3D_normalize(&normal);

	vpn.x = graphics->focus.x - graphics->camera.x;
	vpn.y = graphics->focus.y - graphics->camera.y;
	vpn.z = graphics->focus.z - graphics->camera.z;
        X3D_normalize(&vpn);

	dotprod = X3D_dot_product(&vpn, &normal);
	if (graphics->ncode == NORM_ABSOLUTE)
	   dotprod = kabs(dotprod);
	else if (graphics->ncode == NORM_TRUNCATE && dotprod < 0.0)
	   dotprod = 0.0;

	return(dotprod);
}
