/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>   		3D Surface Drawing Utilities
   >>>>
   >>>>  Private:
   >>>>			X3D_shade_constant()
   >>>>			X3D_shade_ghouraud()
   >>>>			X3D_shade_phong()
   >>>>   Static:
   >>>>			_X3D_constant_imagery()
   >>>>			_X3D_constant_elevations()
   >>>>			_X3D_constant_normals()
   >>>>			_X3D_ghouraud_imagery()
   >>>>			_X3D_ghouraud_elevations()
   >>>>			_X3D_ghouraud_normals()
   >>>>			_X3D_phong_imagery()
   >>>>			_X3D_phong_elevations()
   >>>>			_X3D_phong_normals()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "graphics.h"
#include <xvisual/Plot3D.h>


typedef struct
{
  void (*routine)(X3DGraphics *, Coord, Coord, Coord, XPoint, XPoint, XPoint,
		  Real *, XColor *, int, int, Real, Real, Real);
} ShadingRoutines;

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_constant_imagery
|
|       Purpose: 
|
|         Input: graphics     -
|		 poly1	      -
|		 poly2	      -
|		 poly3	      -
|		 pt1	      -
|		 pt2	      -
|		 pt3	      -
|		 levels	      -
|		 colors	      -
|		 color_offset -
|		 num	      -
|		 n1	      -
|		 n2	      -
|		 n3	      -
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

static void _X3D_constant_imagery(
   X3DGraphics *graphics,
   Coord       poly1,
   Coord       poly2,
   Coord       poly3,
   XPoint      pt1,
   XPoint      pt2,
   XPoint      pt3,
   Real        *levels,
   XColor      *colors,
   int         color_offset,
   int         num,
   Real        n1,
   Real        n2,
   Real        n3)
{
	int	 n;
	XPoint   pts[3];


	n = (poly1.d + poly2.d + poly3.d)/3;
	n = _X3D_find_color(n, colors, color_offset, num);

	pts[0] = pt1; pts[1] = pt2; pts[2] = pt3;
	X3D_draw[graphics->device].fill_color(graphics, &colors[n], TRUE);
	X3D_draw[graphics->device].fill_polygon(graphics, pts, 3, Convex,
		CoordModeOrigin);
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_constant_elevations
|
|       Purpose:
|
|         Input: graphics     -
|		 poly1	      -
|		 poly2	      -
|		 poly3	      -
|		 pt1	      -
|		 pt2	      -
|		 pt3	      -
|		 levels	      -
|		 colors	      -
|		 color_offset -
|		 num	      -
|		 n1	      -
|		 n2	      -
|		 n3	      -
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

static void _X3D_constant_elevations(
   X3DGraphics *graphics,
   Coord       poly1,
   Coord       poly2,
   Coord       poly3,
   XPoint      pt1,
   XPoint      pt2,
   XPoint      pt3,
   Real        *levels,
   XColor      *colors,
   int         color_offset,
   int         num,
   Real        n1,
   Real        n2,
   Real        n3)
{
	int	 n;
	XPoint   pts[3];


	/*n = (n1 + n2 + n3)/3 + color_offset;*/
	n = (n1 + n2 + n3)/3;
	n = _X3D_find_color(n, colors, color_offset, num);

	pts[0] = pt1; pts[1] = pt2; pts[2] = pt3;
	X3D_draw[graphics->device].fill_color(graphics, &colors[n], TRUE);
	X3D_draw[graphics->device].fill_polygon(graphics, pts, 3, Convex,
		CoordModeOrigin);
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_constant_normals
|
|       Purpose:
|
|         Input: graphics     -
|		 poly1	      -
|		 poly2	      -
|		 poly3	      -
|		 pt1	      -
|		 pt2	      -
|		 pt3	      -
|		 levels	      -
|		 colors	      -
|		 color_offset -
|		 num	      -
|		 n1	      -
|		 n2	      -
|		 n3	      -
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

static void _X3D_constant_normals(
   X3DGraphics *graphics,
   Coord       poly1,
   Coord       poly2,
   Coord       poly3,
   XPoint      pt1,
   XPoint      pt2,
   XPoint      pt3,
   Real        *levels,
   XColor      *colors,
   int         color_offset,
   int         num,
   Real        n1,
   Real        n2,
   Real        n3)
{
	int	 n;
	XPoint   pts[3];


	n = _X3D_normal(graphics, &poly1, &poly2, &poly3) *
			num + color_offset;

	pts[0] = pt1; pts[1] = pt2; pts[2] = pt3;
	X3D_draw[graphics->device].fill_color(graphics, &colors[n], TRUE);
	X3D_draw[graphics->device].fill_polygon(graphics, pts, 3, Convex,
		CoordModeOrigin);
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_ghouraud_imagery
|
|       Purpose:
|
|         Input: graphics     -
|		 poly1	      -
|		 poly2	      -
|		 poly3	      -
|		 pt1	      -
|		 pt2	      -
|		 pt3	      -
|		 levels	      -
|		 colors	      -
|		 color_offset -
|		 num	      -
|		 n1	      -
|		 n2	      -
|		 n3	      -
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

static void _X3D_ghouraud_imagery(
   X3DGraphics *graphics,
   Coord       poly1,
   Coord       poly2,
   Coord       poly3,
   XPoint      pt1,
   XPoint      pt2,
   XPoint      pt3,
   Real        *levels,
   XColor      *colors,
   int         color_offset,
   int         num,
   Real        n1,
   Real        n2,
   Real        n3)
{
	Coord	 poly[4];
	XPoint   pts[4];
	int	 i, incr, indx;
	register int indx1, indx2, indx3;


	/*
	 *  Make a quick check to see if the triangle is one color
	 */
	if (poly1.d == poly2.d && poly2.d == poly3.d)
	{
	   indx1 = poly1.d;
	   pts[0] = pt1; pts[1] = pt2; pts[2] = pt3;
	   X3D_draw[graphics->device].fill_color(graphics, &colors[indx1],
				TRUE);
	   X3D_draw[graphics->device].fill_polygon(graphics, pts, 3,
				Convex, CoordModeOrigin);
	   return;
	}

	indx1 = poly1.d;
	indx2 = poly2.d;
	indx3 = poly3.d;

	/*
	 *  Now that we have the three values we are going to interpolate
	 *  the data from each of the corners.
	 */

	/*
	 *  Interpolate from indx1 & indx2 to indx3
	 */
	if ((indx2 < indx1 && indx3 < indx1) ||
	    (indx2 > indx1 && indx3 > indx1))
	{
	   poly[0] = poly[1] = poly1;
	   incr = ((indx1 < indx3) ? 1 : -1);
	   for (i = indx1; i != indx3 && i != indx2; i+= incr)
	   {
	      poly[2].x = kintercept(poly1.x, poly2.x, indx1, indx2, i+incr);
	      poly[2].y = kintercept(poly1.y, poly2.y, indx1, indx2, i+incr);
	      poly[2].z = kintercept(poly1.z, poly2.z, indx1, indx2, i+incr);
	      poly[3].x = kintercept(poly1.x, poly3.x, indx1, indx3, i+incr);
	      poly[3].y = kintercept(poly1.y, poly3.y, indx1, indx3, i+incr);
	      poly[3].z = kintercept(poly1.z, poly3.z, indx1, indx3, i+incr);

	      if (_X3D_convert_wc_to_dc(graphics, graphics->matrix2, poly,
			pts, 4))
	      {
	 	 indx = _X3D_find_color(i, colors, color_offset, num);
	         X3D_draw[graphics->device].fill_color(graphics, &colors[indx],
				TRUE);
	         X3D_draw[graphics->device].fill_polygon(graphics, pts, 4,
				Convex, CoordModeOrigin);
	      }
	      poly[0] = poly[3];
	      poly[1] = poly[2];
	   }
	}

	/*
	 *  Interpolate from indx1 & indx3 to indx2
	 */
	if ((indx1 < indx3 && indx2 < indx3) ||
	    (indx1 > indx3 && indx2 > indx3))
	{
	   poly[0] = poly[1] = poly3;
	   incr = ((indx3 < indx2) ? 1 : -1);
	   for (i = indx3; i != indx2 && i != indx1; i += incr)
	   {
	      poly[2].x = kintercept(poly1.x, poly3.x, indx1, indx3, i+incr);
	      poly[2].y = kintercept(poly1.y, poly3.y, indx1, indx3, i+incr);
	      poly[2].z = kintercept(poly1.z, poly3.z, indx1, indx3, i+incr);
	      poly[3].x = kintercept(poly2.x, poly3.x, indx2, indx3, i+incr);
	      poly[3].y = kintercept(poly2.y, poly3.y, indx2, indx3, i+incr);
	      poly[3].z = kintercept(poly2.z, poly3.z, indx2, indx3, i+incr);

	      if (_X3D_convert_wc_to_dc(graphics, graphics->matrix2, poly,
			pts, 4))
	      {
	 	 indx = _X3D_find_color(i, colors, color_offset, num);
	         X3D_draw[graphics->device].fill_color(graphics, &colors[indx],
				TRUE);
	         X3D_draw[graphics->device].fill_polygon(graphics, pts, 4,
				Convex, CoordModeOrigin);
	      }
	      poly[0] = poly[3];
	      poly[1] = poly[2];
	   }
	}

	/*
	 *  Interpolate from indx2 & indx3 to indx1
	 */
	if ((indx1 < indx2 && indx3 < indx2) ||
	    (indx1 > indx2 && indx3 > indx2))
	{
	   poly[0] = poly[1] = poly2;
	   incr = ((indx2 < indx1) ? 1 : -1);
	   for (i = indx2; i != indx1 && i != indx3; i += incr)
	   {
	      poly[2].x = kintercept(poly2.x, poly3.x, indx2, indx3, i+incr);
	      poly[2].y = kintercept(poly2.y, poly3.y, indx2, indx3, i+incr);
	      poly[2].z = kintercept(poly2.z, poly3.z, indx2, indx3, i+incr);
	      poly[3].x = kintercept(poly1.x, poly2.x, indx1, indx2, i+incr);
	      poly[3].y = kintercept(poly1.y, poly2.y, indx1, indx2, i+incr);
	      poly[3].z = kintercept(poly1.z, poly2.z, indx1, indx2, i+incr);

	      if (_X3D_convert_wc_to_dc(graphics, graphics->matrix2, poly,
			pts, 4))
	      {
	         indx = _X3D_find_color(i, colors, color_offset, num);
	         X3D_draw[graphics->device].fill_color(graphics, &colors[indx],
				TRUE);
	         X3D_draw[graphics->device].fill_polygon(graphics, pts, 4,
				Convex, CoordModeOrigin);
	      }
	      poly[0] = poly[3];
	      poly[1] = poly[2];
	   }
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_ghouraud_elevations
|
|       Purpose:
|
|         Input: graphics     -
|		 poly1	      -
|		 poly2	      -
|		 poly3	      -
|		 pt1	      -
|		 pt2	      -
|		 pt3	      -
|		 levels	      -
|		 colors	      -
|		 color_offset -
|		 num	      -
|		 n1	      -
|		 n2	      -
|		 n3	      -
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

static void _X3D_ghouraud_elevations(
   X3DGraphics *graphics,
   Coord       poly1,
   Coord       poly2,
   Coord       poly3,
   XPoint      pt1,
   XPoint      pt2,
   XPoint      pt3,
   Real        *levels,
   XColor      *colors,
   int         color_offset,
   int         num,
   Real        n1,
   Real        n2,
   Real        n3)
{
	Real    w;
	XPoint	points[4];
	Coord	min, mid, max, temp, pts[4];
	int	indx, indx1, indx2, indx3;

	_X3D_vector_mult(poly1, graphics->imatrix, temp.x, temp.y, temp.z, w);
	poly1 = temp;
	_X3D_vector_mult(poly2, graphics->imatrix, temp.x, temp.y, temp.z, w);
	poly2 = temp;
	_X3D_vector_mult(poly3, graphics->imatrix, temp.x, temp.y, temp.z, w);
	poly3 = temp;
	if (poly1.z < poly2.z)
	{
	   if (poly3.z < poly1.z)
	   {
	      min = poly3;
	      mid = poly1;
	      max = poly2;
	   }
	   else if (poly3.z > poly2.z)
	   {
	      min = poly1;
	      mid = poly2;
	      max = poly3;
	   }
	   else
	   {
	      min = poly1;
	      mid = poly3;
	      max = poly2;
	   }
	}
	else
	{
	   if (poly3.z < poly2.z)
	   {
	      min = poly3;
	      mid = poly2;
	      max = poly1;
	   }
	   else if (poly3.z > poly1.z)
	   {
	      min = poly2;
	      mid = poly1;
	      max = poly3;
	   }
	   else
	   {
	      min = poly2;
	      mid = poly3;
	      max = poly1;
	   }
	}

	indx1 = _X3D_find_level(min.z, levels, num);
	indx2 = _X3D_find_level(mid.z, levels, num);
	indx3 = _X3D_find_level(max.z, levels, num);

	if (indx1 == indx2 && indx2 == indx3)
	{
	   pts[0] = min; pts[1] = mid; pts[2] = max;
	   if (_X3D_convert_wc_to_dc(graphics, graphics->matrix, pts,
			points, 3))
	   {
	      indx = indx1+color_offset;
	      indx = _X3D_find_color(indx, colors, color_offset, num);
	      X3D_draw[graphics->device].fill_color(graphics, &colors[indx],
				TRUE);
	      X3D_draw[graphics->device].fill_polygon(graphics, points, 3,
				Convex, CoordModeOrigin);
	   }
	}
	else
	{
	   pts[0] = pts[1] = min;
	   while (indx1 != indx2)
	   {
	      /*
	       *  Compute the end point for 
	       */
	      pts[3].x = kintercept(pts[0].x,mid.x,pts[0].z,
				    mid.z,levels[indx1+1]);
	      pts[3].y = kintercept(pts[0].y,mid.y,pts[0].z,
				    mid.z,levels[indx1+1]);
	      pts[3].z = levels[indx1 +1];

	      pts[2].x = kintercept(pts[1].x,max.x,pts[1].z,
				    max.z,levels[indx1+1]);
	      pts[2].y = kintercept(pts[1].y,max.y,pts[1].z,
				    max.z,levels[indx1+1]);
	      pts[2].z = levels[indx1 +1];

	      if (_X3D_convert_wc_to_dc(graphics, graphics->matrix, pts,
			points, 4))
	      {
		 indx = indx1+color_offset;
	         indx = _X3D_find_color(indx, colors, color_offset, num);
		 X3D_draw[graphics->device].fill_color(graphics, &colors[indx],
				TRUE);
		 X3D_draw[graphics->device].fill_polygon(graphics, points, 4,
				Convex, CoordModeOrigin);
	      }
	      pts[0] = pts[3];
	      pts[1] = pts[2];
	      indx1++;
	   }

	   if (pts[0].z < mid.z)
	   {
	      pts[2] = mid;
	      if (_X3D_convert_wc_to_dc(graphics, graphics->matrix, pts,
			points, 3))
	      {
		 indx = indx2+color_offset;
	         indx = _X3D_find_color(indx, colors, color_offset, num);
		 X3D_draw[graphics->device].fill_color(graphics, &colors[indx],
				TRUE);
	         X3D_draw[graphics->device].fill_polygon(graphics, points, 3,
			        Convex, CoordModeOrigin);
	      }
	   }
	   pts[0] = mid;

	   while (indx2 != indx3)
	   {
	      /*
	       *  Compute the end point for 
	       */
	      pts[3].x = kintercept(pts[0].x,max.x,pts[0].z,
				    max.z,levels[indx2+1]);
	      pts[3].y = kintercept(pts[0].y,max.y,pts[0].z,
				    max.z,levels[indx2+1]);
	      pts[3].z = levels[indx2 +1];

	      pts[2].x = kintercept(pts[1].x,max.x,pts[1].z,
				    max.z,levels[indx2+1]);
	      pts[2].y = kintercept(pts[1].y,max.y,pts[1].z,
				    max.z,levels[indx2+1]);
	      pts[2].z = levels[indx2 +1];

	      if (_X3D_convert_wc_to_dc(graphics, graphics->matrix, pts,
			points, 4))
	      {
		 indx = indx2+color_offset;
	         indx = _X3D_find_color(indx, colors, color_offset, num);
		 X3D_draw[graphics->device].fill_color(graphics, &colors[indx],
				TRUE);
		 X3D_draw[graphics->device].fill_polygon(graphics, points, 4,
				Convex, CoordModeOrigin);
	      }
	      pts[0] = pts[3];
	      pts[1] = pts[2];
	      indx2++;
	   }

	   /*
	    *  Draw the last part of the triangle in case the max
	    *  point is greater than indx3.
	    */
	   if (levels[indx3] < max.z && levels[indx3] < mid.z)
	   {
	      pts[2] = max;
	      pts[3] = mid;
	      if (_X3D_convert_wc_to_dc(graphics, graphics->matrix, pts,
			points, 4))
	      {
		 indx = indx3+color_offset;
	         indx = _X3D_find_color(indx, colors, color_offset, num);
		 X3D_draw[graphics->device].fill_color(graphics, &colors[indx],
				TRUE);
	         X3D_draw[graphics->device].fill_polygon(graphics, points, 4,
			        Convex, CoordModeOrigin);
	      }
	   }
	   else if (levels[indx3] < max.z)
	   {
	      pts[2] = max;
	      if (_X3D_convert_wc_to_dc(graphics, graphics->matrix, pts,
			points, 3))
	      {
		 indx = indx3+color_offset;
	         indx = _X3D_find_color(indx, colors, color_offset, num);
		 X3D_draw[graphics->device].fill_color(graphics, &colors[indx],
				TRUE);
	         X3D_draw[graphics->device].fill_polygon(graphics, points, 3,
			        Convex, CoordModeOrigin);
	      }
	   }
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_ghouraud_normals
|
|       Purpose:
|
|         Input: graphics     -
|		 poly1	      -
|		 poly2	      -
|		 poly3	      -
|		 pt1	      -
|		 pt2	      -
|		 pt3	      -
|		 levels	      -
|		 colors	      -
|		 color_offset -
|		 num	      -
|		 n1	      -
|		 n2	      -
|		 n3	      -
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

static void _X3D_ghouraud_normals(
   X3DGraphics *graphics,
   Coord       poly1,
   Coord       poly2,
   Coord       poly3,
   XPoint      pt1,
   XPoint      pt2,
   XPoint      pt3,
   Real        *levels,
   XColor      *colors,
   int         color_offset,
   int         num,
   Real        n1,
   Real        n2,
   Real        n3)
{
	Coord	 poly[4];
	XPoint   pts[4];
	int	 i, incr, indx;
	register int indx1, indx2, indx3;


	/*
	 *  Make a quick check to see if the triangle is one color
	 */
	indx1 = n1 * num + color_offset;
	indx2 = n2 * num + color_offset;
	indx3 = n3 * num + color_offset;
	if (indx1 == indx2 && indx2 == indx3)
	{
	   pts[0] = pt1; pts[1] = pt2; pts[2] = pt3;
	   indx = _X3D_find_color(indx1, colors, color_offset, num);
	   X3D_draw[graphics->device].fill_color(graphics, &colors[indx],
				TRUE);
	   X3D_draw[graphics->device].fill_polygon(graphics, pts, 3,
				Convex, CoordModeOrigin);
	   return;
	}

	/*
	 *  Now that we have the three values we are going to interpolate
	 *  the data from each of the corners.
	 */

	/*
	 *  Interpolate from indx1 & indx2 to indx3
	 */
	if ((indx2 < indx1 && indx3 < indx1) ||
	    (indx2 > indx1 && indx3 > indx1))
	{
	   poly[0] = poly[1] = poly1;
	   incr = ((indx1 < indx3) ? 1 : -1);
	   for (i = indx1; i != indx3 && i != indx2; i+= incr)
	   {
	      poly[2].x = kintercept(poly1.x, poly2.x, indx1, indx2, i+incr);
	      poly[2].y = kintercept(poly1.y, poly2.y, indx1, indx2, i+incr);
	      poly[2].z = kintercept(poly1.z, poly2.z, indx1, indx2, i+incr);
	      poly[3].x = kintercept(poly1.x, poly3.x, indx1, indx3, i+incr);
	      poly[3].y = kintercept(poly1.y, poly3.y, indx1, indx3, i+incr);
	      poly[3].z = kintercept(poly1.z, poly3.z, indx1, indx3, i+incr);

	      if (_X3D_convert_wc_to_dc(graphics, graphics->matrix2, poly,
			pts, 4))
	      {
	         indx = _X3D_find_color(i, colors, color_offset, num);
	         X3D_draw[graphics->device].fill_color(graphics, &colors[indx],
				TRUE);
	         X3D_draw[graphics->device].fill_polygon(graphics, pts, 4,
				Convex, CoordModeOrigin);
	      }
	      poly[0] = poly[3];
	      poly[1] = poly[2];
	   }
	}

	/*
	 *  Interpolate from indx1 & indx3 to indx2
	 */
	if ((indx1 < indx3 && indx2 < indx3) ||
	    (indx1 > indx3 && indx2 > indx3))
	{
	   poly[0] = poly[1] = poly3;
	   incr = ((indx3 < indx2) ? 1 : -1);
	   for (i = indx3; i != indx2 && i != indx1; i += incr)
	   {
	      poly[2].x = kintercept(poly1.x, poly3.x, indx1, indx3, i+incr);
	      poly[2].y = kintercept(poly1.y, poly3.y, indx1, indx3, i+incr);
	      poly[2].z = kintercept(poly1.z, poly3.z, indx1, indx3, i+incr);
	      poly[3].x = kintercept(poly2.x, poly3.x, indx2, indx3, i+incr);
	      poly[3].y = kintercept(poly2.y, poly3.y, indx2, indx3, i+incr);
	      poly[3].z = kintercept(poly2.z, poly3.z, indx2, indx3, i+incr);

	      if (_X3D_convert_wc_to_dc(graphics, graphics->matrix2, poly,
			pts, 4))
	      {
	         indx = _X3D_find_color(i, colors, color_offset, num);
	         X3D_draw[graphics->device].fill_color(graphics, &colors[indx],
				TRUE);
	         X3D_draw[graphics->device].fill_polygon(graphics, pts, 4,
				Convex, CoordModeOrigin);
	      }
	      poly[0] = poly[3];
	      poly[1] = poly[2];
	   }
	}

	/*
	 *  Interpolate from indx2 & indx3 to indx1
	 */
	if ((indx1 < indx2 && indx3 < indx2) ||
	    (indx1 > indx2 && indx3 > indx2))
	{
	   poly[0] = poly[1] = poly2;
	   incr = ((indx2 < indx1) ? 1 : -1);
	   for (i = indx2; i != indx1 && i != indx3; i += incr)
	   {
	      poly[2].x = kintercept(poly2.x, poly3.x, indx2, indx3, i+incr);
	      poly[2].y = kintercept(poly2.y, poly3.y, indx2, indx3, i+incr);
	      poly[2].z = kintercept(poly2.z, poly3.z, indx2, indx3, i+incr);
	      poly[3].x = kintercept(poly1.x, poly2.x, indx1, indx2, i+incr);
	      poly[3].y = kintercept(poly1.y, poly2.y, indx1, indx2, i+incr);
	      poly[3].z = kintercept(poly1.z, poly2.z, indx1, indx2, i+incr);

	      if (_X3D_convert_wc_to_dc(graphics, graphics->matrix2, poly,
			pts, 4))
	      {
	         indx = _X3D_find_color(i, colors, color_offset, num);
	         X3D_draw[graphics->device].fill_color(graphics, &colors[indx],
				TRUE);
	         X3D_draw[graphics->device].fill_polygon(graphics, pts, 4,
				Convex, CoordModeOrigin);
	      }
	      poly[0] = poly[3];
	      poly[1] = poly[2];
	   }
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_phong_imagery
|
|       Purpose:
|
|         Input: graphics     -
|		 poly1	      -
|		 poly2	      -
|		 poly3	      -
|		 pt1	      -
|		 pt2	      -
|		 pt3	      -
|		 levels	      -
|		 colors	      -
|		 color_offset -
|		 num	      -
|		 n1	      -
|		 n2	      -
|		 n3	      -
|
|        Output:
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

static void _X3D_phong_imagery(
   X3DGraphics *graphics,
   Coord       poly1,
   Coord       poly2,
   Coord       poly3,
   XPoint      pt1,
   XPoint      pt2,
   XPoint      pt3,
   Real        *levels,
   XColor      *colors,
   int         color_offset,
   int         num,
   Real        n1,
   Real        n2,
   Real        n3)
{
	Coord	 poly[4];
	XPoint   pts[4];
	int	 i, incr, indx;
	register int indx1, indx2, indx3;


	/*
	 *  Make a quick check to see if the triangle is one color
	 */
	if (poly1.d == poly2.d && poly2.d == poly3.d)
	{
	   indx1 = poly1.d;
	   pts[0] = pt1; pts[1] = pt2; pts[2] = pt3;
	   X3D_draw[graphics->device].fill_color(graphics, &colors[indx1],
				TRUE);
	   X3D_draw[graphics->device].fill_polygon(graphics, pts, 3,
				Convex, CoordModeOrigin);
	   return;
	}

	indx1 = poly1.d;
	indx2 = poly2.d;
	indx3 = poly3.d;

	/*
	 *  Now that we have the three values we are going to interpolate
	 *  the data from each of the corners.
	 */

	/*
	 *  Interpolate from indx1 & indx2 to indx3
	 */
	if ((indx2 < indx1 && indx3 < indx1) ||
	    (indx2 > indx1 && indx3 > indx1))
	{
	   poly[0] = poly[1] = poly1;
	   incr = ((indx1 < indx3) ? 1 : -1);
	   for (i = indx1; i != indx3 && i != indx2; i+= incr)
	   {
	      poly[2].x = kintercept(poly1.x, poly2.x, indx1, indx2, i+incr);
	      poly[2].y = kintercept(poly1.y, poly2.y, indx1, indx2, i+incr);
	      poly[2].z = kintercept(poly1.z, poly2.z, indx1, indx2, i+incr);
	      poly[3].x = kintercept(poly1.x, poly3.x, indx1, indx3, i+incr);
	      poly[3].y = kintercept(poly1.y, poly3.y, indx1, indx3, i+incr);
	      poly[3].z = kintercept(poly1.z, poly3.z, indx1, indx3, i+incr);

	      if (_X3D_convert_wc_to_dc(graphics, graphics->matrix2, poly,
			pts, 4))
	      {
	 	 indx = _X3D_find_color(i, colors, color_offset, num);
	         X3D_draw[graphics->device].fill_color(graphics, &colors[indx],
				TRUE);
	         X3D_draw[graphics->device].fill_polygon(graphics, pts, 4,
				Convex, CoordModeOrigin);
	      }
	      poly[0] = poly[3];
	      poly[1] = poly[2];
	   }
	}

	/*
	 *  Interpolate from indx1 & indx3 to indx2
	 */
	if ((indx1 < indx3 && indx2 < indx3) ||
	    (indx1 > indx3 && indx2 > indx3))
	{
	   poly[0] = poly[1] = poly3;
	   incr = ((indx3 < indx2) ? 1 : -1);
	   for (i = indx3; i != indx2 && i != indx1; i += incr)
	   {
	      poly[2].x = kintercept(poly1.x, poly3.x, indx1, indx3, i+incr);
	      poly[2].y = kintercept(poly1.y, poly3.y, indx1, indx3, i+incr);
	      poly[2].z = kintercept(poly1.z, poly3.z, indx1, indx3, i+incr);
	      poly[3].x = kintercept(poly2.x, poly3.x, indx2, indx3, i+incr);
	      poly[3].y = kintercept(poly2.y, poly3.y, indx2, indx3, i+incr);
	      poly[3].z = kintercept(poly2.z, poly3.z, indx2, indx3, i+incr);

	      if (_X3D_convert_wc_to_dc(graphics, graphics->matrix2, poly,
			pts, 4))
	      {
	 	 indx = _X3D_find_color(i, colors, color_offset, num);
	         X3D_draw[graphics->device].fill_color(graphics, &colors[indx],
				TRUE);
	         X3D_draw[graphics->device].fill_polygon(graphics, pts, 4,
				Convex, CoordModeOrigin);
	      }
	      poly[0] = poly[3];
	      poly[1] = poly[2];
	   }
	}

	/*
	 *  Interpolate from indx2 & indx3 to indx1
	 */
	if ((indx1 < indx2 && indx3 < indx2) ||
	    (indx1 > indx2 && indx3 > indx2))
	{
	   poly[0] = poly[1] = poly2;
	   incr = ((indx2 < indx1) ? 1 : -1);
	   for (i = indx2; i != indx1 && i != indx3; i += incr)
	   {
	      poly[2].x = kintercept(poly2.x, poly3.x, indx2, indx3, i+incr);
	      poly[2].y = kintercept(poly2.y, poly3.y, indx2, indx3, i+incr);
	      poly[2].z = kintercept(poly2.z, poly3.z, indx2, indx3, i+incr);
	      poly[3].x = kintercept(poly1.x, poly2.x, indx1, indx2, i+incr);
	      poly[3].y = kintercept(poly1.y, poly2.y, indx1, indx2, i+incr);
	      poly[3].z = kintercept(poly1.z, poly2.z, indx1, indx2, i+incr);

	      if (_X3D_convert_wc_to_dc(graphics, graphics->matrix2, poly,
			pts, 4))
	      {
	         indx = _X3D_find_color(i, colors, color_offset, num);
	         X3D_draw[graphics->device].fill_color(graphics, &colors[indx],
				TRUE);
	         X3D_draw[graphics->device].fill_polygon(graphics, pts, 4,
				Convex, CoordModeOrigin);
	      }
	      poly[0] = poly[3];
	      poly[1] = poly[2];
	   }
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_phong_elevations
|
|       Purpose:
|
|         Input: graphics     -
|		 poly1	      -
|		 poly2	      -
|		 poly3	      -
|		 pt1	      -
|		 pt2	      -
|		 pt3	      -
|		 levels	      -
|		 colors	      -
|		 color_offset -
|		 num	      -
|		 n1	      -
|		 n2	      -
|		 n3	      -
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

static void _X3D_phong_elevations(
   X3DGraphics *graphics,
   Coord       poly1,
   Coord       poly2,
   Coord       poly3,
   XPoint      pt1,
   XPoint      pt2,
   XPoint      pt3,
   Real        *levels,
   XColor      *colors,
   int         color_offset,
   int         num,
   Real        n1,
   Real        n2,
   Real        n3)
{
	Real    w;
	XPoint	points[4];
	Coord	min, mid, max, temp, pts[4];
	int	indx, indx1, indx2, indx3;


	_X3D_vector_mult(poly1, graphics->imatrix, temp.x, temp.y, temp.z, w);
	poly1 = temp;
	_X3D_vector_mult(poly2, graphics->imatrix, temp.x, temp.y, temp.z, w);
	poly2 = temp;
	_X3D_vector_mult(poly3, graphics->imatrix, temp.x, temp.y, temp.z, w);
	poly3 = temp;
	if (poly1.z < poly2.z)
	{
	   if (poly3.z < poly1.z)
	   {
	      min = poly3;
	      mid = poly1;
	      max = poly2;
	   }
	   else if (poly3.z > poly2.z)
	   {
	      min = poly1;
	      mid = poly2;
	      max = poly3;
	   }
	   else
	   {
	      min = poly1;
	      mid = poly3;
	      max = poly2;
	   }
	}
	else
	{
	   if (poly3.z < poly2.z)
	   {
	      min = poly3;
	      mid = poly2;
	      max = poly1;
	   }
	   else if (poly3.z > poly1.z)
	   {
	      min = poly2;
	      mid = poly1;
	      max = poly3;
	   }
	   else
	   {
	      min = poly2;
	      mid = poly3;
	      max = poly1;
	   }
	}

	indx1 = _X3D_find_level(min.z, levels, num);
	indx2 = _X3D_find_level(mid.z, levels, num);
	indx3 = _X3D_find_level(max.z, levels, num);

	if (indx1 == indx2 && indx2 == indx3)
	{
	   pts[0] = min; pts[1] = mid; pts[2] = max;
	   if (_X3D_convert_wc_to_dc(graphics, graphics->matrix, pts,
			points, 3))
	   {
	      indx = indx1+color_offset;
	      indx = _X3D_find_color(indx, colors, color_offset, num);
	      X3D_draw[graphics->device].fill_color(graphics, &colors[indx],
				TRUE);
	      X3D_draw[graphics->device].fill_polygon(graphics, points, 3,
				Convex, CoordModeOrigin);
	   }
	}
	else
	{
	   pts[0] = pts[1] = min;
	   while (indx1 != indx2)
	   {
	      /*
	       *  Compute the end point for 
	       */
	      pts[3].x = kintercept(pts[0].x,mid.x,pts[0].z,
				    mid.z,levels[indx1+1]);
	      pts[3].y = kintercept(pts[0].y,mid.y,pts[0].z,
				    mid.z,levels[indx1+1]);
	      pts[3].z = levels[indx1 +1];

	      pts[2].x = kintercept(pts[1].x,max.x,pts[1].z,
				    max.z,levels[indx1+1]);
	      pts[2].y = kintercept(pts[1].y,max.y,pts[1].z,
				    max.z,levels[indx1+1]);
	      pts[2].z = levels[indx1 +1];

	      if (_X3D_convert_wc_to_dc(graphics, graphics->matrix, pts,
			points, 4))
	      {
	         indx = indx1+color_offset;
	         indx = _X3D_find_color(indx, colors, color_offset, num);
		 X3D_draw[graphics->device].fill_color(graphics, &colors[indx],
				TRUE);
		 X3D_draw[graphics->device].fill_polygon(graphics, points, 4,
				Convex, CoordModeOrigin);
	      }
	      pts[0] = pts[3];
	      pts[1] = pts[2];
	      indx1++;
	   }

	   if (pts[0].z < mid.z)
	   {
	      pts[2] = mid;
	      if (_X3D_convert_wc_to_dc(graphics, graphics->matrix, pts,
			points, 3))
	      {
	         indx = indx2+color_offset;
	         indx = _X3D_find_color(indx, colors, color_offset, num);
		 X3D_draw[graphics->device].fill_color(graphics, &colors[indx],
				TRUE);
	         X3D_draw[graphics->device].fill_polygon(graphics, points, 3,
			        Convex, CoordModeOrigin);
	      }
	   }
	   pts[0] = mid;

	   while (indx2 != indx3)
	   {
	      /*
	       *  Compute the end point for 
	       */
	      pts[3].x = kintercept(pts[0].x,max.x,pts[0].z,
				    max.z,levels[indx2+1]);
	      pts[3].y = kintercept(pts[0].y,max.y,pts[0].z,
				    max.z,levels[indx2+1]);
	      pts[3].z = levels[indx2 +1];

	      pts[2].x = kintercept(pts[1].x,max.x,pts[1].z,
				    max.z,levels[indx2+1]);
	      pts[2].y = kintercept(pts[1].y,max.y,pts[1].z,
				    max.z,levels[indx2+1]);
	      pts[2].z = levels[indx2 +1];

	      if (_X3D_convert_wc_to_dc(graphics, graphics->matrix, pts,
			points, 4))
	      {
	         indx = indx2+color_offset;
	         indx = _X3D_find_color(indx, colors, color_offset, num);
		 X3D_draw[graphics->device].fill_color(graphics, &colors[indx],
				TRUE);
		 X3D_draw[graphics->device].fill_polygon(graphics, points, 4,
				Convex, CoordModeOrigin);
	      }
	      pts[0] = pts[3];
	      pts[1] = pts[2];
	      indx2++;
	   }

	   /*
	    *  Draw the last part of the triangle in case the max
	    *  point is greater than indx3.
	    */
	   if (levels[indx3] < max.z && levels[indx3] < mid.z)
	   {
	      pts[2] = max;
	      pts[3] = mid;
	      if (_X3D_convert_wc_to_dc(graphics, graphics->matrix, pts,
			points, 4))
	      {
	         indx = indx3+color_offset;
	         indx = _X3D_find_color(indx, colors, color_offset, num);
		 X3D_draw[graphics->device].fill_color(graphics, &colors[indx],
				TRUE);
	         X3D_draw[graphics->device].fill_polygon(graphics, points, 4,
			        Convex, CoordModeOrigin);
	      }
	   }
	   else if (levels[indx3] < max.z)
	   {
	      pts[2] = max;
	      if (_X3D_convert_wc_to_dc(graphics, graphics->matrix, pts,
			points, 3))
	      {
	         indx = indx2+color_offset;
	         indx = _X3D_find_color(indx, colors, color_offset, num);
		 X3D_draw[graphics->device].fill_color(graphics, &colors[indx],
				TRUE);
	         X3D_draw[graphics->device].fill_polygon(graphics, points, 3,
			        Convex, CoordModeOrigin);
	      }
	   }
	}
}

/*-----------------------------------------------------------
|
|  Routine Name:
|
|       Purpose: _X3D_phong_normals
|
|         Input: graphics     -
|		 poly1	      -
|		 poly2	      -
|		 poly3	      -
|		 pt1	      -
|		 pt2	      -
|		 pt3	      -
|		 levels	      -
|		 colors	      -
|		 color_offset -
|		 num	      -
|		 n1	      -
|		 n2	      -
|		 n3	      -
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

static void _X3D_phong_normals(
   X3DGraphics *graphics,
   Coord       poly1,
   Coord       poly2,
   Coord       poly3,
   XPoint      pt1,
   XPoint      pt2,
   XPoint      pt3,
   Real        *levels,
   XColor      *colors,
   int         color_offset,
   int         num,
   Real        n1,
   Real        n2,
   Real        n3)
{
	Coord	 poly[4];
	XPoint   pts[4];
	int	 i, incr;
	register int indx, indx1, indx2, indx3;


	indx1 = n1 * num;
	indx2 = n2 * num;
	indx3 = n3 * num;

	/*
	 *  Make a quick check to see if the triangle is one color
	 */
	if (indx1 == indx2 && indx2 == indx3)
	{
	   pts[0] = pt1; pts[1] = pt2; pts[2] = pt3;
	   indx = _X3D_normal(graphics, &poly2, &poly1, &poly3) * num +
			color_offset;
	   indx = _X3D_find_color(indx, colors, color_offset, num);
	   X3D_draw[graphics->device].fill_color(graphics, &colors[indx], TRUE);
	   X3D_draw[graphics->device].fill_polygon(graphics, pts, 3,
				Convex, CoordModeOrigin);
	   return;
	}

	/*
	 *  Now that we have the three values we are going to interpolate
	 *  the data from each of the corners.
	 */

	/*
	 *  Interpolate from indx1 & indx2 to indx3
	 */
	if ((indx2 < indx1 && indx3 < indx1) ||
	    (indx2 > indx1 && indx3 > indx1))
	{
	   poly[0] = poly[1] = poly1;
	   incr = ((indx1 < indx3) ? 1 : -1);
	   for (i = indx1; i != indx3 && i != indx2; i+= incr)
	   {
	      poly[2].x = kintercept(poly1.x, poly2.x, indx1, indx2, i+incr);
	      poly[2].y = kintercept(poly1.y, poly2.y, indx1, indx2, i+incr);
	      poly[2].z = kintercept(poly1.z, poly2.z, indx1, indx2, i+incr);
	      poly[3].x = kintercept(poly1.x, poly3.x, indx1, indx3, i+incr);
	      poly[3].y = kintercept(poly1.y, poly3.y, indx1, indx3, i+incr);
	      poly[3].z = kintercept(poly1.z, poly3.z, indx1, indx3, i+incr);

	      if (_X3D_convert_wc_to_dc(graphics, graphics->matrix2, poly,
			pts, 4))
	      {
	         indx1 = _X3D_normal(graphics, &poly[2], &poly[0], &poly[3]) *
				num + color_offset;
	         indx2 = _X3D_normal(graphics, &poly[2], &poly[1], &poly[3]) *
				num + color_offset;
		 indx = (indx1 + indx2)/2;
	         indx = _X3D_find_color(indx, colors, color_offset, num);
	         X3D_draw[graphics->device].fill_color(graphics, &colors[indx],
				TRUE);
	         X3D_draw[graphics->device].fill_polygon(graphics, pts, 4,
				Convex, CoordModeOrigin);
	      }
	      poly[0] = poly[3];
	      poly[1] = poly[2];
	   }
	}

	/*
	 *  Interpolate from indx1 & indx3 to indx2
	 */
	if ((indx1 < indx3 && indx2 < indx3) ||
	    (indx1 > indx3 && indx2 > indx3))
	{
	   poly[0] = poly[1] = poly3;
	   incr = ((indx3 < indx2) ? 1 : -1);
	   for (i = indx3; i != indx2 && i != indx1; i += incr)
	   {
	      poly[2].x = kintercept(poly1.x, poly3.x, indx1, indx3, i+incr);
	      poly[2].y = kintercept(poly1.y, poly3.y, indx1, indx3, i+incr);
	      poly[2].z = kintercept(poly1.z, poly3.z, indx1, indx3, i+incr);
	      poly[3].x = kintercept(poly2.x, poly3.x, indx2, indx3, i+incr);
	      poly[3].y = kintercept(poly2.y, poly3.y, indx2, indx3, i+incr);
	      poly[3].z = kintercept(poly2.z, poly3.z, indx2, indx3, i+incr);

	      if (_X3D_convert_wc_to_dc(graphics, graphics->matrix2, poly,
			pts, 4))
	      {
	         indx1 = _X3D_normal(graphics, &poly[2], &poly[0], &poly[3]) *
				num + color_offset;
	         indx2 = _X3D_normal(graphics, &poly[2], &poly[1], &poly[3]) *
				num + color_offset;
		 indx = (indx1 + indx2)/2;
	         indx = _X3D_find_color(indx, colors, color_offset, num);
	         X3D_draw[graphics->device].fill_color(graphics, &colors[indx],
				TRUE);
	         X3D_draw[graphics->device].fill_polygon(graphics, pts, 4,
				Convex, CoordModeOrigin);
	      }
	      poly[0] = poly[3];
	      poly[1] = poly[2];
	   }
	}

	/*
	 *  Interpolate from indx2 & indx3 to indx1
	 */
	if ((indx1 < indx2 && indx3 < indx2) ||
	    (indx1 > indx2 && indx3 > indx2))
	{
	   poly[0] = poly[1] = poly2;
	   incr = ((indx2 < indx1) ? 1 : -1);
	   for (i = indx2; i != indx1 && i != indx3; i += incr)
	   {
	      poly[2].x = kintercept(poly2.x, poly3.x, indx2, indx3, i+incr);
	      poly[2].y = kintercept(poly2.y, poly3.y, indx2, indx3, i+incr);
	      poly[2].z = kintercept(poly2.z, poly3.z, indx2, indx3, i+incr);
	      poly[3].x = kintercept(poly1.x, poly2.x, indx1, indx2, i+incr);
	      poly[3].y = kintercept(poly1.y, poly2.y, indx1, indx2, i+incr);
	      poly[3].z = kintercept(poly1.z, poly2.z, indx1, indx2, i+incr);

	      if (_X3D_convert_wc_to_dc(graphics, graphics->matrix2, poly,
			pts, 4))
	      {
	         indx1 = _X3D_normal(graphics, &poly[2], &poly[0], &poly[3]) *
				num + color_offset;
	         indx2 = _X3D_normal(graphics, &poly[2], &poly[1], &poly[3]) *
				num + color_offset;
		 indx = (indx1 + indx2)/2;
	         indx = _X3D_find_color(indx, colors, color_offset, num);
	         X3D_draw[graphics->device].fill_color(graphics, &colors[indx],
				TRUE);
	         X3D_draw[graphics->device].fill_polygon(graphics, pts, 4,
				Convex, CoordModeOrigin);
	      }
	      poly[0] = poly[3];
	      poly[1] = poly[2];
	   }
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_shade_constant
|
|       Purpose: Draws a mesh plot using contour levels for
|		 the different color levels.  Surfaces are built
|		 exactly the same way that X3D_draw_mesh() does,
|		 accept the surfaces are left in 3D coordinate
|		 system.  The surface is then converted to device
|		 coordinates and the area is filled using the gc_fill
|		 color.  The surface is finally passed to the
|		 draw_colorline() routine, which draws the color
|		 border around the filled area.
|
|         Input: id	      -
|		 coords	      -
|		 row_size     -
|		 size	      -
|		 levels	      -
|		 colors	      -
|		 color_offset -
|		 num	      -
|		 wcmin	      -
|		 wcmax	      -
|		 shade_type   -
|
|        Output:
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X3D_shade_constant(
   int    id,
   Coord  *coords,
   int    row_size,
   int    size,
   Real   *levels,
   XColor *colors,
   int    color_offset,
   int    num,
   Coord  wcmin,
   Coord  wcmax,
   int    shade_type)
{
	Real		n[8];
	XPoint		pts[8];
        X3DGraphics     *graphics;
	Coord		polygon[4], ncoords[8];

	register int indx;
	int	 free_levels = FALSE;
	int	 corner, row, col, num_row, num_col, i, j, k, nsize;
	static ShadingRoutines shade[] =
	{
		NULL,
		_X3D_constant_imagery,
		_X3D_constant_elevations,
		_X3D_constant_normals,
	};


	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X3D_shade_imagery:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}

	num_col = row_size;
	num_row = size/num_col;
	row = num_row -1;
	col = num_col -1;

	/*  Compute which corner to start drawing from */
	corner = _X3D_find_corner(graphics, &coords[0], &coords[row_size-1],
			&coords[size-row_size], &coords[size-1]);

	if (levels == NULL && shade_type == KPLOT3D_SHADE_ELEVATION)
	{
	   free_levels = TRUE;
	   levels = X3D_get_contour_levels(wcmin.z, wcmax.z, 0.0, num);
	   if (levels == NULL)
	      return;
	}

	if (corner == BACKLEFT)
	{
	   for (i = 0; i < col; i++)
	   {
	      for (j = row; j > 0; j--)
	      {
		 indx = j * num_col + i;
		 polygon[0] = coords[indx];
		 polygon[1] = coords[++indx];

		 indx -= num_col;
		 polygon[2] = coords[indx];
		 polygon[3] = coords[--indx];

		 if (!_X3D_clip_polygon(graphics, polygon, 4, ncoords, &nsize))
		    continue;

		 if (!(_X3D_convert_wc_to_dc(graphics, graphics->matrix2,
			ncoords, pts, nsize)))
		    continue;

		 if (shade_type == KPLOT3D_SHADE_ELEVATION)
		 {
		    n[0] = n[4] = _X3D_find_level(polygon[0].z, levels, num);
		    n[1] = n[5] = _X3D_find_level(polygon[1].z, levels, num);
		    n[2] = n[6] = _X3D_find_level(polygon[2].z, levels, num);
		    n[3] = n[7] = _X3D_find_level(polygon[3].z, levels, num);
		 }

	         for (k = 1; k < nsize-1; k++)
		 {
		    shade[shade_type].routine(graphics, ncoords[0], ncoords[k],
			ncoords[k+1], pts[0], pts[k], pts[k+1], levels, colors,
			color_offset, num, n[0], n[k], n[k+1]);
		 }
	      }
	   }
	}
	else if (corner == BACKRIGHT)
	{
	   for (j = row; j > 0; j--)
	   {
	      for (i = col; i > 0; i--)
	      {
	         indx = j * num_col + i;
		 polygon[0] = coords[indx];
		 polygon[1] = coords[--indx];

		 indx -= num_col;
		 polygon[2] = coords[indx];
		 polygon[3] = coords[++indx];

		 if (!_X3D_clip_polygon(graphics, polygon, 4, ncoords, &nsize))
		    continue;

		 if (!(_X3D_convert_wc_to_dc(graphics, graphics->matrix2,
			ncoords, pts, nsize)))
		    continue;

		 if (shade_type == KPLOT3D_SHADE_ELEVATION)
		 {
		    n[0] = n[4] = _X3D_find_level(polygon[0].z, levels, num);
		    n[1] = n[5] = _X3D_find_level(polygon[1].z, levels, num);
		    n[2] = n[6] = _X3D_find_level(polygon[2].z, levels, num);
		    n[3] = n[7] = _X3D_find_level(polygon[3].z, levels, num);
		 }

	         for (k = 1; k < nsize-1; k++)
		 {
		    shade[shade_type].routine(graphics, ncoords[0], ncoords[k],
			ncoords[k+1], pts[0], pts[k], pts[k+1], levels, colors,
			color_offset, num, n[0], n[k], n[k+1]);
		 }
	      }
	   }
	}
	else if (corner == FRONTRIGHT)
	{
	   for (i = col; i > 0; i--)
	   {
	      for (j = 0; j < row; j++)
	      {
	         indx = j * num_col + i;
		 polygon[0] = coords[indx];
		 polygon[1] = coords[--indx];

		 indx += num_col;
		 polygon[2] = coords[indx];
		 polygon[3] = coords[++indx];

		 if (!_X3D_clip_polygon(graphics, polygon, 4, ncoords, &nsize))
		    continue;

		 if (!(_X3D_convert_wc_to_dc(graphics, graphics->matrix2,
			ncoords, pts, nsize)))
		    continue;

		 if (shade_type == KPLOT3D_SHADE_ELEVATION)
		 {
		    n[0] = n[4] = _X3D_find_level(polygon[0].z, levels, num);
		    n[1] = n[5] = _X3D_find_level(polygon[1].z, levels, num);
		    n[2] = n[6] = _X3D_find_level(polygon[2].z, levels, num);
		    n[3] = n[7] = _X3D_find_level(polygon[3].z, levels, num);
		 }

	         for (k = 1; k < nsize-1; k++)
		 {
		    shade[shade_type].routine(graphics, ncoords[0], ncoords[k],
			ncoords[k+1], pts[0], pts[k], pts[k+1], levels, colors,
			color_offset, num, n[0], n[k], n[k+1]);
		 }
	      }
	   }
	}
	else if (corner == FRONTLEFT)
	{
	   for (j = 0; j < row; j++)
	   {
	      for (i = 0; i < col; i++)
	      {
	         indx = j * num_col + i;
		 polygon[0] = coords[indx];
		 polygon[1] = coords[++indx];

		 indx += num_col;
		 polygon[2] = coords[indx];
		 polygon[3] = coords[--indx];

		 if (!_X3D_clip_polygon(graphics, polygon, 4, ncoords, &nsize))
		    continue;

		 if (!(_X3D_convert_wc_to_dc(graphics, graphics->matrix2,
			ncoords, pts, nsize)))
		    continue;

		 if (shade_type == KPLOT3D_SHADE_ELEVATION)
		 {
		    n[0] = n[4] = _X3D_find_level(polygon[0].z, levels, num);
		    n[1] = n[5] = _X3D_find_level(polygon[1].z, levels, num);
		    n[2] = n[6] = _X3D_find_level(polygon[2].z, levels, num);
		    n[3] = n[7] = _X3D_find_level(polygon[3].z, levels, num);
		 }

	         for (k = 1; k < nsize-1; k++)
		 {
		    shade[shade_type].routine(graphics, ncoords[0], ncoords[k],
			ncoords[k+1], pts[0], pts[k], pts[k+1], levels, colors,
			color_offset, num, n[0], n[k], n[k+1]);
		 }
	      }
	   }
	}

	/*
	 *  Cleanup time.
	 */
	if (free_levels) kfree(levels);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_shade_ghouraud
|
|       Purpose:
|
|         Input: id	      -
|		 coords	      -
|		 row_size     -
|		 size	      -
|		 levels	      -
|		 colors	      -
|		 color_offset -
|		 num	      -
|		 wcmin	      -
|		 wcmax	      -
|		 shade_type   -
|
|        Output:
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X3D_shade_ghouraud(
   int    id,
   Coord  *coords,
   int    row_size,
   int    size,
   Real   *levels,
   XColor *colors,
   int    color_offset,
   int    num,
   Coord  wcmin,
   Coord  wcmax,
   int    shade_type)
{
	Real		n[8];
	XPoint		pts[8];
        X3DGraphics     *graphics;
	Coord		polygon[4], ncoords[8];

	register int indx;
	int	 free_levels = FALSE;
	int	 corner, row, col, num_row, num_col, i, j, k, nsize;
	static ShadingRoutines shade[] =
	{
                NULL,
		_X3D_ghouraud_imagery,
		_X3D_ghouraud_elevations,
		_X3D_ghouraud_normals,
	};



	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X3D_shade_elevation:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}

	num_col = row_size;
	num_row = size/num_col;
	row = num_row -1;
	col = num_col -1;

	/*  Compute which corner to start drawing from */
	corner = _X3D_find_corner(graphics, &coords[0], &coords[row_size-1],
			&coords[size-row_size], &coords[size-1]);

	if (levels == NULL && shade_type == KPLOT3D_SHADE_ELEVATION)
	{
	   free_levels = TRUE;
	   levels = X3D_get_contour_levels(wcmin.z, wcmax.z, 0.0, num);
	   if (levels == NULL)
	      return;
	}


	if (corner == BACKLEFT)
	{
	   for (i = 0; i < col; i++)
	   {
	      for (j = row; j > 0; j--)
	      {
		 indx = j * num_col + i;
		 polygon[0] = coords[indx];
		 polygon[1] = coords[++indx];

		 indx -= num_col;
		 polygon[2] = coords[indx];
		 polygon[3] = coords[--indx];

		 if (!_X3D_clip_polygon(graphics, polygon, 4, ncoords, &nsize))
		    continue;

		 if (!(_X3D_convert_wc_to_dc(graphics, graphics->matrix2,
			ncoords, pts, nsize)))
		    continue;

		 if (shade_type == KPLOT3D_SHADE_NORMAL)
		 {
		    n[0] = n[4] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i, j);
		    n[1] = n[5] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i+1, j);
		    n[2] = n[6] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i+1, j-1);
		    n[3] = n[7] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i, j-1);
		 }

		 for (k = 1; k < nsize-1; k++)
		 {
		    shade[shade_type].routine(graphics, ncoords[0], ncoords[k],
			ncoords[k+1], pts[0], pts[k], pts[k+1], levels, colors,
			color_offset, num, n[0], n[k], n[k+1]);
		 }
	      }
	   }
	}
	else if (corner == BACKRIGHT)
	{
	   for (j = row; j > 0; j--)
	   {
	      for (i = col; i > 0; i--)
	      {
	         indx = j * num_col + i;
		 polygon[0] = coords[indx];
		 polygon[1] = coords[--indx];

		 indx -= num_col;
		 polygon[2] = coords[indx];
		 polygon[3] = coords[++indx];

		 if (!_X3D_clip_polygon(graphics, polygon, 4, ncoords, &nsize))
		    continue;

		 if (!(_X3D_convert_wc_to_dc(graphics, graphics->matrix2,
			ncoords, pts, nsize)))
		    continue;

		 if (shade_type == KPLOT3D_SHADE_NORMAL)
		 {
		    n[0] = n[4] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i, j);
		    n[1] = n[5] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i-1, j);
		    n[2] = n[6] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i-1, j-1);
		    n[3] = n[7] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i, j-1);
		 }

		 for (k = 1; k < nsize-1; k++)
		 {
		    shade[shade_type].routine(graphics, ncoords[0], ncoords[k],
			ncoords[k+1], pts[0], pts[k], pts[k+1], levels, colors,
			color_offset, num, n[0], n[k], n[k+1]);
		 }
	      }
	   }
	}
	else if (corner == FRONTRIGHT)
	{
	   for (i = col; i > 0; i--)
	   {
	      for (j = 0; j < row; j++)
	      {
	         indx = j * num_col + i;
		 polygon[0] = coords[indx];
		 polygon[1] = coords[--indx];

		 indx += num_col;
		 polygon[2] = coords[indx];
		 polygon[3] = coords[++indx];

		 if (!_X3D_clip_polygon(graphics, polygon, 4, ncoords, &nsize))
		    continue;

		 if (!(_X3D_convert_wc_to_dc(graphics, graphics->matrix2,
			ncoords, pts, nsize)))
		    continue;

		 if (shade_type == KPLOT3D_SHADE_NORMAL)
		 {
		    n[0] = n[4] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i, j);
		    n[1] = n[5] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i-1, j);
		    n[2] = n[6] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i-1, j+1);
		    n[3] = n[7] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i, j+1);
		 }

		 for (k = 1; k < nsize-1; k++)
		 {
		    shade[shade_type].routine(graphics, ncoords[0], ncoords[k],
			ncoords[k+1], pts[0], pts[k], pts[k+1], levels, colors,
			color_offset, num, n[0], n[k], n[k+1]);
		 }
	      }
	   }
	}
	else if (corner == FRONTLEFT)
	{
	   for (j = 0; j < row; j++)
	   {
	      for (i = 0; i < col; i++)
	      {
	         indx = j * num_col + i;
		 polygon[0] = coords[indx];
		 polygon[1] = coords[++indx];

		 indx += num_col;
		 polygon[2] = coords[indx];
		 polygon[3] = coords[--indx];

		 if (!_X3D_clip_polygon(graphics, polygon, 4, ncoords, &nsize))
		    continue;

		 if (!(_X3D_convert_wc_to_dc(graphics, graphics->matrix2,
			ncoords, pts, nsize)))
		    continue;

		 if (shade_type == KPLOT3D_SHADE_NORMAL)
		 {
		    n[0] = n[4] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i, j);
		    n[1] = n[5] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i+1, j);
		    n[2] = n[6] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i+1, j+1);
		    n[3] = n[7] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i, j+1);
		 }

		 for (k = 1; k < nsize-1; k++)
		 {
		    shade[shade_type].routine(graphics, ncoords[0], ncoords[k],
			ncoords[k+1], pts[0], pts[k], pts[k+1], levels, colors,
			color_offset, num, n[0], n[k], n[k+1]);
		 }
	      }
	   }
	}

	/*
	 *  Cleanup time.
	 */
	if (free_levels) kfree(levels);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_shade_phong
|
|       Purpose:
|
|         Input: id	      -
|		 coords	      -
|		 row_size     -
|		 size	      -
|		 levels	      -
|		 colors	      -
|		 color_offset -
|		 num	      -
|		 wcmin	      -
|		 wcmax	      -
|		 shade_type   -
|
|        Output:
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X3D_shade_phong(
   int    id,
   Coord  *coords,
   int    row_size,
   int    size,
   Real   *levels,
   XColor *colors,
   int    color_offset,
   int    num,
   Coord  wcmin,
   Coord  wcmax,
   int    shade_type)
{
	Real		n[8];
	XPoint		pts[8];
        X3DGraphics     *graphics;
	Coord		polygon[4], ncoords[8];

	register int indx;
	int	 free_levels = FALSE;
	int	 corner, row, col, num_row, num_col, i, j, k, nsize;
	static ShadingRoutines shade[] =
	{
		NULL,
		_X3D_phong_imagery,
		_X3D_phong_elevations,
		_X3D_phong_normals,
	};


	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X3D_shade_normals:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}

	num_col = row_size;
	num_row = size/num_col;
	row = num_row -1;
	col = num_col -1;

	/*  Compute which corner to start drawing from */
	corner = _X3D_find_corner(graphics, &coords[0], &coords[row_size-1],
			&coords[size-row_size], &coords[size-1]);

	if (levels == NULL && shade_type == KPLOT3D_SHADE_ELEVATION)
	{
	   free_levels = TRUE;
	   levels = X3D_get_contour_levels(wcmin.z, wcmax.z, 0.0, num);
	   if (levels == NULL)
	      return;
	}

	if (corner == BACKLEFT)
	{
	   for (i = 0; i < col; i++)
	   {
	      for (j = row; j > 0; j--)
	      {
		 indx = j * num_col + i;
		 polygon[0] = coords[indx];
		 polygon[1] = coords[++indx];

		 indx -= num_col;
		 polygon[2] = coords[indx];
		 polygon[3] = coords[--indx];

		 if (!_X3D_clip_polygon(graphics, polygon, 4, ncoords, &nsize))
		    continue;

		 if (!(_X3D_convert_wc_to_dc(graphics, graphics->matrix2,
			ncoords, pts, nsize)))
		    continue;

		 if (shade_type == KPLOT3D_SHADE_NORMAL)
		 {
		    n[0] = n[4] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i, j);
		    n[1] = n[5] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i+1, j);
		    n[2] = n[6] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i+1, j-1);
		    n[3] = n[7] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i, j-1);
		 }

		 for (k = 1; k < nsize-1; k++)
		 {
		    shade[shade_type].routine(graphics, ncoords[0], ncoords[k],
			ncoords[k+1], pts[0], pts[k], pts[k+1], levels, colors,
			color_offset, num, n[0], n[k], n[k+1]);
		 }
	      }
	   }
	}
	else if (corner == BACKRIGHT)
	{
	   for (j = row; j > 0; j--)
	   {
	      for (i = col; i > 0; i--)
	      {
	         indx = j * num_col + i;
		 polygon[0] = coords[indx];
		 polygon[1] = coords[--indx];

		 indx -= num_col;
		 polygon[2] = coords[indx];
		 polygon[3] = coords[++indx];

		 if (!_X3D_clip_polygon(graphics, polygon, 4, ncoords, &nsize))
		    continue;

		 if (!(_X3D_convert_wc_to_dc(graphics, graphics->matrix2,
			ncoords, pts, nsize)))
		    continue;

		 if (shade_type == KPLOT3D_SHADE_NORMAL)
		 {
		    n[0] = n[4] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i, j);
		    n[1] = n[5] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i-1, j);
		    n[2] = n[6] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i-1, j-1);
		    n[3] = n[7] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i, j-1);
		 }

		 for (k = 1; k < nsize-1; k++)
		 {
		    shade[shade_type].routine(graphics, ncoords[0], ncoords[k],
			ncoords[k+1], pts[0], pts[k], pts[k+1], levels, colors,
			color_offset, num, n[0], n[k], n[k+1]);
		 }
	      }
	   }
	}
	else if (corner == FRONTRIGHT)
	{
	   for (i = col; i > 0; i--)
	   {
	      for (j = 0; j < row; j++)
	      {
	         indx = j * num_col + i;
		 polygon[0] = coords[indx];
		 polygon[1] = coords[--indx];

		 indx += num_col;
		 polygon[2] = coords[indx];
		 polygon[3] = coords[++indx];

		 if (!_X3D_clip_polygon(graphics, polygon, 4, ncoords, &nsize))
		    continue;

		 if (!(_X3D_convert_wc_to_dc(graphics, graphics->matrix2,
			ncoords, pts, nsize)))
		    continue;

		 if (shade_type == KPLOT3D_SHADE_NORMAL)
		 {
		    n[0] = n[4] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i, j);
		    n[1] = n[5] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i-1, j);
		    n[2] = n[6] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i-1, j+1);
		    n[3] = n[7] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i, j+1);
		 }

		 for (k = 1; k < nsize-1; k++)
		 {
		    shade[shade_type].routine(graphics, ncoords[0], ncoords[k],
			ncoords[k+1], pts[0], pts[k], pts[k+1], levels, colors,
			color_offset, num, n[0], n[k], n[k+1]);
		 }
	      }
	   }
	}
	else if (corner == FRONTLEFT)
	{
	   for (j = 0; j < row; j++)
	   {
	      for (i = 0; i < col; i++)
	      {
	         indx = j * num_col + i;
		 polygon[0] = coords[indx];
		 polygon[1] = coords[++indx];

		 indx += num_col;
		 polygon[2] = coords[indx];
		 polygon[3] = coords[--indx];

		 if (!_X3D_clip_polygon(graphics, polygon, 4, ncoords, &nsize))
		    continue;

		 if (!(_X3D_convert_wc_to_dc(graphics, graphics->matrix2,
			ncoords, pts, nsize)))
		    continue;

		 if (shade_type == KPLOT3D_SHADE_NORMAL)
		 {
		    n[0] = n[4] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i, j);
		    n[1] = n[5] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i+1, j);
		    n[2] = n[6] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i+1, j+1);
		    n[3] = n[7] = _X3D_vertex_normal(graphics, coords, num_col,
					num_row, i, j+1);
		 }

		 for (k = 1; k < nsize-1; k++)
		 {
		    shade[shade_type].routine(graphics, ncoords[0], ncoords[k],
			ncoords[k+1], pts[0], pts[k], pts[k+1], levels, colors,
			color_offset, num, n[0], n[k], n[k+1]);
		 }
	      }
	   }
	}

	/*
	 *  Cleanup time.
	 */
	if (free_levels) kfree(levels);
}
