/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>              2D Matrix  Utilities
   >>>>
   >>>>  Private:
   >>>>              _X2D_clear_matrix()
   >>>>              _X2D_matrix_set_identity()
   >>>>              _X2D_matrix_mult()
   >>>>              _X2D_matrix_set_ndc()
   >>>>              _X2D_write_matrix()
   >>>>              _X2D_matrix_set_dc()
   >>>>              X2D_matrix_set_translate()
   >>>>              X2D_matrix_set_scale()
   >>>>              X2D_matrix_set_rotate()
   >>>>              X2D_matrix_set_inverse()
   >>>>   Static:
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "graphics.h"


/*-----------------------------------------------------------
|
|  Routine Name: _X2D_clear_matrix
|
|       Purpose: Set a 3x3 matrix to ALL zeros
|
|         Input: matrix -- a 3x3 Real array
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void _X2D_clear_matrix (
   Matrix_2D matrix)
{
	int i,j;

	for (i = 0; i < 3; i++)
	{
	    for (j = 0; j < 3; j++)
	    {
		matrix[i][j] = 0.0;
	    }
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: _X2D_matrix_set_identity
|
|       Purpose: Set a 3x3 matrix to identity matrix.
|
|         Input: matrix -- a 3x3 Real array
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void _X2D_matrix_set_identity (
   Matrix_2D matrix)
{
	_X2D_clear_matrix(matrix);
	matrix[0][0] = 
	matrix[1][1] = 
	matrix[2][2] = 1.0;
}

/*-----------------------------------------------------------
|
|  Routine Name: _X2D_matrix_mult
|
|       Purpose: multiplies 2 3x3 matrices
|
|         Input: matrix1   -
|		 matrix2   -
|
|        Output: outmatrix - resulting matrix
|
|       Returns: noting
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void _X2D_matrix_mult (
   Matrix_2D matrix1,
   Matrix_2D matrix2,
   Matrix_2D outmatrix)
{
	int   i, j;
	Matrix_2D tmp_matrix;

	/*
	 *  Multiply the two matrices together and return the result in
	 *  the out matrix, "outmatrix".
	 */
	for (i = 0; i < 3; i++)
	{
	    for (j = 0; j < 3; j++)
	    {
	        tmp_matrix[i][j]  = matrix1[i][0] * matrix2[0][j] +
	      			    matrix1[i][1] * matrix2[1][j] +
	      			    matrix1[i][2] * matrix2[2][j];
	    }
	}

	/*
	 *  Copy the temporary matrix to the final matrix (outmatrix).  We 
	 *  could have put this directly into the final matrix but this enables
	 *  the user to use the output matrix as either matrix1 or matrix2.
	 *  ie)
	 *      matrix1 = matrix1 * matrix2
	 */
        kmemcpy( (char *) outmatrix, (char *) tmp_matrix, sizeof(Matrix_2D));
}

/*-----------------------------------------------------------
|
|  Routine Name: _X2D_matrix_set_ndc
|
|       Purpose: Sets a transform matrix which will map world
|		 coordinates to normalized device coordinates.
|
|		 Two Dimensional Transform Matrix:
|
|		 ----       ---
|		 | Sx   0    0|
|		 |  0   Sy   0|
|		 | Ax   Ay   1|
|		 ----       ---
|
|         Input: scale	-
|		 xv_min	-
|		 xv_max	-
|		 yv_min	-
|		 yv_max	-
|		 wc_min	-
|		 wc_max	-
|
|        Output: matrix	- resulting 3x3 transformation matrix
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void _X2D_matrix_set_ndc (
   Coord     scale,
   Real      xv_min,
   Real      xv_max,
   Real      yv_min,
   Real      yv_max,
   Coord     wc_min,
   Coord     wc_max,
   Matrix_2D matrix)
{
	Real	scale_x, scale_y, offset_x, offset_y;
        Coord	smin, smax;

	if (!(_X2D_scale_wc_coord(scale, wc_min, &smin)))
	   smin = wc_min;

	if (!(_X2D_scale_wc_coord(scale, wc_max, &smax)))
	   smax = wc_max;

	scale_x = (xv_max - xv_min)/(smax.x - smin.x);
	scale_y = (yv_max - yv_min)/(smax.y - smin.y);

	offset_x = xv_min - (smin.x * scale_x);
	offset_y = yv_min - (smin.y * scale_y);

	/*
	 *  Now set the 2D ndc matrix  
	 */
	_X2D_matrix_set_identity(matrix);

	matrix[0][0] = scale_x;
	matrix[1][1] = scale_y;

	matrix[2][0] = offset_x;
	matrix[2][1] = offset_y;
}

/*-----------------------------------------------------------
|
|  Routine Name: _X2D_write_matrix
|
|       Purpose: Prints out a Matrix.  This is mostly used by
|		 for debugging purposes.
|
|         Input: matrix --  The 3x3 matrix to be printed.
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void _X2D_write_matrix (
   Matrix_2D matrix)
{
     int i,j;

     kprintf ("\n");
     for (i = 0; i < 3; i++)
     {
         for (j = 0; j < 3; j++)
	 {
	     kprintf (" %7.3f ",matrix[i][j]);
	 }
	 kprintf ("\n");
     }
}

/*-----------------------------------------------------------
|
|  Routine Name: _X2D_matrix_set_dc
|
|       Purpose: This routine sets a transform matrix which will map
|		 world coordinates to device coordinates.
|
|         Input: graphics - X3D graphics structure
|
|        Output: matrix	  - a two dimensional (3x3) Real array
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void _X2D_matrix_set_dc (
   X3DGraphics *graphics,
   Matrix_2D   matrix)
{
	/*
	 *  Now set the 2D dc matrix  
	 */
	_X2D_matrix_set_identity(matrix);

	if (graphics->device == X11)
	{
	   matrix[0][0] = graphics->X11_xmax - graphics->X11_xmin;
	   matrix[1][1] = graphics->X11_ymin - graphics->X11_ymax;
	   matrix[2][0] = graphics->X11_xmin;
	   matrix[2][1] = graphics->X11_ymax;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: X2D_matrix_set_translate
|
|       Purpose: Set a translation matrix in order to translate
|		 world coordinates to new world coordinates.
|
|		 Two Dimensional Translation Matrix:
|
|		 ----        ---
|		 | 1    0    0 |
|		 | 0    1    0 |
|		 | Tx   Ty   1 |
|		 ----        ---
|
|         Input: xtrans - a Real containing the x distance to be translated
|		 ytrans - a Real containing the y distance to be translated
|
|        Output: matrix - a two dimensional (3x3) Real array
|			   containing the scaling matrix.
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X2D_matrix_set_translate (
   Real      xtrans,
   Real      ytrans,
   Matrix_2D matrix)
{
        /*
         *  Set up the translation matrix --
         *  these entries will not change
         */
	_X2D_matrix_set_identity(matrix);

        /*
         *  Values in the active 3rd column of the
         *  translation matrix have been provided by the user.
         */
        matrix[2][0] = xtrans; matrix[2][1] = ytrans;
}

/*-----------------------------------------------------------
|
|  Routine Name: X2D_matrix_set_scale
|
|       Purpose: Set a scale matrix in order to scale world
|		 coordinates to new world coordinates.
|
|		 Two Dimensional Scaling Matrix
|
|		 ----        ---
|		 | Sx   0    0 |
|		 |  0   Sy   0 |
|		 |  0   0    1 |
|		 ----        ---
|
|         Input: xscale - a Real containing the x distance to be translated
|		 yscale - a Real containing the y distance to be translated
|
|        Output: matrix - a two dimensional (3x3) float array
|			  containing the scaling matrix.
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X2D_matrix_set_scale (
   Real      xscale,
   Real      yscale,
   Matrix_2D matrix)
{


        /*
         *  set identity matrix
         */
	_X2D_matrix_set_identity(matrix);

        /*
         *  the active entries are on the diagonal, providing the
         *  scaling factor requested by the user
         */
        matrix[0][0] = xscale;
	matrix[1][1] = yscale;
}

/*-----------------------------------------------------------
|
|  Routine Name: X2D_matrix_set_rotate
|
|       Purpose: Set a rotation matrix in order to rotate world
|		 coordinates to new world coordinates.
|
|		 Two Dimensional Rotation Matrix:
|
|		 ----            ---
|		 | cos@  sin@    0 |
|		 |-sin@  cos@    0 |
|		 |  0    0       1 |
|		 ----            ---
|
|         Input: rotation_degree - a Real containing the degree of
|				   rotation
|
|        Output: matrix - a two dimensional (3x3) Real array
|			  containing the rotation matrix.
|
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X2D_matrix_set_rotate (
   Real      rotation_degree,
   Matrix_2D matrix)
{
        Real  rad;

        /*
         *  rotate the x, y, and z arrays & reset world coordinates
         */
        rad = kdegrees_radians(rotation_degree);

        /*
         *  initialize the matrix to the identity matrix, for
         *  the sake of simplicity -- some values will be over-
         *  written.
         */
	_X2D_matrix_set_identity(matrix);

        /*
         *  set up the rotation matrix as described above --
         *  with cos@ and sin@ in the proper place.
         */
	matrix[0][0] =
	matrix[1][1] = cos(rad);
	matrix[0][1] = -sin(rad);
	matrix[1][0] = sin(rad);
}

/*-----------------------------------------------------------
|
|  Routine Name: X2D_matrix_set_inverse
|
|       Purpose: 
|
|         Input: graphics - X3D graphics structure
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

int X2D_matrix_set_inverse(
   Matrix_2D matrix,
   Matrix_2D imatrix)
{
	Matrix_2D tmp;
	Real	  det;


	det =   matrix[0][0]*matrix[1][2]*matrix[2][1]
	      - matrix[0][0]*matrix[1][1]*matrix[2][2]
	      + matrix[1][0]*matrix[0][1]*matrix[2][2]
	      - matrix[1][0]*matrix[0][2]*matrix[2][1]
	      - matrix[2][0]*matrix[0][2]*matrix[1][2]
	      + matrix[2][0]*matrix[0][2]*matrix[1][1];
	if (det <= 0.0)
	{
	   return(FALSE);
	}

	tmp[0][0] = (matrix[1][2]*matrix[2][1] - matrix[1][1]*matrix[2][2])/det;
	tmp[0][1] = (matrix[0][1]*matrix[2][2] - matrix[0][2]*matrix[2][1])/det;
	tmp[0][2] = (matrix[0][2]*matrix[1][1] - matrix[0][1]*matrix[1][2])/det;

	tmp[1][0] = (matrix[1][0]*matrix[2][2] - matrix[1][2]*matrix[2][0])/det;
	tmp[1][1] = (matrix[0][2]*matrix[2][0] - matrix[0][0]*matrix[2][2])/det;
	tmp[1][2] = (matrix[0][0]*matrix[1][2] - matrix[0][2]*matrix[1][0])/det;

	tmp[2][0] = (matrix[1][1]*matrix[2][0] - matrix[1][0]*matrix[2][1])/det;
	tmp[2][1] = (matrix[0][0]*matrix[2][1] - matrix[0][1]*matrix[2][0])/det;
	tmp[2][2] = (matrix[0][1]*matrix[1][0] - matrix[0][0]*matrix[1][1])/det;

	kmemcpy(imatrix, tmp, sizeof(Matrix_2D));
	return(TRUE);
}
