/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>   		3D Viewing System Control Utilities
   >>>>
   >>>>  Private:
   >>>>               _X3D_get_graphics()
   >>>>               _X3D_update_graphics()
   >>>>               X3D_init_graphics()
   >>>>               X3D_set_projection()
   >>>>               X3D_close_graphics()
   >>>>               X3D_set_viewpoint()
   >>>>               X3D_set_view_distance()
   >>>>               X3D_set_focus()
   >>>>               X3D_set_camera()
   >>>>               X3D_set_window()
   >>>>               X3D_set_viewport()
   >>>>               X3D_set_wc_scale()
   >>>>               X3D_set_line_type()
   >>>>               X3D_set_line_width()
   >>>>               X3D_set_clipping()
   >>>>               X3D_set_resize_view()
   >>>>               X3D_set_clipping_volume()
   >>>>   Static:
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "graphics.h"	

#define OldViewing TRUE

static X3DList *graphics_head = NULL;

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_get_graphics
|
|       Purpose: Returns the X3D structure associated with a
|		 particular graphics ID
|
|         Input: id - X3D graphics structure ID
|       Returns: graphics structure on success, NULL otherwise
|    Written By: Mark Young
|          Date: 
|
------------------------------------------------------------*/

X3DGraphics *_X3D_get_graphics (
   int id)
{
	X3DList		*current;

	/*
	 *  Search through the linked graphics list for the desired graphics
	 *  id.
	 */
	if (graphics_head == NULL)
	   return(NULL);
	else if (graphics_head->id == id)
	   return(graphics_head->graphics);

	current = graphics_head;
	while (current->next != NULL && current->id != id)
	      current = current->next;

	if (current->id == id)
	{
	   /*
	    *  Stick the current graphics structure at the top of the list, 
	    *  since it will probably be used again soon.
	    */
	   current->prev->next = current->next;
	   if (current->next != NULL)
	   {
	      current->next->prev = current->prev;
	   }

	   current->next = graphics_head;
	   current->prev = NULL;

	   graphics_head->prev = current;
	   graphics_head = current;

	   return(current->graphics);
	}
	else
	   return(NULL);
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_update_graphics
|
|       Purpose: Recomputes the X3D Grapichs Matrix, used to
|		 transform 3D world coordinates into device
|		 coordinates.
|
|         Input: graphics - X3D graphics structure
|       Returns: nothing
|    Written By: Mark Young
|          Date: 
|
------------------------------------------------------------*/

void _X3D_update_graphics (
   X3DGraphics *graphics)
{
	int     i, j;
	double  *old_values, *new_values;
	Matrix	matrix1, matrix2;
	Coord	wcmin, wcmax, side;
	Real	x, y, z, distance, xmin, xmax, ymin, ymax, zmin,
		zmax, dummy, z0, z1, z2, z3, percentage, dx, dy, dz,
		center_x, center_y, center_z;
#if !OldViewing
	Real    view_dist;
	Coord   camera, focus, up;
#endif


	/*
 	 * If the field of view is to be resized then change the wcmin and
	 * wcmax to fit the view
 	 */
	wcmin = graphics->wc_min;
	wcmax = graphics->wc_max;
	if (graphics->resize_view)
	{
	   x = graphics->focus.x - graphics->camera.x;
	   y = graphics->focus.y - graphics->camera.y;
	   z = graphics->focus.z - graphics->camera.z;
	   distance = X3D_distance(&graphics->camera, &graphics->focus);

	   dx = kabs(x/(wcmax.x - wcmin.x));
	   dy = kabs(y/(wcmax.y - wcmin.y));
	   dz = kabs(z/(wcmax.z - wcmin.z));
	   percentage = kmax3(dx, dy, dz);


	   xmin = graphics->camera.x;
	   ymin = graphics->camera.y;
	   zmin = graphics->camera.z;
	   if (x < 0)
	      xmax = graphics->camera.x - (wcmax.x - wcmin.x)*percentage;
	   else
	      xmax = graphics->camera.x + (wcmax.x - wcmin.x)*percentage;

	   if (y < 0)
	      ymax = graphics->camera.y - (wcmax.y - wcmin.y)*percentage;
	   else
	      ymax = graphics->camera.y + (wcmax.y - wcmin.y)*percentage;

	   if (z < 0)
	      zmax = graphics->camera.z - (wcmax.z - wcmin.z)*percentage;
	   else
	      zmax = graphics->camera.z + (wcmax.z - wcmin.z)*percentage;

	   wcmin.x = kmin(xmin, xmax);
	   wcmin.y = kmin(ymin, ymax);
	   wcmin.z = kmin(zmin, zmax);
	   wcmax.x = kmax(xmin, xmax);
	   wcmax.y = kmax(ymin, ymax);
	   wcmax.z = kmax(zmin, zmax);
	}

	/*
 	 * Convert to unit cube.
 	 */
#if OldViewing
	_X3D_matrix_set_ndc(graphics->scale, wcmin, wcmax, matrix1);
	center_x = center_y = center_z = 0.5;
	X3D_matrix_set_translate(-center_x, -center_y, -center_z, matrix2);
	_X3D_matrix_mult(matrix1, matrix2, matrix1);
	_X3D_matrix_set_oldviewing(graphics->theta, graphics->alpha,
                               graphics->eye_dist, matrix2);
	_X3D_matrix_mult(matrix1, matrix2, graphics->matrix1);

        /*
         * Foley and Van Dam Style
         */
        if (graphics->projection == KGRAPHICS_PERSPECTIVE)
        {
           /*
            *  Set the Perspective transformation
            */
           _X3D_matrix_set_perspective(graphics->view_dist, matrix1);
        }
        else
        {
           /*
            *  Set the Parallel transformation
            */
           _X3D_matrix_set_parallel(graphics->projection, matrix1);
        }

	/*
         *  Set the translation to the origin
         */
        X3D_matrix_set_translate(0.5, 0.5, 0.0, matrix2);
        _X3D_matrix_mult(matrix1, matrix2, matrix1);

#else
	_X3D_matrix_set_ndc(graphics->scale, wcmin, wcmax, matrix1);
	_X3D_vector_mult(graphics->camera, matrix1, camera.x, camera.y,
			camera.z, dummy);
	_X3D_vector_mult(graphics->focus, matrix1, focus.x, focus.y,
			focus.z, dummy);
	_X3D_vector_mult(graphics->up, matrix1, up.x, up.y, up.z, dummy);

	_X3D_matrix_set_viewing(camera, focus, up, matrix2);
	_X3D_matrix_mult(matrix1, matrix2, matrix1);
	X3D_matrix_set_translate(0.5, 0.5, 0.5, matrix2);
	_X3D_matrix_mult(matrix1, matrix2, graphics->matrix1);

	/*
	 * Foley and Van Dam Style
	 */
	if (graphics->projection == KGRAPHICS_PERSPECTIVE)
	{
	   /*
	    *  Set the PERSPECTIVE transformation
	    */
	   distance = X3D_distance(&graphics->camera, &graphics->focus);
	   if (distance != 0.0)
	      view_dist = graphics->view_dist/distance;
	   else
	      view_dist = graphics->view_dist;

	   _X3D_matrix_set_perspective(view_dist, matrix1);
	}
	else
	{
	   /*
	    *  Set the Parallel transformation
	    */
	   _X3D_matrix_set_parallel(graphics->projection, matrix1);
	}
#endif

	/*
	 *  Set the scale about the origin
	 */
	xmin = (graphics->xv_max - graphics->xv_min);
	ymin = (graphics->yv_max - graphics->yv_min);
	zmin = (graphics->zv_max - graphics->zv_min);
	X3D_matrix_set_scale(xmin, ymin, zmin, matrix2);
	_X3D_matrix_mult(matrix1, matrix2, matrix1);

#if OldViewing
	X3D_matrix_set_translate(graphics->xv_min, graphics->yv_min,
			0.0, matrix2);
#else
	X3D_matrix_set_translate(graphics->xv_min, graphics->yv_min,
			graphics->zv_min, matrix2);
#endif
	_X3D_matrix_mult(matrix1, matrix2, matrix1);

	/*
	 *  Set the device coordinate transformation matrix
	 */
	_X3D_matrix_set_dc(graphics, matrix2);
	_X3D_matrix_mult(matrix1, matrix2, graphics->matrix2);
	_X3D_matrix_mult(graphics->matrix1, graphics->matrix2,
			graphics->matrix);

	/*
	 *  Update the 2D part of the graphics structure.  This is really
	 *  bad, but it  is the only way to allow the user to switch 
	 *  between drawing 3D and 2D graphics on the same id.
	 */
	_X2D_update_graphics(graphics);

	/*
	 *  update the reverse clipping window
	 */
#if OldViewing
	old_values = (double *) kmalloc(16*sizeof(double));
	new_values = (double *) kmalloc(16*sizeof(double));
	for (i = 0; i < 4; i++)
	{
	   for (j = 0; j < 4; j++)
	      old_values[i*4+j] = graphics->matrix[i][j];
	}
	kdmatrix_inverse(old_values, 4, new_values);
	for (i = 0; i < 4; i++)
	{
	   for (j = 0; j < 4; j++)
	      graphics->imatrix[i][j] = new_values[i*4+j];
	}
	kfree(old_values);
	kfree(new_values);
#else
	X3D_matrix_set_translate(-0.5, -0.5, -0.5, matrix1);
	_X3D_matrix_set_inviewing(camera, focus, up, matrix2);
	_X3D_matrix_mult(matrix1, matrix2, matrix1);
	_X3D_matrix_set_invndc(graphics->scale, wcmin, wcmax, matrix2);
	_X3D_matrix_mult(matrix1, matrix2, graphics->imatrix);
#endif

	/*
	 *  find out which corner is closest to us
	 */
	side.z = wcmin.z;
	xmin = wcmin.x;
	ymin = wcmin.y;
	xmax = wcmax.x;
	ymax = wcmax.y;

	side.x = (xmax - xmin)/2 + xmin;
	side.y = ymin;
	_X3D_vector_mult(side, graphics->matrix1, dummy, dummy, z0, dummy);

	side.x = xmin;
	side.y = (ymax - ymin)/2 + ymin;
	_X3D_vector_mult(side, graphics->matrix1, dummy, dummy, z1, dummy);

	side.x = (xmax - xmin)/2 + xmin;
	side.y = ymax;
	_X3D_vector_mult(side, graphics->matrix1, dummy, dummy, z2, dummy);

	side.x = xmax;
	side.y = (ymax - ymin)/2 + ymin;
	_X3D_vector_mult(side, graphics->matrix1, dummy, dummy, z3, dummy);

	zmin = kmin4(z0, z1, z2, z3);
	if (zmin == z0)
	   graphics->corner = BACKLEFT;
	else if (zmin == z1)
	   graphics->corner = BACKRIGHT;
	else if (zmin == z2)
	   graphics->corner = FRONTRIGHT;
	else
	   graphics->corner = FRONTLEFT;
}

void X3D_set_spherical_view(
	X3DGraphics *graphics)
{
	Real dx, dy, dz;


	dx = graphics->camera.x - graphics->focus.x;
	dy = graphics->camera.y - graphics->focus.y;
	dz = graphics->camera.z - graphics->focus.z;

#if !OldViewing
	graphics->eye_dist = sqrt(dx * dx + dy * dy + dz * dz);

	if ( graphics->eye_dist == 0.0 )
	{
	   graphics->alpha = 0.0;
	   graphics->theta = 0.0;
	}
	else
	{
	   if ( dx != 0.0 )
	      graphics->alpha = kradians_degrees(atan(dy/dx));
	   else if ( dx != 0.0 && dy >= 0.0 )
	      graphics->alpha = 90.0;
	   else if ( dx != 0.0 && dy < 0.0 )
	      graphics->alpha = 270.0;

	   graphics->theta = kradians_degrees(acos(dz/graphics->eye_dist));
	}
#endif

}
void X3D_set_geo_view(
	X3DGraphics *graphics)
{
#if !OldViewing
	Real dx, dy, dz;

	dx = graphics->camera.x - graphics->focus.x;
	dy = graphics->camera.y - graphics->focus.y;
	dz = graphics->camera.z - graphics->focus.z;

	graphics->eye_dist = sqrt(dx * dx + dy * dy + dz * dz);

	if ( dx != 0.0 )
	   graphics->alpha = kradians_degrees(acos(dx/graphics->eye_dist));
	else if ( dx != 0.0 && dy >= 0.0 )
	   graphics->alpha = 90.0;
	else if ( dx != 0.0 && dy < 0.0 )
	   graphics->alpha = 270.0;

	if ( dy != 0.0 )
	   graphics->theta = kradians_degrees(acos(dy/graphics->eye_dist));
	else if ( dy != 0.0 && dz >= 0.0 )
	   graphics->theta = 90.0;
	else if ( dy != 0.0 && dz < 0.0 )
	   graphics->theta = 270.0;

	if ( dz != 0.0 )
	   graphics->gamma = kradians_degrees(acos(dz/graphics->eye_dist));
	else if ( dz != 0.0 && dx >= 0.0 )
	   graphics->gamma = 90.0;
	else if ( dz != 0.0 && dx < 0.0 )
	   graphics->gamma = 270.0;
#endif
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_init_graphics
|
|       Purpose: Gets the graphics ID number from the user, and
|		 creates the X3DGraphics structure that will be
|		 associated with it, for future reference.
|
|         Input: id	  - X3D graphics structure ID
|		 win_type - Type of window (2D or 3D)
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: 
|
------------------------------------------------------------*/

int X3D_init_graphics (
   int id,
   int win_type)
{
	X3DGraphics	*graphics;
	X3DList		*entry;

	entry = (X3DList *) kmalloc(sizeof(X3DList));
	if (entry == NULL)
	   return(FALSE);

	graphics = (X3DGraphics *) kcalloc(1,sizeof(X3DGraphics));
	if (graphics == NULL)
	   return(FALSE);

	entry->id	= id;
	entry->graphics	= graphics;
	entry->next	= graphics_head;
	entry->prev	= NULL;

	if (graphics_head != NULL)
	   graphics_head->prev	= entry;

	graphics_head = entry;

	graphics->id 	     = id;
	graphics->win_type   = win_type;
	graphics->device     = X11;
	graphics->line_type  = KLINE_SOLID;
	graphics->line_width = KLINE_EXTRA_FINE;
	graphics->fg	     =
	graphics->bg	     = ~0;

	/*
	 *  Set some defaults.  We really should not have to, but just to
	 *  be on the safe side.
	 */
	graphics->projection	 = KGRAPHICS_PERSPECTIVE;
	graphics->eye_dist       = 1.0;
	graphics->view_dist      = 6.0;
	graphics->focus.x        =
	graphics->focus.y        =
	graphics->focus.z        = 0.0;
	graphics->camera.x       =
	graphics->camera.y       =
	graphics->camera.z       = 1.0;
	graphics->up.x           = 1.0;
	graphics->up.y           = 1.0;
	graphics->up.z           = 2.0;
	graphics->clipping       = FALSE;
	graphics->depth_clipping = FALSE;
	graphics->resize_view    = FALSE;

#if OldViewing
	graphics->eye_dist = 8.0;
	graphics->view_dist = 16.0;
	graphics->alpha = 31.8;
	graphics->theta = 63.7;
#endif

	/*
	 *  lighting
	 */
	graphics->ncode = NORM_ABSOLUTE;
	graphics->specular = 0.0;
	graphics->ambient  = 0.0;

	/*
	 *  Set scale type for the World Coordinates to be linear.
	 */
	graphics->scale.x = KGRAPHICS_LINEAR;
	graphics->scale.y = KGRAPHICS_LINEAR;
	graphics->scale.z = KGRAPHICS_LINEAR;

	/*
	 *  Set the graphics device coordinates to be a 100x100 window.
	 */
	graphics->X11_xmin = graphics->X11_ymin = 0;
	graphics->X11_xmax = graphics->X11_ymax = 100;

	/*
	 *  Set the viewport minimums and maximums to be the entire
	 *  viewport.
	 */
	graphics->xv_min = graphics->yv_min = graphics->zv_min = 0.0;
	graphics->xv_max = graphics->yv_max = graphics->zv_max = 1.0;

	/*
	 *  Set the world coordinate minimums and maximums to be 
	 *  between 0.0 and 1.0
	 */
	graphics->wc_min.x = graphics->wc_min.y = graphics->wc_min.z = 0.0;
	graphics->wc_max.x = graphics->wc_max.y = graphics->wc_max.z = 1.0;

	/*
	 *  Set the text defaults.
	 */
	graphics->spacing     = 0.0;
	graphics->aspect      = 0.75;
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_set_projection
|
|       Purpose: Sets the projection type:
|			KGRAPHICS_PERSPECTIVE     1
|			KGRAPHICS_ORTHOGRAPHIC    2
|			KGRAPHICS_CAVALIER        3
|			KGRAPHICS_CABINET         4
|
|         Input: id           - X3D graphics structure ID
|		 projection   - Type of projection.
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: 
|
------------------------------------------------------------*/

int X3D_set_projection (
   int id,
   int projection)
{
        X3DGraphics *graphics;

        if ((graphics = _X3D_get_graphics(id)) == NULL)
        {
           (void) kfprintf (kstderr,"X3D_set_projection:");
	   (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
           return(FALSE);
        }


	switch (projection)
	{
	   case	KGRAPHICS_PERSPECTIVE:  
	        graphics->projection = KGRAPHICS_PERSPECTIVE;
		break;

	   case	KGRAPHICS_ORTHOGRAPHIC: 
		graphics->projection = KGRAPHICS_ORTHOGRAPHIC;
		break;

	   case	KGRAPHICS_CAVALIER:     
		graphics->projection = KGRAPHICS_CAVALIER;
		break;

	   case	KGRAPHICS_CABINET:      
		graphics->projection = KGRAPHICS_CABINET;
		break;

	   default:
	        (void) kfprintf (kstderr,"X3D_set_projection:");
		(void) kfprintf (kstderr,"\t unknown projection");
		(void) kfprintf (kstderr," type %d\n",projection);
		return(FALSE);
	}
	_X3D_update_graphics(graphics);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_close_graphics
|
|       Purpose: Deletes the X3D graphics structure.
|
|         Input: id - X3D graphics structure ID
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: 
|
------------------------------------------------------------*/

int X3D_close_graphics (
   int id)
{
	X3DList		*current;

	/*
	 *  Search through the linked graphics list for the desired graphics
	 *  id.
	 */
	if (graphics_head == NULL)
	   return(FALSE);

	current = graphics_head;
	while (current->next != NULL && current->id != id)
	      current = current->next;

	if (current->id == id)
	{
	   /*
	    */
	   if (current->next == NULL && current->prev == NULL)
	   {
	      graphics_head = NULL;
	   }
	   else if (current->prev == NULL)
	   {
	      current->next->prev = NULL;
	      graphics_head = current->next;
	   }
	   else if (current->next == NULL)
	   {
	      current->prev->next = NULL;
	   }
	   else
	   {
	      current->next->prev = current->prev;
	      current->prev->next = current->next;
	   }
	   kfree(current->graphics);
	   kfree(current);
	   return(TRUE);
	}

	(void) kfprintf (kstderr,"X3D_close_graphics:");
	(void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
	return(FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_set_viewpoint
|
|       Purpose: Sets the eye orientation for a given X3D graphics
|		 structure.
|
|         Input: id        - X3D graphics structure ID
|		 alpha     - the angle describing the position of the
|			     eye between the Z axis and the XY plane
|		 theta     - Angle describing the position of the eye
|			     around Z axis with respect to the XY plane
|		 gammaval  - Angle describing the rotation of the eye
|			     to tilt the view
|		 eye_dist  - Distance of the eye from the object
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: 
|
------------------------------------------------------------*/

int X3D_set_viewpoint (
   int  id,
   Real alpha,
   Real theta,
   Real gammaval,
   Real eye_distance)
{
	X3DGraphics *graphics;
	Real	    x, y, z;


	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   (void) kfprintf (kstderr,"X3D_set_viewpoint:");
	   (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
	   return(FALSE);
	}

	graphics->eye_dist = eye_distance;
	graphics->alpha = alpha;
	graphics->theta = theta;
	graphics->gamma = gammaval;

	alpha = kdegrees_radians(alpha);
	theta = kdegrees_radians(theta);
	gammaval = kdegrees_radians(gammaval);

	x = eye_distance * sin(theta) * cos(alpha);
	y = eye_distance * sin(theta) * sin(alpha);
	z = eye_distance * cos(theta);

	graphics->camera.x = graphics->focus.x + x;
	graphics->camera.y = graphics->focus.y + y;
	graphics->camera.z = graphics->focus.z + z;

	_X3D_update_graphics(graphics);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_set_focus
|
|       Purpose: Sets the point in which the eye is focusing on.
|
|         Input: id    - X3D graphics structure ID
|		 focus - the point in which the user is focusing
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By:
|          Date: 
|
------------------------------------------------------------*/

int X3D_set_focus (
   int   id,
   Coord focus)
{
	X3DGraphics *graphics;


	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   (void) kfprintf (kstderr,"X3D_set_focus:");
	   (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
	   return(FALSE);
	}

	graphics->focus = focus;
	X3D_set_spherical_view(graphics);

	_X3D_update_graphics(graphics);
	return(TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: X3D_set_camera
|
|       Purpose: Sets the point in which the eye is currently at
|
|         Input: id	- X3D graphics structure ID
|		 camera	- the camera or eye position
|		 up	- the camera rotation
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: 
|
------------------------------------------------------------*/

int X3D_set_camera (
   int   id,
   Coord camera,
   Coord up)
{
	X3DGraphics *graphics;


	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   (void) kfprintf (kstderr,"X3D_set_camera:");
	   (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
	   return(FALSE);
	}
	graphics->up	 = up;
	graphics->camera = camera;
	X3D_set_spherical_view(graphics);

	_X3D_update_graphics(graphics);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_set_view_distance
|
|       Purpose: Sets the eye orientation for a given X3D graphics
|		 structure.
|
|         Input: id	   - X3D graphics structure ID
|		 view_dist - Distance of the eye from the viewport
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: 
|
------------------------------------------------------------*/

int X3D_set_view_distance (
   int  id,
   Real view_distance)
{
        X3DGraphics *graphics;

        if ((graphics = _X3D_get_graphics(id)) == NULL)
        {
           (void) kfprintf (kstderr,"X3D_set_view_distance:");
           (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
           return(FALSE);
        }
        graphics->view_dist = view_distance;
        _X3D_update_graphics(graphics);
        return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_set_window
|
|       Purpose: Sets the minimum and maximum device coordinate
|		 for a given X11 Window.
|
|         Input: id	- X3D graphics structure ID
|		 x	- x offset in device coordinates
|		 y	- y offset in device coordinates
|		 width	- width of the window in device coordinates
|		 height	- height of the window in device coordinates
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: 
|
------------------------------------------------------------*/

int X3D_set_window (
   int   id,
   int   x,
   int   y,
   unsigned int width,
   unsigned int height)
{
	X3DGraphics 	*graphics;


	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   (void) kfprintf (kstderr,"X3D_set_window:");
	   (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
	   return(FALSE);
	}

	if (graphics->device == X11)
	{
	   graphics->X11_xmin = x;
	   graphics->X11_ymin = y;
	   graphics->X11_xmax = (width-1)  + x;
	   graphics->X11_ymax = (height-1) + y;
	}
	_X3D_update_graphics(graphics);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_set_viewport
|
|       Purpose: Sets the minimum and maximum viewport values
|		 for a given X3D graphics structure.
|
|         Input: id	 - X3D graphics structure ID
|		 viewmin - the viewport minimum coordinate value.
|		 viewmax - the viewport maximum coordinate value.
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: 
|
------------------------------------------------------------*/

int X3D_set_viewport (
   int   id,
   Coord viewmin,
   Coord viewmax)
{
	X3DGraphics *graphics;

	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   (void) kfprintf (kstderr,"X3D_set_viewport:");
	   (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
	   return(FALSE);
	}
	graphics->xv_min = viewmin.x;
	graphics->xv_max = viewmax.x;
	graphics->yv_min = viewmin.y;
	graphics->yv_max = viewmax.y;
	graphics->zv_min = viewmin.z;
	graphics->zv_max = viewmax.z;
	if (graphics->xv_min == graphics->xv_max)
	{
	   graphics->xv_min -= 0.5;
	   graphics->xv_max += 0.5;
	}
	if (graphics->yv_min == graphics->yv_max)
	{
	   graphics->yv_min -= 0.5;
	   graphics->yv_max += 0.5;
	}
	if (graphics->zv_min == graphics->zv_max)
	{
	   graphics->zv_min -= 0.5;
	   graphics->zv_max += 0.5;
	}
	_X3D_update_graphics(graphics);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_set_wc_min_max
|
|       Purpose: Sets the minimum and maximum world coordinate
|		 points for a given X3D graphics structure.
|
|         Input: id     - X3D graphics structure ID
|		 wc_min - A coord containing the minimum world
|			  coordinate values
|		 wc_max - A coord containing the maximum world
|			  coordinate values
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: 
|
------------------------------------------------------------*/

int X3D_set_wc_min_max (
   int   id,
   Coord wc_min,
   Coord wc_max)
{
	X3DGraphics *graphics;

	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   (void) kfprintf (kstderr,"X3D_set_wc_min_max:");
	   (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
	   return(FALSE);
	}

	if (wc_min.x == wc_max.x)
	{
	   wc_min.x -= 0.5;
	   wc_max.x += 0.5;
	}
	if (wc_min.y == wc_max.y)
	{
	   wc_min.y -= 0.5;
	   wc_max.y += 0.5;
	}
	if (wc_min.z == wc_max.z)
	{
	   wc_min.z -= 0.5;
	   wc_max.z += 0.5;
	}

	graphics->wc_min = wc_min;
	graphics->wc_max = wc_max;

	_X3D_update_graphics(graphics);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_set_wc_scale
|
|       Purpose: Sets the minimum and maximum world coordinate
|		 points for a given X3D graphics structure.
|
|         Input: id    - X3D graphics structure ID
|		 scale - A coord containing the maximum world
|			 coordinate scale types for the x, y,
|			 & z axes.
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: 
|
------------------------------------------------------------*/

int X3D_set_wc_scale(
   int id,
   int x,
   int y,
   int z)
{
	X3DGraphics *graphics;
	Coord wc_min, wc_max;

	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   (void) kfprintf (kstderr,"X3D_set_wc_scale:");
	   (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
	   return(FALSE);
	}

	wc_min = graphics->wc_min;
	wc_max = graphics->wc_max;

	if ( x >= KGRAPHICS_LINEAR && x <= KGRAPHICS_LN )
	  graphics->scale.x = x;

	if ( y >= KGRAPHICS_LINEAR && y <= KGRAPHICS_LN )
	  graphics->scale.y = y;

	if ( z >= KGRAPHICS_LINEAR && z <= KGRAPHICS_LN )
	  graphics->scale.z = z;

	_X3D_update_graphics(graphics);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_set_line_type
|
|       Purpose: Sets the line type for a given graphics context
|
|         Input: id	   - X3D graphics structure ID
|		 line_type - The new line type
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young 
|          Date: 
|
------------------------------------------------------------*/

int X3D_set_line_type(
   int id,
   int line_type)
{
	X3DGraphics *graphics;

	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   (void) kfprintf (kstderr,"X3D_set_line_type:");
	   (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
	   return(FALSE);
	}

	graphics->line_type = line_type;
        X3D_draw[graphics->device].line_type(graphics, line_type);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_set_line_width
|
|       Purpose: Sets the line width for a given graphics context
|
|         Input: id	    - X3D graphics structure ID
|		 line_width - The new line width
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young 
|          Date: 
|
------------------------------------------------------------*/

int X3D_set_line_width(
   int id,
   int line_width)
{
	X3DGraphics *graphics;

	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   (void) kfprintf (kstderr,"X3D_set_line_width:");
	   (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
	   return(FALSE);
	}

	graphics->line_width = line_width;
        X3D_draw[graphics->device].line_width(graphics, line_width);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_set_draw
|
|       Purpose: Sets the foreground draw color
|
|         Input: id - X3D graphics structure ID
|		 fg - The new foreground color
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young 
|          Date: 
|
------------------------------------------------------------*/

int X3D_set_draw(
   int    id,
   XColor *fg)
{
	X3DGraphics *graphics;

	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   (void) kfprintf (kstderr,"X3D_set_draw:");
	   (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
	   return(FALSE);
	}
        X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_set_fill
|
|       Purpose: Sets the background fill color
|
|         Input: id - X3D graphics structure ID
|		 bg - The new background color
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mark Young 
|          Date: 
|
------------------------------------------------------------*/

int X3D_set_fill(
   int    id,
   XColor *bg)
{
	X3DGraphics *graphics;

	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   (void) kfprintf (kstderr,"X3D_set_draw:");
	   (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
	   return(FALSE);
	}
        X3D_draw[graphics->device].draw_color(graphics, bg, FALSE);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_set_clipping
|
|       Purpose: Turns clipping to the viewwport on for 3D
|
|         Input: id       - X3D graphics structure ID
|		 clipping - 1 for on; 0 for off
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mike Lang
|          Date: 
|
------------------------------------------------------------*/

int X3D_set_clipping(
   int id,
   int clipping,
   int depth_clipping)
{
	X3DGraphics *graphics;
	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   (void) kfprintf (kstderr,"X3D_set_clipping:");
	   (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
	   return(FALSE);
	}
	graphics->clipping = clipping;
	graphics->depth_clipping = depth_clipping;
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_set_resize_view
|
|       Purpose: 
|
|         Input: id	     - X3D graphics structure ID
|		 resize_view -
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mike Lang
|          Date: 
|
------------------------------------------------------------*/

int X3D_set_resize_view(
   int id,
   int resize_view)
{
	X3DGraphics *graphics;
	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   (void) kfprintf (kstderr,"X3D_set_resize_view:");
	   (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
	   return(FALSE);
	}
	graphics->resize_view = resize_view;
	_X3D_update_graphics(graphics);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_set_clipping_volume
|
|       Purpose: Sets the clipping volume
|
|         Input: id       - X3D graphics structure ID
|		 clip_min - min clipping values
|		 clip_max - max clipping values
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Mike Lang
|          Date: 
|

int X3D_set_clipping_volume(id, clip_min, clip_max)

int	id;
Coord	clip_min, clip_max;
{
	X3DGraphics *graphics;
	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   (void) kfprintf (kstderr,"X3D_set_clipping:");
	   (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
	   return(FALSE);
	}
	graphics->clip_min = clip_min;
	graphics->clip_max = clip_max;
	return(TRUE);
}
------------------------------------------------------------*/
