/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>   		2D Viewing System Control Utilities
   >>>>
   >>>>  Private:
   >>>>               _X2D_update_graphics()
   >>>>               X2D_set_window()
   >>>>               X2D_set_viewport()
   >>>>               X2D_set_wc_min_max()
   >>>>               X2D_set_wc_scale()
   >>>>               X2D_set_clipping()
   >>>>   Static:
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "graphics.h"


/*-----------------------------------------------------------
|
|  Routine Name: _X2D_update_graphics
|
|       Purpose: Recomputes the X3D Graphics Matrix, used to
|		 transform 2D world coordinates into device
|		 coordinates.
|
|         Input: graphics - X3D graphics structure
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void _X2D_update_graphics (
   X3DGraphics *graphics)
{
	Matrix_2D	matrix1, matrix2;

	/*
	 * Convert to unit cube.
	*/
	_X2D_matrix_set_ndc(graphics->scale,
			    graphics->xv_min, graphics->xv_max,
			    graphics->yv_min, graphics->yv_max,
			    graphics->wc_min, graphics->wc_max, matrix1);
	/*
	 *  Set the device coordinate transformation matrix
	 */
	_X2D_matrix_set_dc(graphics, matrix2);
	_X2D_matrix_mult(matrix1, matrix2, graphics->matrix2D);
}

/*-----------------------------------------------------------
|
|  Routine Name: X2D_set_window
|
|       Purpose: Sets the minimum and maximum device coordinate
*               for a given X11 Window.
|
|         Input: id	- X3D graphics structure ID
|		 x	- x offset in device coordinates
|		 y	- y offset in device coordinates
|		 width	- width of the window in device coordinates
|		 height	- height of the window in device coordinates
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

int X2D_set_window (
   int   id,
   int   x,
   int   y,
   unsigned int width,
   unsigned int height)
{
	X3DGraphics 	*graphics;


	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   (void) kfprintf (kstderr,"X2D_set_window:");
	   (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
	   return(FALSE);
	}


	if (graphics->device == X11)
	{
	   graphics->X11_xmin = x;
	   graphics->X11_ymin = y;
	   graphics->X11_xmax = width  +  x;
	   graphics->X11_ymax = height + y;
	}
	_X2D_update_graphics(graphics);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X2D_set_viewport
|
|       Purpose: Sets the minimum and maximum viewport values
|		 for a given X3D graphics structure.
|
|         Input: id	 - X3D graphics structure ID
|		 viewmin - the viewport minimum coordinate value.
|		 viewmax - the viewport maximum coordinate value.
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

int X2D_set_viewport (
   int   id,
   Coord viewmin,
   Coord viewmax)
{
	X3DGraphics *graphics;

	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   (void) kfprintf (kstderr,"X2D_set_viewport:");
	   (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
	   return(FALSE);
	}
	graphics->xv_min = viewmin.x;
	graphics->xv_max = viewmax.x;
	graphics->yv_min = viewmin.y;
	graphics->yv_max = viewmax.y;
	if (graphics->xv_min == graphics->xv_max)
	{
	   graphics->xv_min -= 0.5;
	   graphics->xv_max += 0.5;
	}
	if (graphics->yv_min == graphics->yv_max)
	{
	   graphics->yv_min -= 0.5;
	   graphics->yv_max += 0.5;
	}
	_X2D_update_graphics(graphics);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X2D_set_wc_min_max
|
|       Purpose: Sets the minimum and maximum world coordinate
|		 points for a given X3D graphics structure.
|
|         Input: id     - X3D graphics structure ID
|		 wc_min - A coord containing the minimum world
|			  coordinate values
|		 wc_max - A coord containing the maximum world
|			  coordinate values
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

int X2D_set_wc_min_max (
   int   id,
   Coord wc_min,
   Coord wc_max)
{
	X3DGraphics *graphics;

	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   (void) kfprintf (kstderr,"X2D_set_wc_min_max:");
	   (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
	   return(FALSE);
	}

	if (wc_min.x == wc_max.x)
	{
	   wc_min.x -= 0.5;
	   wc_max.x += 0.5;
	}
	if (wc_min.y == wc_max.y)
	{
	   wc_min.y -= 0.5;
	   wc_max.y += 0.5;
	}

	graphics->wc_min = wc_min;
	graphics->wc_max = wc_max;

	_X2D_update_graphics(graphics);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X2D_set_wc_scale
|
|       Purpose: Sets the minimum and maximum world coordinate
|		 points for a given X3D graphics structure.
|
|         Input: id    - X3D graphics structure ID
|		 scale - A coord containing the maximum world
|			 coordinate scale types for the x & y,
|			 axes.
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young & Mike Lang
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

int X2D_set_wc_scale(
   int id,
   int x,
   int y)
{
	X3DGraphics *graphics;
	Coord wc_min, wc_max;

	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   (void) kfprintf (kstderr,"X3D_set_wc_scale:");
	   (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
	   return(FALSE);
	}

        wc_min = graphics->wc_min;
        wc_max = graphics->wc_max;

        if ( x >= KGRAPHICS_LINEAR && x <= KGRAPHICS_LN )
          graphics->scale.x = x;

        if ( y >= KGRAPHICS_LINEAR && y <= KGRAPHICS_LN )
          graphics->scale.y = y;

	_X2D_update_graphics(graphics);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X2D_set_clipping
|
|       Purpose: Turns clipping to the viewport on for 2D
|
|         Input: id       - X3D graphics structure ID
|		 clipping - 1 for on; 0 for off
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mike Lang
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

int X2D_set_clipping(
   int id,
   int clipping)
{
	X3DGraphics *graphics;
	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   (void) kfprintf (kstderr,"X2D_set_clipping:");
	   (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
	   return(FALSE);
	}
	graphics->clipping = clipping;
	return(TRUE);
}
