/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>   		3D Conversion Utilities
   >>>>
   >>>>  Private:
   >>>>			_X3D_convert_wc_to_dc()
   >>>>			_X3D_convert_wc_to_dc_seg()
   >>>>			X3D_convert_point_wc_to_dc()
   >>>>			X3D_convert_point_wc_to_ndc()
   >>>>   Static:
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "graphics.h"


/*-----------------------------------------------------------
|
|  Routine Name: _X3D_convert_wc_to_dc
|
|       Purpose: Converts a array of 3D world coordinate points
|                to device coordinates and stores them as XPoints
|                for later calls to X11 routines
|
|         Input: graphics - the graphics structure
|                coords   - array of world coordinates input
|                size     - number of points to convert
|
|        Output: points   - XPoint array of device coordinates
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

int _X3D_convert_wc_to_dc (
   X3DGraphics *graphics,
   Matrix      matrix,
   Coord       *coords,
   XPoint      *points,
   int         size)
{
	register int  i;
	register Real x, y, z, w;
	Coord    *scale;


	if(!(_X3D_scale_wc_coords(graphics->scale, coords, &scale, size)))
	   return(FALSE);

	if (graphics->projection == KGRAPHICS_PERSPECTIVE)
	{
	   for (i = 0; i < size; i++)
	   {
	       _X3D_vector_mult(scale[i], matrix, x, y, z, w);
	       if (w != 0)
	       {
	          points[i].x = x/w + 0.5;
	          points[i].y = y/w + 0.5;
	       }
	       else
	       {
	          points[i].x = x + 0.5;
	          points[i].y = y + 0.5;
	       }
	   }
	}
	else
	{
	   for (i = 0; i < size; i++)
	   {
	       _X3D_vector_mult(scale[i], matrix, x, y, z, w);
	       points[i].x = x + 0.5;
	       points[i].y = y + 0.5;
	   }
	}

	if (coords != scale)
	   kfree(scale);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _X3D_convert_wc_to_dc_seg
|
|       Purpose: Converts a array of 3D world coordinate segments
|                to device coordinates and stores them as XSegments
|                for later calls to X11 routines
|
|         Input: graphics - the graphics structure
|                coords   - array of world coordinates input
|                size     - number of points to convert
|
|        Output: segments - XSegment array of device coordinates
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

int _X3D_convert_wc_to_dc_seg(
   X3DGraphics *graphics,
   Matrix      matrix,
   Coord       *coords,
   XSegment    *segments,
   int         size)
{
	register int   i, j;
	register Real  x, y, z, w;
	Coord    *scale;

	if(!(_X3D_scale_wc_coords(graphics->scale, coords, &scale, size)))
	   return(FALSE);

	j = 0;
	if (graphics->projection == KGRAPHICS_PERSPECTIVE)
	{
	   for (i = 0; i < size; i += 2)
	   {
	       _X3D_vector_mult(scale[i], matrix, x, y, z, w);
	       if (w != 0)
	       {
	          segments[j].x1 = x/w + 0.5;
	          segments[j].y1 = y/w + 0.5;
	       }
	       else
	       {
	          segments[j].x1 = x + 0.5;
	          segments[j].y1 = y + 0.5;
	       }

               _X3D_vector_mult(scale[i +1], matrix, x, y, z, w);
               if (w != 0)
               {
                  segments[j].x2 = x/w + 0.5;
                  segments[j].y2 = y/w + 0.5;
               }
               else
               {
                  segments[j].x2 = x + 0.5;
                  segments[j].y2 = y + 0.5;
               }
	       j++;
	   }
	}
	else
	{
	   for (i = 0; i < size; i += 2)
	   {
	       _X3D_vector_mult(scale[i], matrix, x, y, z, w);
	       segments[j].x1 = x + 0.5;
	       segments[j].y1 = y + 0.5;

	       _X3D_vector_mult(scale[i+1], matrix, x, y, z, w);
	       segments[j].x2 = x + 0.5;
	       segments[j].y2 = y + 0.5;

	       j++;
	   }
	}

	if (coords != scale)
	   kfree(scale);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_convert_point_wc_to_dc
|
|       Purpose: Converts a 3D world coordinate point
|                to device coordinates and stores it the
|                X and Y variables.
|
|         Input: id      - the X2D graphics structure
|                coord   - array of world coordinates input
|
|        Output: x_out - array of x corresponding dc values of the
|                        world coordinate points
|                y_out - array of y corresponding dc values of the
|                        world coordinate point
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

int X3D_convert_point_wc_to_dc (
   int   id,
   Coord coord,
   Real  *x_out,
   Real  *y_out)
{
	register Real  x, y, z, w;
	Coord	 scale;
	X3DGraphics *graphics;


        if ((graphics = _X3D_get_graphics(id)) == NULL)
        {
           (void) kfprintf (kstderr,"X3D_convert_point_wc_to_dc:");
           (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
           return(FALSE);
        }

	if(!(_X3D_scale_wc_coord(graphics->scale, coord, &scale)))
	   return(FALSE);

	if (graphics->projection == KGRAPHICS_PERSPECTIVE)
	{
	   _X3D_vector_mult(scale, graphics->matrix, x, y, z, w);
	   if (w != 0)
	   {
	      *x_out = x/w;
	      *y_out = y/w;
	   }
	   else
	   {
	      *x_out = x;
	      *y_out = y;
	   }
	}
	else
	{
	   _X3D_vector_mult(scale, graphics->matrix, x, y, z, w);
	   *x_out = x;
	   *y_out = y;
	}
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: X3D_convert_point_wc_to_ndc
|
|       Purpose: Converts a 3D world coordinate point
|		 to the unit cube and stores them in the routine
|		 coordinate.
|
|         Input: id    - the graphics structure
|		 coord_in   - the world coordinate input point
|
|        Output: coord_out  - the normalized coordinate output point
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

int X3D_convert_point_wc_to_ndc(
   int   id,
   Coord coord_in,
   Coord *coord_out)
{
	Coord	      scale;
	X3DGraphics   *graphics;
	register Real x, y, z, w;


        if ((graphics = _X3D_get_graphics(id)) == NULL)
        {
           (void) kfprintf (kstderr,"X3D_convert_point_wc_to_dc:");
           (void) kfprintf (kstderr,"\t unknown graphics id %d\n",id);
           return(FALSE);
        }

	if(!(_X3D_scale_wc_coord(graphics->scale, coord_in, &scale)))
	   return(FALSE);

	_X3D_vector_mult(scale, graphics->matrix1, x, y, z, w);
	coord_out->x = x;
	coord_out->y = y;
	coord_out->z = z;
	return(TRUE);
}
