/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            2D Clipping Routines
   >>>>
   >>>>  Private:
   >>>>			_X2D_clip_points()	      	
   >>>>			_X2D_clip_segments()   	 
   >>>>			_X2D_clip_lines() 
   >>>>			_X2D_clip_polygon() 
   >>>>			_X2D_clip_rectangle() 
   >>>>   Static:
   >>>>			compute_outcode_2D()
   >>>>			clip_point_2D()
   >>>>			clip_segment_2D()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "graphics.h"


/*-----------------------------------------------------------
|
|  Routine Name: compute_outcode_2D - determine the outcode for a (x,y)
|                                     point
|
|       Purpose: Computes the outcodes (Cohen-Sutherland style).
|                The outcode will indicate if the point is to the
|                left, right, top, bottom, or in the center of our
|                world coordinate view.
|
|         Input: graphics - the graphics structure
|                x   - the x position
|                y   - the y position
|
|        Output: none
|
|       Returns: the outcode for the (x,y) point
|
|    Written By: Mark Young
|          Date:
| Modifications:
|
------------------------------------------------------------*/

static int compute_outcode_2D(
   X3DGraphics    *graphics,
   register  Real x,
   register  Real y)
{
        register int outcode;


        if (x > graphics->wc_max.x)
           outcode = X2D_RIGHT;
        else if (x < graphics->wc_min.x)
           outcode = X2D_LEFT;
        else
           outcode = X2D_CENTER;

        if (y > graphics->wc_max.y)
           outcode |= X2D_TOP;
        else if (y < graphics->wc_min.y)
           outcode |= X2D_BOTTOM;

        return(outcode);
}


/*-----------------------------------------------------------
|
|  Routine Name: clip_point_2D
|
|       Purpose: clips a point by making sure that it lies within
|		 world coordinate viewing system.  If the point is
|		 to the left of wcmin.x then x is set to wcmin.x.
|		 The process is repeated for points that are to
|		 the right, bottom, & top. 
|
|         Input: graphics - the graphics structure
|		 outcode  - the code that defines the points location
|			    in terms of the clipping region
|		 xin	  - the x point to be clipped
|		 yin	  - the y point to be clipped
|
|        Output: point - the clipped point
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

static void clip_point_2D(
   X3DGraphics    *graphics,
   int            outcode,
   register  Real xin,
   register  Real yin,
   Coord          *point)
{
	if (outcode & X2D_LEFT)
	   point->x = graphics->wc_min.x;
	else if (outcode & X2D_RIGHT)
	   point->x = graphics->wc_max.x;
	else
	   point->x = xin;

	if (outcode & X2D_TOP)
	   point->y = graphics->wc_max.y;
	else if (outcode & X2D_BOTTOM)
	   point->y = graphics->wc_min.y;
	else
	   point->y = yin;
}

/*-----------------------------------------------------------
|
|  Routine Name: clip_segment_2D
|
|       Purpose: computes the outcodes (Cohen-Sutherland style).
|		 The outcode will indicate if the point is to the
|		 left, right, top, bttom, or in the center of our
|		 world coordinate view.
|
|         Input: graphics - the graphics structure
|	         outcode1 - the code that defines the points location
|                           in terms of the clipping region for (px1,py1)
|	         outcode2 - the code that defines the points location
|                           in terms of the clipping region for (px2,py2)
|		 (px1,py1)  - the begin point of the segment to be
|			    clipped
|		 (px2,py2)  - the end point of the segment to be
|			    clipped
|
|        Output: point1, point2 - the resulting clipped output line
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: 
| Modifications: John M. Salas
|		 Sun Apr 26 11:50:21 MDT 1992
|
------------------------------------------------------------*/

int clip_segment_2D(
   X3DGraphics   *graphics,
   int           outcode1,
   int           outcode2,
   register Real px1,
   register Real py1,
   register Real px2,
   register Real py2,
   Coord         *point1,
   Coord         *point2)
{
    int repeat = TRUE;
    int tmp_outcode1, tmp_outcode2;

    tmp_outcode1 = outcode1;
    tmp_outcode2 = outcode2;

/*
  May need to clip twice.
*/
    while ( repeat )
    {

/*
  Only check if the outcode for point if it is not already in the center.
  Only check outcode once, i.e. if TOP-LEFT only the TOP is done the second
  pass if done will check LEFT.  If you combine the two clips you may clip
  incorrectly, the TOP clip may have already moved the point to the CENTER
  and by clipping LEFT you will incorrectly move the point again.  It is
  data and clipping order dependent on whether the first clip moves the point
  to the CENTER or whether two passes are needed.

  Determine the outcode for the clipped point to see if you are done or
  if you need a second pass.  Two passes is the most needed to clip a line
  that passes through the clipping window.

								JMAS
*/
	if ( tmp_outcode1 != X2D_CENTER )
	{
	  if (tmp_outcode1 & X2D_TOP)
	  {
	     px1 = kintercept(px1, px2, py1, py2, graphics->wc_max.y);
	     py1 = graphics->wc_max.y;
	  }
	  else if (tmp_outcode1 & X2D_BOTTOM)
	  {
	     px1 = kintercept(px1, px2, py1, py2, graphics->wc_min.y);
	     py1 = graphics->wc_min.y;
	  }
	  else if (tmp_outcode1 & X2D_RIGHT)
	  {
	     py1 = kintercept(py1, py2, px1, px2, graphics->wc_max.x);
	     px1 = graphics->wc_max.x;
	  }
	  else if (tmp_outcode1 & X2D_LEFT)
	  {
	     py1 = kintercept(py1, py2, px1, px2, graphics->wc_min.x);
	     px1 = graphics->wc_min.x;
	  }
	  tmp_outcode1 = compute_outcode_2D(graphics, px1, py1);
	}
  
	if ( tmp_outcode2 != X2D_CENTER )
	{
	  if (tmp_outcode2 & X2D_TOP)
	  {
	     px2 = kintercept(px1, px2, py1, py2, graphics->wc_max.y);
	     py2 = graphics->wc_max.y;
	  }
	  else if (tmp_outcode2 & X2D_BOTTOM)
	  {
	     px2 = kintercept(px1, px2, py1, py2, graphics->wc_min.y);
	     py2 = graphics->wc_min.y;
	  }
	  else if (tmp_outcode2 & X2D_RIGHT)
	  {
	     py2 = kintercept(py1, py2, px1, px2, graphics->wc_max.x);
	     px2 = graphics->wc_max.x;
	  }
	  else if (tmp_outcode2 & X2D_LEFT)
	  {
	     py2 = kintercept(py1, py2, px1, px2, graphics->wc_min.x);
	     px2 = graphics->wc_min.x;
	  }
	  tmp_outcode2 = compute_outcode_2D(graphics, px2, py2);
	}

/*
  Determine if you are finished:
  1. If both outcode's are CENTER assign points, return TRUE.  Trivial
     case for point's whose outcode is just TOP, BOTTOM, LEFT or RIGHT.
     It will also cover some case that have point's with a combination
     of clip codes, i.e. when a TOP or BOTTOM clip does the job, but
     not all TOP or BOTTOM clips will complete the task.
     NOTE:  It is easier to clip twice in those case where doing a TOP
	    or BOTTOM clip does not complete the clipping processes than 
	    try to figure out whether clipping LEFT or RIGHT the first
	    time will clip the point completely.  And as stated above
	    you do not want to combine the two clipping operation in
	    one pass.
  2. If one or the other is CENTER, in this case a second clip is needed.
     Case: point 1's outcode TOP-LEFT and point 2's outcode CENTER and line
     does pass through the clipping window.  In this case the first clip
     may make point 1's outcode LEFT or CENTER, if it makes it CENTER
     1st if will catch this.  If LEFT than another clip is needed to
     complete the clipping.
  3. If you do a clip and the resulting outcode's become the same you are
     finished, return FALSE.  Case: point 1's outcode TOP-LEFT and point 2's
     outcode BOTTOM-RIGHT and the line does not pass through the clipping
     window.  In this case the first clip makes both outcode's RIGHT.
  4. If you clip and outcode's change, but do not swap values, redefine
     the current outcode's and clip again.  Case: point 1's outcode TOP-LEFT
     and point 2's RIGHT and the line does not pass through the clipping
     window.  In this case the first clip makes point 1's outcode RIGHT and 
     point 2's outcode TOP.  The second clip will swap the two outcode's.
  5. Safty case, return FALSE, may not be need but what the ...

								JMAS
*/
	if ((tmp_outcode1 | tmp_outcode2) == X2D_CENTER)
	{
	  point1->x = px1; point1->y = py1;
	  point2->x = px2; point2->y = py2;
	  return(TRUE);
	}
	else if (tmp_outcode1 == X2D_CENTER || tmp_outcode2 == X2D_CENTER)
	{
	  ;
	}
	else if (tmp_outcode1 == tmp_outcode2)
	{
	  return(FALSE);
	}
	else if ((tmp_outcode1 != outcode1 || tmp_outcode2 != outcode2) &&
		 (tmp_outcode1 != outcode2 && tmp_outcode2 != outcode1))
	{
	  outcode1 = tmp_outcode1;
	  outcode2 = tmp_outcode2;
	}
	else
	{
	  return(FALSE);
	}
    }
    return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _X2D_clip_points - Clips the world coordinate points
|
|       Purpose: Clips the world coordinate points and returns
|		 the clipped points that fall within the view
|		 of the world coordinate viewing system.
|
|         Input: graphics - the graphics structure
|	         coords   - array of world coordinates input
|		 size     - number of points to clipped
|
|        Output: outsize  - the number of clipped points
|
|       Returns: a pointer to the clipped points
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

Coord  *_X2D_clip_points(
   X3DGraphics *graphics,
   Coord       *coords,
   int         size,
   int         clip_code,
   int         *outsize)
{
	Coord   *points;
	int	i, j, outcode;


	if (graphics->clipping == FALSE)
	{
	   *outsize = size;
	   return(coords);
	}

	if ((points = (Coord *) kmalloc((unsigned) size*sizeof(Coord))) == NULL)
	{
	   kfprintf (kstderr,"_X2D_clip_points:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return(NULL);
	}

	for (i = 0, j = 0; i < size; i++)
	{
	   outcode = compute_outcode_2D(graphics, coords[i].x, coords[i].y);
	   if (outcode == X2D_CENTER)
	   {
	      points[j++] = coords[i];
	   }
	   else if ((outcode & ~clip_code) == X2D_CENTER)
	   {
	      clip_point_2D(graphics, outcode, coords[i].x, coords[i].y,
			&points[j]);
	      points[j++].d = coords[i].d;
	   }
	}

	if (j == 0)
	{
	   kfree(points); *outsize = 0;
	   return(NULL);
	}
	*outsize = j;
	return(points);
}

/*-----------------------------------------------------------
|
|  Routine Name: _X2D_clip_segments - Clips the world coordinate segments
|
|       Purpose: Clips the world coordinate segments and returns
|		 the clipped segments that fall within the view
|		 of the world coordinate viewing system.
|
|         Input: graphics - the graphics structure
|	         coords   - array of world coordinates input
|		 size     - number of points to clipped
|
|        Output: outsize  - the number of clipped points
|
|       Returns: points   - the clipped points
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

Coord  *_X2D_clip_segments(
   X3DGraphics *graphics,
   Coord       *coords,
   int         size,
   int         *outsize)
{
	Coord   *points;
	int	i, j, outcode1, outcode2, clipped ;


	if (graphics->clipping == FALSE)
	{
	   *outsize = size;
	   return(coords);
	}

	if ((points = (Coord *)kmalloc((unsigned)size*2*sizeof(Coord))) == NULL)
	{
	   kfprintf (kstderr,"_X2D_clip_points:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return(NULL);
	}

	for (i = 0, j = 0; i+1 < size; i += 2)
	{
	   outcode1 = compute_outcode_2D(graphics, coords[i].x, coords[i].y);
	   outcode2 = compute_outcode_2D(graphics, coords[i+1].x,coords[i+1].y);
	   if ((outcode1 | outcode2) == X2D_CENTER)
	   {
	      points[j] = coords[i-1];
	      points[j+1] = coords[i];
	      j += 2;
	   }
	   else if ((outcode1 & outcode2) == X2D_CENTER)
	   {
	      clipped = clip_segment_2D(graphics, outcode1, outcode2,
					coords[i-1].x, coords[i-1].y,
					coords[i].x, coords[i].y,
					&points[j], &points[j+1]);
	      points[j].d = coords[i-1].d;
	      points[j+1].d = coords[i].d;
	      if ( clipped )
	        j += 2;
	   }
	}

	if (j == 0)
	{
	   kfree(points); *outsize = 0;
	   return(NULL);
	}
	*outsize = j;
	return(points);
}

/*-----------------------------------------------------------
|
|  Routine Name: _X2D_clip_lines - Clip the world coordinate polyline
|
|       Purpose: Clips the world coordinate polyline such that
|		 the resulting segments or set of polylines lie within
|		 the world coordinate viewing system.
|
|         Input: graphics - the graphics structure
|		 coords   - array that defines the polyline
|		 size     - number of points in the polyline to clipped
|
|        Output: outsize  - the number of clipped segment points
|
|       Returns: points   - the clipped segments
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

Coord  *_X2D_clip_lines(
   X3DGraphics *graphics,
   Coord       *coords,
   int         size,
   int         *outsize)
{
	Coord   *points;
	int	i, j, outcode1, outcode2, clipped;

	if ((points = (Coord *)kmalloc((unsigned)size*2*sizeof(Coord))) == NULL)
	{
	   kfprintf (kstderr,"_X2D_clip_lines:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return(NULL);
	}

	if (graphics->clipping == FALSE)
	{
	   for (i = 1, j = 0; i < size; i++)
	   {
	      points[j] = coords[i-1];
	      points[j+1] = coords[i];
	      j += 2;
	   }
	   *outsize = j;
	   return(points);
	}

	outcode1 = compute_outcode_2D(graphics, coords[0].x, coords[0].y);
	for (i = 1, j = 0; i < size; i++)
	{
	   outcode2 = compute_outcode_2D(graphics, coords[i].x, coords[i].y);
	   if ((outcode1 | outcode2) == X2D_CENTER)
	   {
	      points[j] = coords[i-1];
	      points[j+1] = coords[i];
	      j += 2;
	   }
	   else if ((outcode1 & outcode2) == X2D_CENTER)
	   {
	      clipped = clip_segment_2D(graphics, outcode1, outcode2,
					coords[i-1].x, coords[i-1].y,
					coords[i].x, coords[i].y,
					&points[j], &points[j+1]);
	      points[j].d = coords[i-1].d;
	      points[j+1].d = coords[i].d;

	      if ( clipped )
	         j += 2;
	   }
	   outcode1 = outcode2;
	}

	if (j == 0)
	{
	   kfree(points); *outsize = 0;
	   return(NULL);
	}
	*outsize = j;
	return(points);
}

/*-----------------------------------------------------------
|
|  Routine Name: _X2D_clip_polygon - Clips the world coordinate polygon
|
|       Purpose: Clips the world coordinate polygon such that
|		 the resulting polygon or set of polylines lie within
|		 the world coordinate viewing system.
|
|         Input: graphics - the graphics structure
|		 coords   - array that defines the polygon 
|		 size     - number points in the polygon to be clipped
|
|        Output: outsize  - the number of clipped polygon points
|
|       Returns: points   - the clipped polygon points
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

Coord  *_X2D_clip_polygon(
   X3DGraphics *graphics,
   Coord       *coords,
   int         size,
   int         *outsize)
{
	Coord   *points, tmp;
	int	i, j, outcode1, outcode2, clipped;


	if (graphics->clipping == FALSE)
	{
	   *outsize = size;
	   return(coords);
	}

	if ((points = (Coord *)kmalloc((unsigned)size*2*sizeof(Coord))) == NULL)
	{
	   kfprintf (kstderr,"_X2D_clip_lines:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return(NULL);
	}

	outcode1 = compute_outcode_2D(graphics, coords[0].x, coords[0].y);

	j = 0;
	if (outcode1 == X2D_CENTER)
	   points[j++] = coords[0];

	for (i = 1; i < size; i++)
	{
	   outcode2 = compute_outcode_2D(graphics, coords[i].x, coords[i].y);
	   if ((outcode1 | outcode2) == X2D_CENTER)
	   {
	      points[j++] = coords[i];
	   }
	   else if (outcode1 == X2D_CENTER)
	   {
	      points[j].d = coords[i].d;
	      clipped = clip_segment_2D(graphics, outcode1, outcode2,
					coords[i-1].x, coords[i-1].y,
					coords[i].x, coords[i].y,
					&tmp, &points[j]);
	      if ( clipped )
		j++;
	   }
	   else if ((outcode1 & outcode2) == X2D_CENTER)
	   {
	      points[j].d = coords[i-1].d;
	      points[j+1].d = coords[i].d;
	      clipped = clip_segment_2D(graphics, outcode1, outcode2,
					coords[i-1].x, coords[i-1].y,
					coords[i].x, coords[i].y,
					&points[j], &points[j+1]);
	      if ( clipped )
	        j += 2;
	   }
	   outcode1 = outcode2;
	}

	if (j == 0)
	{
	   kfree(points); *outsize = 0;
	   return(NULL);
	}
	*outsize = j;
	return(points);
}

/*-----------------------------------------------------------
|
|  Routine Name: _X2D_clip_rectangles - Clips the world coordinate rectangle
|
|       Purpose: Clips the world coordinate rectangle such that
|		 the resulting rectangle lies within the world
|		 coordinate viewing system.
|
|         Input: graphics - the graphics structure
|		 coords   - array that defines the rectangle
|		 size     - number of points in the rectangle to clipped
|
|        Output: outsize  - the number of clipped rectangle points
|
|       Returns: points   - the clipped rectangle points
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

Coord  *_X2D_clip_rectangles(
   X3DGraphics *graphics,
   Coord       *coords,
   int         size,
   int         *outsize)
{
	Coord   *points;
	int	i, j, outcode1, outcode2;


	if (graphics->clipping == FALSE)
	{
	   *outsize = size;
	   return(coords);
	}

	if ((points = (Coord *)kmalloc((unsigned)size*2*sizeof(Coord))) == NULL)
	{
	   kfprintf (kstderr,"_X2D_clip_lines:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return(NULL);
	}

	outcode1 = compute_outcode_2D(graphics, coords[0].x, coords[0].y);

	j = 0;
	if (outcode1 == X2D_CENTER)
	   points[j++] = coords[0];

	for (i = 1; i < size; i++)
	{
	   outcode2 = compute_outcode_2D(graphics, coords[i].x, coords[i].y);
	   if ((outcode1 | outcode2) == X2D_CENTER)
	   {
	      points[j++] = coords[i];
	   }
/*
	   else if (outcode1 == X2D_CENTER)
	   {
	      clip_point_2D(graphics, outcode2, coords[i].x, coords[i].y,
			&points[j]);
	      points[j++].d = coords[i].d;
	   }
*/
	   else if ((outcode1 & outcode2) == X2D_CENTER)
	   {
	      clip_point_2D(graphics, outcode1, coords[i-1].x, coords[i-1].y,
			&points[j]);
	      clip_point_2D(graphics, outcode2, coords[i].x, coords[i].y,
			&points[j+1]);
	      points[j].d = coords[i-1].d;
	      points[j+1].d = coords[i].d;
	      j += 2;
	   }
	   else if (outcode2 == X2D_TOP )
	   {
	      clip_point_2D(graphics, outcode2, coords[i].x, coords[i].y,
			&points[j]);
	      points[j++].d = coords[i].d;
	   }
	   outcode1 = outcode2;
	}

	if (j == 0)
	{
	   kfree(points); *outsize = 0;
	   return(NULL);
	}
	*outsize = j;
	return(points);
}
