/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>		Attribute Actions For Float Selections        <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>	     		xvf_get_float_gt0_sel()	              <<<<
   >>>>	     		xvf_set_float_gt0_sel()	              <<<<
   >>>>	     		xvf_get_float_ge0_sel()	              <<<<
   >>>>	     		xvf_set_float_ge0_sel()	              <<<<
   >>>>	     		xvf_get_float_lt0_sel()	              <<<<
   >>>>	     		xvf_set_float_lt0_sel()	              <<<<
   >>>>	     		xvf_get_float_le0_sel()	              <<<<
   >>>>	     		xvf_set_float_le0_sel()	              <<<<
   >>>>	     		xvf_get_float_upper()		      <<<<
   >>>>	     		xvf_set_float_upper()		      <<<<
   >>>>	     		xvf_get_float_lower()		      <<<<
   >>>>	     		xvf_set_float_lower()		      <<<<
   >>>>	     		xvf_get_float_def()		      <<<<
   >>>>	     		xvf_set_float_def()		      <<<<
   >>>>	     		xvf_get_float_val()		      <<<<
   >>>>	     		xvf_set_float_val()		      <<<<
   >>>>	     		xvf_get_precision()		      <<<<
   >>>>	     		xvf_set_precision()		      <<<<
   >>>>                                                       <<<<
   >>>>	     		xvf_change_floatbounds()	      <<<<
   >>>>   Static:                                             <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "internals.h"
#include "attributes.h"


/*-----------------------------------------------------------
|
|       Routine: xvf_get_float_gt0_sel
|       Purpose: Gets the boolean TRUE or FALSE depending on whether
|                or not the bounds of an float selection are > 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_FLOAT_GT0_SEL ("xvf_float_gt0_sel")
|        Output: calldata    - passes back TRUE if bounds > 0 set
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_float_gt0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_float_gt0_sel(kformstruct, KVF_FLOAT_GT0_SEL, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_float_gt0_sel
|       Purpose: Sets the bounds of an float selection to > 0
|                according to the value of the "float_gt0" 
|                selection on the pop-up menuform.
|
|                menuform field => current selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_FLOAT_GT0_SEL ("xvf_float_gt0_sel")
|                calldata    - provides value of TRUE if bounds > 0 are set
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_float_gt0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (!(kvf_set_float_gt0_sel(kformstruct, KVF_FLOAT_GT0_SEL, calldata)))
	    return(FALSE);

	xvf_change_floatbounds(kformstruct, "gt0");

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);
	
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_float_ge0_sel
|       Purpose: Gets the boolean TRUE or FALSE depending on whether
|                or not the bounds of an float selection are >= 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_FLOAT_GE0_SEL ("xvf_float_ge0_sel")
|        Output: calldata    - passes back TRUE if bounds are set >= 0
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_float_ge0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_float_ge0_sel(kformstruct, KVF_FLOAT_GE0_SEL, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_float_ge0_sel
|       Purpose: Sets the bounds of an float selection to >= 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_FLOAT_GE0_SEL ("xvf_float_ge0_sel")
|                calldata    - provides boolean value of TRUE if bounds are >= 0
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_float_ge0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (!(kvf_set_float_ge0_sel(kformstruct, KVF_FLOAT_GE0_SEL, calldata)))
	    return(FALSE);

	xvf_change_floatbounds(kformstruct, "ge0");

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);
	
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_float_lt0_sel
|       Purpose: Gets the boolean TRUE or FALSE depending on whether
|                or not the bounds of an float selection are < 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_FLOAT_LT0_SEL ("xvf_float_lt0_sel")
|        Output: calldata    - passes back TRUE if bounds are set to < 0
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_float_lt0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_float_lt0_sel(kformstruct, KVF_FLOAT_LT0_SEL, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_float_lt0_sel
|       Purpose: Sets the bounds of an float selection to < 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_FLOAT_LT0_SEL ("xvf_float_lt0_sel")
|                calldata    - provides boolean value of TRUE if bounds are < 0
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_float_lt0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (!(kvf_set_float_lt0_sel(kformstruct, KVF_FLOAT_LT0_SEL, calldata)))
	    return(FALSE);

	xvf_change_floatbounds(kformstruct, "lt0");

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);
	
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_float_le0_sel
|       Purpose: Gets the boolean TRUE or FALSE depending on whether
|                or not the bounds of an float selection are <= 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_FLOAT_LE0_SEL ("xvf_float_le0_sel")
|        Output: calldata    - passes back TRUE if bounds are set to <= 0
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_float_le0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_float_le0_sel(kformstruct, KVF_FLOAT_LE0_SEL, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_float_le0_sel
|       Purpose: Sets the bounds of an float selection to <= 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_FLOAT_LE0_SEL ("xvf_float_le0_sel")
|                calldata    - provides boolean value of TRUE if bounds are <= 0
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_float_le0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (!(kvf_set_float_le0_sel(kformstruct, KVF_FLOAT_LE0_SEL, calldata)))
	    return(FALSE);

	xvf_change_floatbounds(kformstruct, "le0");

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_float_upper
|       Purpose: Gets the value of the upper bound for an float,
|                assuming that "xvf_float_upper_sel" is TRUE
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_FLOAT_UPPER ("xvf_float_upper")
|        Output: calldata    - passes back float upper bound value
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_float_upper(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_float_upper(kformstruct, KVF_FLOAT_UPPER, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_float_upper
|       Purpose: Sets the value of the upper bound for an float,
|                assuming that "xvf_float_upper_sel" is TRUE
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_FLOAT_UPPER ("xvf_float_upper")
|                calldata    - provides float upper bound value
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_float_upper(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (!(kvf_set_float_upper(kformstruct, KVF_FLOAT_UPPER, calldata)))
	    return(FALSE);

	xvf_change_floatbounds(kformstruct, "bounds");

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_float_lower
|       Purpose: Gets the value of the lower bound for an float,
|                assuming that "xvf_float_lower_sel" is TRUE
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_FLOAT_LOWER ("xvf_float_lower")
|        Output: calldata    - passes back float lower bound value
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_float_lower(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_float_lower(kformstruct, KVF_FLOAT_LOWER, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_float_lower
|       Purpose: Sets the value of the lower bound for an float,
|                assuming that "xvf_float_lower_sel" is TRUE
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_FLOAT_LOWER ("xvf_float_lower")
|                calldata    - provides float lower bound value
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_float_lower(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (!(kvf_set_float_lower(kformstruct, KVF_FLOAT_LOWER, calldata)))
	    return(FALSE);

	xvf_change_floatbounds(kformstruct, "bounds");

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: xvf_get_float_def
|       Purpose: Gets the value of the default for an float
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_FLOAT_DEF ("xvf_float_def")
|        Output: calldata    - passes back float default value
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_float_def(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_float_def(kformstruct, KVF_FLOAT_DEF, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_float_def
|       Purpose: Sets the value of the default for an float
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_FLOAT_DEF ("xvf_float_def")
|                calldata    - provides float default value
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_float_def(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	float *float_def;

	if (!(kvf_set_float_def(kformstruct, KVF_FLOAT_DEF, calldata)))
	    return(FALSE);

	float_def = (float *) calldata;
	if (!(xvf_set_float_val(kformstruct, XVF_FLOAT_VAL, float_def)))
	    return(FALSE);

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: xvf_get_float_val
|       Purpose: Gets the value of the float
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_FLOAT_VAL ("xvf_float_val")
|        Output: calldata    - passes back float value
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_float_val(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_float_val(kformstruct, KVF_FLOAT_VAL, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_float_val
|       Purpose: Sets the value of an float
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_FLOAT_VAL ("xvf_float_val")
|                calldata    - provides float value
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_float_val(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	kselection   *selection;
	char         temp[KLENGTH], scale[KLENGTH];
	Line_Info    line_info;

	if (!(kvf_set_float_val(kformstruct, KVF_FLOAT_VAL, calldata)))
	    return(FALSE);

	selection = kformstruct->Selptr;

	if ((selection->value_object != NULL) ||
            (selection->scroll != NULL))
	{
	    kvf_clear_line_info(&line_info);
	    kvf_parse_float_line(selection->line, &line_info);

	    if (line_info.precision == 0)
	        ksprintf(scale, "%%g");
	    else if (line_info.precision == -1)
	        ksprintf(scale, "%%f");
	    else ksprintf(scale, "%%.%df", line_info.precision);
	    ksprintf(temp, scale, line_info.float_val);

	    if (selection->value_object != NULL)
                xvw_set_attribute(selection->value_object, 
				  XVW_TEXT_STRING, temp);

	    if (selection->scroll != NULL)
            {
                xvw_set_attributes(selection->scroll,
                                   XVW_SCROLL_MIN,   line_info.lower_float,
                                   XVW_SCROLL_MAX,   line_info.upper_float,
                                   XVW_SCROLL_VALUE, line_info.float_val,
                                   NULL);
            }
	    kvf_free_line_info_strings(&line_info);
	}


	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvf_change_floatbounds
|
|       Purpose: Changes the bounds for a float selection
|
|         Input: kformstruct - generic kform_struct pointer
|		 string      - string rep of float bounds, may be one of:
|			      "gt0"        : bounds >  0
|			      "ge0"        : bounds >= 0
|			      "lt0"        : bounds <  0
|			      "le0"        : bounds <= 0
|                             "lower = %f" : lower bound 
|                             "upper = %f" : upper bound 
|			      "nobounds"   : no bounds 
|
|        Output: Returns TRUE on success, FALSE on Failure
|          Date: March 20, 1994 
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/

int xvf_change_floatbounds(
   kform_struct *kformstruct,
   char         *string)
{
	kselection *selection;
	Line_Info  line_info;
	int        scrollbar = FALSE;
	float      sb_width;
	char       temp[KLENGTH], scale[KLENGTH];

	kvf_clear_line_info(&line_info);

	selection = kformstruct->Selptr;
        kvf_parse_float_line(selection->line, &line_info);

	if (kstrcmp(string, "bounds") == 0)
	   scrollbar = TRUE;

	/*
         *  take off scrollbar if there is one
         */
        if ((!scrollbar) && (selection->scroll != NULL))
	{
             xvw_set_attribute(selection->back, XVF_MECHANISM, 0);
        }

        else if ((scrollbar) && (selection->scroll == NULL))
        {
            sb_width = xvf_calculate_sb_width(line_info.width,
                                line_info.optional, line_info.opt_sel,
                                line_info.title, (double) line_info.upper_float,
                                (double) line_info.lower_float,
                                line_info.precision, line_info.typeflag);
            if (sb_width >= 1.0)
                xvw_set_attribute(selection->back, XVF_MECHANISM, 1);
        }

	if (selection->value_object != NULL)
	{
	    if (line_info.precision == 0)
	        ksprintf(scale, "%%g");
	    else if (line_info.precision == -1)
	        ksprintf(scale, "%%f");
	    else ksprintf(scale, "%%.%df", line_info.precision);
	    ksprintf(temp, scale, line_info.float_val);
	    xvw_set_attribute(selection->value_object, XVW_TEXT_STRING, temp);
	}

	kvf_free_line_info_strings(&line_info);

	return(TRUE);
}

