 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";

#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Utilities for Data Services.
   >>>>
   >>>>   Static:
   >>>>
   >>>>  Private:
   >>>>                 _kdms_get_order
   >>>>                 _kdms_set_order
   >>>>
   >>>>                 _kdms_buffer_offset
   >>>>                 _kdms_buffer_index
   >>>>
   >>>>                 _kdms_num_elements
   >>>>                 _kdms_in_buffer
   >>>>                 _kdms_pres2phys
   >>>>
   >>>>                 _kdms_translate_coords
   >>>>                 _kdms_transpose_coords
   >>>>                 _kdms_match_coords
   >>>>
   >>>>                 _kdms_legal_datatype
   >>>>                 _kdms_legal_size
   >>>>                 _kdms_legal_order
   >>>>                 _kdms_legal_vector_def
   >>>>
   >>>>                 _kdms_segment_size
   >>>>
   >>>>                 _kdms_in_space
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <float.h>
extern int _kdms_buffer_threshold;

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_get_order
|
|       Purpose: Extract the value of an element in the info 
|                array which corresponds to the direction 
|                specified by the direction argument with respect to 
|                the index order indicated by the cur_order array
|                argument. 
|
|                Example :
|                  
|                Here, the info array represents the size
|                of the value segment which is ordered in 
|                the order specifed by the cur_order array. 
|                The dimensionality is 5.
|
|       !               direction   = KWIDTH; 
|       !               cur_order   = { KHEIGHT,
|       !                               KWIDTH,
|       !                               KDEPTH,
|       !                               KTIME,
|       !                               KELEMENTS } 
|       !               info        = { 5, 3, 2, 1, 17}
|       !               dim         = 5;
|       !               default     = 45;
|
|                The direction argument specifies the element of 
|                the info array to retrieve.  That is, the element 
|                retrieved will be the one whose index order (according 
|                to the cur_order array) matches the direction argument. 
|           
|                So, for these arguments, the value '3' will be returned.
|
|                If the direction argument had been 'invalid', that is,
|                not equal to any of the directions in the cur_order  
|                array, then the value specified by the default argument 
|                will be returned.
|               
|               
|         Input: direction - direction specifier
|               
|                cur_order - the current index order exhibited by the
|                            elements of the info array
|               
|                info      - the data which you want to extract an element 
|                            from that is in the index order specified 
|                            by cur_order array
|               
|                dim       - dimensionality of the cur_order and info arrays
|               
|                def       - if direction is not one of members of the
|                            cur_order array, this value will be returned.
|
|        Output: none
|
|       Returns: returns the value of the element in the info array
|                whose corresponding "direction" in the cur_order array 
|                matches the requested direction. 
|
|    Written By: Jeremy Worley
|          Date: Feb 08, 1993 09:16
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_get_order(int direction, int *cur_order, int *info, int dim, int def)
{
   int i;

   for (i = 0; i < dim; i++)
      if (direction == cur_order[i])
         return (info[i]);

   return (def);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_set_order
|
|       Purpose: Inserts a value into the info array at the element 
|                position corresponding to the direction specified 
|                by the direction argument with respect to the index 
|                order indicated by the cur_order array argument.
|
|                Example :
|
|                Here, the info array represents the size
|                of the value segment which is ordered in
|                the order specifed by the cur_order array.
|                The dimensionality is 5.
|
|       !               direction   = KWIDTH;
|       !               cur_order   = { KHEIGHT,
|       !                               KWIDTH,
|       !                               KDEPTH,
|       !                               KTIME,
|       !                               KELEMENTS }
|       !               info        = { 5, 3, 2, 1, 17}
|       !               dim         = 5;
|       !               val         = 45;
|
|                The direction argument specifies the element of
|                the info array to insert the value into.  That is, 
|                the element position will be the one whose 
|                index order (according to the cur_order array) matches 
|                the direction argument.
|
|                So, for these arguments, the resulting info array will be
|
|       !               info        = { 5, 45, 2, 1, 17}
|
|
|         Input: direction - direction specifier
|
|                cur_order - the current index order exhibited by the
|                            elements of the info array
|
|                info      - the array in which you want to insert a value 
|                            at the element position that is in the 
|                            index order specified by the cur_order array.
|
|                dim       - dimensionality of the cur_order and info arrays
|
|                val       - value to insert
|
|        Output: Potentially changes a value in the info array.
|
|       Returns: TRUE (1) all the time. Doesn't it make you feel 
|                good to know that you can rely on this routine?
|
|    Written By: Jeremy Worley
|          Date: Feb 24, 1993 10:25
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_set_order(
                  int direction,
                  int *cur_order,
                  int *info,
                  int dim,
                  int val)
{
   int i;

   for (i = 0; i < dim; i++)
      if (direction == cur_order[i])
         info[i] = val;

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_buffer_offset
|
|       Purpose: This routine calculates the offset in
|                data units (i.e. usually 4 bytes units for
|                floats) from the beginning of the segment.
|
|         Input: pos    - position to use for calculation
|                dim    - size to use for calculation
|                order  - index order for calculation
|
|        Output: Nothing
|
|       Returns: The number of units to the specified
|                position.
|
|    Written By: Jeremy Worley
|          Date: Nov 11, 1992 13:31
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_buffer_offset(int *pos, int *size, int dim, int datatype)
{
   int p;
   int i;
   int t;

   /*
    * FIX ME Note that this currently does not take into consideration index
    * order issues.
    */
   if (datatype == KBIT)
   {
      t = (size[0] + 7) >> 3;
      p = (pos[0] + 7) >> 3;
      for (i = 1; i < dim; i++, t *= size[i - 1])
	 p += pos[i] * t;
   }
   else
   {
      for (i = 0, t = 1, p = 0; i < dim; i++, t *= size[i - 1])
	 p += pos[i] * t;
   }
   
   return (p * kdata_size(datatype));
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_num_elements
|
|       Purpose: This routine computes the number of elements
|                in the between the two positions.  It assumes
|                that the two positions describe a 5D cube
|                so dimensionality is implied by the differences.
|
|         Input: x0 - upper-left-front-earlier corner
|                x1 - lower-right-rear-later corner
|
|        Output: nothing
|
|       Returns: Returns the number of points in that subspace
|
|    Written By: Jeremy Worley
|          Date: Oct 07, 1992 10:22
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_num_elements(int *x0, int *x1, int dim)
{
   int i,
      t;

   for (i = 0, t = 1; i < dim; i++)
      t *= x1[i] - x0[i] + 1;

   return (kabs(t));
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_in_buffer
|
|       Purpose: This routine takes a specified region
|                and determines if it is completely
|                contained in the specified segment data.
|
|         Input: dsegment - segment possibly containing
|                           the data.
|                begin    - front corner of data.
|                end      - back corner of data.
|
|        Output: nothing
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 23, 1992 11:30
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_in_buffer(kdsegment * dsegment, int *begin, int *end)
{
   int i;

   if (dsegment->data == NULL)
   {
      return (FALSE);           /* not  an error */
   }

   for (i = 0; i < dsegment->dimension; i++)
      if (begin[i] < dsegment->begin[i] || end[i] > dsegment->end[i])
         return (FALSE);        /* not an error */

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_pres2phys
|
|       Purpose: Translates a presentation position into
|                a physical position.
|
|         Input:
|
|        Output:
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 30, 1992 20:51
| Modifications:
|
------------------------------------------------------------*/

#define Round(x) (x >= 0.0 ? (kfloor(x + 0.5)) : kfloor(x))
int
_kdms_pres2phys(
                  kdsegment * segment,
                  kpresentation * presentation,
                  int *pres,
                  int *phys,
                  int its_a_size)
{
   double stride[KDMS_MAX_DIM];
   int i,
      tmp1[KDMS_MAX_DIM],
      tmp2[KDMS_MAX_DIM];

   for (i = 0; i < KDMS_MAX_DIM; i++)
      tmp1[i] = _kdms_get_order(segment->order[i], presentation->order,
                            presentation->size, presentation->dimension, 1);

   if (presentation->interp != KPAD)
      for (i = 0; i < segment->dimension; i++)
         stride[i] = (double)segment->size[i] / (double)tmp1[i];
   else
      for (i = 0; i < KDMS_MAX_DIM; i++)
         stride[i] = 1.0;

   for (i = 0; i < KDMS_MAX_DIM; i++)
      tmp2[i] = _kdms_get_order(segment->order[i], presentation->order,
                                pres, presentation->dimension, 1);

   if (its_a_size)
      for (i = 0; i < KDMS_MAX_DIM; i++)
	 phys[i] = (stride[i] < 1.0) ?
	    (int)kceil((double)tmp2[i] * stride[i]) :
            (int)kfloor((double)tmp2[i] * stride[i]);
   else
      for (i = 0; i < KDMS_MAX_DIM; i++)
	 phys[i] = (int)kfloor((double)((tmp2[i]) * stride[i])) ;
   
   
   return (TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: _kdms_translate_coords
|
|       Purpose: Translates a n-tuple from one index order
|                and size to its equivelant in another
|                order and size.
|
|         Input:
|
|        Output:
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 30, 1992 20:51
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_translate_coords(
                         int *srcsize,
                         int *srcorder,
                         int *src,
                         int *destsize,
                         int *destorder,
                         int *dest,
                         int dim,
                         int def,
                         int is_a_size)
{
   double stride[KDMS_MAX_DIM];
   int tmp1[KDMS_MAX_DIM];
   int tmp2[KDMS_MAX_DIM];
   int i;

   if (src == NULL)
   {
      for (i = 0; i < dim; i++)
         dest[i] = def;
      return (TRUE);
   }

   for (i = 0; i < dim; i++)
      tmp1[i] = _kdms_get_order(destorder[i], srcorder, srcsize, dim, def);

   for (i = 0; i < dim; i++)
      stride[i] = (double)destsize[i] / (double)tmp1[i];

   for (i = 0; i < dim; i++)
      tmp2[i] = _kdms_get_order(destorder[i], srcorder, src, dim, KNONE);

   if (is_a_size)
   {
      
      for (i = 0; i < dim; i++)
	 dest[i] = (int)kmin(destsize[i],kceil((double)tmp2[i] * stride[i]));
   }
   else
   {
      
      for (i = 0; i < dim; i++)
	 dest[i] = (int)((double)tmp2[i] * stride[i]);
   }
   
   return (TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: _kdms_transpose_coords
|
|       Purpose: Rotates the 5-tuple from one index order
|                to another index order.
|
|         Input:
|
|        Output:
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 30, 1992 20:51
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_transpose_coords(
                         int *srcorder,
                         int *src,
                         int *destorder,
                         int *dest,
                         int dim,
                         int def)
{
   int i;

   for (i = 0; i < dim; i++)
      dest[i] = _kdms_get_order(destorder[i], srcorder, src, dim, def);

   for (i = dim; i < KDMS_MAX_DIM; i++)
      dest[i] = _kdms_get_order(destorder[i], srcorder, src, dim, def);

   return (TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: _kdms_match_coords
|
|       Purpose: Matches two 5-tuples
|
|         Input:
|
|        Output:
|
|       Returns: TRUE (1) on match, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 30, 1992 20:51
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_match_coords(
                     int *arg1,
                     int *arg2,
                     int dim)
{
   int i;

   for (i = 0; i < dim; i++)
      if (arg1[i] != arg2[i])
         return (FALSE);        /* not an error */

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_initialized
|
|       Purpose: This routine returns true if the resource
|                structure has been initialized, and false
|                otherwise.
|
|         Input:
|
|        Output:
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: May 10, 1993 14:53
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_initialized(kobject object)
{
   return (object->phys->resources != NULL);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_preset_data
|
|       Purpose: This function is called whenever you want a
|                chunk of data set to a constant value.
|
|         Input:
|
|        Output:
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: May 17, 1993 11:38
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_preset_data(kaddr data, int type, int num, double rval, double ival)
{
   int i;

   switch ((int)type)
   {
      case KBYTE:
         {
            signed char *tmp = (signed char *)data;
            signed char val = (signed char)rval;

            for (i = 0; i < num; i++)
               *tmp++ = val;
            return (TRUE);
         }
      case KUBYTE:
         {
            unsigned char *tmp = (unsigned char *)data;
            unsigned char val = (unsigned char)rval;

            for (i = 0; i < num; i++)
               *tmp++ = val;
            return (TRUE);
         }
      case KSHORT:
         {
            short *tmp = (short *)data;
            short val = (short)rval;

            for (i = 0; i < num; i++)
               *tmp++ = val;
            return (TRUE);
         }
      case KUSHORT:
         {
            unsigned short *tmp = (unsigned short *)data;
            unsigned short val = (unsigned short)rval;

            for (i = 0; i < num; i++)
               *tmp++ = val;
            return (TRUE);
         }
      case KINT:
         {
            int *tmp = (int *)data;
            int val = (int)rval;

            for (i = 0; i < num; i++)
               *tmp++ = val;
            return (TRUE);
         }
      case KUINT:
         {
            unsigned int *tmp = (unsigned int *)data;
            unsigned int val = (unsigned int)rval;

            for (i = 0; i < num; i++)
               *tmp++ = val;
            return (TRUE);
         }
      case KLONG:
         {
            long *tmp = (long *)data;
            long val = (long)rval;

            for (i = 0; i < num; i++)
               *tmp++ = val;
            return (TRUE);
         }
      case KULONG:
         {
            unsigned long *tmp = (unsigned long *)data;
            unsigned long val = (unsigned long)rval;

            for (i = 0; i < num; i++)
               *tmp++ = val;
            return (TRUE);
         }
      case KFLOAT:
         {
            float *tmp = (float *)data;
            float val = (float)rval;

            for (i = 0; i < num; i++)
               *tmp++ = val;
            return (TRUE);
         }
      case KDOUBLE:
         {
            double *tmp = (double *)data;
            double val = (double)rval;

            for (i = 0; i < num; i++)
               *tmp++ = val;
            return (TRUE);
         }
      case KCOMPLEX:
         {
            kcomplex *tmp = (kcomplex *) data;

            for (i = 0; i < num; i++)
               *tmp++ = kccomp((float)rval, (float)ival);
            return (TRUE);
         }
      case KDCOMPLEX:
         {
            kdcomplex *tmp = (kdcomplex *) data;

            for (i = 0; i < num; i++)
               *tmp++ = kdccomp(rval, ival);
            return (TRUE);
         }
      default:
         _kdms_set_error(KDMS_EINTERNAL);
         return (FALSE);
   }
}



/*-----------------------------------------------------------
|
|  Routine Name: _kdms_legal_datatype
|
|       Purpose: This function determines if the datatype
|                specified is legal.
|
|                Eventually, this oughta check the file format
|                to verify that the data type is representable.
|
|         Input: object - this contains the format information
|                         (hopefully)
|                 pres  - presentation to check data type
|                         on.
|
|        Output:
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: May 20, 1993 11:23
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_legal_datatype(int datatype)
{
   switch (datatype)
   {
      case KBIT:
      case KBYTE:
      case KUBYTE:
      case KSHORT:
      case KUSHORT:
      case KLONG:
      case KULONG:
      case KINT:
      case KUINT:
      case KFLOAT:
      case KDOUBLE:
      case KCOMPLEX:
      case KDCOMPLEX:
      case KSTRUCT:
         return (TRUE);
      default:
         return (FALSE);        /* not an error */
   }
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_legal_size
|
|       Purpose: Determines if the size of the segment
|                indicated by the presentation is legal.
|
|         Input: object - object to check
|                pres   - presentation to check
|
|        Output:
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: May 20, 1993 14:48
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_legal_size(int *size, int dim)
{
   int i;

   for (i = 0; i < dim; i++)
      if (size[i] <= 0)
         return (FALSE);        /* not an error */

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_legal_order
|
|       Purpose: Determines if the index order of the specified
|                segment is legal.
|
|         Input: object - object to check index order on.
|                pres   - presentaiton to check index order on.
|
|        Output:
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: May 20, 1993 14:51
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_legal_order(int *order, int dim)
{
   int i,
      j;

   /*
    * check all combinations to make sure that there is no way that two of
    * the index orders are the same.
    */
   for (i = 0; i < dim - 1; i++)
      for (j = i + 1; j < dim; j++)
         if ((order[i] == order[j] && order[i] != KNONE) || order[i] == -1)
            return (FALSE);     /* not an error */
	 
   if (dim == 1 && order[i] == -1)
      return FALSE;
	 
   /*
    * they be legal
    */
   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_segment_size
|
|       Purpose: Returns the number of bytes in the data
|                set associated with the segment.
|
|         Input:
|
|        Output:
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: May 26, 1993 08:35
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_segment_size(kpresentation * pres)
{
   int tb[KDMS_MAX_DIM],
      te[KDMS_MAX_DIM],
      i;

   for (i = 0; i < pres->dimension; i++)
   {
      tb[i] = 0;
      te[i] = pres->segment->size[i] - 1;
   }

   return (_kdms_num_elements(tb, te, pres->dimension) *
           kdata_size(pres->segment->datatype));
}


/*-----------------------------------------------------------
|
|  Routine Name: _kdms_in_space
|
|       Purpose: This routine takes a specified region
|                and determines if it is even *partially*
|                contained in the specified segments data space.
|
|         Input: dsegment - segment possibly containing
|                           the data.
|                begin    - front corner of data.
|                end      - back corner of data.
|
|        Output: nothing
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 23, 1992 11:30
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_in_space(kdsegment * dsegment, int *begin, int *end)
{
   int i;

   for (i = 0; i < dsegment->dimension; i++)
   {
      if (begin[i] >= dsegment->size[i] || end[i] < 0)
         return (FALSE);        /* not an error */
   }

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_in_data_space
|
|       Purpose: This routine takes a specified region
|                and determines if it is TOTALLY
|                contained in the specified segments data space.
|
|         Input: dsegment - segment possibly containing
|                           the data.
|                begin    - front corner of data.
|                end      - back corner of data.
|
|        Output: nothing
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 23, 1992 11:30
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_in_pres_data_space(kpresentation * pres, int *begin, int *end)
{
   int i;

   for (i = 0; i < pres->dimension; i++)
   {
      if (begin[i] >= pres->size[i] || end[i] < 0 ||
          end[i] >= pres->size[i] || begin[i] < 0)
         return (FALSE);        /* not an error */
   }

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_in_data_space
|
|       Purpose: This routine takes a specified region
|                and determines if it is TOTALLY
|                contained in the specified segments data space.
|
|         Input: dsegment - segment possibly containing
|                           the data.
|                begin    - front corner of data.
|                end      - back corner of data.
|
|        Output: nothing
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 23, 1992 11:30
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_in_phys_data_space(kdsegment * dsegment, int *begin, int *end)
{
   int i;

   for (i = 0; i < dsegment->dimension; i++)
   {
      if (begin[i] >= dsegment->size[i] || end[i] < 0 ||
          end[i] >= dsegment->size[i] || begin[i] < 0)
         return (FALSE);        /* not an error */
   }

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_position_in_space
|
|       Purpose: 
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Aug 31, 1993 14:25
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_position_in_space(int *pos, int *space, int dim)
{
   int i;

   for (i = 0; i < dim; i++)
      if (pos[i] < 0 || pos[i] >= space[i])
         return (FALSE);        /* not an error */

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_get_buffer_threshold
|       Purpose: Returns the buffer threshold that should
|		 be used when fiddling with the transport
|		 or when mucking with an optimal region.
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Jul 12, 1994 15:22
| Modifications:
|
------------------------------------------------------------*/

int 
_kdms_get_buffer_threshold(kpresentation *pres)
{
   return(pres->segment->buffer_threshold >= 0 ?
	  pres->segment->buffer_threshold : _kdms_buffer_threshold);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_compute_chunk_size
|
|       Purpose: This routine determines the largest reasonable
|                chunk of data to get and put in the copy_data
|                operation.  
|
|                This routine will only return whole axes, i.e.
|                if the size is 10x10x10, and the maximum
|                reasonable size is 200, then the resulting
|                size will be 10x10x1...
|
|         Input:
|        Output:
|       Returns: The number of dimensions that are spanned by the
|                new chunk.
|    Written By: Jeremy Worley
|          Date: Mar 10, 1994 08:56
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_compute_chunk_size(
   int *size, 
   int dimension, 
   int *chunk, 
   int datatype,
   int buffer_threshold)
{
   int span = 0;
   int volume = kdata_size(datatype);
   int i;

   /*
    * if the datatype is 0 then the data type has not been properly
    * initialized.  In that case, generate...
    */
   if (datatype == 0)
   {
      kerror("kdataman","_kdms_compute_chunk_size","Improperly initialized "
	     "datatype.\n");
      return FALSE;
   }
   
   /*
    * for various reasons, the real buffer threshold should be at least
    * one unit of the data type specfied, or all hell breaks loose.
    */
   if (buffer_threshold < 0)
      buffer_threshold = kmax(_kdms_buffer_threshold, kdata_size(datatype));
   
   /*
    * optimize the chunk to get back...
    */
   while (volume < buffer_threshold && span < dimension)
   {
      chunk[span] = size[span];
      volume *= chunk[span++];
   }

   /*
    * if the loop above resulted in a chunk that exceeds our maximum
    * size, then adjust the chunk size back downward and subtract 1
    * off the span, which indicates the highest index in which the
    * chunk fully spans the size of the segment.
    */
   if (volume > buffer_threshold)
   {
      span--;
      volume /= chunk[span];
      chunk[span] = 1;
   }

   for (i = span; i < dimension; i++)
      chunk[i] = 1;

   if (span >= dimension)
      return(span);

   /*
    * a great idea from Donna, Mark, and Steve K.
    */
   for (i = 2; i < 100; i++) 
      if (size[span] % i == 0 && 
	  volume * (size[span] / i) <= buffer_threshold) 
      {
	 chunk[span] = size[span] / i;
	 return(span);
      }
	 
   return (span);
}


/*-----------------------------------------------------------
|
|  Routine Name: _kdms_advance_position
|
|       Purpose: This routine advances a given position.  THERE 
|		 IS A VARIANT OF THIS FUNCTION IN KAPPSERV...CODE 
|		 REUSE THROUGH DUPLICATION :^)
|
|		 The position will be incremented by the
|		 given increment unit.  It will optionally
|		 either stop at the end of the data set,
|		 or wrap back around to the beginning of
|		 the data set.
|
|         Input: pos  - current position
|		 size - size of data set
|		 unit - unit size to increment by
|                dim  - the dimension of the position
|                wrap - TRUE if position should wrap back 
|                       to the beginning
|
|        Output: position is advanced in place
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Oct 07, 1993 13:18
| Modifications:
|
------------------------------------------------------------*/
int 
_kdms_advance_position(
   int *begin, 
   int *end,
   int *size, 
   int *unit, 
   int  dim)
{
   int i;

   /*
    * increment to the beginning of the next unit of data... also
    * better check to see if we need to advance the pointer throughout
    * the data segment.  If we are at the end of plane then we need to
    * advance the data segment depth pointer, etc.  This scenario in
    * propagated all the way down to the element number. Where at last
    * the "eof" indicator is set.
    */

   for (i = 0; i < dim - 1; i++)
   {
      begin[i] += unit[i];
      end[i] += unit[i];
      if (begin[i] < size[i])
	 return(TRUE);
      else
      {
	 begin[i] = 0;
	 end[i] = unit[i] - 1;
      }
   }
   begin[dim - 1] += unit[dim - 1];
   end[dim - 1] += unit[dim - 1];

   if (begin[dim - 1] >= size[dim - 1])
      return(FALSE);

   return (TRUE);
}
