 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Functions which would be better
   >>>>               implemented in kappserv
   >>>>
   >>>>		   The hope is to move these there over time...
   >>>>		  
   >>>>
   >>>>   Static:
   >>>>                 _kdms_copy_segment_data()
   >>>>
   >>>>  Private:
   >>>>                 _kdms_copy_data()
   >>>>                 _kdms_copy_remaining_data()
   >>>>                 
   >>>>   Public:
   >>>>                 kdms_copy_segment_data()
   >>>>                 kdms_copy_object()
   >>>>                 kdms_copy_remaining_data()
   >>>>                                  
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

extern int _kdms_compiled;

/*-----------------------------------------------------------
|
|  Routine Name: (static) _kdms_copy_segment_data
|
|       Purpose: copies data in a segment
|
|         Input:
|
|        Output:
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Mar 09, 1993 16:32
| Modifications:
|
------------------------------------------------------------*/

static int
_kdms_copy_segment_data(
                          kobject src_obj,
                          int src_segment,
                          kobject dest_obj,
                          int dest_segment)
{
   kpresentation *pres1;
   kpresentation *pres2;
   int transferable;
   
   /*
    * copy data associated with the segment
    */
   if (!_kdms_get_segment_info(src_obj, src_segment, &pres1) || !pres1)
      return (FALSE);

   if (!kdms_get_attribute(src_obj, ktoken_to_string(src_segment), 
			   KDMS_TRANSFERABLE, &transferable))
      return (FALSE);

   /*
    * if the segment is not transferable, then drop out now.
    */
   if (!transferable)
      return (TRUE);
   
   if (!_kdms_get_segment_info(dest_obj, dest_segment, &pres2) || !pres2)
   {
      if (!_kdms_create_segment(dest_obj, dest_segment, NULL))
         return (FALSE);
      if (!_kdms_get_segment_info(dest_obj, dest_segment, &pres2) || !pres2)
         return (FALSE);
   }

   _kdms_establish_segment(src_obj, src_segment, dest_obj, dest_segment);
   _kdms_update_segment(dest_obj, pres2);
   if (pres1->segment->update == 1)
      _kdms_update_segment(src_obj, pres1);

   return (_kdms_migrate_segment_data(src_obj, pres1, dest_obj, pres2));
}

/*-----------------------------------------------------------
|
|  Routine Name: kdms_copy_data - copy a segment's data from
|                one abstract data object to another.
|
|       Purpose: kdms_copy_data is part of the Data Management
|                Services layer of the Distributed Data
|                Services Toolkit.  This function copies
|                the desired data from an input object to
|                an output object.
|
|                This function copies an entire segment
|                from one object to another object.  This
|                includes physical attributes and data,
|                but does not necessarily include any
|                presentation related attributes.
|
|                If the segment does not currently exist in
|                the destination object, then the physical
|                and are copied from the source to the
|                destination.  These attributes are then
|                used to define the presentation attributes.
|                If the segment already exists in the
|                destination object, then the destination's
|                current segment data is removed and
|                replaced by the source's segment data.
|                In this case, the presentation attributes
|                in the source object's segment are not
|                acquired by the destination object's
|                segment->
|
|                If the segment argument is NULL, then
|                this instructs the kdms_copy_data function
|                to copy the data for all segments to the
|                destination object.
|
|         Input: src_obj - the object that serves as a source
|                          for the segment data.
|                segment - the data segment to be copied.
|
|        Output: dest_obj - the object that will serve as
|                           a destination in the copy
|                           operation.
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|  Restrictions:
|    Written By: Mark Young and Jeremy Worley
|          Date: Aug 20, 1992 15:51
|      Verified:
|  Side Effects:
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_copy_data(kobject src_obj, int segment, kobject dest_obj, int dest_segment)
{
   klist *list;

   /*
    * set pointers and positions in new to indicate how to copy later.
    */
   if (segment != 0)
   {
      return (_kdms_copy_segment_data(src_obj, segment, dest_obj, dest_segment));
   }
   else
   {
      list = src_obj->presentations;
      while (list != NULL)
      {
         if (!_kdms_copy_segment_data(src_obj, klist_token(list), dest_obj,
                                      klist_token(list)))
            return (FALSE);
         list = klist_next(list);
      }
   }

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_copy_remaining_data
|       Purpose:
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Jun 24, 1994 14:58
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_copy_remaining_data(
   kobject src_object,
   kobject dst_object)
{
   klist *list;
   kpresentation *pres;
   int token;
   
   list = dst_object->presentations;

   while (list != NULL)
   {
      token = klist_token(list);
      
      if (!_kdms_get_segment_info(dst_object, token, &pres) || !pres) 
      {
	 list = klist_next(list);
	 continue;
      }
      
      if (pres->segment->data == NULL && pres->segment->file == NULL)
	 if (kdms_query_segment(src_object, ktoken_to_string(token)) &&
	     !_kdms_copy_segment_data(src_object, token, 
				      dst_object, token))
	    return (FALSE);

      list = klist_next(list);
   }

   return (TRUE);
}

/************************************************************
*
*  Routine Name: kdms_copy_data - copy data from
*                one data object to another.
*
*       Purpose: 
*                This function copies data associated with one segment 
*                or all segments from a source object to a destination 
*                output object.  
*
*                The first argument to this function is the source
*                object.  The third argument is the destination
*                object.  The second argument to this function 
*                determines which data segment will be copied. 
*                If this argument is NULL, then this instructs the 
8                kdms_copy_data function to copy the data for all 
*                segments to the destination object.
*
*                If a segment that is being copied already exists in 
*                the destination object, then the segment data is 
*                removed and replaced by the source's segment data.
*                In this event, the physical attributes of the original
*                data in the destination object will remain unchanged.
*                The data will be automatically translated to fit the
*                physical attributes that are already present.
*
*                If the destination object does not
*                contain a segment that exists on the source object
*                and that is being copied with this function, then
*                the said segment will be created.  In this event,
*                the physical attributes that describe the new segment
*                will be used to initialize the presentation attributes
*                for the destination object.
*
*                Data Services attempts to minimize the number of
*                attributes that are copied with the data, as there
*                another function for that purpose (kdms_copy_attributes).
*                However, as mentioned above, there are cases when it
*                is necessary to copy certain attributes that are 
*                critical to the interpretation of the data.  However,
*                this should only occur when the destination does not
*                have the segment or segments being copied.
*
*         Input: src_obj - the object that serves as a source
*                          for the segment data.
*                segment - the data segment to be copied.
*
*        Output: dest_obj - the object that will serve as
*                           a destination in the copy
*                           operation.
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young and Jeremy Worley
*          Date: Aug 20, 1992 15:51
*      Verified:
*  Side Effects: As of the second alpha release of Khoros 2.0, 
*                this function will lock the destination object.
*                This side effect will be removed in future
*                releases.
* Modifications:
*
*************************************************************/

int
kdms_copy_segment_data(
                         kobject src_obj,
                         char *src_segment,
                         kobject dest_obj,
                         char *dest_segment)
{
   int srctoken = kstring_to_token(src_segment),
      desttoken = kstring_to_token(dest_segment);

   if (!_kdms_compiled)
      _kdms_init();

   /*
    * sanity check
    */
   if (src_obj == NULL || dest_obj == NULL)
   {
      _kdms_set_error(KDMS_EOBJ_INVALID);
      return (FALSE);
   }

   /*
    * check the case where nothing really happens
    */
   if (src_obj == dest_obj && srctoken == desttoken)
   {
      _kdms_set_error(KDMS_ENULLOP);
      return (FALSE);
   }

   return (_kdms_copy_data(src_obj, srctoken, dest_obj, desttoken));
}

/*-----------------------------------------------------------
|
|  Routine Name: kdms_copy_object
|
|       Purpose: 
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Dec 01, 1993 09:00
| Modifications:
|
------------------------------------------------------------*/
int
kdms_copy_object(
                   kobject src_obj,
                   kobject dest_obj,
                   int copy_attributes,
                   int copy_data)
{
   char **segments = NULL;
   int i;
   int j;
   int nsegments = 0;

   segments = kdms_get_segment_names(src_obj, &nsegments);

   for (i = 0; i < nsegments; i++)
   {

      /*  if the segment appears to exists in the src, then copy it */
      if (kdms_query_segment(src_obj, segments[i]))
      {

         /* copy the attributes if so desired */
         if (copy_attributes)
            if (!kdms_copy_segment_attributes(src_obj, segments[i],
                                              dest_obj, segments[i])) 
	    {
	       for (j = 0; j < nsegments; j++)
		  kfree(segments[j]);
	       kfree(segments);
	       
	       return (FALSE);
	    }
	 
         /* copy the data if so desired */
         if (copy_data)
            if (!kdms_copy_segment_data(src_obj, segments[i],
                                        dest_obj, segments[i]))
	    {
	       for (j = 0; j < nsegments; j++)
		  kfree(segments[j]);
	       kfree(segments);
	       
               return (FALSE);
	    }
	 
      }
   }

   /* copy the object level attributes if desired */
   if (copy_attributes)
      if (!kdms_copy_segment_attributes(src_obj, NULL, dest_obj, NULL))
      {
	 for (j = 0; j < nsegments; j++)
	    kfree(segments[j]);
	 kfree(segments);
	 return (FALSE);
      }
   
   for (j = 0; j < nsegments; j++)
      kfree(segments[j]);
   kfree(segments);

   return (TRUE);
}

/************************************************************
*
*  Routine Name: kdms_copy_remaining_data - copy remaining data
*       Purpose: kdms_copy_remaining_data copies any data 
*		 that has not been previously written to
*		 in the destination from a specified source.
*         Input: src_object - object to get data from
*		 dst_object - object to copy data to.
*        Output:
*       Returns:
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Jun 24, 1994 14:57
*      Verified:
*  Side Effects:
* Modifications:
*************************************************************/

int 
kdms_copy_remaining_data(
   kobject src_object,
   kobject dst_object)
{
   if (!_kdms_compiled)
      _kdms_init();

   return (_kdms_copy_remaining_data(src_object, dst_object));
}
